/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.connector.rmi;

import java.rmi.NoSuchObjectException;
import java.rmi.Remote;
import java.rmi.RemoteException;

import javax.management.Notification;
import javax.management.NotificationFilter;

/**
 * This class is serialized from client to the server, and forwards calls to remote filter. <p>
 * It shadows the fact that the listener is an RMI remote object.
 * It implements java.rmi.Remote as a tagging interface, to avoid dependencies of the specification 1.0
 * implementation from mx4j classes
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.1 $
 */
public abstract class RemoteNotificationFilterSupport implements NotificationFilter, Remote
{
	/**
	 * When this class is serialized to the server, the stub of this remote object
	 * will be sent to the server, enabling it to call back the client.
	 * This must be the ONLY data member.
	 */
	private RemoteNotificationFilter m_clientSideFilter;

	public RemoteNotificationFilterSupport(NotificationFilter clientFilter)
	{
		m_clientSideFilter = new RemoteNotificationFilterImpl(clientFilter);
	}

	protected abstract void export() throws RemoteException;

	protected abstract void unexport() throws NoSuchObjectException;

	protected RemoteNotificationFilter getRemoteNotificationFilter()
	{
		return m_clientSideFilter;
	}

	public boolean isNotificationEnabled(Notification notification)
	{
		// We are on the server
		try
		{
			// This is a remote call from the server to the client
			return m_clientSideFilter.isNotificationEnabled(notification);
		}
		catch (RemoteException x)
		{
			// We are on the server, we rethrow so that the server will log this exception
			throw new RuntimeException(x.toString());
		}
	}

   public int hashCode()
   {
      return m_clientSideFilter.hashCode();
   }

   public boolean equals(Object obj)
   {
      if (obj == null) {return false;}
      if (obj == this) {return true;}

      try
      {
         RemoteNotificationFilterSupport other = (RemoteNotificationFilterSupport)obj;
         return m_clientSideFilter.equals(other.m_clientSideFilter);
      }
      catch (ClassCastException x)
      {
         return false;
      }
   }
}
