package org.jboss.cache.marshall;

import org.jboss.cache.Cache;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.cache.Region;
import org.jboss.cache.config.CacheLoaderConfig;
import org.jboss.cache.config.CacheLoaderConfig.IndividualCacheLoaderConfig;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.eviction.LRUConfiguration;
import org.jboss.cache.eviction.LRUPolicy;
import org.jboss.cache.loader.FileCacheLoaderConfig;
import org.jboss.cache.util.TestingUtil;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNotNull;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Tests marshalling/unmarshalling during cache loader operations involving types
 * not visible to the cache's default classloader.
 *
 * @author <a href="mailto:brian.stansberry@jboss.org">Brian Stansberry</a>
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 * @since 2.1.0
 */
@Test(groups = "functional")
public class CacheLoaderMarshallingTest extends RegionBasedMarshallingTestBase
{
   private static final String tmpDir = System.getProperty("java.io.tmpdir") + File.separatorChar + "CacheLoaderMarshallingTest";

   private Cache<Object, Object> cache;
   private Fqn fqn = Fqn.fromString("/a");

   @BeforeMethod(alwaysRun = true)
   protected void setUp() throws Exception
   {
      originalClassLoader = Thread.currentThread().getContextClassLoader();
   }

   @AfterMethod(alwaysRun = true)
   protected void tearDown()
   {
      resetContextClassLoader();
      TestingUtil.killCaches(cache);

      File f = new File(tmpDir);
      if (f.exists())
         if (!f.delete())
            f.deleteOnExit();
   }

   public void testCacheLoaderMarshalling() throws Exception
   {
      cacheLoaderMarshallingTest(false);
   }

   public void testCacheLoaderRegionBasedMarshalling() throws Exception
   {
      cacheLoaderMarshallingTest(true);
   }
   
   public void testLoadNodesAtRootOfRegion() throws Exception
   {
      String rootRegionName = "/myregion";
      String hereFqn = rootRegionName + "/here";
      
      cache = createCache(true);
      cache.start();

      Region r = cache.getRegion(Fqn.fromString(rootRegionName), true);
      r.registerContextClassLoader(Thread.currentThread().getContextClassLoader());
      r.activate();
      
      cache.put(rootRegionName, "a key", "a value");
      cache.put(hereFqn, "another key", "another value");

      r.deactivate();
      r.unregisterContextClassLoader();
      
      cache.stop();
      
      cache.start();
      
      r = cache.getRegion(Fqn.fromString(rootRegionName), true);
      r.registerContextClassLoader(Thread.currentThread().getContextClassLoader());
      r.activate();
      
      Node<Object, Object> rootRegionNode = cache.getNode(rootRegionName);
      Node<Object, Object> hereNode = cache.getNode(hereFqn);
      assertNotNull(rootRegionNode);
      assertNotNull(hereNode);
      
      assertEquals(hereNode.get("another key"), "another value");
      assertEquals(rootRegionNode.get("a key"), "a value");
   }

   private void cacheLoaderMarshallingTest(boolean useRegionBased) throws Exception
   {
      cache = createCache(useRegionBased);
      cache.start();

      FooClassLoader loader = new FooClassLoader(originalClassLoader);

      if (useRegionBased)
      {
         Region r = cache.getRegion(Fqn.ROOT, true);
         r.registerContextClassLoader(loader);
         r.activate();
      }

      Class clazz = loader.loadFoo();
      Object obj = clazz.newInstance();

      Thread.currentThread().setContextClassLoader(loader);
      cache.put(fqn, "key", obj);

      this.resetContextClassLoader();
      cache.evict(fqn);

      Thread.currentThread().setContextClassLoader(loader);
      assertEquals(obj, cache.get(fqn, "key"));
   }

   private Cache createCache(boolean useRegionBased)
   {
      Cache cache = new DefaultCacheFactory<Object, Object>().createCache(false);
      Configuration config = cache.getConfiguration();
      config.setUseRegionBasedMarshalling(useRegionBased);
      config.setInactiveOnStartup(useRegionBased);

      EvictionConfig ec = new EvictionConfig();
      ec.setDefaultEvictionPolicyClass(LRUPolicy.class.getName());
      ec.setWakeupIntervalSeconds(1000);  // a long time; really disabled
      EvictionRegionConfig erc = new EvictionRegionConfig();
      erc.setRegionFqn(Fqn.ROOT);
      erc.setRegionName("_default_");
      LRUConfiguration epc = new LRUConfiguration();
      epc.setMaxNodes(1000);
      epc.setTimeToLiveSeconds(1000);
      erc.setEvictionPolicyConfig(epc);
      List<EvictionRegionConfig> ercs = new ArrayList<EvictionRegionConfig>();
      ercs.add(erc);
      ec.setEvictionRegionConfigs(ercs);
      config.setEvictionConfig(ec);

      CacheLoaderConfig clc = new CacheLoaderConfig();
      clc.setPassivation(true);
      clc.setShared(false);
      FileCacheLoaderConfig fclc = new FileCacheLoaderConfig();
      fclc.setLocation(tmpDir);

      clc.setIndividualCacheLoaderConfigs(Collections.<IndividualCacheLoaderConfig>singletonList(fclc));
      config.setCacheLoaderConfig(clc);

      return cache;
   }

}
