#define _XOPEN_SOURCE 500 /* For strdup(3) */
#include "../test-tools.h"
#include "http.h"
#include "services.h"
#include "system.h"
#include <string.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <libxml/xmlsave.h>

static const char *soap_mime_type = "text/xml"; /* SOAP/1.1 requires text/xml */

/* Used to choose proper name space for message elements.
 * See _isds_register_namespaces(). */
typedef enum {
    MESSAGE_NS_1,
    MESSAGE_NS_UNSIGNED,
    MESSAGE_NS_SIGNED_INCOMING,
    MESSAGE_NS_SIGNED_OUTGOING,
    MESSAGE_NS_SIGNED_DELIVERY,
    MESSAGE_NS_OTP
} message_ns_type;

#define SOAP_NS "http://schemas.xmlsoap.org/soap/envelope/"
#define SOAP2_NS "http://www.w3.org/2003/05/soap-envelope"
#define ISDS1_NS "http://isds.czechpoint.cz"
#define ISDS_NS "http://isds.czechpoint.cz/v20"
#define OISDS_NS "http://isds.czechpoint.cz/v20/asws"
#define SISDS_INCOMING_NS "http://isds.czechpoint.cz/v20/message"
#define SISDS_OUTGOING_NS "http://isds.czechpoint.cz/v20/SentMessage"
#define SISDS_DELIVERY_NS "http://isds.czechpoint.cz/v20/delivery"
#define SCHEMA_NS "http://www.w3.org/2001/XMLSchema"
#define DEPOSIT_NS "urn:uschovnaWSDL"


struct service {
    service_id id;
    const char *end_point;
    const xmlChar *name_space;
    const xmlChar *name;
    http_error (*function) (const struct http_connection *, xmlDocPtr,
            xmlXPathContextPtr, xmlNodePtr, xmlDocPtr, xmlNodePtr,
            const void *arguments);
};

/* Following EXTRACT_* macros expect @xpath_ctx, @error, @message,
 * and leave label. */
#define ELEMENT_EXISTS(element, allow_multiple) { \
    xmlXPathObjectPtr result = NULL; \
    result = xmlXPathEvalExpression(BAD_CAST element, xpath_ctx); \
    if (NULL == result) { \
        error = HTTP_ERROR_SERVER; \
        goto leave; \
    } \
    if (xmlXPathNodeSetIsEmpty(result->nodesetval)) { \
            xmlXPathFreeObject(result); \
            test_asprintf(&message, "Element %s does not exist", element); \
            error = HTTP_ERROR_CLIENT; \
            goto leave; \
    } else { \
        if (!allow_multiple && result->nodesetval->nodeNr > 1) { \
            xmlXPathFreeObject(result); \
            test_asprintf(&message, "Multiple %s element", element); \
            error = HTTP_ERROR_CLIENT; \
            goto leave; \
        } \
    } \
    xmlXPathFreeObject(result); \
}

#define EXTRACT_STRING(element, string) { \
    xmlXPathObjectPtr result = NULL; \
    result = xmlXPathEvalExpression(BAD_CAST element "/text()", xpath_ctx); \
    if (NULL == result) { \
        error = HTTP_ERROR_SERVER; \
        goto leave; \
    } \
    if (!xmlXPathNodeSetIsEmpty(result->nodesetval)) { \
        if (result->nodesetval->nodeNr > 1) { \
            xmlXPathFreeObject(result); \
            test_asprintf(&message, "Multiple %s element", element); \
            error = HTTP_ERROR_CLIENT; \
            goto leave; \
        } \
        (string) = (char *) \
            xmlXPathCastNodeSetToString(result->nodesetval); \
        if (!(string)) { \
            xmlXPathFreeObject(result); \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
    } \
    xmlXPathFreeObject(result); \
}

#define EXTRACT_BOOLEAN(element, booleanPtr) { \
    char *string = NULL; \
    EXTRACT_STRING(element, string); \
     \
    if (NULL != string) { \
        (booleanPtr) = calloc(1, sizeof(*(booleanPtr))); \
        if (NULL == (booleanPtr)) { \
            free(string); \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
         \
        if (!xmlStrcmp((xmlChar *)string, BAD_CAST "true") || \
                !xmlStrcmp((xmlChar *)string, BAD_CAST "1")) \
            *(booleanPtr) = 1; \
        else if (!xmlStrcmp((xmlChar *)string, BAD_CAST "false") || \
                !xmlStrcmp((xmlChar *)string, BAD_CAST "0")) \
            *(booleanPtr) = 0; \
        else { \
            test_asprintf(&message, \
                    "%s value is not valid boolean: %s", \
                    element, string); \
            free(string); \
            error = HTTP_ERROR_CLIENT; \
            goto leave; \
        } \
         \
        free(string); \
    } \
} 


#define EXTRACT_DATE(element, tmPtr) { \
    char *string = NULL; \
    EXTRACT_STRING(element, string); \
    if (NULL != string) { \
        (tmPtr) = calloc(1, sizeof(*(tmPtr))); \
        if (NULL == (tmPtr)) { \
            free(string); \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
        error = _server_datestring2tm(string, (tmPtr)); \
        if (error) { \
            if (error == HTTP_ERROR_CLIENT) { \
                test_asprintf(&message, "%s value is not a valid date: %s", \
                        element, string); \
            } \
            free(string); \
            goto leave; \
        } \
        free(string); \
    } \
}

/* Following INSERT_* macros expect @error and leave label */
#define INSERT_STRING_WITH_NS(parent, ns, element, string) \
    { \
        xmlNodePtr node = xmlNewTextChild(parent, ns, BAD_CAST (element), \
                (xmlChar *) (string)); \
        if (NULL == node) { \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
    }

#define INSERT_STRING(parent, element, string) \
    { INSERT_STRING_WITH_NS(parent, NULL, element, string) }

#define INSERT_LONGINTPTR(parent, element, longintPtr) { \
    if ((longintPtr)) { \
        char *buffer = NULL; \
        /* FIXME: locale sensitive */ \
        if (-1 == test_asprintf(&buffer, "%ld", *(longintPtr))) { \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
        INSERT_STRING(parent, element, buffer) \
        free(buffer); \
    } else { INSERT_STRING(parent, element, NULL) } \
}

#define INSERT_TIMEVALPTR(parent, element, timevalPtr) { \
    if (NULL != (timevalPtr)) { \
        char *buffer = NULL; \
        error = timeval2timestring(timevalPtr, &buffer); \
        if (error) { \
            free(buffer); \
            goto leave; \
        } \
        INSERT_STRING(parent, element, buffer); \
        free(buffer); \
    } else { \
        INSERT_STRING(parent, element, NULL); \
    } \
}

#define INSERT_TMPTR(parent, element, tmPtr) { \
    if (NULL != (tmPtr)) { \
        char *buffer = NULL; \
        error = tm2datestring(tmPtr, &buffer); \
        if (error) { \
            free(buffer); \
            goto leave; \
        } \
        INSERT_STRING(parent, element, buffer); \
        free(buffer); \
    } else { \
        INSERT_STRING(parent, element, NULL); \
    } \
}

#define INSERT_ELEMENT(child, parent, element) \
    { \
        (child) = xmlNewChild((parent), NULL, BAD_CAST (element), NULL); \
        if (NULL == (child)) { \
            error = HTTP_ERROR_SERVER; \
            goto leave; \
        } \
    }


/* Insert dmStatus or similar subtree
 * @parent is element to insert to
 * @dm is true for dmStatus, otherwise dbStatus
 * @code is status code as string
 * @message is UTF-8 encoded message
 * @db_ref_number is optinal reference number propagated if not @dm
 * @return 0 on success, otherwise non-0. */
static http_error insert_isds_status(xmlNodePtr parent, _Bool dm,
        const xmlChar *code, const xmlChar *message,
        const xmlChar *db_ref_number) {
    http_error error = HTTP_ERROR_SUCCESS;
    xmlNodePtr status;
    
    if (NULL == code || NULL == message) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    INSERT_ELEMENT(status, parent, (dm) ? "dmStatus" : "dbStatus");
    INSERT_STRING(status, (dm) ? "dmStatusCode" : "dbStatusCode", code);
    INSERT_STRING(status, (dm) ? "dmStatusMessage" : "dbStatusMessage", message);
    if (!dm && NULL != db_ref_number) {
        INSERT_STRING(status, "dbStatusRefNumber", db_ref_number);
    }

leave:
    return error;
}


/* Convert struct tm *@time to UTF-8 ISO 8601 date @string. */
static http_error tm2datestring(const struct tm *time, char **string) {
    if (NULL == time || NULL == string) return HTTP_ERROR_SERVER;

    if (-1 == test_asprintf(string, "%d-%02d-%02d",
                time->tm_year + 1900, time->tm_mon + 1, time->tm_mday))
        return HTTP_ERROR_SERVER;

    return HTTP_ERROR_SUCCESS;
}


/* Convert struct timeval *@time to UTF-8 ISO 8601 date-time @string. It
 * respects the @time microseconds too. */
static http_error timeval2timestring(const struct timeval *time,
        char **string) {
    struct tm broken;

    if (!time || !string) return HTTP_ERROR_SERVER;

    if (!gmtime_r(&time->tv_sec, &broken)) return HTTP_ERROR_SERVER;
    if (time->tv_usec < 0 || time->tv_usec > 999999) return HTTP_ERROR_SERVER;

    /* TODO: small negative year should be formatted as "-0012". This is not
     * true for glibc "%04d". We should implement it.
     * TODO: What's type of time->tv_usec exactly? Unsigned? Absolute?
     * See <http://www.w3.org/TR/2001/REC-xmlschema-2-20010502/#dateTime> */ 
    if (-1 == test_asprintf(string,
                "%04d-%02d-%02dT%02d:%02d:%02d.%06ld",
                broken.tm_year + 1900, broken.tm_mon + 1, broken.tm_mday,
                broken.tm_hour, broken.tm_min, broken.tm_sec,
                time->tv_usec))
        return HTTP_ERROR_SERVER;

    return HTTP_ERROR_SUCCESS;
}


/* Compare dates represented by pointer to struct tm.
 * @return 0 if equalued, non-0 otherwise. */
static int datecmp(const struct tm *a, const struct tm *b) {
    if (NULL == a && b == NULL) return 0;
    if ((NULL == a && b != NULL) || (NULL != a && b == NULL)) return 1;
    if (a->tm_year != b->tm_year) return 1;
    if (a->tm_mon != b->tm_mon) return 1;
    if (a->tm_mday != b->tm_mday) return 1;
    return 0;
}


/* Implement DummyOperation */
static http_error service_DummyOperation(
        const struct http_connection *connection, const xmlDocPtr soap_request,
        xmlXPathContextPtr xpath_ctx, xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    return insert_isds_status(isds_response, 1, BAD_CAST "0000",
            BAD_CAST "Success", NULL);
}


/* Implement Re-signISDSDocument.
 * It sends document from request back.
 * @arguments is pointer to struct arguments_DS_Dz_ResignISDSDocument */
static http_error service_ResignISDSDocument(
        const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    http_error error = HTTP_ERROR_SUCCESS;
    const char *code = "9999";
    char *message = NULL;
    const struct arguments_DS_Dz_ResignISDSDocument *configuration =
        (const struct arguments_DS_Dz_ResignISDSDocument *)arguments;
    char *data = NULL;

    if (NULL == configuration || NULL == configuration->status_code ||
            NULL == configuration->status_message) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    EXTRACT_STRING("isds:dmDoc", data);
    if (NULL == data) {
        message = strdup("Missing isds:dmDoc");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }


    /* dmResultDoc is mandatory in response */
    if (xmlStrcmp(BAD_CAST configuration->status_code, BAD_CAST "0000")) {
        free(data);
        data = NULL;
    }
    INSERT_STRING(isds_response, "dmResultDoc", data);

    if (configuration->valid_to != NULL) {
        error = tm2datestring(configuration->valid_to, &data);
        if (error) {
            message = strdup("Could not format date");
            goto leave;
        }
        INSERT_STRING(isds_response, "dmValidTo", data);
    }

    code = configuration->status_code;
    message = strdup(configuration->status_message);

leave:
    if (HTTP_ERROR_SERVER != error) {
        http_error next_error = insert_isds_status(isds_response, 1,
                BAD_CAST code, BAD_CAST message, NULL);
        if (HTTP_ERROR_SUCCESS != next_error) error = next_error;
    }
    free(data);
    free(message);
    return error;
}


/* Implement EraseMessage.
 * @arguments is pointer to struct arguments_DS_DsManage_ChangeISDSPassword */
static http_error service_EraseMessage(const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    http_error error = HTTP_ERROR_SUCCESS;
    char *code = "9999", *message = NULL;
    const struct arguments_DS_Dx_EraseMessage *configuration =
        (const struct arguments_DS_Dx_EraseMessage *)arguments;
    char *message_id = NULL;
    _Bool *incoming = NULL;

    if (NULL == configuration || NULL == configuration->message_id) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    EXTRACT_STRING("isds:dmID", message_id);
    if (NULL == message_id) {
        message = strdup("Missing isds:dmID");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    EXTRACT_BOOLEAN("isds:dmIncoming", incoming);
    if (NULL == incoming) {
        message = strdup("Missing isds:dmIncoming");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    if (xmlStrcmp((const xmlChar *) configuration->message_id,
                (const xmlChar *) message_id)) {
        code = "1219";
        message = strdup("Message is not in the long term storage");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    if (configuration->incoming != *incoming) {
        code = "1219";
        message = strdup("Message direction mismatches");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    
    code = "0000";
    message = strdup("Success");
leave:
    if (HTTP_ERROR_SERVER != error) {
        http_error next_error = insert_isds_status(isds_response, 1,
                BAD_CAST code, BAD_CAST message, NULL);
        if (HTTP_ERROR_SUCCESS != next_error) error = next_error;
    }
    free(incoming);
    free(message_id);
    free(message);
    return error;
}


/* Insert list of credit info as XSD:tCiRecord XML tree.
 * @isds_response is XML node with the response
 * @history is list of struct server_credit_event. If NULL, no ciRecord XML
 * subtree will be created. */
static http_error insert_ciRecords(xmlNodePtr isds_response,
        const struct server_list *history) {
    http_error error = HTTP_ERROR_SUCCESS;
    xmlNodePtr records, record;

    if (NULL == isds_response) return HTTP_ERROR_SERVER;
    if (NULL == history) return HTTP_ERROR_SUCCESS;

    INSERT_ELEMENT(records, isds_response, "ciRecords");
    for (const struct server_list *item = history; NULL != item;
            item = item->next) {
        const struct server_credit_event *event =
            (struct server_credit_event*)item->data;
        
        INSERT_ELEMENT(record, records, "ciRecord");
        if (NULL == event) continue;

        INSERT_TIMEVALPTR(record, "ciEventTime", event->time);
        switch(event->type) {
            case SERVER_CREDIT_CHARGED:
                INSERT_STRING(record, "ciEventType", "1");
                INSERT_STRING(record, "ciTransID",
                        event->details.charged.transaction);
                break;
            case SERVER_CREDIT_DISCHARGED:
                INSERT_STRING(record, "ciEventType", "2");
                INSERT_STRING(record, "ciTransID",
                        event->details.discharged.transaction);
                break;
            case SERVER_CREDIT_MESSAGE_SENT:
                INSERT_STRING(record, "ciEventType", "3");
                INSERT_STRING(record, "ciRecipientID",
                        event->details.message_sent.recipient);
                INSERT_STRING(record, "ciPDZID",
                        event->details.message_sent.message_id);
                break;
            case SERVER_CREDIT_STORAGE_SET:
                INSERT_STRING(record, "ciEventType", "4");
                INSERT_LONGINTPTR(record, "ciNewCapacity",
                        &event->details.storage_set.new_capacity);
                INSERT_TMPTR(record, "ciNewFrom",
                        event->details.storage_set.new_valid_from);
                INSERT_TMPTR(record, "ciNewTo",
                        event->details.storage_set.new_valid_to);
                INSERT_LONGINTPTR(record, "ciOldCapacity",
                        event->details.storage_set.old_capacity);
                INSERT_TMPTR(record, "ciOldFrom",
                        event->details.storage_set.old_valid_from);
                INSERT_TMPTR(record, "ciOldTo",
                        event->details.storage_set.old_valid_to);
                INSERT_STRING(record, "ciDoneBy",
                        event->details.storage_set.initiator);
                break;
            case SERVER_CREDIT_EXPIRED:
                INSERT_STRING(record, "ciEventType", "5");
                break;
            default:
                error = HTTP_ERROR_SERVER;
                goto leave;
        }
        INSERT_LONGINTPTR(record, "ciCreditChange", &event->credit_change);
        INSERT_LONGINTPTR(record, "ciCreditAfter", &event->new_credit);

    }

leave:
    return error;
}


/* Implement DataBoxCreditInfo.
 * @arguments is pointer to struct arguments_DS_df_DataBoxCreditInfo */
static http_error service_DataBoxCreditInfo(
        const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    http_error error = HTTP_ERROR_SUCCESS;
    const char *code = "9999";
    char *message = NULL;
    const struct arguments_DS_df_DataBoxCreditInfo *configuration =
        (const struct arguments_DS_df_DataBoxCreditInfo *)arguments;
    char *box_id = NULL;
    struct tm *from_date = NULL, *to_date = NULL;

    if (NULL == configuration || NULL == configuration->status_code ||
            NULL == configuration->status_message) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    EXTRACT_STRING("isds:dbID", box_id);
    if (NULL == box_id) {
        message = strdup("Missing isds:dbID");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    if (NULL != configuration->box_id &&
            xmlStrcmp(BAD_CAST configuration->box_id,
                    BAD_CAST box_id)) {
        code = "9999";
        message = strdup("Unexpected isds:dbID value");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    
    ELEMENT_EXISTS("isds:ciFromDate", 0);
    EXTRACT_DATE("isds:ciFromDate", from_date);
    if (datecmp(configuration->from_date, from_date)) {
        code = "9999";
        message = strdup("Unexpected isds:ciFromDate value");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    ELEMENT_EXISTS("isds:ciTodate", 0);
    EXTRACT_DATE("isds:ciTodate", to_date);
    if (datecmp(configuration->to_date, to_date)) {
        code = "9999";
        message = strdup("Unexpected isds:ciTodate value");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    INSERT_LONGINTPTR(isds_response, "currentCredit",
        &configuration->current_credit);
    INSERT_STRING(isds_response, "notifEmail", configuration->email);
    if ((error = insert_ciRecords(isds_response, configuration->history))) {
        goto leave;
    }

    code = configuration->status_code;
    message = strdup(configuration->status_message);
leave:
    if (HTTP_ERROR_SERVER != error) {
        http_error next_error = insert_isds_status(isds_response, 0,
                BAD_CAST code, BAD_CAST message, NULL);
        if (HTTP_ERROR_SUCCESS != next_error) error = next_error;
    }
    free(box_id);
    free(from_date);
    free(to_date);
    free(message);
    return error;
}


/* Common part for ChangeISDSPassword and ChangePasswordOTP.
 * @code is output pointer to static string
 * @pass_message is output pointer to auto-allocated string
 * @arguments is pointer to struct arguments_DS_DsManage_ChangeISDSPassword */
static http_error check_passwd(
        const char *username, const char *current_password,
        xmlXPathContextPtr xpath_ctx,
        char **code, char **pass_message) {
    http_error error = HTTP_ERROR_SUCCESS;
    char *message = NULL;
    char *old_password = NULL, *new_password = NULL;
    size_t length;

    if (NULL == username || NULL == current_password ||
            NULL == code || NULL == pass_message) {
        return HTTP_ERROR_SERVER;
    }

    *code = "9999";


    /* Parse request */
    EXTRACT_STRING("isds:dbOldPassword", old_password);
    if (NULL == old_password) {
        message = strdup("Empty isds:dbOldPassword");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    EXTRACT_STRING("isds:dbNewPassword", new_password);
    if (NULL == new_password) {
        message = strdup("Empty isds:dbOldPassword");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    /* Check defined cases */
    if (strcmp(current_password, old_password)) {
        *code = "1090";
        message = strdup("Bad current password");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    length = strlen(new_password);

    if (length < 8 || length > 32) {
        *code = "1066";
        message = strdup("Too short or too long");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    {
        const char lower[] = "abcdefghijklmnopqrstuvwxyz";
        const char upper[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        const char digit[] = "0123456789";
        const char special[] = "!#$%&()*+,-.:=?@[]_{}|~";
        _Bool has_lower = 0, has_upper = 0, has_digit=0;

        for (int i = 0; i < length; i++) {
            if (NULL != strchr(lower, new_password[i]))
                has_lower = 1;
            else if (NULL != strchr(upper, new_password[i]))
                has_upper = 1;
            else if (NULL != strchr(digit, new_password[i]))
                has_digit = 1;
            else if (NULL == strchr(special, new_password[i])) {
                *code = "1079";
                message = strdup("Password contains forbidden character");
                error = HTTP_ERROR_CLIENT;
                goto leave;
            }
        }

        if (!has_lower || !has_upper || !has_digit) {
            *code = "1080";
            message = strdup("Password does not contain lower cased letter, "
                    "upper cased letter and a digit");
            error = HTTP_ERROR_CLIENT;
            goto leave;
        }
    }

    if (!strcmp(old_password, new_password)) {
        *code = "1067";
        message = strdup("New password same as current one");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    if (NULL != strstr(new_password, username)) {
        *code = "1082";
        message = strdup("New password contains user ID");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    for (int i = 0; i < length - 2; i++) {
        if (new_password[i] == new_password[i+1] &&
                new_password[i] == new_password[i+2]) {
            *code = "1083";
            message = strdup("Password contains sequence "
                    "of three identical characters");
            error = HTTP_ERROR_CLIENT;
            goto leave;
        }
    }
    
    {
        const char *forbidden_prefix[] = { "qwert", "asdgf", "12345" };
        for (int i = 0; i < sizeof(forbidden_prefix)/sizeof(*forbidden_prefix);
                i++) {
            if (!strncmp(new_password, forbidden_prefix[i],
                        strlen(forbidden_prefix[i]))) {
                *code = "1083";
                message = strdup("Password has forbidden prefix");
                error = HTTP_ERROR_CLIENT;
                goto leave;
            }
        }
    }

    *code = "0000";
    message = strdup("Success");
leave:
    free(old_password);
    free(new_password);
    *pass_message = message;
    return error;
}


/* Implement ChangeISDSPassword.
 * @arguments is pointer to struct arguments_DS_DsManage_ChangeISDSPassword */
static http_error service_ChangeISDSPassword(
        const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    http_error error = HTTP_ERROR_SUCCESS;
    char *code = "9999", *message = NULL;
    const struct arguments_DS_DsManage_ChangeISDSPassword *configuration =
        (const struct arguments_DS_DsManage_ChangeISDSPassword *)arguments;

    if (NULL == configuration || NULL == configuration->username ||
            NULL == configuration->current_password) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    /* Check for common password rules */
    error = check_passwd(
        configuration->username, configuration->current_password,
        xpath_ctx, &code, &message);

leave:
    if (HTTP_ERROR_SERVER != error) {
        http_error next_error = insert_isds_status(isds_response, 0,
                BAD_CAST code, BAD_CAST message, NULL);
        if (HTTP_ERROR_SUCCESS != next_error) error = next_error;
    }
    free(message);
    return error;
}


/* Implement ChangePasswordOTP.
 * @arguments is pointer to struct
 * arguments_asws_changePassword_ChangePasswordOTP */
static http_error service_ChangePasswordOTP(
        const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    http_error error = HTTP_ERROR_SUCCESS;
    char *code = "9999", *message = NULL;
    const struct arguments_asws_changePassword_ChangePasswordOTP *configuration
        = (const struct arguments_asws_changePassword_ChangePasswordOTP *)
        arguments;
    char *method = NULL;

    if (NULL == configuration || NULL == configuration->username ||
            NULL == configuration->current_password) {
        error = HTTP_ERROR_SERVER;
        goto leave;
    }

    /* Chek for OTP method */
    EXTRACT_STRING("isds:dbOTPType", method);
    if (NULL == method) {
        message = strdup("Empty isds:dbOTPType");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }
    if ((configuration->method == AUTH_OTP_HMAC && strcmp(method, "HOTP")) ||
        (configuration->method == AUTH_OTP_TIME && strcmp(method, "TOTP"))) {
        message = strdup("isds:dbOTPType does not match OTP method");
        error = HTTP_ERROR_CLIENT;
        goto leave;
    }

    /* Check for common password rules */
    error = check_passwd(
        configuration->username, configuration->current_password,
        xpath_ctx, &code, &message);

leave:
    if (HTTP_ERROR_SERVER != error) {
        http_error next_error = insert_isds_status(isds_response, 0,
                BAD_CAST code, BAD_CAST message,
                BAD_CAST configuration->reference_number);
        if (HTTP_ERROR_SUCCESS != next_error) error = next_error;
    }
    free(message);
    free(method);
    return error;
}


/* Implement SendSMSCode.
 * @arguments is pointer to struct arguments_asws_changePassword_SendSMSCode */
static http_error service_SendSMSCode(
        const struct http_connection *connection,
        const xmlDocPtr soap_request, xmlXPathContextPtr xpath_ctx,
        const xmlNodePtr isds_request,
        xmlDocPtr soap_response, xmlNodePtr isds_response,
        const void *arguments) {
    const struct arguments_asws_changePassword_SendSMSCode *configuration
        = (const struct arguments_asws_changePassword_SendSMSCode *)
        arguments;

    if (NULL == configuration || NULL == configuration->status_code ||
            NULL == configuration->status_message) {
        return HTTP_ERROR_SERVER;
    }

    return insert_isds_status(isds_response, 0,
                BAD_CAST configuration->status_code,
                BAD_CAST configuration->status_message,
                BAD_CAST configuration->reference_number);
}


/* List of implemented services */
static struct service services[] = {
    { SERVICE_DS_Dz_DummyOperation,
        "DS/dz", BAD_CAST ISDS_NS, BAD_CAST "DummyOperation",
        service_DummyOperation },
    { SERVICE_DS_Dz_ResignISDSDocument,
        "DS/dz", BAD_CAST ISDS_NS, BAD_CAST "Re-signISDSDocument",
        service_ResignISDSDocument },
    { SERVICE_DS_df_DataBoxCreditInfo,
        "DS/df", BAD_CAST ISDS_NS, BAD_CAST "DataBoxCreditInfo",
        service_DataBoxCreditInfo },
    { SERVICE_DS_DsManage_ChangeISDSPassword,
        "DS/DsManage", BAD_CAST ISDS_NS, BAD_CAST "ChangeISDSPassword",
        service_ChangeISDSPassword },
    { SERVICE_DS_Dx_EraseMessage,
        "DS/dx", BAD_CAST ISDS_NS, BAD_CAST "EraseMessage",
        service_EraseMessage },
    { SERVICE_asws_changePassword_ChangePasswordOTP,
        "/asws/changePassword", BAD_CAST OISDS_NS, BAD_CAST "ChangePasswordOTP",
        service_ChangePasswordOTP },
    { SERVICE_asws_changePassword_SendSMSCode,
        "/asws/changePassword", BAD_CAST OISDS_NS, BAD_CAST "SendSMSCode",
        service_SendSMSCode },
};


/* Makes known all relevant namespaces to given XPath context
 * @xpath_ctx is XPath context
 * @otp_ns selects name space for the request and response know as "isds".
 * Use true for OTP-authenticated password change services, otherwise false.
 * @message_ns selects proper message name space. Unsigned and signed
 * messages and delivery info's differ in prefix and URI.
 * @return 0 in success, otherwise not 0. */
static int register_namespaces(xmlXPathContextPtr xpath_ctx,
        const _Bool otp_ns, const message_ns_type message_ns) {
    const xmlChar *service_namespace = NULL;
    const xmlChar *message_namespace = NULL;

    if (!xpath_ctx) return -1;

    if (otp_ns) {
        service_namespace = BAD_CAST OISDS_NS;
    } else {
        service_namespace = BAD_CAST ISDS_NS;
    }

    switch(message_ns) {
        case MESSAGE_NS_1:
            message_namespace = BAD_CAST ISDS1_NS; break;
        case MESSAGE_NS_UNSIGNED:
            message_namespace = BAD_CAST ISDS_NS; break;
        case MESSAGE_NS_SIGNED_INCOMING:
            message_namespace = BAD_CAST SISDS_INCOMING_NS; break;
        case MESSAGE_NS_SIGNED_OUTGOING:
            message_namespace = BAD_CAST SISDS_OUTGOING_NS; break;
        case MESSAGE_NS_SIGNED_DELIVERY:
            message_namespace = BAD_CAST SISDS_DELIVERY_NS; break;
        default:
            return -1;
    }

    if (xmlXPathRegisterNs(xpath_ctx, BAD_CAST "soap", BAD_CAST SOAP_NS))
        return -1;
    if (xmlXPathRegisterNs(xpath_ctx, BAD_CAST "isds", service_namespace))
        return -1;
    if (xmlXPathRegisterNs(xpath_ctx, BAD_CAST "sisds", message_namespace))
        return -1;
    if (xmlXPathRegisterNs(xpath_ctx, BAD_CAST "xs", BAD_CAST SCHEMA_NS))
        return -1;
    if (xmlXPathRegisterNs(xpath_ctx, BAD_CAST "deposit", BAD_CAST DEPOSIT_NS))
        return -1;
    return 0;
}


/* Parse soap request, pass it to service endpoint and respond to it.
 * It sends final HTTP response. */
void soap(const struct http_connection *connection,
        const struct service_configuration *configuration,
        const void *request, size_t request_length, const char *end_point) {
    xmlDocPtr request_doc = NULL;
    xmlXPathContextPtr xpath_ctx = NULL;
    xmlXPathObjectPtr request_soap_body = NULL;
    xmlNodePtr isds_request = NULL; /* pointer only */
    _Bool service_handled = 0, service_passed = 0;
    xmlDocPtr response_doc = NULL;
    xmlNodePtr response_soap_envelope = NULL, response_soap_body = NULL,
               isds_response = NULL;
    xmlNsPtr soap_ns = NULL, isds_ns = NULL;
    char *response_name = NULL;
    xmlBufferPtr http_response_body = NULL;
    xmlSaveCtxtPtr save_ctx = NULL;


    if (NULL == configuration) {
        http_send_response_500(connection,
                "Second argument of soap() is NULL");
        return;
    }

    if (NULL == request || request_length == 0) {
        http_send_response_400(connection, "Client sent empty body");
        return;
    }

    request_doc = xmlParseMemory(request, request_length);
    if (NULL == request_doc) {
        http_send_response_400(connection, "Client sent invalid XML document");
        return;
    }
    
    xpath_ctx = xmlXPathNewContext(request_doc);
    if (NULL == xpath_ctx) {
        xmlFreeDoc(request_doc);
        http_send_response_500(connection, "Could not create XPath context");
        return;
    }

    if (register_namespaces(xpath_ctx, 0, MESSAGE_NS_UNSIGNED)) {
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_500(connection,
                "Could not register name spaces to the XPath context");
        return;
    }

    /* Get SOAP Body */
    request_soap_body = xmlXPathEvalExpression(
            BAD_CAST "/soap:Envelope/soap:Body", xpath_ctx);
    if (NULL == request_soap_body) {
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_400(connection, "Client sent invalid SOAP request");
        return;
    }
    if (xmlXPathNodeSetIsEmpty(request_soap_body->nodesetval)) {
        xmlXPathFreeObject(request_soap_body);
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_400(connection,
                "SOAP request does not contain SOAP Body element");
        return;
    }
    if (request_soap_body->nodesetval->nodeNr > 1) {
        xmlXPathFreeObject(request_soap_body);
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_400(connection,
                "SOAP response has more than one Body element");
        return;
    }
    isds_request = request_soap_body->nodesetval->nodeTab[0]->children;
    if (isds_request->next != NULL) {
        xmlXPathFreeObject(request_soap_body);
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_400(connection, "SOAP body has more than one child");
        return;
    }
    if (isds_request->type != XML_ELEMENT_NODE || isds_request->ns == NULL ||
            NULL == isds_request->ns->href) {
        xmlXPathFreeObject(request_soap_body);
        xmlXPathFreeContext(xpath_ctx);
        xmlFreeDoc(request_doc);
        http_send_response_400(connection,
                "SOAP body does not contain a name-space-qualified element");
        return;
    }

    /* Build SOAP response envelope */
    response_doc = xmlNewDoc(BAD_CAST "1.0");
    if (!response_doc) {
        http_send_response_500(connection,
                "Could not build SOAP response document");
        goto leave;
    }
    response_soap_envelope = xmlNewNode(NULL, BAD_CAST "Envelope");
    if (!response_soap_envelope) {
        http_send_response_500(connection,
                "Could not build SOAP response envelope");
        goto leave;
    }
    xmlDocSetRootElement(response_doc, response_soap_envelope);
    /* Only this way we get namespace definition as @xmlns:soap,
     * otherwise we get namespace prefix without definition */
    soap_ns = xmlNewNs(response_soap_envelope, BAD_CAST SOAP_NS, NULL);
    if(NULL == soap_ns) {
        http_send_response_500(connection, "Could not create SOAP name space");
        goto leave;
    }
    xmlSetNs(response_soap_envelope, soap_ns);
    response_soap_body = xmlNewChild(response_soap_envelope, NULL,
            BAD_CAST "Body", NULL);
    if (!response_soap_body) {
        http_send_response_500(connection,
                "Could not add Body to SOAP response envelope");
        goto leave;
    }
    /* Append ISDS response element */
    if (-1 == test_asprintf(&response_name, "%s%s", isds_request->name,
                "Response")) {
        http_send_response_500(connection,
                "Could not buld ISDS resposne element name");
        goto leave;
    }
    isds_response = xmlNewChild(response_soap_body, NULL,
            BAD_CAST response_name, NULL);
    free(response_name);
    if (NULL == isds_response) {
        http_send_response_500(connection,
                "Could not add ISDS response element to SOAP response body");
        goto leave;
    }
    isds_ns = xmlNewNs(isds_response, isds_request->ns->href, NULL);
    if(NULL == isds_ns) {
        http_send_response_500(connection,
                "Could not create a name space for the response body");
        goto leave;
    }
    xmlSetNs(isds_response, isds_ns);

    /* Dispatch request to service */
    for (int i = 0; i < sizeof(services)/sizeof(services[0]); i++) {
        if (!strcmp(services[i].end_point, end_point) &&
                !xmlStrcmp(services[i].name_space, isds_request->ns->href) &&
                !xmlStrcmp(services[i].name, isds_request->name)) {
            /* Check if the configuration is enabled and find configuration */
            for (const struct service_configuration *service = configuration;
                    service->name != SERVICE_END; service++) {
                if (service->name == services[i].id) {
                    service_handled = 1;
                    if (!xmlStrcmp(services[i].name_space, BAD_CAST OISDS_NS)) {
                        /* Alias "isds" XPath identifier to OISDS_NS */
                        if (register_namespaces(xpath_ctx, 1,
                                    MESSAGE_NS_UNSIGNED)) {
                            http_send_response_500(connection,
                                    "Could not register name spaces to the "
                                    "XPath context");
                            break;
                        }
                    }
                    xpath_ctx->node = isds_request;
                    if (HTTP_ERROR_SERVER != services[i].function(connection,
                                request_doc, xpath_ctx, isds_request,
                                response_doc, isds_response,
                                service->arguments)) {
                        service_passed = 1;
                    } else {
                        http_send_response_500(connection,
                                "Internal server error while processing "
                                "ISDS request");
                    }
                }
            }
            break;
        }
    }
    
    /* Send response */
    if (service_passed) {
        /* Serialize the SOAP response */
        http_response_body = xmlBufferCreate();
        if (NULL == http_response_body) {
            http_send_response_500(connection,
                    "Could not create xmlBuffer for response serialization");
            goto leave;
        }
        /* Last argument 1 means format the XML tree. This is pretty but it breaks
         * XML document transport as it adds text nodes (indentiation) between
         * elements. */
        save_ctx = xmlSaveToBuffer(http_response_body, "UTF-8", 0);
        if (NULL == save_ctx) {
            http_send_response_500(connection, "Could not create XML serializer");
            goto leave;
        }
        /* XXX: According LibXML documentation, this function does not return
         * meaningful value yet */
        xmlSaveDoc(save_ctx, response_doc);
        if (-1 == xmlSaveFlush(save_ctx)) {
            http_send_response_500(connection,
                    "Could not serialize SOAP response");
            goto leave;
        }

        http_send_response_200(connection, http_response_body->content,
                http_response_body->use, soap_mime_type);
    }

leave:
    xmlSaveClose(save_ctx);
    xmlBufferFree(http_response_body);

    xmlFreeDoc(response_doc);

    xmlXPathFreeObject(request_soap_body);
    xmlXPathFreeContext(xpath_ctx);
    xmlFreeDoc(request_doc);

    if (!service_handled) {
        http_send_response_500(connection,
                "Requested ISDS service not implemented");
    }

}


