/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  toolbar.cc - GtkToolbar C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "toolbar.h"
#include "private/toolbar_p.h"
#include "label.h"
#include "radiobutton.h"

using namespace Inti;

/*  Gtk::ToolbarChild
 */
 
Gtk::ToolbarChildType
Gtk::ToolbarChild::type() const
{
	return (ToolbarChildType)child_.type;
}

Gtk::Widget*
Gtk::ToolbarChild::widget() const
{
	return G::Object::wrap<Widget>(child_.widget);
}

Gtk::Widget*
Gtk::ToolbarChild::icon() const
{
	return gtk_toolbar_child()->icon ? G::Object::wrap<Widget>(child_.icon) : 0;
}

Gtk::Label*
Gtk::ToolbarChild::label() const
{
	return gtk_toolbar_child()->label ? G::Object::wrap<Label>((GtkLabel*)child_.label) : 0;
}

/*  Gtk::Toolbar
 */

Gtk::Toolbar::Toolbar(GtkToolbar *toolbar, bool reference)
: Container((GtkContainer*)toolbar, reference)
{
}

Gtk::Toolbar::Toolbar(Orientation orientation, ToolbarStyle style)
: Container((GtkContainer*)ToolbarClass::create())
{
	set_orientation(orientation);
	set_style(style);
}

Gtk::Toolbar::~Toolbar() 
{
}
	
GtkToolbarClass*
Gtk::Toolbar::gtk_toolbar_class() const 
{ 
	return get_class<GtkToolbarClass>(); 
}
	
Gtk::Toolbar::operator GtkToolbar* () const 
{ 
	return this ? gtk_toolbar() : 0; 
}

bool
Gtk::Toolbar::children(std::vector<ToolbarChild*>& child_list) const
{
	g_return_val_if_fail(child_list.empty(), false);
	GList *first = gtk_toolbar()->children;
	GList *next = first;
	while (next != 0)
	{
		child_list.push_back(reinterpret_cast<ToolbarChild*>(next->data));
		next = g_list_next(next);
	}
	return !child_list.empty();
}

Gtk::Orientation
Gtk::Toolbar::get_orientation() const
{
	return (Orientation)gtk_toolbar_get_orientation(gtk_toolbar());
}

Gtk::ToolbarStyle
Gtk::Toolbar::get_style() const
{
	return (ToolbarStyle)gtk_toolbar_get_style(gtk_toolbar());
}

Gtk::IconSize
Gtk::Toolbar::get_icon_size() const
{
	return (IconSize)gtk_toolbar_get_icon_size(gtk_toolbar());
}

bool 
Gtk::Toolbar::get_tooltips() const
{
	return gtk_toolbar_get_tooltips(gtk_toolbar());
}

Gtk::Button*
Gtk::Toolbar::append_button(const String& text, Widget *icon, const Slot0<void> *callback,
                            const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_button(text, icon, callback, gtk_toolbar()->num_children, tooltip_text, tooltip_private_text);
}

Gtk::Button*
Gtk::Toolbar::prepend_button(const String& text, Widget *icon, const Slot0<void> *callback,
                             const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_button(text, icon, callback, 0, tooltip_text, tooltip_private_text);
}

Gtk::Button*
Gtk::Toolbar::insert_button(const String& text, Widget *icon, const Slot0<void> *callback, int position,
                            const String& tooltip_text, const String& tooltip_private_text)
{
	GtkWidget *widget = gtk_toolbar_insert_element(gtk_toolbar(), GTK_TOOLBAR_CHILD_BUTTON, 0, text.c_str(), tooltip_text.c_str(), 
	                                               tooltip_private_text.c_str(), *icon, 0, 0, position);
	if (!widget) return 0;

	Button *button = G::Object::wrap_new<Button>((GtkButton*)widget);
	if (callback)
		button->sig_clicked().connect(callback);
	return button;	
}

Gtk::ToggleButton*
Gtk::Toolbar::append_toggle_button(const String& text, Widget *icon, const Slot0<void> *callback,
                                   const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_toggle_button(text, icon, callback, gtk_toolbar()->num_children, tooltip_text, tooltip_private_text);
}

Gtk::ToggleButton*
Gtk::Toolbar::prepend_toggle_button(const String& text, Widget *icon, const Slot0<void> *callback,
                                    const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_toggle_button(text, icon, callback, 0, tooltip_text, tooltip_private_text);
}

Gtk::ToggleButton*
Gtk::Toolbar::insert_toggle_button(const String& text, Widget *icon, const Slot0<void> *callback, int position,
                                   const String& tooltip_text, const String& tooltip_private_text)
{
	GtkWidget *widget = gtk_toolbar_insert_element(gtk_toolbar(), GTK_TOOLBAR_CHILD_TOGGLEBUTTON, 0, text.c_str(),
	                                               tooltip_text.c_str(), tooltip_private_text.c_str(), *icon, 0, 0, position);
	if (!widget)
		return 0;
	ToggleButton *button = G::Object::wrap_new<ToggleButton>((GtkToggleButton*)widget);
	if (callback)
		button->sig_toggled().connect(callback);
	return button;	
}

Gtk::RadioButton*
Gtk::Toolbar::append_radio_button(const RadioButton *group, const String& text, Widget *icon, const Slot0<void> *callback,
                                  const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_radio_button(group, text, icon, callback, gtk_toolbar()->num_children, tooltip_text, tooltip_private_text);
}

Gtk::RadioButton*
Gtk::Toolbar::prepend_radio_button(const RadioButton *group, const String& text, Widget *icon, const Slot0<void> *callback,
                                   const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_radio_button(group, text, icon, callback, 0, tooltip_text, tooltip_private_text);
}

Gtk::RadioButton*
Gtk::Toolbar::insert_radio_button(const RadioButton *group, const String& text, Widget *icon, const Slot0<void> *callback,
                                  int position, const String& tooltip_text, const String& tooltip_private_text)
{
	GtkWidget *widget = gtk_toolbar_insert_element(gtk_toolbar(), GTK_TOOLBAR_CHILD_RADIOBUTTON, *group, text.c_str(),
	                                               tooltip_text.c_str(), tooltip_private_text.c_str(), *icon, 0, 0, position);
	if (!widget)
		return 0;
	RadioButton *button = G::Object::wrap_new<RadioButton>((GtkRadioButton*)widget);
	if (callback)
		button->sig_toggled().connect(callback);
	return button;
}

Gtk::Button*
Gtk::Toolbar::append_stock(const char *stock_id, const Slot0<void> *callback,
                           const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_stock(stock_id, callback, gtk_toolbar()->num_children, tooltip_text.c_str(), tooltip_private_text.c_str());
}

Gtk::Button*
Gtk::Toolbar::prepend_stock(const char *stock_id, const Slot0<void> *callback,
                            const String& tooltip_text, const String& tooltip_private_text)
{
	return insert_stock(stock_id, callback, 0, tooltip_text.c_str(), tooltip_private_text.c_str());
}

Gtk::Button*
Gtk::Toolbar::insert_stock(const char *stock_id, const Slot0<void> *callback, int position,
                           const String& tooltip_text, const String& tooltip_private_text)
{
	GtkWidget *widget = gtk_toolbar_insert_stock(gtk_toolbar(), stock_id, tooltip_text.c_str(),
	                                             tooltip_private_text.c_str(), 0, 0, position);
	if (!widget)
		return 0;
	Button *button = G::Object::wrap_new<Button>((GtkButton*)widget);
	if (callback)
		button->sig_clicked().connect(callback);
	return button;	
}

void 
Gtk::Toolbar::append_space()
{
	gtk_toolbar_append_space(gtk_toolbar());
}

void 
Gtk::Toolbar::prepend_space()
{
	gtk_toolbar_prepend_space(gtk_toolbar());
}

void 
Gtk::Toolbar::insert_space(int position)
{
	gtk_toolbar_insert_space(gtk_toolbar(), position);
}

void 
Gtk::Toolbar::remove_space(int position)
{
	gtk_toolbar_remove_space(gtk_toolbar(), position);
}

void 
Gtk::Toolbar::append_widget(Widget& widget, const String& tooltip_text, const String& tooltip_private_text)
{
	gtk_toolbar_append_widget(gtk_toolbar(), widget.gtk_widget(), tooltip_text.c_str(), tooltip_private_text.c_str());
}

void 
Gtk::Toolbar::prepend_widget(Widget& widget, const String& tooltip_text, const String& tooltip_private_text)
{
	gtk_toolbar_prepend_widget(gtk_toolbar(), widget.gtk_widget(), tooltip_text.c_str(), tooltip_private_text.c_str());
}

void 
Gtk::Toolbar::insert_widget(Widget& widget, int position, const String& tooltip_text, const String& tooltip_private_text)
{
	gtk_toolbar_insert_widget(gtk_toolbar(), widget.gtk_widget(), tooltip_text.c_str(), tooltip_private_text.c_str(), position);
}

void 
Gtk::Toolbar::set_orientation(Orientation orientation)
{
	gtk_toolbar_set_orientation(gtk_toolbar(), (GtkOrientation)orientation);
}

void 
Gtk::Toolbar::set_style(ToolbarStyle style)
{
	gtk_toolbar_set_style(gtk_toolbar(), (GtkToolbarStyle)style);
}

void 
Gtk::Toolbar::set_icon_size(IconSize icon_size)
{
	gtk_toolbar_set_icon_size(gtk_toolbar(), (GtkIconSize)icon_size);
}

void 
Gtk::Toolbar::set_tooltips(bool enable)
{
	gtk_toolbar_set_tooltips(gtk_toolbar(), enable);
}

void 
Gtk::Toolbar::unset_style()
{
	gtk_toolbar_unset_style(gtk_toolbar());
}

void 
Gtk::Toolbar::unset_icon_size()
{
	gtk_toolbar_unset_icon_size(gtk_toolbar());
}

/*  Gtk::ToolbarClass
 */

void
Gtk::ToolbarClass::init(GtkToolbarClass *g_class)
{
	ContainerClass::init((GtkContainerClass*)g_class);
	g_class->orientation_changed = &orientation_changed_proxy;
	g_class->style_changed = &style_changed_proxy;
}

GType
Gtk::ToolbarClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TOOLBAR, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ToolbarClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::ToolbarClass::orientation_changed_proxy(GtkToolbar *toolbar, GtkOrientation orientation)
{
	Toolbar *tmp_toolbar = G::Object::pointer<Toolbar>(toolbar);
	if (tmp_toolbar)
		tmp_toolbar->on_orientation_changed((Orientation)orientation);
	else
	{
		GtkToolbarClass *g_class = G::TypeInstance::class_peek_parent<GtkToolbarClass>(GTK_TOOLBAR_GET_CLASS(toolbar));
		if (g_class->orientation_changed)
			g_class->orientation_changed(toolbar, orientation);
	}
}

void
Gtk::ToolbarClass::style_changed_proxy(GtkToolbar *toolbar, GtkToolbarStyle style)
{
	Toolbar *tmp_toolbar = G::Object::pointer<Toolbar>(toolbar);
	if (tmp_toolbar)
		tmp_toolbar->on_style_changed((ToolbarStyle)style);
	else
	{
		GtkToolbarClass *g_class = G::TypeInstance::class_peek_parent<GtkToolbarClass>(GTK_TOOLBAR_GET_CLASS(toolbar));
		if (g_class->style_changed)
			g_class->style_changed(toolbar, style);
	}
}

/*  Signal handlers
 */

void
Gtk::Toolbar::on_orientation_changed(Orientation orientation)
{
	GtkToolbarClass *g_class = class_peek_parent<GtkToolbarClass>(gtk_toolbar_class());
	if (g_class->orientation_changed)
		g_class->orientation_changed(gtk_toolbar(), (GtkOrientation)orientation);
}

void
Gtk::Toolbar::on_style_changed(ToolbarStyle style)
{
	GtkToolbarClass *g_class = class_peek_parent<GtkToolbarClass>(gtk_toolbar_class());
	if (g_class->style_changed)
		g_class->style_changed(gtk_toolbar(), (GtkToolbarStyle)style);
}

/*  Properties
 */

const Gtk::Toolbar::OrientationPropertyType Gtk::Toolbar::orientation_property("orientation");

const Gtk::Toolbar::ToolbarStylePropertyType Gtk::Toolbar::toolbar_style_property("toolbar_style");

/*  Signals
 */

const Gtk::Toolbar::OrientationChangedSignalType Gtk::Toolbar::orientation_changed_signal("orientation_changed");

const Gtk::Toolbar::StyleChangedSignalType Gtk::Toolbar::style_changed_signal("style_changed");

