/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gconf/client.h
//! @brief GConfClient C++ wrapper interface.
//!
//! Provides GConf::Client, a G::Object subclass that should be used instead of GConfEngine
//! in most applications. GConf::Client tries to minimize the need to make remote calls to
//! the GConf daemon by keeping more information on the client side. It also has a default
//! error handler that comes into play if you pass null for the G::Error* argument to any
//! GConf::Client methods.

#ifndef INTI_GCONF_CLIENT_H
#define INTI_GCONF_CLIENT_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef GCONF_GCONF_CLIENT_H
#include <gconf/gconf-client.h>
#endif

#ifndef CPP_VECTOR_H
#include <vector>
#endif

namespace Inti {

namespace G {
class Error;
}

namespace GConf {

class ChangeSet;
class ClientClass;
class Entry;
class Schema;
class Value;

//! @enum Client::PreloadType
//! PreloadType is used to tell GConf::Client how to preload one of its directories.
//! As a rule of thumb, if you plan to get the value of almost all the keys in a directory,
//! preloading that directory will probably enhance performance. If you plan to use only
//! half the keys, preloading is likely a bad idea.

//! @enum Client::ErrorHandlingMode
//! ErrorHandlingMode is used to control GConf::Client's default error handling.
//! GConf::Client can pop up a dialog in the default signal handler for "error" or "unreturned_error". 
//! You can specify that no errors are handled, only unreturned errors are handled, or all errors are
//! handled with this enumeration. You can prevent specific errors from being handled automatically
//! by stopping the signal emission before the default signal handler is called (see the GTK+ 
//! documentation, gtk_signal_emit_stop_by_name() for example).

//! @class Client client.h inti/gconf/client.h
//! @brief A GConfClient C++ wrapper class.
//!
//! GConf::Client adds to plain GConf a client-side cache for a specified list of directories
//! you're interested in. You can <EM>preload</EM> entire directories into the cache, speeding
//! things up even more. There is automatic error handling, if you request it, and signals when
//! a value changes or an error occurs.

class Client : public G::Object
{
	friend class G::Object;
	friend class ClientClass;

	Client(const Client&);
	Client& operator=(const Client&);
	
protected:
	explicit Client(GConfClient *client, bool reference = true);
	//!< Construct a new Client from an existing GConfClient.
	//!< @param client A pointer to a GConfClient.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>client</EM> can be a newly created GConfClient or an existing
	//!< GConfClient. (see G::Object::Object).

//! @name Signal Handlers
//! @{

	virtual void on_value_changed(const String& key, const Value *value);
	//!< Emitted when a key below one of the directories added with add_dir()
	//!< receives a new value or is unset.
	//!< @param key The GConf key whose value has changed.
	//!< @param value The new value if set, or null if unset.
	//!<
	//!< Called when a key below one of the directories added with add_dir()
	//!< receives a new value or is unset. It can be much more efficient to
	//!< use notify_add().

	virtual void on_unreturned_error(const G::Error& error);
	//!< Emitted when you pass null for the G::Error* argument to any Client method,
	//!< and an error occurs.
	//!< @param error The unhandled error.
	//!<
	//!< <BR>Depending on the ErrorHandlingMode, the default handler for this signal
	//!< may or may not display an error dialog.
	
	virtual void on_error(const G::Error& error);
	//!< Emitted whenever an error occurs inside a Client method.
	//!< @param error An Error object.
	//!<
	//!< <BR>Depending on the ErrorHandlingMode, the default handler for this signal
	//!< may or may not display an error dialog.

//! @}
//  Signals

	typedef G::Signal2<void, const char*, GConfValue*> ValueChangedSignalType;
	typedef G::SignalProxy<G::TypeInstance, ValueChangedSignalType> ValueChangedSignalProxy;
	static const ValueChangedSignalType value_changed_signal;

	typedef G::Signal1<void, GError*> UnreturnedErrorSignalType;
	typedef G::SignalProxy<G::TypeInstance, UnreturnedErrorSignalType> UnreturnedErrorSignalProxy;
	static const UnreturnedErrorSignalType unreturned_error_signal;

	typedef G::Signal1<void, GError*> ErrorSignalType;
	typedef G::SignalProxy<G::TypeInstance, ErrorSignalType> ErrorSignalProxy;
	static const ErrorSignalType error_signal;

public:
	typedef Slot2<void, unsigned int, const Entry&> NotifySlot;
	//!< Signature for the notify method invoked when the value of a key changes.
	//!<
	//!< <B>Example:</B> Method signature for NotifySlot.
	//!< @code
	//!< void method(unsigned int cnxn_id, const Entry& entry);
	//!<
	//!< // cnxn_id: The connection indentifier returned by notify_add().
	//!< // entry: The Entry holding the new value. Entry::is_set() returns false if the value was unset.
	//!< @endcode
	//!<
	//!< The value argument should not be modified, and should be copied if you want to keep
	//!< it around (the Client will destroy it sometime after your notify method is called).

	enum PreloadType
	{
		PRELOAD_NONE = GCONF_CLIENT_PRELOAD_NONE, 
		//!< No preload occurs.

		PRELOAD_ONELEVEL = GCONF_CLIENT_PRELOAD_ONELEVEL, 
		//!< Loads the immediate children of the directory.

		PRELOAD_RECURSIVE = GCONF_CLIENT_PRELOAD_RECURSIVE 
		//!< Loads all children of the directory and its subdirectories, recursively.
	};

	enum ErrorHandlingMode
	{
		HANDLE_NONE = GCONF_CLIENT_HANDLE_NONE, 
		//!< never run a default error handler.

		HANDLE_UNRETURNED = GCONF_CLIENT_HANDLE_UNRETURNED, 
		//!< run a default error handler for unreturned errors ("unreturned_error" signal).

		HANDLE_ALL = GCONF_CLIENT_HANDLE_ALL 
		//!< run a default error handler for all errors ("error" signal).
	};
	
//! @name Constructors
//! @{

	virtual ~Client();
	//!< Destructor.
	
//! @}

	static Pointer<Client> get_default();
	//!< Creates a new Client using the default GConfEngine; normally this is the engine you want.
	//!< @return A smart pointer to the new Client.
	//!<
	//!< <BR>If someone else is already using the default Client, this function returns the same one
	//!< they're using, but with the reference count incremented. So it has to be unreferenced
	//!< either way. To save you the trouble a smart pointer is returned. The smart pointer will
	//!< unreference the Client when it goes out of scope.

	static Pointer<Client> get_for_engine(GConfEngine *engine);
	//!< Creates a new GConfClient with a specific GConfEngine.
	//!< @param engine The GConfEngine to use.
	//!< @return A smart pointer to the new Client.
	//!<
	//!< <BR>Only specialized configuration-related programs should need to call this function.
	//!< Remember to avoid using the GConfEngine directly once you have a GConfClient wrapper.
	//!< The smart pointer will unreference the Client when it goes out of scope.

//! @name Accessors
//! @{

	GConfClient* gconf_client() const { return (GConfClient*)instance; }
	//!< Get a pointer to the GConfClient structure.

	GConfClientClass* gconf_client_class() const;
	//!< Get a pointer to the GConfClientClass structure.

	operator GConfClient* () const;
	//!< Conversion operator; safely converts a GConf::Client into a GConfClient pointer.

	bool is_gconf_client() const;
	//!< Returns true if the object instance is of the type GCONF_TYPE_CLIENT

	Value get(const String& key, G::Error *error = 0) const;
	//!< Get the current value stored at key, or null if no value is set.
	//!< @param key The key to get.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return a Value object.
	//!<
	//!< <BR>If you know the expected type of the value, you probably want to use the type-specific
	//!< convenience wrappers (get_int(), etc.) because they will do the type-checking for you and
	//!< return the appropriate type. Automatically returns the default value for a key, if the key
	//!< is unset and a default exists.

	Value get_without_default(const String& key, G::Error *error = 0) const;
	//!< Get the current value for the key.
	//!< @param key The key to get.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return a Value object.
	//!<
	//!< <BR>Identical to get(), except that it will return null in place of the default value
	//!< if the key is unset. Note that get() can also return null if no default exists or
	//!< an error occurs.

	Entry get_entry(const String& key, const String& locale, bool use_schema_default, G::Error *error = 0) const;
	//!< Obtain the full GConf::Entry for a value.
	//!< @param key The key to get.
	//!< @param locale Locales not implemented yet; pass null for this value (not "").
	//!< @param use_schema_default Set <EM>true</EM> to return default value for key, instead of VALUE_INVALID.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return an Entry object.
	//!<
	//!< <BR> An Entry object stores an entry from a GConf <EM>directory</EM>, including a key-value
	//!< pair, the name of the schema applicable to this entry, whether the value is a default value,
	//!< and whether GConf can write a new value at this key. <EM>key</EM> should be an absolute key,
	//!< not a relative key. To access the key and value, use Entry::get_key() and Entry::get_value().

	Value get_default_from_schema(const String& key, G::Error *error = 0) const;
	//!< Get the value for key in the default schema.
	//!< @param key The key to get.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return A Value object.
	//!<
	//!< <BR>Returns the default value stored in the key's schema, if the key has a schema associated and
	//!< the schema exists and the schema contains a default value. Note that get(), get_string(), and
	//!< so on already return the default value if no other value is found, so normally you do not need
	//!< this function. This function is just for convenience; you could also get the GConf::MetaInfo
	//!< for the key, read the schema name from there, then look up the schema by name and extract the
	//!< default value.

	bool dir_exists(const String& dir, G::Error *error = 0) const;
	//!< Queries whether the directory dir exists in the GConf database.
	//!< @param dir The directory to check for.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> if the directory exists.

	bool key_is_writable(const String& key, G::Error *error = 0) const;
	//!< Checks wether the <EM>key</EM> is writable.
	//!< @param key The key to check.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> if the key is writable.

	double get_float(const String& key, G::Error *error = 0) const;
	//!< Requests the floating point number (VALUE_FLOAT) stored at key.
	//!< @param key The key you want the value of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return The value of key, or 0.0 if no value is obtained.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-float is stored at key,
	//!< an error is returned. On error, or if key is unset, 0.0 is returned.

	int get_int(const String& key, G::Error *error = 0) const;
	//!< Requests the integer (VALUE_INT) stored at key.
	//!< @param key The key you want the value of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return The value of key, or 0 if no value is obtained.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-integer is stored at key,
	//!< an error is returned. On error, or if key is unset, 0 is returned.

	String get_string(const String& key, G::Error *error = 0) const;
	//!< Requests the string (VALUE_STRING) stored at key. 
	//!< @param key The key you want the value of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return The string (value of key), or a null string if no value is obtained.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-string is stored at key,
	//!< an error is returned. On error, or if key is unset, a null string is returned.

	bool get_bool(const String& key, G::Error *error = 0) const;
	//!< Requests the bool value (VALUE_BOOL) stored at key.
	//!< @param key The key you want the value of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return The value of key, or <EM>false</EM> if no value is obtained.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-bool is stored at key,
	//!< an error is returned. On error, or if key is unset, <EM>false</EM> is returned.

	Schema get_schema(const String& key, G::Error *error = 0) const;
	//!< Requests the schema (VALUE_SCHEMA) stored at key.
	//!< @param key The key you want the value of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return The value of key as a Schema object, or null if no value was obtained.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-schema is stored at key,
	//!< an error is returned. If no value is set or an error occurs, null is returned.
	//!< Note that this returns the schema stored at key, NOT the schema that key conforms to.

	bool get_list(const String& key, std::vector<int>& list, G::Error *error = 0) const;
	//!< Requests the list (VALUE_INT) stored at key.
	//!< @param key The key you want the value of.
	//!< @param list A reference to a vector of int, to hold the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-list is stored at key, or the
	//!< list does not contain elements of type VALUE_INT, an error is returned.
	//!< If no value is set or an error occurs, <EM>false</EM> is returned.

	bool get_list(const String& key, std::vector<bool>& list, G::Error *error = 0) const;
	//!< Requests the list (VALUE_BOOL) stored at key.
	//!< @param key The key you want the value of.
	//!< @param list A reference to a vector of bool, to hold the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-list is stored at key, or the
	//!< list does not contain elements of type VALUE_BOOL, an error is returned.
	//!< If no value is set or an error occurs, <EM>false</EM> is returned.

	bool get_list(const String& key, std::vector<double>& list, G::Error *error = 0) const;
	//!< Requests the list (VALUE_FLOAT) stored at key.
	//!< @param key The key you want the value of.
	//!< @param list A reference to a vector of double, to hold the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-list is stored at key, or the
	//!< list does not contain elements of type VALUE_FLOAT, an error is returned.
	//!< If no value is set or an error occurs, <EM>false</EM> is returned.

	bool get_list(const String& key, std::vector<String>& list, G::Error *error = 0) const;
	//!< Requests the list (VALUE_STRING) stored at key.
	//!< @param key The key you want the value of.
	//!< @param list A reference to a vector of String, to hold the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-list is stored at key, or the
	//!< list does not contain elements of type VALUE_STRING, an error is returned.
	//!< If no value is set or an error occurs, <EM>false</EM> is returned.

	bool get_list(const String& key, std::vector<Schema>& list, G::Error *error = 0) const;
	//!< Requests the list (VALUE_SCHEMA) stored at key.
	//!< @param key The key you want the value of.
	//!< @param list A reference to a vector of Schema, to hold the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-list is stored at key, or the
	//!< list does not contain elements of type VALUE_SCHEMA, an error is returned.
	//!< If no value is set or an error occurs, <EM>false</EM> is returned.

	bool get_pair(const String& key, int& car_data, int& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_INT/VALUE_INT) stored at key.
	//!< @param key The key you want the value of.
	//!< @param car_data Reference to the return location for the car.
	//!< @param cdr_data Reference to the return location for the cdr.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Automatically performs type-checking, so if a non-pair is stored at key,
	//!< an error is returned. Remember that the car of a pair is its first value, and
	//!< the cdr is its second value, in the Lisp tradition. get_pair() stores the two
	//!< fields of the pair in the locations pointed to by car_data and cdr_data. If
	//!< there's an error or the value is unset, car_data and cdr_data are left unchanged.
	//!< 
	//!< <B>Example:</B> An example of get_pair() in action.
	//!< @code
	//!< #include <iostream>
	//!< int car = 10;
	//!< String cdr;
	//!< G::Error error;
	//!<
	//!< Pointer<GConf::Client> client = GConf::Client::get_default();
	//!< if (!client->get_pair("/foo", car, cdr, &error))
	//!< {
	//!<     // Note: car/cdr should be untouched, because an error occurred.
	//!<     if (error.get())
	//!<         std::cout << error.mesage() << std::endl;
	//!< }
	//!< else
	//!< {
	//!<     // Note: car/cdr may be untouched even though there was no error, if no value was set for "/foo".
	//!<     std::cout << "Found pair (" << car << "," << cdr << ")" << std::endl;
	//!< }
	//!< @endcode

	bool get_pair(const String& key, int& car_data, bool& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_INT/VALUE_BOOL) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, int& car_data, double& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_INT/VALUE_FLOAT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, int& car_data, String& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_INT/VALUE_STRING) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, int& car_data, Schema &cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_INT/VALUE_SCHEMA) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, bool& car_data, int& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_BOOL/VALUE_INT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, bool& car_data, bool& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_BOOL/VALUE_BOOL) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, bool& car_data, double& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_BOOL/VALUE_FLOAT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, bool& car_data, String& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_BOOL/VALUE_STRING) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, bool& car_data, Schema &cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_BOOL/VALUE_SCHEMA) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, double& car_data, int& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_FLOAT/VALUE_INT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, double& car_data, bool& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_FLOAT/VALUE_BOOL) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, double& car_data, double& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_FLOAT/VALUE_FLOAT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, double& car_data, String& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_FLOAT/VALUE_STRING) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, double& car_data, Schema &cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_FLOAT/VALUE_SCHEMA) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, String& car_data, int& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_STRING/VALUE_INT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, String& car_data, bool& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_STRING/VALUE_BOOL) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, String& car_data, double& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_STRING/VALUE_FLOAT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, String& car_data, String& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_STRING/VALUE_STRING) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, String& car_data, Schema &cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_STRING/VALUE_SCHEMA) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, Schema &car_data, int& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_SCHEMA/VALUE_INT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, Schema &car_data, bool& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_SCHEMA/VALUE_BOOL) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, Schema &car_data, double& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_SCHEMA/VALUE_FLOAT) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, Schema &car_data, String& cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_SCHEMA/VALUE_STRING) stored at key (see get_pair(INT/INT)).

	bool get_pair(const String& key, Schema &car_data, Schema &cdr_data, G::Error *error = 0) const;
	//!< Requests the pair (VALUE_SCHEMA/VALUE_SCHEMA) stored at key (see get_pair(INT/INT)).

//! @}
//! @name Methods
//! @{

	void add_dir(const char* dir, PreloadType preload, G::Error *error = 0);
	//!< Add a directory to the list of directories the Client will watch.
	//!< @param dir Directory to add to the list.
	//!< @param preload Degree of preload.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>Any changes to keys below this directory will cause the <EM>value_changed</EM> signal
	//!< to be emitted. When you add the directory, you can request that the Client to preload its
	//!< contents; see ClientPreloadType for details.
	//!<
	//!< Added directories may not overlap. That is, if you add "/foo", you may not add "/foo/bar".
	//!< However you can add "/foo" and "/bar". You can also add "/foo" multiple times; if you add a
	//!< directory multiple times, it will not be removed until you call gconf_client_remove_dir() an
	//!< equal number of times.

	void add_dir(const String& dir, PreloadType preload, G::Error *error = 0);
	//!< Add a directory to the list of directories the Client will watch.
	//!< @param dir Directory to add to the list.
	//!< @param preload Degree of preload.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>See add_dir(const char*, PreloadType, G::Error*) for details.

	void remove_dir(const char* dir, G::Error *error = 0);
	//!< Remove a directory from the list created with add_dir().
	//!< @param dir Directory to remove from the list.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>If any notifications have been added below this directory with notify_add(), those
	//!< notifications will be disabled until you re-add the removed directory. Note that if a
	//!< directory has been added multiple times, you must remove it the same number of times 
	//!< before the remove takes effect.

	void remove_dir(const String& dir, G::Error *error = 0);
	//!< Remove a directory from the list created with add_dir().
	//!< @param dir Directory to remove from the list.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>See remove_dir(const char*, PreloadType, G::Error*) for details.

	unsigned int notify_add(const String& namespace_section, NotifySlot *slot, G::Error *error = 0);
	//!< Request notification of changes to namespace_section.
	//!< @param namespace_section Where to listen for changes.
	//!< @param slot A NotifySlot to call when changes occur.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return A connection ID for removing the notification.
	//!<
	//!< <BR>This includes the key namespace_section itself, and any keys below it (the behavior is
	//!< identical to gconf_engine_notify_add(), but while gconf_engine_notify_add() places a 
	//!< notification request on the server for every notify function, GConf::Client requests
	//!< server notification for directories added with add_dir() and keeps the list of 
	//!< NotifySlots on the client side).
	//!<
	//!< For the notification to happen, namespace_section must be equal to or below one of
	//!< the directories added with add_dir(). You can still call notify_add() for other 
	//!< directories, but no notification will be received until you add a directory above or
	//!< equal to namespace_section. One implication of this is that remove_dir() temporarily
	//!< disables notifications that were below the removed directory.
	//!<
	//!< The method returns a connection ID you can use to call notify_remove().See the
	//!< description of NotifySlot for details on how the notification method is called.

	void notify_remove(unsigned int connect_id);
	//!< Remove a notification using the ID returned from notify_add().
	//!< @param connect_id The connection indentifier.

	void set_error_handling(ErrorHandlingMode mode);
	//!< Controls the default error handling for the Client. 
	//!< @param mode The error handling mode.
	//!<
	//!< <BR>If you don't want the default handler, set the error handling to CLIENT_HANDLE_NONE.
	//!< Error handling happens in the default signal handler, so you can selectively override the
	//!< default handling by connecting to the error signal and calling g_signal_emission_stop().

	void clear_cache();
	//!< Dumps everything out of the Client client-side cache. 
	//!< If you know you're done using the Client for a while, you can call this method to save
	//!< some memory. Note that this nullifies the effect of any preloading you may have done.

	void preload(const String& dirname, PreloadType type, G::Error *error = 0);
	//!< Preloads a directory. 
	//!< @param dirname The name of the directory to preload.
	//!< @param type The degree of preload.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>Normally you do this when you call add_dir(), but if you've called clear_cache() there
	//!< may be a reason to do it again. Not useful otherwise.

	void set(const String& key, const Value& value, G::Error *error = 0);
	//!< Sets the value for a configuration key.
	//!< @param key A key to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>Just like gconf_engine_set(), but uses the Client caching and error-handling features.
	//!< The value argument will not be modified. These keys do <EM>not</EM> have to be in the
	//!< client's directory list, but they won't be cached unless they are.

	bool unset(const String& key, G::Error *error = 0);
	//!< Unsets the value of key; if key is already unset, it has no effect.
	//!< @param key A key to unset.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> on error.
	//!<
	//!< <BR>An error of note is GCONF_OVERRIDDEN, indicating that the system administrator has
	//!< <EM>forced</EM> a value for this key.

	std::vector<Entry> all_entries(const String& dir, G::Error *error = 0);
	//!< Lists the key-value pairs in dir. Does not list subdirectories; for that use all_dirs(). 
	//!< @param dir The directory to list.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return A standard vector of Entry objects.
	//!<
	//!< <BR>The returned vector contains GConf::Entry objects. A GConf::Entry contains a relative
	//!< key and a value. The list is not recursive, it contains only the immediate children of dir.

	std::vector<String> all_dirs(const String& dir, G::Error *error = 0);
	//!< Lists the subdirectories in dir.
	//!< @param dir The directory to get subdirectories from.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return a standard vector of Strings that are subdirectory names.

	void suggest_sync(G::Error *error = 0);
	//!< Suggests to gconfd that you've just finished a block of changes, and it would be an
	//!< optimal time to sync to permanent storage. 
	//!< @param error A G::Error object, or null to ignore errors.
	//!<
	//!< <BR>This is only a suggestion; and gconfd will eventually sync even if you don't call
	//!< suggest_sync(). This method is just a "hint" provided to gconfd to maximize efficiency
	//!< and minimize data loss.

	bool set_float(const String& key, double value, G::Error *error = 0);
	//!< Sets the floating point value (VALUE_FLOAT) for a key.
	//!< @param key The key you want to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, or <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Note that you still should call suggest_sync().

	bool set_int(const String& key, int value, G::Error *error = 0);
	//!< Sets the integer value (VALUE_INT) for key.
	//!< @param key The key you want to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, or <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Note that you still should call suggest_sync().

	bool set_string(const String& key, const String& value, G::Error *error = 0);
	//!< Sets the String value (VALUE_STRING) for a key.
	//!< @param key The key you want to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, or <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Note that you still should call suggest_sync().

	bool set_bool(const String& key, bool value, G::Error *error = 0);
	//!< Sets the bool value (VALUE_BOOL) for a key.
	//!< @param key The key you want to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, or <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Note that you still should call suggest_sync().

	bool set_schema(const String& key, const Schema& value, G::Error *error = 0);
	//!< Sets the Schema value (VALUE_SCHEMA) for a key.
	//!< @param key The key you want to set.
	//!< @param value The new value.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, or <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>Note that you still should call suggest_sync().

	bool set_list(const String& key, const std::vector<int>& list, G::Error *error = 0);
 	//!< Sets the list (VALUE_INT) for a key.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of int that contains the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_list(const String& key, const std::vector<bool>& list, G::Error *error = 0);
 	//!< Sets the list (VALUE_BOOL) for a key.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of bool that contains the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_list(const String& key, const std::vector<double>& list, G::Error *error = 0);
 	//!< Sets the list (VALUE_FLOAT) for a key.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of double that contains the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_list(const String& key, const std::vector<String>& list, G::Error *error = 0);
 	//!< Sets the list (VALUE_STRING) for a key.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of String that contains the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_list(const String& key, const std::vector<Schema>& list, G::Error *error = 0);
 	//!< Sets the list (VALUE_SCHEMA) for a key.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of String that contains the list elements.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_pair(const String& key, int car_data, int cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_INT/VALUE_INT) for a key.
	//!< @param key The key you want to set.
	//!< @param car_data The new value for the first field (car).
	//!< @param cdr_data The new value for the second field (cdr).
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.

	bool set_pair(const String& key, int car_data, bool cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_INT/VALUE_BOOL) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, int car_data, double cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_INT/VALUE_FLOAT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, int car_data, const String& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_INT/VALUE_STRING) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, int car_data, const Schema& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_INT/VALUE_SCHEMA) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, bool car_data, int cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_BOOL/VALUE_INT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, bool car_data, bool cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_BOOL/VALUE_BOOL) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, bool car_data, double cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_BOOL/VALUE_FLOAT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, bool car_data, const String& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_BOOL/VALUE_STRING) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, bool car_data, const Schema& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_BOOL/VALUE_SCHEMA) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, double car_data, int cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_FLOAT/VALUE_INT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, double car_data, bool cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_FLOAT/VALUE_BOOL) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, double car_data, double cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_FLOAT/VALUE_FLOAT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, double car_data, const String& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_FLOAT/VALUE_STRING) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, double car_data, const Schema& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_FLOAT/VALUE_SCHEMA) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const String& car_data, int cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_STRING/VALUE_INT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const String& car_data, bool cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_STRING/VALUE_BOOL) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const String& car_data, double cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_STRING/VALUE_FLOAT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const String& car_data, const String& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_STRING/VALUE_STRING) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const String& car_data, const Schema& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_STRING/VALUE_SCHEMA) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const Schema& car_data, int cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_SCEHEMA/VALUE_INT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const Schema& car_data, bool cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_SCEHEMA/VALUE_BOOL) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const Schema& car_data, double cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_SCEHEMA/VALUE_FLOAT) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const Schema& car_data, const String& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_SCEHEMA/VALUE_STRING) for a key (see set_pair(INT/INT).

	bool set_pair(const String& key, const Schema& car_data, const Schema& cdr_data, G::Error *error = 0);
	//!< Sets the pair (VALUE_SCEHEMA/VALUE_SCHEMA) for a key (see set_pair(INT/INT).

	void error(G::Error& error);
	//!< Emits the "error" signal. Rarely useful.
	//!< @param error A reference to the error to pass to signal handlers.

	void unreturned_error(G::Error& error);
	//!< Emits the "unreturned_error" signal. Rarely useful.

	void value_changed(const String& key, Value& value);
	//!< Emits the "value_changed" signal. Rarely useful.
	//!< @param key The key to pass to signal handlers.
	//!< @param value The value of key to pass to signal handlers.

	bool commit_change_set(ChangeSet& cs, bool remove_committed, G::Error *error = 0);
	//!< Applies the changes in the change set to the Client.
	//!< @param cs A ChangeSet.
	//!< @param remove_committed Whether to remove successfully-committed changes from the set.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return <EM>true</EM> on success, <EM>false</EM> if an error occurs.
	//!<
	//!< <BR>If remove_committed is <EM>true</EM>, then any successfully-committed changes are
	//!< removed from the change set. If remove_committed is <EM>false</EM>, the ChangeSet is
	//!< left unmodified. If any set or unset operation results in an error, then processing
	//!< terminates and the error is returned in <EM>error</EM> (unless error was was). If
	//!< remove_committed was <EM>true</EM>, then all the changes committed before the error
	//!< occurred will have been removed from the set. If any error occurs, <EM>false</EM> 
	//!< is returned.

	Pointer<ChangeSet> reverse_change_set(ChangeSet& cs, G::Error *error = 0);
	//!< Create a change set that would revert the given change set for the Client.
	//!< @param cs The change set to create the reverse of.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return A new change set that would restore the current state of each key in cs.
	//!<
	//!< <BR>Creates a change set that would reverse cs. That is, for each change in cs,
	//!< save the current state of that key in the returned change set.

	Pointer<ChangeSet> change_set_from_current(const std::vector<String> keys, G::Error *error = 0);
	//!< Creates a change set that will change the keys in the vector of keys to their current state.
	//!< @param keys A vector of key names.
	//!< @param error A G::Error object, or null to ignore errors.
	//!< @return A new change set that changes all the keys to their current state.
	//!<
	//!< <BR>Use this to save the current state of a collection of keys; then you can later revert to the
	//!< current state by committing the returned change set.

//! @}
//! @name Signal Proxies
//! @{

	const ValueChangedSignalProxy sig_value_changed()
	{
		return ValueChangedSignalProxy(this, &value_changed_signal);
	}
	//!< Connect to the value_changed_signal; emitted when a key below one of the directories
	//!< added with add_dir() receives a new value or is unset.

	const UnreturnedErrorSignalProxy sig_unreturned_error()
	{
		return UnreturnedErrorSignalProxy(this, &unreturned_error_signal);
	}
	//!< Connect to the Error_signal; emitted when you pass null for the G::Error* argument to
	//!< any Client method, and an error occurs.

	const ErrorSignalProxy sig_error()
	{
		return ErrorSignalProxy(this, &error_signal);
	}
	//!< Connect to the Error_signal; emitted whenever an error occurs inside a Client method.

//! @}
};

} // namespace GConf

} // namespace Inti

#endif // INTI_GCONF_CLIENT_H

