/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  client.cc - GConfClient C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "client.h"
#include "private/client_p.h"
#include "changeset.h"
#include "schema.h"
#include "value.h"
#include "internals.h"
#include <inti/glib/error.h>

using namespace Inti;

/*  GConf::Client
 */

GConf::Client::Client(GConfClient *client, bool reference)
: G::Object((GObject*)client, reference)
{
}
	
GConf::Client::~Client()
{
}

GConfClientClass* 
GConf::Client::gconf_client_class() const
{
	return get_class<GConfClientClass>();
}

GConf::Client::operator GConfClient* () const
{
	return this ? gconf_client() : 0;
}

bool 
GConf::Client::is_gconf_client() const
{
	return is_a(GCONF_TYPE_CLIENT);
}
	
GConf::Value 
GConf::Client::get(const String& key, G::Error *error) const
{
	GConfValue *tmp_value = gconf_client_get(gconf_client(), key.c_str(), *error);
	Value value(tmp_value);
	gconf_value_free(tmp_value);
	return value;
}

GConf::Value
GConf::Client::get_without_default(const String& key, G::Error *error) const
{
	GConfValue *tmp_value = gconf_client_get_without_default(gconf_client(), key.c_str(), *error);
	Value value(tmp_value);
	gconf_value_free(tmp_value);
	return value;
}

GConf::Entry 
GConf::Client::get_entry(const String& key, const String& locale, bool use_schema_default, G::Error *error) const
{
	GConfEntry *tmp_entry = gconf_client_get_entry(gconf_client(), key.c_str(), locale.c_str(), use_schema_default, *error);
	Entry entry(tmp_entry);
	gconf_entry_free(tmp_entry);
	return entry;
}

GConf::Value 
GConf::Client::get_default_from_schema(const String& key, G::Error *error) const
{
	GConfValue *tmp_value = gconf_client_get_default_from_schema(gconf_client(), key.c_str(), *error);
	Value value(tmp_value);
	gconf_value_free(tmp_value);
	return value;
}

bool 
GConf::Client::dir_exists(const String& dir, G::Error *error) const
{
	return gconf_client_dir_exists(gconf_client(), dir.c_str(), *error);
}

bool 
GConf::Client::key_is_writable(const String& key, G::Error *error) const
{
	return gconf_client_key_is_writable(gconf_client(), key.c_str(), *error);
}

double 
GConf::Client::get_float(const String& key, G::Error *error) const
{
	return gconf_client_get_float(gconf_client(), key.c_str(), *error);
}

int 
GConf::Client::get_int(const String& key, G::Error *error) const
{
	return gconf_client_get_int(gconf_client(), key.c_str(), *error);
}

String 
GConf::Client::get_string(const String& key, G::Error *error) const
{
	char *value = gconf_client_get_string(gconf_client(), key.c_str(), *error);
	String s(value);
	g_free(value);
	return s;
}

bool 
GConf::Client::get_bool(const String& key, G::Error *error) const
{
	return gconf_client_get_bool(gconf_client(), key.c_str(), *error);
}

GConf::Schema 
GConf::Client::get_schema(const String& key, G::Error *error) const
{
	return Schema(gconf_client_get_schema(gconf_client(), key.c_str(), *error));
}

bool 
GConf::Client::get_list(const String& key, std::vector<int>& list, G::Error *error) const
{
	g_return_val_if_fail(list.empty(), false);
	GSList *first = gconf_client_get_list(gconf_client(), key.c_str(), GCONF_VALUE_INT, *error);
	GSList *next = first;

	while (next != 0)
	{
		list.push_back((int)next->data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Client::get_list(const String& key, std::vector<bool>& list, G::Error *error) const
{
	g_return_val_if_fail(list.empty(), false);
	GSList *first = gconf_client_get_list(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, *error);
	GSList *next = first;

	while (next != 0)
	{
		list.push_back((bool)next->data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Client::get_list(const String& key, std::vector<double>& list, G::Error *error) const
{
	g_return_val_if_fail(list.empty(), false);
	GSList *first = gconf_client_get_list(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, *error);
	GSList *next = first;

	while (next != 0)
	{
		double *data = static_cast<double*>(next->data);
		list.push_back(*data);
		g_free(data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool
GConf::Client::get_list(const String& key, std::vector<String>& list, G::Error *error) const
{
	g_return_val_if_fail(list.empty(), false);
	GSList *first = gconf_client_get_list(gconf_client(), key.c_str(), GCONF_VALUE_STRING, *error);
	GSList *next = first;

	while (next != 0)
	{
		char *data = static_cast<char*>(next->data);
		list.push_back(data);
		g_free(data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Client::get_list(const String& key, std::vector<Schema>& list, G::Error *error) const
{
	g_return_val_if_fail(list.empty(), false);
	GSList *first = gconf_client_get_list(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, *error);
	GSList *next = first;

	while (next != 0)
	{
		list.push_back((GConfSchema*)next->data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Client::get_pair(const String& key, int& car_data, int& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, int& car_data, bool& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, int& car_data, double& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, int& car_data, String& cdr_data, G::Error *error) const
{
	char *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                                    &car_data, &cdr, *error);
	cdr_data = cdr;
	g_free(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, int& car_data, Schema &cdr_data, G::Error *error) const
{
	GConfSchema *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                                    &car_data, &cdr, *error);
	cdr_data = Schema(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, bool& car_data, int& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, bool& car_data, bool& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, bool& car_data, double& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, bool& car_data, String& cdr_data, G::Error *error) const
{
	char *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, GCONF_VALUE_STRING,
	                                    &car_data, &cdr, *error);
	cdr_data = cdr;
	g_free(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, bool& car_data, Schema &cdr_data, G::Error *error) const
{
	GConfSchema *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, GCONF_VALUE_SCHEMA,
	                                    &car_data, &cdr, *error);
	cdr_data = Schema(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, double& car_data, int& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, double& car_data, bool& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, double& car_data, double& cdr_data, G::Error *error) const
{
	return gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::get_pair(const String& key, double& car_data, String& cdr_data, G::Error *error) const
{
	char *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, GCONF_VALUE_STRING,
	                                    &car_data, &cdr, *error);
	cdr_data = cdr;
	g_free(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, double& car_data, Schema &cdr_data, G::Error *error) const
{
	GConfSchema *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, GCONF_VALUE_SCHEMA,
	                                    &car_data, &cdr, *error);
	cdr_data = Schema(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, String& car_data, int& cdr_data, G::Error *error) const
{
	char *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_STRING, GCONF_VALUE_INT,
	                                    &car, &cdr_data, *error);
	car_data = car;
	g_free(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, String& car_data, bool& cdr_data, G::Error *error) const
{
	char *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_STRING, GCONF_VALUE_BOOL,
	                                    &car, &cdr_data, *error);
	car_data = car;
	g_free(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, String& car_data, double& cdr_data, G::Error *error) const
{
	char *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_STRING, GCONF_VALUE_FLOAT,
	                                    &car, &cdr_data, *error);
	car_data = car;
	g_free(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, String& car_data, String& cdr_data, G::Error *error) const
{
	char *car = 0;
	char *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_STRING, GCONF_VALUE_STRING,
	                                    &car, &cdr, *error);
	car_data = car;
	cdr_data = cdr;
	g_free(car);
	g_free(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, String& car_data, Schema &cdr_data, G::Error *error) const
{
	char *car = 0;
	GConfSchema *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_STRING, GCONF_VALUE_SCHEMA,
	                                    &car, &cdr, *error);
	car_data = car;
	cdr_data = Schema(cdr);
	g_free(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, Schema &car_data, int& cdr_data, G::Error *error) const
{
	GConfSchema *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, GCONF_VALUE_INT,
	                                    &car, &cdr_data, *error);
	car_data = Schema(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, Schema &car_data, bool& cdr_data, G::Error *error) const
{
	GConfSchema *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, GCONF_VALUE_BOOL,
	                                    &car, &cdr_data, *error);
	car_data = Schema(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, Schema &car_data, double& cdr_data, G::Error *error) const
{ const
	GConfSchema *car = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, GCONF_VALUE_FLOAT,
	                                    &car, &cdr_data, *error);
	car_data = Schema(car);
	return result;
}

bool
GConf::Client::get_pair(const String& key, Schema &car_data, String& cdr_data, G::Error *error) const
{
	GConfSchema *car = 0;
	char *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, GCONF_VALUE_STRING,
	                                    &car, &cdr, *error);
	car_data = Schema(car);
	cdr_data = cdr;
	g_free(cdr);
	return result;
}

bool
GConf::Client::get_pair(const String& key, Schema &car_data, Schema &cdr_data, G::Error *error) const
{
	GConfSchema *car = 0;
	GConfSchema *cdr = 0;
	bool result = gconf_client_get_pair(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, GCONF_VALUE_SCHEMA,
	                                    &car, &cdr, *error);
	car_data = Schema(car);
	cdr_data = Schema(cdr);
	return result;
}

Pointer<GConf::Client>
GConf::Client::get_default()
{
	return G::Object::wrap<Client>(gconf_client_get_default());
}
	
Pointer<GConf::Client>
GConf::Client::get_for_engine(GConfEngine *engine)
{
	return G::Object::wrap<Client>(gconf_client_get_for_engine(engine));
}
	
void
GConf::Client::add_dir(const char* dir, PreloadType preload, G::Error *error)
{
	gconf_client_add_dir(gconf_client(), dir, (GConfClientPreloadType)preload, *error);
}

void
GConf::Client::add_dir(const String& dir, PreloadType preload, G::Error *error)
{
	gconf_client_add_dir(gconf_client(), dir.c_str(), (GConfClientPreloadType)preload, *error);
}

void
GConf::Client::remove_dir(const char* dir, G::Error *error)
{
	gconf_client_remove_dir(gconf_client(), dir, *error);
}

void
GConf::Client::remove_dir(const String& dir, G::Error *error)
{
	gconf_client_remove_dir(gconf_client(), dir.c_str(), *error);
}

namespace { // NotifySlot callback

void notify_slot_callback(GConfClient*, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	GConf::Client::NotifySlot *slot = static_cast<GConf::Client::NotifySlot*>(user_data);
	GConf::Entry tmp_entry(entry);
	slot->call(cnxn_id, tmp_entry);
}

} // NotifySlot callback

unsigned int
GConf::Client::notify_add(const String& namespace_section, NotifySlot *slot, G::Error *error)
{
	return gconf_client_notify_add(gconf_client(), namespace_section.c_str(), &notify_slot_callback, slot, 0, *error);
}

void 
GConf::Client::notify_remove(unsigned int connect_id)
{
	gconf_client_notify_remove(gconf_client(), connect_id);
}

void 
GConf::Client::set_error_handling(ErrorHandlingMode mode)
{
	gconf_client_set_error_handling(gconf_client(), (GConfClientErrorHandlingMode)mode);
}

void
GConf::Client::clear_cache()
{
	gconf_client_clear_cache(gconf_client());
}

void
GConf::Client::preload(const String& dirname, PreloadType type, G::Error *error)
{
	gconf_client_preload(gconf_client(), dirname.c_str(), (GConfClientPreloadType)type, *error);
}

void 
GConf::Client::set(const String& key, const Value& value, G::Error *error)
{
 	gconf_client_set(gconf_client(), key.c_str(), value.gconf_value(), *error);
}

bool 
GConf::Client::unset(const String& key, G::Error *error)
{
	return gconf_client_unset(gconf_client(), key.c_str(), *error);
}

std::vector<GConf::Entry>
GConf::Client::all_entries(const String& dir, G::Error *error)
{
	std::vector<Entry> entries;
	GSList *first = gconf_client_all_entries(gconf_client(), dir.c_str(), *error);
	GSList *next = first;

	while (next != 0)
	{
		entries.push_back((GConfEntry*)next->data);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return entries;
}

std::vector<String>
GConf::Client::all_dirs(const String& dir, G::Error *error)
{
	std::vector<String> dirs;
	GSList *first = gconf_client_all_dirs(gconf_client(), dir.c_str(), *error);
	GSList *next = first;

	while (next != 0)
	{
		char * dir_name = (char*)next->data;
		dirs.push_back(dir_name);
		g_free(dir_name);
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return dirs;
}

void 
GConf::Client::suggest_sync(G::Error *error)
{
	gconf_client_suggest_sync(gconf_client(), *error);
}

bool 
GConf::Client::set_float(const String& key, double value, G::Error *error)
{
	return gconf_client_set_float(gconf_client(), key.c_str(), value, *error);
}

bool
GConf::Client::set_int(const String& key, int value, G::Error *error)
{
	return gconf_client_set_int(gconf_client(), key.c_str(), value, *error);
}

bool 
GConf::Client::set_string(const String& key, const String& value, G::Error *error)
{
	return gconf_client_set_string(gconf_client(), key.c_str(), value.c_str(), *error);
}

bool
GConf::Client::set_bool(const String& key, bool value, G::Error *error)
{
	return gconf_client_set_bool(gconf_client(), key.c_str(), value, *error);
}

bool
GConf::Client::set_schema(const String& key, const Schema& value, G::Error *error)
{
	return gconf_client_set_schema(gconf_client(), key.c_str(), value.gconf_schema(), *error);
}

bool
GConf::Client::set_list(const String& key, const std::vector<int>& list, G::Error *error)
{
	g_return_val_if_fail(!list.empty(), false);
	GSList *tmp_list = vector_to_gslist(key, list);
	bool result = gconf_client_set_list(gconf_client(), key.c_str(), GCONF_VALUE_INT, tmp_list, *error); 
 	g_slist_free(tmp_list);
	return result;
}

bool
GConf::Client::set_list(const String& key, const std::vector<bool>& list, G::Error *error)
{
	g_return_val_if_fail(!list.empty(), false);
	GSList *tmp_list = vector_to_gslist(key, list);
	bool result = gconf_client_set_list(gconf_client(), key.c_str(), GCONF_VALUE_BOOL, tmp_list, *error); 
 	g_slist_free(tmp_list);
	return result;
}

bool 
GConf::Client::set_list(const String& key, const std::vector<double>& list, G::Error *error)
{
	g_return_val_if_fail(!list.empty(), false);
	GSList *tmp_list = vector_to_gslist(key, list);
	bool result = gconf_client_set_list(gconf_client(), key.c_str(), GCONF_VALUE_FLOAT, tmp_list, *error); 
	g_slist_free(tmp_list);
	return result;
}

bool
GConf::Client::set_list(const String& key, const std::vector<String>& list, G::Error *error)
{
	g_return_val_if_fail(!list.empty(), false);
	GSList *tmp_list = vector_to_gslist(key, list);
	bool result = gconf_client_set_list(gconf_client(), key.c_str(), GCONF_VALUE_STRING, tmp_list, *error);
 	g_slist_free(tmp_list);
	return result;
}

bool
GConf::Client::set_list(const String& key, const std::vector<Schema>& list, G::Error *error)
{
	g_return_val_if_fail(!list.empty(), false);
	GSList *tmp_list = vector_to_gslist(key, list);
	bool result = gconf_client_set_list(gconf_client(), key.c_str(), GCONF_VALUE_SCHEMA, tmp_list, *error);  
 	g_slist_free(tmp_list);
	return result;
}

bool
GConf::Client::set_pair(const String& key, int car_data, int cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, int car_data, bool cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, int car_data, double cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, int car_data, const String& cdr_data, G::Error *error)
{
	const char *tmp_cdr = cdr_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, int car_data, const Schema& cdr_data, G::Error *error)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, bool car_data, int cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, bool car_data, bool cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, bool car_data, double cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, bool car_data, const String& cdr_data, G::Error *error)
{
	const char *tmp_cdr = cdr_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, bool car_data, const Schema& cdr_data, G::Error *error)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, double car_data, int cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, double car_data, bool cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, double car_data, double cdr_data, G::Error *error)
{
	return gconf_client_set_pair(gconf_client(), key.c_str(), 
	                             GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &car_data, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, double car_data, const String& cdr_data, G::Error *error)
{
	const char *tmp_cdr = cdr_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, double car_data, const Schema& cdr_data, G::Error *error)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                             &car_data, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, const String& car_data, int cdr_data, G::Error *error)
{
	const char *tmp_car = car_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const String& car_data, bool cdr_data, G::Error *error)
{
	const char *tmp_car = car_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const String& car_data, double cdr_data, G::Error *error)
{
	const char *tmp_car = car_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const String& car_data, const String& cdr_data, G::Error *error)
{
	const char *tmp_car = car_data.c_str();
	const char *tmp_cdr = cdr_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                             &tmp_car, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, const String& car_data, const Schema& cdr_data, G::Error *error)
{
	const char *tmp_car = car_data.c_str();
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                             &tmp_car, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, const Schema& car_data, int cdr_data, G::Error *error)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_INT,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const Schema& car_data, bool cdr_data, G::Error *error)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const Schema& car_data, double cdr_data, G::Error *error)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                             &tmp_car, &cdr_data, *error);
}

bool
GConf::Client::set_pair(const String& key, const Schema& car_data, const String& cdr_data, G::Error *error)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	const char *tmp_cdr = cdr_data.c_str();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                             &tmp_car, &tmp_cdr, *error);
}

bool
GConf::Client::set_pair(const String& key, const Schema& car_data, const Schema& cdr_data, G::Error *error)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	return gconf_client_set_pair(gconf_client(), key.c_str(),
	                             GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                             &tmp_car, &tmp_cdr, *error);
}

void 
GConf::Client::error(G::Error& error)
{
	gconf_client_error(gconf_client(), error);
}

void 
GConf::Client::unreturned_error(G::Error& error)
{
	gconf_client_unreturned_error(gconf_client(), error);
}

void 
GConf::Client::value_changed(const String& key, Value& value)
{
	gconf_client_value_changed(gconf_client(), key.c_str(), value.gconf_value());
}

bool 
GConf::Client::commit_change_set(ChangeSet& cs, bool remove_committed, G::Error *error)
{
	return gconf_client_commit_change_set(gconf_client(), cs.gconf_change_set(), remove_committed, *error);
}

Pointer<GConf::ChangeSet>
GConf::Client::reverse_change_set(ChangeSet& cs, G::Error *error)
{
	return new ChangeSet(gconf_client_reverse_change_set(gconf_client(), cs.gconf_change_set(), *error));
}

Pointer<GConf::ChangeSet>
GConf::Client::change_set_from_current(const std::vector<String> keys, G::Error *error)
{
	g_return_val_if_fail(!keys.empty(), 0);
	int count = keys.size();
	const char **tmp_keys = new const char*[count + 1];

	int i = 0;
	while (i < count)
	{
		tmp_keys[i] = keys[i].c_str();
		i++;
	}

	tmp_keys[count] = '\0';
	GConfChangeSet *change_set = gconf_client_change_set_from_currentv(gconf_client(), tmp_keys, *error);
	delete [] tmp_keys;
	return new ChangeSet(change_set);
}

/*  GConf::ClientClass
 */
 
#ifndef GCONF_CLIENT_GET_CLASS
#define GCONF_CLIENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GCONF_TYPE_CLIENT, GConfClientClass))
#endif

void
GConf::ClientClass::init(GConfClientClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->value_changed = &value_changed_proxy;
	g_class->unreturned_error = &unreturned_error_proxy;
	g_class->error = &error_proxy;
}

GType
GConf::ClientClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GConfClientClass),
			(GBaseInitFunc)0,
			(GBaseFinalizeFunc)0,
			(GClassInitFunc)init,
			0, 0, sizeof(GConfClient), 0,
			(GInstanceInitFunc)0
		};
		type = g_type_register_static(GCONF_TYPE_CLIENT, "Inti_GConf_Client", &info, GTypeFlags(0));
	}
	return type;
}

void*
GConf::ClientClass::create()
{
	return g_object_new(get_type(), 0);
}

void
GConf::ClientClass::value_changed_proxy(GConfClient* client, const gchar* key, GConfValue* value)
{
	GConf::Client *tmp_client = G::Object::pointer<GConf::Client>(client);
	if (tmp_client)
	{
		std::string tmp_key(key);
		Value tmp_value(value);
		tmp_client->on_value_changed(tmp_key, value ? &tmp_value : 0);
	}
	else
	{
		GConfClientClass *tmp_class = GCONF_CLIENT_GET_CLASS(client);
		GConfClientClass *g_class = G::TypeInstance::class_peek_parent<GConfClientClass>(tmp_class);
		if (g_class->value_changed)
			g_class->value_changed(client, key, value);
	}
}

void
GConf::ClientClass::unreturned_error_proxy(GConfClient* client, GError* error)
{
	GConf::Client *tmp_client = G::Object::pointer<GConf::Client>(client);
	if (tmp_client)
	{
		G::Error tmp_error(error);
		tmp_client->on_unreturned_error(tmp_error);
	}
	else
	{
		GConfClientClass *tmp_class = GCONF_CLIENT_GET_CLASS(client);
		GConfClientClass *g_class = G::TypeInstance::class_peek_parent<GConfClientClass>(tmp_class);
		if (g_class->unreturned_error)
			g_class->unreturned_error(client, error);
	}
}

void
GConf::ClientClass::error_proxy(GConfClient* client, GError* error)
{
	GConf::Client *tmp_client = G::Object::pointer<GConf::Client>(client);
	if (tmp_client)
	{
		G::Error tmp_error(error);
		tmp_client->on_error(tmp_error);
	}
	else
	{
		GConfClientClass *tmp_class = GCONF_CLIENT_GET_CLASS(client);
		GConfClientClass *g_class = G::TypeInstance::class_peek_parent<GConfClientClass>(tmp_class);
		if (g_class->error)
			g_class->error(client, error);
	}
}

/*  Signal handlers
 */

void 
GConf::Client::on_value_changed(const String& key, const Value *value)
{
	GConfClientClass *g_class = class_peek_parent<GConfClientClass>(gconf_client_class());
	if (g_class->value_changed)
		g_class->value_changed(gconf_client(), key.c_str(), value ? value->gconf_value() : 0);
}

void 
GConf::Client::on_unreturned_error(const G::Error& error)
{
	GConfClientClass *g_class = class_peek_parent<GConfClientClass>(gconf_client_class());
	if (g_class->unreturned_error)
		g_class->unreturned_error(gconf_client(), error);
}

void
GConf::Client::on_error(const G::Error& error)
{
	GConfClientClass *g_class = class_peek_parent<GConfClientClass>(gconf_client_class());
	if (g_class->error)
		g_class->error(gconf_client(), error);
}

/*  Signals
 */

const GConf::Client::ValueChangedSignalType GConf::Client::value_changed_signal("value_changed");

const GConf::Client::UnreturnedErrorSignalType GConf::Client::unreturned_error_signal("unreturned_error");

const GConf::Client::ErrorSignalType GConf::Client::error_signal("error");














