/*******************************************************************************************************************************************
 cframe.c
*******************************************************************************************************************************************/

#include "cframe.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CFrame);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFrame::CFrame 	   (CContainer *inOwner, const CString &inCaption, const CWidgetListener *inListener)
       :CContainer (inOwner, inListener),
	m_Caption  (inCaption==CString()?"id"+CString(GetId()):inCaption)
{
	// gtk instanciation requested
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFrame::~CFrame ()
{
	// gtk/gtkol deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CFrame::PerformWidgetInstanciate ()
{
	// ok
	return gtk_frame_new (m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFrame::PerformWidgetInitialize ()
{
	// show the frame
	Show();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CFrame::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFrame::SetCaption (const CString &inCaption)
{
	// attribute local copy
	m_Caption = inCaption;

	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_frame_set_label (GTK_FRAME(inGtkWidget), m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CFrame::GetCaption () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok ?
	return inGtkWidget == NULL ? CString() : CString(::gtk_frame_get_label (GTK_FRAME(inGtkWidget)));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set the caption alignment
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFrame::SetCaptionAlignment (const float inX, const float inY)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_frame_set_label_align (GTK_FRAME(inGtkWidget), inX, inY);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get x alignment
//-----------------------------------------------------------------------------------------------------------------------------------------
float CFrame::GetCaptionAlignmentX () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return 0.0f;

	// ok
	float ioX, ioY; ::gtk_frame_get_label_align (GTK_FRAME(inGtkWidget), &ioX, &ioY);

	// return it
	return ioX;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get y alignment
//-----------------------------------------------------------------------------------------------------------------------------------------
float CFrame::GetCaptionAlignmentY () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return 0.0f;

	// ok
	float ioX, ioY; ::gtk_frame_get_label_align (GTK_FRAME(inGtkWidget), &ioX, &ioY);

	// return it
	return ioY;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// shadow type affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFrame::SetShadowType (const GtkShadowType inGtkShadowType)
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_frame_set_shadow_type (GTK_FRAME(inGtkWidget), inGtkShadowType);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// shadow type access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkShadowType CFrame::GetShadowType () const
{
	// get our widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok ?
	return inGtkWidget == NULL ? GTK_SHADOW_NONE : ::gtk_frame_get_shadow_type (GTK_FRAME(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFrame::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CContainer::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_FRAME_ELEMENT);
		
			// add the xml attributes
			newXMLElement -> AddAttribute (XML_FRAME_ATTR_CAPTION, GetCaption());
			newXMLElement -> AddAttribute (XML_FRAME_ATTR_CAPTION_ALIGN_X, CString(GetCaptionAlignmentX()));
			newXMLElement -> AddAttribute (XML_FRAME_ATTR_CAPTION_ALIGN_Y, CString(GetCaptionAlignmentY()));
			switch (GetShadowType())
			{
				case GTK_SHADOW_NONE 	   : 
					newXMLElement -> AddAttribute (XML_FRAME_ATTR_SHADOW_TYPE, CString("none")); 
					break;
				case GTK_SHADOW_IN 	   :
					newXMLElement -> AddAttribute (XML_FRAME_ATTR_SHADOW_TYPE, CString("in")); 
					break;
				case GTK_SHADOW_OUT        :
					newXMLElement -> AddAttribute (XML_FRAME_ATTR_SHADOW_TYPE, CString("out")); 
					break;
				case GTK_SHADOW_ETCHED_IN  :
					newXMLElement -> AddAttribute (XML_FRAME_ATTR_SHADOW_TYPE, CString("etched-in")); 
					break;
				case GTK_SHADOW_ETCHED_OUT :
					newXMLElement -> AddAttribute (XML_FRAME_ATTR_SHADOW_TYPE, CString("etched-out")); 
					break;
			}

			// change the io xml element node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_FRAME_ELEMENT);

			// check we got an expected cframe node
			if (inXMLNode == NULL)
				throw new CException (CString("CFrame::Serialize, specified xml node is not a \"") + 
							      XML_FRAME_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cframe node
			ioXMLElementNode = inXMLNode;

			// set the frame caption and alignment from the expected attributes
			SetCaption (::xml_node_get_attribute (inXMLNode, XML_FRAME_ATTR_CAPTION).GetValue());
			SetCaptionAlignment (::xml_node_get_attribute (inXMLNode, XML_FRAME_ATTR_CAPTION_ALIGN_X).GetValue().ToDouble(), 
					     ::xml_node_get_attribute (inXMLNode, XML_FRAME_ATTR_CAPTION_ALIGN_Y).GetValue().ToDouble());

			// get the shadow type and set it
			CString inShadowType (::xml_node_get_attribute (inXMLNode, XML_FRAME_ATTR_SHADOW_TYPE).GetValue());
			if (inShadowType == CString("none"))
				SetShadowType (GTK_SHADOW_NONE);
			else if (inShadowType == CString("in"))
				SetShadowType (GTK_SHADOW_IN);
			else if (inShadowType == CString("out"))
				SetShadowType (GTK_SHADOW_OUT);
			else if (inShadowType == CString("etched-in"))
				SetShadowType (GTK_SHADOW_ETCHED_IN);
			else if (inShadowType == CString("etched-out"))
				SetShadowType (GTK_SHADOW_ETCHED_OUT);
		}
		break;
	}
}

