/*******************************************************************************************************************************************
 cform.c
*******************************************************************************************************************************************/

#include "cform.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CFormListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFormListener::CFormListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFormListener::~CFormListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// specific OnQueryDestroy
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CForm::OnQueryDestroy (GtkWidget *inGtkWidget, GdkEvent *inGdkEvent, gpointer inData)
{
	// defaults to destroy permission
	bool outDoDestroy = true;

	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check the instance
	if (theWidget == NULL) return true;

	// send notification to the listener if any
	if (theWidget -> GetListener() != NULL)
		static_cast <CFormListener *> (theWidget -> GetListener()) -> OnQueryClose (theWidget, outDoDestroy);

	// check the notification result
	if (!outDoDestroy) return true;

	// get the application's (id 1L) main forms (direct children of the CApplication instance)
	CComponents inAbsoluteForms (CComponent::GetComponent ((UInt32)1L) -> GetChildren (__metaclass(CForm)));

	// if the form is the main one
	if (theWidget == *inAbsoluteForms[0])
	{
		// send notification to the listener if any
		if (theWidget -> GetListener() != NULL)
			static_cast <CFormListener *> (theWidget -> GetListener()) -> OnClose (theWidget);
		
		// generic specifications
		return CContainer::OnQueryDestroy (inGtkWidget, inGdkEvent, inData);
	}
	// the form is not the main one
	else
	{
		// the gtk widget is not destroyed
		outDoDestroy = false;

		// send the notification to the listener if any
		if (theWidget -> GetListener() != NULL)
			static_cast <CFormListener *> (theWidget -> GetListener()) -> OnClose (theWidget);

		// it is only hidden
		theWidget -> Hide ();
	}

	// ok
	return !outDoDestroy;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// specific OnDestroy
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::OnDestroy (GtkWidget *inGtkWidget, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// check the pointer
	if (theWidget == NULL) return;

	// get the main application's (Id 1L) forms (direct children of the CApplication instance)
	CComponents inAbsoluteForms (CComponent::GetComponent ((UInt32)1L) -> GetChildren (__metaclass(CForm)));

	// is the destroyed widget the main form ?
	if (theWidget == *inAbsoluteForms[0])

		// delete all of the forms
		for (size_t i=inAbsoluteForms.GetLength(); i>1; i--) delete *inAbsoluteForms[i-1];

	// generic specifications
	CContainer::OnDestroy (inGtkWidget, inData);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnConfigure
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CForm::OnConfigure (GtkWidget *inGtkWidget, GdkEventConfigure *inGdkEvent, gpointer inData)
{
	// GtkAllocation declaration
        GtkAllocation theGtkAllocation;

	// get the root bounds of the window
	gint x, y; ::gdk_window_get_root_origin (inGtkWidget->window, &x, &y);
                                                                                                                                                
        // allocation fill in
        theGtkAllocation.x      = x;
        theGtkAllocation.y      = y;
        theGtkAllocation.width  = inGdkEvent -> width;
        theGtkAllocation.height = inGdkEvent -> height;

        // allocation validation, indirect call of CWidget::OnResize
        ::gtk_widget_size_allocate (inGtkWidget, &theGtkAllocation);

        // propagate the event further
        return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnStateEvent
//-----------------------------------------------------------------------------------------------------------------------------------------
gboolean CForm::OnStateEvent (GtkWidget *, GdkEvent *inGdkEvent, gpointer inData)
{
	// retreive our gtkol instance
	CForm *inForm = reinterpret_cast <CForm *> (inData);

	// check pointer
	if (inForm == NULL) return false;

	// event analyse (don't look at withdrawn states that are already handled by the OnShow and OnHide listening)
	if (inGdkEvent -> type == GDK_WINDOW_STATE && ((GdkEventWindowState*)inGdkEvent) -> changed_mask != GDK_WINDOW_STATE_WITHDRAWN)
	{
		// the new form state
		int newFormState = 0;

		// state analyse
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_ICONIFIED) 
			newFormState |= FORMSTATE_ICONIFIED;
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_MAXIMIZED) 
			newFormState |= FORMSTATE_MAXIMIZED;
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_STICKY) 
			newFormState |= FORMSTATE_STICKY;
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_FULLSCREEN) 
			newFormState |= FORMSTATE_FULLSCREEN;
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_ABOVE) 
			newFormState |= FORMSTATE_ABOVE;
		if (((GdkEventWindowState*)inGdkEvent) -> new_window_state & GDK_WINDOW_STATE_BELOW) 
			newFormState |= FORMSTATE_BELOW;

		// send notification to the listener if any
		if (inForm -> GetListener() != NULL)
			static_cast <CFormListener *> (inForm -> GetListener()) -> OnStateChange (inForm, newFormState);

		// keep a form states copy on the instance
		inForm -> m_FormState = newFormState;
	}

	// propagate event further
	return false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CForm);

//-----------------------------------------------------------------------------------------------------------------------------------------
// static variable resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkAccelGroup * CForm::m_AccelGroup = NULL;

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CForm::CForm 	   	(CComponent *inOwner, const CFormListener *inListener)
      :CContainer  	(inOwner, inListener),
       m_FormState 	(0)
{
	// the form should handle the serialization process on the bounds too, so set the appropriate flag to do so
	CControl::m_SerializeBounds = true;

	// launch the encapsulated gtk widget instanciation process if the component owner is specified
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CForm::~CForm ()
{
	// widget and subwidgets deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CForm::PerformWidgetInstanciate ()
{
	// the gtk widget
	GtkWidget *outGtkWidget = ::gtk_window_new (GTK_WINDOW_TOPLEVEL);
	
	// allocate hotkeys table if not defined yet
	if (CForm::m_AccelGroup == NULL) CForm::m_AccelGroup = ::gtk_accel_group_new ();

	// hotkeys table assignation
	::gtk_window_add_accel_group (GTK_WINDOW(outGtkWidget), CForm::m_AccelGroup);

	// ok
	return outGtkWidget;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::PerformWidgetInitialize ()
{
	// get the gtk widget pointer of this gtkol instance
	GtkWidget *inGtkWidget = GetGtkWidget();

	// check the gtk widget pointer...
	if (inGtkWidget == NULL) return;

	// static events disconnections
        ::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CContainer::OnConfigure),    this);
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CContainer::OnDestroy),      this);
        ::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CContainer::OnQueryDestroy), this);

	// specific static events connections
        ::g_signal_connect (G_OBJECT(inGtkWidget), "configure-event", 	 G_CALLBACK(CForm::OnConfigure),    this);
        ::g_signal_connect (G_OBJECT(inGtkWidget), "destroy", 	     	 G_CALLBACK(CForm::OnDestroy),      this);
        ::g_signal_connect (G_OBJECT(inGtkWidget), "delete-event",    	 G_CALLBACK(CForm::OnQueryDestroy), this);
	::g_signal_connect (G_OBJECT(inGtkWidget), "window-state-event", G_CALLBACK(CForm::OnStateEvent),   this);

	// default caption affectation
	SetCaption ("id"+CString(GetId()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CForm::ListenerMustBe () const
{
	return __metaclass(CFormListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CForm::OwnerMustBe () const
{
	return __metaclasses(CComponent);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// bounds affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::SetBounds (const TBounds &inBounds)
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// check the pointer
	if (inGtkWidget == NULL) return;

	// check bounds entry
	if (inBounds.x == -1 && inBounds.y == -1)
        {
                ::gtk_window_set_default_size (GTK_WINDOW(inGtkWidget), inBounds.w, inBounds.h);
        }
        else if (inBounds.w == -1 && inBounds.h == -1)
        {
                CContainer::SetBounds (inBounds);
        }
        else
        {
                ::gtk_window_resize (GTK_WINDOW(inGtkWidget), inBounds.w, inBounds.h);
                ::gtk_window_move   (GTK_WINDOW(inGtkWidget), inBounds.x, inBounds.y);
        }
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// hotkeys table access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkAccelGroup *	CForm::GetAccelGroup ()
{
	return CForm::m_AccelGroup;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::SetCaption (const CString &inCaption)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// caption affectation
	::gtk_window_set_title (GTK_WINDOW(inGtkWidget), inCaption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// caption access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CForm::GetCaption () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? CString (::gtk_window_get_title (GTK_WINDOW(inGtkWidget))) : "id"+CString(GetId());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// maximize
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Maximize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// maximize
	if (inGtkWidget != NULL) ::gtk_window_maximize (GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// minimize
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Minimize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// maximize
	if (inGtkWidget != NULL) ::gtk_window_iconify (GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// full screen
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::FullScreen ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// maximize
	if (inGtkWidget != NULL) ::gtk_window_fullscreen (GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// restore
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Restore ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// state restore
	if (m_FormState&FORMSTATE_ICONIFIED ) ::gtk_window_deiconify  	(GTK_WINDOW(inGtkWidget));
	if (m_FormState&FORMSTATE_MAXIMIZED ) ::gtk_window_unmaximize	(GTK_WINDOW(inGtkWidget));
	if (m_FormState&FORMSTATE_FULLSCREEN) ::gtk_window_unfullscreen	(GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// stick
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Stick ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL) ::gtk_window_stick (GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// unstick
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::UnStick ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL) ::gtk_window_unstick (GTK_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// above
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::KeepAbove (const bool inAbove)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL) ::gtk_window_set_keep_above (GTK_WINDOW(inGtkWidget), inAbove);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// below
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::KeepBelow (const bool inBelow)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL) ::gtk_window_set_keep_below (GTK_WINDOW(inGtkWidget), inBelow);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// properties affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::SetProperties (const int inProperties)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL)
	{
		if (inProperties&FORMPROPERTIE_RESIZEABLE) 
			::gtk_window_set_resizable (GTK_WINDOW(inGtkWidget), true);
		else
			::gtk_window_set_resizable (GTK_WINDOW(inGtkWidget), false);
		if (inProperties&FORMPROPERTIE_DECORATED)
			::gtk_window_set_decorated (GTK_WINDOW(inGtkWidget), true);
		else
			::gtk_window_set_decorated (GTK_WINDOW(inGtkWidget), false);
		if (inProperties&FORMPROPERTIE_TASKBARHINT)
			::gtk_window_set_skip_taskbar_hint (GTK_WINDOW(inGtkWidget), false);
		else
			::gtk_window_set_skip_taskbar_hint (GTK_WINDOW(inGtkWidget), true);
		if (inProperties&FORMPROPERTIE_PAGERHINT)
			::gtk_window_set_skip_pager_hint (GTK_WINDOW(inGtkWidget), false);
		else
			::gtk_window_set_skip_pager_hint (GTK_WINDOW(inGtkWidget), true);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// properties access
//-----------------------------------------------------------------------------------------------------------------------------------------
int CForm::GetProperties () const
{
	// out properties
	int outProperties=0;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget != NULL)
	{
		if (!::gtk_window_get_skip_taskbar_hint (GTK_WINDOW(inGtkWidget))) outProperties |= FORMPROPERTIE_TASKBARHINT;
		if (!::gtk_window_get_skip_pager_hint	(GTK_WINDOW(inGtkWidget))) outProperties |= FORMPROPERTIE_PAGERHINT;
		if ( ::gtk_window_get_resizable 	(GTK_WINDOW(inGtkWidget))) outProperties |= FORMPROPERTIE_RESIZEABLE;
		if ( ::gtk_window_get_decorated 	(GTK_WINDOW(inGtkWidget))) outProperties |= FORMPROPERTIE_DECORATED;
	}

	// ok
	return outProperties;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// form state access
//-----------------------------------------------------------------------------------------------------------------------------------------
int CForm::GetState () const
{
	return m_FormState;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// close request
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Close ()
{
	// defaults to do close
	Bool doClose = true;

	// send notification to the listener if any
	if (m_Listener != NULL)
		static_cast <CFormListener *> (m_Listener) -> OnQueryClose (this, doClose);

	// check the notification's result
	if (!doClose) return;

	// send the notification to the listener if any
	if (m_Listener != NULL)
		static_cast <CFormListener *> (m_Listener) -> OnClose (this);

	// is this the main form ?
	if ((*CComponent::GetComponents (__metaclass(CForm))[0]) == this)

		// destroy it
		CWidget::DestroyWidget (this);

	// the widget is not the main form
	else

		// just hide the form
		Hide();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// cform xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CForm::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization call
	CContainer::Serialize (ioXMLElementNode, inMode);
	
	// serialization request analyse
	switch (inMode)
	{
		// serialization dump
		case XML_WRITE :
		{
			// allocate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_FORM_ELEMENT);

			// add the specific form attributes
			newXMLElement -> AddAttribute (XML_FORM_ATTR_CAPTION, GetCaption());
			newXMLElement -> AddAttribute (XML_FORM_ATTR_MAXIMIZE, CString((m_FormState&FORMSTATE_MAXIMIZED) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_MINIMIZE, CString((m_FormState&FORMSTATE_ICONIFIED) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_FULLSCREEN, CString((m_FormState&FORMSTATE_FULLSCREEN) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_STICKY, CString((m_FormState&FORMSTATE_STICKY) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_ABOVE, CString((m_FormState&FORMSTATE_ABOVE) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_BELOW, CString((m_FormState&FORMSTATE_BELOW) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_RESIZEABLE, CString((GetProperties()&FORMPROPERTIE_RESIZEABLE) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_DECORATED, CString((GetProperties()&FORMPROPERTIE_DECORATED) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_TASKBARHINT, CString((GetProperties()&FORMPROPERTIE_TASKBARHINT) ?
							"true":"false"));
			newXMLElement -> AddAttribute (XML_FORM_ATTR_PAGERHINT, CString((GetProperties()&FORMPROPERTIE_PAGERHINT) ?
							"true":"false"));

			// modify the input/output pointer so that the serialization process of overwritten definition continues under the
			// current cform xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// serialization load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_FORM_ELEMENT);

			// check we got an expected cform node
			if (inXMLNode == NULL)
				throw new CException (CString("CForm::Serialize, specified xml node is not a \"") + 
							      XML_FORM_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cform node
			ioXMLElementNode = inXMLNode;

			// retreive and set the form caption
			SetCaption (::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_CAPTION).GetValue());

			// get the form state and properties
			bool inMaximize	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_MAXIMIZE).GetValue().ToBool();
			bool inMinimize   = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_MINIMIZE).GetValue().ToBool();
			bool inFullScreen = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_FULLSCREEN).GetValue().ToBool();
			bool inSticky	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_STICKY).GetValue().ToBool();
			bool inAbove	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_ABOVE).GetValue().ToBool();
			bool inBelow	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_BELOW).GetValue().ToBool();
			bool inResizeable = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_RESIZEABLE).GetValue().ToBool();
			bool inDecorated  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_DECORATED).GetValue().ToBool();
			bool inTaskBar	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_TASKBARHINT).GetValue().ToBool();
			bool inPager	  = ::xml_node_get_attribute (inXMLNode, XML_FORM_ATTR_PAGERHINT).GetValue().ToBool();

			// properties flag construction
			int outProperties = 0;
			outProperties |= (inResizeable) ? FORMPROPERTIE_RESIZEABLE  : 0;
			outProperties |= (inDecorated ) ? FORMPROPERTIE_DECORATED   : 0;
			outProperties |= (inTaskBar   ) ? FORMPROPERTIE_TASKBARHINT : 0;
			outProperties |= (inPager     ) ? FORMPROPERTIE_PAGERHINT   : 0;

			// state affectation
			if (inMaximize)   Maximize   ();
			if (inMinimize)   Minimize   ();
			if (inFullScreen) FullScreen ();
			if (inSticky)	  Stick      ();
			if (inAbove)	  KeepAbove  (true);
			if (inBelow)	  KeepBelow  (true);

			// properties affectation
			SetProperties (outProperties);
		}
		break;
	}
}



