/*****************************************************************************
 *                                                                           *
 * Widget:    gtkimreg                                                       *
 * Uses:      GTK+1.2, GdkImlib                                              *
 * Purpose:   get user defined coordinates inside an image                   *
 * Author:    Andreas Tille <tille@debian.org>
 * Date:      May 15, 2001                                                   *
 * Copyright: Andreas Tille, 1998, 1999, 2000, 2001                          *
 * License:   LGPL                                                           *
 *                                                                           *
 *****************************************************************************/

#include "gtk/gtkbutton.h"
#include "gtk/gtkcontainer.h"
#include "gtk/gtkdialog.h"
#include "gtk/gtkdrawingarea.h"
#include "gtk/gtkframe.h"
#include "gtk/gtkhbbox.h"
#include "gtk/gtkhbox.h"
#include "gtk/gtkhruler.h"
#include "gtk/gtklabel.h"
#include "gtk/gtksignal.h"
#include "gtk/gtkspinbutton.h"
#include "gtk/gtktable.h"
#include "gtk/gtkvbox.h"
#include "gtk/gtkvruler.h"
#include "gtkimreg.h"

#include "gtkintl.h"

static void gtk_imreg_class_init     (GtkImRegClass *class);
static void gtk_imreg_init           (GtkImReg *imreg);
static void gtk_imreg_destroy        (GtkObject *object);

static void gtk_imreg_button_press   (GtkWidget *widget, GdkEventButton *event, GtkImReg *imreg);
static void gtk_imreg_button_release (GtkWidget *widget, GdkEventButton *event, GtkImReg *imreg);
static void gtk_imreg_configure      (GtkWidget *widget, GdkEventConfigure *event, GtkImReg *imreg);
static void gtk_imreg_expose         (GtkWidget *widget, GdkEventExpose *event, GtkImReg *imreg);
static void gtk_imreg_leave          (GtkWidget *widget, GdkEvent *event, GtkImReg *imreg);
static void gtk_imreg_motion         (GtkWidget *widget, GdkEvent *event, GtkImReg *imreg);
static void gtk_imreg_value_changed  (GtkAdjustment *adj, GtkWidget *spinner);

static GtkAdjustment*  _create_adjustment      (GtkWidget *table, GtkImReg *imreg, gchar *name, 
                                                gint pos, gint *value, gint max);
static void            _draw_rubber            (GtkImReg *imreg);
static void            _handle_rubber          (GtkImReg *imreg, gint x, gint y, 
                                                GdkModifierType state);
static void            _set_cursor             (GdkWindow *window, gint type);
static void            _set_cursor_rel         (GtkWidget *widget, _CursorLocation pos);
static gint            _set_parameter_if_valid (gint *val2set, gint val, gint cond, 
						GtkAdjustment *adj);
static void            _set_rubber_corner      (gint x, gint *x1, gint *x2, gint len, 
                                                GtkAdjustment *adj1, GtkAdjustment *adj2);
static _CursorLocation _where                  (GtkImReg *imreg, gint x, gint y);

#define _is_variable(imreg)                    ( imreg->var == GTK_IMREG_SIZE_VARIABLE ) 
#define _gint_swap(x, y)                       {register gint tmp = x; x = y; y = tmp;}

static GtkWindowClass *parent_class = NULL;

#define FRAME  3

GtkType
gtk_imreg_get_type (void)
{
  static GtkType IMREG_type = 0;

  if (!IMREG_type)
    {
      GtkTypeInfo imreg_info =
      {
	"GtkImReg",
	sizeof (GtkImReg),
	sizeof (GtkImRegClass),
	(GtkClassInitFunc) gtk_imreg_class_init,
	(GtkObjectInitFunc) gtk_imreg_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      IMREG_type = gtk_type_unique (GTK_TYPE_WINDOW, &imreg_info);
    }

  return IMREG_type;
}

static void
gtk_imreg_class_init (GtkImRegClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  parent_class = gtk_type_class (GTK_TYPE_DIALOG);

  object_class->destroy = gtk_imreg_destroy;
}

static void
gtk_imreg_init (GtkImReg *imreg)
{
  GtkWidget *confirm_area;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *table;    /* 2x2 hold rulers in (2,1) and (1,2), pixmap in (2,2) */
  GtkWidget *viewport;

  /* The dialog-sized vertical box  */
  imreg->main_vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (imreg), 10);
  gtk_container_add (GTK_CONTAINER (imreg), imreg->main_vbox);
  gtk_widget_show (imreg->main_vbox);

  /**********************************************************************************
   * construct the cut area (table with sliders & image window)                     *
   **********************************************************************************/

  table = gtk_table_new(2, 2, /* homogeneous */ FALSE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 1);
  gtk_table_set_row_spacing(GTK_TABLE(table), 0, 1);
  gtk_container_set_border_width(GTK_CONTAINER (table), 2);
  gtk_box_pack_start(GTK_BOX(imreg->main_vbox), table, FALSE, FALSE, 0);
                              /* do not allow resizing! ^^^^^  ^^^^^  */
  gtk_widget_show(table);

  /* the empty box in the top-left corner */
  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_OUT);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(frame);

  /**********************************************************************************
   * the image itself                                                               *
   **********************************************************************************/
  viewport = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(viewport), GTK_SHADOW_IN);
  gtk_table_attach(GTK_TABLE(table), viewport, 1, 2, 1, 2, 0, 0, 0, 0);
  gtk_widget_show (viewport);

  hbox = gtk_hbox_new(FALSE, 0);  /* this seems to be neccessary to avoid resizing of drawing area */
  gtk_container_add(GTK_CONTAINER(viewport), hbox);
  gtk_widget_show (hbox);

  imreg->cut = gtk_drawing_area_new();
  gtk_box_pack_start(GTK_BOX(hbox), imreg->cut, FALSE, FALSE, 0);
  gtk_widget_show (imreg->cut);

  /**********************************************************************************
   * the rulers to know where we are                                                *
   **********************************************************************************/
  imreg->hruler = gtk_hruler_new();
  gtk_table_attach(GTK_TABLE(table), imreg->hruler, 1, 2, 0, 1, GTK_FILL, 0, 0, 0);
  gtk_signal_connect_object(GTK_OBJECT(imreg->cut), "motion_notify_event",
         GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS(GTK_OBJECT(imreg->hruler)->klass)->motion_notify_event),
         GTK_OBJECT(imreg->hruler));
  gtk_widget_show (imreg->hruler);
  
  imreg->vruler = gtk_vruler_new();
  gtk_table_attach(GTK_TABLE(table), imreg->vruler, 0, 1, 1, 2, 0, GTK_FILL, 0, 0);
  gtk_signal_connect_object(GTK_OBJECT(imreg->cut), "motion_notify_event",
         GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS(GTK_OBJECT(imreg->vruler)->klass)->motion_notify_event),
         GTK_OBJECT (imreg->vruler));
  gtk_widget_show (imreg->vruler);

  /**********************************************************************************
   * all the signals and events of the pixmap area                                  *
   **********************************************************************************/
  gtk_widget_set_events(imreg->cut, GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK | GDK_ENTER_NOTIFY_MASK );
  /* Signals used to handle backing pixmap */
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "expose_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_expose), imreg);
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "configure_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_configure), imreg);

  /* Event signals */
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "motion_notify_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_motion), imreg);
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "enter_notify_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_motion), imreg);
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "leave_notify_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_leave), imreg);
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "button_release_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_button_release), imreg);
  gtk_signal_connect(GTK_OBJECT(imreg->cut), "button_press_event", 
                     GTK_SIGNAL_FUNC(gtk_imreg_button_press), imreg);

  /**********************************************************************************
   * area for several parameters in the control part                                *
   **********************************************************************************/

  imreg->parms = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(imreg->main_vbox), imreg->parms, TRUE, TRUE, 0);
  gtk_widget_show (imreg->parms);

#define LABEL_WIDTH 65
  /**********************************************************************************
   * current cursor position                                                        *
   **********************************************************************************/
  table = gtk_table_new (2, 2, /* homogeneous */ FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_box_pack_start(GTK_BOX(imreg->parms), table, TRUE, FALSE, 0);
  gtk_widget_show (table);
  
  label = gtk_label_new(_("current:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 0, 0, 0, 0);
  gtk_widget_show (label);

  frame = gtk_frame_new(NULL);
  gtk_widget_set_usize(frame, LABEL_WIDTH, 0);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_table_attach(GTK_TABLE(table), frame, 1, 2, 0, 1, 0, 0, 0, 0);
  gtk_widget_show (frame);

  imreg->pos = gtk_label_new("");
  gtk_container_add(GTK_CONTAINER(frame), imreg->pos);
  gtk_widget_show (imreg->pos);

  /**********************************************************************************
   * current width and height                                                       *
   **********************************************************************************/

  label = gtk_label_new(_("size:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 0, 0);
  gtk_widget_show (label);

  frame = gtk_frame_new(NULL);
  gtk_widget_set_usize(frame, LABEL_WIDTH, 0);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_table_attach(GTK_TABLE(table), frame, 1, 2, 1, 2, 0, 0, 0, 0);
  gtk_widget_show (frame);

  imreg->size = gtk_label_new("");
  gtk_container_add(GTK_CONTAINER(frame), imreg->size);
  gtk_widget_show (imreg->size);
#undef LABEL_WIDTH 

  /**********************************************************************************
   * The OK/Cancel button area                                                      *
   **********************************************************************************/
   
  confirm_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(confirm_area), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(confirm_area), 5);
  gtk_box_pack_end (GTK_BOX (imreg->main_vbox), confirm_area, FALSE, FALSE, 0);
  gtk_widget_show (confirm_area);

  /****************************************************************************
   * The OK button                                                            *
   ****************************************************************************/
  imreg->ok_button = gtk_button_new_with_label (_("OK"));
  GTK_WIDGET_SET_FLAGS (imreg->ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), imreg->ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (imreg->ok_button);
  gtk_widget_show (imreg->ok_button);

  /**********************************************************************************
   * The Cancel button                                                              *
   **********************************************************************************/
  imreg->cancel_button = gtk_button_new_with_label (_("Cancel"));
  GTK_WIDGET_SET_FLAGS (imreg->cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), imreg->cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (imreg->cancel_button);

  /**********************************************************************************
   * allocate memory for rubber boxes (do free only ->r)                            *
   **********************************************************************************/
  imreg->r         = imreg->old = NULL;
  imreg->adj_le    = imreg->adj_to = imreg->adj_ri = imreg->adj_bo = NULL;
  imreg->pm        = NULL;
  imreg->XorGc     = NULL;
  imreg->where     = NOWHERE;
  imreg->im        = NULL;
  imreg->rw        = imreg->rh = 0;
  imreg->xi        = imreg->yi = -1;
  imreg->var       = GTK_IMREG_SIZE_VARIABLE;
  imreg->user_area = NULL;

/*  gtk_widget_grab_focus (imreg->cut); */
}

GtkWidget*
gtk_imreg_new (const gchar *title, GdkImlibImage *im, gint rw, gint rh, GtkWidget *user_area)
/* 
 * --- Parameter: ---
 * const gchar   *title    : widget title
 * GdkImlibImage *im       : image to cut or insert into
 * gint           rw       : width of region to insert
 * gint           rh       : width of region to insert
 * GtkWidget     *user_area: user defined widget to insert in the control area
 */
{
  GtkImReg     *imreg;
  GtkWidget    *table;
  gchar        *buf;
  gint          max_right, max_bottom;
  register gint w, h;
  gchar        *unable = _("Unable to select region of %s %i from image of only %s %i."),
               *width  = _("width"),
	       *height = _("height");

  g_return_val_if_fail ( im != NULL, NULL );
  g_return_val_if_fail ( im->rgb_width > 0, NULL );
  g_return_val_if_fail ( im->rgb_height > 0, NULL );
  g_return_val_if_fail ( im->rgb_data , NULL ) ;
  if ( im->rgb_width < rw ) {
    g_warning(unable, width, rw, width, im->rgb_width);
    return NULL;
  }
  if ( im->rgb_height < rh ) {
    g_warning(unable, height, rh, height, im->rgb_height);
    return NULL;
  }
  if ( im->rgb_width == rw && im->rgb_height == rh ) {
    g_warning(_("The requested section has the same dimension as the original image (%i,%i)."), rw, rh);
    return NULL;
  }
  
  imreg = gtk_type_new (GTK_TYPE_IMREG);
  g_return_val_if_fail ( imreg != NULL, NULL );

  gtk_window_set_title (GTK_WINDOW (imreg), title);
  imreg->im = im;
  imreg->r  = g_malloc (sizeof(_Rubber));

  w         = im->rgb_width;
  h         = im->rgb_height;
  if ( rw > 0 && rh > 0 ) {
    imreg->var   = GTK_IMREG_SIZE_FIX;
    imreg->rw    = rw;
    imreg->rh    = rh;
    buf          = g_strdup_printf("%4ix%4i", imreg->rw, imreg->rh);
    imreg->r->le = (w - imreg->rw) >> 1;  
    imreg->r->to = (h - imreg->rh) >> 1;  
    imreg->r->ri = imreg->r->le + imreg->rw - 1;
    imreg->r->bo = imreg->r->to + imreg->rh - 1;
    max_right    = w - imreg->rw - 1;
    max_bottom   = h - imreg->rh - 1;
  } else {
    imreg->var   = GTK_IMREG_SIZE_VARIABLE;
    imreg->rw    = w;
    imreg->rh    = h;
    buf          = g_strdup_printf("%4ix%4i", w, h);
    max_right    = w - 1;
    max_bottom   = h - 1;
    imreg->r->le = 0;
    imreg->r->to = 0;
    imreg->r->ri = max_right;
    imreg->r->bo = max_bottom;
  }
  if ( user_area && GTK_IS_WIDGET (user_area) ) imreg->user_area = user_area;
  
  gtk_drawing_area_size(GTK_DRAWING_AREA(imreg->cut), w, h);
  gtk_ruler_set_range(GTK_RULER(imreg->hruler), 0, w - 1, 0, h - 1);
  gtk_ruler_set_range(GTK_RULER(imreg->vruler), 0, h - 1, 0, w - 1);
  gtk_label_set(GTK_LABEL(imreg->size), buf);
  if ( buf ) g_free(buf);
  
  /**********************************************************************************
   * editables for left and top position                                            *
   **********************************************************************************/
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_box_pack_start(GTK_BOX(imreg->parms), table, TRUE, FALSE, 0);
  
  imreg->adj_le = _create_adjustment(table, imreg, _("left:"), 0, &(imreg->r->le), max_right);
  imreg->adj_to = _create_adjustment(table, imreg, _("top:"), 1,  &(imreg->r->to), max_bottom);
  gtk_widget_show(table);

  /**********************************************************************************
   * editables for right and bottom position only for variable sized region         *
   **********************************************************************************/
  if ( _is_variable(imreg) ) {
    table = gtk_table_new (2, 2, FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);
    gtk_box_pack_start(GTK_BOX(imreg->parms), table, TRUE, FALSE, 0);

    imreg->adj_ri = _create_adjustment(table, imreg, _("right:"), 0, &(imreg->r->ri), max_right);
    imreg->adj_bo = _create_adjustment(table, imreg, _("bottom:"), 1,  &(imreg->r->bo), max_bottom);
    gtk_widget_show(table);
  }

  /**********************************************************************************
   * any user defined area                                                          *
   **********************************************************************************/
  if ( imreg->user_area ) {
    gtk_box_pack_start(GTK_BOX(imreg->parms), imreg->user_area, TRUE, FALSE, 0);
    gtk_widget_show(imreg->user_area);
  }

  return GTK_WIDGET (imreg);
}

gint
gtk_imreg_get_left (GtkImReg *imreg)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);

  return MAX(imreg->r->le, 0);
}

gint
gtk_imreg_get_top (GtkImReg *imreg)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);

  return MAX(imreg->r->to, 0);
}

gint
gtk_imreg_get_right (GtkImReg *imreg)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);

  if ( _is_variable(imreg) ) return MIN(imreg->r->ri, imreg->im->rgb_width - 1);
  else                       return MIN(imreg->r->le + imreg->rw, imreg->im->rgb_height - 1);
}

gint
gtk_imreg_get_bottom (GtkImReg *imreg)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);

  if ( _is_variable(imreg) ) return MIN(imreg->r->bo, imreg->im->rgb_height - 1);
  else                       return MIN(imreg->r->to + imreg->rh, imreg->im->rgb_height - 1);
}

gint
gtk_imreg_get (GtkImReg *imreg, gint *le, gint *to, gint *ri, gint *bo)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);

  *le = MAX(imreg->r->le, 0);
  *to = MAX(imreg->r->to, 0);
  if ( _is_variable(imreg) ) {
    *ri = MIN(imreg->r->ri, imreg->im->rgb_width - 1);
    *bo = MIN(imreg->r->bo, imreg->im->rgb_height - 1);
  } else {
    *ri = MIN(imreg->r->le + imreg->rw, imreg->im->rgb_width - 1);
    *bo = MIN(imreg->r->to + imreg->rh, imreg->im->rgb_height - 1);
  }
  return 0;
}

gint
gtk_imreg_set_left (GtkImReg *imreg, gint left)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);
  if ( left < 0 || left >= imreg->im->rgb_width ) return -1;

  if ( _is_variable(imreg) ) {
    if ( left > imreg->r->ri ) {
      gint tmp = imreg->r->ri;
      gtk_adjustment_set_value(imreg->adj_ri, imreg->r->ri);
      left = tmp;
    } 
  } else {
    if ( left >= imreg->im->rgb_width - imreg->rw ) 
      left = imreg->im->rgb_width - imreg->rw - 1;
  }
  gtk_adjustment_set_value(imreg->adj_le, left);
  return 0;
}

gint
gtk_imreg_set_top (GtkImReg *imreg, gint top)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);
  if ( top < 0 || top >= imreg->im->rgb_height ) return -1;

  if ( _is_variable(imreg) ) {
    if ( top > imreg->r->bo ) {
      gint tmp = imreg->r->bo;
      gtk_adjustment_set_value(imreg->adj_bo, top);
      top = tmp;
    } 
  } else {
    if ( top >= imreg->im->rgb_height - imreg->rh ) 
      top = imreg->im->rgb_height - imreg->rh - 1;
  }

  gtk_adjustment_set_value(imreg->adj_to, top);
  return 0;
}

gint
gtk_imreg_set_right (GtkImReg *imreg, gint right)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);
  if ( right < 0 || right >= imreg->im->rgb_width ) return -1;

  if ( _is_variable(imreg) ) {
    if ( right < imreg->r->le ) {
      gint tmp = imreg->r->le;
      gtk_adjustment_set_value(imreg->adj_le, imreg->r->le);
      right = tmp;
    } 
    gtk_adjustment_set_value(imreg->adj_ri, right);
  } else {
    if ( right < imreg->rw ) right  = 0;
    else                     right -= imreg->rw;
    gtk_adjustment_set_value(imreg->adj_le, right);    
  }
  return 0;
}

gint
gtk_imreg_set_bottom (GtkImReg *imreg, gint bottom)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);
  if ( bottom < 0 || bottom >= imreg->im->rgb_height ) return -1;

  if ( _is_variable(imreg) ) {
    if ( bottom < imreg->r->to ) {
      gint tmp = imreg->r->to;
      gtk_adjustment_set_value(imreg->adj_to, bottom);
      bottom = tmp;
    } 
    gtk_adjustment_set_value(imreg->adj_bo, bottom);
  } else {
    if ( bottom < imreg->rh ) bottom  = 0;
    else                      bottom -= imreg->rh;
    gtk_adjustment_set_value(imreg->adj_to, bottom);
  }

  return 0;
}

gint
gtk_imreg_set (GtkImReg *imreg, gint left, gint top, gint right, gint bottom)
{
  g_return_val_if_fail (imreg != NULL, -1);
  g_return_val_if_fail (GTK_IS_IMREG (imreg), -1);
  g_return_val_if_fail (imreg->r != NULL, -1);
  if ( left < 0 || left >= imreg->im->rgb_width ||
       top  < 0 || top  >= imreg->im->rgb_height ) return -1;
  if ( _is_variable(imreg) ) {
    if ( right  < 0 || right  >= imreg->im->rgb_width  || left == right ||
         bottom < 0 || bottom >= imreg->im->rgb_height || top  == bottom ) return -1;
    if ( right  < left ) _gint_swap(left, right);
    if ( bottom < top )  _gint_swap(top,  bottom);
    
  } else {
    if ( left >= imreg->im->rgb_width  - imreg->rw ) 
      left = imreg->im->rgb_width  - imreg->rw - 1;
    if ( top  >= imreg->im->rgb_height - imreg->rh ) 
      top  = imreg->im->rgb_height - imreg->rh - 1;
  }
  gtk_adjustment_set_value(imreg->adj_le, left);
  gtk_adjustment_set_value(imreg->adj_to, top);
  if ( _is_variable(imreg) ) {
    gtk_adjustment_set_value(imreg->adj_ri, right);
    gtk_adjustment_set_value(imreg->adj_bo, bottom);
  }
  return 0;
}

static void
gtk_imreg_destroy (GtkObject *object)
{
  GtkImReg *imreg;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_IMREG (object));

  imreg = GTK_IMREG (object);
  if ( imreg->pm ) gdk_imlib_free_pixmap(imreg->pm);
  imreg->pm = NULL;
  if ( imreg->XorGc ) gdk_gc_destroy(imreg->XorGc);
  imreg->XorGc = NULL;
  g_free(imreg->r);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/**********************************************************************
 *	   image region selection callbacks                           *
 **********************************************************************/

static void
gtk_imreg_button_press(GtkWidget *widget, GdkEventButton *event,
                                        GtkImReg *imreg)
/* Handle button press event
 * left button: if inside box: move whole box
 *              if on border/corner: move border corner
 *              if outside: mark new corner and move it
 * middle button: delete box
 * right button: mark new corner and move it
 * GtkWidget      *widget : drawing area with image to cut and rubber box
 * GdkEventButton *event  : button_press_event
 * GtkImReg       *imreg  : image region selection structure
 */
{
  register gint x, y, w, h;

  g_return_if_fail ( widget );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_BUTTON_PRESS || event->type == GDK_2BUTTON_PRESS || 
                     event->type == GDK_3BUTTON_PRESS );

  x = (int)event->x;
  y = (int)event->y;
   
  imreg->where = _where(imreg, x, y);
  if ( imreg->where == INNER && event->button == 1 ) {
    g_return_if_fail ( imreg->old );
    imreg->xi = x - imreg->old->le;
    imreg->yi = y - imreg->old->to;
    if ( _is_variable(imreg) ) {
       w = imreg->old->ri - imreg->old->le;
       h = imreg->old->bo - imreg->old->to;
    } else {
       w = imreg->rw;
       h = imreg->rh;
    }
    if ( imreg->xi < 0 || w < imreg->xi || imreg->yi < 0 || h < imreg->yi ) {
      g_warning(_("Invalid position while trying to move box."));
      imreg->xi = imreg->yi = -1;
    }
  }

  /**************************************************************************************
   * delete rubberbox if mouse button 2 (middle) was pressed                            *
   **************************************************************************************/
  if ( event->button == 2 ) {
    if ( !imreg->old ) return;
    if ( !_is_variable(imreg) ) { /* middle button isn't handled for fixed sized images */
      imreg->where = NOWHERE;
      imreg->xi    = imreg->yi = -1;
      return;
    }
    gtk_adjustment_set_value(imreg->adj_le, 0);
    gtk_adjustment_set_value(imreg->adj_to, 0);
    gtk_adjustment_set_value(imreg->adj_ri, imreg->im->rgb_width - 1);
    gtk_adjustment_set_value(imreg->adj_bo, imreg->im->rgb_height - 1);
    g_free(imreg->old);
    imreg->old = NULL;

    /* Instead of repainting the whole pixmap it might be good to draw rectangle  *
     * only if it is not equal to the image borders.                              */
    if ( imreg->pm ) gdk_imlib_free_pixmap(imreg->pm);
    g_return_if_fail ( gdk_imlib_render(imreg->im, imreg->im->rgb_width, imreg->im->rgb_height) ) ;
    imreg->pm = gdk_imlib_copy_image(imreg->im);
    gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], imreg->pm,
                    0, 0, 0, 0, imreg->im->rgb_width - 1, imreg->im->rgb_height - 1);
    return;
  }

  /**************************************************************************************
   * when initial state (that means also after pressing button 2) draw a new box with   *
   * the left mouse button                                                              *
   **************************************************************************************/

  if ( event->button == 1 && !(imreg->old) ) {
    imreg->r->le = x;
    gtk_adjustment_set_value(imreg->adj_le, x);
    imreg->r->to = y;
    gtk_adjustment_set_value(imreg->adj_to, y);
    if ( _is_variable(imreg) ) {
      imreg->r->ri = x;
      gtk_adjustment_set_value(imreg->adj_ri, x);
      imreg->r->bo = y;
      gtk_adjustment_set_value(imreg->adj_bo, y);
    }
  } else {
    if ( (event->button == 1 && imreg->where == OUT) || (event->button == 3 ) ) {
      if ( _is_variable(imreg) ) {
        _set_rubber_corner(x, &(imreg->r->le), &(imreg->r->ri), 0, imreg->adj_le, imreg->adj_ri);
        _set_rubber_corner(y, &(imreg->r->to), &(imreg->r->bo), 0, imreg->adj_to, imreg->adj_bo);
      } else {
        _set_rubber_corner(x, &(imreg->r->le), NULL, imreg->rw, imreg->adj_le, NULL);
        _set_rubber_corner(y, &(imreg->r->to), NULL, imreg->rh, imreg->adj_to, NULL);
      }
    }
  }
  _draw_rubber(imreg);
}

static void
gtk_imreg_button_release(GtkWidget *widget, GdkEventButton *event,
                                          GtkImReg *imreg)
/* Handle button release event
 * GtkWidget      *widget : drawing area with image to cut and rubber box
 * GdkEventButton *event  : button_release_event
 * GtkImReg       *imreg  : image region selection structure
 */
{
  g_return_if_fail ( widget );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_BUTTON_RELEASE );

  _set_cursor_rel(widget, _where(imreg, (int)event->x, (int)event->y));
  imreg->where = NOWHERE;
  imreg->xi    = imreg->yi = -1;
}

static void
gtk_imreg_configure(GtkWidget *widget, GdkEventConfigure *event, 
                                     GtkImReg *imreg)
/* Create a new backing pixmap of the appropriate size 
 * --- Parameter: ---
 * GtkWidget         *widget : drawing area with image to cut and rubber box
 * GdkEventConfigure *event  : configure_event
 * GtkImReg          *imreg  : image region selection structure
 */
{
  g_return_if_fail ( widget );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_CONFIGURE );
   
  if ( imreg->pm ) return; /* pixmap already created, so nothing is to do */
  g_return_if_fail ( gdk_imlib_render(imreg->im, imreg->im->rgb_width, imreg->im->rgb_height) ) ;
  imreg->pm = gdk_imlib_copy_image(imreg->im);
  return;
}

static void
gtk_imreg_expose(GtkWidget *widget, GdkEventExpose *event, GtkImReg *imreg)
/* Redraw the screen from the backing pixmap 
 * --- Parameter: ---
 * GtkWidget      *widget : drawing area with image to cut and rubber box
 * GdkEventExpose *event  : expose_event
 * GtkImReg       *imreg  : image region selection structure
 */
{
  GdkColor   color;

  g_return_if_fail ( widget );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( imreg->pm );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_EXPOSE );

  gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], imreg->pm,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  _set_cursor(widget->window, GDK_CROSSHAIR);
   
  if ( imreg->XorGc ) gdk_gc_destroy(imreg->XorGc);
  imreg->XorGc = gdk_gc_new (widget->window);
  gdk_color_white(gdk_window_get_colormap(widget->window), &color);
  gdk_gc_set_foreground(imreg->XorGc, &color);
  gdk_gc_set_function(imreg->XorGc, GDK_XOR);
  if ( !_is_variable(imreg) && !imreg->old &&
       !(imreg->r->le == 0 && imreg->r->to == 0   &&  /* if _set() was called _draw */
         imreg->r->ri == imreg->im->rgb_width  - 1 &&
	 imreg->r->bo == imreg->im->rgb_height - 1) ) 
    _draw_rubber(imreg);
}
  

static void
gtk_imreg_leave(GtkWidget *widget, GdkEvent *event, GtkImReg *imreg)
/* Handle leave event
 * GtkWidget               *widget : drawing area with image to cut and rubber box
 * GdkEventLeave           *event  : motion_event
 * GtkImReg *imreg : image region selection structure
 */
{
  gint             x, y;
  GdkModifierType  state;

  g_return_if_fail ( widget );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( imreg->pos );
  g_return_if_fail ( GTK_IS_LABEL(imreg->pos) );
  g_return_if_fail ( imreg->im );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_LEAVE_NOTIFY );

  gtk_label_set(GTK_LABEL(imreg->pos), "");
  if ( imreg->where == NOWHERE ) return;
  /* if rubber box is resized check for clean leaving of the window */
  gdk_window_get_pointer(widget->window, &x, &y, &state);
  if ( x < 0 ) x = 0;
  else if ( x >= imreg->im->rgb_width  ) x = imreg->im->rgb_width  - 1;
  if ( y < 0 ) y = 0;
  else if ( y >= imreg->im->rgb_height ) y = imreg->im->rgb_height - 1;
  _handle_rubber(imreg, x, y, state);
}

static void
gtk_imreg_motion(GtkWidget *widget, GdkEvent *event, GtkImReg *imreg)
/* Handle motion and notify events
 * GtkWidget               *widget : drawing area with image to cut and rubber box
 * GdkEvent                *event  : motion_event
 * GtkImReg *imreg : image region selection structure
 */
{
  gchar          *buf;
  gint            x, y;
  GdkModifierType state;

  g_return_if_fail ( widget );
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( imreg->pm );
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_ENTER_NOTIFY || event->type == GDK_MOTION_NOTIFY );
  
  if ( event->type == GDK_ENTER_NOTIFY ) 
     gdk_window_get_pointer(widget->window, &x, &y, &state);
  else if ( event->type == GDK_MOTION_NOTIFY ) {
     x     = event->motion.x;
     y     = event->motion.y;
     state = event->motion.state;
  }

  buf = g_strdup_printf("%4i, %4i", x, y);
  gtk_label_set(GTK_LABEL(imreg->pos), buf);
  if ( buf ) g_free(buf);
  gtk_widget_show(imreg->pos);

  if ( !state ) {
    _set_cursor_rel(widget, _where(imreg, x, y));
    return ;
  }
  _handle_rubber(imreg, x, y, state);
}

static void
gtk_imreg_value_changed(GtkAdjustment *adj, GtkWidget *spinner)
/* called while editing box parameters
 * --- Parameter: ---
 * GtkAdjustment *adj     : target for new value
 * GtkWidget     *spinner :
 */
{
  gint                    *chg, new;
  GtkImReg *imreg;
  
  g_return_if_fail ( adj );
  g_return_if_fail ( spinner );
  g_return_if_fail ( GTK_IS_OBJECT (adj) );
  g_return_if_fail ( GTK_IS_SPIN_BUTTON (spinner) );
  imreg = gtk_object_get_user_data(GTK_OBJECT(spinner));
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );

  chg    = gtk_object_get_user_data(GTK_OBJECT(adj));
  new    = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner));

  /* avoid that left or top exceed right or bottom, respectively */
  if ( _is_variable(imreg) ) {
    if        ( chg == &(imreg->r->le) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->le), new, new < imreg->r->ri, adj) ) return;
    } else if ( chg == &(imreg->r->to) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->to), new, new < imreg->r->bo, adj) ) return;
    } else if ( chg == &(imreg->r->ri) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->ri), new, new > imreg->r->le, adj) ) return;
    } else if ( chg == &(imreg->r->bo) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->bo), new, new > imreg->r->to, adj) ) return;
    } else return;
  } else {
    if        ( chg == &(imreg->r->le) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->le), new, new < imreg->im->rgb_width - imreg->rw, adj) ) 
        return;
    } else if ( chg == &(imreg->r->to) ) {
      if ( !_set_parameter_if_valid(&(imreg->r->to), new, new < imreg->im->rgb_height - imreg->rh, adj) )
        return;
    } else return;
  }

  _draw_rubber (imreg);
}

/**********************************************************************
 *	   image region selection internal utilities                  *
 **********************************************************************/

static GtkAdjustment*
_create_adjustment(GtkWidget *table, GtkImReg *imreg, gchar *name, gint pos, 
                           gint *value, gint max)
/* create spinners for coordinate input
 * --- Parameters: ---
 * GtkImReg *imreg : image region structure
 * gchar    *name  : name of parameter
 * gint      pos   : position
 * gint      value : initial value of parameter
 * gint      max   : maximum value of parameter
 */
{
  GtkAdjustment *adj;
  GtkWidget     *label, *spinner;
   
  label   = gtk_label_new(name);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, pos, pos+1);
  gtk_widget_show(label);

  adj     = GTK_ADJUSTMENT(gtk_adjustment_new((gfloat)(*value), 0, max, 1, 0.0, 0.0));
  spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
  gtk_widget_set_usize(spinner, 80, 0);
  gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spinner), GTK_UPDATE_ALWAYS);
  gtk_object_set_user_data(GTK_OBJECT(adj), value);
  gtk_object_set_user_data(GTK_OBJECT(spinner), imreg);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed", 
                     GTK_SIGNAL_FUNC(gtk_imreg_value_changed), spinner);
  gtk_table_attach_defaults(GTK_TABLE(table), spinner, 1, 2, pos, pos+1);
  gtk_widget_show(spinner);
  
  return adj;
}

static void 
_draw_rubber(GtkImReg *imreg)
/* Draw the rubber box */
{
  register int w, h;
   
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( imreg->r );
  if ( !(imreg->XorGc) ) return;
   
  w = imreg->rw;
  h = imreg->rh;
  if ( !imreg->old ) imreg->old = g_malloc(sizeof(_Rubber));
  else {
    if ( imreg->old->le == imreg->r->le && imreg->old->to == imreg->r->to &&
         ( ! _is_variable(imreg) || 
           ( imreg->old->ri == imreg->r->ri && imreg->old->bo == imreg->r->bo ) )
       ) return;

    if ( _is_variable(imreg) ) {
       w = imreg->old->ri - imreg->old->le;
       h = imreg->old->bo - imreg->old->to;
    }  
    gdk_draw_rectangle(imreg->cut->window, imreg->XorGc, FALSE, imreg->old->le, imreg->old->to, w, h);
    gdk_draw_rectangle(imreg->pm, imreg->XorGc, FALSE, imreg->old->le, imreg->old->to, w, h);
  }

  imreg->old->le = imreg->r->le;
  imreg->old->to = imreg->r->to;
  if ( _is_variable(imreg) ) {
    w = (imreg->old->ri = imreg->r->ri) - imreg->old->le;
    h = (imreg->old->bo = imreg->r->bo) - imreg->old->to;
  }
  
  gdk_draw_rectangle(imreg->cut->window, imreg->XorGc, FALSE, imreg->old->le, imreg->old->to, w, h);
  gdk_draw_rectangle(imreg->pm, imreg->XorGc, FALSE, imreg->old->le, imreg->old->to, w, h);

  if ( !_is_variable(imreg) ) return;
  
  g_return_if_fail ( GTK_IS_LABEL (imreg->size) );
  {
    gchar *buf;
     
    buf = g_strdup_printf("%4ix%4i", w + 1, h + 1);
    gtk_label_set(GTK_LABEL(imreg->size), buf);
    g_free(buf);
    gtk_widget_show(imreg->size);
  }
}

static void
_handle_rubber(GtkImReg *imreg, gint x, gint y, GdkModifierType state)
/* Do necessary things with a rubber box
 * --- Parameter: ---
 * GtkImReg *imreg : image region selection structure
 * gint                     x      : current x
 * gint                     y      : current y
 * GdkModifierType          state  : state (to check whether button is pressed or not)
 */
{
  gint              i, w = 0, h = 0;
  register gint     xi, yi;
  register _Rubber *r;
   
  g_return_if_fail ( imreg );
  g_return_if_fail ( GTK_IS_IMREG (imreg) );
  g_return_if_fail ( imreg->r );
  g_return_if_fail ( imreg->im );

  r = imreg->r;
  /************************************************************************************
   * Set correct adjustment values                                                    *
   ************************************************************************************/
  switch ( i = imreg->where ) {
    case INNER:
      if ( _is_variable(imreg) ) w = r->ri - r->le;
      else                       w = imreg->rw;
      if ( !(state & GDK_BUTTON1_MASK) ) break;
      if      ( (xi = x - imreg->xi) < 0 )      xi = 0;
      else if ( xi + w > imreg->im->rgb_width - 1 ) xi = imreg->im->rgb_width - 1 - w;
      gtk_adjustment_set_value(imreg->adj_le, xi);

      if ( _is_variable(imreg) ) h = r->bo - r->to;
      else                       h = imreg->rh;
      if      ( (yi = y - imreg->yi) < 0 )       yi = 0;
      else if ( yi + h > imreg->im->rgb_height - 1 ) yi = imreg->im->rgb_height - 1 - h;
      gtk_adjustment_set_value(imreg->adj_to, yi);

      if ( _is_variable(imreg) ) {
        gtk_adjustment_set_value(imreg->adj_ri, xi + w);
        gtk_adjustment_set_value(imreg->adj_bo, yi + h);
      }
      break;

    case OUT: 
      if ( !_is_variable(imreg) ) {
        _set_rubber_corner(x, &(r->le), NULL, imreg->rw, imreg->adj_le, NULL);
        _set_rubber_corner(y, &(r->to), NULL, imreg->rh, imreg->adj_to, NULL);
      }
      _set_cursor_rel(imreg->cut, imreg->where = _where(imreg, x, y));
      break;

    case TOP:
      if ( !(state & GDK_BUTTON1_MASK) || y < 0 ) break;
      gtk_adjustment_set_value(imreg->adj_to, y);
      break;

    case BOTTOM:
      if ( !(state & GDK_BUTTON1_MASK) || y >= imreg->im->rgb_height ) break;
      if ( _is_variable(imreg) ) gtk_adjustment_set_value(imreg->adj_bo, y);
      else                       gtk_adjustment_set_value(imreg->adj_to, y - imreg->rh);
      break;

    case LEFT:        
      if ( !(state & GDK_BUTTON1_MASK) || x < 0 ) break;
      gtk_adjustment_set_value(imreg->adj_le, x);
      break;

    case RIGHT:
      if ( !(state & GDK_BUTTON1_MASK) || x >= imreg->im->rgb_width ) break;
      if ( _is_variable(imreg) ) gtk_adjustment_set_value(imreg->adj_ri, x);
      else                       gtk_adjustment_set_value(imreg->adj_le, x - imreg->rw);
      break;

    case LEFTTOP: 
      if ( _is_variable(imreg) ) {
	i = CORNER;
        break;
      }
      if ( !(state & GDK_BUTTON1_MASK) || x < 0 || y < 0 ) break;
      gtk_adjustment_set_value(imreg->adj_le, x);
      gtk_adjustment_set_value(imreg->adj_to, y);
      break;

    case RIGHTBOTTOM: 
      if ( _is_variable(imreg) ) {
	i = CORNER;
        break;
      }
      if ( !(state & GDK_BUTTON1_MASK) || 
           x >= imreg->im->rgb_width || y >= imreg->im->rgb_height ) break;
      gtk_adjustment_set_value(imreg->adj_le, x - imreg->rw);
      gtk_adjustment_set_value(imreg->adj_to, y - imreg->rh);
      break;

    case RIGHTTOP:
      if ( _is_variable(imreg) ) {
	i = CORNER;
        break;
      }
      if ( !(state & GDK_BUTTON1_MASK) || x >= imreg->im->rgb_width || y < 0 ) break;
      gtk_adjustment_set_value(imreg->adj_le, x - imreg->rw);
      gtk_adjustment_set_value(imreg->adj_to, y);
      break;

    case LEFTBOTTOM:
      if ( _is_variable(imreg) ) {
	i = CORNER;
        break;
      }
      if ( !(state & GDK_BUTTON1_MASK) || x < 0 || y >= imreg->im->rgb_height ) break;
      gtk_adjustment_set_value(imreg->adj_le, x);
      gtk_adjustment_set_value(imreg->adj_to, y - imreg->rh);
      break;

    case OUTSIDE: 
    case NOWHERE:     break;
   }

  if ( ((state & GDK_BUTTON1_MASK) && (i == OUT || i == CORNER)) || (state & GDK_BUTTON3_MASK) ) {
    if ( _is_variable(imreg) ) {
      _set_rubber_corner(x, &(r->le), &(r->ri), 0, imreg->adj_le, imreg->adj_ri);
      _set_rubber_corner(y, &(r->to), &(r->bo), 0, imreg->adj_to, imreg->adj_bo);
    }
  }

  if ( (state & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) ) 
    _draw_rubber (imreg);
}

static void 
_set_cursor(GdkWindow *window, gint type)
{
  GdkCursor *cursor;

  g_return_if_fail ( window );
  
  cursor = gdk_cursor_new(type);
  gdk_window_set_cursor(window, cursor);
  gdk_cursor_destroy(cursor);
}

static void 
_set_cursor_rel(GtkWidget *widget, _CursorLocation pos)
{
  g_return_if_fail ( widget );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );

  switch ( pos ) {
    case INNER:       _set_cursor(widget->window, GDK_FLEUR);
                      break;
    case OUT:         _set_cursor(widget->window, GDK_CROSSHAIR);
                      break;
    case TOP:         _set_cursor(widget->window, GDK_SB_V_DOUBLE_ARROW);
                      break;
    case BOTTOM:      _set_cursor(widget->window, GDK_SB_V_DOUBLE_ARROW);
                      break;
    case LEFT:        _set_cursor(widget->window, GDK_SB_H_DOUBLE_ARROW);
                      break;
    case RIGHT:       _set_cursor(widget->window, GDK_SB_H_DOUBLE_ARROW);
                      break;
    case LEFTTOP:     _set_cursor(widget->window, GDK_TOP_LEFT_CORNER);
                      break;
    case RIGHTBOTTOM: _set_cursor(widget->window, GDK_BOTTOM_RIGHT_CORNER);
                      break;
    case RIGHTTOP:    _set_cursor(widget->window, GDK_TOP_RIGHT_CORNER);
                      break;
    case LEFTBOTTOM:  _set_cursor(widget->window, GDK_BOTTOM_LEFT_CORNER);
                      break;
    case OUTSIDE:
    case NOWHERE:     _set_cursor(widget->window, GDK_DOT);
  }
}

static gint 
_set_parameter_if_valid(gint *val2set, gint val, gint cond, GtkAdjustment *adj)
/* Set value in address if condition is true
 */
{
  g_return_val_if_fail ( adj , FALSE );
  g_return_val_if_fail ( GTK_IS_ADJUSTMENT (adj), FALSE );

  if ( cond ) {
    *val2set = val;
    return TRUE;
  }
  if ( *val2set == val ) return FALSE;  /* it's not necessary to set the value */
  gdk_beep();
  gtk_adjustment_set_value(adj, *val2set);
  return FALSE;
}

static void 
_set_rubber_corner(gint x, gint *x1, gint *x2, gint len, GtkAdjustment *adj1, GtkAdjustment *adj2)
/* set the nearest corner to click (default behaviour for right mouse button)
 * --- Parameter: ---
 * gint           x   : current x or y position
 * gint          *x1  : pointer to old left or top value
 * gint          *x2  : pointer to old right or bottom value
 * gint           len : if len == 0 then variable box, else fixed box of width/height len
 * GtkAdjustment *adj1: adjustment for *x1
 * GtkAdjustment *adj2: adjustment for *x2
 * --- Return: ---
 * gint *x1           : new left or top value
 * gint *x2           : new right or bottom value
 * GtkAdjustment *adj1: adjusted for *x1 which caused resizing of box via signal
 * GtkAdjustment *adj2: adjusted for *x2 which caused resizing of box via signal
 */
{
  g_return_if_fail ( GTK_IS_ADJUSTMENT (adj1) );

  if ( x <= *x1 ) { /* left or top */
    if ( x < 0 ) x = 0 ;
    gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x)); 
    return;
  }
  if ( !len ) {
    g_return_if_fail ( *x2 );
    g_return_if_fail ( GTK_IS_ADJUSTMENT (adj2) );

    if ( x >= *x2 ) { /* right or bottom */
      if ( x > adj2->upper ) x = adj2->upper ; 
      gtk_adjustment_set_value(adj2, (gfloat)(*x2 = x)); 
      return;
    }
    if ( x - *x1 < *x2 - x ) { /* inner, but near to left or top */
      gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x)); 
      return;
    }
    gtk_adjustment_set_value(adj2, (gfloat)(*x2 = x)); /* inner, but near to right or bottom */
    return;
  }
  if ( x >= *x1 + len ) { /* right or bottom */
    gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x - len)); 
    return;
  }
  if ( x - *x1 < len >> 1 ) { /* inner, but near to left or top */
    gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x)); 
    return;
  }
  gtk_adjustment_set_value(adj1, (gfloat)(*x1 = x - len)); /* inner, but near to right or bottom */
}

static _CursorLocation
_where(GtkImReg *imreg, gint x, gint y)
/* obtain cursor position relative to the selected rubber box
 */
{
  register _Rubber *r;
  register gint    ri, bo;

  g_return_val_if_fail ( imreg, NOWHERE );
  g_return_val_if_fail ( GTK_IS_IMREG (imreg), NOWHERE );
  g_return_val_if_fail ( imreg->im, NOWHERE );

  if ( !imreg->old ) return OUT;
  r = imreg->old;
  
  if ( x < 0 || y < 0 || x >= imreg->im->rgb_width || y >= imreg->im->rgb_height ) return OUTSIDE;

  if ( _is_variable(imreg) ) {
    ri = r->ri;
    bo = r->bo;
  } else {
    ri = r->le + imreg->rw;
    bo = r->to + imreg->rh;
  }

  if ( r->le + FRAME <= x && x <= ri - FRAME &&
        r->to + FRAME <= y && y <= bo - FRAME ) return INNER;
   if ( r->le - FRAME <= x && x <= r->le + FRAME ) {
      if ( r->to - FRAME <= y && y <= r->to + FRAME ) return LEFTTOP;
      if ( bo - FRAME <= y && y <= bo + FRAME ) return LEFTBOTTOM;
      if ( r->to + FRAME <  y && y <  bo + FRAME ) return LEFT;
      return OUT;
   }
   if ( ri - FRAME <= x && x <= ri + FRAME ) {
      if ( r->to - FRAME <= y && y <= r->to + FRAME ) return RIGHTTOP;
      if ( bo - FRAME <= y && y <= bo + FRAME ) return RIGHTBOTTOM;
      if ( r->to + FRAME <  y && y <  bo + FRAME ) return RIGHT;
      return OUT;
   }
   if ( r->to - FRAME <= y && y <= r->to + FRAME &&
        r->le + FRAME <  x && x <  ri + FRAME ) return TOP;
   if ( bo - FRAME <= y && y <= bo + FRAME &&
        r->le + FRAME <  x && x <  ri + FRAME ) return BOTTOM;
   if ( r->le + 2 > x || x > ri - 2 ||
        r->to + 2 > y || y > bo - 2 ) return OUT;
   return NOWHERE;
}







