/*
 * Copyright (C) 2003, 2004 Laurent Sansonetti <lrz@gnome.org>
 *
 * This file is part of Ruby/GtkTrayIcon.
 *
 * Ruby/GtkTrayIcon is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GtkTrayIcon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GtkTrayIcon; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgtk.h"
#include "eggtrayicon.h"

/*
 * Class: Gtk::TrayIcon
 *
 * This class provides a Ruby access to the System Tray Protocol
 * Specification.
 * The class is based on Gtk::Plug, so you can easily add widgets
 * in the GNOME notification area using Gtk::Plug#add.
 * Here is an example:
 *
 *	require 'gtktrayicon'
 *
 *	Gtk.init
 *
 *	tray = Gtk::TrayIcon.new("test")
 *	tray.add(Gtk::Label.new("Hello, World!"))
 *	tray.show_all
 *
 *	Gtk.main
 */
#define _SELF(s) (EGG_TRAY_ICON (RVAL2GOBJ (s)))

/*
 * Class method: new(name, screen=nil)
 * name: the name of the tray icon.
 * screen: a Gdk::Screen (can be nil).
 *
 * Constructs a new Gtk::TrayIcon object using the given parameters.
 * 
 * Returns: a newly created Gtk::TrayIcon object.
 */
static VALUE 
ti_initialize (int argc, VALUE *argv, VALUE self)
{
	VALUE name, screen;
	EggTrayIcon *ti;

	rb_scan_args (argc, argv, "11", &name, &screen);
	if (NIL_P (screen))
		ti = egg_tray_icon_new (RVAL2CSTR (name));
	else
		ti = egg_tray_icon_new_for_screen (GDK_SCREEN (RVAL2GOBJ (screen)),
						   RVAL2CSTR (name));
	if (ti != NULL)
		RBGTK_INITIALIZE (self, ti);

	return Qnil;
}

/*
 * Method: send_message(timeout, message)
 * timeout: a timeout value (Integer).
 * message: a message to send (String).
 *
 * Asks the tray to display the given message for the given timeout milliseconds.
 * If timeout is 0, the message will not expire.
 *
 * Returns: an Integer identifier for the message.
 */
static VALUE
ti_send_message (VALUE self, VALUE timeout, VALUE message)
{
	return INT2NUM (egg_tray_icon_send_message (_SELF (self),
						    NUM2INT (timeout),
						    RVAL2CSTR (message),
						    RSTRING (message)->len));
}

/*
 * Method: cancel_message(id)
 * id: a message ID.
 *
 * Cancels a message which is currently displayed on the tray, giving its
 * ID.
 *
 * Returns: self.
 */
static VALUE
ti_cancel_message (VALUE self, VALUE id)
{
	egg_tray_icon_cancel_message (_SELF (self), NUM2INT (id));
	return self;
}

/* Method: orientation
 * Returns: the orientation of the tray, as a Gdk::Orientation. 
 */
static VALUE
ti_get_orientation (VALUE self)
{
	return GENUM2RVAL (egg_tray_icon_get_orientation (_SELF (self)),
			   GTK_TYPE_ORIENTATION);
}

void
Init_gtktrayicon (void)
{
	VALUE eggTrayIcon = G_DEF_CLASS (EGG_TYPE_TRAY_ICON, "TrayIcon", mGtk);
	
	rb_define_method (eggTrayIcon, "initialize", ti_initialize, -1);
	rb_define_method (eggTrayIcon, "send_message", ti_send_message, 2);
	rb_define_method (eggTrayIcon, "cancel_message", ti_cancel_message, 1);
	rb_define_method (eggTrayIcon, "orientation", ti_get_orientation, 0);
}
