#include <ept/tests/test-utils.h>
#include <ept/cache/debtags/update.h>
#include <ept/cache/debtags/vocabularymerger.h>
#include <ept/path.h>

#include <tagcoll/stream/sink.h>


namespace tut {

using namespace ept;
using namespace ept::configuration::apt;

using namespace t::cache::debtags;

struct cache_debtags_update_shar : common {
	typedef ept::configuration::Apt C;
	cache::Aggregator agg;
};
TESTGRP(cache_debtags_update);

// RAII-style class to ensure cleanup of a temporary test file
class TempTestFile
{
	std::string pathname;
public:
	TempTestFile(const std::string& pathname) : pathname(pathname) { unlink(pathname.c_str()); }
	~TempTestFile() { unlink(pathname.c_str()); }
};

struct DebtagsTempTestFiles
{
	TempTestFile tf1;
	TempTestFile tf2;
	TempTestFile tf3;
	TempTestFile tf4;
	TempTestFile tf5;
	TempTestFile tf6;
	TempTestFile tf7;
	TempTestFile tf8;
	TempTestFile tf9;
	TempTestFile tfA;

	DebtagsTempTestFiles()
		: tf1(Path::vocabulary()), tf2(Path::vocabularyIndex()),
		  tf3(Path::userVocabulary()), tf4(Path::userVocabularyIndex()),
		  tf5(Path::tagdb()), tf6(Path::tagdbIndex()),
		  tf7(Path::userTagdb()), tf8(Path::userTagdbIndex()),
		  tf9(Path::pkgidx()), tfA(Path::userPkgidx()) {}
};

static bool isEmpty(const std::string& dir)
{
	Directory d(dir);
	for (Directory::const_iterator i = d.begin(); i != d.end(); ++i)
	{
		std::string n(i->d_name);
		if (n != "." && n != "..")
			return false;
	}
	return true;
}

//#include <iostream>

// Test Directory
template<> template<>
void to::test< 1 >()
{
	Directory dir(Path::debtagsSourceDir());

	ensure_equals(Path::debtagsSourceDir(), dir.path());
	ensure(dir.valid());

	//std::cerr << dir.path() << std::endl;

	std::set<string> files;
	Directory::const_iterator i = dir.begin();
	ensure(i != dir.end());
	for ( ; i != dir.end(); ++i)
	{
		ensure(i->d_name[0] != 0);
		//std::cerr << "FILE:" << i->d_name << std::endl;
		files.insert(i->d_name);
	}
	
	ensure(files.size() > 4);
	ensure(files.find(".") != files.end());
	ensure(files.find("..") != files.end());
	ensure(files.find("test.voc") != files.end());
	ensure(files.find("test.tag") != files.end());
}

// Check that valid() works on nonexisting directories and non directories
template<> template<>
void to::test< 2 >()
{
	Directory dir1(Path::debtagsSourceDir() + "/does-not-exist");
	ensure(!dir1.valid());

	Directory dir2(Path::debtagsSourceDir() + "/test.voc");
	ensure(!dir2.valid());
}

// Test SourceDir
template<> template<>
void to::test< 3 >()
{
	SourceDir main = SourceDir(Path::debtagsSourceDir());

	// Check that the timestamps/files are properly found
	ensure(main.timestamp() > 0);
	ensure(main.tagTimestamp() > 0);
	ensure(main.vocTimestamp() > 0);

	// Check that it can properly read the tags
	int items = 0, tags = 0;
	main.readTags(tagcoll::stream::countingSink(items, tags));
	ensure(items > 10);
	ensure(tags > 10);

	VocabularyMerger voc;
	main.readVocabularies(voc);
	ensure(!voc.empty());
}

// Timestamps should return 0 on nonexisting directories
template<> template<>
void to::test< 4 >()
{
	SourceDir main = SourceDir(Path::debtagsSourceDir() + "/does-not-exist");

	// Check that the timestamps/files are 0
	ensure_equals(main.timestamp(), 0);
	ensure_equals(main.tagTimestamp(), 0);
	ensure_equals(main.vocTimestamp(), 0);
}

// If sources are missing, instantiating indexes should fail
template<> template<>
void to::test< 5 >()
{
	using namespace std;

	Path::OverrideDebtagsSourceDir o1(testDebtagsCacheDir + "empty/");
	Path::OverrideDebtagsUserSourceDir o2(testDebtagsCacheDir + "empty/");

	DebtagsTempTestFiles tf;

	ensure(isEmpty(Path::debtagsSourceDir()));
	ensure(isEmpty(Path::debtagsUserSourceDir()));

	string a, b;

	IndexManager<>::Vocabulary v;
	ensure_equals(v.sourceTimestamp(), 0);
	ensure_equals(v.needsRebuild(), true);
	ensure_equals(v.getUpToDateVocabulary(a, b), false);
	ensure_equals(v.userIndexIsRedundant(), false);

	bool succeeded;

	try {
		// This should throw an exception
		IndexManager<>::obtainWorkingVocabulary(a, b);
		succeeded = true;
	} catch (wibble::exception::Consistency& e) {
		succeeded = false;
		//cerr << e.what() << endl;
	}
	ensure(!succeeded);
	ensure(isEmpty(Path::debtagsSourceDir()));
	ensure(isEmpty(Path::debtagsUserSourceDir()));

	IndexManager<>::Tagdb<cache_debtags_update_shar::C> t(agg);
	// There is always the APT database as source index
	ensure(t.sourceTimestamp() != 0);
	ensure_equals(t.needsRebuild(), true);
	ensure_equals(t.getUpToDateTagdb(a, b), false);
	ensure_equals(t.userIndexIsRedundant(), false);
	try {
		// This should throw an exception
		IndexManager<>::obtainWorkingTagdb<cache_debtags_update_shar::C>(agg, a, b);
		succeeded = true;
	} catch (wibble::exception::Consistency& e) {
		succeeded = false;
		//cerr << e.what() << endl;
	}
	ensure(!succeeded);
	ensure(isEmpty(Path::debtagsSourceDir()));
	ensure(isEmpty(Path::debtagsUserSourceDir()));
}

// Test creating pkgidx indexes
template<> template<>
void to::test< 6 >()
{
	// Empty RC directory, writable system directory
	Path::OverrideDebtagsUserSourceDir o1(testDebtagsCacheDir + "empty/");
	DebtagsTempTestFiles tf;

	std::string a;

	IndexManager<>::Pkgidx<cache_debtags_update_shar::C> p(agg);
	ensure(p.sourceTimestamp() > 0);
	ensure_equals(p.needsRebuild(), true);
	ensure_equals(p.getUpToDatePkgidx(a), false);
	ensure_equals(p.userIndexIsRedundant(), false);

	IndexManager<>::obtainWorkingPkgidx<cache_debtags_update_shar::C>(agg, a);

	ensure_equals(a, Path::pkgidx());

	// There is always APT as a data source for this filter
	ensure(p.sourceTimestamp() != 0);
	ensure_equals(p.needsRebuild(), true);
	ensure_equals(p.getUpToDatePkgidx(a), false);
	ensure_equals(p.userIndexIsRedundant(), false);

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(Path::userPkgidx(), F_OK) != 0);
}
template<> template<>
void to::test< 7 >()
{
	// Empty RC directory, nonwritable/nonexisting system directory
	Path::OverrideDebtagsUserSourceDir o1(testDebtagsCacheDir + "empty/");
	Path::OverrideDebtagsIndexDir o2(testDebtagsCacheDir + "does-not-exist/");
	DebtagsTempTestFiles tf;

	std::string a;
	IndexManager<>::obtainWorkingPkgidx<cache_debtags_update_shar::C>(agg, a);

	ensure_equals(a, Path::userPkgidx());

	ensure(Path::access(Path::pkgidx(), F_OK) != 0);
	ensure(Path::access(a, R_OK) == 0);
}

// Test creating vocabulary indexes
template<> template<>
void to::test< 8 >()
{
	// No user sources, writable system directory
	DebtagsTempTestFiles tf;

	std::string a, b;

	IndexManager<>::Vocabulary v;
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), true);
	ensure_equals(v.getUpToDateVocabulary(a, b), false);
	ensure_equals(v.userIndexIsRedundant(), false);

	IndexManager<>::obtainWorkingVocabulary(a, b);

	ensure_equals(a, Path::vocabulary());
	ensure_equals(b, Path::vocabularyIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::userVocabulary(), F_OK) != 0);
	ensure(Path::access(Path::userVocabularyIndex(), F_OK) != 0);

	v.rescan();
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), false);
	ensure_equals(v.getUpToDateVocabulary(a, b), true);
	ensure_equals(a, Path::vocabulary());
	ensure_equals(b, Path::vocabularyIndex());
	ensure_equals(v.userIndexIsRedundant(), false);
}
template<> template<>
void to::test< 9 >()
{
	// No user sources, nonwritable system directory
	Path::OverrideDebtagsIndexDir o1(testDebtagsCacheDir + "does-not-exist/");
	DebtagsTempTestFiles tf;

	std::string a, b;

	IndexManager<>::Vocabulary v;
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), true);
	ensure_equals(v.getUpToDateVocabulary(a, b), false);
	ensure_equals(v.userIndexIsRedundant(), false);

	IndexManager<>::obtainWorkingVocabulary(a, b);

	ensure_equals(a, Path::userVocabulary());
	ensure_equals(b, Path::userVocabularyIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::vocabulary(), F_OK) != 0);
	ensure(Path::access(Path::vocabularyIndex(), F_OK) != 0);

	v.rescan();
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), false);
	ensure_equals(v.getUpToDateVocabulary(a, b), true);
	ensure_equals(a, Path::userVocabulary());
	ensure_equals(b, Path::userVocabularyIndex());
	ensure_equals(v.userIndexIsRedundant(), false);
}
template<> template<>
void to::test< 10 >()
{
	// User and system sources, writable system directory, should create user index
	Path::OverrideDebtagsUserSourceDir o1(testDebtagsCacheDir);
	DebtagsTempTestFiles tf;

	std::string a, b;

	IndexManager<>::Vocabulary v;
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), true);
	ensure_equals(v.getUpToDateVocabulary(a, b), false);
	ensure_equals(v.userIndexIsRedundant(), false);

	IndexManager<>::obtainWorkingVocabulary(a, b);

	ensure_equals(a, Path::userVocabulary());
	ensure_equals(b, Path::userVocabularyIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::vocabulary(), F_OK) != 0);
	ensure(Path::access(Path::vocabularyIndex(), F_OK) != 0);

	v.rescan();
	ensure(v.sourceTimestamp() > 0);
	ensure_equals(v.needsRebuild(), false);
	ensure_equals(v.getUpToDateVocabulary(a, b), true);
	ensure_equals(a, Path::userVocabulary());
	ensure_equals(b, Path::userVocabularyIndex());
	ensure_equals(v.userIndexIsRedundant(), false);
}


// Test creating tag indexes
template<> template<>
void to::test< 11 >()
{
	// No user sources, writable system directory
	TempTestFile dl1(Path::tagdb());
	TempTestFile dl2(Path::tagdbIndex());

	std::string a, b;
	IndexManager<>::obtainWorkingTagdb<cache_debtags_update_shar::C>(agg, a, b);

	ensure_equals(a, Path::tagdb());
	ensure_equals(b, Path::tagdbIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::userTagdb(), F_OK) != 0);
	ensure(Path::access(Path::userTagdbIndex(), F_OK) != 0);
}
template<> template<>
void to::test< 12 >()
{
	// No user sources, nonwritable system directory
	Path::OverrideDebtagsIndexDir o1(testDebtagsCacheDir + "does-not-exist/");
	TempTestFile dl1(Path::userTagdb());
	TempTestFile dl2(Path::userTagdbIndex());

	std::string a, b;
	IndexManager<>::obtainWorkingTagdb<cache_debtags_update_shar::C>(agg, a, b);

	ensure_equals(a, Path::userTagdb());
	ensure_equals(b, Path::userTagdbIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::tagdb(), F_OK) != 0);
	ensure(Path::access(Path::tagdbIndex(), F_OK) != 0);
}
template<> template<>
void to::test< 13 >()
{
	// User and system sources, writable system directory, should create user index
	Path::OverrideDebtagsUserSourceDir o1(testDebtagsCacheDir);
	TempTestFile dl1(Path::userTagdb());
	TempTestFile dl2(Path::userTagdbIndex());

	std::string a, b;
	IndexManager<>::obtainWorkingTagdb<cache_debtags_update_shar::C>(agg, a, b);

	ensure_equals(a, Path::userTagdb());
	ensure_equals(b, Path::userTagdbIndex());

	ensure(Path::access(a, R_OK) == 0);
	ensure(Path::access(b, R_OK) == 0);
	ensure(Path::access(Path::tagdb(), F_OK) != 0);
	ensure(Path::access(Path::tagdbIndex(), F_OK) != 0);
}

#if 0
	using namespace std;

	OverrideRcdir o2(Path::stateDir() + "empty/");
	OverrideStateDir o1(Path::stateDir() + "empty/");

	ensure(isEmpty(Path::stateDir()));
	ensure(isEmpty(Path::rcdir()));

	string a, b;
	bool succeeded;

	try {
		// This should throw an exception
		IndexManager<>::obtainWorkingVocabulary(a, b);
		succeeded = true;
	} catch (wibble::exception::Consistency& e) {
		succeeded = false;
		//cerr << e.what() << endl;
	}
	ensure(!succeeded);
	ensure(isEmpty(Path::stateDir()));
	ensure(isEmpty(Path::rcdir()));

	try {
		// This should throw an exception
		IndexManager<>::obtainWorkingTagmap<cache_debtags_update_shar::C>(agg, a, b);
		succeeded = true;
	} catch (wibble::exception::Consistency& e) {
		succeeded = false;
		//cerr << e.what() << endl;
	}
	ensure(!succeeded);
	ensure(isEmpty(Path::stateDir()));
	ensure(isEmpty(Path::rcdir()));
}
#endif


}

#include <tagcoll/TextFormat.tcc>
#include <ept/cache/debtags/update.tcc>

// vim:set ts=4 sw=4:
