// -*- mode: c++; tab-width: 4; indent-tabs-mode: t -*-
/**
 * @file cache/component/tagmap.cpp
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

/*
 * System tag database
 *
 * Copyright (C) 2003-2006  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <ept/tests/test-utils.h>
#include <ept/cache/debtags/tagmap.h>

#include <ept/cache/cache.h>
#include <ept/cache/tag.h>
#include <ept/cache/debtags/vocabulary.h>
#include <tagcoll/utils/set.h>
#include <tagcoll/stream/sink.h>

using namespace tagcoll;
using namespace ept;
using namespace ept::configuration::apt;
using namespace wibble::operators;
using namespace cache;

namespace tut {

struct cache_debtags_tagmap_shar : common {
	Index& index() { return apt.index(); }
    TagMap& pt() { return apt.tagMap(); }
    Vocabulary& voc() { return apt.vocabulary(); }
};

TESTGRP( cache_debtags_tagmap );

template<> template<>
void to::test<1>()
{
    for (TagMap::const_iterator i = pt().begin(); i != pt().end(); ++i)
	{
		*i;
		i->first;
		i->second;
	}
	int items = 0, tags = 0;
	pt().outputSystem(stream::countingSink(items, tags));

	int pitems = 0, ptags = 0;
	pt().outputPatched(stream::countingSink(pitems, ptags));

	ensure(items > 10);
	ensure(tags > 10);
	ensure(items <= pitems);
	ensure(tags <= ptags);
}

template<> template<>
void to::test<2>()
{
    Package p = index().packageByName( "debtags" );
    ensure(p.valid());
	std::set<Tag> tags = pt().getTagsOfItem(p);
	ensure( !tags.empty() );

	for ( std::set< Tag >::iterator i = tags.begin(); i != tags.end(); ++ i ) {
		std::cerr << i->id() << ": " << i->fullname() << std::endl;
	}
	std::cerr << "---" << std::endl;
	Tag t = voc().tagByName( "interface::commandline" );
	std::cerr << t.id() << ": " << t.fullname() << std::endl;

	ensure_equals( tags.size(), 5u );
	ensure( voc().tagByName( "interface::commandline" ) <= tags );
	ensure( voc().tagByName( "made-of::lang:c++" ) <= tags );
	ensure( voc().tagByName( "suite::debian" ) <= tags );
	ensure( voc().tagByName( "use::searching" ) <= tags );
	ensure( voc().tagByName( "works-with::software:package" ) <= tags );
}

#include <iostream>

void ppset(const std::set<Package>& pkgs)
{
	using namespace std;
	for (std::set<Package>::const_iterator i = pkgs.begin(); i != pkgs.end(); ++i)
		if (i == pkgs.begin())
			cerr << i->name() << ":" << i->ondiskId();
		else
			cerr << ", " << i->name() << ":" << i->ondiskId();
}
void piset(const std::set<int>& ints)
{
	using namespace std;
	for (std::set<int>::const_iterator i = ints.begin(); i != ints.end(); ++i)
		if (i == ints.begin())
			cerr << *i;
		else
			cerr << ", " << *i;
}

template<> template<>
void to::test<3>()
{
	using namespace std;

    /* Get the 'debtags' package */
    Package p = index().packageByName( "debtags" );
    ensure(p.valid());

    /* Get its tags */
	std::set<Tag> tags = pt().getTagsOfItem(p);
    ensure(!tags.empty());

	/*
	cerr << "Intersection size: " << endl;
	using namespace wibble::operators;
	std::set<Tag>::const_iterator dbgi = tags.begin();
	cerr << "* " << dbgi->fullname() << ": " << dbgi->id() << endl;
	std::set<int> dbgres = pt().tagdb().getItemsHavingTag(dbgi->id());
	std::set<Package> dbgpres = pt().getItemsHavingTag(*dbgi);
	cerr << " #pkgs " << dbgres.size() << " == " << dbgpres.size() << endl;
	cerr << " #isec " << dbgres.size() << " == " << dbgpres.size() << endl;
	cerr << "  "; ppset(dbgpres); cerr << endl;
	cerr << "  "; piset(dbgres); cerr << endl;
	for (++dbgi ; dbgi != tags.end(); ++dbgi)
	{
		cerr << "* " << dbgi->fullname() << ": " << dbgi->id() << endl;
		std::set<Package> dbgpkgs = pt().getItemsHavingTag(*dbgi);
		std::set<int> dbgids = pt().tagdb().getItemsHavingTag(dbgi->id());
		cerr << "  "; ppset(dbgpkgs); cerr << endl;
		cerr << "  "; piset(dbgids); cerr << endl;
		cerr << " #pkgs " << dbgpkgs.size() << " == " << dbgids.size() << endl;
		dbgres &= dbgids;
		dbgpres &= dbgpkgs;
		cerr << " #isec " << dbgres.size() << " == " << dbgpres.size() << endl;
	}
	cerr << " " << dbgres.size() << endl << "Results: " << endl;
	for (std::set<int>::const_iterator i = dbgres.begin(); i != dbgres.end(); ++i)
		cerr << "   " << *i << endl;
	*/


//	cerr << "Tags of debtags: ";
//	for (std::set<Tag>::const_iterator i = tags.begin(); i != tags.end(); ++i)
//	{
//		cerr << " " + i->fullname() << endl;
//		std::set<Package> packages = pt().getItemsHavingTag(*i);
//		for (std::set<Package>::const_iterator p = packages.begin();
//				p != packages.end(); ++p)
//			cerr << "   PKG " << p->name() << endl;
//	}
//	cerr << endl;

    /* Get the items for the tagset of 'debtags' */
	std::set<Package> packages = pt().getItemsHavingTags(tags);
	cerr << packages.size() << endl;
    ensure(!packages.empty());
	for ( std::set< Package >::iterator i = packages.begin(); i != packages.end(); ++ i )
		std::cerr << i->name() << std::endl;
	std::cerr << "---" << std::endl;
	std::cerr << p.name() << std::endl;
    /* They should at least contain 'debtags' */
    ensure( p <= packages );

    /* Get one of the tags of 'debtags' */
    Tag tag = *tags.begin();
    ensure(tag);

    /* Get its items */
    {
        /* Need this workaround until I figure out how to tell the new GCC
         * that TagDB is a TDBReadonlyDiskIndex and should behave as such
         */
		std::set<Tag> ts;
		ts.insert(tag);
        packages = pt().getItemsHavingTags(ts);
    }
    //packages = c.debtags().tagdb().getItems(tag);
    ensure(!packages.empty());
    /* They should at least contain 'debtags' */
    ensure( p <= packages );

    //c.debtags().getTags(""); // XXX HACK AWW!
}

template<> template<>
void to::test<4>()
{
	std::string patchfile = Path::debtagsUserSourceDir() + "patch";
	unlink(patchfile.c_str());

	Package p = index().packageByName( "debtags" );
	ensure(p.valid());

    /* Get its tags */
	std::set<Tag> tags = pt().getTagsOfItem(p);
    ensure(!tags.empty());

	// Ensure that it's not tagged with gameplaying
	Tag t = voc().tagByName("use::gameplaying");
    ensure(! (t <= tags) );

	// Add the gameplaying tag
	PatchList<Package, Tag> change;
	change.addPatch(Patch<Package, Tag>(p, wibble::singleton(t), wibble::Empty<Tag>()));
	pt().applyChange(change);

	// See that the patch is non empty
	PatchList<Package, Tag> tmp = pt().changes();
	ensure(tmp.size() > 0);
	ensure_equals(tmp.size(), 1u);

	// Ensure that the tag has been added
	tags = pt().getTagsOfItem(p);
    ensure(!tags.empty());

	t = voc().tagByName("use::gameplaying");
    ensure( t <= tags );

	// Save the patch
	pt().savePatch();

	// Check that the saved patch is correct
	FILE* in = fopen(patchfile.c_str(), "r");
	string writtenPatch;
	int c;
	while ((c = getc(in)) != EOF)
		writtenPatch += c;
	fclose(in);

	ensure_equals(writtenPatch, string("debtags: +use::gameplaying\n"));

	unlink(patchfile.c_str());

	// Reapply the patch and see that it doesn't disrupt things
	pt().applyChange(change);

	// The patch should not have changed
	tmp = pt().changes();
	ensure_equals(tmp.size(), 1u);
	ensure_equals(tmp.begin()->first.id(), p.id());
	ensure_equals(tmp.begin()->second.item.id(), p.id());
}

}

#include <ept/cache/debtags/tagmap.tcc>
#include <ept/cache/debtags/vocabulary.tcc>

// vim:set ts=4 sw=4:

