/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

/*
 * mod_iptos: set IP type-of-service bits according to directory
 *
 * v1.1
 * 
 * author: dean gaudet <dean@arctic.org>
 *
 * 0.1: initial implementation
 * 0.2: added keyword "none"
 * 1.0: what the heck, it's official
 * 1.1: support IPTOSthreshold; support integer TOS specifications
 */

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_request.h"

#include <netinet/ip.h>

// byte me apache, i don't care about whether sunos4 has strtoul or not!
#undef strtoul

typedef uint8_t iptos_t;

typedef struct {
	iptos_t tos;
	iptos_t large_tos;
	uint8_t tos_set : 1;
	uint8_t large_tos_set : 1;
	off_t large_threshold;
} iptos_config;

module MODULE_VAR_EXPORT iptos_module;


static const char *iptos_parse_iptos(char *word, iptos_t *ptos)
{
	iptos_t tos;

	if (!*word) {
usage:
		return "unrecognized TOS.  use none, lowdelay, throughput, reliability, lowcost, or an integer (0x supported).";
	}
	if (strcasecmp(word, "lowdelay") == 0) {
		tos = IPTOS_LOWDELAY;
	}
	else if (strcasecmp(word, "throughput") == 0) {
		tos = IPTOS_THROUGHPUT;
	}
	else if (strcasecmp(word, "reliability") == 0) {
		tos = IPTOS_RELIABILITY;
	}
	else if (strcasecmp(word, "lowcost") == 0) {
		tos = IPTOS_LOWCOST;
	}
	else if (strcasecmp(word, "none") == 0) {
		tos = 0;
	}
	else if (isdigit(word[0])) {
		tos = strtoul(word, 0, 0);
	}
	else {
		goto usage;
	}
	*ptos = tos;
	return NULL;
}


static const char *iptos_cmd_iptos(cmd_parms *cmd, void *mconfig, char *word)
{
	iptos_config *config = mconfig;

	config->tos_set = 1;
	return iptos_parse_iptos(word, &config->tos);
}


static const char *iptos_cmd_iptos_threshold(cmd_parms *cmd, void *mconfig, char *bytes, char *word)
{
	iptos_config *config = mconfig;

	config->large_tos_set = 1;
	config->large_threshold = strtoul(bytes, 0, 0);
	return iptos_parse_iptos(word, &config->large_tos);
}


static void *iptos_create_dir_config(pool *p, char *dirspec)
{
	iptos_config *config;

	config = ap_pcalloc(p, sizeof(iptos_config));
	memset(config, 0, sizeof(config));
	return config;
}


static void *iptos_merge_dir_config(pool *p, void *basev, void *overridesv)
{
	iptos_config *config = ap_pcalloc(p, sizeof(iptos_config));
	iptos_config *overrides = (iptos_config *)overridesv;

	*config = *(iptos_config *)basev;
	if (overrides->tos_set) {
		config->tos = overrides->tos;
	}
	if (overrides->large_tos_set) {
		config->large_tos = overrides->large_tos;
		config->large_threshold = overrides->large_threshold;
	}
	return config;
}



static void iptos_set_tos(request_rec *r, iptos_t _tos)
{
	unsigned tos = _tos;
	if (setsockopt(r->connection->client->fd, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(tos))) {
		ap_log_rerror(APLOG_MARK, APLOG_INFO, r, "setsockopt(IP_TOS, %u)", tos);
	}
}


static int iptos_fixup(request_rec *r)
{
	iptos_config *config = ap_get_module_config(r->per_dir_config, &iptos_module);

	/* don't want to do this for subrequests */
	if (!ap_is_initial_req(r)) {
		return DECLINED;
	}
	iptos_set_tos(r, config->tos);
	return DECLINED;
}


static int iptos_handler(request_rec *r)
{
	iptos_config *config;

	/* we don't handle anything but GET */
	if (r->method_number != M_GET) return DECLINED;

	/* file doesn't exist, we won't be dealing with it */
	if (r->finfo.st_mode == 0) return DECLINED;

	config = ap_get_module_config(r->per_dir_config, &iptos_module);
	if (config->large_threshold == 0) return DECLINED;
	if (r->finfo.st_size < config->large_threshold) return DECLINED;

	iptos_set_tos(r, config->large_tos);
	return DECLINED;
}


static const command_rec iptos_cmds[] =
{
	{ "IPTOS", iptos_cmd_iptos, NULL, OR_OPTIONS, TAKE1,
		"arg is none, lowdelay, throughput, reliability, or lowcost -- and can be bitwise-or'd together using '|'"
	},
	{ "IPTOSthreshold", iptos_cmd_iptos_threshold, NULL, OR_OPTIONS, TAKE2,
		"arg1 is a number of bytes, arg2 is a TOS specification"
	},
	{NULL}
};


static const handler_rec iptos_handlers[] =
{
    { "*/*", iptos_handler },
    { NULL }
};


module MODULE_VAR_EXPORT iptos_module =
{
    STANDARD_MODULE_STUFF,
    NULL,			/* module initializer */
    iptos_create_dir_config,	/* per-directory config creator */
    iptos_merge_dir_config,	/* dir config merger */
    NULL,			/* server config creator */
    NULL,			/* server config merger */
    iptos_cmds,			/* command table */
    iptos_handlers,		/* [9] list of handlers */
    NULL,			/* [2] filename-to-URI translation */
    NULL,			/* [5] check/validate user_id */
    NULL,			/* [6] check user_id is valid *here* */
    NULL,			/* [4] check access by host address */
    NULL,			/* [7] MIME type checker/setter */
    iptos_fixup,		/* [8] fixups */
    NULL			/* [10] logger */
#if MODULE_MAGIC_NUMBER >= 19970103
    ,NULL			/* [3] header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
    ,NULL			/* process initializer */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
    ,NULL			/* process exit/cleanup */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
    ,NULL			/* [1] post read_request handling */
#endif
};
