/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/time.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "transfer.h"
#include "global.h"
#include "search.h"
#include "hotlist.h"
#include "share.h"
#include "resume.h"
#include "support.h"
#include "chat.h"
#include "handler.h"
#include "browse.h"

socket_t *browse_search(char *nick);
gint await_conn_ack2(gpointer data, gint source,
		     GdkInputCondition condition);
gint send_browse_send(gpointer data, gint source,
		      GdkInputCondition condition);
gint send_browse_get(gpointer data, gint source,
		     GdkInputCondition condition);

gint send_browse_files(gpointer data, gint source,
		       GdkInputCondition condition)
{
  char *command;
  socket_t *socket = (socket_t *) data;
  file_t *file;
  GList *dlist = (GList *) (socket->data);

  if (condition != GDK_INPUT_WRITE) {
    socket_destroy(socket, NULL);
    return 1;
  }

  if (socket->data) {
    file = (file_t *) (dlist->data);
    if ((file->flags & FLAG_SHARED) || (file->flags & FLAG_TO_SHARE)) {
      command = l_strdup_printf("\"%s\" %s %lu %d %d %d\n",
				file->winname, file->md5, file->size,
				file->bitrate, file->frequency,
				file->duration);
      send(source, command, strlen(command), 0);
      global.up_width.bytes += strlen(command);
      global.statistic.total[1] += (double)strlen(command);
      l_free(command);
      socket->cnt = 0;
    }
    socket->data = (void *) (dlist->next);
  }

  //  printf("%lu %lu\n", global.up_width.bytes, global.up_width.limit);
  if (!socket->data) {
    send(source, "\n", 1, 0);
    socket_destroy(socket, (void *) 1);
  }
  if (global.up_width.bytes >= global.up_width.limit) {
    uploads_disable();
    return 1;
  }
  return 1;
}

gint send_browse_nick(gpointer data, gint source,
		      GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;
  char *temp_str;

  if (!SERVER || (condition != GDK_INPUT_WRITE)) {
    socket_destroy(socket, NULL);
    return 1;
  }

  socket->data = global.user.files;

  temp_str = l_strdup_printf("%s\n", SERVER->nick);
  send(source, temp_str, strlen(temp_str), 0);

  gdk_input_remove(socket->input);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(send_browse_files), socket);
  
  return 1;
}

////////

gint get_browse_files(gpointer data, gint source,
		      GdkInputCondition condition)
{
  int res;
  char buf[2049];
  char *filename;
  char *md5;
  char *size;
  char *bitrate;
  char *frequency;
  char *duration;
  socket_t *socket = (socket_t *) data;
  file_t *file;
  char *pos1;
  char *pos2;
  browse_t *browse = (browse_t *) (socket->data);

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, NULL);
    return 1;
  }
  switch (res = recv(source, browse->buf + browse->pos, 
		     2048 - browse->pos - 1, 0)) {
  case -1:
    socket_destroy(socket, NULL);
    return 1;
  case 0:
    socket_destroy(socket, NULL);
    return 1;
  default:
    break;
  }

  browse->buf[res + browse->pos] = 0;

  // test
  global.down_width.bytes += res;
  global.statistic.total[0] += (double)res;
  // test end

  pos1 = browse->buf;
  while ((pos2 = strchr(pos1, '\n'))) {
    *pos2 = 0;

    filename = arg(pos1, 0);
    if (!filename)
      goto close;
    md5 = arg(NULL, 0);
    if (!md5)
      goto close;
    size = arg(NULL, 0);
    if (!size)
      goto close;
    bitrate = arg(NULL, 0);
    if (!bitrate)
      goto close;
    frequency = arg(NULL, 0);
    if (!frequency)
      goto close;
    duration = arg(NULL, 0);
    if (!duration)
      goto close;

    file = (file_t *) l_malloc(sizeof(file_t));
    file->user = l_strdup(browse->nick);
    file->winname = l_strdup(filename);
    file->md5 = l_strdup(md5);
    file->size = strtoul(size, NULL, 10);
    file->bitrate = atoi(bitrate);
    file->frequency = atoi(frequency);
    file->duration = atoi(duration);
    file->longname = l_strdup(file->winname);
    convert_to_unix(file->longname);
    file->shortname = l_strdup(extract_short_name(file->longname));
    file->mime_type = get_mimetype(file->longname);
    file->flags = 0;
    file->ip = 0;
    file->linespeed = 0;
    file->local = 0;		//remote file
    browse_insert_file(file);
    socket->cnt = 0;
    pos1 = pos2 + 1;
  }

  if (pos1 != browse->buf)
    strcpy(browse->buf, pos1);
  browse->pos = strlen(browse->buf);

  if (global.up_width.bytes >= global.up_width.limit) {
    downloads_disable();
    return 1;
  }

  return 1;

close:
  sprintf(buf, "%s %lu", browse->nick, socket->ip_long);
  cmd_browse_end(buf);
  socket_destroy(socket, (void *) 1);
  return 1;
}

gint get_browse_nick(gpointer data, gint source,
		     GdkInputCondition condition)
{
  unsigned char buffer[1025];
  int res;
  int cnt;
  char *pos;
  socket_t *socket = (socket_t *) data;
  socket_t *socket2;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, NULL);
    return 1;
  }
  gdk_input_remove(socket->input);
  socket->input = -1;

  switch (res = recv(source, buffer, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, NULL);
    return 1;
  case 0:
    socket_destroy(socket, NULL);
    return 1;
  default:
    break;
  }

  buffer[res] = 0;
  pos = strchr(buffer, '\n');
  if (!pos) {
    socket_destroy(socket, NULL);
    return 1;
  }
  cnt = (long) pos - (long) buffer;
  recv(source, buffer, cnt + 1, 0);
  buffer[cnt] = 0;

  socket2 = browse_search(buffer);
  if (!socket2) {
    socket_destroy(socket, NULL);
    return 1;
  } else if (socket != socket2) {
    // socket is an incoming connection, mapping to found one
    browse_t *browse;

    socket_destroy(socket2, (void *) 1);
    browse = (browse_t *) l_malloc(sizeof(browse_t));
    browse->buf[0] = 0;
    browse->pos = 0;
    browse->nick = l_strdup(buffer);
    socket->data = browse;
    socket2 = socket;
  }

  socket2->cnt = 0;

  socket2->input =
      gdk_input_add(socket2->fd, GDK_INPUT_READ,
		    GTK_SIGNAL_FUNC(get_browse_files), socket2);

  return 1;
}

///////////////

void browse_connect_and_start(socket_t * socket)
{

  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    return;
  }
  //  printf("connecting browse\n");
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_READ,
		    GTK_SIGNAL_FUNC(await_conn_ack2), socket);
}

gint await_conn_ack3(gpointer data, gint source,
		     GdkInputCondition condition)
{
  int res;
  char c;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, NULL);
    return 1;
  }
  switch (res = recv(source, &c, 1, 0)) {
  case -1:
    socket_destroy(socket, NULL);
    return 1;
  case 0:
    socket_destroy(socket, NULL);
    return 1;
  default:
    break;
  }

  if (c != '1') {
    socket_destroy(socket, NULL);
    return 1;
  }

  gdk_input_remove(socket->input);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(send_browse_send), socket);

  return 1;
}

gint await_conn_ack2(gpointer data, gint source,
		     GdkInputCondition condition)
{
  int res;
  char c;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, NULL);
    return 1;
  }
  switch (res = recv(source, &c, 1, 0)) {
  case -1:
    socket_destroy(socket, NULL);
    return 1;
  case 0:
    socket_destroy(socket, NULL);
    return 1;
  default:
    break;
  }

  if (c != '1') {
    socket_destroy(socket, NULL);
    return 1;
  }

  gdk_input_remove(socket->input);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(send_browse_get), socket);

  return 1;
}

gint send_browse_get(gpointer data, gint source,
		     GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_WRITE) {
    socket_destroy(socket, NULL);
    return 1;
  }

  send(source, "GETLIST", 7, 0);

  gdk_input_remove(socket->input);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_READ,
		    GTK_SIGNAL_FUNC(get_browse_nick), socket);

  return 1;
}

gint send_browse_send(gpointer data, gint source,
		      GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_WRITE) {
    socket_destroy(socket, NULL);
    return 1;
  }
  send(source, "SENDLIST", 8, 0);

  gdk_input_remove(socket->input);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(send_browse_nick), socket);

  return 1;
}

////////////

socket_t *browse_search(char *nick)
{
  GList *dlist;
  socket_t *socket;
  browse_t *browse;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (socket->type == S_BROWSE) {
      browse = socket->data;
      if (!l_strcasecmp(nick, browse->nick))
	return socket;
    }
  }
  return NULL;
}

void browse_direct(char *nick)
{
  socket_t *socket;
  browse_t *browse;

  socket = browse_search(nick);
  if (socket)
    return;			// already browsing

  socket = socket_new(S_BROWSE);
  browse = (browse_t *) l_malloc(sizeof(browse_t));
  browse->buf[0] = 0;
  browse->pos = 0;
  browse->nick = l_strdup(nick);
  socket->data = browse;
  send_command(CMD_CLIENT_BROWSE_DIRECT, nick);
}

GtkWidget *create_browse_popup()
{
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *trennlinie20;
  GtkWidget *customize_list6;
  file_t *file;
  GtkWidget *item;
  GtkWidget *user_popup;
  GtkObject* ctree;
  user_t* userinfo;
  GList* resume_list;
  GList* dlist;
  resume_t* resume;

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));
  if (global.popup_row != -1)
    file = (file_t *) gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
					     global.popup_row);
  else
    file = NULL;

  if (global.popup_row != -1) {
    if (!file) {
      item = gtk_menu_item_new_with_label(_("Download Folder"));
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(popup), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(on_download_folder_activate),
			 (void *) 0);
    } else {
      item = gtk_menu_item_new_with_label(_("Download Selected"));
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(popup), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(on_download2_activate),
			 (void *) 0);
      item =
	  gtk_menu_item_new_with_label(_("Download Selected with Foldername"));
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(popup), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(on_download2_activate),
			 (void *) 1);

      if ((resume_list = resume_search_size(file->size)) != NULL) {
	item = gtk_menu_item_new_with_label(_("Resume File"));
	gtk_widget_show(item);
	gtk_container_add(GTK_CONTAINER(popup), item);
	
	user_popup = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);
	for (dlist = resume_list; dlist; dlist = dlist->next) {
	  resume = dlist->data;
	  item = gtk_menu_item_new_with_label(extract_filename(resume->filename));
	  gtk_widget_show(item);
	  gtk_container_add(GTK_CONTAINER(user_popup), item);
	  gtk_signal_connect(GTK_OBJECT(item), "activate",
			     GTK_SIGNAL_FUNC(on_download_resume_activate),
			     (gpointer)resume);
	}
	g_list_free(resume_list);
      }
    }

    trennlinie20 = gtk_menu_item_new();
    gtk_widget_show(trennlinie20);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie20);
    gtk_widget_set_sensitive(trennlinie20, FALSE);
  }
  item = gtk_menu_item_new_with_label(_("Remove Browse"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_remove_browse_activate), NULL);
  item = gtk_menu_item_new_with_label(_("Refresh Files"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_browse_files_activate),
		     (gpointer) M_BROWSE);
  trennlinie20 = gtk_menu_item_new();
  gtk_widget_show(trennlinie20);
  gtk_container_add(GTK_CONTAINER(popup), trennlinie20);
  gtk_widget_set_sensitive(trennlinie20, FALSE);


  item = gtk_menu_item_new_with_label(_("User Menu"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);

  user_popup = create_user_popup(M_BROWSE);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

  trennlinie20 = gtk_menu_item_new();
  gtk_widget_show(trennlinie20);
  gtk_container_add(GTK_CONTAINER(popup), trennlinie20);
  gtk_widget_set_sensitive(trennlinie20, FALSE);

  ctree = GTK_OBJECT(global.popup_list);
  userinfo = (user_t *) gtk_object_get_data(ctree, "userinfo");

  if (userinfo && (global.status.connection == 2) && 
      !l_strcasecmp(SERVER->nick, userinfo->username)) {
    item = gtk_menu_item_new_with_label(_("Synchronize with Library"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_sync_activate),
		       (gpointer)userinfo);
    trennlinie20 = gtk_menu_item_new();
    gtk_widget_show(trennlinie20);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie20);
    gtk_widget_set_sensitive(trennlinie20, FALSE);
  }

  customize_list6 = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(customize_list6);
  gtk_container_add(GTK_CONTAINER(popup), customize_list6);
  gtk_signal_connect(GTK_OBJECT(customize_list6), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

file_t *file_create_from_browse_response(char *data)
{
  char *pos1;
  file_t *file;

  file = (file_t *) l_malloc(sizeof(file_t));

  file->user = l_strdup(arg(data, 0));
  file->winname = l_strdup(arg(NULL, 0));
  file->md5 = l_strdup(arg(NULL, 0));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%ld", &(file->size));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(file->bitrate));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(file->frequency));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(file->duration));

  file->longname = l_strdup(file->winname);
  convert_to_unix(file->longname);
  pos1 = extract_short_name(file->longname);
  file->shortname = l_strdup(pos1);
  file->mime_type = get_mimetype(file->longname);
  file->flags = 0;
  file->ip = 0;
  file->linespeed = 0;
  file->local = 0;		//remote file

  return file;
}

void file_create_from_browse_new(char *data)
{
  file_t *file;
  char *pos1;
  char *dir;
  char *name;
  char *user;

  user = arg(data, 0);
  dir = arg(NULL, 0);

  while ((name = arg(NULL, 0)) != NULL) {
    file = (file_t *) l_malloc(sizeof(file_t));

    file->user = l_strdup(user);

    file->md5 = l_strdup(arg(NULL, 0));

    pos1 = arg(NULL, 0);
    sscanf(pos1, "%ld", &(file->size));

    pos1 = arg(NULL, 0);
    sscanf(pos1, "%d", &(file->bitrate));

    pos1 = arg(NULL, 0);
    sscanf(pos1, "%d", &(file->frequency));

    pos1 = arg(NULL, 0);
    sscanf(pos1, "%d", &(file->duration));

    file->winname = (char *) l_malloc(strlen(dir) + 1 + strlen(name) + 1);
    sprintf(file->winname, "%s\\%s", dir, name);
    file->mime_type = get_mimetype(file->winname);

    file->longname = l_strdup(file->winname);
    convert_to_unix(file->longname);
    pos1 = extract_short_name(file->longname);
    file->shortname = l_strdup(pos1);
    file->mime_type = get_mimetype(file->longname);

    file->ip = 0;
    file->flags = 0;
    file->linespeed = 0;
    file->local = 0;		//remote file

    browse_insert_file(file);
  }
}

int browse_insert_file_real(file_t * file, GtkCTree * ctree)
{
  GtkWidget *temp;
  char *text;
  char *filename;
  char *pos;
  GtkCTreeNode *node;
  GtkCTreeNode *node2 = NULL;
  file_t *file2;

  //  printf("inserting real\n");
  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;
  node = NULL;

  temp = lookup_widget(global.win, "checkbutton18");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    filename = l_strdup(file->longname);
  else
    filename = l_strdup(file->shortname);

  //  strcpy(t, file->shortname);

  /////////////////
  pos = strtok(filename, "/");
  while (pos) {
    if (strlen(pos) <= 0) {
      pos = strtok(NULL, "/");
      continue;
    }
    node2 = node;
    if (node)
      node = GTK_CTREE_ROW(node)->children;
    else
      node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
    while (node) {
      text = GTK_CELL_PIXTEXT
	  (GTK_CTREE_ROW(node)->row.cell[ctree->tree_column])->text;
      if (strcmp(pos, text) == 0)
	break;
      node = GTK_CTREE_ROW(node)->sibling;
    }
    if (!node)
      break;
    if (GTK_CTREE_ROW(node)->is_leaf)
      break;
    pos = strtok(NULL, "/");
  }

  // do not insert identical files
  if (node && GTK_CTREE_ROW(node)->is_leaf) {
    file2 = (file_t *) gtk_ctree_node_get_row_data(ctree, node);
    if (!strcmp(file2->longname, file->longname)) {
      //      printf("do not insert [%s]\n", file->longname);
      return 0;
    }
  }

  node = node2;
  do {
    strcpy(tstr[0], pos);
    pos = strtok(NULL, "/");
    if (pos) {
      node =
	  gtk_ctree_insert_node(ctree, node, NULL, list, 5,
				global.pix.folder,
				global.pix.folderb,
				global.pix.folder_open,
				global.pix.folder_openb, FALSE, FALSE);
      gtk_ctree_node_set_row_data(ctree, node, NULL);
    } else {
      sprintf(tstr[1], "%.2f MB", (double) (file->size) / 1024 / 1024);
      sprintf(tstr[2], "%d", file->bitrate);
      sprintf(tstr[3], "%d", file->frequency);
      sprintf(tstr[4], "%d:%s%d", file->duration / 60,
	      (file->duration % 60) < 10 ? "0" : "", file->duration % 60);

      node = gtk_ctree_insert_node(ctree,
				   node, NULL, list, 5,
				   NULL, NULL, NULL, NULL, TRUE, FALSE);

      gtk_ctree_node_set_row_data(ctree, node, (gpointer) file);
    }
  } while (pos);

  l_free(filename);
  return 1;
}

void browse_insert_file(file_t * file)
{
  GtkWidget *temp;
  user_t *userinfo;
  user_t *stats;

  userinfo = browse_search_nick(file->user);
  if (!userinfo) {
    destroy_file_row(file);
    return;
  }

  userinfo->shared++;
  userinfo->bytes += file->size;

  userinfo->filesize_info[file->mime_type] += file->size;
  userinfo->filecount_info[file->mime_type]++;

  browse_update_user(userinfo);
  userinfo->files = g_list_prepend(userinfo->files, file);

  temp = lookup_widget(global.win, "notebook6");
  stats = (user_t *) gtk_object_get_data(GTK_OBJECT(temp), "browse_stats");
  stats->shared++;
  stats->bytes += file->size;
  browse_update_stats(stats);
}

void browse_user_files(char *nick, int max)
{
  user_t *userinfo;
  char *command;
  GtkWidget *temp;

  if (!nick || !(*nick))
    return;

  userinfo = browse_search_nick(nick);
  if (userinfo) {
    browse_clear_files(userinfo);
  } else {
    userinfo = browse_create_new(nick);
  }

  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;
  strcpy(tstr[0], _("Performing Browse...."));
  gtk_clist_append(GTK_CLIST(userinfo->link), list);

  if (max > 0)
    command = l_strdup_printf("%s %d", nick, max);
  else
    command = l_strdup_printf("%s", nick);

  temp = lookup_widget(global.win, "checkbutton44");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)) &&
      (opennap_version(0, 39) || !opennap_version(0, 0))) {
    browse_direct(nick);
  } else {
    send_command(CMD_CLIENT_BROWSE, command);
  }
  l_free(command);
}

/////
user_t *browse_search_nick(char *nick)
{
  GList *dlist;
  user_t *user;

  for (dlist = global.browses; dlist; dlist = dlist->next) {
    user = (user_t *) (dlist->data);
    if (!l_strcasecmp(nick, user->username))
      return user;
  }
  return NULL;
}

void browse_clear_files(user_t * userinfo)
{
  GtkWidget *temp;
  GtkCList *clist;
  user_t *stats;
  GList *dlist;
  file_t *file;
  int i1;

  if (!userinfo)
    return;

  // first updating browse stats
  temp = lookup_widget(global.win, "notebook6");
  stats = (user_t *) gtk_object_get_data(GTK_OBJECT(temp), "browse_stats");
  stats->shared -= userinfo->shared;
  stats->bytes -= userinfo->bytes;
  browse_update_stats(stats);

  // updating user stats
  userinfo->shared = 0;
  userinfo->bytes = 0;
  for (i1 = 0; i1 < MIME_SIZE; i1++) {
    userinfo->filecount_info[i1] = 0;
    userinfo->filesize_info[i1] = 0.;
  }
  browse_update_user(userinfo);

  // releasing file list
  for (dlist = (GList *) (userinfo->files); dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    destroy_file_row(file);
  }
  g_list_free(userinfo->files);
  userinfo->files = NULL;

  // clearing the ctree
  if (userinfo->link) {
    clist = GTK_CLIST(userinfo->link);
    gtk_clist_clear(clist);
  }
}

user_t *browse_create_new(char *nick)
{
  user_t *userinfo;
  int i1;

  userinfo = (user_t *) l_malloc(sizeof(user_t));
  userinfo->files = NULL;
  userinfo->timeout = 0;
  userinfo->bytes = 0;
  userinfo->shared = 0;
  for (i1 = 0; i1 < MIME_SIZE; i1++) {
    userinfo->filecount_info[i1] = 0;
    userinfo->filesize_info[i1] = 0.;
  }
  strcpy(userinfo->username, nick);
  global.browses = g_list_append(global.browses, userinfo);

  userinfo->link = browse_create_page(nick);
  gtk_object_set_data(GTK_OBJECT(userinfo->link), "userinfo", userinfo);
  browse_update_user(userinfo);

  return userinfo;
}

void on_button_files_clicked(GtkButton * button, gpointer user_data);

void on_expand_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data)
{
  GtkCTree *ctree;

  ctree = GTK_CTREE(user_data);
  gtk_ctree_expand_recursive(ctree, NULL);
}

void on_collapse_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data)
{
  GtkCTree *ctree;

  ctree = GTK_CTREE(user_data);
  gtk_ctree_collapse_recursive(ctree, NULL);
}

GtkWidget *browse_create_page(char *nick)
{
  GtkWidget *vbox;
  GtkWidget *scrolledwindow;
  GtkWidget *ctree;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *hbox2;
  GtkWidget *hbox3;
  GtkWidget *button;
  GtkNotebook *notebook;
  GtkTooltips *tips;
  GtkWidget* pixmap;
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  ctree = gtk_ctree_new(5, 0);
  gtk_widget_show(ctree);
  gtk_widget_set_events(ctree,
			GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), ctree);
  gtk_clist_set_column_width(GTK_CLIST(ctree), 0, global.browse_width[0]);
  gtk_clist_set_column_visibility(GTK_CLIST(ctree), 0,
				  global.browse_show[0]);
  gtk_clist_set_column_width(GTK_CLIST(ctree), 1, global.browse_width[1]);
  gtk_clist_set_column_visibility(GTK_CLIST(ctree), 1,
				  global.browse_show[1]);
  gtk_clist_set_column_width(GTK_CLIST(ctree), 2, global.browse_width[2]);
  gtk_clist_set_column_visibility(GTK_CLIST(ctree), 2,
				  global.browse_show[2]);
  gtk_clist_set_column_width(GTK_CLIST(ctree), 3, global.browse_width[3]);
  gtk_clist_set_column_visibility(GTK_CLIST(ctree), 3,
				  global.browse_show[3]);
  gtk_clist_set_column_width(GTK_CLIST(ctree), 4, global.browse_width[4]);
  gtk_clist_set_column_visibility(GTK_CLIST(ctree), 4,
				  global.browse_show[4]);


  gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_EXTENDED);
  gtk_clist_column_titles_show(GTK_CLIST(ctree));
  gtk_clist_set_compare_func(GTK_CLIST(ctree), browse_compare);

  label = gtk_label_new(_("Filename"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(ctree), 0, label);

  label = gtk_label_new(_("Size"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(ctree), 1, label);

  label = gtk_label_new(_("Bitrate"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(ctree), 2, label);

  label = gtk_label_new(_("Frequency"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(ctree), 3, label);

  label = gtk_label_new(_("Length"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(ctree), 4, label);

  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox2);
  gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 3);
  gtk_widget_show(hbox);
  gtk_box_pack_end(GTK_BOX(hbox2), hbox, FALSE, FALSE, 0);

  ///////

  tips =
      (GtkTooltips *) gtk_object_get_data(GTK_OBJECT(global.win),
					  "tooltips");
  button = gtk_button_new_with_label(_("Remove Browse"));
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  //  gtk_widget_set_usize(button, -2, 15);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_remove_browse_clicked),
		     (gpointer) ctree);

  button = gtk_button_new_with_label(_("  X  "));
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  //  gtk_widget_set_usize(button, -2, 15);
  gtk_tooltips_set_tip(tips, button, _("Expand whole tree"), NULL);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_expand_clicked), (gpointer) ctree);

  button = gtk_button_new_with_label(_("  C  "));
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  //  gtk_widget_set_usize(button, -2, 15);
  gtk_tooltips_set_tip(tips, button, _("Collapse whole tree"), NULL);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_collapse_clicked),
		     (gpointer) ctree);

  //////
  button = gtk_button_new ();
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
  gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_button_files_clicked),
                      (gpointer)ctree);

  hbox3 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox3);
  gtk_container_add (GTK_CONTAINER (button), hbox3);

  pixmap = create_pixmap (global.win, "arrowu.xpm");
  gtk_widget_show (pixmap);
  gtk_box_pack_start (GTK_BOX (hbox3), pixmap, TRUE, TRUE, 0);

  label = gtk_label_new (_("Files"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox3), label, FALSE, FALSE, 0);
  gtk_misc_set_padding (GTK_MISC (label), 3, 0);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  gtk_widget_set_usize(frame, 50, -2);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);

  label = gtk_label_new("");
  gtk_widget_show(label);
  gtk_container_add(GTK_CONTAINER(frame), label);
  gtk_object_set_data(GTK_OBJECT(ctree), "label2", label);

  label = gtk_label_new (_("MB"));
  gtk_widget_show (label);
  gtk_misc_set_padding (GTK_MISC (label), 3, 0);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  gtk_widget_set_usize(frame, 55, -2);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);

  label = gtk_label_new("");
  gtk_widget_show(label);
  gtk_container_add(GTK_CONTAINER(frame), label);
  gtk_object_set_data(GTK_OBJECT(ctree), "label3", label);
  gtk_object_set_data(GTK_OBJECT(ctree), "child", vbox);
  gtk_object_set_data(GTK_OBJECT(vbox), "clist", ctree);

  gtk_signal_connect(GTK_OBJECT(ctree), "click_column",
		     GTK_SIGNAL_FUNC(on_list_click_column), NULL);
  gtk_signal_connect(GTK_OBJECT(ctree), "button_press_event",
		     GTK_SIGNAL_FUNC(on_browse_button_press_event), NULL);
  gtk_signal_connect(GTK_OBJECT(ctree), "motion_notify_event",
		     GTK_SIGNAL_FUNC(on_browse_motion_notify_event), NULL);
  gtk_signal_connect(GTK_OBJECT(ctree), "leave_notify_event",
		     GTK_SIGNAL_FUNC(on_list_leave_notify_event), NULL);

  gtk_object_set_data(GTK_OBJECT(vbox), "tree", (void *) ctree);

  label = gtk_label_new(nick);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook6"));
  gtk_notebook_append_page(notebook, vbox, label);
  gtk_notebook_set_page(notebook,
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
					      scrolledwindow));

  return ctree;
}

void browse_update_user(user_t * userinfo)
{
  GtkObject *ctree;
  GtkWidget *temp;
  char str[1024];

  if (!userinfo)
    return;
  if (userinfo->link) {
    ctree = GTK_OBJECT(userinfo->link);
    if (ctree) {
      temp = gtk_object_get_data(ctree, "label2");
      sprintf(str, "%d", userinfo->shared);
      gtk_label_set_text(GTK_LABEL(temp), str);
      temp = gtk_object_get_data(ctree, "label3");
      sprintf(str, "%.0f", userinfo->bytes / 1048576.);
      gtk_label_set_text(GTK_LABEL(temp), str);
    }
  }
}

void browse_update_stats(user_t * stats)
{
  GtkWidget *temp;
  char str[1024];

  temp = lookup_widget(global.win, "label229");
  sprintf(str, "%d", stats->shared);
  gtk_label_set_text(GTK_LABEL(temp), str);
  temp = lookup_widget(global.win, "label231");
  sprintf(str, "%.0f", stats->bytes / 1048576.);
  gtk_label_set_text(GTK_LABEL(temp), str);
}

void browse_remove(user_t * userinfo)
{
  GtkObject *ctree;
  GtkWidget *child;
  GtkNotebook *notebook;
  int i1;

  if (!userinfo)
    return;

  // clearing file list
  browse_clear_files(userinfo);

  // removing browse page
  ctree = GTK_OBJECT(userinfo->link);
  if (ctree) {
    notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook6"));
    child = GTK_WIDGET(gtk_object_get_data(ctree, "child"));
    i1 = gtk_notebook_page_num(notebook, child);
    gtk_notebook_remove_page(notebook, i1);
  }
  l_free(userinfo);

  global.browses = g_list_remove(global.browses, userinfo);
}

file_t *browse_search_file(user_t * info, char *winname)
{
  GList *dlist;
  file_t *file;

  for (dlist = info->files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    if (!strcmp(winname, file->winname))
      return file;
  }
  return NULL;
}

///////////
void search_browse(user_t * userinfo, search_t * search, int *cnt)
{
  file_t *file;
  GList *dlist;

  for (dlist = userinfo->files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    if (search_pattern_fits_file(search->pattern, file, 1)) {
      search_insert_file(search, file);
      (*cnt)++;
    }
    if (search->pattern->max_results && (*cnt) >= search->pattern->max_results)
      break;
  }
}

void browse_do_search(search_t * search)
{
  GtkCList *clist;
  user_t *userinfo;
  GList *dlist;
  int cnt = 0;

  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  for (dlist = global.browses; dlist; dlist = dlist->next) {
    userinfo = (user_t *) (dlist->data);
    search_browse(userinfo, search, &cnt);
    if (search->pattern->max_results && (cnt >= search->pattern->max_results))
      break;;
  }

  search_finish(search);
}

void browse_show(user_t* userinfo, int mediatype) {
  file_t* file;
  GList* dlist;

  if (userinfo->link) {
    GtkCTree *ctree = GTK_CTREE(userinfo->link);
    gtk_clist_clear(GTK_CLIST(ctree));
    gtk_clist_freeze(GTK_CLIST(ctree));
    for (dlist = userinfo->files; dlist; dlist = dlist->next) {
      file = dlist->data;
      if ((mediatype != MIME_SIZE) && (file->mime_type != mediatype)) continue;
      browse_insert_file_real(file, ctree);
    }
    gtk_clist_thaw(GTK_CLIST(ctree));
    gtk_ctree_sort_recursive(ctree, NULL);
  }
  
}
