
/*
    Copyright (c) 2008 Andrew Caudwell (acaudwell@gmail.com)
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:
    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. The name of the author may not be used to endorse or promote products
       derived from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
    OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
    DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
    THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "fontpack.h"

//FPChar

FPChar::FPChar() {
    this->id=0;
    this->font_size=0;
    this->base=0;
    this->w=0;
    this->h=0;
    tmin.x=-1.0f;
}

FPChar::FPChar(const char* line, TextureResource* texture) {

    int x, y;

    sscanf( line, "%d %d %d %d %d %d %d", &id, &font_size, &base, &x, &y, &w, &h);

    float tx = 1.0f / texture->w;
    float ty = 1.0f / texture->h;

    tmin = vec2f(x * tx, y * ty);
    tmax = vec2f((x + w) * tx, (y + h) * ty);
}

void FPChar::draw(vec2f pos) {
    if(tmin.x<0.0f) return;

    pos.y -= base;

    glBegin(GL_QUADS);
        glTexCoord2fv(tmin);
        glVertex2fv(pos);

        glTexCoord2f(tmax.x,tmin.y);
        glVertex2f(pos.x+w, pos.y);

        glTexCoord2fv(tmax);
        glVertex2f(pos.x+w, pos.y+h);

        glTexCoord2f(tmin.x,tmax.y);
        glVertex2f(pos.x, pos.y+h);
    glEnd();
}

//FPFont

FPSize::FPSize(int size, TextureResource* texture) {
    this->size    = size;
    this->texture = texture;

    space       = size/3;
}

void FPSize::setChar(int charid, FPChar fpchar) {
	chars[charid] = fpchar;
}

vec3f FPSize::charColour(int c) {
    switch(c) {
        case '0':
            return vec3f(0.0f,0.0f,0.0f);
        case '1':
            return vec3f(1.0f,0.0f,0.0f);
        case '2':
            return vec3f(0.0f,1.0f,0.0f);
        case '3':
            return vec3f(1.0f,1.0f,0.0f);
        case '4':
            return vec3f(0.0f,0.0f,1.0f);
        case '5':
            return vec3f(0.0f,1.0f,1.0f);
        case '6':
            return vec3f(1.0f,0.0f,1.0f);
        case '7':
            return vec3f(1.0f,1.0f,1.0f);
        default:
            break;
    }

    return vec3f(1.0f,1.0f,1.0f);
}

int FPSize::render(int x, int y, const char *text, bool force_colour, bool test) {

    vec2f corner(x, y + size);

    int max_width  = 0;
    int line_width = 0;

    const char *cp = text;

    int c;

    while ((c = *cp) != 0) {

        //new line
        if (c == '\n') {
            corner.y += size;
            corner.x  = x;

            if (line_width>max_width) max_width = line_width;
            line_width = 0;
        }

        //space
        else if(c == 32) {
            corner.x   += space;
            line_width += space;
        }

        //color change
        else if(c == '^' && *(cp+1) != 0) {
            cp++;
            c= *cp;

            if(!force_colour && !test) {
                vec3f textcolour = charColour(c);
                glColor3fv(textcolour);
            }
        }

        //draw character
        else {
            FPChar* chr = &(chars[c]);

            if(!test) {
                chr->draw(corner);
            }

            int char_width = chr->w + 2;

            corner.x   += char_width;
            line_width += char_width;
        }

        cp++;
    }

    if(line_width>max_width) max_width = line_width;

    return max_width;
}

//FPInstance

FPInstance::FPInstance() {
}

FPInstance::FPInstance(FontPack* fp, int size, bool shadow) {
    this->fp = fp;
    this->shadow      = shadow;

    soffset     = vec2f(2.0f,2.0f);
    sstrength   = 0.7f;

    setSize(size);
}

int  FPInstance::getSize() {
    return font->size;
}

void FPInstance::setSize(int size) {
    this->font = fp->getFontSize(size);

    if(this->font==0) {
        printf("font has no size %d\n", size);
        exit(1);
    }
}

void FPInstance::shadowStrength(float s) {
    sstrength = s;
}

void FPInstance::shadowOffset(float x, float y) {
    soffset = vec2f(x,y);
}

void FPInstance::dropShadow(bool shadow) {
    this->shadow = shadow;
}

int FPInstance::width(const char *text) {
    return font->render(0,0, text, false, true);
}

int FPInstance::print(int x, int y, const char *str, ...) {
    char buf[4096];

    va_list vl;

    va_start (vl, str);
    vsnprintf (buf, 4096, str, vl);
    va_end (vl);

    return draw(x, y, buf);
}

int FPInstance::draw(int x, int y, const char* text) {

	glBindTexture(GL_TEXTURE_2D, fp->getTexture()->textureid);

    if(shadow) {
        vec4f current = display.currentColour();
        glColor4f(0.0f, 0.0f, 0.0f, current.w * sstrength);
        font->render(x + (int) soffset.x, y + (int) soffset.y, text, true);
        glColor4fv(current);
    }

    return font->render(x, y, text);
}

//FontPack

FontPack::FontPack(std::string tgafile, std::string infofile) {

    texture = (TextureResource*) texturemanager.grab(tgafile, 1, 1, 1);

    FILE* fp = fopen(infofile.c_str(), "rt");

    if(fp==0) {
        printf("error loading font info file %s\n", infofile.c_str());
        exit(1);
    }


    char line[256];
	while (fgets( line, 256, fp )) {
		if (line[0]=='#') continue;

        FPChar chr(line, texture);

        int size = chr.font_size;

		FPSize* fs = fsize[size];

		if(fs==0) {
			fs = new FPSize(size, texture);
			fsize[size] = fs;
		}

		fs->setChar(chr.id, chr);
	}

    fclose(fp);
}

FPInstance FontPack::getInstance() {
    return FPInstance(this, 0);
}

FPInstance FontPack::getInstance(int size, bool shadow) {
    return FPInstance(this, size, shadow);
}

FPSize* FontPack::getFontSize(int size) {

    if(size==0) {
        int min_size = -1;

        std::map<int, FPSize*>::iterator it;
        for(it=fsize.begin();it!=fsize.end();it++) {
            FPSize* fs = it->second;

            if(min_size==-1 || fs->size < min_size) {
                min_size = fs->size;
            }
        }

        if(min_size==-1) {
            return 0;
        }

        size = min_size;
    }

    return fsize[size];
}

TextureResource* FontPack::getTexture() {
    return texture;
}

FontPack::~FontPack() {

    std::map<int, FPSize*>::iterator it;
    for(it=fsize.begin();it!=fsize.end();it++) {
        FPSize* fs = it->second;
        it->second=0;
        delete fs;
    }

	//free texture
	texturemanager.release(texture);
}

