/*
    chips.c - Part of libsensors, a Linux library for reading sensor data.
    Copyright (c) 1998, 1999  Frodo Looijaard <frodol@dds.nl> 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "chips.h"
#include "data.h"
#include "sensors.h"
#include "kernel/include/sensors.h"

/* For each chip, a list of features;
   For each feature, the number, its name, the logical mapping (for example,
   fan1_div belongs to fan1), the compute mapping, an access field, the 
   SYSCTL id of the /proc field, the value location of the /proc field,
   and the magnitude.
   The access field  indicates whether resources are readable and writable.
   Sometimes, a resource is neither; in this case, it is just a catch-all
   category, for mappings. 
   The value location is the byte-offset where this value can be found in
   the sysctl buffer.
   The magnitude is the 10-power through which the read value must be
   divided. */


#define VALUE(x) (x-1) * sizeof(long)

static sensors_chip_feature lm78_features[] =
  { 
    { SENSORS_LM78_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM78_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM78_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM78_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM78_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM78_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM78_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM78_IN0_MIN, "in0_min", SENSORS_LM78_IN0, SENSORS_LM78_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM78_IN1_MIN, "in1_min", SENSORS_LM78_IN1, SENSORS_LM78_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM78_IN2_MIN, "in2_min", SENSORS_LM78_IN2, SENSORS_LM78_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM78_IN3_MIN, "in3_min", SENSORS_LM78_IN3, SENSORS_LM78_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM78_IN4_MIN, "in4_min", SENSORS_LM78_IN4, SENSORS_LM78_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM78_IN5_MIN, "in5_min", SENSORS_LM78_IN5, SENSORS_LM78_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM78_IN6_MIN, "in6_min", SENSORS_LM78_IN6, SENSORS_LM78_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM78_IN0_MAX, "in0_max", SENSORS_LM78_IN0, SENSORS_LM78_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM78_IN1_MAX, "in1_max", SENSORS_LM78_IN1, SENSORS_LM78_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM78_IN2_MAX, "in2_max", SENSORS_LM78_IN2, SENSORS_LM78_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM78_IN3_MAX, "in3_max", SENSORS_LM78_IN3, SENSORS_LM78_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM78_IN4_MAX, "in4_max", SENSORS_LM78_IN4, SENSORS_LM78_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM78_IN5_MAX, "in5_max", SENSORS_LM78_IN5, SENSORS_LM78_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM78_IN6_MAX, "in6_max", SENSORS_LM78_IN6, SENSORS_LM78_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM78_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM78_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM78_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM78_FAN1_MIN, "fan1_min", SENSORS_LM78_FAN1, SENSORS_LM78_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM78_FAN2_MIN, "fan2_min", SENSORS_LM78_FAN2, SENSORS_LM78_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM78_FAN3_MIN, "fan3_min", SENSORS_LM78_FAN3, SENSORS_LM78_FAN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM78_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM78_TEMP_HYST, "temp_hyst", SENSORS_LM78_TEMP,
                              SENSORS_LM78_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM78_TEMP_OVER, "temp_over", SENSORS_LM78_TEMP,
                              SENSORS_LM78_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM78_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM78_FAN1_DIV, "fan1_div", SENSORS_LM78_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM78_FAN2_DIV, "fan2_div", SENSORS_LM78_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM78_FAN3_DIV, "fan3_div", SENSORS_LM78_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, LM78_SYSCTL_FAN_DIV, VALUE(3), 
                              0 },
    { SENSORS_LM78_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm78j_features[] =
  { 
    { SENSORS_LM78J_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM78J_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM78J_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM78J_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM78J_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM78J_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM78J_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM78J_IN0_MIN, "in0_min", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM78J_IN1_MIN, "in1_min", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM78J_IN2_MIN, "in2_min", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM78J_IN3_MIN, "in3_min", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM78J_IN4_MIN, "in4_min", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM78J_IN5_MIN, "in5_min", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM78J_IN6_MIN, "in6_min", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM78J_IN0_MAX, "in0_max", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM78J_IN1_MAX, "in1_max", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM78J_IN2_MAX, "in2_max", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM78J_IN3_MAX, "in3_max", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM78J_IN4_MAX, "in4_max", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM78J_IN5_MAX, "in5_max", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM78J_IN6_MAX, "in6_max", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM78J_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM78J_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM78J_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM78J_FAN1_MIN, "fan1_min", SENSORS_LM78J_FAN1,
                              SENSORS_LM78J_FAN1, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM78J_FAN2_MIN, "fan2_min", SENSORS_LM78J_FAN2,
                              SENSORS_LM78J_FAN2, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM78J_FAN3_MIN, "fan3_min", SENSORS_LM78J_FAN3,
                              SENSORS_LM78J_FAN3, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM78J_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM78J_TEMP_HYST, "temp_hyst", SENSORS_LM78J_TEMP,
                               SENSORS_LM78J_TEMP, SENSORS_MODE_RW, 
                               LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM78J_TEMP_OVER, "temp_over", SENSORS_LM78J_TEMP,
                               SENSORS_LM78J_TEMP, SENSORS_MODE_RW, 
                               LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM78J_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM78J_FAN1_DIV, "fan1_div", SENSORS_LM78J_FAN1,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_LM78J_FAN2_DIV, "fan2_div", SENSORS_LM78J_FAN2,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              LM78_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_LM78J_FAN3_DIV, "fan3_div", SENSORS_LM78J_FAN3,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              LM78_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_LM78J_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                            SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm79_features[] =
  { 
    { SENSORS_LM79_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM79_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM79_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM79_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM79_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM79_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM79_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM79_IN0_MIN, "in0_min", SENSORS_LM79_IN0, SENSORS_LM79_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM79_IN1_MIN, "in1_min", SENSORS_LM79_IN1, SENSORS_LM79_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM79_IN2_MIN, "in2_min", SENSORS_LM79_IN2, SENSORS_LM79_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM79_IN3_MIN, "in3_min", SENSORS_LM79_IN3, SENSORS_LM79_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM79_IN4_MIN, "in4_min", SENSORS_LM79_IN4, SENSORS_LM79_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM79_IN5_MIN, "in5_min", SENSORS_LM79_IN5, SENSORS_LM79_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM79_IN6_MIN, "in6_min", SENSORS_LM79_IN6, SENSORS_LM79_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM79_IN0_MAX, "in0_max", SENSORS_LM79_IN0, SENSORS_LM79_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM79_IN1_MAX, "in1_max", SENSORS_LM79_IN1, SENSORS_LM79_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM79_IN2_MAX, "in2_max", SENSORS_LM79_IN2, SENSORS_LM79_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM79_IN3_MAX, "in3_max", SENSORS_LM79_IN3, SENSORS_LM79_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM79_IN4_MAX, "in4_max", SENSORS_LM79_IN4, SENSORS_LM79_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM79_IN5_MAX, "in5_max", SENSORS_LM79_IN5, SENSORS_LM79_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM79_IN6_MAX, "in6_max", SENSORS_LM79_IN6, SENSORS_LM79_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM79_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM79_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM79_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM79_FAN1_MIN, "fan1_min", SENSORS_LM79_FAN1, SENSORS_LM79_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM79_FAN2_MIN, "fan2_min", SENSORS_LM79_FAN2, SENSORS_LM79_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM79_FAN3_MIN, "fan3_min", SENSORS_LM79_FAN3, SENSORS_LM79_FAN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM79_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM79_TEMP_HYST, "temp_hyst", SENSORS_LM79_TEMP,
                              SENSORS_LM79_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM79_TEMP_OVER, "temp_over", SENSORS_LM79_TEMP,
                              SENSORS_LM79_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM79_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM79_FAN1_DIV, "fan1_div", SENSORS_LM79_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM79_FAN2_DIV, "fan2_div", SENSORS_LM79_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM79_FAN3_DIV, "fan3_div", SENSORS_LM79_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, LM78_SYSCTL_FAN_DIV, VALUE(3), 
                              0 },
    { SENSORS_LM79_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature lm75_features[] =
  {
    { SENSORS_LM75_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM75_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM75_TEMP_HYST, "temp_hyst", SENSORS_LM75_TEMP,
                              SENSORS_LM75_TEMP, SENSORS_MODE_RW, 
                              LM75_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM75_TEMP_OVER, "temp_over", SENSORS_LM75_TEMP,
                              SENSORS_LM75_TEMP, SENSORS_MODE_RW, 
                              LM75_SYSCTL_TEMP, VALUE(1), 1 },
    { 0 }
  };

static sensors_chip_feature adm1021_features[] =
  {
    { SENSORS_ADM1021_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_ADM1021_TEMP_HYST, "temp_low", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_ADM1021_TEMP_OVER, "temp_over", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_ADM1021_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM1021_DIE_CODE, "die_code", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature max1617_features[] =
  {
    { SENSORS_MAX1617_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_MAX1617_TEMP_HYST, "temp_low", SENSORS_MAX1617_TEMP,
                              SENSORS_MAX1617_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617_TEMP_OVER, "temp_over", SENSORS_MAX1617_TEMP,
                              SENSORS_MAX1617_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_MAX1617_REMOTE_TEMP,
                              SENSORS_MAX1617_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_MAX1617_REMOTE_TEMP,
                              SENSORS_MAX1617_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature max1617a_features[] =
  {
    { SENSORS_MAX1617A_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_MAX1617A_TEMP_HYST, "temp_low", SENSORS_MAX1617A_TEMP,
                              SENSORS_MAX1617A_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617A_TEMP_OVER, "temp_over", SENSORS_MAX1617A_TEMP,
                              SENSORS_MAX1617A_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_MAX1617A_REMOTE_TEMP,
                              SENSORS_MAX1617A_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_MAX1617A_REMOTE_TEMP,
                              SENSORS_MAX1617A_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617A_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature gl518_features[] =
  {
    { SENSORS_GL518_VDD, "vdd", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                            SENSORS_MODE_R, GL518_SYSCTL_VDD, VALUE(3), 2 },
    { SENSORS_GL518_VIN1, "vin1", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN1, VALUE(3), 
                             2 },
    { SENSORS_GL518_VIN2, "vin2", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN2, VALUE(3), 
                             2 },
    { SENSORS_GL518_VIN3, "vin3", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN3, VALUE(3), 2 },
    { SENSORS_GL518_VDD_MIN, "vdd_min", SENSORS_GL518_VDD,
                                SENSORS_GL518_VDD, SENSORS_MODE_RW,
                                GL518_SYSCTL_VDD, VALUE(1), 2 },
    { SENSORS_GL518_VIN1_MIN, "vin1_min", SENSORS_GL518_VIN1, 
                                 SENSORS_GL518_VIN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN1, VALUE(1), 2 },
    { SENSORS_GL518_VIN2_MIN, "vin2_min", SENSORS_GL518_VIN2, 
                                 SENSORS_GL518_VIN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN2, VALUE(1), 2 },
    { SENSORS_GL518_VIN3_MIN, "vin3_min", SENSORS_GL518_VIN3, 
                                 SENSORS_GL518_VIN3, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN3, VALUE(1), 2 },
    { SENSORS_GL518_VDD_MAX, "vdd_max", SENSORS_GL518_VDD,
                                SENSORS_GL518_VDD, SENSORS_MODE_RW,
                                GL518_SYSCTL_VDD, VALUE(2), 2 },
    { SENSORS_GL518_VIN1_MAX, "vin1_max", SENSORS_GL518_VIN1, 
                                 SENSORS_GL518_VIN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN1, VALUE(2), 2 },
    { SENSORS_GL518_VIN2_MAX, "vin2_max", SENSORS_GL518_VIN2, 
                                 SENSORS_GL518_VIN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN2, VALUE(2), 2 },
    { SENSORS_GL518_VIN3_MAX, "vin3_max", SENSORS_GL518_VIN3, 
                                 SENSORS_GL518_VIN3, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN3, VALUE(2), 2 },
    { SENSORS_GL518_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_GL518_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_GL518_FAN1_MIN, "fan1_min", SENSORS_GL518_FAN1, 
                                 SENSORS_GL518_FAN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_GL518_FAN2_MIN, "fan2_min", SENSORS_GL518_FAN2, 
                                 SENSORS_GL518_FAN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_GL518_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_GL518_TEMP_HYST, "temp_hyst", SENSORS_GL518_TEMP, 
                                  SENSORS_GL518_TEMP, SENSORS_MODE_RW,
                                  GL518_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_GL518_TEMP_OVER, "temp_over", SENSORS_GL518_TEMP, 
                                  SENSORS_GL518_TEMP, SENSORS_MODE_RW,
                                  GL518_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_GL518_FAN1_DIV, "fan1_div", SENSORS_GL518_FAN1, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_GL518_FAN2_DIV, "fan2_div", SENSORS_GL518_FAN2, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_GL518_ALARMS, "alarms", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_R,
                               GL518_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_GL518_BEEP_ENABLE, "beep_enable", SENSORS_GL518_ALARMS,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_GL518_BEEPS, "beeps", SENSORS_GL518_ALARMS,
                               SENSORS_GL518_ALARMS, SENSORS_MODE_RW,
                               GL518_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_GL518_FAN1OFF, "fan1_off", SENSORS_GL518_FAN1,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_FAN1OFF, VALUE(1), 0 },
    { SENSORS_GL518_FAN1PIN, "fan1_off_pin", SENSORS_GL518_FAN1OFF,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_FAN1OFF, VALUE(2), 0 },
    { SENSORS_GL518_ITERATE, "iterate", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_ITERATE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature gl520_features[] =
  {
    { SENSORS_GL520_VDD, "vdd", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                            SENSORS_MODE_R, GL520_SYSCTL_VDD, VALUE(3), 2 },
    { SENSORS_GL520_VIN1, "vin1", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN1, VALUE(3), 
                             2 },
    { SENSORS_GL520_VIN2, "vin2", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN2, VALUE(3), 
                             2 },
    { SENSORS_GL520_VIN3, "vin3", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN3, VALUE(3), 2 },
    { SENSORS_GL520_VIN4, "vin4", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN4, VALUE(3), 2 },
    { SENSORS_GL520_VDD_MIN, "vdd_min", SENSORS_GL520_VDD,
                                SENSORS_GL520_VDD, SENSORS_MODE_RW,
                                GL520_SYSCTL_VDD, VALUE(1), 2 },
    { SENSORS_GL520_VIN1_MIN, "vin1_min", SENSORS_GL520_VIN1, 
                                 SENSORS_GL520_VIN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN1, VALUE(1), 2 },
    { SENSORS_GL520_VIN2_MIN, "vin2_min", SENSORS_GL520_VIN2, 
                                 SENSORS_GL520_VIN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN2, VALUE(1), 2 },
    { SENSORS_GL520_VIN3_MIN, "vin3_min", SENSORS_GL520_VIN3, 
                                 SENSORS_GL520_VIN3, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN3, VALUE(1), 2 },
    { SENSORS_GL520_VIN4_MIN, "vin4_min", SENSORS_GL520_VIN4, 
                                 SENSORS_GL520_VIN4, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN4, VALUE(1), 2 },
    { SENSORS_GL520_VDD_MAX, "vdd_max", SENSORS_GL520_VDD,
                                SENSORS_GL520_VDD, SENSORS_MODE_RW,
                                GL520_SYSCTL_VDD, VALUE(2), 2 },
    { SENSORS_GL520_VIN1_MAX, "vin1_max", SENSORS_GL520_VIN1, 
                                 SENSORS_GL520_VIN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN1, VALUE(2), 2 },
    { SENSORS_GL520_VIN2_MAX, "vin2_max", SENSORS_GL520_VIN2, 
                                 SENSORS_GL520_VIN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN2, VALUE(2), 2 },
    { SENSORS_GL520_VIN3_MAX, "vin3_max", SENSORS_GL520_VIN3, 
                                 SENSORS_GL520_VIN3, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN3, VALUE(2), 2 },
    { SENSORS_GL520_VIN4_MAX, "vin4_max", SENSORS_GL520_VIN4, 
                                 SENSORS_GL520_VIN4, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN4, VALUE(2), 2 },
    { SENSORS_GL520_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_GL520_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_GL520_FAN1_MIN, "fan1_min", SENSORS_GL520_FAN1, 
                                 SENSORS_GL520_FAN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_GL520_FAN2_MIN, "fan2_min", SENSORS_GL520_FAN2, 
                                 SENSORS_GL520_FAN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_GL520_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_GL520_TEMP1_HYST, "temp1_hyst", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_GL520_TEMP1_OVER, "temp1_over", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_GL520_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_GL520_TEMP2_HYST, "temp2_hyst", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_GL520_TEMP2_OVER, "temp2_over", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_GL520_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, GL520_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_GL520_FAN1_DIV, "fan1_div", SENSORS_GL520_FAN1, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_GL520_FAN2_DIV, "fan2_div", SENSORS_GL520_FAN2, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_GL520_ALARMS, "alarms", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_R,
                               GL520_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_GL520_BEEP_ENABLE, "beep_enable", SENSORS_GL520_ALARMS,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_GL520_BEEPS, "beeps", SENSORS_GL520_ALARMS,
                               SENSORS_GL520_ALARMS, SENSORS_MODE_RW,
                               GL520_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_GL520_FAN1OFF, "fan1_off", SENSORS_GL520_FAN1,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_FAN1OFF, VALUE(1), 0 },
    { SENSORS_GL520_TWOTEMPS, "two_temps", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_CONFIG, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm80_features[] =
  { 
    { SENSORS_LM80_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM80_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM80_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM80_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM80_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM80_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM80_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM80_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM80_IN0_MIN, "in0_min", SENSORS_LM80_IN0, SENSORS_LM80_IN0,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM80_IN1_MIN, "in1_min", SENSORS_LM80_IN1, SENSORS_LM80_IN1,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM80_IN2_MIN, "in2_min", SENSORS_LM80_IN2, SENSORS_LM80_IN2,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM80_IN3_MIN, "in3_min", SENSORS_LM80_IN3, SENSORS_LM80_IN3,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM80_IN4_MIN, "in4_min", SENSORS_LM80_IN4, SENSORS_LM80_IN4,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_LM80_IN5_MIN, "in5_min", SENSORS_LM80_IN5, SENSORS_LM80_IN5,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM80_IN6_MIN, "in6_min", SENSORS_LM80_IN6, SENSORS_LM80_IN6,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM80_IN0_MAX, "in0_max", SENSORS_LM80_IN0, SENSORS_LM80_IN0,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM80_IN1_MAX, "in1_max", SENSORS_LM80_IN1, SENSORS_LM80_IN1,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM80_IN2_MAX, "in2_max", SENSORS_LM80_IN2, SENSORS_LM80_IN2,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM80_IN3_MAX, "in3_max", SENSORS_LM80_IN3, SENSORS_LM80_IN3,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM80_IN4_MAX, "in4_max", SENSORS_LM80_IN4, SENSORS_LM80_IN4,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM80_IN5_MAX, "in5_max", SENSORS_LM80_IN5, SENSORS_LM80_IN5,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM80_IN6_MAX, "in6_max", SENSORS_LM80_IN6, SENSORS_LM80_IN6,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM80_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM80_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM80_FAN1_MIN, "fan1_min", SENSORS_LM80_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM80_FAN2_MIN, "fan2_min", SENSORS_LM80_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM80_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_TEMP, VALUE(5), 2 },
    { SENSORS_LM80_TEMP_HOT_HYST, "temp_hot_hyst", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(2), 2 },
    { SENSORS_LM80_TEMP_HOT_MAX, "temp_hot_max", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(1), 2 },
    { SENSORS_LM80_TEMP_OS_HYST, "temp_os_hyst", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(4), 2 },
    { SENSORS_LM80_TEMP_OS_MAX, "temp_os_max", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(3), 2 },
    { SENSORS_LM80_FAN1_DIV, "fan1_div", SENSORS_LM80_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM80_FAN2_DIV, "fan2_div", SENSORS_LM80_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM80_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM80_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature w83781d_features[] =
  { 
    { SENSORS_W83781D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83781D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83781D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83781D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83781D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83781D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83781D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83781D_IN0_MIN, "in0_min", SENSORS_W83781D_IN0, 
                        SENSORS_W83781D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83781D_IN1_MIN, "in1_min", SENSORS_W83781D_IN1, 
                        SENSORS_W83781D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83781D_IN2_MIN, "in2_min", SENSORS_W83781D_IN2, 
                        SENSORS_W83781D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83781D_IN3_MIN, "in3_min", SENSORS_W83781D_IN3, 
                        SENSORS_W83781D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83781D_IN4_MIN, "in4_min", SENSORS_W83781D_IN4, 
                        SENSORS_W83781D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_W83781D_IN5_MIN, "in5_min", SENSORS_W83781D_IN5, 
                        SENSORS_W83781D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83781D_IN6_MIN, "in6_min", SENSORS_W83781D_IN6, 
                        SENSORS_W83781D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83781D_IN0_MAX, "in0_max", SENSORS_W83781D_IN0, 
                        SENSORS_W83781D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83781D_IN1_MAX, "in1_max", SENSORS_W83781D_IN1, 
                        SENSORS_W83781D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83781D_IN2_MAX, "in2_max", SENSORS_W83781D_IN2, 
                        SENSORS_W83781D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83781D_IN3_MAX, "in3_max", SENSORS_W83781D_IN3, 
                        SENSORS_W83781D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83781D_IN4_MAX, "in4_max", SENSORS_W83781D_IN4, 
                        SENSORS_W83781D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83781D_IN5_MAX, "in5_max", SENSORS_W83781D_IN5, 
                        SENSORS_W83781D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83781D_IN6_MAX, "in6_max", SENSORS_W83781D_IN6, 
                        SENSORS_W83781D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83781D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83781D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83781D_FAN1_MIN, "fan1_min", SENSORS_W83781D_FAN1,
                        SENSORS_W83781D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83781D_FAN2_MIN, "fan2_min", SENSORS_W83781D_FAN2, 
                        SENSORS_W83781D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83781D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83781D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP1_HYST, "temp1_hyst", SENSORS_W83781D_TEMP1,
                         SENSORS_W83781D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP1_OVER, "temp1_over", SENSORS_W83781D_TEMP1,
                         SENSORS_W83781D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83781D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP2_HYST, "temp2_hyst", SENSORS_W83781D_TEMP2,
                         SENSORS_W83781D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP2_OVER, "temp2_over", SENSORS_W83781D_TEMP2,
                         SENSORS_W83781D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83781D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP3_HYST, "temp3_hyst", SENSORS_W83781D_TEMP3,
                         SENSORS_W83781D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP3_OVER, "temp3_over", SENSORS_W83781D_TEMP3,
                         SENSORS_W83781D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },

    { SENSORS_W83781D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_W83781D_FAN1_DIV, "fan1_div", SENSORS_W83781D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83781D_FAN2_DIV, "fan2_div", SENSORS_W83781D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83781D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83781D_BEEP_ENABLE, "beep_enable", SENSORS_W83781D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83781D_BEEPS, "beeps", SENSORS_W83781D_ALARMS,
                         SENSORS_W83781D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { 0 }
  };
  
static sensors_chip_feature as99127f_features[] =
  { 
    { SENSORS_W83782D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83782D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83782D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83782D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83782D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83782D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83782D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83782D_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_W83782D_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83782D_IN7_MIN, "in7_min", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_W83782D_IN8_MIN, "in8_min", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83782D_IN7_MAX, "in7_max", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_W83782D_IN8_MAX, "in8_max", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { SENSORS_W83782D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83782D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1,
                        SENSORS_W83782D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                        SENSORS_W83782D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83782D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83782D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },

    { SENSORS_W83782D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83782D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS,
                         SENSORS_W83782D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83782D_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83782D_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { SENSORS_W83782D_SENS3, "sensor3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS3, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature w83782d_features[] =
  { 
    { SENSORS_W83782D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83782D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83782D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83782D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83782D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83782D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83782D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83782D_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_W83782D_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83782D_IN7_MIN, "in7_min", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_W83782D_IN8_MIN, "in8_min", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83782D_IN7_MAX, "in7_max", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_W83782D_IN8_MAX, "in8_max", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { SENSORS_W83782D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83782D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1,
                        SENSORS_W83782D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                        SENSORS_W83782D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83782D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83782D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },

    { SENSORS_W83782D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83782D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS,
                         SENSORS_W83782D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83782D_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83782D_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { SENSORS_W83782D_SENS3, "sensor3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS3, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature w83783s_features[] =
  { 
    { SENSORS_W83783S_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83783S_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83783S_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83783S_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83783S_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83783S_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83783S_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83783S_IN0_MIN, "in0_min", SENSORS_W83783S_IN0, 
                        SENSORS_W83783S_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83783S_IN1_MIN, "in1_min", SENSORS_W83783S_IN1, 
                        SENSORS_W83783S_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83783S_IN2_MIN, "in2_min", SENSORS_W83783S_IN2, 
                        SENSORS_W83783S_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83783S_IN3_MIN, "in3_min", SENSORS_W83783S_IN3, 
                        SENSORS_W83783S_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83783S_IN4_MIN, "in4_min", SENSORS_W83783S_IN4, 
                        SENSORS_W83783S_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_W83783S_IN5_MIN, "in5_min", SENSORS_W83783S_IN5, 
                        SENSORS_W83783S_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83783S_IN6_MIN, "in6_min", SENSORS_W83783S_IN6, 
                        SENSORS_W83783S_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83783S_IN0_MAX, "in0_max", SENSORS_W83783S_IN0, 
                        SENSORS_W83783S_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83783S_IN1_MAX, "in1_max", SENSORS_W83783S_IN1, 
                        SENSORS_W83783S_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83783S_IN2_MAX, "in2_max", SENSORS_W83783S_IN2, 
                        SENSORS_W83783S_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83783S_IN3_MAX, "in3_max", SENSORS_W83783S_IN3, 
                        SENSORS_W83783S_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83783S_IN4_MAX, "in4_max", SENSORS_W83783S_IN4, 
                        SENSORS_W83783S_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83783S_IN5_MAX, "in5_max", SENSORS_W83783S_IN5, 
                        SENSORS_W83783S_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83783S_IN6_MAX, "in6_max", SENSORS_W83783S_IN6, 
                        SENSORS_W83783S_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83783S_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83783S_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83783S_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83783S_FAN1_MIN, "fan1_min", SENSORS_W83783S_FAN1,
                        SENSORS_W83783S_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83783S_FAN2_MIN, "fan2_min", SENSORS_W83783S_FAN2, 
                        SENSORS_W83783S_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83783S_FAN3_MIN, "fan3_min", SENSORS_W83783S_FAN3, 
                        SENSORS_W83783S_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83783S_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83783S_TEMP1_HYST, "temp1_hyst", SENSORS_W83783S_TEMP1,
                         SENSORS_W83783S_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83783S_TEMP1_OVER, "temp1_over", SENSORS_W83783S_TEMP1,
                         SENSORS_W83783S_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83783S_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83783S_TEMP2_HYST, "temp2_hyst", SENSORS_W83783S_TEMP2,
                         SENSORS_W83783S_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83783S_TEMP2_OVER, "temp2_over", SENSORS_W83783S_TEMP2,
                         SENSORS_W83783S_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83783S_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_W83783S_FAN1_DIV, "fan1_div", SENSORS_W83783S_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83783S_FAN2_DIV, "fan2_div", SENSORS_W83783S_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83783S_FAN3_DIV, "fan3_div", SENSORS_W83783S_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83783S_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83783S_BEEP_ENABLE, "beep_enable", SENSORS_W83783S_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83783S_BEEPS, "beeps", SENSORS_W83783S_ALARMS,
                         SENSORS_W83783S_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83783S_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83783S_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature adm9240_features[] =
  { 
    { SENSORS_ADM9240_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM9240_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM9240_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM9240_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM9240_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM9240_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM9240_IN0_MIN, "2.5V_min", SENSORS_ADM9240_IN0, 
                        SENSORS_ADM9240_IN0, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM9240_IN1_MIN, "Vccp1_min", SENSORS_ADM9240_IN1, 
                        SENSORS_ADM9240_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM9240_IN2_MIN, "3.3V_min", SENSORS_ADM9240_IN2, 
                        SENSORS_ADM9240_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM9240_IN3_MIN, "5V_min", SENSORS_ADM9240_IN3, 
                        SENSORS_ADM9240_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM9240_IN4_MIN, "12V_min", SENSORS_ADM9240_IN4, 
                        SENSORS_ADM9240_IN4, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM9240_IN5_MIN, "Vccp2_min", SENSORS_ADM9240_IN5, 
                        SENSORS_ADM9240_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM9240_IN0_MAX, "2.5V_max", SENSORS_ADM9240_IN0, 
                        SENSORS_ADM9240_IN0, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM9240_IN1_MAX, "Vccp1_max", SENSORS_ADM9240_IN1, 
                        SENSORS_ADM9240_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM9240_IN2_MAX, "3.3V_max", SENSORS_ADM9240_IN2, 
                        SENSORS_ADM9240_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM9240_IN3_MAX, "5V_max", SENSORS_ADM9240_IN3, 
                        SENSORS_ADM9240_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM9240_IN4_MAX, "12V_max", SENSORS_ADM9240_IN4, 
                        SENSORS_ADM9240_IN4, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM9240_IN5_MAX, "Vccp2_max", SENSORS_ADM9240_IN5, 
                        SENSORS_ADM9240_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM9240_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_ADM9240_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_ADM9240_FAN1_MIN, "fan1_min", SENSORS_ADM9240_FAN1,
                        SENSORS_ADM9240_FAN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_ADM9240_FAN2_MIN, "fan2_min", SENSORS_ADM9240_FAN2, 
                        SENSORS_ADM9240_FAN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_ADM9240_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM9240_TEMP_HYST, "temp_hyst", SENSORS_ADM9240_TEMP,
                         SENSORS_ADM9240_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM9240_TEMP_OVER, "temp_over", SENSORS_ADM9240_TEMP,
                         SENSORS_ADM9240_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(1), 1 },

    { SENSORS_ADM9240_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_ADM9240_FAN1_DIV, "fan1_div", SENSORS_ADM9240_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_ADM9240_FAN2_DIV, "fan2_div", SENSORS_ADM9240_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_ADM9240_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM9240_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM9240_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM9240_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature adm1024_features[] =
  { 
    { SENSORS_ADM1024_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM1024_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM1024_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM1024_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM1024_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM1024_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM1024_IN0_MIN, "2.5V_min", SENSORS_ADM1024_IN0, 
                        SENSORS_ADM1024_IN0, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM1024_IN1_MIN, "Vccp1_min", SENSORS_ADM1024_IN1, 
                        SENSORS_ADM1024_IN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM1024_IN2_MIN, "3.3V_min", SENSORS_ADM1024_IN2, 
                        SENSORS_ADM1024_IN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM1024_IN3_MIN, "5V_min", SENSORS_ADM1024_IN3, 
                        SENSORS_ADM1024_IN3, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM1024_IN4_MIN, "12V_min", SENSORS_ADM1024_IN4, 
                        SENSORS_ADM1024_IN4, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM1024_IN5_MIN, "Vccp2_min", SENSORS_ADM1024_IN5, 
                        SENSORS_ADM1024_IN5, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM1024_IN0_MAX, "2.5V_max", SENSORS_ADM1024_IN0, 
                        SENSORS_ADM1024_IN0, SENSORS_MODE_RW,
                        ADM1024_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM1024_IN1_MAX, "Vccp1_max", SENSORS_ADM1024_IN1, 
                        SENSORS_ADM1024_IN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM1024_IN2_MAX, "3.3V_max", SENSORS_ADM1024_IN2, 
                        SENSORS_ADM1024_IN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM1024_IN3_MAX, "5V_max", SENSORS_ADM1024_IN3, 
                        SENSORS_ADM1024_IN3, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM1024_IN4_MAX, "12V_max", SENSORS_ADM1024_IN4, 
                        SENSORS_ADM1024_IN4, SENSORS_MODE_RW,
                        ADM1024_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM1024_IN5_MAX, "Vccp2_max", SENSORS_ADM1024_IN5, 
                        SENSORS_ADM1024_IN5, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM1024_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_ADM1024_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_ADM1024_FAN1_MIN, "fan1_min", SENSORS_ADM1024_FAN1,
                        SENSORS_ADM1024_FAN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_ADM1024_FAN2_MIN, "fan2_min", SENSORS_ADM1024_FAN2, 
                        SENSORS_ADM1024_FAN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_ADM1024_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP_HYST, "temp_hyst", SENSORS_ADM1024_TEMP,
                         SENSORS_ADM1024_TEMP, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP_OVER, "temp_over", SENSORS_ADM1024_TEMP,
                         SENSORS_ADM1024_TEMP, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_ADM1024_TEMP1_HYST, "temp1_hyst", SENSORS_ADM1024_TEMP1,
                         SENSORS_ADM1024_TEMP1, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP1_OVER, "temp1_over", SENSORS_ADM1024_TEMP1,
                         SENSORS_ADM1024_TEMP1, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_ADM1024_TEMP2_HYST, "temp2_hyst", SENSORS_ADM1024_TEMP2,
                         SENSORS_ADM1024_TEMP2, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP2_OVER, "temp2_over", SENSORS_ADM1024_TEMP2,
                         SENSORS_ADM1024_TEMP2, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP2, VALUE(1), 1 },

    { SENSORS_ADM1024_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_ADM1024_FAN1_DIV, "fan1_div", SENSORS_ADM1024_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_ADM1024_FAN2_DIV, "fan2_div", SENSORS_ADM1024_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_ADM1024_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM1024_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM1024_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM1024_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ds1780_features[] =
  { 
    { SENSORS_DS1780_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_DS1780_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_DS1780_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_DS1780_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_DS1780_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_DS1780_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_DS1780_IN0_MIN, "2.5V_min", SENSORS_DS1780_IN0, 
                        SENSORS_DS1780_IN0, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_DS1780_IN1_MIN, "Vccp1_min", SENSORS_DS1780_IN1, 
                        SENSORS_DS1780_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_DS1780_IN2_MIN, "3.3V_min", SENSORS_DS1780_IN2, 
                        SENSORS_DS1780_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_DS1780_IN3_MIN, "5V_min", SENSORS_DS1780_IN3, 
                        SENSORS_DS1780_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_DS1780_IN4_MIN, "12V_min", SENSORS_DS1780_IN4, 
                        SENSORS_DS1780_IN4, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_DS1780_IN5_MIN, "Vccp2_min", SENSORS_DS1780_IN5, 
                        SENSORS_DS1780_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_DS1780_IN0_MAX, "2.5V_max", SENSORS_DS1780_IN0, 
                        SENSORS_DS1780_IN0, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_DS1780_IN1_MAX, "Vccp1_max", SENSORS_DS1780_IN1, 
                        SENSORS_DS1780_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_DS1780_IN2_MAX, "3.3V_max", SENSORS_DS1780_IN2, 
                        SENSORS_DS1780_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_DS1780_IN3_MAX, "5V_max", SENSORS_DS1780_IN3, 
                        SENSORS_DS1780_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_DS1780_IN4_MAX, "12V_max", SENSORS_DS1780_IN4, 
                        SENSORS_DS1780_IN4, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_DS1780_IN5_MAX, "Vccp2_max", SENSORS_DS1780_IN5, 
                        SENSORS_DS1780_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_DS1780_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_DS1780_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_DS1780_FAN1_MIN, "fan1_min", SENSORS_DS1780_FAN1,
                        SENSORS_DS1780_FAN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_DS1780_FAN2_MIN, "fan2_min", SENSORS_DS1780_FAN2, 
                        SENSORS_DS1780_FAN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_DS1780_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_DS1780_TEMP_HYST, "temp_hyst", SENSORS_DS1780_TEMP,
                         SENSORS_DS1780_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_DS1780_TEMP_OVER, "temp_over", SENSORS_DS1780_TEMP,
                         SENSORS_DS1780_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(1), 1 },

    { SENSORS_DS1780_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_DS1780_FAN1_DIV, "fan1_div", SENSORS_DS1780_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_DS1780_FAN2_DIV, "fan2_div", SENSORS_DS1780_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_DS1780_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM9240_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_DS1780_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM9240_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature sis5595_features[] =
  { 
    { SENSORS_SIS5595_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_SIS5595_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_SIS5595_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_SIS5595_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_SIS5595_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_SIS5595_IN0_MIN, "in0_min", SENSORS_SIS5595_IN0, SENSORS_SIS5595_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_SIS5595_IN1_MIN, "in1_min", SENSORS_SIS5595_IN1, SENSORS_SIS5595_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_SIS5595_IN2_MIN, "in2_min", SENSORS_SIS5595_IN2, SENSORS_SIS5595_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_SIS5595_IN3_MIN, "in3_min", SENSORS_SIS5595_IN3, SENSORS_SIS5595_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_SIS5595_IN4_MIN, "in4_min", SENSORS_SIS5595_IN4, SENSORS_SIS5595_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_SIS5595_IN0_MAX, "in0_max", SENSORS_SIS5595_IN0, SENSORS_SIS5595_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_SIS5595_IN1_MAX, "in1_max", SENSORS_SIS5595_IN1, SENSORS_SIS5595_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_SIS5595_IN2_MAX, "in2_max", SENSORS_SIS5595_IN2, SENSORS_SIS5595_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_SIS5595_IN3_MAX, "in3_max", SENSORS_SIS5595_IN3, SENSORS_SIS5595_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_SIS5595_IN4_MAX, "in4_max", SENSORS_SIS5595_IN4, SENSORS_SIS5595_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_SIS5595_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_SIS5595_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_SIS5595_FAN1_MIN, "fan1_min", SENSORS_SIS5595_FAN1, SENSORS_SIS5595_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_SIS5595_FAN2_MIN, "fan2_min", SENSORS_SIS5595_FAN2, SENSORS_SIS5595_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_SIS5595_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_SIS5595_TEMP_HYST, "temp_hyst", SENSORS_SIS5595_TEMP,
                              SENSORS_SIS5595_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_SIS5595_TEMP_OVER, "temp_over", SENSORS_SIS5595_TEMP,
                              SENSORS_SIS5595_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_SIS5595_FAN1_DIV, "fan1_div", SENSORS_SIS5595_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_SIS5595_FAN2_DIV, "fan2_div", SENSORS_SIS5595_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_SIS5595_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_cg_features[] =
  { 
    { SENSORS_MAXI_CG_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN1_MIN, "fan1_min", SENSORS_MAXI_CG_FAN1, SENSORS_MAXI_CG_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN2_MIN, "fan2_min", SENSORS_MAXI_CG_FAN2, SENSORS_MAXI_CG_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN3_MIN, "fan3_min", SENSORS_MAXI_CG_FAN3, SENSORS_MAXI_CG_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN1_DIV, "fan1_div", SENSORS_MAXI_CG_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_CG_FAN2_DIV, "fan2_div", SENSORS_MAXI_CG_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_CG_FAN3_DIV, "fan3_div", SENSORS_MAXI_CG_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_CG_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CG_TEMP1,
                              SENSORS_MAXI_CG_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CG_TEMP2,
                              SENSORS_MAXI_CG_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CG_TEMP3,
                              SENSORS_MAXI_CG_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CG_TEMP4,
                              SENSORS_MAXI_CG_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CG_TEMP5,
                              SENSORS_MAXI_CG_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CG_TEMP1,
                              SENSORS_MAXI_CG_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CG_TEMP2,
                              SENSORS_MAXI_CG_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CG_TEMP3,
                              SENSORS_MAXI_CG_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CG_TEMP4,
                              SENSORS_MAXI_CG_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CG_TEMP5,
                              SENSORS_MAXI_CG_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_CG_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_CG_PLL_MIN, "pll_min", SENSORS_MAXI_CG_PLL, SENSORS_MAXI_CG_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_CG_PLL_MAX, "pll_max", SENSORS_MAXI_CG_PLL, SENSORS_MAXI_CG_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_CG_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID1_MIN, "vid1_min", SENSORS_MAXI_CG_VID1, SENSORS_MAXI_CG_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID2_MIN, "vid2_min", SENSORS_MAXI_CG_VID2, SENSORS_MAXI_CG_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID3_MIN, "vid3_min", SENSORS_MAXI_CG_VID3, SENSORS_MAXI_CG_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID4_MIN, "vid4_min", SENSORS_MAXI_CG_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID1_MAX, "vid1_max", SENSORS_MAXI_CG_VID1, SENSORS_MAXI_CG_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID2_MAX, "vid2_max", SENSORS_MAXI_CG_VID2, SENSORS_MAXI_CG_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CG_VID3, SENSORS_MAXI_CG_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID4_MAX, "vid4_max", SENSORS_MAXI_CG_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_CG_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_co_features[] =
  { 
    { SENSORS_MAXI_CO_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN1_MIN, "fan1_min", SENSORS_MAXI_CO_FAN1, SENSORS_MAXI_CO_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN2_MIN, "fan2_min", SENSORS_MAXI_CO_FAN2, SENSORS_MAXI_CO_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN3_MIN, "fan3_min", SENSORS_MAXI_CO_FAN3, SENSORS_MAXI_CO_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN1_DIV, "fan1_div", SENSORS_MAXI_CO_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_CO_FAN2_DIV, "fan2_div", SENSORS_MAXI_CO_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_CO_FAN3_DIV, "fan3_div", SENSORS_MAXI_CO_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_CO_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CO_TEMP1,
                              SENSORS_MAXI_CO_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CO_TEMP2,
                              SENSORS_MAXI_CO_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CO_TEMP3,
                              SENSORS_MAXI_CO_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CO_TEMP4,
                              SENSORS_MAXI_CO_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CO_TEMP5,
                              SENSORS_MAXI_CO_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CO_TEMP1,
                              SENSORS_MAXI_CO_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CO_TEMP2,
                              SENSORS_MAXI_CO_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CO_TEMP3,
                              SENSORS_MAXI_CO_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CO_TEMP4,
                              SENSORS_MAXI_CO_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CO_TEMP5,
                              SENSORS_MAXI_CO_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_CO_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_CO_PLL_MIN, "pll_min", SENSORS_MAXI_CO_PLL, SENSORS_MAXI_CO_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_CO_PLL_MAX, "pll_max", SENSORS_MAXI_CO_PLL, SENSORS_MAXI_CO_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_CO_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID1_MIN, "vid1_min", SENSORS_MAXI_CO_VID1, SENSORS_MAXI_CO_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID2_MIN, "vid2_min", SENSORS_MAXI_CO_VID2, SENSORS_MAXI_CO_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID3_MIN, "vid3_min", SENSORS_MAXI_CO_VID3, SENSORS_MAXI_CO_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID4_MIN, "vid4_min", SENSORS_MAXI_CO_VID4, SENSORS_MAXI_CO_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID1_MAX, "vid1_max", SENSORS_MAXI_CO_VID1, SENSORS_MAXI_CO_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_CO_VID2_MAX, "vid2_max", SENSORS_MAXI_CO_VID2, SENSORS_MAXI_CO_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CO_VID3, SENSORS_MAXI_CO_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_CO_VID4_MAX, "vid4_max", SENSORS_MAXI_CO_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_CO_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_as_features[] =
  { 
    { SENSORS_MAXI_AS_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN1_MIN, "fan1_min", SENSORS_MAXI_AS_FAN1, SENSORS_MAXI_AS_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN2_MIN, "fan2_min", SENSORS_MAXI_AS_FAN2, SENSORS_MAXI_AS_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN3_MIN, "fan3_min", SENSORS_MAXI_AS_FAN3, SENSORS_MAXI_AS_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN1_DIV, "fan1_div", SENSORS_MAXI_AS_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_AS_FAN2_DIV, "fan2_div", SENSORS_MAXI_AS_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_AS_FAN3_DIV, "fan3_div", SENSORS_MAXI_AS_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_AS_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP1_MAX, "temp1_max", SENSORS_MAXI_AS_TEMP1,
                              SENSORS_MAXI_AS_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP2_MAX, "temp2_max", SENSORS_MAXI_AS_TEMP2,
                              SENSORS_MAXI_AS_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP3_MAX, "temp3_max", SENSORS_MAXI_AS_TEMP3,
                              SENSORS_MAXI_AS_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP4_MAX, "temp4_max", SENSORS_MAXI_AS_TEMP4,
                              SENSORS_MAXI_AS_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP5_MAX, "temp5_max", SENSORS_MAXI_AS_TEMP5,
                              SENSORS_MAXI_AS_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_AS_TEMP1,
                              SENSORS_MAXI_AS_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_AS_TEMP2,
                              SENSORS_MAXI_AS_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_AS_TEMP3,
                              SENSORS_MAXI_AS_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_AS_TEMP4,
                              SENSORS_MAXI_AS_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_AS_TEMP5,
                              SENSORS_MAXI_AS_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_AS_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_AS_PLL_MIN, "pll_min", SENSORS_MAXI_AS_PLL, SENSORS_MAXI_AS_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_AS_PLL_MAX, "pll_max", SENSORS_MAXI_AS_PLL, SENSORS_MAXI_AS_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_AS_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID1_MIN, "vid1_min", SENSORS_MAXI_AS_VID1, SENSORS_MAXI_AS_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID2_MIN, "vid2_min", SENSORS_MAXI_AS_VID2, SENSORS_MAXI_AS_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID3_MIN, "vid3_min", SENSORS_MAXI_AS_VID3, SENSORS_MAXI_AS_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID4_MIN, "vid4_min", SENSORS_MAXI_AS_VID4, SENSORS_MAXI_AS_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID1_MAX, "vid1_max", SENSORS_MAXI_AS_VID1, SENSORS_MAXI_AS_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID2_MAX, "vid2_max", SENSORS_MAXI_AS_VID2, SENSORS_MAXI_AS_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID3_MAX, "vid3_max", SENSORS_MAXI_AS_VID3, SENSORS_MAXI_AS_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID4_MAX, "vid4_max", SENSORS_MAXI_AS_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_AS_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature thmc50_features[] =
  {
    { SENSORS_THMC50_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, THMC50_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_THMC50_TEMP_HYST, "temp_hyst", SENSORS_THMC50_TEMP,
                              SENSORS_THMC50_TEMP, SENSORS_MODE_RW, 
                              THMC50_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_THMC50_TEMP_OVER, "temp_over", SENSORS_THMC50_TEMP,
                              SENSORS_THMC50_TEMP, SENSORS_MODE_RW,
                              THMC50_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_THMC50_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_THMC50_REMOTE_TEMP_HYST, "remote_temp_hyst", 
                              SENSORS_THMC50_REMOTE_TEMP,
                              SENSORS_THMC50_REMOTE_TEMP, SENSORS_MODE_RW, 
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_THMC50_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_THMC50_REMOTE_TEMP,
                              SENSORS_THMC50_REMOTE_TEMP, SENSORS_MODE_RW,
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_THMC50_INTER_MASK, "analog output", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              THMC50_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { SENSORS_THMC50_INTER, "interrupts", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              THMC50_SYSCTL_INTER, VALUE(1), 0 },
    { SENSORS_THMC50_INTER_MASK, "interrupt mask", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              THMC50_SYSCTL_INTER_MASK, VALUE(1), 0 },
    { SENSORS_THMC50_DIE_CODE, "die_code", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              THMC50_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature adm1025_features[] =
  { 
    { SENSORS_ADM1025_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM1025_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM1025_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM1025_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM1025_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM1025_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM1025_IN0_MIN, "in0_min", SENSORS_ADM1025_IN0, 
                        SENSORS_ADM1025_IN0, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM1025_IN1_MIN, "in1_min", SENSORS_ADM1025_IN1, 
                        SENSORS_ADM1025_IN1, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM1025_IN2_MIN, "in2_min", SENSORS_ADM1025_IN2, 
                        SENSORS_ADM1025_IN2, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM1025_IN3_MIN, "in3_min", SENSORS_ADM1025_IN3, 
                        SENSORS_ADM1025_IN3, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM1025_IN4_MIN, "in4_min", SENSORS_ADM1025_IN4, 
                        SENSORS_ADM1025_IN4, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM1025_IN5_MIN, "in5_min", SENSORS_ADM1025_IN5, 
                        SENSORS_ADM1025_IN5, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM1025_IN0_MAX, "in0_max", SENSORS_ADM1025_IN0, 
                        SENSORS_ADM1025_IN0, SENSORS_MODE_RW,
                        ADM1025_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM1025_IN1_MAX, "in1_max", SENSORS_ADM1025_IN1, 
                        SENSORS_ADM1025_IN1, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM1025_IN2_MAX, "in2_max", SENSORS_ADM1025_IN2, 
                        SENSORS_ADM1025_IN2, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM1025_IN3_MAX, "in3_max", SENSORS_ADM1025_IN3, 
                        SENSORS_ADM1025_IN3, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM1025_IN4_MAX, "in4_max", SENSORS_ADM1025_IN4, 
                        SENSORS_ADM1025_IN4, SENSORS_MODE_RW,
                        ADM1025_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM1025_IN5_MAX, "in5_max", SENSORS_ADM1025_IN5, 
                        SENSORS_ADM1025_IN5, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM1025_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM1025_TEMP1_HYST, "temp1_hyst", SENSORS_ADM1025_TEMP1,
                         SENSORS_ADM1025_TEMP1, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM1025_TEMP1_OVER, "temp1_over", SENSORS_ADM1025_TEMP1,
                         SENSORS_ADM1025_TEMP1, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_ADM1025_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_RTEMP, VALUE(3), 1 },
    { SENSORS_ADM1025_TEMP2_HYST, "temp2_hyst", SENSORS_ADM1025_TEMP2,
                         SENSORS_ADM1025_TEMP2, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_RTEMP, VALUE(2), 1 },
    { SENSORS_ADM1025_TEMP2_OVER, "temp2_over", SENSORS_ADM1025_TEMP2,
                         SENSORS_ADM1025_TEMP2, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_RTEMP, VALUE(1), 1 },
    { SENSORS_ADM1025_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_ADM1025_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM1025_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature via686a_features[] =
  { 
    { SENSORS_VIA686A_IN0, "2.0V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_VIA686A_IN1, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_VIA686A_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_VIA686A_IN3, "5.0V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_VIA686A_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_VIA686A_IN0_MIN, "in0_min", SENSORS_VIA686A_IN0,
                            SENSORS_VIA686A_IN0,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_VIA686A_IN1_MIN, "in1_min", SENSORS_VIA686A_IN1,
                            SENSORS_VIA686A_IN1,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_VIA686A_IN2_MIN, "in2_min", SENSORS_VIA686A_IN2,
                            SENSORS_VIA686A_IN2,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_VIA686A_IN3_MIN, "in3_min", SENSORS_VIA686A_IN3,
                            SENSORS_VIA686A_IN3,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_VIA686A_IN4_MIN, "in4_min", SENSORS_VIA686A_IN4,
                            SENSORS_VIA686A_IN4,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_VIA686A_IN0_MAX, "in0_max", SENSORS_VIA686A_IN0,
                            SENSORS_VIA686A_IN0,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_VIA686A_IN1_MAX, "in1_max", SENSORS_VIA686A_IN1,
                            SENSORS_VIA686A_IN1,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_VIA686A_IN2_MAX, "in2_max", SENSORS_VIA686A_IN2,
                            SENSORS_VIA686A_IN2,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_VIA686A_IN3_MAX, "in3_max", SENSORS_VIA686A_IN3,
                            SENSORS_VIA686A_IN3,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_VIA686A_IN4_MAX, "in4_max", SENSORS_VIA686A_IN4,
                            SENSORS_VIA686A_IN4,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_VIA686A_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_VIA686A_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_VIA686A_FAN1_MIN, "fan1_min", SENSORS_VIA686A_FAN1,
                             SENSORS_VIA686A_FAN1,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_VIA686A_FAN2_MIN, "fan2_min", SENSORS_VIA686A_FAN2,
                             SENSORS_VIA686A_FAN2,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_VIA686A_TEMP, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP_HYST, "temp1_hyst", SENSORS_VIA686A_TEMP,
                              SENSORS_VIA686A_TEMP, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP_OVER, "temp1_over", SENSORS_VIA686A_TEMP,
                              SENSORS_VIA686A_TEMP, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_VIA686A_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP2_HYST, "temp2_hyst", SENSORS_VIA686A_TEMP2,
                              SENSORS_VIA686A_TEMP2, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP2_OVER, "temp2_over", SENSORS_VIA686A_TEMP2,
                              SENSORS_VIA686A_TEMP2, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_VIA686A_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP3_HYST, "temp3_hyst", SENSORS_VIA686A_TEMP3,
                              SENSORS_VIA686A_TEMP3, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP3_OVER, "temp3_over", SENSORS_VIA686A_TEMP3,
                              SENSORS_VIA686A_TEMP3, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_VIA686A_FAN1_DIV, "fan1_div", SENSORS_VIA686A_FAN1,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_VIA686A_FAN2_DIV, "fan2_div", SENSORS_VIA686A_FAN2,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_VIA686A_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VIA686A_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ddcmon_features[] =
  {
    { SENSORS_DDCMON_ID, "Monitor ID",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_ID, VALUE(1), 0 },
    { SENSORS_DDCMON_VERSIZE, "Monitor Size (cm)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SIZE, VALUE(1), 0 },
    { SENSORS_DDCMON_HORSIZE, "hsize",
                         SENSORS_DDCMON_VERSIZE, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SIZE, VALUE(2), 0 },
    { SENSORS_DDCMON_VERSYNCMIN, "Vertical Sync (Hz)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(1), 0 },
    { SENSORS_DDCMON_VERSYNCMAX, "v_sync_max",
                         SENSORS_DDCMON_VERSYNCMIN, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(2), 0 },
    { SENSORS_DDCMON_HORSYNCMIN, "Horizontal Sync (KHz)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(3), 0 },
    { SENSORS_DDCMON_HORSYNCMAX, "h_sync_max",
                         SENSORS_DDCMON_HORSYNCMIN, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(4), 0 },
    { SENSORS_DDCMON_TIMINGS, "timings",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMINGS, VALUE(1), 0 },
    { SENSORS_DDCMON_SERIAL, "Serial number",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SERIAL, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature eeprom_features[] =
  {
    { SENSORS_EEPROM_TYPE, "Memory type",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(3), 0 },
/* cheat and use the row-address-bits label for the total size in sensors */
    { SENSORS_EEPROM_ROWADDR, "SDRAM Size (MB)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(4), 0 },
    { SENSORS_EEPROM_COLADDR, "col",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(5), 0 },
    { SENSORS_EEPROM_NUMROWS, "num rows",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(6), 0 },
    { SENSORS_EEPROM_BANKS, "banks",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL2, VALUE(2), 0 },
    { 0 }
  };

static sensors_chip_feature lm87_features[] =
  { 
    { SENSORS_LM87_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM87_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM87_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM87_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM87_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM87_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM87_AIN1, "AIN1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_AIN1, VALUE(3), 0 },
    { SENSORS_LM87_AIN2, "AIN2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_AIN2, VALUE(3), 0 },
    { SENSORS_LM87_IN0_MIN, "AmbTemp_min", SENSORS_LM87_IN0, 
                        SENSORS_LM87_IN0, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM87_IN1_MIN, "Vccp1_min", SENSORS_LM87_IN1, 
                        SENSORS_LM87_IN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM87_IN2_MIN, "3.3V_min", SENSORS_LM87_IN2, 
                        SENSORS_LM87_IN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM87_IN3_MIN, "5V_min", SENSORS_LM87_IN3, 
                        SENSORS_LM87_IN3, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM87_IN4_MIN, "12V_min", SENSORS_LM87_IN4, 
                        SENSORS_LM87_IN4, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_LM87_IN5_MIN, "Vccp2_min", SENSORS_LM87_IN5, 
                        SENSORS_LM87_IN5, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM87_AIN1_MIN, "AIN1_min", SENSORS_LM87_AIN1, 
                        SENSORS_LM87_AIN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN1, VALUE(1), 0 },
    { SENSORS_LM87_AIN2_MIN, "AIN2_min", SENSORS_LM87_AIN2, 
                        SENSORS_LM87_AIN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN2, VALUE(1), 0 },
    { SENSORS_LM87_IN0_MAX, "AmbTemp_max", SENSORS_LM87_IN0, 
                        SENSORS_LM87_IN0, SENSORS_MODE_RW,
                        LM87_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM87_IN1_MAX, "Vccp1_max", SENSORS_LM87_IN1, 
                        SENSORS_LM87_IN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM87_IN2_MAX, "3.3V_max", SENSORS_LM87_IN2, 
                        SENSORS_LM87_IN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM87_IN3_MAX, "5V_max", SENSORS_LM87_IN3, 
                        SENSORS_LM87_IN3, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM87_IN4_MAX, "12V_max", SENSORS_LM87_IN4, 
                        SENSORS_LM87_IN4, SENSORS_MODE_RW,
                        LM87_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM87_IN5_MAX, "Vccp2_max", SENSORS_LM87_IN5, 
                        SENSORS_LM87_IN5, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM87_AIN1_MAX, "AIN1_max", SENSORS_LM87_AIN1, 
                        SENSORS_LM87_AIN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN1, VALUE(2), 0 },
    { SENSORS_LM87_AIN2_MAX, "AIN2_max", SENSORS_LM87_AIN2, 
                        SENSORS_LM87_AIN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN2, VALUE(2), 0 },
    { SENSORS_LM87_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM87_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM87_FAN1_MIN, "fan1_min", SENSORS_LM87_FAN1,
                        SENSORS_LM87_FAN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM87_FAN2_MIN, "fan2_min", SENSORS_LM87_FAN2, 
                        SENSORS_LM87_FAN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM87_TEMP3, "CPU2_Temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_LM87_TEMP2, "CPU_Temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM87_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_LM87_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_LM87_TEMP1_HYST, "temp1_min", SENSORS_LM87_TEMP1,
                         SENSORS_LM87_TEMP1, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_LM87_TEMP1_OVER, "temp1_max", SENSORS_LM87_TEMP1,
                         SENSORS_LM87_TEMP1, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_LM87_TEMP2_HYST, "temp2_min", SENSORS_LM87_TEMP2,
                         SENSORS_LM87_TEMP2, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_LM87_TEMP2_OVER, "temp2_max", SENSORS_LM87_TEMP2,
                         SENSORS_LM87_TEMP2, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_LM87_TEMP3_HYST, "temp3_min", SENSORS_LM87_TEMP3,
                         SENSORS_LM87_TEMP3, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_LM87_TEMP3_OVER, "temp3_max", SENSORS_LM87_TEMP3,
                         SENSORS_LM87_TEMP3, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_LM87_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_VID, VALUE(1), 2 },
/* Ho hum, this will be wrong if fan1 is disabled, but fan2 isn't.. fix?? */
    { SENSORS_LM87_FAN1_DIV, "fan1_div", SENSORS_LM87_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM87_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_LM87_FAN2_DIV, "fan2_div", SENSORS_LM87_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM87_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_LM87_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, LM87_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_LM87_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         LM87_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature mtp008_features[] =
  { 
    { SENSORS_MTP008_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_MTP008_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_MTP008_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_MTP008_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_MTP008_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_MTP008_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_MTP008_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_MTP008_IN0_MIN, "in0_min", SENSORS_MTP008_IN0, 
                        SENSORS_MTP008_IN0, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_MTP008_IN1_MIN, "in1_min", SENSORS_MTP008_IN1, 
                        SENSORS_MTP008_IN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_MTP008_IN2_MIN, "in2_min", SENSORS_MTP008_IN2, 
                        SENSORS_MTP008_IN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_MTP008_IN3_MIN, "in3_min", SENSORS_MTP008_IN3, 
                        SENSORS_MTP008_IN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_MTP008_IN4_MIN, "in4_min", SENSORS_MTP008_IN4, 
                        SENSORS_MTP008_IN4, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_MTP008_IN5_MIN, "in5_min", SENSORS_MTP008_IN5, 
                        SENSORS_MTP008_IN5, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_MTP008_IN6_MIN, "in6_min", SENSORS_MTP008_IN6, 
                        SENSORS_MTP008_IN6, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_MTP008_IN0_MAX, "in0_max", SENSORS_MTP008_IN0, 
                        SENSORS_MTP008_IN0, SENSORS_MODE_RW,
                        MTP008_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_MTP008_IN1_MAX, "in1_max", SENSORS_MTP008_IN1, 
                        SENSORS_MTP008_IN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_MTP008_IN2_MAX, "in2_max", SENSORS_MTP008_IN2, 
                        SENSORS_MTP008_IN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_MTP008_IN3_MAX, "in3_max", SENSORS_MTP008_IN3, 
                        SENSORS_MTP008_IN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_MTP008_IN4_MAX, "in4_max", SENSORS_MTP008_IN4, 
                        SENSORS_MTP008_IN4, SENSORS_MODE_RW,
                        MTP008_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_MTP008_IN5_MAX, "in5_max", SENSORS_MTP008_IN5, 
                        SENSORS_MTP008_IN5, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_MTP008_IN6_MAX, "in6_max", SENSORS_MTP008_IN6, 
                        SENSORS_MTP008_IN6, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_MTP008_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MTP008_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MTP008_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MTP008_FAN1_MIN, "fan1_min", SENSORS_MTP008_FAN1,
                        SENSORS_MTP008_FAN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MTP008_FAN2_MIN, "fan2_min", SENSORS_MTP008_FAN2, 
                        SENSORS_MTP008_FAN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MTP008_FAN3_MIN, "fan3_min", SENSORS_MTP008_FAN3, 
                        SENSORS_MTP008_FAN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MTP008_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP1_OVER, "temp1_over", SENSORS_MTP008_TEMP1,
			SENSORS_MTP008_TEMP1, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP2_OVER, "temp2_over", SENSORS_MTP008_TEMP2,
			SENSORS_MTP008_TEMP2, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP3_OVER, "temp3_over", SENSORS_MTP008_TEMP3,
			SENSORS_MTP008_TEMP3, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP1_HYST, "temp1_hyst", SENSORS_MTP008_TEMP1,
			SENSORS_MTP008_TEMP1, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MTP008_TEMP2_HYST, "temp2_hyst", SENSORS_MTP008_TEMP2,
			SENSORS_MTP008_TEMP2, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MTP008_TEMP3_HYST, "temp3_hyst", SENSORS_MTP008_TEMP3,
			SENSORS_MTP008_TEMP3, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MTP008_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MTP008_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_MTP008_FAN1_DIV, "fan1_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_MTP008_FAN2_DIV, "fan2_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_MTP008_FAN3_DIV, "fan3_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_MTP008_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, MTP008_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_MTP008_BEEP, "beeps", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, MTP008_SYSCTL_BEEP, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ds1621_features[] =
  {
    { SENSORS_DS1621_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DS1621_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_DS1621_TEMP_HYST, "temp_hyst", SENSORS_DS1621_TEMP,
                              SENSORS_DS1621_TEMP, SENSORS_MODE_RW, 
                              DS1621_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_DS1621_TEMP_OVER, "temp_over", SENSORS_DS1621_TEMP,
                              SENSORS_DS1621_TEMP, SENSORS_MODE_RW, 
                              DS1621_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_DS1621_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                           SENSORS_MODE_R, DS1621_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature it87_features[] =
  {
    { SENSORS_IT87_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_IT87_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_IT87_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_IT87_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_IT87_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_IT87_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_IT87_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_IT87_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_IT87_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_IT87_IN0_MIN, "in0_min", SENSORS_IT87_IN0, SENSORS_IT87_IN0,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_IT87_IN1_MIN, "in1_min", SENSORS_IT87_IN1, SENSORS_IT87_IN1,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_IT87_IN2_MIN, "in2_min", SENSORS_IT87_IN2, SENSORS_IT87_IN2,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_IT87_IN3_MIN, "in3_min", SENSORS_IT87_IN3, SENSORS_IT87_IN3,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_IT87_IN4_MIN, "in4_min", SENSORS_IT87_IN4, SENSORS_IT87_IN4,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_IT87_IN5_MIN, "in5_min", SENSORS_IT87_IN5, SENSORS_IT87_IN5,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_IT87_IN6_MIN, "in6_min", SENSORS_IT87_IN6, SENSORS_IT87_IN6,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_IT87_IN7_MIN, "in7_min", SENSORS_IT87_IN7, SENSORS_IT87_IN7,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_IT87_IN0_MAX, "in0_max", SENSORS_IT87_IN0, SENSORS_IT87_IN0,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_IT87_IN1_MAX, "in1_max", SENSORS_IT87_IN1, SENSORS_IT87_IN1,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_IT87_IN2_MAX, "in2_max", SENSORS_IT87_IN2, SENSORS_IT87_IN2,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_IT87_IN3_MAX, "in3_max", SENSORS_IT87_IN3, SENSORS_IT87_IN3,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_IT87_IN4_MAX, "in4_max", SENSORS_IT87_IN4, SENSORS_IT87_IN4,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_IT87_IN5_MAX, "in5_max", SENSORS_IT87_IN5, SENSORS_IT87_IN5,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_IT87_IN6_MAX, "in6_max", SENSORS_IT87_IN6, SENSORS_IT87_IN6,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_IT87_IN7_MAX, "in7_max", SENSORS_IT87_IN7, SENSORS_IT87_IN7,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_IT87_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_IT87_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_IT87_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_IT87_FAN1_MIN, "fan1_min", SENSORS_IT87_FAN1, SENSORS_IT87_FAN1,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_IT87_FAN2_MIN, "fan2_min", SENSORS_IT87_FAN2, SENSORS_IT87_FAN2,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_IT87_FAN3_MIN, "fan3_min", SENSORS_IT87_FAN3, SENSORS_IT87_FAN3,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_IT87_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_IT87_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_IT87_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_IT87_TEMP1_LOW, "temp1_hyst", SENSORS_IT87_TEMP1,
                              SENSORS_IT87_TEMP1, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_IT87_TEMP2_LOW, "temp2_hyst", SENSORS_IT87_TEMP2,
                              SENSORS_IT87_TEMP2, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_IT87_TEMP3_LOW, "temp3_hyst", SENSORS_IT87_TEMP3,
                              SENSORS_IT87_TEMP3, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_IT87_TEMP1_HIGH, "temp1_over", SENSORS_IT87_TEMP1,
                              SENSORS_IT87_TEMP1, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_IT87_TEMP2_HIGH, "temp2_over", SENSORS_IT87_TEMP2,
                              SENSORS_IT87_TEMP2, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_IT87_TEMP3_HIGH, "temp3_over", SENSORS_IT87_TEMP3,
                              SENSORS_IT87_TEMP3, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_IT87_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_IT87_FAN1_DIV, "fan1_div", SENSORS_IT87_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN_DIV, VALUE(1),
                             0 },
    { SENSORS_IT87_FAN2_DIV, "fan2_div", SENSORS_IT87_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN_DIV, VALUE(2),
                             0 },
    { SENSORS_IT87_FAN3_DIV, "fan3_div", SENSORS_IT87_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, IT87_SYSCTL_FAN_DIV, VALUE(3),
                              0 },
    { SENSORS_IT87_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                           SENSORS_MODE_R, IT87_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature fscpos_features[] =
  {
    { SENSORS_FSCPOS_REV, "rev", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_REV, VALUE(1), 0 },
    { SENSORS_FSCPOS_EVENT, "event", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_EVENT, VALUE(1), 0 },
    { SENSORS_FSCPOS_CONTROL, "control", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_RW, FSCPOS_SYSCTL_CONTROL, VALUE(1), 0 },
    { SENSORS_FSCPOS_VOLTAGE1, "volt12", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_VOLT0, VALUE(1), 2 },
    { SENSORS_FSCPOS_VOLTAGE2, "volt5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_VOLT1, VALUE(1), 2 },
    { SENSORS_FSCPOS_VOLTAGE3, "voltbatt", SENSORS_NO_MAPPING,
			SENSORS_NO_MAPPING, SENSORS_MODE_R, 
			FSCPOS_SYSCTL_VOLT2, VALUE(1), 2 },
    { SENSORS_FSCPOS_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP0, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP2, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP1_STATE, "temp1_state", SENSORS_FSCPOS_TEMP1,
			SENSORS_FSCPOS_TEMP1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP0, VALUE(1), 0 },
    { SENSORS_FSCPOS_TEMP2_STATE, "temp2_state", SENSORS_FSCPOS_TEMP2, 
			SENSORS_FSCPOS_TEMP2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_FSCPOS_TEMP3_STATE, "temp3_state", SENSORS_FSCPOS_TEMP3, 
			SENSORS_FSCPOS_TEMP3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP2, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN0, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN1, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN2, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN1_MIN, "fan1_min", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(2), 0 },
    { SENSORS_FSCPOS_FAN2_MIN, "fan2_min", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_FSCPOS_FAN1_STATE, "fan1_state", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN2_STATE, "fan2_state", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN3_STATE, "fan3_state", SENSORS_FSCPOS_FAN3, 
			SENSORS_FSCPOS_FAN3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(3), 0 },
    { SENSORS_FSCPOS_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_FSCPOS_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCPOS_FAN3, 
			SENSORS_FSCPOS_FAN3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_FSCPOS_WDOG_PRESET, "wdog_preset", SENSORS_NO_MAPPING, 
			SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(1), 0 },
    { SENSORS_FSCPOS_WDOG_STATE, "wdog_state", SENSORS_FSCPOS_WDOG_PRESET, 
			SENSORS_FSCPOS_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(2), 0 },
    { SENSORS_FSCPOS_WDOG_CONTROL, "wdog_control", SENSORS_FSCPOS_WDOG_PRESET, 
			SENSORS_FSCPOS_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(3), 0 },
    { 0 }
  };

static sensors_chip_feature fscscy_features[] =
  {
    { SENSORS_FSCSCY_REV, "rev", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_REV, VALUE(1), 0 },
    { SENSORS_FSCSCY_EVENT, "event", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_EVENT, VALUE(1), 0 },
    { SENSORS_FSCSCY_CONTROL, "control", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_RW, FSCSCY_SYSCTL_CONTROL, VALUE(1), 0 },
    { SENSORS_FSCSCY_VOLTAGE1, "volt12", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_VOLT0, VALUE(1), 2 },
    { SENSORS_FSCSCY_VOLTAGE2, "volt5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_VOLT1, VALUE(1), 2 },
    { SENSORS_FSCSCY_VOLTAGE3, "voltbatt", SENSORS_NO_MAPPING,
			SENSORS_NO_MAPPING, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_VOLT2, VALUE(1), 2 },
    { SENSORS_FSCSCY_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP0, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP2, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP3, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP1_STATE, "temp1_state", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP0, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP2_STATE, "temp2_state", SENSORS_FSCSCY_TEMP2, 
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP3_STATE, "temp3_state", SENSORS_FSCSCY_TEMP3, 
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP2, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP4_STATE, "temp4_state", SENSORS_FSCSCY_TEMP4, 
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP3, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP1_LIM, "temp1_lim", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP2_LIM, "temp2_lim", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP3_LIM, "temp3_lim", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP4_LIM, "temp4_lim", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP1_MIN, "temp1_min", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP2_MIN, "temp2_min", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP3_MIN, "temp3_min", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP4_MIN, "temp4_min", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP1_MAX, "temp1_max", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP2_MAX, "temp2_max", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP3_MAX, "temp3_max", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP4_MAX, "temp4_max", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(5), 0 },
    { SENSORS_FSCSCY_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN0, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN1, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN2, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN4, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN3, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN5, "fan5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN4, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN6, "fan6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN5, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN1_MIN, "fan1_min", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN2_MIN, "fan2_min", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN3_MIN, "fan3_min", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN4_MIN, "fan4_min", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN5_MIN, "fan5_min", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN6_MIN, "fan6_min", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN1_STATE, "fan1_state", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN2_STATE, "fan2_state", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN3_STATE, "fan3_state", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN4_STATE, "fan4_state", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN5_STATE, "fan5_state", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN6_STATE, "fan6_state", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN4_RIPPLE, "fan4_ripple", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN5_RIPPLE, "fan5_ripple", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN6_RIPPLE, "fan6_ripple", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(3), 0 },
    { SENSORS_FSCSCY_WDOG_PRESET, "wdog_preset", SENSORS_NO_MAPPING, 
			SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(1), 0 },
    { SENSORS_FSCSCY_WDOG_STATE, "wdog_state", SENSORS_FSCSCY_WDOG_PRESET, 
			SENSORS_FSCSCY_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(2), 0 },
    { SENSORS_FSCSCY_WDOG_CONTROL, "wdog_control", SENSORS_FSCSCY_WDOG_PRESET, 
			SENSORS_FSCSCY_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(3), 0 },
    { 0 }
  };

  static sensors_chip_feature pcf8591_features[] =
  {
    { SENSORS_PCF8591_AIN_CONF, "ain_conf", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_RW, PCF8591_SYSCTL_AIN_CONF, VALUE(1), 0 },
    { SENSORS_PCF8591_CH0, "ch0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH0, VALUE(1), 0 },
    { SENSORS_PCF8591_CH1, "ch1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH1, VALUE(1), 0 },
    { SENSORS_PCF8591_CH2, "ch2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH2, VALUE(1), 0 },
    { SENSORS_PCF8591_CH3, "ch3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH3, VALUE(1), 0 },
    { SENSORS_PCF8591_AOUT_ENABLE, "aout_enable", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                                 SENSORS_MODE_RW, PCF8591_SYSCTL_AOUT_ENABLE, VALUE(1), 0 },
    { SENSORS_PCF8591_AOUT, "aout", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_RW, PCF8591_SYSCTL_AOUT, VALUE(1), 0 },
    { 0 }
  };


sensors_chip_features sensors_chip_features_list[] =
{
 { SENSORS_LM78_PREFIX, lm78_features },
 { SENSORS_LM78J_PREFIX, lm78j_features },
 { SENSORS_LM79_PREFIX, lm79_features },
 { SENSORS_LM75_PREFIX, lm75_features },
 { SENSORS_GL518_PREFIX, gl518_features },
 { SENSORS_GL520_PREFIX, gl520_features },
 { SENSORS_LM80_PREFIX, lm80_features },
 { SENSORS_W83781D_PREFIX, w83781d_features },
 { SENSORS_W83782D_PREFIX, w83782d_features },
 { SENSORS_W83783S_PREFIX, w83783s_features },
 { SENSORS_W83697HF_PREFIX, w83782d_features }, /* same as 782d */
		/* Cheat on 83627HF for now - no separate #defines */
 { SENSORS_W83627HF_PREFIX, w83782d_features },
 { SENSORS_AS99127F_PREFIX, as99127f_features },
 { SENSORS_ADM9240_PREFIX, adm9240_features },
		/* Cheat on LM81 for now - no separate #defines */
 { SENSORS_LM81_PREFIX, adm9240_features },
 { SENSORS_DS1780_PREFIX, ds1780_features },
 { SENSORS_ADM1021_PREFIX, adm1021_features },
 { SENSORS_MAX1617_PREFIX, max1617_features },
 { SENSORS_MAX1617A_PREFIX, max1617a_features },
		/* Cheat on LM84 for now - no separate #defines */
 { SENSORS_LM84_PREFIX, adm1021_features },
		/* Cheat on GL523 for now - no separate #defines */
 { SENSORS_GL523_PREFIX, adm1021_features },
		/* Cheat on 1023,THMC10 for now - no separate #defines */
 { SENSORS_ADM1023_PREFIX, adm1021_features },
 { SENSORS_THMC10_PREFIX, adm1021_features },
 { SENSORS_SIS5595_PREFIX, sis5595_features },
 { SENSORS_MAXI_CG_PREFIX, maxi_cg_features },
 { SENSORS_MAXI_CO_PREFIX, maxi_co_features },
 { SENSORS_MAXI_AS_PREFIX, maxi_as_features },
		/* Cheat on NBA for now - no separate #defines */
 { SENSORS_MAXI_NBA_PREFIX, maxi_as_features },
 { SENSORS_THMC50_PREFIX, thmc50_features },
		/* Cheat on ADM1022 for now - no separate #defines */
 { SENSORS_ADM1022_PREFIX, thmc50_features },
 { SENSORS_ADM1025_PREFIX, adm1025_features },
 { SENSORS_VIA686A_PREFIX, via686a_features },
 { SENSORS_DDCMON_PREFIX, ddcmon_features },
 { SENSORS_EEPROM_PREFIX, eeprom_features },
 { SENSORS_LM87_PREFIX, lm87_features },
 { SENSORS_MTP008_PREFIX, mtp008_features },
 { SENSORS_DS1621_PREFIX, ds1621_features },
 { SENSORS_ADM1024_PREFIX, adm1024_features },
 { SENSORS_IT87_PREFIX, it87_features },
 { SENSORS_FSCPOS_PREFIX, fscpos_features },
 { SENSORS_FSCSCY_PREFIX, fscscy_features },
 { SENSORS_PCF8591_PREFIX, pcf8591_features }, 
 { 0 }
};
