package Lire::ReportSpec;

use strict;

use vars qw( $VERSION @ISA );

use Lire::Config;
use Lire::DataTypes qw( :special );
use Lire::XMLSpecContainer;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.22 $' =~ m!Revision: ([.\d]+)!;

    @ISA = qw/Lire::XMLSpecContainer/;
}

########################################################################
#			 Lire::XMLSpecContainer METHODS
########################################################################

sub root_element {
    my ( $self ) = @_;

    return "report-spec";
}

sub root_xml_attrs {
    my ( $self ) = @_;
    if (defined $self->{charttype}) {
	return qq{ charttype="$self->{charttype}"}
    } else {
	return "";
    }
}

sub file_from_id {
    my ( $self, $super, $id ) = @_;

    return "$Lire::Config::reportdir/$super/$id.xml";
}

sub print_children {
    my ( $self, $fh, $indent ) = @_;

    my $pfx = ' ' x $indent;
    if ( defined $self->{filter_spec} ) {
	print $fh "$pfx<lire:filter-spec>\n";
	$self->{filter_spec}->print( $fh, $indent + 1);
	print $fh "\n$pfx</lire:filter-spec>\n\n";
    }

    if ( defined $self->{calc_spec} ) {
	print $fh "$pfx<lire:report-calc-spec>\n";
	$self->{calc_spec}->print( $fh, $indent + 1 );
    } else {
	$self->{extension}->print( $fh, $indent );
    }
}

########################################################################
#			 Lire::ReportSpec METHODS
########################################################################

sub charttype {
    my ( $self, $chart ) = @_;

    if ( @_ == 2 ) {
	if ( $chart) {
	    croak "invalid chart type : $chart"
	      unless ( check_chart( $chart ));

	    $self->{charttype} = $chart;
	} else {
	    delete $self->{charttype};
	}
    }

    $self->{charttype};
}

sub filter_spec {
    my ( $self, $filter_spec ) = @_;

    if ( @_ == 2 ) {
	if ( defined $filter_spec ) {
	    croak "filter specification must be of type Lire::FilterExpr"
	      unless UNIVERSAL::isa( $filter_spec, "Lire::FilterExpr" );
	    $self->{filter_spec} = $filter_spec;
	} else {
	    delete $self->{filter_spec};
	}
    }

    $self->{filter_spec};
}

sub calc_spec {
    my ( $self, $calc_spec ) = @_;

    if ( @_ == 2 ) {
	croak "calc specification must be of type Lire::Aggregator"
	  unless UNIVERSAL::isa( $calc_spec, "Lire::Aggregator" );
	$self->{calc_spec} = $calc_spec;
	delete $self->{extension};
    }

    $self->{calc_spec};
}

sub extension {
    my ( $self, $extension ) = @_;
    if ( @_ == 2 ) {
	croak "extension must be of type Lire::Extension"
	  unless UNIVERSAL::isa( $extension, "Lire::Extension" );
	$self->{extension} = $extension;
	delete $self->{calc_spec};
    }

    $self->{extension};
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = ();
    my $filter_spec = $self->filter_spec;
    push @fields, $filter_spec->needed_fields
      if defined $filter_spec;

    my $calc_spec = $self->calc_spec;
    push @fields, $calc_spec->needed_fields
      if defined $calc_spec;

    my $extension = $self->extension;
    push @fields, $calc_spec->needed_fields
      if defined $extension;

    @fields;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportSpec - 

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: ReportSpec.pm,v 1.22 2002/01/16 21:48:24 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
