package Lire::ReportConfig;

use strict;

use vars qw( $VERSION @ISA  );

use Lire::DataTypes qw/:special/;
use Lire::ReportSpec;
use Lire::FilterSpec;
use Lire::ReportSection;
use Lire::ReportSpecFactory;
use Lire::XMLSpecContainer;

use Text::ParseWords qw/shellwords/;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.2 $' =~ m!Revision: ([.\d]+)!;
}

my $debug = 0;
sub debug {
    $debug and warn $_[0];
}

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $super ) = @_;

    croak "invalid superservice: $super"
      unless check_superservice( $super );
    
    bless { superservice => $super,
	    sections	 => [],
	  }, $class;
}

sub new_from_file {
    my $proto	= shift;
    my $class	= ref $proto || $proto;
    my ($superservice, $report_cfg, $factory) =	@_;

    my $self = $class->new( @_ );

    debug( "new_from_file: calling load_from_file" );
    $self->load_from_file( $report_cfg, $factory );

    $self;
}

sub superservice {
    my ( $self ) = @_;

    $self->{superservice};
}

sub load_from_file {
    my ( $self, $report_cfg, $factory ) = @_;

    # Create the factory
    $factory ||= new Lire::ReportSpecFactory;

    # Reset the sections
    $self->{sections} = [];

    # Format of the configuration file is
    # ( (=section <title>)?
    #   (|filter_id <param>)*
    #   (report_id  <param>))+
    #  )+

    my $state = "section"; # Can also be filter or report
    my $curr_section  = undef;

    # Load the report configuration file
    open CFG_FILE, $report_cfg
      or die "can't open report configuration file $report_cfg: $!\n";
    debug( "load_from_file: reading report configuration file " .
        "'$report_cfg'\n" );

    my $line;
  CFG_LINE:
    while ( defined( $line = <CFG_FILE> ) ) {
	next if $line =~ /^\s*#/; # Skip comments
	next if $line =~ /^\s*$/; # Skip blank lines

	chomp $line;

      debug( "load_from_file: processing line '$line'\n" );

      STATE:
	while ( $_ = $state ) {
	    /^section$/ && do {
		if ( $line =~ /^=section (.*)$/ ) {
		    $curr_section = new Lire::ReportSection( $self->superservice, $1 );
		    $self->add_section( $curr_section );
		    $state = "filter";
		    next CFG_LINE;
		} else {
		    # Create default section
		    $curr_section = 
		      new Lire::ReportSection( $self->superservice );
		    $self->add_section( $curr_section );
		    $state = "filter";
		    next STATE;
		}
	    };
	    /^filter$/ && do {
		if ( $line =~ /^\|(.*)/ ) {
		    my ( $id, $params ) = parse_param_line( $1 );

		    eval {
			my $spec = Lire::FilterSpec->load( $self->superservice,
							   $id, $factory );

			while ( my ($name, $value) = each %$params ) {
			    $spec->param( $name )->value( $value );
			}
			$curr_section->add_filter( $spec );
		    };
		    if ( $@ ) {
			warn( "error at line $.: $@\n" );
			warn( "Omitting filter $id defined at line $.\n" );
		    }
		    next CFG_LINE;
		} elsif ( $line =~ /^=section/ ) {
		    warn "unexpected =section at line $.. Ignoring.\n";
		    next CFG_LINE;
		} else {
		    # Report
		    $state = "report";
		    next STATE;
		}
	    };
	    /^report$/ && do {
		if ( $line =~ /^=section/ ) {
		    # Some reports were defined
		    $state = "section";
		    next STATE;
		} elsif ( $line =~ /^\|/ ) {
		    # Filter -> create a new section
		    $curr_section = 
		      new Lire::ReportSection( $self->superservice );
		    $self->add_section( $curr_section );
		    $state = "filter";
		    next STATE;
		} else {
		    my ( $id, $params ) = parse_param_line( $line );

		    eval {
			my $report_spec =
			  Lire::ReportSpec->load( $self->superservice, $id,
						  $factory );

			while ( my ($name, $value) = each %$params ) {
			    $report_spec->param( $name )->value( $value );
			}
			$curr_section->add_report( $report_spec );
		    };
		    if ( $@ ) {
			warn( "error at line $.: $@\n" );
			warn( "Omitting report $id defined at line $.\n" );
		    }
		    next CFG_LINE;
		}
	    };
	    die "unknown state: $state";
	}
    }
    close CFG_FILE;
}

sub parse_param_line {
    my ( $id, @p ) = shellwords( $_[0] );
    my %params = ();
    foreach my $param_str ( @p ) {
	my ( $param, $value ) = $param_str =~ /^([-.\w]+)=(.*)$/;
	unless ( defined $param ) {
	    warn( "error parsing parameter $param_str at line $.. Ignoring parameter.\n" );
	    next;
	}

	$value ||= "";
	$params{$param} = $value;
    }

    return ( $id, \%params );
}

sub create_param_line {
    my ( $id, $spec ) = @_;

    my @line = ( $id );
    foreach my $name ( $spec->param_names ) {
	my $value = $spec->param( $name )->value;
	$value =~ s/\\/\\\\/g;	# Escape backslashes
	$value =~ s/"/\\"/g;	# and double quotes
	
	push @line, $name . '="' . $value . '"';
    }
    return join( " ", @line );
}

sub sections {
    my ( $self ) = @_;

    return @{$self->{sections}};
}

sub add_section {
    my ( $self, $section ) = @_;

    croak ( "section should be of type Lire::ReportSection (not $section)" )
      unless UNIVERSAL::isa( $section, "Lire::ReportSection" );

    push @{$self->{sections}}, $section;
}

sub print {
    my ( $self, $fh ) = @_;
    
    $fh ||= \*STDOUT;

    foreach my $section ( $self->sections ) {
	print $fh "=section ", $section->title, "\n";

	foreach my $filter ( $section->filters ) {
	    print $fh create_param_line( "|" . $filter->id, $filter ), "\n";
	}

	foreach my $report ( $section->reports ) {
	    print $fh create_param_line( $report->id, $report ), "\n";
	}

	# Empty line
	print $fh "\n"
    }
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportConfiguration -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: ReportConfig.pm,v 1.2 2002/01/19 15:18:05 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
