package Lire::AsciiDlf::Group;

use strict;

use vars qw( $VERSION @ISA %key_db %defined_keys );

use Lire::Group;
use Lire::AsciiDlf::GroupOp;
use Lire::DataTypes qw( :basic :time );
use Lire::XMLUtils qw/xml_encode/;

use POSIX qw( strftime );
use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.15 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::Group Lire::AsciiDlf::GroupOp );
}

sub init_report {
    my $self = shift;

    my @fields = map { $_->field->name} @{$self->group_fields};
    $self->{time_fields} = [ map { is_time_type( $_->field->type ) } @{$self->group_fields} ];
    $self->{key_maker} =
      $self->{report_spec}->schema->make_key_access_func( @fields  );
    $self->{data}   = $self->init_group_data;

    if ( $self->sort_fields ) {
	# Build sort function
	my @sort_ops = ();
	foreach my $f ( @{$self->sort_fields} ) {
	    my ($a, $b) = ('$a', '$b' );
	    if ( $f =~ /^-/ ) {
		$f = substr $f, 1;
		$a = '$b';
		$b = '$a';
	    }

	    my $index;	# This will contains the index of the field in the array
	    my $cmp = '<=>'; # Default to numeric comparison
	    my $i = 0;
	    foreach my $group_field ( @{$self->group_fields} ) {
		if ( $group_field->name eq $f ) {
		    $index = $i;
		    if ( is_numeric_type( $group_field->field->type ) ) {
			$cmp = "<=>";
		    } else {
			$cmp = "cmp";
		    }
		    last;
		}
		$i++;
	    }
	    $i = @{$self->group_fields};
	    unless (defined $index) {
		foreach my $op ( @{$self->ops} ) {
		    if ( $op->name eq $f ) {
			$index = $i;
			last;
		    }
		    $i++;
		}
	    }
	    push @sort_ops, $a ."->[1][$index] $cmp " . $b ."->[1][$index]";
	}
	my $sort_code = "sub { " . join( " || ", @sort_ops ) . " }";
	$self->{sort_cmp} = eval $sort_code;
	croak "error compiling sort comparison ($sort_code): $@" if $@;
    }

    foreach my $op ( @{$self->{ops}}) {
	$op->init_report( @_ );
    }
}

sub update_report {
    my ( $self, $dlf ) = @_;

    $self->update_group_data( $dlf, $self->{data} );
}

sub end_report {
    my ( $self ) = @_;

    $self->end_group_data( $self->{data} );
}

sub init_group_data {
    my ( $self ) = @_;

    # We group the data by key in an hash.
    # Each key as an array.
    # The layout of the array is
    # [ group field, group field, ..., op data, op data, op data ]
    return {};
}

sub update_group_data {
    my ( $self, $dlf, $data ) = @_;

    my $key = $self->{key_maker}->( $dlf );
    my $key_data = $data->{$key};
    unless ( exists $data->{$key} ) {
	# Save the grouped field's value
	my $i = 0;
	foreach my $f ( @{$self->group_fields} ) {
	    $key_data->[$i++] = $dlf->[$f->field->pos];
	}

	foreach my $op ( @{$self->ops} ) {
	    $key_data->[$i++] = $op->init_group_data();
	}

	$data->{$key} = $key_data;
    }

    my $i = @{$self->group_fields};
    foreach my $op ( @{$self->ops} ) {
	$op->update_group_data( $dlf, $key_data->[$i++] );
    }
}

sub end_group_data {
    my ( $self, $data ) = @_;

    foreach my $key ( keys %$data ) {
	my $item = $data->{$key};

	my $i = @{$self->group_fields};
	foreach my $op ( @{$self->ops} ) {
	    $op->end_group_data( $item->[$i++] );
	}
    }

    # Sort the keys according to the sort value
    my @sorted_keys;
    if ( $self->sort_fields ) {
	my $cmp = $self->{sort_cmp};
	# This use schwartzian transform
	@sorted_keys =  map { $_->[0] } sort $cmp 
	  map {
	      my $v = [];
	      foreach my $f ( @{$data->{$_}} ) {
		  if ( ref $f eq 'SCALAR' ) {
		      push @$v, $$f;
		  } elsif (ref $f eq 'ARRAY' ) {
		      push @$v, $f->[0];
		  } elsif (ref $f eq 'HASH' ) {
		      push @$v, $f->{value};
		  } else {
		      push @$v, $f;
		  }
	      }
	      [ $_, $v ];
	  } keys %$data;
    } else {
	@sorted_keys = keys %$data;
    }

    # Keep only limit records
    if ( defined $self->limit ) {
	my $limit = $self->limit;
	if ( $limit =~ /^\$/ ) {
	    $limit = substr $limit, 1;
	    $limit = $self->{report_spec}->param( $limit )->value;
	}
	splice @sorted_keys, $limit
	  if ($limit < @sorted_keys );
    }

    # Delete unused keys
    %$data = map { $_ => $data->{$_} } @sorted_keys;
    $data->{_lr_sorted_keys} = \@sorted_keys;
}

sub write_group_data {
    my ( $self, $fh, $prefix, $data ) = @_;

    my $pfx = ' ' x $prefix;

    my $sorted_keys = $data->{_lr_sorted_keys};
    if ( $self->{parent} && @$sorted_keys ) {
	print $fh $pfx, qq{<lire:group>\n};
        $prefix++;
	$pfx .= ' ';
    }
    my $field_count = @{$self->group_fields};
    foreach my $key ( @$sorted_keys ) {
	my $item = $data->{$key};
	print $fh $pfx, qq{<lire:entry>\n};
	my $i = 0;
	while  ( $i < $field_count ) {
	    # XXX: This is an hack. We should handle the time 
	    # format in the style layer, and we shouldn't group on
	    # time field. We should add Record operation which would display
	    # record's fields based on certain criteria.
	    if ( $self->{time_fields}[$i] ) {
		print $fh $pfx, " <lire:name>",
		  strftime( '%Y-%m-%d %H:%M:%S', localtime $item->[$i++] ),
		    "</lire:name>\n";
	    } else {
		print $fh $pfx, " <lire:name>", xml_encode( $item->[$i++] ),
		  "</lire:name>\n";
	    }
	}
	foreach my $op ( @{$self->ops} ) {
	    $op->write_group_data( $fh, $prefix + 1, $item->[$i++] );
	}
	print $fh $pfx, qq{</lire:entry>\n};
    }
    if ( $self->{parent} && @$sorted_keys ) {
	$pfx = substr $pfx, 0, length($pfx) -1;
	print $fh $pfx, qq{</lire:group>\n};
    }
}

sub write_report {
    my ( $self, $fh, $prefix ) = @_;
    $fh	    ||= \*STDOUT;
    $prefix ||= 0;

    $self->write_group_data( $fh, $prefix, $self->{data} );

    $self;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Group -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Group.pm,v 1.15 2002/01/16 20:21:21 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
