/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/time.h>
#include <math.h>

#include "common/FFT.h"
#include "common/SFFT.h"
#include "common/SoundFile.h"

#include "FFTSpeed.h"

int main(int argc, char **argv)
{
	if (argc < 6) {
		::fprintf(stderr, "Usage: FFTSpeed <filename> <overlap> <fft length> <sfft first> <sfft last>\n");
		return 1;
	}

	wxString fileName = wxString(argv[1]);
	int       overlap = ::atoi(argv[2]);
	int     fftLength = ::atoi(argv[3]);
	int     sfftFirst = ::atoi(argv[4]);
	int      sfftLast = ::atoi(argv[5]);

	CFFTSpeed speed(fileName, overlap, fftLength, sfftFirst, sfftLast);
	speed.run();

	return 0;
}

CFFTSpeed::CFFTSpeed(const wxString& fileName, int overlap, int fftLength, int sfftFirst, int sfftLast) :
m_fileName(fileName),
m_overlap(overlap),
m_fftLength(fftLength),
m_sfftFirst(sfftFirst),
m_sfftLast(sfftLast)
{
}

CFFTSpeed::~CFFTSpeed()
{
}

void CFFTSpeed::run()
{
	CSoundDev* file = new CSoundFile;

	file->openRead(m_fileName, 11025, 16);

	double* audio = new double[30 * 11025];

	int length = 0;
	while (length < (30 * 11025)) {
		int len = 4096;

		if (!file->read(audio + length, len))
			break;

		length += len;
	}

	file->close();
	delete file;

	processFFT(audio, length);

	processSFFT(audio, length);

	delete[] audio;
}

void CFFTSpeed::processSFFT(double* audio, int length) const
{
	struct timeval tvBegin;
	::gettimeofday(&tvBegin, NULL);

	CSFFT sfft(m_fftLength, m_sfftFirst, m_sfftLast + 1);

	for (int i = 0; i < (length - m_fftLength); i++)
		sfft.process(audio[i]);

	struct timeval tvEnd;
	::gettimeofday(&tvEnd, NULL);

//	::printf("SFFT: start at %ld:%ld end at %ld:%ld\n", tvBegin.tv_sec, tvBegin.tv_usec, tvEnd.tv_sec, tvEnd.tv_usec);

	::printf("SFFT: elapsed: %ld seconds\n", tvEnd.tv_sec - tvBegin.tv_sec);
}

void CFFTSpeed::processFFT(double* audio, int length) const
{
	struct timeval tvBegin;
	::gettimeofday(&tvBegin, NULL);

	CFFT fft(m_fftLength);

	int done = 0;

	for (int i = 0; i < (length - m_fftLength); i += m_overlap) {
		done++;
		fft.process(audio + i);
	}

	struct timeval tvEnd;
	::gettimeofday(&tvEnd, NULL);

//	::printf("FFT: start at %ld:%ld end at %ld:%ld\n", tvBegin.tv_sec, tvBegin.tv_usec, tvEnd.tv_sec, tvEnd.tv_usec);

	::printf("FFT: elapsed: %ld seconds\n", tvEnd.tv_sec - tvBegin.tv_sec);
	::printf("FFT: performed %d FFTs\n", done);
}
