/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <ctype.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "curse.h"
#include "linuxtrade.h"
#include "news.h"
#include "article.h"
#include "minihtml.h"

/*
 * TODO. Oh yeah, there's bugs in here.
 */
static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	Cursor;		// Article line number at top of display
static int	MaxCursor;	// Maximum value for above
static int	NumLines;	// Number of lines in the article

//
// TODO: make this asynchronous
// TODO: handle basic HTML
//

void
recenter(void)
{
	int	y, x;
	int	i, cnt;

	y = getcury(Subwin);
	x = getcurx(Subwin);

	cnt = (getmaxx(Subwin) - x) / 2;
	wmove(Subwin, y, 0);

	for (i = 0; i < cnt; ++i)
		winsch(Subwin, ' ');
	wmove(Subwin, y, x+cnt);
}

static void
newline()
{
	attr_t	boldorrev = Reverse ? A_REVERSE : A_BOLD;

	waddch(Subwin, ' ' | boldorrev);	// hack
	waddch(Subwin, '\n');
}


static void
display_more(void)
{

	mvwaddch(Win, 1, getmaxx(Win)-1,
		Cursor ? ACS_UARROW : ACS_VLINE);

	mvwaddch(Win, getmaxy(Win)-2, getmaxx(Win)-1,
		(Cursor < MaxCursor) ? ACS_DARROW : ACS_VLINE);

}

void
article_display_schwab(char *article)
{
	char	ch;
	int	x, y;
	char	*ep;
	int	wlen;
	int	escs;

	if (!Subwin)
		return;

	wmove(Subwin, 0, 0);
	escs = 0;
	while ((ch = *article++))
	{
		if (ch != 0x1b)
		{
			if (escs == 2)
				waddch(Subwin, ' ');
			escs = 0;
		}

		switch (ch)
		{
		case 0x1b:
			++escs;
			if (escs == 4)
			{
				newline();
				newline();
				escs = 0;
			}
			break;
		case 0x1d:
		case 0x08:
			x = getcurx(Subwin); y = getcury(Subwin);
			if (x != 0 || y != 0)
				newline();
			break;
		case ' ':
			waddch(Subwin, ' ');
			break;
		default:
			for (ep = article; *ep && *ep > ' '; ++ep)
				{}
			wlen = ep - article + 1;
			x = getcurx(Subwin);
			if ((x+wlen) >= getmaxx(Subwin)-2)
				newline();
			waddch(Subwin, ch);
			break;
		}
	}

	Cursor = 0;
	NumLines = getcury(Subwin) + 1;
	MaxCursor = NumLines - (getmaxy(Win) - 2);
	if (MaxCursor < 0)
		MaxCursor = 0;

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

static int
prewordhook(int *tblcolp, char *wbuf, int wlen)
{
	if (strcmp(wbuf, "Money.net") == 0)
		return -1;
	else
		return wlen;
}

void
article_display_mini(char *url, char *skip)
{
	FILE	*fp;
	char	buf[BUFSIZ];

	sprintf(buf, "%s \"%s\"", SUCKURL, url);
	if (Debug)
		fprintf(stderr, "URL: %s\n", buf);
	fp = popen(buf, "r");
	if (!fp)
	{
		mvwprintw(Subwin, getmaxy(Subwin)/2, 20,
				"Can't access headlines");
		touchwin(Win);
		return;
	}

	if (skip)
		minihtml_skip_past_line(fp, skip);

	minihtml_parse(Subwin, fp, NULL, 0, MHP_DEFAULT,
			prewordhook, NULL, NULL, NULL);

	pclose(fp);

	Cursor = 0;
	NumLines = getcury(Subwin) + 1;
	MaxCursor = NumLines - (getmaxy(Win) - 2);
	if (MaxCursor < 0)
		MaxCursor = 0;

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

void
article_display_url(char *url, char *skip)
{
	int	x, y;
	FILE	*fp;
	char	buf[BUFSIZ];
	int	state;
	int	c;
	char	*tagp;
	char	*parmp;
	int	slash;
	int	pre;
	int	center;
	int	anylets;

	sprintf(buf, "%s \"%s\"", SUCKURL, url);
	if (Debug)
		fprintf(stderr, "URL: %s\n", buf);
	fp = popen(buf, "r");
	if (!fp)
	{
		mvwprintw(Subwin, getmaxy(Subwin)/2, 20,
				"Can't access headlines");
		touchwin(Win);
		return;
	}

	if (skip)
		minihtml_skip_past_line(fp, skip);

	/*
	 * Mini-parser for a subset of HTML
	 */
	state = 0;
	pre = 0;
	center = 0;
	slash = 0;
	anylets = 0;
	tagp = buf;
	y = 0; 
	x = 0;
	wmove(Subwin, y, x);

	while ( (c = fgetc(fp)) != EOF)
	{
		if (Debug >= 5)
			fprintf(stderr, "state=%c c=%c tagp=%d\n",
					state, c, tagp-buf);
		switch (state)
		{
		case 0:
			if (c == '<')
			{
				state = '<';
				tagp = buf;
				slash = 0;
				anylets = 0;
			}
			else if (c == '&')
			{
				state = '&';
				tagp = buf;
			}
			else if (!pre && c == '\r')
				{}
			else if (c == '\n')
			{
				if (pre)
					newline();
			}
			else
				waddch(Subwin, c);
			break;
		case '&':
			if (c == ';')
			{
				*tagp = 0;
				state = 0;
				if (strcmp(buf, "AMP") == 0)
					waddch(Subwin, '&');
			}
			else
				*tagp++ = toupper(c);
			break;
		case '<':
			if (c == '>')
			{
				*tagp = 0;
				parmp = strchr(buf, ' ');
				if (parmp)
					*parmp++ = 0;
				state = 0;
				if (strcmp(buf, "P") == 0 && !slash)
				{
					if (center)
						recenter();
					newline();
					newline();
				}
				else if (strcmp(buf, "B") == 0)
				{
					wattrset(Subwin,
						slash ? A_NORMAL : A_BOLD);
				}
				else if (strcmp(buf, "I") == 0)
				{
					wattrset(Subwin,
						slash ? A_NORMAL :
						A_UNDERLINE);
				}
				else if (strcmp(buf, "CENTER") == 0)
				{
					if (center)
						recenter();
					center = !slash;
				}
				else if (strcmp(buf, "PRE") == 0)
				{
					pre = !slash;
					newline();
				}
				else if (strcmp(buf, "TR") == 0 && !slash)
				{
					newline();
				}
				else if (strcmp(buf, "TD") == 0 && !slash)
				{
					waddch(Subwin, '\t');
				}
				else if (strcmp(buf, "TH") == 0 && !slash)
				{
					waddch(Subwin, '\t');
				}
			}
			else if (tagp == buf && c == '/')
			{
				anylets = 1;
				slash = 1;
			}
			else if (anylets == 0 && c >= '0' && c <= '9')
			{
				// < num is not HTML, just bad coding
				waddch(Subwin, '<');
				waddch(Subwin, c);
				state = 0;
			}
			else
			{
				anylets = 1;
				*tagp++ = toupper(c);
				if ( (tagp-buf) > 1024)
				{
					// Bad HTML
					tagp = buf;
					state = 0;
				}
			}
			break;
		}
	}

	pclose(fp);

	Cursor = 0;
	NumLines = getcury(Subwin) + 1;
	MaxCursor = NumLines - (getmaxy(Win) - 2);
	if (MaxCursor < 0)
		MaxCursor = 0;

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();
}

void
//article_popup(int artno, char *source, char *date, char *time)
article_popup(ARTINFO *ai)
{
	int	cols;
	char	buf[BUFSIZ];
	int	n;
	char    *datefmt = "%m/%d %H:%M";
	char    datetimebuf[64];
	struct tm *tmp;

	n = LINES - 4 - 2 - NumStock - 12;
	if (n < 24)
		n = 24;
	Win = bestwin(n);
	if (!Win)
		error(1, "Can't create article window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);

	// format date and time
	tmp = localtime(&ai->utime);
	strftime(datetimebuf, sizeof(datetimebuf), datefmt, tmp);

	sprintf(buf, " %s %s ", ai->source, datetimebuf);
	mvwcenter(Win, 0, buf);

	//
	// Turns out, this works best for most news articles and an 80
	// column xterm. We will clip columns 78 and 79, but there are
	// only blanks there in the news articles I've seen.  For smaller
	// displays, we are screwed anyway.
	//
	if (ai->sr && ai->sr->send_article)
		Subwin = newpad(600, cols - 2);
	else if (cols == 80)
		Subwin = newpad(600, cols);
	else
		Subwin = newpad(600, cols - 2);
	if (!Subwin)
		error(1, "Can't create article pad\n");
	wbkgd(Subwin, Reverse ? A_REVERSE : A_NORMAL);

	Panel = new_panel(Win);

	if (ai->sr && ai->sr->send_article)
	{
		(*ai->sr->send_article)(ai->sr, ai->artkey);
	}
	else
	{
		sprintf(buf,
			"http://fast.quote.com/fq/quotetracker/news?story=%s",
			ai->artkey);
		article_display_url(buf, NULL);
	}
}

static void
popdown(void)
{
	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
}

int
article_command(int c, STREAMER sr)
{
	MEVENT	m;

	switch (c)
	{
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return -1;
	case 'j':
	case KEY_DOWN:
		if (++Cursor > MaxCursor)
		{
			--Cursor;
			beep();
			break;
		}
		break;
	case 'k':
	case KEY_UP:
		if (--Cursor < 0)
		{
			++Cursor;
			beep();
			break;
		}
		break;
	case '-':
	case KEY_PPAGE:
		if (Cursor == 0)
		{
			beep();
			break;
		}
		Cursor -= getmaxy(Win) - 2 - 1;
		if (Cursor < 0)
			Cursor = 0;
		break;
	case '+':
	case ' ':
	case KEY_NPAGE:
		if (Cursor == MaxCursor)
		{
			beep();
			break;
		}
		Cursor += getmaxy(Win) - 2 - 1;
		if (Cursor > MaxCursor)
			Cursor = MaxCursor;
		break;
	case '0':
	case KEY_HOME:
		Cursor = 0;
		break;
	case '$':
	case KEY_END:
		Cursor = MaxCursor;
		break;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;
	case KEY_PRINT:
	case KEY_F(12):
	case CTRL('P'):
		print_window(Subwin, NumLines,
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		// TODO: lookup symbol at click point
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case 033:
	case 'q':
	case 'x':
		popdown();
		return 2;
	default:
		beep();
		break;
	}

	display_more();
	copywin(Subwin, Win,
			Cursor, 0,
			1, 1,
			getmaxy(Win)-2, getmaxx(Win)-2,
			FALSE);
	move(LINES-1, CursorX);
	update_panels();
	doupdate();

	return (-1);
}
