/*
 * $Id: disk_io.h,v 1.9 2001/06/01 02:07:24 antona Exp $
 *
 * disk_io.h - Exports for disk io. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef DISK_IO_H
#define DISK_IO_H

#include "types.h"

/**
 * ntfs_pwrite - positioned write to disk
 * @fd:		file descriptor to write to
 * @b:		data buffer to write to disk
 * @count:	number of bytes to write
 * @pos:	position in file descriptor to write to
 * 
 * This function will write @count bytes of data @b to file descriptor @fd at
 * position @pos.
 *
 * On success return bytes written (0 means nothing written).
 * On error return -1 with errno set appropriately to the return code of 
 * either lseek, malloc, write or fdatasync. Or errno is EINVAL if b is null.
 */
__s64 ntfs_pwrite(int fd, const void *b, __s64 count, const __s64 pos);

/**
 * mst_pwrite - multi sector transfer (mst) positioned write
 * @fd:		file descriptor to write to
 * @b:		data buffer to write to disk
 * @count:	number of bytes to write
 * @pos:	position in file descriptor to write to
 * 
 * Multi sector transfer (mst) positioned write. This function will write
 * @count bytes of data @b to file descriptor @fd at position @pos.
 * 
 * On success return bytes written (0 means nothing written).
 * On error return -1 with errno set appropriately to the return code of 
 * either lseek, malloc, write or fdatasync. Or errno is EINVAL if b is null.
 */
__s64 mst_pwrite(int fd, const void *b, __s64 count, const __s64 pos);

/**
 * ntfs_pread - positioned read from disk
 * @fd:		file descriptor to read from
 * @b:		output data buffer
 * @bksize:	size of each block
 * @count:	number of blocks to read
 * @pos:	position in file descriptor to read from
 * 
 * This function will read @count @blocks of size @bksize bytes each from file
 * descriptor @fd at position @pos into the buffer @b.
 * 
 * On success, return the number of successfully read blocks. If this number is
 * lower than @count this means that we have reached end of file. 0 means end
 * of file or nothing to read (@count or @bksize is 0). On error, return -1
 * with errno set appropriately to the return code of either lseek or read, or
 * to EINVAL in case the data pointer @b was NULL or the block size (@bksize)
 * was not a multiple of NTFS_SECTOR_SIZE or @bksize was not a power of two.
 * 
 * NOTE: If a read error is encountered we return -1 even if we have already
 * 	 read some blocks successfully (this is subject to change!!!).
 */
__s64 ntfs_pread(const int fd, void *b, const __u32 bksize, __s64 count,
	         const __s64 pos);

/**
 * mst_pread - multi sector transfer (mst) positioned read
 * @fd:		file descriptor to read from
 * @b:		output data buffer
 * @bksize:	size of each block
 * @count:	number of blocks to read
 * @pos:	position in file descriptor to read from
 * 
 * Multi sector transfer (mst) positioned read. This function will read @count
 * @blocks of size @bksize bytes each from file descriptor @fd at position @pos
 * into the buffer @b.
 * 
 * On success, return the number of successfully read blocks. If this number is
 * lower than @count this means that we have reached end of file. 0 means end
 * of file or nothing to read (@count or @bksize is 0). On error, return -1
 * with errno set appropriately to the return code of either lseek or read, or
 * to EINVAL in case the data pointer @b was NULL or the block size (@bksize)
 * was not a multiple of NTFS_SECTOR_SIZE or @bksize was not a power of two.
 * 
 * NOTE: If an incomplete multi sector transfer has been detected the magic
 * 	 will have been changed to magic_BAAD but no error will be returned.
 * 	 I.e. it is possible that we return count blocks as being read but that
 * 	 any number (between zero and count!) of these blocks is actually
 * 	 subject to a multi sector transfer error. This should be detected by
 * 	 the caller by checking for the magic being "BAAD".
 * 
 * NOTE: If a read error is encountered we return -1 even if we have already
 * 	 read some blocks successfully (this is subject to change!!!).
 */
__s64 mst_pread(const int fd, void *b, const __u32 bksize, __s64 count,
		const __s64 pos);

/**
 * get_clusters - read ntfs clusters
 * @vol:	volume to read from
 * @buf:	output data buffer
 * @lcn:	starting logical cluster number
 * @count:	number of clusters to read
 *
 * Read @count ntfs clusters starting at logical cluster number @lcn from volume
 * @vol into buffer @buf. Return number of clusters read or -ERRNO on error, 
 * where ERRNO is the error code. This function will do the error reporting so
 * caller really only needs to check for sucess / failure.
 *
 * NOTE: @buf has to be at least of size vol->ClusterSize * @count. No checking
 * is performed for this!
 */
int get_clusters(const ntfs_volume *vol, __u8 *buf, const __s64 lcn,
		 const int count);

/**
 * put_clusters - write ntfs clusters
 * @vol:	volume to write to
 * @buf:	data buffer to write to disk
 * @lcn:	starting logical cluster number
 * @count:	number of clusters to write
 *
 * Write @count ntfs clusters starting at logical cluster number @lcn from
 * buffer @buf to volume @vol. Return @count on success or -ERRNO on error,
 * where ERRNO is the error code. This function will do the error reporting so
 * caller really only needs to check for sucess / failure.
 */
int put_clusters(ntfs_volume *vol, const __u8 *buf, const __s64 lcn, int count);

int get_mft_records(const ntfs_volume *vol, const MFT_RECORD *mrec,
		    const MFT_REF *mref, const int count);

extern __inline__ int get_mft_record(const ntfs_volume *vol,
				     const MFT_RECORD *mrec,
				     const MFT_REF *mref);

int __read_file_record(const ntfs_volume *vol, const MFT_REF *mref,
			MFT_RECORD **mrec, ATTR_RECORD **attr);

/**
 * put_mft_record - write a record to the mft
 * @vol:	volume to write to
 * @buf:	data buffer containing the mft record to write
 * @mref:	mft record number to write to
 *
 * Write mft record @mref from buffer @buf to volume @vol. Return 1 on
 * success or -ERRNO on error, where ERRNO is the error code. This function
 * will do the error reporting so caller really only needs to check for sucess
 * / failure.
 */
int put_mft_record(ntfs_volume *vol, const __u8 *buf,
		   const MFT_REF *mref);

#endif /* defined DISK_IO_H */

