/*
 * PQAdecompile
 * Steven James <pyro@linuxlabs.com>
 * Linux Labs http://www.linuxlabs.com
 *
 * This is Free software, licensed under the GNU Public License V2.
 *
 * Based on pilot-file
 */

/*
 * Pilot File dump utility
 * Pace Willisson <pace@blitz.com> December 1996
 * Additions by Kenneth Albanowski
 *
 * This is free software, licensed under the GNU Public License V2.
 * See the file COPYING for details.
 */
          
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include <libpisock/pi-source.h>
#include <libpisock/pi-socket.h>
#include <libpisock/pi-dlp.h>
#include <libpisock/pi-file.h>

#ifdef sun
  extern char* optarg;
  extern int optind;
#endif

void dump_header (struct pi_file *pf, struct DBInfo *ip);
void dump_app_info (struct pi_file *pf, struct DBInfo *ip);
void list_records (struct pi_file *pf, struct DBInfo *ip);
void dump_record (struct pi_file *pf, struct DBInfo *ip, int record);
void dump_pqa(char *buf,int size);

char *iso_time_str (time_t t);
void dump (void *buf, int size);

char * progname;

void
usage (void)
{
  fprintf (stderr, "usage: %s [-v] file\n", progname);
  exit (1);
}

int vflag=0;

char *htmls[100];
int num_htmls = 0;

int
main (int argc, char **argv)
{
  struct pi_file *pf;
  int c;
  char *name;
  struct DBInfo info;
  
  progname = argv[0];

  while ((c = getopt (argc, argv, "v")) != EOF) {
    switch (c) {
    case 'v':
      vflag = 1;
      break;
    default:
      usage ();
    }
  }

  if (optind >= argc)
    usage ();
  
  name = argv[optind++];

  if (optind != argc)
    usage ();

  if ((pf = pi_file_open (name)) == NULL) {
    fprintf (stderr, "can't open %s\n", name);
    exit (1);
  }

  if (pi_file_get_info (pf, &info) < 0) {
    fprintf (stderr, "can't get info\n\n");
    exit (1);
  }
  
	if(vflag)
		dump_header (pf, &info);

    dump_app_info (pf, &info);

    list_records(pf, &info);
  
  return (0);
}

char *
iso_time_str (time_t t)
{
  struct tm tm;
  static char buf[50];

  tm = *localtime (&t);
  sprintf (buf, "%04d-%02d-%02d %02d:%02d:%02d",
	   tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
	   tm.tm_hour, tm.tm_min, tm.tm_sec);
  return (buf);
}

void
dump (void *buf, int n)
{
  int i, j, c;

  for (i = 0; i < n; i += 16) {
    printf ("%04x: ", i);
    for (j = 0; j < 16; j++) {
      if (i+j < n)
	printf ("%02x ", ((unsigned char *)buf)[i+j]);
      else
	printf ("   ");
    }
    printf ("  ");
    for (j = 0; j < 16 && i+j < n; j++) {
      c = ((unsigned char *)buf)[i+j] & 0x7f;
      if (c < ' ' || c >= 0x7f)
	putchar ('.');
      else
	putchar (c);
    }
    printf ("\n");
  }
}

//	PalmOS 5-bit compression handling

typedef struct {
	char 	*buffer;
	int		length;
	int 	bitpos;
} bit_buffer;

typedef struct asciiz_t {
	struct asciiz_t *next;
	char 			*string;
}	asciiz_stack;
	
asciiz_stack *stack_top = NULL;

int
stack_push(char *string)
{
	asciiz_stack *entry = (asciiz_stack *) malloc(sizeof(asciiz_stack));

	if(!entry)
		return(-1);

	entry->string = string;
	entry->next = stack_top;
	stack_top = entry;
	return(0);
}

char *stack_pop(void)
{
	asciiz_stack *entry = stack_top;
	char *ret;

	if(!entry)
		return(NULL);

	stack_top = entry->next;

	ret = entry->string;
	free(entry);

	return(ret);
}
	

bit_buffer *InitBitBuffer(char *buf, int length)
{
	bit_buffer *buffer;

	buffer = (bit_buffer *) malloc(sizeof(bit_buffer));
	if(!buffer)
		return(NULL);

	buffer->buffer = buf;
	buffer->length = length;
	buffer->bitpos = 0;

	return(buffer);
}

int BitSkip(bit_buffer *buffer,int bits) {
	int charpos;

	charpos = (buffer->bitpos += bits)/8;
	if(charpos > buffer->length) {
		buffer->bitpos -= bits;
		return(-1);
	}

	return(0);
}

int GetBits(bit_buffer *buffer, int bits) {
	int charpos;
	int bitoff;
	unsigned char mask = 1;
	unsigned char bbuf[2];


	if(bits >8)
		return(-2);

	if((buffer->bitpos + bits)/8 > buffer->length)
		return(-1);

	charpos = buffer->bitpos/8;
	bitoff = buffer->bitpos - charpos*8;

	bbuf[0] = buffer->buffer[charpos];
	bbuf[1] = buffer->buffer[charpos+1];

	bbuf[0] <<= bitoff;
	bbuf[1] >>= 8-bitoff;
//printf("%02x : %02x\n",bbuf[0], bbuf[1]);
	bbuf[0] |= bbuf[1];

	// now mask off all but the requested bits!
	bbuf[0] >>= (8-bits);
	mask <<= bits;
	mask -=1;

	bbuf[0] &= mask;

	buffer->bitpos += bits;
	return(bbuf[0]);
}


unsigned int UintV(bit_buffer *buffer)
{
	unsigned int value=0;

	if(!GetBits(buffer,1))
		return(value);

	if(!GetBits(buffer,1)) {
		value = GetBits(buffer,3);
		return(value);
	}

	if(!GetBits(buffer,1)) {
		value = GetBits(buffer,6);
		return(value);
	}

	if(!GetBits(buffer,1)) {
		value = GetBits(buffer,8);
		value <<=8;
		value += GetBits(buffer,8);
		return(value);
	}

	if(!GetBits(buffer,1)) {
		value = GetBits(buffer,32);
		return(value);
	}
}

char *alignment[] = { "left", "right", "top", "middle", "bottom"};
char *halign[]	= {"left", "center", "right"};
char *valign[]	= {"top", "center", "bottom"};
		
int __dump_compressed(bit_buffer *buffer,FILE *out,char recursed)
{
	char quit =0;
	int bbuf;
	int ext1,ext2, ext3;
	char *tmp;

	while ((bbuf = GetBits(buffer,5)) >=0) {

		switch(bbuf) {
			case 0:
				if(recursed)
					return(0);

				tmp = stack_pop();
				if(tmp) 
					fputs(tmp,out);
				else
					fprintf(out,"\nspecial = %02x\n",bbuf);
				break;

			case 1:
				bbuf = GetBits(buffer,8);

				

				if(bbuf == 0x00)	{	// anchor tag
					fprintf(out, "<A name=\"");
					__dump_compressed(buffer,out,1);
					fprintf(out,"\">");
					stack_push("</A>");

				} else if(bbuf == 0x34) {		// an image tag!
					fprintf(out,"<IMG");
					bbuf = GetBits(buffer,8);

					if(bbuf & 0x80) {
						ext1 = UintV(buffer);
						fprintf(out," index=%04x",ext1);
					}

					if(bbuf & 0x40) {
						fprintf(out," alt=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					if(bbuf & 0x20) {
						fprintf(out," src=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\" ");
					} 

					if(bbuf & 0x10) {
						ext1 = UintV(buffer);
						fprintf(out," vspace=\"%u\"", ext1+1);
					}

					if(bbuf & 0x08) {
						ext1 = UintV(buffer);
						fprintf(out," hspace=\"%u\"", ext1+1);
					}

					if(bbuf & 0x04) {
						ext1 = UintV(buffer);
						fprintf(out," border=\"%u\"", ext1+1);
					}

					if(bbuf & 0x02) {
						ext1 = UintV(buffer);
						fprintf(out," align=\"%s\"", alignment[ext1]);
					}

					if(bbuf & 0x01) {
						fprintf(out," embedded! (unsupported)  prepare for the junk! ");
					}

					fprintf(out,">");

				} else if(bbuf == 0x0a) {	// header
					fprintf(out,"<H1");
					ext1 = GetBits(buffer,1);

					if(ext1) {
						fprintf(out," align=\"");
						ext2 = GetBits(buffer,2);
						switch(ext2) {
							case 0:
								fprintf(out,"left");
								break;
							case 1:
								fprintf(out,"center");
								break;
							case 2:
								fprintf(out,"right");
								break;
							default:
								fprintf(out,"unknown");
								break;
						}
						fprintf(out,"\" ");
					}

					fprintf(out,">");
					stack_push("</H1>");
					
				} else if(bbuf == 0x0f) {	// header
					fprintf(out,"<H6");
					ext1 = GetBits(buffer,1);

					if(ext1) {
						fprintf(out," align=\"");
						ext2 = GetBits(buffer,2);
						switch(ext2) {
							case 0:
								fprintf(out,"left");
								break;
							case 1:
								fprintf(out,"center");
								break;
							case 2:
								fprintf(out,"right");
								break;
							default:
								fprintf(out,"unknown");
								break;
						}
						fprintf(out,"\" ");
					}

					fprintf(out,">");
					stack_push("</H6>");


				} else if(bbuf == 0x70) {	// escape to 8-bit text!
					while(bbuf !=0) {
						bbuf = GetBits(buffer,8);
						if(bbuf)
							fputc(bbuf,out);
					}

				} else if(bbuf == 0x71) {	// cmlEnd!
					return(0);

				} else if (bbuf == 0x04) { // font size
					bbuf = GetBits(buffer,3);

					fprintf(out," <font size=\"%u\">",bbuf);

				} else if (bbuf == 0x05) { /* bold */
					fprintf(out,"<B>");
					stack_push("</B>");

				} else if (bbuf == 0x06) { /* italic */
					fprintf(out,"<I>");
					stack_push("</I>");

				} else if (bbuf == 0x09) {		// horizontal rule
					ext1 = GetBits(buffer,5);

					fprintf(out,"<HR align=\"");
					switch(ext1 & 0x06) {
						case 0:
							fprintf(out,"left");
							break;
						case 2:
							fprintf(out,"center");
							break;
						case 4:
							fprintf(out,"right");
							break;
						default:
							fprintf(out,"unknown");
							break;
					}
					fprintf(out,"\"");

				
					if(ext1 & 0x08)
						fprintf(out," noshade");

					if(ext1 & 0x01) {
						ext2 = UintV(buffer);	/* height */
						if(ext1 & 0x10)
							ext3 = GetBits(buffer,8);
						else
							ext3 = UintV(buffer);

						fprintf(out," height=%u width=%u",ext2,ext3);
						if(ext1 & 0x10)
							fprintf(out,"%%");
					}

					fprintf(out,">");
					

//					fprintf(out,"<superscript>");
//					stack_push("</superscript>");

				} else if(bbuf == 0x08) { // wierd hyperlink!
					fprintf(out,"<P align=\"");

					ext1 = GetBits(buffer,2);
					
					switch(ext1 & 0x06) {
						case 0:
							fprintf(out,"left");
							break;
						case 2:
							fprintf(out,"center");
							break;
						case 4:
							fprintf(out,"right");
							break;
						default:
							fprintf(out,"unknown");
							break;
					}
					fprintf(out,"\">");
					continue;


//					fprintf(out,"[link flags = %01x, %02x] ",ext1, ext2);

					if(ext1) {
						ext2 = UintV(buffer);
						fprintf(out," flag=%u index=\"%02x\"",ext1,ext2);
						continue;
					}

//					} else {
						ext2 = GetBits(buffer,8);

						if(ext2 & 0x40) 	{	// samedoc
							if(ext2 & 0x08) 	{	// is fragment
								fprintf(out," href=\"#");
								__dump_compressed(buffer,out,1);
								fprintf(out,"\"");
							}
						} else if(ext2 & 0x04) {	// internal link
							fprintf(out," index=\"%u\"",UintV(buffer));
							if(ext2 & 0x08) 	{	// is fragment
								fprintf(out," href=\"#");
								__dump_compressed(buffer,out,1);
								fprintf(out,"\"");
							}
						} else {
					if(bbuf == 0x08)
						BitSkip(buffer,10);
							if(ext2 && 0x20)	{	// has href
								fprintf(out," href=\"");
								__dump_compressed(buffer,out,1);
								fprintf(out,"\"");
							} else {
								fprintf(out," extindex=\"%u\"",UintV(buffer));
								fprintf(out," hash=\"%u\"",UintV(buffer));
							}

							if(ext2 & 0x02)	{	// title
								fprintf(out," title=\"");
								__dump_compressed(buffer,out,1);
								fprintf(out,"\"");
							}
						}	// end else

//					}

					fprintf(out,">");
					stack_push("</a>");
						
				} else if(bbuf == 0x11) { // hyperlink!
					fprintf(out,"<A");

					ext1 = GetBits(buffer,2);
					ext2 = GetBits(buffer,8);


//					fprintf(out,"[link flags = %01x, %02x] ",ext1, ext2);

					if(ext1) {

						if(ext2 & 0x20) {
							fprintf(out," href=\"");
							__dump_compressed(buffer,out,1);
							fprintf(out,"\"");
						} else {
							ext3 = UintV(buffer);
							fprintf(out," index=\"%02x\" flags=%02x",ext3,ext2);
						}
						if(ext2 & 0x08) {
							fprintf(out," name=");
							__dump_compressed(buffer,out,1);
							fprintf(out,"\"");
						}
					} else {
						fprintf(out," href=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					fprintf(out,">");
					stack_push("</a>");
						

				} else if (bbuf == 0x18) {
					fprintf(out,"<blockquote>");
					stack_push("</blockquote>");

				} else if (bbuf == 0x2a) {	// selection
					ext1 = GetBits(buffer,2);
					ext2 = UintV(buffer);

					fprintf(out,"<select");
					if(ext1 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					if(ext2)
						fprintf(out," size=\"%u\"",ext2);

					fprintf(out,">");
					stack_push("</select>\n");

				} else if (bbuf == 0x2b) {	// select item normal
					fprintf(out,"<option>");
					stack_push("</option>\n");

				} else if (bbuf == 0x2c) {	// select item custom
					ext1 = GetBits(buffer,2);

					fprintf(out,"<option");
					if(ext1 & 0x01)
						fprintf(out," selected");

					if(ext1 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					fprintf(out,">");
					stack_push("</option>\n");

				} else if (bbuf == 0x30) {
				 	ext1 = GetBits(buffer,1);

					if(ext1) {
						ext2 = GetBits(buffer,2);
						ext3 = GetBits(buffer,2);
					} else {
						ext2 = ext3 =0;
					}

					fprintf(out,"<tr halign = %1x valign = %1x>", ext2, ext3);
					stack_push("</tr>\n");
				} else if (bbuf == 0x32) {		// td
					fprintf(out,"<td");
					ext1 = GetBits(buffer,7);

					if(ext1 & 0x01)		{	// hAlign
						ext2 = GetBits(buffer,2);
						fprintf(out," halign=\'%s\'",halign[ext2]);
					}
			
					if(ext1 & 0x02)		{	// vAlign
						ext2 = GetBits(buffer,2);
						fprintf(out," valign=\'%s\'",valign[ext2]);
					}
					
					if(ext1 & 0x04)	 {		// colspan
						ext3 = UintV(buffer);
						fprintf(out," colspan=\'%u\'",ext2);
					}

					if(ext1 & 0x08)	 {		// rowspan
						ext3 = UintV(buffer);
						fprintf(out," rowspan=\'%u\'",ext2);
					}

					if(ext1 & 0x10)	 {		// height
						ext3 = UintV(buffer);
						fprintf(out," height=\'%u\'",ext2);
					}

					if(ext1 & 0x20)	 {		// width
						ext3 = UintV(buffer);
						fprintf(out," width=\'%u\'",ext2);
					}

					fprintf(out,">");
					stack_push("</td>");

				} else if (bbuf == 0x21) {	// form
					fprintf(out,"<form");

					ext1 = UintV(buffer);	// form index
					ext2 = GetBits(buffer,3);	// flags
					fprintf(out," index=\"%u\"", ext1);

					if(ext2 & 0x01)
						ext3 = GetBits(buffer,1);	// post?
					else
						ext3 = 0;

					if(ext3)
						fprintf(out," method=\"post\"");

//					fprintf(out," encoding=\"");
					__dump_compressed(buffer,out,1);
					fprintf(out," ACTION=\"");
					__dump_compressed(buffer,out,1);
					fprintf(out,"\">");
					stack_push("</form>");

				} else if (bbuf == 0x22) {	// input text
					ext1 = UintV(buffer);
					ext2 = UintV(buffer);
					ext3 = GetBits(buffer,2);

					fprintf(out,"<INPUT type=\"text\" size=\"%u\" maxlength=\"%u\"",ext1, ext2);
					if( ext3 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					if( ext3 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					fprintf(out,">");
		

				} else if (bbuf == 0x23) {	// input password
					ext1 = UintV(buffer);
					ext2 = UintV(buffer);
					ext3 = GetBits(buffer,2);

					fprintf(out,"<INPUT type='password' size=\"%u\" maxlength=\"%u\"",ext1, ext2);
					if( ext3 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					if( ext3 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					fprintf(out,">");
		

				} else if (bbuf == 0x25) {		// checkbox
					ext1 = GetBits(buffer,4);
					fprintf(out,"<INPUT type=checkbox");
					if(ext1 & 0x04)
						fprintf(out," CHECKED");

					if(ext1 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					if(ext1 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					if(ext1 & 0x08) {
						fprintf(out," text=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					fprintf(out,">");
					
		
				} else if (bbuf == 0x28) {
					fprintf(out,"<INPUT type=\"hidden\"");
					ext1 = GetBits(buffer,2);
					if( ext1 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					if( ext1 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					fprintf(out,">");
		
//					fprintf(out,"[hidden input flags = %1x]\n",GetBits(buffer,2));

				} else if (bbuf == 0x26) {	// Submit
					fprintf(out,"<INPUT type=\"submit\"");

					ext1 = GetBits(buffer,2);
					if( ext1 & 0x01) {
						fprintf(out," name=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					if( ext1 & 0x02) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}
			
					fprintf(out,">");
		
//					fprintf(out,"[Submit flags = %1x]\n",GetBits(buffer,2));
				} else if (bbuf == 0x27) {
					fprintf(out,"<INPUT type=\"reset\"");

					ext1 = GetBits(buffer,1);
					if(ext1) {
						fprintf(out," value=\"");
						__dump_compressed(buffer,out,1);
						fprintf(out,"\"");
					}

					fprintf(out,">");
//					fprintf(out,"[Reset flags = %1x]\n",GetBits(buffer,1));
				} else if (bbuf == 0x2f) {			// table

					fprintf(out,"<table");

					ext1 = GetBits(buffer,7);
	

					if(ext1 & 0x01) {				// align
						ext2 = GetBits(buffer,2);
						fprintf(out," align=\'%s\'", alignment[ext2]);
					}

					if(ext1 & 0x02) {				// width
						ext2 = UintV(buffer);
						fprintf(out," width=%u",ext2);
					}

					if(ext1 & 0x04) {				// border
						ext2 = UintV(buffer);
						fprintf(out," border=%u", ext2);
					}

					if(ext1 & 0x08) {				// cell spacing
						ext2 = UintV(buffer);
						fprintf(out," cellspacing=%u", ext2);
					}

					if(ext1 & 0x10) {				// cell padding
						ext2 = UintV(buffer);
						fprintf(out," cellpadding=%u", ext2);
					}

					fprintf(out,">\n");
				} else
					fprintf(out," (%02x) ",bbuf);

				break;

			case 2:
				bbuf = GetBits(buffer,8);
				switch(bbuf) {
					case 0xa0:
						fprintf(out,"&nbsp;");
						break;

					case 0x99:
						fprintf(out,"&tm;");
						break;

					case '<':
						fprintf(out,"&lt;");
						break;

					case '>':
						fprintf(out,"&gt;");
						break;

					default:
						fputc(bbuf,out);
						break;
				}
				break;

			case 3:
				fputs("<BR>\n",out);
			case 4:
				fputs("<BR>\n",out);
				break;

			case 5:
				fputc(' ',out);
				break;

			default:
				fputc('a' + (bbuf-6),out);
				break;
		}
	}
	return(0);
}

void
dump_compressed(char *buf, int n,FILE *out)
{
	bit_buffer *buffer;

	if(vflag)
		dump(buf,n);

	fprintf(out,"<HTML>\n\t<HEAD>\n\t\t<TITLE>");
	stack_push("</TITLE>\n\t</HEAD>\n\t<BODY>\n");

	buffer = InitBitBuffer(buf,n);
	if(!buffer)
		return;	/* should never happen */
	
	__dump_compressed(buffer,out,0);
	fprintf(out,"\n\t</BODY>\n</HTML>\n");
	return;
}
		

void
dump_header (struct pi_file *pf, struct DBInfo *ip)
{
  printf ("name: \"%s\"\n", ip->name);
  printf ("flags: 0x%x", ip->flags);
  if (ip->flags & dlpDBFlagNewer) printf (" NEWER");
  if (ip->flags & dlpDBFlagReset) printf (" RESET");
  if (ip->flags & dlpDBFlagResource) printf (" RESOURCE");
  if (ip->flags & dlpDBFlagReadOnly) printf (" READ_ONLY");
  if (ip->flags & dlpDBFlagAppInfoDirty) printf (" APP-INFO-DIRTY");
  if (ip->flags & dlpDBFlagBackup) printf (" BACKUP");
  if (ip->flags & dlpDBFlagOpen) printf (" OPEN");
  printf ("\n");
  printf ("version: %d\n", ip->version);
  printf ("creation_time: %s\n", iso_time_str (ip->createDate));
  printf ("modified_time: %s\n", iso_time_str (ip->modifyDate));
  printf ("backup_time: %s\n", iso_time_str (ip->backupDate));
  printf ("modification_number: %ld\n", ip->modnum);
  printf ("type: '%s', ", printlong(ip->type));
  printf ("creator: '%s'\n", printlong(ip->creator));
  printf ("\n");
}

int 
dump_bitmap(char *buffer,int inlength, char *filename)
{
int i, j, k;
int pos, x, y, tmp;
unsigned char *buf;
int length;
int line,valid_line;
unsigned int flags;
int bits;
int shift;
unsigned int bit_buf;
char *dbuf = NULL,*dbufp;
FILE *out = NULL;


	if(filename)
		if((out = fopen(filename,"w")) == NULL)
			return(-2);

	buf = buffer;

	if(!inlength) {
		length = *(buf++);
		length *=256;
		length += *(buf++);

		length*=2;

		if(!length)
			return(0);
	} else
		length = inlength;

	x=0;
	for(i=0;i<2;i++) {
		x*=256;
		x+= *((unsigned char *) buf++);
	}

	y=0;
	for(i=0;i<2;i++) {
		y*=256;
		y+= *(buf++);
	}

	line=0;
	for(i=0;i<2;i++) {
		line*=256;
		line+= *(buf++);
	}


	flags = *((unsigned char *) buf++) * 256;
	flags |= *(buf++);


	bits = *(buf++);

	valid_line = (x * bits)/8;
	if((x * bits)%8)
		valid_line+=1;

	if(vflag) {
		printf("length (bytes): %u\n",length);
		printf("x= %u\n",x);
		printf("y= %u\n",y);
		printf("byte/row= %u\n",line);
		printf("flags = %04x\n",flags);
		printf("bits per pixel: %u\n",bits);
		printf("version: %u\n",*(buf++));

		for(i=0;i<6;i++) {
			printf("%02x ",*((unsigned char *) buf++));
		}
		putchar('\n');
	} else 
		buf += 7;

// SMJ test
	x = (valid_line * 8)/bits;
	if(out)
		fprintf(out,"P6\n#Created by LinPQA decompile\n%u %u\n255\n",x,y);

	if(vflag) {
		printf("static char * icon[] = {\n");
	
		if(bits ==1) {
			printf("\"%u %u 2 1\",\n",x,y);
			printf("\".      c White\",\n");
			printf("\"*      c Black\",\n");
			printf("\"");
		} else {
			printf("\"%u %u 4 1\",\n",x,y);
			printf("\"*      c Black\",\n");
			printf("\"X      c gray66\",\n");
			printf("\"!      c gray33\",\n");
			printf("\".      c White\",\n");
			printf("\"");
		}
	}

	if(flags & 0x8000) {	 // compressed!!!
		dbuf = malloc(line * (y+5));

		dbufp = dbuf;

		for(i=0; i<y; i++) {	// y
			for(j=0; j<(line +7)/8; j++) {
				bit_buf = *((unsigned char *) buf++);		// read flags
				if(j<line/8)
					tmp = 8;
				else
					tmp = line%8;

				for(k=0; k<tmp; k++) {
					if(bit_buf & 0x80)
						*(dbufp++) = *(buf++);
					else
						*(dbufp++) = *(dbufp - line);
					bit_buf <<=1;
				}
			}
		}
				
	buf = dbuf;
	}
			
	
	for(i=0; i<y; i++) {
		if(vflag)
			printf("\",\n\"");
		for(j=0; j<valid_line ; j++) {
			if(bits ==1)
				for(x = 0x80; x ; x>>=1) {
					if(buf[i*line +j] &x) {
						if(vflag)
							putchar('*');
						if(out) {
							fputc(0x00,out);
							fputc(0x00,out);
							fputc(0x00,out);
						}
					} else {
						if(vflag)
							putchar('.');
						if(out) {
							fputc(0xff,out);
							fputc(0xff,out);
							fputc(0xff,out);
						}
					}
				}
			else 
				for(shift =6, x=0xc0; x; x>>=2,shift-=2)	{
					bit_buf = buf[i * line +j] & x;
					bit_buf >>= shift;
		
					switch(bit_buf & 0x03) {
						case 0:
							if(vflag)
								putchar('.');
							if(out) {
								fputc(0xff,out);
								fputc(0xff,out);
								fputc(0xff,out);
							}
							break;
							
						case 1:
							if(vflag)
								putchar('!');
							if(out) {
								fputc(0xa0,out);
								fputc(0xa0,out);
								fputc(0xa0,out);
							}
							break;
		
						case 2:
							if(vflag)
								putchar('X');
							if(out) {
								fputc(0x54,out);
								fputc(0x54,out);
								fputc(0x54,out);
							}
							break;
		
						default:
						case 3:
							if(vflag)
								putchar('*');
							if(out) {
								fputc(0x00,out);
								fputc(0x00,out);
								fputc(0x00,out);
							}
							break;
					}
				} // for x
		}	// for j
	}	// for i


	if(vflag)
		printf("\"\n};\n");
	if(!inlength)
		length +=2;

	if(dbuf)
		free(dbuf);

	if(out)
		fclose(out);

	return(length);	/* +2 to make it include the size word */
}


void
dump_app_info (struct pi_file *pf, struct DBInfo *ip)
{
  void *app_info;
  int app_info_size;

int i, pos,x, y, tmp;
char *buf;

  if (pi_file_get_app_info (pf, &app_info, &app_info_size) < 0) {
    printf ("can't get app_info\n\n");
    return;
  }

  printf ("app_info_size %d\n", app_info_size);

	if(vflag) {
		dump (app_info, app_info_size);
		printf ("\n");
	}

	buf = app_info;

	pos = 8;

	tmp = buf[pos++];
	tmp *=256;
	tmp += buf[pos++];

	tmp*=2;

	printf("Version: %s\n",buf+pos);
	pos += tmp;

	tmp = buf[pos++];
	tmp *=256;
	tmp += buf[pos++];

	tmp*=2;

	printf("Title: %s\n",buf+pos);
	pos += tmp;

	if(vflag)
		printf("ICON:\n");

	pos += dump_bitmap(buf+pos,0,"LgIcon.pnm");

	if(vflag)
		printf("SMALL ICON:\n");

	pos += dump_bitmap(buf+pos,0,"SmIcon.pnm");
}

void
list_records (struct pi_file *pf, struct DBInfo *ip)
{
int entnum;
int size;
unsigned long type, uid;
int id;
void *buf;
int nentries;
int attrs, cat;
char line[255];


	pi_file_get_entries (pf, &nentries);

	printf ("entries\n");
	printf ("index\tsize\tattrs\tcat\tuid\n");
	for (entnum = 0; entnum < nentries; entnum++) {
		if (pi_file_read_record (pf, entnum, &buf, &size,
			&attrs, &cat, &uid) < 0) {
   			printf ("error reading %d\n\n", entnum);
   			return;
   		}

		if(vflag)
   			printf ("%d\t%d\t0x%x\t%d\t0x%lx\n", entnum, size, attrs, cat, uid);

		dump_pqa(buf,size);
   		printf ("\n");
   	}

	printf ("\n");

	((char *)buf)[0]=0;
	strcat(buf,"for i in *.html ; do sed -e '");

	for(id=0;id<num_htmls;id++) {
		sprintf(line,"s/index=\"%02u\" flags=04/HREF=\"%s\"/g ; ",id,htmls[id]);
		strcat(buf,line);
	}

	strcat(buf,"' <$i >$i.new ; mv $i.new $i ; done");

//	printf("%s\n",buf);

	system(buf);
	
	printf("\n");
	
}

void
dump_record (struct pi_file *pf, struct DBInfo *ip, int record)
{
int size;
unsigned long type, uid;
int id;
void *buf;
int attrs, cat;

    printf ("entries\n");
    printf ("index\tsize\tattrs\tcat\tuid\n");
    if (pi_file_read_record (pf, record, &buf, &size,
			     &attrs, &cat, &uid) < 0) {
      printf ("error reading record #%d\n\n", record);
      return;
    }

	if(vflag)
    	printf ("%d\t%d\t0x%x\t%d\t0x%lx\n", record, size, attrs, cat, uid);

	dump_pqa(buf,size);

	printf ("\n");
}

typedef struct {
	int		urlOffset;
	short	urlLength;
	int		dataOffset;
	short	dataLength;
	char	contentType;
	char	compressionType;
	int		uncompressedSize;
	char	flags;
	char 	reserved;
}	__attribute__ ((packed)) pqa_header;

void dump_pqa(char *buf,int size)
{
	pqa_header	*head;
	int i,start,end;
	char url[100];
	FILE *outfile;

	head = (pqa_header *) buf;

	printf("Content Type = %u\n",head->contentType);
	printf("Compression Type = %u\n",head->compressionType);

	if(head->compressionType) 
		printf("Uncompressed size = %u bytes\n",ntohl(head->uncompressedSize));

	printf("flags = %08x\n",head->flags);

	if(head->contentType == 4) {
		printf("URL: ");
		start = ntohl(head->urlOffset);
		end = start + ntohs(head->urlLength);

		memcpy(url,&(buf[start]),end-start);
		url[end-start] =0;

		printf("%s\n",url);

		htmls[num_htmls++] = strdup(url);

		outfile = fopen(url,"w");

		dump_compressed(&(buf[ntohl(head->dataOffset)]), ntohs(head->dataLength),outfile);
		fclose(outfile);

		if(vflag)
			dump_compressed(&(buf[ntohl(head->dataOffset)]), ntohs(head->dataLength),stdout);
	}

	if(head->contentType == 5) {
		printf("URL: ");
		start = ntohl(head->urlOffset);
		end = start + ntohs(head->urlLength);

		memcpy(url,&(buf[start]),end-start);
		url[end-start] =0;

		printf("%s\n",url);

		if(vflag) {
			printf("data:\n\n");
			dump(&(buf[ntohl(head->dataOffset)]), ntohs(head->dataLength));
			printf("\nimage:\n");
		}

		dump_bitmap(&(buf[ntohl(head->dataOffset)]), ntohs(head->dataLength),url);
	}

}


void
dump_pqa_record (struct pi_file *pf, struct DBInfo *ip, int record)
{
  int size;
  unsigned long type, uid;
  int id;
  void *buf;
  int attrs, cat;

  if (ip->flags & dlpDBFlagResource) {
    printf ("entries\n");
    printf ("index\tsize\ttype\tid\n");
    if (pi_file_read_resource (pf, record, &buf, &size, &type, &id) < 0) {
      printf ("error reading resource #%d\n\n", record);
      return;
    }
    printf ("%d\t%d\t%s\t%d\n", record, size, printlong(type), id);
    dump(buf,size);
  } else {
    printf ("entries\n");
    printf ("index\tsize\tattrs\tcat\tuid\n");
    if (pi_file_read_record (pf, record, &buf, &size,
			     &attrs, &cat, &uid) < 0) {
      printf ("error reading record #%d\n\n", record);
      return;
    }
    printf ("%d\t%d\t0x%x\t%d\t0x%lx\n", record, size, attrs, cat, uid);
    dump_pqa(buf,size);
  }

  printf ("\n");
}
