/* $Id: gtklesson.c,v 1.18 2005/03/28 16:24:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "../li18n.h"
#include "../lesson.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "gtkconfig.h"
#include "util.h"
#include "gtklesson.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

enum
{
     LESSON_NAME,
     LESSON_MEANINGS,
     LESSON_PATH,
     LESSON_COLUMNS
};

static GtkListStore* create_lesson_list (void);
static void refresh_list (GtkTreeView *tree, void *data);
static gboolean check_sessions_for_lesson (lingLesson *lesson);

static GtkListStore*
create_lesson_list (void)
{
     GtkListStore *list;
     GtkTreeIter iter;
     lingLesson *tmp = settings->prefs->lessons;

     /* create the list and fill it with data */
     list = gtk_list_store_new (LESSON_COLUMNS, G_TYPE_STRING, G_TYPE_INT,
                                G_TYPE_STRING);

     while (tmp)
     {
          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              LESSON_NAME, tmp->type,
                              LESSON_MEANINGS, 
                              ling_lesson_get_meaning_amount (tmp),
                              LESSON_PATH, ling_lesson_get_path (tmp),
                              -1);
          tmp = tmp->next;
     }
     return list;
}

static gboolean
check_sessions_for_lesson (lingLesson *lesson)
{
     /* currently there are only learn and training sessions */
     /* TODO: training sessions */
     if (settings->learn_session)
          return learn_session_check_lesson (settings->learn_session, lesson);
     return FALSE;
}

static void
refresh_list (GtkTreeView *tree, void *data)
{
     GtkListStore *list = create_lesson_list ();
     gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
     g_object_unref (list);
     return;
}

void 
gtklesson_activate (void)
{
     GtkWidget *win_lesson;
     GtkWidget *box_main;
     GtkWidget *frm_main;
     GtkWidget *align;

     GtkWidget *box_btns;
     GtkWidget *box_addrem;
     GtkWidget *btn_add;
     GtkWidget *btn_remove;
     GtkWidget *btn_close;

     GtkWidget *view_tree;
     GtkCellRenderer *renderer;
     GtkListStore *list;
     GtkTreeSelection *select;

     GtkWidget *scr_win;

     /* main window */
     win_lesson = hig_window_new (_("Lessons"));
     gtk_window_set_transient_for (GTK_WINDOW (win_lesson),
                                   GTK_WINDOW (main_app->window));
     gtk_window_set_position (GTK_WINDOW (win_lesson),
                              GTK_WIN_POS_CENTER_ON_PARENT);
     gtk_window_set_modal (GTK_WINDOW (win_lesson), TRUE);
     gtk_window_set_default_size (GTK_WINDOW (win_lesson), 300, 200);

     /* main box */
     box_main = hig_vbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_main), 2 * BOX_SPACING);
     gtk_container_add (GTK_CONTAINER (win_lesson), box_main);

     /* frame */
     frm_main = hig_frame_new (_("Lessons"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_main), 0);
     align = gtk_bin_get_child (GTK_BIN (frm_main));
     gtk_box_pack_start (GTK_BOX (box_main), frm_main, TRUE, TRUE, 0);
     
     /* create tree view */
     list = create_lesson_list ();
     view_tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (list);
     
     /* register object */
     main_app->objects = event_append_listener (main_app->objects,
                                                view_tree,
                                                "lessons-changed",
                                                EV_CALLBACK (refresh_list),
                                                NULL);
     g_signal_connect (G_OBJECT (view_tree), "destroy",
                       G_CALLBACK (util_event_unregister), NULL);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_tree),
                                                  LESSON_NAME, 
                                                  _("Lesson Name"),
                                                  renderer, "text",
                                                  LESSON_NAME, NULL);
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_tree),
                                                  LESSON_MEANINGS, 
                                                  _("Meaning Amount"),
                                                  renderer, "text",
                                                  LESSON_MEANINGS, NULL);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_tree),
                                                  LESSON_PATH, 
                                                  _("Lesson Path"),
                                                  renderer, "text",
                                                  LESSON_PATH, NULL);
     /* set selection mode */
     select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view_tree));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

     /* scrolled window */
     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (scr_win), view_tree);
     gtk_container_add (GTK_CONTAINER (align), scr_win);

     /* button boxes */
     box_btns = hig_hbox_new ();
     gtk_box_set_spacing (GTK_BOX (box_btns), 2 * BOX_SPACING);
     gtk_container_set_border_width (GTK_CONTAINER (box_btns), 0);
     gtk_box_pack_start (GTK_BOX (box_main), box_btns, FALSE, TRUE, 0);

     /* button box */
     box_addrem = gtk_hbutton_box_new ();
     gtk_box_pack_start (GTK_BOX (box_btns), box_addrem, TRUE, TRUE, 0);
     gtk_button_box_set_layout (GTK_BUTTON_BOX (box_addrem),
                                GTK_BUTTONBOX_START);
     gtk_box_set_spacing (GTK_BOX (box_addrem), 2 * BOX_SPACING);
     
     /* buttons add and remove */
     btn_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
     gtk_container_add (GTK_CONTAINER (box_addrem), btn_add);
     gtk_tooltips_set_tip (main_app->tips, btn_add,
                           _("Adds a lesson file to the active session"),
                           NULL);
     g_signal_connect_swapped (G_OBJECT (btn_add), "clicked",
                               G_CALLBACK (gtklesson_add_lesson), view_tree);

     btn_remove = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
     gtk_container_add (GTK_CONTAINER (box_addrem), btn_remove);
     gtk_tooltips_set_tip (main_app->tips, btn_remove,
                           _("Removes a selected lesson file from the "
                             "active session"), NULL);
     g_signal_connect_swapped (G_OBJECT (btn_remove), "clicked",
                               G_CALLBACK (gtklesson_remove_lesson),
                               view_tree);

     /* button close */
     btn_close = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
     gtk_box_pack_start (GTK_BOX (box_btns), btn_close, FALSE, FALSE, 0);
     g_signal_connect_swapped (G_OBJECT (btn_close), "clicked",
                               G_CALLBACK (gtk_widget_destroy), win_lesson);
     gtk_widget_show_all (win_lesson);
     return;
}

void
gtklesson_add_lesson (void)
{
     gchar *file = util_run_fileselection (_("Choose Lesson..."),
                                           GTK_FILE_CHOOSER_ACTION_OPEN);
     if (!file)
          return;

     if (lesson_lesson_in_list (settings->prefs->lessons, file))
     {
          util_info (_("Lesson is already loaded!"));
          g_free (file);
          return;
     }

     if (!lesson_load_lesson_to_prefs (settings->prefs, file))
     {
          error_warning (_("Lesson could not be loaded!"),
                         _("The selected lesson could not be loaded."),
                         _("Please make sure, the lesson is a proper "
                           "LingoTeach lesson file."));
          g_free (file);
          return;
     }
     g_free (file);

     /* emit state change */
     event_emit_signal (main_app->objects, "lessons-changed");

     return;
}

void
gtklesson_remove_lesson (GtkTreeView *tree)
{
     GtkTreeSelection *select = gtk_tree_view_get_selection (tree);
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     GtkTreeIter iter;
     GtkListStore *list;
     gchar *name = NULL;
     gchar *path = NULL;
     lingLesson *tmp = settings->prefs->lessons;

     /* TODO maybe set the button insensitive, if no row is selected */
     if (gtk_tree_selection_get_selected (select, &model, &iter))
     {
          gtk_tree_model_get (model, &iter, 
                              LESSON_NAME, &name, LESSON_PATH, &path, -1);
          while (tmp)
          {
               /* remove lesson on match */
               if (g_utf8_collate (path, ling_lesson_get_path (tmp)) == 0)
               {
                    /* check for active sessions using the lesson */
                    if (!check_sessions_for_lesson (tmp))
                         settings->prefs->lessons = 
                              ling_lesson_remove (settings->prefs->lessons,
                                                  tmp);
                    else
                         error_warning (_("Lesson cannot be removed!"),
                                        _("The lesson is used by a session or "
                                          "another part of the application "
                                          "and thus cannot be removed."),
                                        _("Try to close the part of the "
                                          "application, which makes use of "
                                          "the lesson and then remove it."));
                    break;
               }
               tmp = tmp->next;
          }
          g_free (name);
          g_free (path);
          
          list = create_lesson_list ();
          gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
          g_object_unref (list);

          /* emit state change */
          event_emit_signal (main_app->objects, "lessons-changed");
     }
     return;
}
