#include <arpa/inet.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include "cmd.h"
#include "cmd_lcp3.h"
#include "global_vars.h"

struct list_anchor_t lcp3_ordq;

/* outgoing */
void *get_output_buffer(int cmd)
{
    void *buf = malloc(get_lcp3_pack_size(cmd));
    if ( !buf )
    {
	syslog(LOG_ERR, "get_output_buffer(): not enough memory");
    }
    return(buf);
}

int lcp3_cmd_queue(int cmd, struct client_t *who, void* info, int infosize)
{
    struct t_lcp3_ordq_item* newcmd;
    if ( !(newcmd = (struct t_lcp3_ordq_item*)ll_list_add(&lcp3_ordq, sizeof(struct t_lcp3_ordq_item))) )
    {
		syslog(LOG_ERR, "Out of memory. Unable to queue command.");
		return(-1);
    }
#ifdef DEBUG
    syslog(LOG_DEBUG, "queueing cmd %d", cmd);
#endif
    newcmd->cmd = cmd;
    newcmd->who = who;
    newcmd->cnt = 0;
    newcmd->prev_send = 0;
    newcmd->infosize = infosize;
    newcmd->specline = NULL;
    if ( infosize )
		memcpy(newcmd->info, info, infosize);
    // the new cmd is initialized, now try to send it.
    lcp3_ordq_checksend(who); // sends it if possible
    return(0);
}

int lcp3_cmd_direct(int cmd, int tcp_socket, void* info, int infosize)
{
    int sent, len;
    struct t_lcp3_cmd *pcmd;
    void *buf = get_output_buffer(cmd); // logs LOG_ERR on error
    if ( !buf ) return(-1);
    // fill buffer
    pcmd = buf;
    len = get_lcp3_pack_size(cmd);
//    memset(pcmd, 0, len);
    if ( infosize )
    {
	void *pinfo = (void*)((char*)buf + sizeof(struct t_lcp3_cmd));
	memcpy(pinfo, info, infosize);
    }
    pcmd->magic = LC_MAGIC_NR;
    pcmd->version = LCP3_PROTO_VERSION;
    pcmd->cmd = cmd;
    pcmd->info_is_valid = 0;
#ifdef DEBUG
    syslog(LOG_DEBUG, "lcp3_cmd_direct(%d,...)", cmd);
    if ( get_lcp3_pack_size(pcmd->cmd) != len )
	syslog(LOG_DEBUG, "wrong sized packet! cmd %d", cmd);
#endif
    // send the packet
    sent = secure_write(tcp_socket, buf, len);
//    sent = write(tcp_socket, buf, len);
    free(buf);
    if ( sent != len ) return(-1);
    return(0);
}

int lcp3_cmd_send(struct t_lcp3_ordq_item *item)
{
    int sent, len;
    struct t_lcp3_cmd *pcmd;
    void *buf = get_output_buffer(item->cmd); // logs LOG_ERR on error
    if ( !buf ) return(-1);
    // fill buffer
    pcmd = buf;
    len = get_lcp3_pack_size(item->cmd);
//    memset(pcmd, 0, len);
    if ( item->infosize )
    {
        void *pinfo = (void*)((char*)buf + sizeof(struct t_lcp3_cmd));
        memcpy(pinfo, item->info, item->infosize);
    }
    pcmd->magic = LC_MAGIC_NR;
    pcmd->version = LCP3_PROTO_VERSION;
    pcmd->cmd = item->cmd;
    pcmd->info_is_valid = 1;
    pcmd->client_stat = item->who->status;
	if ( item->specline )
	{
		pcmd->line_id = item->specline->id;
		pcmd->line_stat = item->specline->con_stat;
		pcmd->line_channels = item->specline->used_channels;
	}
    else if ( item->who->line )
    {
		pcmd->line_id = item->who->line->id;
		pcmd->line_stat = item->who->line->con_stat;	// LINE (a marker for grep...)
		pcmd->line_channels = item->who->line->used_channels;	// LINE
    }
    else pcmd->info_is_valid = 0;
#ifdef DEBUG
    syslog(LOG_DEBUG, "lcp3_cmd_send(%d,...)", item->cmd);
    if ( get_lcp3_pack_size(pcmd->cmd) != len )
		syslog(LOG_DEBUG, "wrong sized packet! cmd %d", item->cmd);
#endif
    // send the packet
    sent = secure_write(item->who->tcp_sock, buf, len);
//    sent = secure_write(item->who->tcp_sock, item->who->ip, item->who->port, buf, len);
    free(buf);
    item->cnt++;
    item->prev_send = time(NULL);
    if ( sent != len ) return(-1);
    return(0);
}

int lcp3_cmd_broadcast(int cmd, void* info, int infosize, struct line_t *line)
{
    struct client_t *clt = (struct client_t*)cltlist.first;
    struct t_lcp3_ordq_item *item = malloc(sizeof(struct t_lcp3_ordq_item));
    struct t_lcp3_info_tput* tput;
    time_t now = time(NULL);
    if (!item)
    {
		syslog(LOG_ERR, "lcp3_cmd_broadcast(): not enough memory");
		return(-1);
    }
    item->prev = 0;
    item->next = 0;
    item->cmd = cmd;
    item->infosize = infosize;
//    if ( cmd == CBR3_LINESTAT )	item->specline = line;
//    else item->specline = NULL;
	item->specline = line;
    if ( infosize && info )
	memcpy(item->info, info, infosize);
#ifdef DEBUG
    syslog(LOG_DEBUG, "lcp3_cmd_broadcast(%d,...)", cmd);
#endif
    while ( clt )
    {	// only handle LCP3 clients
	// line selected??
	// new in 2.1.3: send broadcast either if the client selected this
	// line or if it has set the getall flag.
	if ( (clt->type == LCS_CLT_LCP3) && ((!line) || (clt->line == line)
			|| (clt->getall == 1)) )
	{
	    item->who = clt;
	    item->cnt = 0;
	    item->prev_send = 0;
	    switch ( cmd )
	    {
		case CBR3_TPUT:
		    tput = (void*)item->info;
		    if ( clt->status == CLT_ONLINE )
			tput->client = now - clt->started;
		    else
			tput->client = 0;
		    break;
	    }
	    lcp3_cmd_send(item);
	}
#ifdef DEBUG
	else if ( clt->type != LCS_CLT_LCP3 )
	    syslog(LOG_DEBUG, "lcp3_cmd_broadcast(): skipping non-LCP3 client");
#endif
	clt = (struct client_t*)clt->next;
    }
    free(item);
    return(0);
}

int lcp3_ack_send(int cmd, struct client_t *who)
{
    int sent, len = get_lcp3_pack_size(CMD3_ACK);
    struct t_lcp3_cmd* pcmd;
    struct t_lcp3_info_ack* pinfoack;
    void *buf = get_output_buffer(CMD3_ACK);
    if ( !buf )
    {
	kick_client(who, 4);
	return(-1);
    }
    pcmd = buf;
    // load data into buffer
    pinfoack = (void*)((char*)buf + sizeof(struct t_lcp3_cmd));
    pcmd->magic = LC_MAGIC_NR;
    pcmd->version = LCP3_PROTO_VERSION;
    pcmd->cmd = CMD3_ACK;
    pcmd->info_is_valid = 1;
    pcmd->client_stat = who->status;
    if ( (pcmd->line_id = who->line->id) )
    {    
	pcmd->line_stat = who->line->con_stat;	// LINE (a marker for grep...)
	pcmd->line_channels = who->line->used_channels;	// LINE
    }
    pinfoack->ackcmd = cmd;
    // send buffer
    sent = secure_write(who->tcp_sock, buf, len);
//    sent = secure_write(who->tcp_sock, who->ip, who->port, buf, len);
    free(buf);
    if ( sent != len ) return(-1);
    return(0);
}


/* incoming */
int lcp3_proc_ack(int cmd, struct client_t *who)
{
    struct t_lcp3_ordq_item *item = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
    void *dum;
#ifdef DEBUG
    syslog(LOG_DEBUG, "got ack for %d", cmd);
#endif
    while ( item )
    {
		if ( (item->who == who) && (item->cmd == cmd) && (item->cnt) )
		{ // remove it
	    	dum = item->next;
	    	list_del(&lcp3_ordq, (struct list_hdr_t*)item);
	    	item = dum;
//20010906	    	continue; // check for errors
			break;
		}
		item = (struct t_lcp3_ordq_item*)item->next;
    }
    lcp3_ordq_checksend(who); // check whether to send a cmd to this client
    return(0);
}

/* init */
int lcp3_init()
{
    list_init(&lcp3_ordq);
    return(0); // OK
}

/* manipulating */
void lcp3_ordq_check() // checks the lcp3_ordq for timeouts
{
    time_t now = time(NULL);
    struct client_t* clt;
    struct t_lcp3_ordq_item *cmd = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
    while ( cmd )
    {
		if ( cmd->cnt )
		{ // cmd has been sent already
	    	if ( cmd->prev_send + LCP3_TMO_ACK < now )
	    	{
				if ( !client_exists(cmd->who) )
				{
		    		list_del(&lcp3_ordq, (struct list_hdr_t*)cmd);
		    		cmd = NULL;
				}
				else if ( cmd->cnt < LCP3_TMO_CNT )
				{	// send it again
				    lcp3_cmd_send(cmd);
				} // cmd resent
				else
				{
		    		clt = cmd->who;
		    		kick_client(clt, 4);
		    		cmd = NULL;
				} // clt kicked
	    	} // timed out?
		} // already sent?
		if ( cmd ) cmd = (struct t_lcp3_ordq_item*)cmd->next;
		else cmd = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
    } // list loop
}

void lcp3_ordq_checksend(struct client_t *who)
{
    int sent_found = 0;
    struct t_lcp3_ordq_item *cmd = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
#ifdef DEBUG
syslog(LOG_DEBUG, "lcp3_ordq_checksend(): entering");
#endif
    while ( cmd && (!sent_found) )
    {
		if ( (cmd->who == who) && (cmd->cnt) ) { sent_found++; break; }
		cmd = (struct t_lcp3_ordq_item*)cmd->next;
    }
    if ( !sent_found ) // check for a unsent cmd of this clt and send it
    {
		cmd = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
		while ( cmd )
		{
		    if ( (cmd->who == who) && (!cmd->cnt) )
		    {
				lcp3_cmd_send(cmd);
				break;
	    	}
	    	cmd = (struct t_lcp3_ordq_item*)cmd->next;
		}
    }
#ifdef DEBUG
    else
    {
		syslog(LOG_DEBUG, "can't send cmd! An already sent cmd is in the queue (cmd %d).",
			    cmd->cmd);
    }
    syslog(LOG_DEBUG, "lcp3_ordq_checksend(): leaving");
#endif
}
