#include <assert.h>
#include <getopt.h>

#include <fstream>
#include <iomanip>
#include <iostream>

#include <libmpeg3.h>

using namespace std;


#define PODXTPRO_RATE 39062.5
#define PODXTPRO_BYTES_PER_SAMPLE 6

#define MP3_BYTES_PER_SAMPLE 4


void
usage()
{
  cerr << "usage: mp32pod [-d <device-file>] <mp3-file>\n";
  exit(1);
}

int
main(int argc, char *argv[])
{
  int bufsamples_pod = 10000;
  float speed = 1;
  float amp = 1;
  string device = "/sys/bus/usb/devices/1-1:1.0/audio_out";
  
  for(;;)
    {
      int c = getopt(argc, argv, "a:d:hs:");

      if(c < 0)
	break;

      switch(c)
	{
	case 'a':
	  amp = atof(optarg);
	  break;

	case 'd':
	  device = optarg;
	  break;

	case 'h':
	  usage();
	  break;

	case 's':
	  speed = atof(optarg);
	  break;

	default:
	  usage();
	}
    }

  argc -= optind;
  argv += optind;

  if(argc != 1)
    usage();

  char *mp3name = argv[0];

  if(!mpeg3_check_sig(mp3name))
    {
      cerr << "File \"" << mp3name << "\" doesn't exist or is not an MP3 file\n";
      return 2;
    }
  
  mpeg3_t *file = mpeg3_open(mp3name);

  if(!file)
    {
      cerr << "File \"" << mp3name << "\" couldn't be opened\n";
      return 3;
    }

  ofstream pod;

  if(device.substr(3) == "wh:")
    {
      // open ALSA device:
    }

  ofstream usb(device.c_str());

  if(!usb)
    {
      cerr << "Couldn't open USB device \"" << device << "\"\n";
      return 3;
    }

  // query format data:
  if(!mpeg3_has_audio(file))
    {
      cerr << "File \"" << mp3name << "\" doesn't contain audio information\n";
      return 4;
    }

  if(mpeg3_total_astreams(file) != 1)
    {
      cerr << "Only one audio stream supported\n";
      return 4;
    }

  int stream = 0;
  int channels = mpeg3_audio_channels(file, stream);

  if(channels != 2)
    {
      cerr << "Only stereo streams supported\n";
      return 4;
    }

  const int rate_mp3 = mpeg3_sample_rate(file, stream);
  long samples_mp3 = mpeg3_audio_samples(file, stream);
  // cout << rate << endl << samples << endl;

  const double rr = PODXTPRO_RATE / speed / rate_mp3;  // rate ratio
  const int bufsamples_mp3 = (int)ceil(bufsamples_pod / rr) + 2;
  const int bufsize_pod = bufsamples_pod * PODXTPRO_BYTES_PER_SAMPLE;
  const int bufsize_mp3 = bufsamples_mp3 * MP3_BYTES_PER_SAMPLE;
  short *buf0_mp3 = (short *)malloc(bufsize_mp3);  // right buffer
  short *buf1_mp3 = buf0_mp3 + bufsamples_mp3;     // left buffer
  unsigned char *buf_pod = (unsigned char *)malloc(bufsize_pod);
  memset(buf_pod, 0, bufsize_pod);

  int pos_pod = 0;
  int pos1i_mp3_prev = -1;
  int read_mp3_prev = 0;

  while(samples_mp3 > 0)
    {
      // fractional positions in MP3 stream:
      double pos0_mp3 = pos_pod / rr;
      double pos1_mp3 = (pos_pod + bufsamples_pod - 1) / rr;

      // compute integer positions in MP3 stream needed for this request:
      int pos0i_mp3 = (int)floor(pos0_mp3);
      int pos1i_mp3 = (int)ceil(pos1_mp3);

      // decode MP3 stream:
      int ofs;
      int read_mp3 = pos1i_mp3 - pos0i_mp3 + 1;
      // cout << read_mp3 << ' ' << bufsamples_mp3 << endl;
      assert(read_mp3 <= bufsamples_mp3);

      if(pos0i_mp3 == pos1i_mp3_prev)
	{
	  // last sample can be reused, so read one less:
	  --read_mp3;
	  ofs = 1;

	  // and copy last sample:
	  buf0_mp3[0] = buf0_mp3[read_mp3_prev - 1];
	  buf1_mp3[0] = buf1_mp3[read_mp3_prev - 1];
	  // cout << "here1\n";
	}
      else
	{
	  ofs = 0;
	  // cout << "here2\n";
	}

      if(read_mp3 > samples_mp3)
	read_mp3 = samples_mp3;

      // read and decode the MP3 stream:
      int ret = 0;
      /*
      buf0_mp3[ofs + read_mp3 - 1] = 12345;
      buf0_mp3[ofs + read_mp3 - 0] = 12345;
      buf1_mp3[ofs + read_mp3 - 1] = 12345;
      buf1_mp3[ofs + read_mp3 - 0] = 12345;
      */
      ret |= mpeg3_read_audio(file, 0, buf0_mp3 + ofs, 0, read_mp3, stream);
      ret |= mpeg3_reread_audio(file, 0, buf1_mp3 + ofs, 1, read_mp3, stream);
      /*
      cout << buf0_mp3[ofs + read_mp3 - 1] << ' ' << buf0_mp3[ofs + read_mp3 - 0] << ' ' << buf1_mp3[ofs + read_mp3 - 1] << ' ' << buf1_mp3[ofs + read_mp3 - 0] << endl;
      */

      // usb.write((const char *)(buf1_mp3 + ofs), read_mp3 * 2);

      if(ret)
	{
	  cerr << "Error decoding MP3 stream\n";
	  return 5;
	}

      read_mp3_prev = read_mp3;
      pos1i_mp3_prev = pos1i_mp3;

      // convert data to PodXT Pro format:
      // cout << "converting MP3 from " << pos0_mp3 << " to " << pos1_mp3 << " (" << read_mp3 << " samples read)\n";
      // memset(buf_pod, 0, bufsize_pod);

      for(int i = 0; i < bufsamples_pod; i++)
	{
	  double pos_mp3 = (pos_pod + i) / rr;
	  int ofs_mp3 = (int)floor(pos_mp3) - pos0i_mp3;
	  assert(ofs_mp3 >= 0);
	  assert(ofs_mp3 < bufsize_mp3 - 1);
	  double t1 = pos_mp3 - floor(pos_mp3);
	  double t0 = 1 - t1;
	  // cout << ofs_mp3 << ' ' << t << endl;
	  // int v[2];
	  int v0 = (int)((t0 * buf0_mp3[ofs_mp3] + t1 * buf0_mp3[ofs_mp3 + 1]) * 256 * amp + 0.5);
	  int v1 = (int)((t0 * buf1_mp3[ofs_mp3] + t1 * buf1_mp3[ofs_mp3 + 1]) * 256 * amp + 0.5);

	  /*
	  for(int c = 2; c--;)
	    for(int b = 3; b--;)
	      {
		buf_pod[(2 * i + c) * PODXTPRO_BYTES_PER_SAMPLE + b] = *((unsigned char *)(v + c) + b);
		// int oo = (2 * i + c) * PODXTPRO_BYTES_PER_SAMPLE + b;
		// assert((oo & 0xf) < 3);
	      }

	  // memset(buf_pod + i * 6 + 3, 0, 3);
	  */
	  memcpy(buf_pod + i * PODXTPRO_BYTES_PER_SAMPLE, &v0, 3);
	  memcpy(buf_pod + i * PODXTPRO_BYTES_PER_SAMPLE + 3, &v1, 3);
	}

      usb.write((const char *)buf_pod, bufsize_pod);

      pos_pod += bufsamples_pod;
      samples_mp3 -= read_mp3;
    }

  return 0;
}
