/*!!***************************************************************************
 *!! FILE NAME
 *!!	$Source: /var/cvs/pub/repository/LinCVS/src/LogDialogImpl.cpp,v $
 *!!
 *!! AUTHOR
 *!!	$Author: joseh $
 *!!
 *!! DATE, REVISION AND STATE
 *!!	$Date: 2001/10/10 21:13:24 $
 *!!	$Revision: 1.14 $
 *!!	$State: Exp $
 *!!
 *!! DESCRIPTION
 *!!    Implementation of LinCVS log dialog.
 *!!
 *!!    Based on code create by Bernd Gehrmann
 *!!    Copyright (C) 1999 Bernd Gehrmann
 *!!    bernd@@physik.hu-berlin.de
 *!!
 *!!	Copyright (C) 2001 The LinCVS development team.
 *!!	This program is free software; you can redistribute it and/or modify
 *!!	it under the terms of the GNU General Public License as published by
 *!!	the Free Software Foundation; either version 2, or (at your option)
 *!!	any later version.
 *!!
 *!!	This program is distributed in the hope that it will be useful,
 *!!	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *!!	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *!!	GNU General Public License for more details.
 *!!
 *!!**************************************************************************/

#include "config.h"
#include "ac_system_defs.h"

#include <unistd.h>

#include <qtextcodec.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qwhatsthis.h>
#include <qlayout.h>
#include <qcolordialog.h>

#include "DiffDialogImpl.h"
#include "AnnotateDialogImpl.h"
#include "LogDialogImpl.h"
#include "CmdThread.h"
#include "globals.h"
#include "PixmapCache.h"
#include "colortab.h"



LogDialogImpl::LogDialogImpl( QWidget* parent,  const char* name, bool modal, WFlags fl )
    : LogDialog( parent, name, modal, fl )
{
    if (bUseSmallIcons) setIcon( findEmbeddedPixmap( "LogTree16x16" ) );
    else setIcon( findEmbeddedPixmap( "LogTree32x32" ) );

    basedir= "";

    items.setAutoDelete(true);
    tags.setAutoDelete(true);

    for (int i=0; i<2; i++)
    {
	    tree->setSelectionColor (i, SELECTION_COLOR_OF_LOGDLG[i]);
	    list->setSelectionColor (i, SELECTION_COLOR_OF_LOGDLG[i]);
    }

    /* Strech the revision, author, date and comment boxes a bit more
     * than the rest of the neighbouring widgets
     */
    RevFrameLayout->setColStretch(1,1);
    RevFrameLayout->setColStretch(3,3);

    pColorBtnPalette = new QPalette(buttonAColor->palette());
    pColorBtnPalette->setColor(QColorGroup::Button, SELECTION_COLOR_OF_LOGDLG[0]);
    buttonAColor->setPalette(*pColorBtnPalette);
    pColorBtnPalette->setColor(QColorGroup::Button, SELECTION_COLOR_OF_LOGDLG[1]);
    buttonBColor->setPalette(*pColorBtnPalette);
}


/*  
 *  Destroys the object and frees any allocated resources
 */
LogDialogImpl::~LogDialogImpl()
{
	delete(pColorBtnPalette);
}


bool LogDialogImpl::parseCvsLog (QString BaseDir, QString name, QString connectMethod)
{
    m_connectMethod = connectMethod;	//hack
    QString command;
    QStringList strlist;
    QString tag, rev, date, author, state, comment;
    enum { Begin, Tags, Admin, Revision,
	   Author, Branches, Comment, Finished } ParserState;

    fname = name;
    
    setCaption ("CVS Log - " + name);
    basedir = BaseDir;

    bool usingRsh = false;
    bool usingSsh = false;
    
    if( (QString::compare(connectMethod, "ext") == 0)
	|| (QString::compare(connectMethod, "") == 0) )
    {//rsh access
	    usingRsh = true;
		    
       command = "cd " + BaseDir + " && ";
						 
       if (cvsRsh.find("ssh") == 0) {
          if (!sshAgentIsRunning) startSshAgent();
          usingSsh = true;

          command += "set-ssh-commit-env.sh \"";
          command += "env CVS_RSH=" + cvsRsh + " cvs log " + name + "\"";
       } else {//rsh
          command += "env CVS_RSH=" + cvsRsh + " cvs log " + name + " 2> /dev/null";
       }
    }
								    
    if(!usingRsh) {
       command = "cd " + BaseDir + " && cvs log ";
       command += name;
       command += " 2> /dev/null";
    }
    
    CmdThread * Thread = new CmdThread( command );
    Thread->start();
    Thread->wait();
    
    long len = Thread->m_outputQueue.count();

    ParserState = Begin;
    QString line;

    QTextCodec *eucjp, *sjis;

    QString lang = QString(QTextCodec::locale()).left(2);
    if(lang == "ja"){
        eucjp = QTextCodec::codecForName("EUC-JP");
        sjis = QTextCodec::codecForName("Shift_JIS");
    }
    else {
        eucjp = NULL;
        sjis = NULL;
    }
    // must skip 6 lines
    for (long i = 0; i < len; i ++ ) {
        line = Thread->m_outputQueue[i];

        if(lang == "ja"){
            QString line_codec = line;
            line_codec = eucjp->toUnicode(line.data(), line.length());
            if (!eucjp->canEncode(line_codec))
                line_codec = sjis->toUnicode(line.data(), line.length());
            LogOutput->insertLine (line_codec);
        }
        else
            LogOutput->insertLine (line);

	switch (ParserState)
	{
	case Begin:
	  if (line == "symbolic names:")
	    ParserState = Tags;

	  /* We'll drop through here because the log output
	   * doesn't always have a symbolic names section.
	   * This is particularly true if we have a .cvsrc file
	   * with the following line: "log -N"
	   */

	case Admin:
	  if (line == "----------------------------") {
	    ParserState = Revision;
	  }
	  break;

	case Tags:
	  if (line[0] == '\t') {
	    strlist = QStringList::split (':', line);
	    QString rev = strlist[1].simplifyWhiteSpace();
	    QString tag = strlist[0].simplifyWhiteSpace();
	    QString branchpoint = "";

	    if (rev.contains (".0."))
	      {
		/* For a revision number such as 2.10.0.6, we want:
		 * branchpoint = "2.10"
		 * rev = "2.10.6"
		 */
		branchpoint = QString(rev).replace (QRegExp ("\\.0\\.[0-9]+$"), "");
		rev.replace (QRegExp ("\\.0\\."), ".");
	      }

	    if (rev != "1.1.1")
	      {
		TagInfo *taginfo = new TagInfo;
		taginfo->rev = rev;
		taginfo->tag = tag;
		taginfo->branchpoint = branchpoint;
		tags.append(taginfo);
	      }
	  }
	  else ParserState = Admin;
	  break;

	case Revision:
	  strlist = QStringList::split (' ', line);
	  rev = strlist[1];
	  ParserState = Author;
	  break;

	case Author:
	  /* Extract the date, author and state subcomponents of a
	   * line that looks something like this:
	   * date: 2001/05/16 22:04:07;  author: joseh;  state: Exp;  lines: +11 -11 
	   */
	  strlist = QStringList::split (';', line);
	  date = (QStringList::split (": ", strlist[0]))[1];
	  author = (QStringList::split (": ", strlist[1]))[1];
	  state = (QStringList::split (": ", strlist[2]))[1];
	  comment = "";
	  ParserState = Branches;
	  break;

	case Branches:
	  if (qstrncmp(line, "branches:", 9) != 0)
	    {
	      comment = line;
	      ParserState = Comment;
	    }
	  break;

	case Comment:
	  if (line == "----------------------------") {
	    ParserState = Revision;
	  } else if (line == "=============================================================================") {
	    ParserState = Finished;
	  }
	  if (ParserState == Comment) {// still in message
	    comment += QString("\n") + QString(line);
	  } else {
	    // Create tagcomment
	    QString tagcomment, BranchTagList, RegularTagList, branch;

	    branch = QString(rev).replace (QRegExp ("\\.[0-9]+$"), "");

	    // Build tagcomment and taglist:
	    // tagcomment contains Tags, Branchpoints and 'On Branch's
	    // taglist contains tags (without prefix) and Branchpoints
	    QListIterator<TagInfo> it(tags);
	    for (; it.current(); ++it)
	      {
		if (rev == it.current()->rev)
		  {
		    // regular tag
		    tagcomment += "\nTag: " + it.current()->tag;
		    RegularTagList += "\nTag: " + it.current()->tag;
		  }
		if (rev == it.current()->branchpoint)
		  {
		    // branch origin tag
		    tagcomment += "\nBranch-Point: " + it.current()->tag;
		    BranchTagList += "\nBranch-Point: " + it.current()->tag;
		  }
		else if (branch == it.current()->rev)
		  {
		    // branch tag
		    tagcomment += "\nBranch-Tip: " + it.current()->tag;
		    BranchTagList += "\nBranch-Tip: " + it.current()->tag;

		    // Since branch tags are floating tags, will associate
		    // the tag with the revision at the tip of the branch.
		    // Note that cvs always reports the latest revision first.
		    it.current()->rev = rev;
		  }
	      }

	    // remove leading '\n'
	    if (!tagcomment.isEmpty())
	      tagcomment.remove(0, 1);
	    if (!BranchTagList.isEmpty())
	      BranchTagList.remove(0, 1);
	    if (!RegularTagList.isEmpty())
	      RegularTagList.remove(0, 1);

       if (lang == "ja"){
           QString comment_codec = comment;
           comment = eucjp->toUnicode(comment_codec.data(), comment_codec.length());
           if (!eucjp->canEncode(comment))
               comment = sjis->toUnicode(comment_codec.data(), comment_codec.length());
       }

	    list->addRevision(rev, author, comment, date);
	    tree->addRevision(rev, author, date, state, comment,
			      BranchTagList, RegularTagList, tagcomment);

	    RevisionInfo *item = new RevisionInfo;
	    item->rev = rev;
	    item->date = date;
	    item->author = author;
	    item->state = state;
	    item->comment = comment;
	    item->tagcomment = tagcomment;
	    items.append(item);
	    // Holding data redundant is bad...
	    // ... but I don't have a better idea
	  }

	default :
	  break;
	}
    }

    delete Thread;

    tree->collectConnections();
    tree->recomputeCellSizes();
    layout()->activate();

    return true; // successfull
};


/* 
 * protected slot
 */
void LogDialogImpl::annotateClicked()
{
    AnnotateDialogImpl *pAnnotateDialogImpl = new AnnotateDialogImpl ();

    if (pAnnotateDialogImpl->parseCvsAnnotate (basedir, fname, selectionA))
        pAnnotateDialogImpl->show();
    else
        delete pAnnotateDialogImpl;
}


void LogDialogImpl::displayFileRevision (QString rev)
{
    AnnotateDialogImpl *pAnnotateDialogImpl = new AnnotateDialogImpl ();

    if (pAnnotateDialogImpl->parseCvsRevision (basedir, fname, selectionA))
        pAnnotateDialogImpl->show();
    else
        delete pAnnotateDialogImpl;
}


/* 
 * protected slot
 */
void LogDialogImpl::diffClicked()
{
  QString RealA, RealB;
 
  RealA =selectionA;
  RealB =selectionB;
 
  if( selectionA.isEmpty() ) {
    if( ! selectionB.isEmpty() ) {
      /* must swap -> can only compare currentdir version
       * with repository.
       */
      RealA = selectionB;
      RealB = selectionA;
    }
  }


  if (bUseExternalDiffForSideBySide) {
    QString tmpDiffFileNameA = createTempFile() + "-" + fname + "-" + RealA;
    QString tmpDiffFileNameB = createTempFile() + "-" + fname + "-" + RealB;

    bool usingRsh = false;
    bool usingSsh = false;
    
    QString command = "";

    QString tmpRevision = RealA;
    QString tmpDiffFileName = tmpDiffFileNameA;
    int i;
    for (i = 0; i < 2; i++) {
      if( (QString::compare(m_connectMethod, "ext") == 0)
	  || (QString::compare(m_connectMethod, "") == 0) ) {//rsh access
	usingRsh = true;
		    
	command = "cd " + basedir + " && ";
						 
	if (cvsRsh.find("ssh") == 0) {
	  if (!sshAgentIsRunning) startSshAgent();
	  usingSsh = true;

	  command += "set-ssh-commit-env.sh \"";
	  command += "env CVS_RSH=" + cvsRsh + " cvs -Q update -p -r " + tmpRevision;
	  command += " " + fname + " > " + tmpDiffFileName + "\"";
	} else {//rsh
	  command += "env CVS_RSH=" + cvsRsh + " cvs -Q update -p -r " + tmpRevision;
	  command += " " + fname + " > " + tmpDiffFileName + " 2> /dev/null";
	}
      }
								    
      if(!usingRsh) {
	command = "cd " + basedir + " && cvs -Q update -p -r " + tmpRevision + " ";
	command += fname + " > " + tmpDiffFileName;
	command += " 2> /dev/null";
      }
    
      CmdThread * Thread = new CmdThread( command );
      Thread->start();
      Thread->wait();

      tmpRevision = RealB;
      tmpDiffFileName = tmpDiffFileNameB;
    }

    if (fork() == 0) {
      execlp (externalDiffProgram, externalDiffProgram, 
	      tmpDiffFileNameA.latin1(), 
	      tmpDiffFileNameB.latin1(), 
	      0);
    }
  } else {
    DiffDialogImpl *l = new DiffDialogImpl();
    if(l->parseCvsDiff(basedir, fname, m_connectMethod, RealA, RealB))
      l->show();
    else
      delete l;
  }
}


/* 
 * protected slot
 */
void LogDialogImpl::revisionSelected(QString rev, bool rmb)
{
    QListIterator<RevisionInfo> it(items);

    for (; it.current(); ++it)
	if (it.current()->rev == rev)
	    {
		if (rmb)
		{
		    RevBBox->setText (selectionB = rev);
		    AuthorBBox->setText(it.current()->author);
		    DateBBox->setText(it.current()->date);
		    CommentBBox->setText(it.current()->comment);
		}
		else
		{
		    RevABox->setText (selectionA = rev);
		    AuthorABox->setText(it.current()->author);
		    DateABox->setText(it.current()->date);
		    CommentABox->setText(it.current()->comment);
		}

		/* Enable or disable the view diffs button depending
		 * on whether there are two different revisions
		 * currently selected.
		 */
		buttonDiffs->setEnabled (bool (!selectionA.isNull() &&
					       !selectionB.isNull() &&
					       selectionA != selectionB));

		/* Enable or disable the annotation button depending
		 * on whether there is a revision currently selected.
		 */
		buttonAnnotate->setEnabled (bool (!selectionA.isNull()));

                tree->setSelectedPair(selectionA, selectionB);
		list->setSelectedPair(selectionA, selectionB);
		return;
	    }
}


/* 
 * protected slot
 */
void LogDialogImpl::tag1Selected(const char *tag)
{
    tagSelected(tag, false);
}


/* 
 * protected slot
 */
void LogDialogImpl::tag2Selected(const char *tag)
{
    tagSelected(tag, true);
}


/* 
 * protected slot
 */
void LogDialogImpl::tagSelected(QString tag, bool rmb)
{
    QListIterator<TagInfo> it(tags);

    for (; it.current(); ++it)
        if (tag == it.current()->tag)
            {
                if (it.current()->branchpoint.isEmpty())
                    revisionSelected(it.current()->rev, rmb);
                else
                    revisionSelected(it.current()->branchpoint, rmb);
                return;
            }
}


/* 
 * protected slot
 */
void LogDialogImpl::SelectSelectionAColor ()
{
	SelectSelectionColor(0);
	pColorBtnPalette->setColor(QColorGroup::Button, SELECTION_COLOR_OF_LOGDLG[0]);
	buttonAColor->setPalette(*pColorBtnPalette);
}


void LogDialogImpl::SelectSelectionBColor ()
{
	SelectSelectionColor(1);
	pColorBtnPalette->setColor(QColorGroup::Button, SELECTION_COLOR_OF_LOGDLG[1]);
	buttonBColor->setPalette(*pColorBtnPalette);
}


void LogDialogImpl::SelectSelectionColor (unsigned char selection)
{
	QColor color = QColorDialog::getColor (SELECTION_COLOR_OF_LOGDLG [selection]);

	if (color.isValid())
	{
		tree->setSelectionColor (selection, color);
		tree->repaint (true);
		list->setSelectionColor (selection, color);
		list->repaint (true);

		SELECTION_COLOR_OF_LOGDLG [selection] = color;
	}
}
