/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeograph.hpp"
#include "app_window.hpp"
#include "panel_diary.hpp"
#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;


PanelDiary::PanelDiary()
: m_entry_count( 0 )
{
    Gtk::Button *button_today;
    try
    {
        Lifeograph::builder->get_widget( "button_logout", m_button_logout );
        Lifeograph::builder->get_widget( "button_today", button_today );

        Gtk::MenuButton *mb_tools;
        Lifeograph::builder->get_widget( "menubutton_tools", mb_tools );
        Lifeograph::builder->get_widget( "menuitem_sort_by_date", m_menuitem_sort_by_date );
        Lifeograph::builder->get_widget( "menuitem_sort_by_size", m_menuitem_sort_by_size );
        Lifeograph::builder->get_widget( "menuitem_sort_by_change", m_menuitem_sort_by_change );

        Lifeograph::builder->get_widget_derived( "entry_search", m_entry_search );
        Lifeograph::builder->get_widget_derived( "entry_replace", m_entry_replace );
        Lifeograph::builder->get_widget( "button_match_prev", m_button_match_prev );
        Lifeograph::builder->get_widget( "button_match_next", m_button_match_next );
        Lifeograph::builder->get_widget( "grid_replace", m_grid_replace );
        Lifeograph::builder->get_widget( "button_replace", m_button_replace );
        Lifeograph::builder->get_widget( "button_replace_all", m_button_replace_all );

        // FALLBACK ICON FOR SYSTEMS WITHOUT SYMBOLIC ICONS
        Glib::RefPtr< Gtk::IconTheme > theme = Gtk::IconTheme::get_default();
        if( ! theme->has_icon( "preferences-system-symbolic" ) &&
            theme->has_icon( "gtk-preferences" ) )
        {
            Glib::RefPtr< Gdk::Pixbuf > icon_tools = theme->load_icon( "gtk-preferences", 16 );
            Gtk::Image *img_tools = Gtk::manage( new Gtk::Image( icon_tools ) );
            mb_tools->set_image( *img_tools );
        }

        Lifeograph::builder->get_widget( "calendar_main", m_calendar );
        Lifeograph::builder->get_widget_derived( "treeview_main", m_treeview_entries );

        Lifeograph::builder->get_widget( "menubutton_diary_add", m_button_add );
        Lifeograph::builder->get_widget( "mnitem_add_entry", m_menuitem_add_entry );
        Lifeograph::builder->get_widget( "mnitem_add_topic", m_menuitem_add_topic );
        Lifeograph::builder->get_widget( "mnitem_add_todo_group", m_menuitem_add_todo_group );
        Lifeograph::builder->get_widget( "mnitem_add_chapter", m_menuitem_add_chapter );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the diary panel" );
    }

    // SIGNALS
    m_button_logout->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( AppWindow::p, &AppWindow::logout ), true ) );

    m_menuitem_sort_by_date->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_DATE ) );
    m_menuitem_sort_by_size->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_SIZE ) );
    m_menuitem_sort_by_change->signal_activate().connect(
            sigc::bind(
                    sigc::mem_fun( m_treeview_entries,
                                   &WidgetEntryList::handle_sorting_criteria_changed ),
                    SC_CHANGE ) );

    m_calendar->signal_day_selected_double_click().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_entry ) );
    m_calendar->signal_month_changed().connect(
            sigc::mem_fun( this, &PanelDiary::handle_calendar_monthchanged ) );
    m_calendar->signal_day_selected().connect(
            sigc::mem_fun( this, &PanelDiary::handle_calendar_dayselected ) );

     m_treeview_entries->signal_row_expanded().connect(
            sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );
    m_treeview_entries->signal_row_collapsed().connect(
            sigc::mem_fun( this, &PanelDiary::handle_treeview_row_expanded ) );

    m_menuitem_add_entry->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_entry ) );
    m_menuitem_add_topic->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_topic ) );
    m_menuitem_add_todo_group->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_todo_group ) );
    m_menuitem_add_chapter->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::handle_add_chapter ) );

    m_entry_search->signal_changed().connect(
            sigc::mem_fun( this, &PanelDiary::handle_search_string_changed ) );
    m_entry_search->signal_activate().connect(
            sigc::mem_fun( AppWindow::p->panel_diary, &PanelDiary::go_next_match ) );

    m_entry_replace->signal_activate().connect(
            sigc::mem_fun( this, &PanelDiary::replace_match ) );
    m_button_replace->signal_clicked().connect(
            sigc::mem_fun( this, &PanelDiary::replace_match ) );
    m_button_replace_all->signal_clicked().connect(
            sigc::mem_fun( this, &PanelDiary::replace_all_matches ) );

    // ACTIONS
    Lifeograph::create_action(
            true, m_action_today, "Today", "x-office-calendar", _( "Today" ),
            _( "Go to today (creates a new entry if there isn't any or when already at today)" ),
            Gtk::AccelKey( GDK_KEY_d, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::show_today ),
            button_today );

    Lifeograph::create_action(
            false, m_action_previous_list, "ListPrevious", Gtk::Stock::GO_UP, _( "Previous" ),
            _( "Go to the previous entry in the list" ),
            Gtk::AccelKey( GDK_KEY_Page_Up, Gdk::CONTROL_MASK ),
            sigc::bind( sigc::mem_fun( m_treeview_entries, &WidgetEntryList::go_up ), false ) );

    Lifeograph::create_action(
            false, m_action_next_list, "ListNext", Gtk::Stock::GO_DOWN, _( "Next" ),
            _( "Go to the next entry in the list" ),
            Gtk::AccelKey( GDK_KEY_Page_Down, Gdk::CONTROL_MASK ),
            sigc::bind( sigc::mem_fun( m_treeview_entries, &WidgetEntryList::go_down ), false ) );

    Lifeograph::create_action(
            false, m_action_jump2current, "Jump", Gtk::Stock::JUMP_TO,
            "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_j, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::present_current_row ) );

    Lifeograph::create_action(
            false, m_action_show_prev_session_elem, "PrevSessElem", Gtk::Stock::JUMP_TO,
            "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_l, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &PanelDiary::show_prev_session_elem ) );

    m_button_logout->add_accelerator(
            "clicked",
            AppWindow::p->get_accel_group(),
            GDK_KEY_Escape,
            Gdk::CONTROL_MASK,
            Gtk::ACCEL_VISIBLE );

    // ACTIONS
    Lifeograph::create_action(
            false, m_action_focusfilter, "FocusFilter",
            Gtk::Stock::FIND, "", "",     // no grachical interface yet!
            Gtk::AccelKey( GDK_KEY_f, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_entry_search, &Gtk::Widget::grab_focus ) );

    Lifeograph::create_action(
            false, m_action_previous_match, "MatchPrevious", Gtk::Stock::GO_BACK,
// TRANSLATORS: label and tooltip of the go-to-previous-match-of-filtered-text action
            _( "Previous Match" ),
            _( "Go to previous match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::SHIFT_MASK ),
            sigc::mem_fun( this, &PanelDiary::go_prev_match ),
            m_button_match_prev );

    Lifeograph::create_action(
            false, m_action_next_match, "MatchNext", Gtk::Stock::GO_FORWARD,
// TRANSLATORS: label and tooltip of the go-to-next-match-of-filtered-text action
            _( "Next Match" ),
            _( "Go to next match" ),
            Gtk::AccelKey( GDK_KEY_F3, Gdk::ModifierType( 0 ) ),
            sigc::mem_fun( this, &PanelDiary::go_next_match ),
            m_button_match_next );

    // becoming proxy widgets makes them visible:
    m_button_match_prev->set_visible( false );
    m_button_match_next->set_visible( false );
}

void
PanelDiary::add_chapter_category_to_list( const CategoryChapters* chapters )
{
    Gtk::TreeRow	row_container;
    Chapter			*chapter = NULL;

    for( CategoryChapters::const_iterator iter_chapter = chapters->begin();
         iter_chapter != chapters->end();
         ++iter_chapter )
    {
        chapter = iter_chapter->second;
        if( ! chapter->is_initialized() )
            continue;

        // show all folders even when they are empty:
        row_container = * m_treeview_entries->m_treestore->append( l_row_diary.children() );
        row_container[ ListData::colrec->ptr ] = chapter;	// ptr must be first:
        row_container[ ListData::colrec->info ] = chapter->get_list_str();
        row_container[ ListData::colrec->icon ] = chapter->get_icon();

        chapter->reset_size();

        if( l_entry == NULL )
            continue;

        while( l_entry->get_date() >= chapter->get_date() )
        {
            chapter->increase_size();
            if( l_entry->get_filtered_out() == false )
            {
                m_entry_count++;
                l_row_entry = * m_treeview_entries->m_treestore->append(
                        row_container.children() );
                l_row_entry[ ListData::colrec->ptr ] = l_entry;
                l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();
                l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
            }

            ++l_itr_entry;
            if( l_itr_entry != Diary::d->m_entries.end() )
                l_entry = l_itr_entry->second;
            else
            {
                l_entry = NULL;
                break;
            }
        }
    }
}

void
PanelDiary::fill_child_rows( const Gtk::TreeModel::Children& children )
{
    for( Gtk::TreeModel::Children::const_iterator iter = children.begin();
            iter != children.end(); ++iter )
    {
        Gtk::TreeModel::Row row = *iter;
        DiaryElement *ptr = row[ ListData::colrec->ptr ];

        if( ptr )
            ptr->m_list_data->treepath = m_treeview_entries->m_treestore->get_path( row );

        switch( ptr->get_type() )
        {
            case DiaryElement::ET_CHAPTER:
            case DiaryElement::ET_TOPIC:
            case DiaryElement::ET_TODO_GRP:
                if( dynamic_cast< Chapter* >( ptr )->get_expanded() )
                    m_treeview_entries->expand_to_path( ptr->m_list_data->treepath );
                break;
            default:
                break;
        }

        fill_child_rows( row.children() );
    }
}

void
PanelDiary::update_entry_list()
{
    PRINT_DEBUG( "update_entry_list()" );
    Lifeograph::m_internaloperation++;

    AppWindow::p->m_entry_view->sync();

    l_itr_entry = Diary::d->m_entries.begin();
    l_entry = ( l_itr_entry != Diary::d->m_entries.end() ? l_itr_entry->second : NULL );

    m_treeview_entries->m_treestore->clear();
    m_entry_count = 0;

    // REMOVE MODEL AND DISBLE SORTING FOR SPEED-UP
    m_treeview_entries->unset_model();
    m_treeview_entries->m_treestore->set_sort_column(
            Gtk::TreeSortable::DEFAULT_UNSORTED_COLUMN_ID, Gtk::SORT_ASCENDING );

    // DIARY ITEM
    l_row_diary = * m_treeview_entries->m_treestore->append();
    l_row_diary[ ListData::colrec->ptr ] = Diary::d; // ptr must be first
    l_row_diary[ ListData::colrec->info ] = Diary::d->get_list_str();
    l_row_diary[ ListData::colrec->icon ] = Lifeograph::icons->diary_16;

    // CHAPTERS
    if( Diary::d->get_sorting_criteria() == SC_DATE /* TODO: maybe later: &&
        !( Diary::d->get_filtering_status() & Filter::FS_FILTER_TRASH )*/ )
    {
        add_chapter_category_to_list( Diary::d->get_todo_groups() );
        add_chapter_category_to_list( Diary::d->get_topics() );
        add_chapter_category_to_list( Diary::d->get_current_chapter_ctg() );
    }

    // ENTRIES NOT PART OF A CHAPTER
    for( ; l_itr_entry != Diary::d->m_entries.end(); ++l_itr_entry )
    {
        l_entry = ( *l_itr_entry ).second;
        if( l_entry->get_filtered_out() == false )
        {
            m_entry_count++;

            l_row_entry = *m_treeview_entries->m_treestore->append(
                    l_row_diary.children() );
            l_row_entry[ ListData::colrec->ptr ] = l_entry; // ptr must be first
            l_row_entry[ ListData::colrec->icon ] = l_entry->get_icon();
            l_row_entry[ ListData::colrec->info ] = l_entry->get_list_str();
        }
    }

    // RESTORE MODEL AND SORTING (FOR SPEED-UP)
    m_treeview_entries->m_treestore->set_sort_column(
            Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING );
    m_treeview_entries->set_model( m_treeview_entries->m_treestore );

    // set treepath values in of listed elements
    fill_child_rows( m_treeview_entries->m_treestore->children() );

    // always expand diary
    m_treeview_entries->expand_element( Diary::d );

    // UPDATE CURRENT ELEM IF LAST ONE IS NOT IN THE LIST ANYMORE
    DiaryElement* elem = AppWindow::p->panel_main->get_cur_elem();
    if( elem != NULL )
    {
        if( elem->get_filtered_out() )
        {
            elem = Diary::d->get_entry_first();
            if( elem == NULL ) // if there is no entry
                elem = Diary::d;
            elem->show();
        }
        else if( elem->get_type() >= DiaryElement::ET_DIARY )
            m_treeview_entries->present_element( elem );
    }
    else // the case at the start-up
    {
        elem = Diary::d->get_startup_elem();
        if( elem->get_filtered_out() ) // if startup item is trashed
            elem = Diary::d->get_entry_first();
        if( elem == NULL ) // if there is no unfiltered entry
            elem = Diary::d;
        elem->show();
    }

    Diary::d->get_filter()->set_status_applied();

    Lifeograph::m_internaloperation--;
}

void
PanelDiary::update_entry_list_style() // TODO: is this function really needed?
{
    Lifeograph::m_internaloperation++;

    update_entry_list();

    if( AppWindow::p->panel_main->get_cur_elem_type() != DiaryElement::ET_ENTRY )
        Diary::d->show();

    m_treeview_entries->columns_autosize();

    Lifeograph::m_internaloperation--;
}

void
PanelDiary::update_calendar()
{
    guint year, month, day;

    m_calendar->get_date( year, month, day );
    month++;	// fix month
    Entry *entry;
    m_calendar->clear_marks();

    for( EntryIterConst itr_entry = Diary::d->get_entries().begin();
         itr_entry != Diary::d->get_entries().end();
         ++itr_entry )
    {
        entry = itr_entry->second;
        if( entry->get_filtered_out() || entry->get_date().is_ordinal() )
            continue;

        if( entry->get_date().get_year() == year )
        {
            if( entry->get_date().get_month() == month )
            {
                m_calendar->mark_day( entry->get_date().get_day() );
            }
            else
            if( entry->get_date().get_month() < month )
                break;
        }
        else
        if( entry->get_date().get_year() < year )
            break;
    }
}

void
PanelDiary::handle_login()
{
    switch( Diary::d->get_sorting_criteria() )
    {
        default: // future-proofing
        case SC_DATE: // no break
            m_menuitem_sort_by_date->set_active();
            break;
        case SC_SIZE:
            m_menuitem_sort_by_size->set_active();
            break;
        case SC_CHANGE:
            m_menuitem_sort_by_change->set_active();
            break;
    }

    // not called automatically because of m_internaloperation:
    m_treeview_entries->reset_sorting_criteria();

    if( Diary::d->is_read_only() )
    {
        m_treeview_entries->unset_rows_drag_source();
        m_treeview_entries->unset_rows_drag_dest();
    }
    else
    {
        m_treeview_entries->enable_model_drag_source( Lifeograph::p->drag_targets_entry );
        m_treeview_entries->enable_model_drag_dest( Lifeograph::p->drag_targets_entry,
                                                    Gdk::ACTION_MOVE );
    }

    update_entry_list();

    m_button_add->set_visible( ! Diary::d->is_read_only() );
}

void
PanelDiary::handle_logout()
{
    Lifeograph::m_internaloperation++;

    m_treeview_entries->clear();

    m_entry_search->set_text( "" );
    m_entry_replace->set_text( "" );

    m_button_match_prev->hide();
    m_button_match_next->hide();
    m_grid_replace->hide();

    Lifeograph::m_internaloperation--;
}

void
PanelDiary::show( const DiaryElement* element )
{
    if( element->get_type() >= DiaryElement::ET_DIARY ) // if a list element
        m_treeview_entries->present_element( element );
    else
        m_treeview_entries->get_selection()->unselect_all();

    if( element->get_type() > DiaryElement::ET_DIARY )
    {
        select_date_in_calendar( element->get_date() );
        m_menuitem_add_entry->set_sensitive( true );

        if( element->get_date().is_ordinal() )
        {
            Chapter *c = element->get_date().is_todo() ?
                    Diary::d->get_todo_groups()->get_chapter( element->get_date().get_pure() ) :
                    Diary::d->get_topics()->get_chapter( element->get_date().get_pure() );
            if( c == NULL )
                m_menuitem_add_entry->set_sensitive( false );
            else
                m_menuitem_add_entry->set_tooltip_text(
                        Glib::ustring::compose( _( "Create a new entry in %1" ), c->get_name() ) );
            m_menuitem_add_chapter->set_sensitive( false );
            m_menuitem_add_chapter->set_tooltip_text( _( "Please select a date" ) );
        }
        else
        {
            m_menuitem_add_entry->set_tooltip_text(
                    Glib::ustring::compose( _( "Create a new entry on %1" ),
                                            element->get_date().format_string() ) );
            m_menuitem_add_chapter->set_tooltip_text(
                    Glib::ustring::compose( _( "Create a new chapter on %1" ),
                                            element->get_date().format_string() ) );
            m_menuitem_add_chapter->set_sensitive(
                    Diary::d->get_current_chapter_ctg()->get_chapter( element->get_date().m_date )
                    == NULL );
        }
    }
    else
    {
        m_calendar->select_day( 0 );
        m_menuitem_add_entry->set_sensitive( false );
        m_menuitem_add_chapter->set_sensitive( false );

        m_menuitem_add_entry->set_tooltip_text( _( "Please select a date or topic" ) );
        m_menuitem_add_chapter->set_tooltip_text( _( "Please select a date" ) );
    }
}

void
PanelDiary::show_today()
{
    Entry *entry_today = Diary::d->get_entry_today();
    bool flag_add = false;

    if( entry_today == NULL )   // add new entry if no entry exists on selected date
    {
        if( Diary::d->is_read_only() )
            return;
        else
            flag_add = true;
    }
    else                        // or current entry is already at today
    if( entry_today->get_date().get_pure() ==
            AppWindow::p->panel_main->get_cur_elem()->get_date().get_pure() )
        flag_add = true;

    if( flag_add )
    {
        entry_today = Diary::d->add_today();
        update_entry_list();
    }

    entry_today->show();
}

void
PanelDiary::show_prev_session_elem()
{
    DiaryElement *elem( Diary::d->get_prev_session_elem() );
    if( elem != NULL )
        elem->show();
    else
        print_info( "Previous sesion element cannot be found!" );
}

void
PanelDiary::present_current_row()
{
    Gtk::TreeIter iter = m_treeview_entries->get_selection()->get_selected();
    if( iter == m_treeview_entries->m_treestore->children().end() )
        return;
    Gtk::TreePath path = m_treeview_entries->m_treestore->get_path( iter );

    m_treeview_entries->scroll_to_row( path );
}

void
PanelDiary::handle_calendar_dayselected()
{
    if( Lifeograph::m_internaloperation ) return;

    guint year, month, day;
    m_calendar->get_date( year, month, day );
    Date date( year, month + 1, day );

    m_menuitem_add_entry->set_sensitive( day > 0 );
    m_menuitem_add_chapter->set_sensitive( day > 0 &&
            Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) == NULL );

    Entry *entry;
    if( date.m_date == ( AppWindow::p->panel_main->get_cur_elem()->get_date().get_pure() ) )
    {
        entry = Diary::d->get_entry_next_in_day(
                AppWindow::p->panel_main->get_cur_elem()->get_date() );
    }
    else
    {
        entry = Diary::d->get_entry( date.m_date + 1 ); // 1 increment to fix order
    }

    if( entry )
        entry->show();
    else
    {
        m_menuitem_add_entry->set_tooltip_text(
                Glib::ustring::compose( _( "Create a new entry on %1" ), date.format_string() ) );
        m_menuitem_add_chapter->set_tooltip_text(
                Glib::ustring::compose( _( "Create a new chapter on %1" ),
                                        date.format_string() ) );
    }
}

void
PanelDiary::handle_calendar_monthchanged()
{
    if( Lifeograph::m_internaloperation ) return;

    update_calendar();
}

void
PanelDiary::handle_add_entry()
{
    if( Diary::d->is_read_only() )
        return;

    Date date( 0 );
    guint year, month, day;
    m_calendar->get_date( year, month, day );
    if( !day ) // no day is selected in the calendar
    {
        date = AppWindow::p->panel_main->get_cur_elem()->get_date();
        if( !date.is_ordinal() )
            return;
    }
    else
        date.m_date = Date::make_date( year, month + 1, day );

    Entry *entry( Diary::d->create_entry( date.m_date ) );

    update_entry_list();
    update_calendar();

    entry->show();

    // this is the only way i could find to give focus to textview
    //FIXME
//  Glib::signal_timeout().connect_once(
//          sigc::mem_fun( m_textviewdiary, &Gtk::Widget::grab_focus ), 200 );
}

void
PanelDiary::handle_add_todo_group()
{
    Chapter *c( Diary::d->add_todo_group( _( "New todo group" ) ) );
    update_entry_list();
    c->show();
    AppWindow::p->panel_main->start_title_edit();
}

void
PanelDiary::handle_add_topic()
{
    Chapter *c( Diary::d->add_topic( _( "New topic" ) ) );
    update_entry_list();
    c->show();
    AppWindow::p->panel_main->start_title_edit();
}

void
PanelDiary::handle_add_chapter()
{
    guint year, month, day;
    m_calendar->get_date( year, month, day );
    Date date( year, month + 1, day );

    if( day > 0 && Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) == NULL )
    {
        Chapter *c( Diary::d->get_current_chapter_ctg()->create_chapter(
                _( "Untitled chapter" ), date ) );

        update_entry_list();
        c->show();
        AppWindow::p->panel_main->start_title_edit();
    }
}

void
PanelDiary::select_date_in_calendar( const Date& date )
{
    ++Lifeograph::m_internaloperation;

    // CALENDAR
    // the current day has to be deselected first to avoid gtk's complaints
    m_calendar->select_day( 0 );
    if( ! date.is_ordinal() )
    {
        guint year, month, day;
        m_calendar->get_date( year, month, day );

        if( year != date.get_year() || month != date.get_month() + 1 )
        {
            m_calendar->select_month( date.get_month() - 1, date.get_year() );
            update_calendar();
        }
        //if( date.get_day() ) should not be necessary
        m_calendar->select_day( date.get_day() );
    }

    --Lifeograph::m_internaloperation;
}

void
PanelDiary::go_prev_match()
{
    if( m_entry_count < 1 )
        return;

    // TODO: move to Entry
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
        if( AppWindow::p->m_entry_view->get_buffer()->select_searchstr_previous() )
            return;

    int i = 0;
    do
    {
        m_treeview_entries->go_up( true );
        // put cursor to the end:
        AppWindow::p->m_entry_view->get_buffer()->select_range(
                AppWindow::p->m_entry_view->get_buffer()->end(),
                AppWindow::p->m_entry_view->get_buffer()->end() );
        i++;
    }
    while( ! AppWindow::p->m_entry_view->get_buffer()->select_searchstr_previous() &&
           i < m_entry_count );
}

void
PanelDiary::go_next_match()
{
    if( m_entry_count < 1 )
        return;

    // TODO: move to Entry
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
        if( AppWindow::p->m_entry_view->get_buffer()->select_searchstr_next() )
            return;

    int i = 0;
    do
    {
        m_treeview_entries->go_down( true );
        i++;
    }
    while( ! AppWindow::p->m_entry_view->get_buffer()->select_searchstr_next() && i < m_entry_count );
}

void
PanelDiary::handle_entry_title_changed( DiaryElement* element )
{
    if( Lifeograph::m_internaloperation ) return;

    Gtk::TreeRow row = * get_row( element->m_list_data->treepath );
    row[ ListData::colrec->info ] = element->get_list_str();
}

Gtk::TreeRow
PanelDiary::get_row( const Gtk::TreePath& path )
{
    return( * m_treeview_entries->m_treestore->get_iter( path ) );
    // TODO: check validity
}

void
PanelDiary::handle_treeview_row_expanded( const Gtk::TreeIter& iter,
                                          const Gtk::TreePath& path )
{
    //if( Lifeograph::m_internaloperation )
        //return;

    DiaryElement* element = ( * iter )[ ListData::colrec->ptr ];
    switch( element->get_type() )
    {
        case DiaryElement::ET_CHAPTER:
        case DiaryElement::ET_TOPIC:
        case DiaryElement::ET_TODO_GRP:
        {
            Chapter* chapter = dynamic_cast< Chapter* >( element );
            chapter->set_expanded( m_treeview_entries->row_expanded( path ) );
        }
        break;
    default:
        break;
    }
}

// SEARCH & REPLACE
void
PanelDiary::handle_search_string_changed()
{
    if( Lifeograph::m_internaloperation ) return;
    // current entry must be closed here or else it loses...
    // ...changes since it was selected:
    //AppWindow::p->m_entry_view->sync(); update_entry_list() does the syncing

    const std::string filterstring( m_entry_search->get_text().lowercase() );

    Diary::d->set_search_text( filterstring );
    AppWindow::p->panel_diary->update_entry_list();    // syncs current entry

    if( filterstring.size() > 0 )
    {
        if( AppWindow::p->panel_diary->get_entry_count() > 0 )
        {
            AppWindow::p->m_entry_view->get_buffer()->set_search_str( filterstring );
            if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
                AppWindow::p->m_entry_view->get_buffer()->select_searchstr_next();
        }

        m_entry_search->set_tooltip_markup(
                Glib::ustring::compose(
                        _( "Found in <b>%1</b> of <b>%2</b> entrie(s)" ),
                        AppWindow::p->panel_diary->get_entry_count(), Diary::d->get_size() ) );
    }
    else
    {
        AppWindow::p->m_entry_view->get_buffer()->set_search_str( "" );
        m_entry_search->set_tooltip_text( _( "Enter text to be filtered" ) );
    }

    AppWindow::p->panel_diary->update_calendar();

    if( AppWindow::p->panel_diary->get_entry_count() < 1 || filterstring.size() < 1 )
    {
        m_button_match_prev->hide();
        m_button_match_next->hide();
        m_grid_replace->hide();
    }
    else
    {
        m_button_match_prev->show();
        m_button_match_next->show();
        if( ! Diary::d->is_read_only() )
            m_grid_replace->show();
    }
}

void
PanelDiary::replace_match()
{
    if( ! AppWindow::p->m_entry_view->get_buffer()->get_has_selection() )
        AppWindow::p->panel_diary->go_next_match();
    if( AppWindow::p->m_entry_view->get_buffer()->get_has_selection() )
    {
        AppWindow::p->m_entry_view->get_buffer()->erase_selection();
        AppWindow::p->m_entry_view->get_buffer()->insert_at_cursor( m_entry_replace->get_text() );
        AppWindow::p->panel_diary->go_next_match();
    }
}

void
PanelDiary::replace_all_matches()
{
    AppWindow::p->m_entry_view->sync();

    Diary::d->replace_text( m_entry_replace->get_text() );
    // update current entry:
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
    {
        Entry *entry = dynamic_cast< Entry* >( AppWindow::p->panel_main->get_cur_elem() );
        AppWindow::p->m_entry_view->get_buffer()->set_richtext( entry );
    }
}

