/***************************************************************************
                          window.c  -  description
                             -------------------
    begin                : Sat Jun 23 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "sdl.h"
#include "dynlist.h"
#include "tools.h"
#include "config.h"
#include "date.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "theme.h"
#include "map.h"
#include "scenario.h"
#include "gui.h"
#include "ai_action.h"
#include "engine_tools.h"
#include "deploy.h"
#include "slot.h"
#include "engine.h"
#include "window.h"

extern Sdl sdl;
extern Config config;
extern Scen scen;
extern Map map;
extern int mode_count; /* number of video modes defined in sdl.c */

/*
====================================================================
Forwarded
====================================================================
*/
void show_unit_info( Engine *engine, Unit *unit );
void update_scen_info_window( Engine *engine );

/* create windows by using current theme */
/* NOTE: to change theme you must delete/recreate engine!!!! */
void create_windows( Engine *engine )
{
    int gfx_offset;
    int game_offset;
    int audio_offset;
    Window *label = 0;
    int i;

    create_upper_bar( engine );
    create_lower_bar( engine );

    /* confirm window */
    engine->conf_window = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 240, 80 );
    create_text_button( engine->conf_window, ID_ACCEPT, engine->theme->menu_text_button,
                        engine->conf_window->width / 2 - 80, engine->conf_window->height - 30, 80, 20,
                        engine->theme->stan_font, "Ok" );
    create_text_button( engine->conf_window, ID_CANCEL, engine->theme->menu_text_button,
                        engine->conf_window->width / 2, engine->conf_window->height - 30, 80, 20,
                        engine->theme->stan_font, "Cancel" );
    create_label( engine->conf_window, ID_UPPER_LABEL, 10, 10, engine->conf_window->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, ""  );
    create_label( engine->conf_window, ID_LOWER_LABEL, 10, 26, engine->conf_window->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, ""  );

    /* briefing window */
    engine->brief_window = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 330, 340 );
    create_list( engine->brief_window, 101, engine->theme->list_frame, 20, 20,
                 engine->brief_window->width - 40, engine->brief_window->height - 80, 6,
                 engine->theme->stan_font, 0, NO_SELECT , 0x0000ff,
                 0, engine->theme->down_arrow, engine->theme->up_arrow, engine->theme->track_ball,
                 20 );
    create_text_button( engine->brief_window, ID_ACCEPT, engine->theme->menu_text_button,
                        engine->brief_window->width / 2 - 40, engine->brief_window->height - 40, 80, 20,
                        engine->theme->stan_font, "Ok" );

    /* edit window */
    engine->edit_window = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 240, 100 );
    create_label( engine->edit_window, ID_LABEL, 10, 10, engine->edit_window->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, ""  );
    create_edit( engine->edit_window, ID_EDIT,
                 engine->theme->edit_back,
                 10, 36, engine->edit_window->width - 20, 20,
                 5, engine->theme->stan_font, "", 10, /*TRANS_EDIT*/0 );
    create_text_button( engine->edit_window, ID_ACCEPT, engine->theme->menu_text_button,
                        engine->edit_window->width / 2 - 80, engine->edit_window->height - 30, 80, 20,
                        engine->theme->stan_font, "Ok" );
    create_text_button( engine->edit_window, ID_CANCEL, engine->theme->menu_text_button,
                        engine->edit_window->width / 2, engine->edit_window->height - 30, 80, 20,
                        engine->theme->stan_font, "Cancel" );

    /* game menu */
    engine->menu = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 200, 180 );
    create_text_button( engine->menu, ID_LOAD_MENU, engine->theme->menu_text_button,
                        20, 20, 160, 20,
                        engine->theme->stan_font, "Load Game" );
    create_text_button( engine->menu, ID_SAVE_MENU, engine->theme->menu_text_button,
                        20, 40, 160, 20,
                        engine->theme->stan_font, "Save Game" );
    create_text_button( engine->menu, ID_OPTIONS, engine->theme->menu_text_button,
                        20, 60, 160, 20,
                        engine->theme->stan_font, "Options" );
    create_text_button( engine->menu, ID_RESTART, engine->theme->menu_text_button,
                        20, 80, 160, 20,
                        engine->theme->stan_font, "Restart Scenario" );
    create_text_button( engine->menu, ID_CANCEL, engine->theme->menu_text_button,
                        20, 100, 160, 20,
                        engine->theme->stan_font, "Quit Scenario" );
    create_text_button( engine->menu, ID_ACCEPT, engine->theme->menu_text_button,
                        20, 140, 160, 20,
                        engine->theme->stan_font, "Resume Game" );

    /* save(load) menu */
    engine->save_menu = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 300, 290 );
    create_label( engine->save_menu, ID_LABEL, 10, 14, engine->save_menu->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, ""  );
    create_text_button( engine->save_menu, ID_CANCEL, engine->theme->menu_text_button,
                        engine->save_menu->width / 2 - 40, engine->save_menu->height - 35, 80, 20,
                        engine->theme->stan_font, "Cancel" );
    /* ten slots */
    for ( i = 0; i < SLOT_COUNT; i++ ) {
        create_text_button( engine->save_menu, ID_SLOT1 + i, engine->theme->menu_text_button,
                            10, 40 + i *20,
                            engine->save_menu->width - 20, 20,
                            engine->theme->stan_font, "" );
        /* usually text is centered in a button but not in these */
        get_child( engine->save_menu, ID_SLOT1 + i )->align = ALIGN_X_LEFT | ALIGN_Y_CENTER;
    }

    /* options */
    engine->options = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 240, 200 );
    create_text_button( engine->options, ID_ACCEPT, engine->theme->menu_text_button,
                        engine->options->width / 2 - 80, engine->options->height - 35, 80, 20,
                        engine->theme->stan_font, "Ok" );
    create_text_button( engine->options, ID_CANCEL, engine->theme->menu_text_button,
                        engine->options->width / 2, engine->options->height - 35, 80, 20,
                        engine->theme->stan_font, "Cancel" );
    label = create_label( engine->options, ID_LABEL,
                          10, engine->options->height - 90, engine->options->width - 20, 14,
                          engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, "Test"  );
    create_text_button( engine->options, ID_MODE_BUTTON, engine->theme->menu_text_button,
                        20, engine->options->height - 65, engine->options->width - 40, 20,
                        engine->theme->stan_font, cur_video_mode()->name );
    /* config buttons */
    /* game buttons */
    game_offset = 30;
    create_label( engine->options, ID_NONE,
                  10, game_offset - 20, engine->options->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, "Game"  );
    create_switch( engine->options, ID_SUPPLY_BUTTON,
                   engine->theme->supply_button,
                   10, game_offset, 24, 24,
                   "Units Must Supply", label,
                   config.supply );
    create_switch( engine->options, ID_WEATHER_BUTTON,
                   engine->theme->weather_button,
                   10 + 30, game_offset, 24, 24,
                   "Weather Influence", label,
                   config.supply );
    create_switch( engine->options, ID_CPU_BUTTON,
                   engine->theme->cpu_button,
                   10 + 60, game_offset, 24, 24,
                   "Show CPU Turn", label,
                   config.supply );
    /* gfx buttons: added behind game buttons: looks better :-) */
/*    create_label( engine->options, ID_NONE,
                  10, 60, engine->options->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, "Graphics"  );*/
    gfx_offset = 30;
    create_switch( engine->options, ID_GRID_BUTTON,
                   engine->theme->grid_button,
                   10 + 90, gfx_offset, 24, 24,
                   "Grid", label,
                   config.grid );
    create_switch( engine->options, ID_LIFE_BAR_BUTTON,
                   engine->theme->life_bar_button,
                   10 + 120, gfx_offset, 24, 24,
                   "Show Unit Life Bar", label,
                   config.show_bar );
    /* audio stuff */
    audio_offset = 80;
    create_label( engine->options, ID_NONE,
                  10, audio_offset - 20, engine->options->width - 20, 14,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, "Audio"  );
#ifdef WITH_SOUND
    create_switch( engine->options, ID_SOUND_BUTTON,
                   engine->theme->sound_button,
                   10 , audio_offset, 24, 24,
                   "Sound", label,
                   config.supply );
#else
    /* deactivated button */
    create_button( engine->options, ID_SOUND_BUTTON,
                   engine->theme->sound_button,
                   10 , audio_offset, 24, 24,
                   "Sound", label )->active = 0;
#endif
    create_scroll( engine->options, ID_SOUND_VOLUME_SCROLL,
                   engine->theme->hori_ctrl,
                   10 + 30, audio_offset + 2, 70, 20,
                   HORI,
                   0, 0, engine->theme->hori_ctrl_button,
                   0, 127, 16, config.sound_volume );
#ifdef WITH_SOUND
    create_button( engine->options, ID_MUSIC_BUTTON,
                   engine->theme->music_button,
                   10 + 110, audio_offset, 24, 24,
                   "Select Music Track", label );
#else
    /* deactivated button */
    create_button( engine->options, ID_MUSIC_BUTTON,
                   engine->theme->music_button,
                   10 + 110, audio_offset, 24, 24,
                   "Select Music Track", label )->active = 0;
#endif
    create_scroll( engine->options, ID_MUSIC_VOLUME_SCROLL,
                   engine->theme->hori_ctrl,
                   10 + 140, audio_offset + 2, 70, 20,
                   HORI,
                   0, 0, engine->theme->hori_ctrl_button,
                   0, 127, 16, config.sound_volume );

    /* scen info window */
    engine->scen_info_window = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 400, 300 );
    create_text_button( engine->scen_info_window, ID_ACCEPT, engine->theme->menu_text_button,
                        engine->scen_info_window->width / 2 - 40, engine->scen_info_window->height - 40, 80, 20,
                        engine->theme->stan_font, "Ok" );
    /* title */
    create_label( engine->scen_info_window, ID_TITLE_LABEL,
                  20, 20, engine->scen_info_window->width - 40, 20,
                  engine->theme->stan_font, ALIGN_X_CENTER | ALIGN_Y_CENTER, ""  );
    create_list( engine->scen_info_window, ID_INFO_LIST, engine->theme->list_frame, 20, 40,
                 engine->scen_info_window->width - 40, engine->scen_info_window->height - 100, 6,
                 engine->theme->stan_font, 0, NO_SELECT , 0x0,
                 0, engine->theme->down_arrow, engine->theme->up_arrow, engine->theme->track_ball,
                 20 );

    /* video mode window */
    engine->mode_window = create_mode_window( engine->theme );

    /* deploy */
    engine->deploy_window = create_deploy_window( engine );
}

/* called in delete_engine to delete all created windows */
void delete_windows( Engine *engine )
{
    if ( engine->upper_bar ) delete_window( engine->upper_bar );
    if ( engine->lower_bar ) delete_window( engine->lower_bar );
    if ( engine->conf_window ) delete_window( engine->conf_window );
    if ( engine->brief_window ) delete_window( engine->brief_window );
    if ( engine->edit_window ) delete_window( engine->edit_window );
    if ( engine->menu ) delete_window( engine->menu );
    if ( engine->save_menu ) delete_window( engine->save_menu );
    if ( engine->options ) delete_window( engine->options );
    if ( engine->scen_info_window ) delete_window( engine->scen_info_window );
    if ( engine->mode_window ) delete_window( engine->mode_window );
    if ( engine->deploy_window ) delete_window( engine->deploy_window );
}

/* check window_id and run the appropiate window */
int run_window( Engine *engine )
{
    int result = ID_NONE;
    char *str = 0;
    SDL_Cursor *old_cursor;
    int type;
    Unit *unit;

    switch ( engine->window_id ) {

        case ID_DEPLOY_UNITS:
            prep_deploy_window( engine );
            run_deploy_window( engine, &type, &unit );
            switch ( type ) {
                case CANCEL_DEPLOY:
                    cancel_deploy( engine );
                    engine->action = END_DEPLOY_UNITS;
                    break;
                case APPLY_DEPLOY:
                    apply_deploy( engine );
                    engine->action = END_DEPLOY_UNITS;
                    break;
                case DEPLOY_UNIT:
                    engine->sel_unit = unit;
                    set_deploy_mask( unit );
                    update_full_map( engine, CLIP );
                    break;
            }
            break;

        case ID_SCEN_INFO:
            update_scen_info_window( engine );
            run_stan_window( engine->scen_info_window );
            break;

        case ID_CONF_END_TURN:
            set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ), "Do you really want",
                         NO_REFRESH );
            set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ), "to end your turn?",
                         NO_REFRESH );
            if ( run_stan_window( engine->conf_window ) == ID_ACCEPT )
                result = END_TURN;
            break;

        case ID_UNIT_INFO:
            show_unit_info( engine, engine->sel_info_unit );
            break;

        case ID_RENAME_UNIT:
            update_edit_window( engine, "Enter new unit name:", engine->sel_unit->name, 20 );
            if ( run_stan_window( engine->edit_window ) == ID_ACCEPT ) {

                /* use entered name if valid */
                str = get_child( engine->edit_window, ID_EDIT )->text->lines[0];
                if ( str[0] != 0 )
                    strcpy( engine->sel_unit->name, str );

            }
            break;

        case ID_MENU:
            old_cursor = SDL_GetCursor();
            SDL_SetCursor( engine->stan_cursor );
            run_menu( engine );
            SDL_SetCursor( old_cursor );
            /* check if scen was loaded and run reinit_engine() */
            if ( engine->scen_loaded )
                reinit_engine( engine );
            else
                /* check if restart wanted */
                if ( engine->restart_scen )
                    restart_scen( engine );
                else
                    if ( engine->redraw_map ) {
                        engine->redraw_map = 0;
                        if ( engine->status != SHOWING_STRAT_MAP )
                            update_full_screen( engine );
                    }
            /* maybe resolution changed? */
            if ( engine->action == APPLY_VIDEO_MODE ) {
                engine->action = ID_NONE;
                result = APPLY_VIDEO_MODE;
            }
            break;

        default:
            fprintf( stderr, "run_window: id %i is unknown\n", engine->window_id );
            break;

    }

    engine->window_id = ID_NONE;
    return result;
}

/* run window and only check if ACCEPT or CANCEL is the result */
int run_stan_window( Window *window )
{
    SDL_Event event;
    Window *used_child = 0;
    int leave = 0;
    int result = ID_CANCEL;

    open_window( window, REFRESH );
    while ( !leave ) {

        if ( SDL_WaitEvent( &event ) ) {

            if ( event.type == SDL_KEYUP && event.key.keysym.sym == SDLK_ESCAPE ) leave = 1;

            handle_window_event( window, &event, &used_child );
            update_window( window, IF_NEEDED );
            if ( used_child )
                switch ( used_child->id ) {

                    case ID_ACCEPT: result = ID_ACCEPT; leave = 1; break;
                    case ID_CANCEL: result = ID_CANCEL; leave = 1; break;

                }

        }

    }
    close_window( window );

    return result;
}

/* run menu includes the handling of all subwindows, too
this means handling of options/saving/resuming games */
void run_menu( Engine *engine )
{
    int leave = 0;
    SDL_Event event;
    Window *child = 0;
    Video_Mode old_mode = config.video_mode;
    Video_Mode new_mode = old_mode;
    int i;
    int old_weather;

    open_window( engine->menu, REFRESH );
    while ( !leave ) {

        if ( SDL_WaitEvent( &event ) ) {

            if ( event.type == SDL_KEYUP && event.key.keysym.sym == SDLK_ESCAPE ) leave = 1;

            handle_window_event( engine->menu, &event, &child );
            update_window( engine->menu, IF_NEEDED );
            if ( child )
                switch ( child->id ) {

                    /* back to game */
                    case ID_ACCEPT: leave = 1; break;
                    /* leave scenario */
                    case ID_CANCEL:
                        set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ),
                                         "Do you really want",
                                         NO_REFRESH );
                        set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ),
                                         "to quit ?",
                                         NO_REFRESH );
                        if ( run_stan_window( engine->conf_window ) == ID_ACCEPT )
                            engine->leave_scen = leave = 1;
                        break;
                    /* run save/load window */
                    case ID_LOAD_MENU:
                        run_save_menu( engine, ID_LOAD_MENU );
                        if ( engine->scen_loaded ) leave = 1;
                        break;
                    case ID_SAVE_MENU:
                        /* run save */
                        run_save_menu( engine, ID_SAVE_MENU );
                        break;
                    case ID_RESTART:
                        set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ),
                                         "Do you really want",
                                         NO_REFRESH );
                        set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ),
                                         "to restart ?",
                                         NO_REFRESH );
                        if ( run_stan_window( engine->conf_window ) == ID_ACCEPT ) {
                            leave = 1;
                            engine->restart_scen = 1;
                        }
                        break;
                    case ID_OPTIONS:
                        old_weather = config.weather;
                        prep_options_window( engine );
                        if ( run_options_window( engine, &new_mode ) ) {
                            apply_options( engine );
                            engine->redraw_map = 1;
                            if ( old_weather != config.weather ) {
                                /* update spot mask and unit properties as these change if */
                                /* weather influence was turned on/off */
                                set_spot_mask( engine );
                                for ( i = 0; i < scen.units.count; i++ )
                                    update_cur_unit_prop( (Unit*)dl_get( &scen.units, i ) );
                            }
                            /* video mode */
                            config.video_mode = new_mode;
                        }
                        break;

                }

        }

    }
    close_window( engine->menu );

    /* if video mode has changed inform the engine about that */
    /* video mode is applied as action */
    if ( old_mode.id != config.video_mode.id )
        engine->action = APPLY_VIDEO_MODE;
}

/* run save/load menu and allow the action defined by type: ID_SAVE_MENU,ID_LOAD_MENU */
void run_save_menu( Engine *engine, int type )
{
    Text *slot_names;
    int slot_id;
    SDL_Event event;
    Window *child = 0;
    int leave = 0;
    int i;
    char str[256];

    /* change label due to type */
    if ( type == ID_SAVE_MENU )
        set_simple_window_text( get_child( engine->save_menu, ID_LABEL ), "Choose slot to save game:",
                         NO_REFRESH );
    else
        set_simple_window_text( get_child( engine->save_menu, ID_LABEL ), "Choose slot to load game:",
                         NO_REFRESH );
    /* get a list of save slots */
    slot_names = get_slot_names( engine );
    /* set the names of the buttons according to this list */
    for ( i = 0; i < slot_names->count; i++ )
        set_simple_window_text( get_child( engine->save_menu, ID_SLOT1 + i ), slot_names->lines[i], NO_REFRESH );
    delete_text( slot_names );
    /* run window */
    open_window( engine->save_menu, REFRESH );
    while ( !leave ) {

        if ( SDL_WaitEvent( &event ) ) {

            handle_window_event( engine->save_menu, &event, &child );
            update_window( engine->save_menu, IF_NEEDED );
            if ( child ) {
                if ( child->id == ID_CANCEL )
                    leave = 1;
                else
                    if ( child->id >= ID_SLOT1 && child->id < ID_SLOT1 + SLOT_COUNT ) {
                        /* get the slot id */
                        slot_id = child->id - ID_SLOT1;
                        /* valid slot to load? */
                        if ( type == ID_LOAD_MENU && !engine->slots[slot_id].valid ) continue;
                        /* if you want to save run and edit to query the new slot name */
                        if ( type == ID_SAVE_MENU ) {
                            update_edit_window( engine, "Enter new slot name:", engine->slots[slot_id].name, 20 );
                            if ( run_stan_window( engine->edit_window ) == ID_ACCEPT ) {
                                update_slot_name( engine, slot_id,
                                                  get_child( engine->edit_window, ID_EDIT )->text->lines[0] );
                                if ( save_game( engine, slot_id ) )
                                    engine->slots[slot_id].valid = 1;
                                leave = 1;
                            }
                        }
                        else {
                            /* load data */
                            set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ),
                                             "Do you really want to", NO_REFRESH );
                            sprintf( str, "load slot '%s'?", engine->slots[slot_id].name );
                            set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ),
                                             str, NO_REFRESH );
                            if ( run_stan_window( engine->conf_window ) == ID_ACCEPT )
                                if ( load_game( engine, slot_id ) ) leave = 1;
                        }
                    }
                }
        }
    }
    close_window( engine->save_menu );
}

/* update edit window; set label, exit contents and max length */
void update_edit_window( Engine *engine, char *label_str, char *edit_str, int max )
{
    Window *edit = 0;

    edit = engine->edit_window;
    set_simple_window_text( get_child( edit, ID_LABEL ), label_str, NO_REFRESH );
    set_simple_window_text( get_child( edit, ID_EDIT ), edit_str, NO_REFRESH  );
    get_child( edit, ID_EDIT )->max = max;
}

/* cur_v is the current value whereas v is the basic value */
char* get_info_str( char *str, int v, int cur_v )
{
    char aux[128];
    char diff_str[32];
    int abs_v = abs( v );
    int i;
    int offset = 20;
    char space_str[32]; /* spaces to show values at the same offset */

    /* get spacer */
    for ( i = 0; i < offset - strlen( str ); i++ )
        space_str[i] = 32;
    space_str[i] = 0;

    /* basic value; in brackets if only as defence */
    cur_v = abs( cur_v );
    if ( v == 0 )
        sprintf( aux, "%s%s -", str, space_str );
    else
        if ( v >= 0 )
            sprintf( aux, "%s%s %i", str, space_str, abs_v );
        else
            sprintf( aux, "%s%s [%i]", str, space_str, abs_v );
    /* stop here if current value is the same */
    if ( cur_v != 999 && abs_v != cur_v ) {
        /* else show difference in brackets */
        strcat( aux, " (" );
        if ( cur_v - abs_v > 0 )
           strcat( aux, "+" );
        sprintf( diff_str, "%i)", cur_v - abs_v );
        strcat( aux, diff_str );
    }
    return strdup( aux );
}

/*
====================================================================
Update the contents of the passed listbox with the unit info.
====================================================================
*/
void set_unit_info( Window *list, Unit *unit, int ref )
{
    char str[128];
    Text *text;
    int i, j;

    /* use briefing window for this */
    /* layout:
        name
        caption(unit class)
        origin
        target type

        intiative
        movement
        spotting
        range
        attackvalues
        defence
        ammo
        fuel

        (transporter if any)
        caption
        target_type

        initative
        ...
        (no range)
        (no attack values as transporter got none)
        ...
        (no ammo)
        ...
    */

    /* allocate memory and compute number of lines */
    text = calloc( 1, sizeof( Text ) );
    text->count = 14 + unit->prop.unit_def->type_count;
    if ( unit->tran_prop_set )
        text->count += 11;
    text->lines = calloc( text->count, sizeof( char* ) );

    i = 0;
    /* basic unit information */
    text->lines[i++] = strdup( unit->name );
    sprintf( str, "%s (%s)", unit->prop.cap, scen.unit_def->classes[unit->prop.class].name );
    text->lines[i++] = strdup( str );
    text->lines[i++] = strdup( scen.nations[unit->prop.nation]->name );
    sprintf( str, "%s Target", scen.unit_def->type_names[unit->prop.type] );
    str[0] = toupper( str[0] );
    text->lines[i++] = strdup( str );
    text->lines[i++] = strdup( " " );

    /* experience */
    text->lines[i++] = get_info_str( "Experience:", unit->exp, 999);//unit->cur_prop.ini );
    text->lines[i++] = strdup( " " );

    /* unit properties */
    text->lines[i++] = get_info_str( "Initiative:", unit->prop.ini, 999);//unit->cur_prop.ini );
    text->lines[i++] = get_info_str( "Movement:", unit->prop.mov, 999 );
    text->lines[i++] = get_info_str( "Spotting:", unit->prop.spot, 999 );
    text->lines[i++] = get_info_str( "Range:", unit->prop.range, 999 );
    for ( j = 0; j < unit->prop.unit_def->type_count; j++ ) {
        sprintf( str, "%s Attack:", scen.unit_def->type_names[j] );
        str[0] = toupper( str[0] );
        text->lines[i++] = get_info_str( str, unit->prop.attack[j], 999);//unit->cur_prop.attack[i] );
    }
    text->lines[i++] = get_info_str( "Defence:", unit->prop.def, 999);//unit->cur_prop.def );
    text->lines[i++] = get_info_str( "Ammo:", unit->prop.ammo, 999 );
    text->lines[i++] = get_info_str( "Fuel:", unit->prop.fuel, 999 );

    /* transporter info if any */
    if ( unit->tran_prop_set ) {
        /* basic info */
        text->lines[i++] = strdup( " " );
        if ( unit->sel_prop == &unit->tran_prop )
            sprintf( str, "Transporter: (in use)");
        else
            sprintf( str, "Transporter: (not in use)");
        text->lines[i++] = strdup( str );
        text->lines[i++] = strdup( " " );
        text->lines[i++] = strdup( unit->tran_prop.cap );
        sprintf( str, "%s Target", scen.unit_def->type_names[unit->tran_prop.type] );
        str[0] = toupper( str[0] );
        text->lines[i++] = strdup( str );
        text->lines[i++] = strdup( " " );
        /* properties though no attack properties */
        text->lines[i++] = get_info_str( "Initiative:", unit->tran_prop.ini, 999 );
        text->lines[i++] = get_info_str( "Movement:", unit->tran_prop.mov, 999 );
        text->lines[i++] = get_info_str( "Spotting:", unit->tran_prop.spot, 999 );
        text->lines[i++] = get_info_str( "Defence:", unit->tran_prop.def, 999 );
        text->lines[i++] = get_info_str( "Fuel:", unit->tran_prop.fuel, 999 );
    }

    set_window_text( list, text, ref );
}

/*
====================================================================
Show info of unit by using briefing window.
====================================================================
*/
void show_unit_info( Engine *engine, Unit *unit )
{
    SDL_Cursor *old_cursor = SDL_GetCursor();

    SDL_SetCursor( engine->stan_cursor );
    set_unit_info( get_child( engine->brief_window, 101 ), unit, NO_REFRESH );
    run_stan_window( engine->brief_window );
    SDL_SetCursor( old_cursor );
}

/*
====================================================================
Update scenario info window.
====================================================================
*/
void update_scen_info_window( Engine *engine )
{
    char text_str[1024*3];
    char str[256];
    Player *next_player = 0;
    char major_limit_str[64];
    char minor_limit_str[64];
    char forecast_str[12];
    int i;
    Player **att_team;
    int att_team_count;

    /* title */
    sprintf( str, "Scenario: %s", scen.title );
    set_simple_window_text( get_child( engine->scen_info_window, ID_TITLE_LABEL ), str, NO_REFRESH );

    /* get next player */
    if ( engine->player_id + 1 == scen.player_count )
        next_player = scen.players[0];
    else
        next_player = scen.players[engine->player_id + 1];
    /* get minor, major turn limit */
    if ( scen.major_turn_limit - scen.cur_turn <= 0 )
        sprintf( major_limit_str, "Major Victory Turn Limit: -" );
    else
        if ( scen.major_turn_limit - scen.cur_turn == 1 )
            sprintf( major_limit_str, "Major Victory Turn Limit: Last Turn" );
        else
            sprintf( major_limit_str, "Major Victory Turn Limit: %i Turns",
                     scen.major_turn_limit - scen.cur_turn );
    if ( scen.minor_turn_limit - scen.cur_turn > 1 )
        sprintf( minor_limit_str, "Minor Victory Turn Limit: %i Turns", scen.minor_turn_limit - scen.cur_turn );
    else
        sprintf( minor_limit_str, "Minor Victory Turn Limit: Last Turn" );
    /* forcast string */
    if ( scen.cur_turn + 1 >= scen.minor_turn_limit )
        sprintf( forecast_str, "-" );
    else
        sprintf( forecast_str, map.def->weather_names[scen.weather[scen.cur_turn + 1]] );

    /* setup list */
    sprintf( text_str,
             "Current Player: %s#Next Player: %s##"
             "%s#%s##"
             "Weather: %s (Forecast: %s)##"
             "Description:#%s##",
             engine->player->name, next_player->name,
             major_limit_str, minor_limit_str,
             map.def->weather_names[scen.weather[scen.cur_turn]], forecast_str,
             scen.desc );
    /* attackers */
    att_team = get_att_team( &att_team_count );
    strcat( text_str, "Attackers: " );
    for ( i = 0; i < att_team_count; i++ ) {
        if ( i > 0 ) strcat( text_str, ", " );
        strcat( text_str, att_team[i]->name );
    }
    strcat ( text_str, "##" );
    /* explicit objectives */
    strcat( text_str, "Explicit Objectives:#" );
    if ( scen.victory_cond.type == ALL_COND )
        strcat( text_str, "The attackers must fullfill ALL of the following objectives:" );
    else
        strcat( text_str, "The attackers must fullfill at least ONE of the following objectives:" );
    /* required obj count */
    sprintf( str, "#-%i strategic flags of choice captured", scen.victory_cond.req_obj_count );
    strcat( text_str, str );
    /* primary objectives */
    for ( i = 0; i < scen.victory_cond.prim_obj_count; i++ ) {
        sprintf( str, "#-strategic flag at %i,%i captured",
                 scen.victory_cond.prim_obj[i].x, scen.victory_cond.prim_obj[i].y );
        strcat( text_str, str );
    }

    set_window_text( get_child( engine->scen_info_window, ID_INFO_LIST ),
                     create_text( text_str,
                                  get_child( engine->scen_info_window, ID_INFO_LIST )->char_limit ),
                     NO_REFRESH );
}

/*
====================================================================
Run options window.
====================================================================
*/
void prep_options_window( Engine *engine )
{
    /* set config */
    get_child( engine->options, ID_SUPPLY_BUTTON )->is_on =
        config.supply;
    get_child( engine->options, ID_WEATHER_BUTTON )->is_on =
        config.weather;
    get_child( engine->options, ID_CPU_BUTTON )->is_on =
        config.show_cpu_turn;
    get_child( engine->options, ID_LIFE_BAR_BUTTON )->is_on =
        config.show_bar;
    get_child( engine->options, ID_GRID_BUTTON )->is_on =
        config.grid;
    get_child( engine->options, ID_SOUND_BUTTON )->is_on =
        config.sound_on;
    /* set mode button caption */
    set_simple_window_text( get_child( engine->options, ID_MODE_BUTTON ), config.video_mode.name, NO_REFRESH );
    /* set position of volume controllers */
    get_child( engine->options, ID_SOUND_VOLUME_SCROLL )->pos = config.sound_volume;
    adjust_track_button( get_child( engine->options, ID_SOUND_VOLUME_SCROLL ), NO_REFRESH );
    get_child( engine->options, ID_MUSIC_VOLUME_SCROLL )->pos = config.music_volume;
    adjust_track_button( get_child( engine->options, ID_MUSIC_VOLUME_SCROLL ), NO_REFRESH );
}
int run_options_window( Engine *engine, Video_Mode *new_mode )
{
    SDL_Event event;
    int leave = 0;
    int ret = 0;
    Window *child = 0;
    int mode_id;

    open_window( engine->options, REFRESH  );
    while( !leave ) {
        if ( SDL_WaitEvent( &event ) ) {
            handle_window_event( engine->options, &event, &child );
            if ( child )
                switch ( child->id ) {
                    case ID_ACCEPT: ret = 1; leave = 1; break;
                    case ID_CANCEL: ret = 0; leave = 1; break;
                    case ID_MODE_BUTTON:
                        if ( run_mode_window( engine, &mode_id ) ) {
                            *new_mode = std_video_mode( mode_id );
                            set_simple_window_text( get_child( engine->options, ID_MODE_BUTTON ),
                                                    new_mode->name, REFRESH );
                        }
                        break;
                    case ID_SOUND_VOLUME_SCROLL:
#ifdef WITH_SOUND
                        sound_volume( get_child( engine->options, ID_SOUND_VOLUME_SCROLL )->pos );
#endif
                        break;
                    case ID_SOUND_BUTTON:
#ifdef WITH_SOUND
                        sound_enable( is_on( get_child( engine->options, ID_SOUND_BUTTON ) ) );
#endif
                        break;
                    case ID_MUSIC_BUTTON:
                        /* get to track menu */
                        printf( "Sorry, but music's not implemented so far.\n" );
                        break;
                    case ID_MUSIC_VOLUME_SCROLL:
#ifdef WITH_SOUND
                        music_volume( get_child( engine->options, ID_MUSIC_VOLUME_SCROLL )->pos );
#endif
                        break;
                }
        }
        update_window( engine->options, IF_NEEDED );
    }
    close_window( engine->options );

    /* always check sound */
    if ( ret ) {
        config.sound_on =
            is_on( get_child( engine->options, ID_SOUND_BUTTON ) );
        config.sound_volume =
            get_child( engine->options, ID_SOUND_VOLUME_SCROLL )->pos;
        config.music_volume =
            get_child( engine->options, ID_MUSIC_VOLUME_SCROLL )->pos;
    }
#ifdef WITH_SOUND
    sound_enable( config.sound_on );
    sound_volume( config.sound_volume );
    music_volume( config.music_volume );
#endif

    return ret;
}
void apply_options( Engine *engine )
{
    /* apply_changes */
    config.supply =
        is_on( get_child( engine->options, ID_SUPPLY_BUTTON ) );
    config.weather =
        is_on( get_child( engine->options, ID_WEATHER_BUTTON ) );
    config.show_cpu_turn =
        is_on( get_child( engine->options, ID_CPU_BUTTON ) );
    config.show_bar =
        is_on( get_child( engine->options, ID_LIFE_BAR_BUTTON ) );
    config.grid =
        is_on( get_child( engine->options, ID_GRID_BUTTON ) );
}

/*
====================================================================
Run video mode window.
====================================================================
*/
Window *create_mode_window( Theme *theme )
{
    int width, height;
    Window *window;
    int i;
    Video_Mode mode;
    Window *button;

    width = 340;
    height = mode_count * 20 + 110;

    window = create_wallpaper_window( theme->parent_back, theme->parent_frame, width, height );
    create_label( window, ID_NONE, 20, 20, window->width - 40, 14, theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "Select A Resolution:" );
    create_text_button( window, ID_CANCEL, theme->menu_text_button,
                        window->width / 2 - 40, window->height - 40, 80, 20,
                        theme->stan_font, "Cancel" );
    for ( i = 0; i < mode_count; i++ ) {
        mode = std_video_mode( i );
        button = create_text_button( window, 101 + i, theme->menu_text_button,
                                     20, 50 + i * 20, width - 40, 20,
                                     theme->stan_font, mode.name );
        if ( !mode.ok ) button->active = 0;
    }
    return window;
}
int run_mode_window( Engine *engine, int *mode_id )
{
    SDL_Event event;
    int leave = 0;
    int ret = 0;
    Window *child = 0;

    open_window( engine->mode_window, REFRESH  );
    while( !leave ) {
        if ( SDL_WaitEvent( &event ) ) {
            handle_window_event( engine->mode_window, &event, &child );
            update_window( engine->mode_window, IF_NEEDED );
            if ( child ) {
                if ( child->id == ID_CANCEL ) {
                    ret = 0; leave = 1;
                    break;
                }
                else
                    if ( child->id >= 101 && child->id < 101 + mode_count ) {
                        ret = 1;
                        leave = 1;
                        *mode_id = child->id - 101;
                        break;
                    }
            }
        }
    }
    close_window( engine->mode_window );

    return ret;
}

/*
====================================================================
Create upper/lower bar.
====================================================================
*/
void create_upper_bar( Engine *engine )
{
    int label_length;
    int middle_label_length;
    /* create upper bar and reset position */
    engine->upper_bar = create_wallpaper_window( engine->theme->upper_bar_back, 0, sdl.screen->w, 24 );
    engine->upper_bar->x = 0; engine->upper_bar->y = 0;
    label_length = ( sdl.screen->w - 20 ) / 3;
    middle_label_length = (sdl.screen->w - 20 ) - 2 * label_length;
    create_label( engine->upper_bar, ID_LEFT_LABEL, 10, 4, label_length, 16, engine->theme->stan_font,
                  ALIGN_X_LEFT | ALIGN_Y_CENTER, "" );
    create_label( engine->upper_bar, ID_MIDDLE_LABEL, 10 + label_length, 4, middle_label_length, 16, engine->theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    create_label( engine->upper_bar, ID_RIGHT_LABEL, 10 + label_length + middle_label_length, 4, label_length, 16, engine->theme->stan_font,
                  ALIGN_X_RIGHT | ALIGN_Y_CENTER, "" );
}
void create_lower_bar( Engine *engine )
{
    int right_border;
    int button_width, button_index;

    /* create lower bar and reset position */
    engine->lower_bar = create_wallpaper_window( engine->theme->lower_bar_back, 0, sdl.screen->w, 40 );
    engine->lower_bar->x = 0; engine->lower_bar->y = sdl.screen->h - engine->lower_bar->height;
    /* add children */
    create_button( engine->lower_bar, ID_MENU_BUTTON, engine->theme->menu_button, 6, 8, 24, 24,
                   "Main Menu", get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_icon( engine->lower_bar, ID_ATTACK_ICON, 48, 8,
                 engine->theme->status_icon_width,
                 engine->theme->status_icon_height,
                 0, 0, 0, 0, 0 );
    create_icon( engine->lower_bar, ID_MOVE_ICON, 48, 8 + engine->theme->status_icon_height,
                 engine->theme->status_icon_width,
                 engine->theme->status_icon_height,
                 0, 0, 0, 0, 0 );
    create_label( engine->lower_bar, ID_STATUS_LABEL, 60, 6, 110, 14, engine->theme->status_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    create_label( engine->lower_bar, ID_EXP_LABEL, 60, 20, 110, 14, engine->theme->status_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    create_label( engine->lower_bar, ID_NAME_LABEL, 170, 6, 160, 14, engine->theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );
    create_label( engine->lower_bar, ID_CAP_LABEL, 170, 20, 160, 14, engine->theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "" );

    right_border = 30;
    button_width = 30;
    button_index = -1;

    create_button( engine->lower_bar, ID_END_TURN_BUTTON, engine->theme->end_turn_button,
                   sdl.screen->w - right_border  - (++button_index) * button_width, 8, 24, 24, "End Turn",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_SCEN_INFO_BUTTON, engine->theme->scen_info_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Scenario Information",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_STRAT_MAP_BUTTON, engine->theme->map_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Strategic Map",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_AIR_MODE_BUTTON, engine->theme->air_mode_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Switch Air/Ground Mode",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );

    create_button( engine->lower_bar, ID_DEPLOY_BUTTON, engine->theme->deploy_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Deploy Units",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );

    create_button( engine->lower_bar, ID_MERGE_BUTTON, engine->theme->melt_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Merge Units",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_UNDO_BUTTON, engine->theme->undo_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Undo Unit Move",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_RENAME_BUTTON, engine->theme->rename_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Rename Unit",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_EMBARK_BUTTON, engine->theme->embark_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Embark/Debark Unit",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
    create_button( engine->lower_bar, ID_SUPPLY_BUTTON, engine->theme->supply_button,
                   sdl.screen->w - right_border - (++button_index) * button_width, 8, 24, 24,
                   "Supply Unit",
                   get_child( engine->upper_bar, ID_MIDDLE_LABEL ) );
}

/*
====================================================================
Deploy units window
====================================================================
*/
Window* create_deploy_window( Engine *engine )
{
    Window *window = 0;
    int scroll_x;
    int icon_height, icon_width;
    int icon_count;
    int i;

    window = create_wallpaper_window( engine->theme->parent_back, engine->theme->parent_frame, 500, 400 );

    /* title label */
    create_label( window, ID_NONE, 20, 20, window->width - 40, 20, engine->theme->stan_font,
                  ALIGN_X_CENTER | ALIGN_Y_CENTER, "Select A Unit To Deploy:" );

    /* unit info list */
    create_list( window, ID_INFO_LIST, engine->theme->list_frame, window->width - 300, 60,
                 280, window->height - 120, 6,
                 engine->theme->stan_font, 0, NO_SELECT , 0x0000ff,
                 0, engine->theme->down_arrow, engine->theme->up_arrow, engine->theme->track_ball,
                 20 );
    /* deploy button */
    create_text_button( window, ID_DEPLOY_UNIT, engine->theme->menu_text_button,
                        20, window->height - 40, 120, 20,
                        engine->theme->stan_font, "Deploy" );
    /* apply */
    create_text_button( window, ID_APPLY_DEPLOY, engine->theme->menu_text_button,
                        window->width / 2 - 60, window->height - 40, 120, 20,
                        engine->theme->stan_font, "Apply Deploy" );
    /* cancel button */
    create_text_button( window, ID_CANCEL_DEPLOY, engine->theme->menu_text_button,
                        window->width - 140, window->height - 40, 120, 20,
                        engine->theme->stan_font, "Cancel Deploy" );
    /* create scrollbar */
    scroll_x = 160;
    create_scroll( window, ID_UNIT_SCROLL, 0,
                   scroll_x, 60, 20, window->height - 120,
                   VERT,
                   engine->theme->down_arrow, engine->theme->up_arrow, engine->theme->track_ball,
                   0, 0, 0, 0 );

    /* wanted height per tile is tile_height + 20; not known here so assume 60 */
    icon_count = ( window->height - 120 ) / ( 60 );
    icon_height = ( window->height - 120 ) / icon_count;
    icon_width = scroll_x - 20;

    /* create icons */
    for ( i = 0; i < icon_count; i++ ) {
        create_frame_icon( window, ID_UNIT_ICON1 + i, engine->theme->list_frame,
                           20, 60 + i * icon_height, icon_width, icon_height,
                           0, 0, 0, 0, 0 );
        /* connect to scrollbar */
        get_child( window, ID_UNIT_ICON1 + i )->list_scroll = get_child( window, ID_UNIT_SCROLL );
    }

    /* set engine stuff */
    engine->deploy_icon_count = icon_count;
    engine->deploy_unit_offset = 0;
    engine->deploy_sel_id = -1;

    return window;
}
/*
====================================================================
Update the unit icons containing the pictures of the reinforcements
by engine::deploy_unit_offset
====================================================================
*/
void update_deploy_units( Engine *engine, int ref )
{
    int i;
    Unit *unit;

    /* clear icons */
    for ( i = 0;i < engine->deploy_icon_count; i++ )
        set_icon( get_child( engine->deploy_window, ID_UNIT_ICON1 + i ),
                  0, 0, 0, 0, 0, ref );
    /* set new icons */
    for ( i = 0; i < engine->deploy_icon_count; i++ )
        if ( i + engine->deploy_unit_offset < engine->deploy_unit_count ) {
            unit = engine->deploy_units[i + engine->deploy_unit_offset];
            set_icon( get_child( engine->deploy_window, ID_UNIT_ICON1 + i ),
                      unit->prop.pic, 0, 0, unit->prop.width, unit->prop.height,
                      ref );
        }
}
/*
====================================================================
Unselect currently selected icon if any. Does not change selected
unit!
====================================================================
*/
void unselect_deploy_icons( Engine *engine, int ref )
{
    int i;
    Window *icon;
    for ( i = 0; i < engine->deploy_icon_count; i++ ) {
        icon = get_child( engine->deploy_window, ID_UNIT_ICON1 + i );
        if ( icon->is_on )
            unselect_icon( icon, ref );
    }
}
/*
====================================================================
Select a deploy unit and refresh if wanted.
====================================================================
*/
void select_deploy_unit( Engine *engine, Unit *unit, int ref )
{
    int icon_id;
    int unit_id;
    int info_changed = engine->deploy_units[engine->deploy_sel_id] != unit;

    /* get unit id for engine::deploy_units */
    for ( unit_id = 0; unit_id < engine->deploy_unit_count; unit_id++ )
        if ( engine->deploy_units[unit_id] == unit ) break;
    if ( unit_id == engine->deploy_unit_count ) return;

    /* get valid icon id if unit is visible */
    icon_id = unit_id - engine->deploy_unit_offset;
    if ( icon_id < 0 ) icon_id = -1;
    if ( icon_id >= engine->deploy_icon_count ) icon_id = -1;

    /* if unit seen update frame icon */
    if ( icon_id != -1 )
        select_icon( get_child( engine->deploy_window, ID_UNIT_ICON1 + icon_id ), ref);
    /* new selection id */
    engine->deploy_sel_id = unit_id;

    /* display info */
    if ( info_changed )
        set_unit_info( get_child( engine->deploy_window, ID_INFO_LIST ),
                       engine->deploy_units[engine->deploy_sel_id], ref );
}
/*
====================================================================
Update deploy unit scroll bar. As this is only
done in prep_deploy use NO_REFRESH
====================================================================
*/
void update_deploy_scroll( Engine *engine )
{
    int max;

    max = engine->deploy_unit_count - engine->deploy_icon_count;
    if ( max < 0 ) max = 0;

    set_scroll_range( get_child( engine->deploy_window, ID_UNIT_SCROLL ),
                      0, max, engine->deploy_icon_count, engine->deploy_unit_offset,
                      NO_REFRESH );
}
/*
====================================================================
Run unit window.
====================================================================
*/
void prep_deploy_window( Engine *engine )
{
    create_deploy_list( engine );

    /* if no unit selected so far: select first unit */
    if ( engine->deploy_sel_id == -1 )
        select_deploy_unit( engine, engine->deploy_units[0], NO_REFRESH );
    else {
        /* display selected unit at top if there are any other units left to fill othwe icons */
        if ( engine->deploy_unit_offset < engine->deploy_unit_count - engine->deploy_icon_count )
            engine->deploy_unit_offset = engine->deploy_sel_id;
        else
            engine->deploy_unit_offset = 0;
        /* select unit */
        unselect_deploy_icons( engine, NO_REFRESH );
        if ( engine->deploy_unit_count > 0 )
            select_deploy_unit( engine, engine->deploy_units[engine->deploy_sel_id], NO_REFRESH );
    }

    /* update scroll */
    update_deploy_scroll( engine );

    /* update unit list: show only units whose position x != -1 */
    update_deploy_units( engine, NO_REFRESH );

    /* update unit info */
    if ( engine->deploy_unit_count > 0 )
        set_unit_info( get_child( engine->deploy_window, ID_INFO_LIST ),
                       engine->deploy_units[engine->deploy_sel_id], NO_REFRESH );
    else
        set_window_text( get_child( engine->deploy_window, ID_INFO_LIST ), create_text( "", 10 ), NO_REFRESH );
}
void run_deploy_window( Engine *engine, int *type, Unit **unit )
{
    SDL_Event event;
    int leave = 0;
    Window *child = 0;

    SDL_SetCursor( engine->stan_cursor );

    open_window( engine->deploy_window, REFRESH  );
    while( !leave ) {
        if ( SDL_WaitEvent( &event ) ) {
            handle_window_event( engine->deploy_window, &event, &child );
            if ( child ) {
                switch ( child->id ) {
                    case ID_CANCEL_DEPLOY:
                        if ( run_conf_window( engine, "Do you really want to", "discard all changes?" ) ) {
                            *type = CANCEL_DEPLOY;
                            leave = 1;
                        }
                        break;
                    case ID_APPLY_DEPLOY:
                        if ( run_conf_window( engine, "Do you really want to", "deploy these units?" ) ) {
                            *type = APPLY_DEPLOY;
                            leave = 1;
                        }
                        break;
                    case ID_DEPLOY_UNIT:
                        *type = DEPLOY_UNIT;
                        leave = 1;
                        /* set unit */
                        *unit = engine->deploy_units[engine->deploy_sel_id];
                        break;
                    case ID_UNIT_SCROLL:
                        /* change offset */
                        if ( child->pos == engine->deploy_unit_offset ) break;
                        engine->deploy_unit_offset = child->pos;
                        /* unselect current icon */
                        unselect_deploy_icons( engine, NO_REFRESH );
                        /* update icons as offset may have changed */
                        update_deploy_units( engine, REFRESH );
                        /* reselect current unit */
                        select_deploy_unit( engine, engine->deploy_units[engine->deploy_sel_id], REFRESH );
                        break;
                    default:
                        /* clicked on icon? */
                        if ( child->id >= ID_UNIT_ICON1 && child->id < ID_UNIT_ICON1 + engine->deploy_icon_count ) {
                            /* check if this tile contains a unit, too */
                            if ( child->id - ID_UNIT_ICON1 + engine->deploy_unit_offset < engine->deploy_unit_count ) {
                                unselect_deploy_icons( engine, REFRESH );
                                select_deploy_unit( engine,
                                                    engine->deploy_units[child->id - ID_UNIT_ICON1 + engine->deploy_unit_offset],
                                                    REFRESH );
                            }
                            else
                                /* unselect this icon */
                                unselect_icon( child, REFRESH );
                        }
                        break;
                }
            }
            update_window( engine->deploy_window, IF_NEEDED );
        }
    }
    close_window( engine->deploy_window );
}
