/***************************************************************************
                          game_loop.c  -  description
                             -------------------
    begin                : Thu Jun 28 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                                                                                                  *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.                                                                                *
 *                                                                                                                                                  *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "sdl.h"
#include "config.h"
#include "tools.h"
#include "dynlist.h"
#include "file.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "date.h"
#include "map.h"
#include "scenario.h"
#include "campaign.h"
#include "gui.h"
#include "theme.h"
#include "ai_action.h"
#include "engine_tools.h"
#include "window.h"
#include "engine.h"

extern Scen scen;
extern Camp camp;
extern Sdl sdl;

/*
====================================================================
Parse the loaded campaign for a scenario with this name and return its id else -1.
====================================================================
*/
int get_scen_id( char *name )
{
    int i;
    /* maybe no scenario follows? */
    if ( equal_str( "none", name ) )
        return -1;
	//search list//
    for ( i = 0; i < camp.scen_count; i++ )
        if ( equal_str( camp.scen_info[i].name, name ) )
            return i;
    fprintf( stderr, "get_next_scen_info: can't find campaign entry '%s'\n", name );
    return -1;
}

/*
====================================================================
Set next scenario by checking the victory type of the current scenario. Return 1
if a scenario follows else 0.
====================================================================
*/
int set_next_scen_id( int vic_type ) {
	/* check if attackers won and if it's a major victory */
	switch ( vic_type ) {
		/* if you were the attacker a defeat means: retry this scenario */
		case DEFEAT:
			if ( scen.human_is_att ) return 1; /* reinitiate same scenario */
			/* defending always means minor victory */
			camp.scen_id = get_scen_id( camp.scen_info[camp.scen_id].next_name_minor );
			if ( camp.scen_id == -1 ) /* leave scenario we're done */
				return 0;
			return 1;
		case MAJOR_VIC:
			if ( !scen.human_is_att ) return 1;
			camp.scen_id = get_scen_id( camp.scen_info[camp.scen_id].next_name_major );
			if ( camp.scen_id == -1 ) /* leave scenario we're done */
				return 0;
			return 1;
		case MINOR_VIC:
			if ( !scen.human_is_att ) return 1;
			camp.scen_id = get_scen_id( camp.scen_info[camp.scen_id].next_name_minor );
			if ( camp.scen_id == -1 ) /* leave scenario we're done */
				return 0;
			return 1;
		default:
			/* just leave the game */
			return 0;
	}
}

/*
====================================================================
Prepares the main game loop for run_game().
file_name specfies the main resource which is either a campaign  or a scenario
depending on if single_scen is true. If it's a single scenario the default controls
may be overwritten by passing player_cont != 0 which is checked and
freed by init_engine(...).
====================================================================
*/
void prep_game( Engine *engine,
	            char *file_name, int single_scen, int *player_cont )
{
    char *first_scen = 0;

    /* load either campaign or scenario */
    if ( single_scen )
        first_scen = file_name;
    else {
        load_camp( file_name );
        first_scen = camp.scen_info[camp.scen_id].name;
        if ( player_cont ) {
            fprintf( stderr, "prep_game: you're not allowed to overwrite player controls in a campaign\n" );
            FREE( player_cont );
        }
    }

    /* in the first scenario is this must already be done before entering run_game() */
    init_engine( engine, first_scen, player_cont );
}	
	
/*
====================================================================
Set background for briefing
====================================================================
*/
void set_brief_back( Engine *engine )
{
    int x, y;	
    /* if title window is smaller fill whole screen with parent_back wallpaper */
    for ( x = 0; x < sdl.screen->w; x += engine->theme->parent_back->w )
        for ( y = 0; y < sdl.screen->h; y += engine->theme->parent_back->h ) {
            DEST( sdl.screen, x, y, engine->theme->parent_back->w, engine->theme->parent_back->h );
            SOURCE( engine->theme->parent_back, 0, 0 );
            blit_surf();
        }
    /* show title picture */
    DEST( sdl.screen,
          ( sdl.screen->w - engine->theme->title_pic->w ) / 2,
          ( sdl.screen->h - engine->theme->title_pic->h ) / 2,
          engine->theme->title_pic->w, engine->theme->title_pic->h );
    FULL_SOURCE( engine->theme->title_pic );
    blit_surf();
    refresh_screen( 0, 0, 0, 0 );
}

/*
====================================================================
file_name specfies the main resource which is either a campaign  or a scenario
depending on if single_scen is true. If it's a single scenario the default controls
may be overwritten by passing player_cont != 0 which is checked and
freed by init_engine(...).
====================================================================
*/
void run_game( Engine *engine )
{
    int leave = 0;
    char *cur_scen_name = 0; /* file name of current scenario */
	/* campaign stuff */
    int final_msg = 1; /* display final message  */
    int vic_type; /* return value of run_engine(..) used to get next scenario */
    int first_scen = 1;

    /* if camp.scen_count is >0 a campaign is running */
    while ( !leave ) {
        /* campaign briefing if not loaded a previous game */
        if ( camp.scen_count > 0 && !engine->scen_loaded ) {
            set_brief_back( engine );
            run_brief_window( engine, camp.scen_info[camp.scen_id].brief );
        }
        /* as the first scenario is already loaded init_engine() must not be called again
           if this function is ever called again it's a campaign and therefore 0 may be
           passed as player_cont */
        if ( !first_scen )
            init_engine( engine, cur_scen_name, 0 );
        first_scen = 0;
        /* run scenario */
        vic_type = run_engine( engine );
		/* check victory_type and determine next scenario (only if campaign is running) */
        if ( camp.scen_count > 0 ) {
			/* if victory type is NONE the game has been left */
			if ( vic_type == NONE ) {
				final_msg = 0;
				leave = 1;
			}
			else
				if ( !set_next_scen_id( vic_type )  )
					leave = 1;
				else
					/*  get next scenario name */
					cur_scen_name = camp.scen_info[camp.scen_id].name;
        }
        else
            leave = 1;
        /* clean up */
        clear_engine( engine );
    }

    /* display final message if any */
    if ( camp.scen_count > 0 && final_msg )
        run_brief_window( engine, camp.final_msg );
    /* clean up campaign */
    if ( camp.scen_count > 0 )
        clear_camp();
}
