/*
 * kl_dump_arm.h
 * 
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Fleming Feng (fleming.feng@intel.com)
 *
 * Copyright (C) 2003 Intel Corp. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

/* This header file holds the architecture specific crash dump header */
#ifndef __KL_DUMP_ARM_H
#define __KL_DUMP_ARM_H

/* macros
 */
#ifndef KL_NR_CPUS
# define KL_NR_CPUS 32     /* max number CPUs */
#endif

#define KL_DUMP_MAGIC_NUMBER_ARM     0xdeaddeadULL  /* magic number         */
#define KL_DUMP_VERSION_NUMBER_ARM   0x3        /* version number          */

/* KL_DUMP_REVERSE_MAGIC_NUMBER is used to determine the byte order */
#define KL_DUMP_REVERSE_MAGIC_NUMBER 0xed238f61730119a8ULL

#define DUMP_MAX_NUM_CPUS 32

#define MODE_BIT 		  0x1f
#define USER_MODE		  0x10
#define SYSTEM_MODE		  0x1f

/* structures
 */

/* some strcutures that linux kernel and lkcd defined */
/* The following structure is the exception frame
 * defined by ARM linux kernel */
struct kl_pt_regs_arm {
	uint32_t uregs[18];
};

#define ARM_cpsr	uregs[16]
#define ARM_pc		uregs[15]
#define ARM_lr		uregs[14]
#define ARM_sp		uregs[13]
#define ARM_ip		uregs[12]
#define ARM_fp		uregs[11]
#define ARM_r10		uregs[10]
#define ARM_r9		uregs[9]
#define ARM_r8		uregs[8]
#define ARM_r7		uregs[7]
#define ARM_r6		uregs[6]
#define ARM_r5		uregs[5]
#define ARM_r4		uregs[4]
#define ARM_r3		uregs[3]
#define ARM_r2		uregs[2]
#define ARM_r1		uregs[1]
#define ARM_r0		uregs[0]
#define ARM_ORIG_r0	uregs[17]

/* Structure: _arm_cp15_regs_t
 *  Function: This is the structure to hold registers for coprocessor 15
 *            of ARM. Some cp15 registers are specific for XSCale Core.
 */
typedef struct _arm_cp15_regs_s {
#ifdef CONFIG_CPU_XSCALE
	uint32_t uregs[16];
#else	/* CONFIG_CPU_XSCALE */
	uint32_t uregs[9];
#endif	/* CONFIG_CPU_XSCALE */
} _arm_cp15_regs_t;

#define ARM_cp15_id		uregs[0]	/* ID */
#define ARM_cp15_ct		uregs[1]	/* Cache type */
#define ARM_cp15_ctr		uregs[2]	/* Control */
#define ARM_cp15_ttb		uregs[3]	/* Translation table base */
#define ARM_cp15_dac		uregs[4]	/* Domain access control */
#define ARM_cp15_fsr		uregs[5]	/* Fault status register */
#define ARM_cp15_far		uregs[6]	/* Fault address register */
#define ARM_cp15_clm		uregs[7]	/* Data cache lock mode */
#define ARM_cp15_pid		uregs[8]	/* Processor ID */

#ifdef CONFIG_CPU_XSCALE
/* The following are registers only exists on XScale Core */
#define XSCALE_cp15_actr	uregs[9]	/* Auxillary Control */
#define XSCALE_cp15_ibcr0	uregs[10]	/* Instruction breakpoint
						   control register 0 */
#define XSCALE_cp15_ibcr1	uregs[11]	/* Instruction breakpoint
						   control register 1 */
#define XSCALE_cp15_dbr0	uregs[12]	/* Data breakpoint address
						   register 0 */
#define XSCALE_cp15_dbr1	uregs[13]	/* Data breakpoint address
						   register 1 */
#define XSCALE_cp15_dbctr	uregs[14]	/* Data breakpoint control
						   register */
#define XSCALE_cp15_car		uregs[15]	/* Coprocessor access */

/* Structure: _xscale_cp14_regs_t
 *  Function: This is the structure to hold registers for coprocessor 14
 *            of XScale Core.
 */
typedef struct _xscale_cp14_regs_s {
	uint32_t uregs[11];
} _xscale_cp14_regs_t;

#define XSCALE_cp14_pmnc	uregs[0]	/* Perfmon control */
#define XSCALE_cp14_ccnt	uregs[1]	/* Clock counter */
#define XSCALE_cp14_pmn0	uregs[2]	/* Performance monitor
						   event counter 0 */
#define XSCALE_cp14_pmn1	uregs[3]	/* Performance monitor
						   event counter 1 */
#define XSCALE_cp14_cclk	uregs[4]	/* Core clock config */
#define XSCALE_cp14_tx		uregs[5]	/* Transmit regiser */
#define XSCALE_cp14_rx		uregs[6]	/* Receive register */
#define XSCALE_cp14_dcsr	uregs[7]	/* Debug control and
						   status register */
#define XSCALE_cp14_chkpt0	uregs[8]	/* Checkpoint register 0 */
#define XSCALE_cp14_chkpt1	uregs[9]	/* Checkpoint register 1 */
#define XSCALE_cp14_txrxctl	uregs[10]	/* Tx/Rx control register */

/* asm independent version of struct dump_regs for arm */
/* Structure: _xscale_trace_buf_t
 *  Function: This is the structure to hold registers and contents for
 *            trace buffer of XScale Core.
 */
#define TRACE_BUF_LEN		256
typedef struct _xscale_trace_buf_s {
	unsigned char tb_entry[TRACE_BUF_LEN];
} _xscale_trace_buf_t;
#endif /* CONFIG_CPU_XSCALE */

struct kl_dump_regs_arm_s {
	/* the dump registers */
	struct kl_pt_regs_arm       dha_ARM_regs;

	/* the saved psr register */
	uint32_t	     dha_ARM_spsr;

	/* coprocessor registers for ARM & XScle */
	_arm_cp15_regs_t  dha_cp15_regs;

#ifdef CONFIG_CPU_XSCALE	
	/* coprocessor registers for XScale */
	_xscale_cp14_regs_t  dha_cp14_regs;
	
	/* trace buffer registers and contents */
	_xscale_trace_buf_t  dha_trace_buf;
#endif /*CONFIG_CPU_XSCALE */
} __attribute__((packed));


/*  Header for architecture-specific stuff.
 *  It follows right after the dump header.
 */

/*
 * Structure: dump_header_asm_t
 *  Function: This is the header for architecture-specific stuff.  It
 *            follows right after the dump header.
 */
typedef struct kl_dump_header_arm_s {
        /* the dump magic number -- unique to verify dump is valid */
        uint64_t              magic_number;

        /* the version number of this dump */
        uint32_t              version;

        /* the size of this header (in case we can't read it) */
        uint32_t              header_size;

	/* The start physical memory address */
	uint32_t                physaddr_start;

	/* the registers to be dumpped */
	struct kl_dump_regs_arm_s regs;

	/* FixMe: Although currently no smp system is available,
	 * still defined here for future extension
	 */
	uint32_t                smp_num_cpus;
	uint32_t                dumping_cpu;
	struct kl_dump_regs_arm_s smp_regs[DUMP_MAX_NUM_CPUS];
	uint32_t                smp_current_task[DUMP_MAX_NUM_CPUS];
	uint32_t                stack[DUMP_MAX_NUM_CPUS];
	uint32_t                stack_ptr[DUMP_MAX_NUM_CPUS];
} kl_dump_header_arm_t;

/* Global varialbes */
extern uint32_t kl_start_physaddr;

/* function declarations
 */
kaddr_t kl_dumpeip_arm(kaddr_t);
kaddr_t kl_dumpesp_arm(kaddr_t);
int kl_get_dump_header_arm(kl_dump_header_arm_t*);
kaddr_t kl_smp_dumptask_arm(kaddr_t);
int kl_set_dumparch_arm(void);
int kl_get_target_byteorder(char* dump);
uint32_t kl_get_start_physaddr(void);

#endif /* __KL_DUMP_ARM_H */
