/*
 * $Id: lcrash.h,v 1.1 2004/12/21 23:26:19 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#ifndef __LCRASH_H
#define __LCRASH_H

#include <klib.h>
#include <alloc.h>
#include <lc_command.h>
#include <lc_dis.h>
#include <lc_trace.h>
#include <lc_report.h>
#include <lc_util.h>
#include <lc_dump_arch.h>

/*
 * indicate wether debug output should be displayed or not
 */
extern uint64_t lcrash_debug;

/*
 * Following is used when stepping through chained lists to avoid endless loops.
 */
extern uint64_t def_iter_threshold;   /* default iteration threshold */
extern uint64_t iter_threshold;       /* current iteration threshold */

/* COUNT must occur once in macro, ohterwise trouble with increment etc.
 */
#define CHECK_ITER_THRESHOLD(COUNT)\
 ((iter_threshold == 0) || ((int64_t) COUNT < (int64_t) iter_threshold))

/* COUNTER must occur once in macro, ohterwise trouble with increment etc.
 */
#define ITER_THRESHOLD_MSG(OFP, COUNTER)\
do{\
	if(!(CHECK_ITER_THRESHOLD(COUNTER))){\
		fprintf(OFP, "\nWARNING: Iteration threshold reached. "\
			"Current threshold: %"FMT64"u.\n         "\
			"Use \"-i\" to change threshold.\n", iter_threshold);\
	}\
}while(0)

#define	NEW_UTSNAME     "new_utsname"
#define BLKSZ		4096    /* maximum block size */
#define LCRASH_OPT_MAX   256    /* maximum argument size */

#ifndef FALSE
#define FALSE		   0		
#endif
#ifndef TRUE
#define TRUE		   1		
#endif

#define ADDR64(X) ((uint64_t)((kaddr_t)X))
#define INT64(X) ((int64_t)((int32_t)X))
#define UINT64(X) ((uint64_t)((uint32_t)X))

#define ADDR_FLG        0x01
#define HEX_FLG         0x02
#define OCTAL_FLG       0x04
#define SIGNED_FLG      0x08
#define UNSIGNED_FLG    0x10
#define CHAR_FLG        0x20

/* lcrash flags (used e.g. in main routine) */
#define LC_BOUNDS_FLG       0x0001
#define LC_REPORT_FLG       0x0002
#define LC_SAVE_FLG         0x0004
#define LC_ERASE_FLG        0x0008
#define LC_PROGRESS_FLG     0x0010
#define LC_INFO_FLG         0x0020

/* Function prototypes
 */

/* from struct.c */
void print_banner(FILE *, char *, int);
void print_value(FILE *, char *, uint64_t, int, int);
void mm_struct_banner(FILE *, int);
void print_mm_struct(kaddr_t, void *, int, FILE *);
void page_banner(FILE *, int);
void print_page(kaddr_t, void *, int, FILE *);
void print_page_struct(kaddr_t, void *, int, unsigned long, FILE *);
void task_struct_banner(FILE *, int);
void print_task_struct(kaddr_t, void *, int, FILE *);
int  print_active_tasks(int, FILE *);
void vm_area_struct_banner(FILE *, int);
void print_vm_area_struct(kaddr_t, void *, int, FILE *);

/* from module.c */
void module_banner(FILE *, int);
void print_module(kaddr_t, void *, int, FILE *);
void print_modsyms(void *, const char *, int, FILE *);
void print_modrefs(void *, const char *, FILE *);

/* from util.c */
void init_liballoc(int, int, int);
void dump_memory(kaddr_t, uint64_t, int, FILE *);
int  get_string(char *, int);
kaddr_t symaddr_per_cpu_2_6(const char * symname, int cpu);
int  task_has_cpu(uint64_t, void*); 
int  task_cpu(void*);

char *get_utsname(void);
void symbol_banner(
	FILE *		/* output file pointer */,
	int 		/* flags */);

void print_symbol(
	kaddr_t 	/* PC */,
	syment_t *	/* syment_t pointer */,
	int 		/* flags */, 
	FILE *		/* output file pointer */);

/* from arch_*.c */
int  dumparch_init(int, FILE *);
void print_kaddr(
	kaddr_t		/* kernel address */,
	FILE *		/* output file pointer */,
	int		/* flag */);

/* from struct.c */
void structlist(FILE*);
int  walk_structs(char*, char*, int, kaddr_t, int, FILE*);

/* from vmdump.c */
void live_vmdump(int, int);

/* from sial.c */
void init_sial(char *, char *);

/* from stabs.c */
void stab_type_banner(FILE*, int);
void ktype_banner(FILE*, int);
void print_stab_type(dbg_type_t*, int, FILE*);
void print_ktype(kltype_t*, int, FILE*);
void walk_ktype(kltype_t*, int, FILE*);
int  list_stab_types(int, FILE*);
int  list_ktypes(int, FILE*);

#endif /* __LCRASH_H */
