/***************************************************************************
                          balls.c  -  description
                             -------------------
    begin                : Sun Sep 9 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lbreakout.h"
#include "list.h"
#include "config.h"
#include "event.h"
#include "difficulty.h"
#include "levels.h"
#include "shrapnells.h"
#include "bricks.h"
#include "shots.h"
#include "paddle.h"
#include "balls.h"
#ifdef SOUND
#include "audio.h"
#endif

#define BALLS_IDLE_LIMIT 10000

#define TOARC(d) (((float)d/180)*M_PI)
#define TODEG(a) (((float)a/M_PI)*180)
#define VLEN(x, y) ( sqrt( (x)*(x) + (y)*(y) ) )
#define REC_SQRT_2 (0.707106781)

extern SDL_Surface *ball_pic; /* ball pictures */
extern SDL_Surface *ball_shadow;
int ball_pic_x_offset = 0; /* display ball at this position */
float ball_vhmask = 0.363970234; /* twenty degrees */
float ball_vvmask = 5.67128182; /* ten degrees */
int ball_rad = 6;
int ball_dia = 12;
int ball_w = 12;
int ball_h = 12;
List *balls = 0;
float ball_vm; /* v max */
float ball_v; /* ball's speed */
float ball_old_v; /* old speed (before extra used) */
float ball_v_add;
int ball_metal = 0; /* ball gets through metal */
int ball_weak = 0; /* 30% that a ball is unable to damage a brick */
int ball_expl = 0; /* like an explosibve brick a ball causes explosion */
float ball_metal_alpha_change = 1.2; /* pulse factor */
float ball_metal_alpha = 0; /* alpha of ball when blue */
int ball_chaos = 0; /* chaotic ball reflection */
extern Paddle paddle;
extern int paddle_resized;
extern SDL_Surface *offscreen;
extern Sdl sdl;
extern Config config;
extern Brick bricks[MAP_WIDTH][MAP_HEIGHT];
extern int keystate[SDLK_LAST];
extern int buttonstate[BUTTON_COUNT];
extern int shadow_size;
#ifdef SOUND
extern Sound_Chunk *wav_reflect;
extern Sound_Chunk *wav_attach;
#endif

extern int active[EX_NUMBER];

extern char circle_msg[256];
extern Font *font;

/*
====================================================================
Locals
====================================================================
*/

/*
====================================================================
Return x squared
====================================================================
*/
float square(float x)
{
    return (x * x);
}
/*
====================================================================
Clear contents of target.
====================================================================
*/
void ball_clear_target( Target *t )
{
    memset(t, 0, sizeof(Target));
    t->side = SIDE_UNDEFINED;
}
/*
====================================================================
Compute motion vector from angle and velocity
====================================================================
*/
void ball_compute_vec(Ball *b, float a, float v)
{
    b->vel.x = sin(TOARC(a)) * v;
    b->vel.y = cos(TOARC(a)) * v;
    /* y-coordinates are upside down */
    b->vel.y = -b->vel.y;
    /* mask angles */
    ball_mask_vel(b, b->vel.x);
}
/*
====================================================================
Attach ball to paddle.
====================================================================
*/
void ball_attach( Ball *ball )
{
    /* relative position */
    ball->attached = 1;
    ball->idle_time = ball->moving_back = ball->return_allowed = 0;
    ball->cur.x -= paddle.x;
    ball->cur.y -= paddle.y;
    ball->x = (int)ball->cur.x;
    ball->y = (int)ball->cur.y;
#ifdef SOUND
    sound_play( wav_attach );
#endif
}
/*
====================================================================
Reflect ball at brick assume normed perp_vector.
====================================================================
*/
void ball_check_brick_reflection( Ball *b )
{
    float old_vx;
    Vector n;
    int reflect;
    int chaos_reflect;
    
	/* time left? */
    if (b->t.cur_tm < b->t.time) return;

    /* if the brick is destructible (thus it'll take damage)
     * we must reset the idle time
     */
    if ( bricks[b->t.mx][b->t.my].dur != -1 || ( ball_metal && bricks[b->t.mx][b->t.my].type != MAP_WALL ) )
        b->idle_time = 0;
    /* or if it is within the last four rows and no chaotic malus is active it will
       hit the paddle soon so reset here too */
    if ( b->t.my >= MAP_HEIGHT - 4 && b->vel.y > 0 && !active[EX_CHAOS] )
        b->idle_time = 0;
    
    /* will reflect? */
    reflect = 1;
    if ( ball_metal && bricks[b->t.mx][b->t.my].type != MAP_WALL )
        reflect = 0;
    
    /* will reflect chaotic? */
    chaos_reflect = 0;
    if ( active[EX_CHAOS] || bricks[b->t.mx][b->t.my].type == MAP_BRICK_CHAOS )
        chaos_reflect = 1;
    
    /* remove brick -- if weak ball there is a 40% chance that no damage is done to the brick */
    if ( !ball_weak || rand() % 10 < 6 ) {
        /* if explosive ball und brick is destructible by normal means set as explosive */
        if ( ball_expl && bricks[b->t.mx][b->t.my].dur > 0 ) {
            bricks[b->t.mx][b->t.my].type = MAP_BRICK_EXP;
            bricks[b->t.mx][b->t.my].dur = 1;
        }
        /* hit brick */
        if ( ball_metal )
            brick_hit( b->t.mx, b->t.my, ball_metal, SHR_BY_ENERGY_BALL, b->vel );
        else
            brick_hit( b->t.mx, b->t.my, ball_metal, SHR_BY_NORMAL_BALL, b->vel );
    }
//    if (b->t2.exists) brick_hit( b->t2.mx, b->t2.my, ball_metal, SHR_HIGH_RES );

	/* we have a target and so we have a reset position and even if the ball's
	   not reflected the position must be reset */
    b->cur.x = b->t.x; b->x = (int)b->cur.x;
    b->cur.y = b->t.y; b->y = (int)b->cur.y;
	
    if ( reflect ) {
#ifdef SOUND
        sound_play( wav_reflect );
#endif
        old_vx = b->vel.x;
		if ( !chaos_reflect ) {
        	/* normal reflection */
	        n.x = (1 - 2 * b->t.perp_vector.x * b->t.perp_vector.x) * b->vel.x + (  - 2 * b->t.perp_vector.x * b->t.perp_vector.y) * b->vel.y;
    	    n.y = (  - 2 * b->t.perp_vector.x * b->t.perp_vector.y) * b->vel.x + (1 - 2 * b->t.perp_vector.y * b->t.perp_vector.y) * b->vel.y;
        	b->vel.x = n.x;
        	b->vel.y = n.y;
		}
		else {
			b->vel.x = ((float)RANDOM( -10000, 10000 )) / 10000;
			b->vel.y = (float)(RANDOM( -10000, 10000 )) / 10000;
	        ball_mask_vel( b, old_vx );
		}	
    }
    /* mark target as disabled so it won't get stuck at the bottom of the screen but keep the target position so we know that we need
       an update. */
    b->t.exists = 0;
    /* check targets */
   	balls_get_new_targets( b->t.mx, b->t.my );
   	shots_get_new_targets( b->t.mx, b->t.my );
    /*if ( b->t2.exists ) {
        balls_get_new_targets(b->t2.mx, b->t2.my);
        shots_get_new_targets(b->t2.mx, b->t2.my);
    }*/
}
/*
====================================================================
Handle ball's contact with paddle: reflect at perpendicular (normed)
or attach.
====================================================================
*/
void ball_handle_paddle_contact( Ball *ball, Vector perp_vector )
{
    float  old_vx = ball->vel.x;
    Vector c; /* A(perp_vector) = c; */

    /* attach ball or reflect */
    if ( paddle_slimy() ) {
        ball_attach( ball );
    }
    else {
        /* assume that perpendicular is normed */
        /* reflect */
        if ( perp_vector.x || perp_vector.y ) {
            /* a simple 2x2 matrix does this for us */
            c.x = (1 - 2 * perp_vector.x * perp_vector.x) * ball->vel.x + (-2 * perp_vector.x * perp_vector.y) * ball->vel.y;
            c.y = (-2 * perp_vector.x * perp_vector.y) * ball->vel.x + (1 - 2 * perp_vector.y * perp_vector.y) * ball->vel.y;
            ball->vel.x = c.x; ball->vel.y = c.y;
#ifdef PADDLE_FRICTION
            /* transfer friction to ball's velocity if not convex */
            if ( !config.convex )
                ball->vel.x += paddle.v_x * paddle.friction;
#endif
            /* adjust velocity */
            ball_mask_vel( ball, old_vx );
            /* or reset position if in wall */
            if ( ball->x < BRICK_WIDTH ) {
                ball->cur.x = BRICK_WIDTH;
                ball->x = (int)ball->cur.x;
            }
            else
                if ( ball->x + ball_dia >= sdl.screen->w - BRICK_WIDTH ) {
                    ball->cur.x = sdl.screen->w - BRICK_WIDTH - ball_dia;
                    ball->x = (int)ball->cur.x;
                }
            /* get new target */
            ball_get_target( ball );
#ifdef SOUND
            sound_play( wav_reflect );
#endif
        }
		else
			fprintf( stderr, "ERROR: Can't reflect at perpendicular (0,0)!\n" );
    }
}
/*
====================================================================
Sorry.
Berechnung der Schnittpunkte der Geraden, die orthogonal zur
Geraden der Ballgeschwindigkeit durch den Ballmittelpunkt verluft,
also der tangentialen Geschwindigkeitspunkte.
Der Geschwindigkeitsvektor wird auf 1 genormt. Ausgehend vom
Mittelpunkt wird der dazu orthogonale Vektor jeweils mit gendertem
Vorzeichen addiert und ergibt so die Tangentialpunkte.

If you're able and willing to translate this please send me your
result. ;-)
====================================================================
*/
void ball_get_tangents( Ball *ball, Coord *left, Coord *right )
{
    Vector norm_vel = ball->vel;
    int center_x = ball->cur.x + ball_rad, center_y = ball->cur.y + ball_rad;
    vector_norm( &norm_vel );
    left->x = center_x + norm_vel.y * ball_rad;
    left->y = center_y - norm_vel.x * ball_rad;
    right->x = center_x - norm_vel.y * ball_rad;
    right->y = center_y + norm_vel.x * ball_rad;
}
/*
====================================================================
Check if the ball is on paddle's level and an reflect is
possible (ball moves down).
====================================================================
*/
int ball_paddle_contact_possible( Ball *ball, Vector old )
{
	if ( !paddle_solid() ) return 0;
	if ( ball->vel.y < 0 ) return 0; /* ball moves up so no contact possible because if it was below the paddle it has been 
										reflected by the bonus floor and MUST ignore the paddle */
    if ( ball->y + ball_dia < paddle.y ) return 0; /* above paddle */
    if ( ball->y >= paddle.y + ( paddle.h >> 1 ) )
        if ( old.y >= paddle.y + ( paddle.h >> 1 ) ) return 0; /* already behind paddle */
	return 1;			
}
/*
====================================================================
Check reflection of ball at paddle. 'old' is the position of
the ball before update. Used to compute direction.
====================================================================
*/
enum { CONTACT_LEFT = 1, CONTACT_MIDDLE, CONTACT_RIGHT };
void ball_check_paddle_reflection( Ball *ball )
{
    Line    ball_line; /* balls velocity line */
    Line    paddle_line; /* paddle line */
    Coord   pt, pt2; /* auxiliary point (result of intersection) */
    int     contact = 0; /* paddle contact */
    Vector  perp_vector; /* perpendicular of ball's direction change */
    Coord   center = { ball->cur.x + ball_rad, ball->cur.y + ball_rad }; /* center of ball */
    Vector  norm_vel; /* normed ball velocity vector */
    /* paddle is constructed as two hemispheres at the side and a cylinder in the middle */
    Coord   right_hemi_center = { paddle.x + paddle.w - ( paddle.h >> 1 ), paddle.y + ( paddle.h >> 1 ) };
    Coord   left_hemi_center = { paddle.x + ( paddle.h >> 1 ), paddle.y + ( paddle.h >> 1 ) };
    /* radius of hemispheres */
    int     hemi_r = ( ball_rad ) + ( paddle.h >> 1 );
    /* if paddle's treated as convex these are the perpendiculars through the hemisphere centers */
    Vector  left_convex_perp = { 1, 1 };
    /* paddle center */
    Coord   paddle_center = { paddle.x + ( paddle.w >> 1 ), paddle.y + ( paddle.h >> 1 ) };
    /*  center of the convex behaviour -- computed when reflecting by using
        left/right_convex_perp and paddle_center */
    Coord   convex_center;
    /* perpendicular line used for convex behaviour */
    Line    convex_line;

	/* the simple check for the y-position of ball and paddle is done in ball_paddle_contact_possible() */		
	/* so if we got here it's possible by velocity and position of ball that it hits the paddle */

    /*  basic idea:
        The paddle is constructed of a middle rectangle and two hemispheres.
        We check the center line of the ball with the imaginary paddle that's size
        is paddle_size + ball_rad. The intersection with this paddle is the reset point
        for the ball at the same time (if slimy).
        The perpendicular is computed as convex thing. (overwrites the perpendicular
        set by the reflection)
    */
    /* ball line */
    line_set( &ball_line, center.x, center.y, ball->vel.y / ball->vel.x );
    /* imaginary paddle upper line -- we'll decide at intersection which hemipshere to check */
    line_set_hori( &paddle_line, paddle.y - ball_rad );
    line_intersect( &paddle_line, &ball_line, &pt );
    if ( pt.x < left_hemi_center.x ) {
        /* intersect left */
        norm_vel = ball->vel; vector_norm( &norm_vel );
        if ( circle_intersect( left_hemi_center, hemi_r,
                               center, norm_vel,
                               &pt, &pt2 ) ) {
            if ( VEC_DIST( center, left_hemi_center ) <= hemi_r ) {
                /* use the higher point as this is the upper intersection */
                if ( pt.y > pt2.y ) pt = pt2;
                /* use vector between hemi_sphere center and ball center as reflection */
                perp_vector = vector_get( center.x - left_hemi_center.x,
                                          center.y - left_hemi_center.y );
                vector_norm( &perp_vector );
                /* had contact */
                contact = CONTACT_LEFT;
            }
        }
    }
    else
        if ( pt.x > right_hemi_center.x ) {
            /* intersect right */
            norm_vel = ball->vel; vector_norm( &norm_vel );
            if ( circle_intersect( right_hemi_center, hemi_r,
                                   center, norm_vel,
                                   &pt, &pt2 ) ) {
                if ( VEC_DIST( center, right_hemi_center ) <= hemi_r ) {
                    /* use the higher point as this is the upper intersection */
                    if ( pt.y > pt2.y ) pt = pt2;
                    /* use vector between hemi_sphere center and ball center as reflection */
                    perp_vector = vector_get( center.x - right_hemi_center.x,
                                              center.y - right_hemi_center.y );
                    vector_norm( &perp_vector );
                    /* had contact */
                    contact = CONTACT_RIGHT;
                }
            }
        }
        else {
            contact = CONTACT_MIDDLE; /* contact with middle part */
            perp_vector = vector_get( 0, 1 ); /* reflect at horizontal line */
        }
    /* if we got here 'pt' contains the intersection with the imaginary paddle so reset ball
    to this position */
    if ( contact ) {
        /* reset idle time */
        ball->idle_time = 0;
        /* reset position if ball will be attached */
        if ( paddle_slimy() ) {
            ball->cur.x = pt.x - ( ball_rad );
            ball->cur.y = pt.y - ( ball_rad );
            ball->x = (int)ball->cur.x; ball->y = (int)ball->cur.y;
        }
        /* convex perpendicular */
        if ( config.convex ) {
            line_set_vert( &paddle_line, paddle_center.x );
            line_set( &convex_line, left_hemi_center.x, left_hemi_center.y,
                      vector_monotony( left_convex_perp ) );
            line_intersect( &paddle_line, &convex_line, &convex_center );
            /* get actual perp_vector */
            perp_vector.x = pt.x - convex_center.x;
            perp_vector.y = pt.y - convex_center.y;
        }
        /* handle contact: attach, reflect, sound... */
        ball_handle_paddle_contact( ball, perp_vector );
    }
}
/*
====================================================================
Intersect ball line with imaginary brick line.
Use target's map position and 
set reset position (centered) and perp_vector of target.
ball_rad is substracted later in ball_get_target()
====================================================================
*/
enum { LINE_HORI = 0, LINE_VERT };
int check_line( Line *ball_line, int type, int anchor, int range_start, int range_end, Coord *pt ) {
    Line line;
    if ( type == LINE_HORI )
        line_set_hori( &line, anchor );
    else
        line_set_vert( &line, anchor );
    line_intersect( &line, ball_line, pt );
    if ( type == LINE_HORI ) {
        if ( pt->x >= range_start && pt->x <= range_end ) return 1;
        return 0;
    }
    else {
        if ( pt->y >= range_start && pt->y <= range_end ) return 1;
        return 0;
    }
}
void ball_intersect_brick( Ball *ball, Target *target )
{
    Line  ball_line;
    Coord pt; /* auxiliary point */
    int   x = target->mx * BRICK_WIDTH, y = target->my * BRICK_HEIGHT; /* left upper corner of brick */
    int   intersect = 0; /* intersected? */

    /* ball_line */
    line_set( &ball_line, ball->cur.x + ball_rad, ball->cur.y + ball_rad, ball->vel.y / ball->vel.x );

    if ( ball->vel.x > 0 ) {
        /* left */
        if ( check_line( &ball_line,
                         LINE_VERT,
                         x - ball_rad,
                         y - ball_rad, y + BRICK_HEIGHT + ball_rad,
                         &pt ) ) {
            intersect = 1;
            target->perp_vector = vector_get( 1, 0 );
        }
    }
    else {
        /* right */
        if ( check_line( &ball_line,
                         LINE_VERT,
                         x + BRICK_WIDTH + ball_rad,
                         y - ball_rad, y + BRICK_HEIGHT + ball_rad,
                         &pt ) ) {
            intersect = 1;
            target->perp_vector = vector_get( 1, 0 );
        }
    }
    if ( !intersect ) {
    if ( ball->vel.y > 0 ) {
        /* top */
        if ( check_line( &ball_line,
                         LINE_HORI,
                         y - ball_rad,
                         x - ball_rad, x + BRICK_WIDTH + ball_rad,
                         &pt ) ) {
            intersect = 1;
            target->perp_vector = vector_get( 0, 1 );
        }
    }
    else {
        /* bottom */
        if ( check_line( &ball_line,
                         LINE_HORI,
                         y + BRICK_HEIGHT + ball_rad,
                         x - ball_rad, x + BRICK_WIDTH + ball_rad,
                         &pt ) ) {
            intersect = 1;
            target->perp_vector = vector_get( 0, 1 );
        }
    }
    }
    /* intersected */
    if ( intersect ) {
        target->x = pt.x;
        target->y = pt.y;
        /* perp_vector is set */
    }
}
/*
====================================================================
Reflect ball at target at target->side and set perp_vector
and reset position x,y.
====================================================================
*/
void ball_reflect_at_side( Ball *ball, Target *target )
{
	Line  ball_line;
	Line  brick_line;
	Coord pt;
	/* ball line */
    line_set( &ball_line, ball->cur.x + ball_rad, ball->cur.y + ball_rad, ball->vel.y / ball->vel.x );
	/* brick line and perp vector */
	switch ( target->side ) {
		case SIDE_LEFT:
			line_set_vert( &brick_line, target->mx * BRICK_WIDTH - ball_rad );
			target->perp_vector = vector_get( 1, 0 );
			break;
		case SIDE_RIGHT:
			line_set_vert( &brick_line, target->mx * BRICK_WIDTH + BRICK_WIDTH + ball_rad );
			target->perp_vector = vector_get( 1, 0 );
			break;
		case SIDE_TOP:
			line_set_hori( &brick_line, target->my * BRICK_HEIGHT - ball_rad );
			target->perp_vector = vector_get( 0, 1 );
			break;
		case SIDE_BOTTOM:
			line_set_hori( &brick_line, target->my * BRICK_HEIGHT + BRICK_HEIGHT + ball_rad );
			target->perp_vector = vector_get( 0, 1 );
			break;
		default:
			fprintf( stderr, "Unknown side: %i\n", target->side );
			break;
	}
	/* intersect, it's already assured that we hit this brick so just get the reset position */
	line_intersect( &brick_line, &ball_line, &pt );
	target->x = pt.x;
	target->y = pt.y;
}
/*
====================================================================
Reflect ball at target but ignore target::side and reflect at 
corner instead.
====================================================================
*/
void ball_reflect_at_corner( Ball *ball, Target *target, int corner )
{
	Coord corner_center; /* center of corner circle */
	Coord ball_center = { ball->cur.x + ball_rad, ball->cur.y + ball_rad };
	Vector norm_vel = ball->vel;
	Coord pt, pt2; /* intersection points */
	/* norm velocity */
	vector_norm( &norm_vel );
	/* set up center of corner */
	switch ( corner ) {
		case CORNER_UPPER_LEFT:
			corner_center = vector_get( target->mx * BRICK_WIDTH, target->my * BRICK_HEIGHT );
			break;
		case CORNER_UPPER_RIGHT:
			corner_center = vector_get( target->mx * BRICK_WIDTH + BRICK_WIDTH - 1, target->my * BRICK_HEIGHT );
			break;
		case CORNER_LOWER_LEFT:
			corner_center = vector_get( target->mx * BRICK_WIDTH, target->my * BRICK_HEIGHT + BRICK_HEIGHT - 1);
			break;
		case CORNER_LOWER_RIGHT:
			corner_center = vector_get( target->mx * BRICK_WIDTH + BRICK_WIDTH - 1, target->my * BRICK_HEIGHT + BRICK_HEIGHT - 1);
			break;
	}
	/* intersect */
	circle_intersect( corner_center, ball_rad + 2, ball_center, norm_vel, &pt, &pt2 );
	/* use nearest point for reset and perp vector */
	if ( VEC_DIST( ball_center, pt ) < VEC_DIST( ball_center, pt2 ) ) {
		target->x = pt.x;
		target->y = pt.y;
	}
	else {
		target->x = pt2.x;
		target->y = pt2.y;
	}
	/* don't use the real perp vector but assume a 45 degree angle */
	switch ( corner ) {
		case CORNER_UPPER_LEFT:
		case CORNER_LOWER_RIGHT:
			target->perp_vector = vector_get( REC_SQRT_2, REC_SQRT_2 );
			break;
		case CORNER_UPPER_RIGHT:
		case CORNER_LOWER_LEFT:
			target->perp_vector = vector_get( REC_SQRT_2, -REC_SQRT_2 );
			break;
	}
}
/*
====================================================================
Reflect ball at target ball:t and decide by ball::t::side wether
to use reflect_at_side or reflect_at_corner.
====================================================================
*/
void ball_reflect( Ball *ball )
{
	if ( !ball->t.exists ) return;
	if ( ball->t.side <= SIDE_LEFT ) 
		ball_reflect_at_side( ball, &ball->t );
	else
		ball_reflect_at_corner( ball, &ball->t, ball->t.side );
}

/*
====================================================================
Check if ball hit a corner and update target's side.
====================================================================
*/
void corner_check( Ball *ball, Target *target_left_tang, Target *target_right_tang, Target *target ) {
	/* balls moving ... */
	if ( ball->vel.y > 0 ) {
		if ( ball->vel.x < 0 ) {
			/* ... down left */
			if ( target == target_right_tang && target->side == SIDE_TOP   ) target->side = CORNER_UPPER_RIGHT;
			if ( target == target_left_tang  && target->side == SIDE_RIGHT ) target->side = CORNER_UPPER_RIGHT;
		}	
		else {
			/* ... down right */
			if ( target == target_left_tang  && target->side == SIDE_TOP   ) target->side = CORNER_UPPER_LEFT;
			if ( target == target_right_tang && target->side == SIDE_LEFT  ) target->side = CORNER_UPPER_LEFT;
		}
	}
	else {
		if ( ball->vel.x < 0 ) {
			/* ... up left */
			if ( target == target_right_tang && target->side == SIDE_RIGHT  ) target->side = CORNER_LOWER_RIGHT;
			if ( target == target_left_tang  && target->side == SIDE_BOTTOM ) target->side = CORNER_LOWER_RIGHT;
		}
		else {
			/* ... up right */
			if ( target == target_left_tang  && target->side == SIDE_LEFT   ) target->side = CORNER_LOWER_LEFT;
			if ( target == target_right_tang && target->side == SIDE_BOTTOM ) target->side = CORNER_LOWER_LEFT;
		}
	}
}

/*
====================================================================
Public
====================================================================
*/

/*
====================================================================
Load/delete ball graphics
====================================================================
*/
void ball_load()
{
    balls = list_create( LIST_AUTO_DELETE, NO_CALLBACK );
}
void ball_delete()
{
    if ( balls ) list_delete( balls ); balls = 0;
}
/*
====================================================================
Create ball at position with starting angle and velocity
====================================================================
*/
Ball* ball_create( int x, int y, float a, float v )
{
    Ball *b = calloc( 1, sizeof( Ball ) );
    b->cur.x = x;
    b->x = x;
    b->cur.y = y;
    b->y = y;
    b->attached = 0;
    b->idle_time = 0;
    b->moving_back = 0;
    b->return_allowed = 0;
    ball_compute_vec(b, a, v);
    ball_clear_target(&b->t);
    ball_clear_target(&b->t2);
    return b;
}
/*
====================================================================
Set a special ball property like metal ball.
====================================================================
*/
void balls_set_type( int type )
{
    ball_metal = 0;
    ball_expl  = 0;
    ball_weak  = 0;
    switch ( type ) {
        case BALL_NORMAL:
            ball_pic_x_offset = 0;
            break;
        case BALL_METAL:
            ball_metal = 1;
            ball_pic_x_offset = ball_w;
            ball_metal_alpha = 0;
            if ( ball_metal_alpha_change < 0 ) 
                ball_metal_alpha_change = -ball_metal_alpha_change;
            break;
       case BALL_EXPL:
           ball_expl = 1;
           ball_pic_x_offset = ball_w + ball_w;
           break;
       case BALL_WEAK:
           ball_weak = 1;
           ball_pic_x_offset = ball_w + ball_w + ball_w;
           break;
	}		
}
/*
====================================================================
Set chaotic behaviour (random relfection)
====================================================================
*/
void balls_set_chaos( int chaos )
{
    ball_chaos = chaos;
}
/*
====================================================================
Clear ball list and attach one ball to paddle
====================================================================
*/
void balls_reset()
{
    Ball *ball;
    list_clear( balls );
    ball = ball_create((paddle.w - ball_w) / 2, -ball_dia, (rand() % 120) - 60, ball_v);
    ball->attached = 1;
    list_add( balls, ball );
    balls_set_type( BALL_NORMAL );
}
/*
====================================================================
Initate velocity values (for all balls one value )
====================================================================
*/
void balls_set_vel( float v_start, float v_change, float v_max )
{
    ball_v = v_start;
    ball_vm = v_max;
    ball_v_add = v_change;
}
/*
====================================================================
Show/hide all balls
====================================================================
*/
void balls_hide()
{
    List_Entry *entry = balls->head.next;
    Ball *b;
    int bx, by;
    int w = ball_w, h = ball_h;
    if ( config.shadow ) { w += shadow_size; h += shadow_size; }
    while ( entry != &balls->tail ) {
        b = entry->item;
        /* balls position; add paddle pos if attached */
        bx = b->x;
        by = b->y;
        if (b->attached) {
            bx += paddle.x;
            by += paddle.y;
        }
        /* put background */
        DEST(sdl.screen, bx, by, w, h);
        SOURCE(offscreen, bx, by);
        blit_surf();
        add_refresh_rect(bx, by, w, h );
        entry = entry->next;
    }
}
void balls_show()
{
    List_Entry *entry = balls->head.next;
    Ball *b;
    int bx, by;
    int mx, my;
    while ( entry != &balls->tail ) {
        b = entry->item;
        /* balls position; add paddle pos if attached */
        bx = b->x;
        by = b->y;
        if (b->attached) {
            bx += paddle.x;
            by += paddle.y;
        }
        /* show ball -- no shadow if darkness -- no shadow if going back home */
        if ( config.shadow && !active[EX_DARKNESS] && !b->moving_back ) {
            set_surf_clip( sdl.screen, 0, 0, sdl.screen->w - BRICK_WIDTH, sdl.screen->h );
            DEST(sdl.screen, bx + shadow_size, by + shadow_size, ball_w, ball_h);
            SOURCE(ball_shadow, 0, 0);
			if ( ball_metal )
				alpha_blit_surf( ball_metal_alpha );
			else
            	alpha_blit_surf( SHADOW_ALPHA );
            /* redraw nearby bricks */
            set_surf_clip( sdl.screen, bx + shadow_size, by + shadow_size, ball_w, ball_h );
            /* check the three outer ocrners of the shadow if there's a brick */
            mx = ( bx + shadow_size + ball_w ) / BRICK_WIDTH;
            my = ( by + shadow_size ) / BRICK_HEIGHT;
            if ( my < MAP_HEIGHT - 1 )
                if ( mx < MAP_WIDTH - 1 && bricks[mx][my].type != MAP_EMPTY )
                    brick_draw( sdl.screen, mx, my, 0 );
            mx = ( bx + shadow_size + ball_w ) / BRICK_WIDTH;
            my = ( by + shadow_size + ball_h ) / BRICK_HEIGHT;
            if ( my < MAP_HEIGHT - 1 )
                if ( mx < MAP_WIDTH - 1 && bricks[mx][my].type != MAP_EMPTY )
                    brick_draw( sdl.screen, mx, my, 0 );
            mx = ( bx + shadow_size ) / BRICK_WIDTH;
            my = ( by + shadow_size + ball_h ) / BRICK_HEIGHT;
            if ( my < MAP_HEIGHT - 1 )
                if ( mx < MAP_WIDTH - 1 && bricks[mx][my].type != MAP_EMPTY )
                    brick_draw( sdl.screen, mx, my, 0 );
            set_surf_clip( sdl.screen, 0, 0, 0, 0 );
        }
        DEST(sdl.screen, bx, by, ball_w, ball_h);
        SOURCE(ball_pic, ball_pic_x_offset, 0);
		if ( ball_metal ) {
			alpha_blit_surf( ball_metal_alpha );
		}
		else {
			if ( !active[EX_DARKNESS] ) {
                if ( b->moving_back )
                    alpha_blit_surf( 128 );
                else
                    blit_surf();
            }
			else
				alpha_blit_surf( 128 );
		}			
        if ( config.shadow )
            add_refresh_rect(bx, by, ball_w + shadow_size, ball_h + shadow_size);
        else
            add_refresh_rect(bx, by, ball_w, ball_h);
        entry = entry->next;
    }
}
void balls_alphashow( int alpha )
{
    List_Entry *entry = balls->head.next;
    Ball *b;
    int bx, by;
    while ( entry != &balls->tail ) {
        b = entry->item;
        /* balls position; add paddle pos if attached */
        bx = b->x;
        by = b->y;
        if (b->attached) {
            bx += paddle.x;
            by += paddle.y;
        }
        /* show ball */
        DEST(sdl.screen, bx, by, ball_w, ball_h);
        SOURCE(ball_pic, ball_pic_x_offset, 0);
        alpha_blit_surf( alpha );
        add_refresh_rect(bx, by, ball_w, ball_h);
        entry = entry->next;
    }
}
/*
====================================================================
Update balls and detach attached balls if fire was pressed.
Return False if all balls are lost.
====================================================================
*/
int balls_update( int ms )
{
    List_Entry *entry = balls->head.next;
    Ball *b;
    Vector old; /* old position of ball before update */
	/* modify alpha when metal */
	if ( ball_metal && config.trp ) {
		ball_metal_alpha += ball_metal_alpha_change * ms;
		if ( ball_metal_alpha >= 255 || ball_metal_alpha <= 0 ) {
			ball_metal_alpha_change = -ball_metal_alpha_change;
			if ( ball_metal_alpha < 0 ) ball_metal_alpha = 0;
			if ( ball_metal_alpha > 255 ) ball_metal_alpha = 255;
		}	
	}
    while ( entry != &balls->tail ) {
        b = entry->item;
        old.x = b->cur.x;
        old.y = b->cur.y;
		/* adjust position if paddle_resize and attached */
		if ( b->attached && paddle_resized ) {
			if ( paddle_resized < 0 ) {
				/* shrinked */
				if ( b->cur.x > ( paddle.w >> 1 ) ) {
					b->cur.x -= 2;
					b->x = (int)b->cur.x;
				}	
			}
			else {
				/* expanded */
				if ( b->cur.x > ( paddle.w >> 1 ) ) {
					b->cur.x += 2;
					b->x = (int)b->cur.x;
				}	
			}
		}
		/* fire balls */
        if (b->attached && (buttonstate[LEFT_BUTTON] || buttonstate[RIGHT_BUTTON]|| keystate[config.k_fire])) {
            /* if not in wall */
            if (b->x + paddle.x >= BRICK_WIDTH && b->x + ball_dia + paddle.x < sdl.screen->w - BRICK_WIDTH) {
                /* release ball */
#ifdef SOUND
#endif
                b->attached = 0;
                b->moving_back = b->idle_time = b->return_allowed = 0;
                b->x += paddle.x;
                b->y += paddle.y;
                b->cur.x = b->x;
                b->cur.y = b->y;
                if (buttonstate[LEFT_BUTTON])
                    ball_compute_vec(b, -50, ball_v);
                else
                    ball_compute_vec(b, 50, ball_v);
                ball_get_target(b);
            }
        }
        /* update ball when moving back */
        if ( b->moving_back ) {
            /* update velocity */
            b->vel.x = ( paddle.x + ( paddle.w >> 1 ) ) - ( b->cur.x + ball_rad );
            b->vel.y = ( paddle.y - ball_rad + 2 ) - ( b->cur.y + ball_rad );
            ball_adjust_vel( b, ball_vm );
            /* new position */
            b->cur.x += b->vel.x * ms;
            b->cur.y += b->vel.y * ms;
            b->x = (int)b->cur.x;
            b->y = (int)b->cur.y;
            /* check if paddle is reached and attach the ball */
            if ( b->x + ball_rad >= paddle.x && b->x + ball_rad < paddle.x + paddle.w )
                if ( b->y + ball_dia >= paddle.y && b->y + ball_dia < paddle.y + paddle.h ) {
                    b->cur.x = paddle.x + ( paddle.w >> 1 ) - ball_rad;
                    b->cur.y = paddle.y - ball_dia;
                    b->x = (int)b->cur.x;
                    b->y = (int)b->cur.y;
                    ball_attach( b );
                }
        }
        /* update ball if not attached and not moving back */
        if (!b->attached && !b->moving_back) {
            /* increase idle time -- paddle and brick_check will reset this value */
            if ( !b->return_allowed )
                b->idle_time += ms;
            /* new position */
            b->cur.x += b->vel.x * ms;
            b->cur.y += b->vel.y * ms;
            b->x = (int)b->cur.x;
            b->y = (int)b->cur.y;
            /* check if reflected by paddle */
			if ( ball_paddle_contact_possible( b, old ) ) ball_check_paddle_reflection( b );
            /* or by brick */
            /* quick hack to handle the case when the ball was just attached but touches the wall and the 
               slimy paddle in the same instant. - Patrick Hohmeyer 19.12.01 */
            if ( b->t.exists && !b->attached ) {
                b->t.cur_tm += ms;
                ball_check_brick_reflection(b);
            }
            /* check if idle time is above limit and the ball has a target because if there
             * is no target the ball moves out of the window and should not go back to the 
             * paddle as it's moving into this direction by itself
             */
            if ( b->idle_time >= BALLS_IDLE_LIMIT && !b->return_allowed && b->t.exists ) {
                /* okay send this ball back home or allow to do so by click */
                if ( !config.return_on_click ) {
                    b->idle_time = 0;
                    b->moving_back = 1;
                    b->t.exists = 0; /* no target */
                }
                else {
                    b->idle_time = 0;
                    b->return_allowed = 1;
                }
            }
        }
        entry = entry->next;
        /* delete ball if outside of window */
        if (!b->attached && (b->x >= sdl.screen->w || b->x + ball_dia < 0 || b->y >= sdl.screen->h))
            list_delete_entry( balls, entry->prev );
    }
	/* clear paddle_resize flag */
	paddle_resized = 0;
    /* all balls lost?*/
    return !list_empty( balls );
}
/*
====================================================================
Compute new targets for all balls (as bricks where destroyed
meanwhile by other balls or shots)
====================================================================
*/
void balls_get_new_targets( int mx, int my )
{
    Ball *ball;
    int reset = 0;
    list_reset( balls );
    while ( ( ball = list_next( balls ) ) ) 
        if ( !ball->attached && !ball->moving_back )
        if ( mx == -1 || ( ball->t.mx == mx && ball->t.my == my ) ) {
            /*
             * As we don't have a constant velocity but assume one it is possible that
             * the ball is within a wall when this function is called because it actually
             * passed it's reset position without time expiration because of the velocity
             * change. So we have to check here if it is already behind this position and if so
             * simply reset here. This doesn't hurt as this would happen before reflection, too.
             */
            if ( ball->t.exists ) {
                if ( ball->vel.y > 0 ) {
                    if ( ball->cur.y > ball->t.y ) 
                        reset = 1;
                }
                else {
                    if ( ball->cur.y < ball->t.y ) 
                        reset = 1;
                }
                if ( ball->vel.x > 0 ) {
                    if ( ball->cur.x > ball->t.x ) 
                        reset = 1;
                }
                else {
                    if ( ball->cur.x < ball->t.x ) 
                        reset = 1;
                }
                if ( reset ) {
                    ball->cur.x = ball->t.x;
                    ball->cur.y = ball->t.y;
                    ball->x = (int)ball->cur.x;
                    ball->y = (int)ball->cur.y;
                }
            }
            /* get target */
            ball_get_target(ball);
        }
}
/*
====================================================================
Adjust ball's velocity to passed velocity
====================================================================
*/
void ball_adjust_vel(Ball *b, float v)
{
    float d = sqrt(b->vel.x * b->vel.x + b->vel.y * b->vel.y);
    b->vel.x /= d; b->vel.y /= d;
    b->vel.x *= v; b->vel.y *= v;
}
/*
====================================================================
Adjust velocity of ball to spare out any illegal values
====================================================================
*/
void ball_mask_vel(Ball *b, float old_vx)
{
    float m;

    /* b->vel.x == 0 would cause seg faults */
    if (b->vel.x == 0) {
        if (old_vx < 0)
            b->vel.x = 0.01;
        else
            b->vel.x = -0.01;
    }
    /* avoid 45 angles */
    if (b->vel.x == b->vel.y)
        b->vel.x *= 0.99;

    m = b->vel.y / b->vel.x;

    /* mask angles from 70 to 110 and -110 to -70 */
    if (fabs(m) < ball_vhmask) {
        if (b->vel.y < 0)
            b->vel.y = fabs(ball_vhmask * b->vel.x) * -1;
        else
            b->vel.y = fabs(ball_vhmask * b->vel.x);
    }
    /* mask angles from -10 to 10 and 170 to 190 */
    if (fabs(m) > ball_vvmask) {
        if (b->vel.x < 0)
            b->vel.x = fabs(b->vel.y / ball_vvmask) * -1;
        else
            b->vel.x = fabs(b->vel.y / ball_vvmask);
    }
    /* adjust speed */
    ball_adjust_vel(b, ball_v);
}
/*
====================================================================
Get target for a ball.
====================================================================
*/
enum { TANG_LEFT = 0, TANG_RIGHT };
enum { DIR_UP = 0, DIR_DOWN, DIR_LEFT, DIR_RIGHT };
void ball_get_target( Ball *ball )
{
    int    i;
    int    cur_tang;
    float  mono; /* monotony */
    Coord  tang_pts[2]; /* tangential points */
    Line   tang; /* current tangent */
    Coord  center = { ball->cur.x + ball_rad, ball->cur.y + ball_rad }; /* ball center */
    int    start, end, dir, line_pos, change; /* used to intersect the brick grid */
    Line   cur_line; /* dito */
    Coord  pt; /* auxiliary point. used for this 'n' that */
    Target targets[2]; /* targets hit by the tangents: nearest is the actual target */
    Target hori_target[2], vert_target[2]; /* used to get target of tangent */
    float  dist; /* distance between two points */
    Vector norm_vel; /* normed ball velocity */
	int	   corner = -1; /* id of corner sued for corner reflection */
#ifdef WITH_BUG_REPORT
	char	tang_target_chosen_str[2][128]; /* either hori or vert target chosen */
	char	side_str[128];
#endif
	Target	*prim, *sec; /* primary, secondary target */
	int     maybe_corner;
	
#ifdef WITH_BUG_REPORT
	side_str[0] = 0;
#endif
	
    /* balls moving back to paddle must not be reflected */
    if ( ball->moving_back ) return;
	/* attached balls MUST NOT be reflected!!!! */
	if ( ball->attached ) return;
	/* balls already out of the screen though still visible don't need new reflection, too */
	if ( ball->cur.y + ball_dia >= sdl.screen->h - 1 ) return;
	
    /* clear ball targets */
    ball_clear_target( &ball->t );
    ball_clear_target( &ball->t2 );
    ball_clear_target( &targets[TANG_LEFT] );
    ball_clear_target( &targets[TANG_RIGHT] );
    /* monotony */
    mono = ball->vel.y / ball->vel.x;
    /* normed velocity */
    norm_vel = ball->vel; vector_norm( &norm_vel );
    /* tangential points */
    ball_get_tangents( ball, &tang_pts[TANG_LEFT], &tang_pts[TANG_RIGHT] );
    /* get all map bricks the tangents intersect and check target */
    for ( cur_tang = 0; cur_tang < 2; cur_tang++ ) {
        /* clear targets */
        ball_clear_target( &hori_target[cur_tang] );
        ball_clear_target( &vert_target[cur_tang] );
        /* current tangent */
        line_set( &tang, tang_pts[cur_tang].x, tang_pts[cur_tang].y, mono );
        /* intersect horizontal lines */
        /* get direction */
        dir = DIR_DOWN;
        if ( ball->vel.y < 0 ) dir = DIR_UP;
        /* get starting line */
        start = ((int)( tang_pts[cur_tang].y / BRICK_HEIGHT )) * BRICK_HEIGHT;
        /* get end line */
        if ( dir == DIR_UP )
            end = 0;
        else
            end = ( MAP_HEIGHT - 1 ) * BRICK_HEIGHT;
        /* adjust lines if ball moves up */
        if ( dir == DIR_UP ) {
            start += BRICK_HEIGHT - 1;
            end += BRICK_HEIGHT - 1;
        }
        /* get position change */
        change = BRICK_HEIGHT;
        if ( dir == DIR_UP ) change = -change;
        /* we're at this brick so we can't reflect here */
        start += change;
        /* intersect */
        line_pos = start;
        /*  end specifies the last line to be checked to we have to add
            another line to state the break condition.
            this last line is not checked */
        end += change;
        while ( line_pos != end ) {
            line_set_hori( &cur_line, line_pos );
            if ( line_intersect( &cur_line, &tang, &pt ) && ( pt.x >= 0 && pt.x < sdl.screen->w ) )
                if ( bricks[(int)pt.x / BRICK_WIDTH][(int)pt.y / BRICK_HEIGHT].type != MAP_EMPTY ) {
                    /* we got our horizontal target */
                    hori_target[cur_tang].exists = 1;
                    hori_target[cur_tang].x = pt.x;
                    hori_target[cur_tang].y = pt.y;
                    hori_target[cur_tang].mx = pt.x / BRICK_WIDTH;
                    hori_target[cur_tang].my = pt.y / BRICK_HEIGHT;
                    if ( ball->vel.y < 0 )
                        hori_target[cur_tang].side = SIDE_BOTTOM;
                    else
                        hori_target[cur_tang].side = SIDE_TOP;
                    break; /* we got our target for this tangent */
                }
            line_pos += change;
        }
        /* intersect vertical lines */
        /* get direction */
        dir = DIR_RIGHT;
        if ( ball->vel.x < 0 ) dir = DIR_LEFT;
        /* get starting line */
        start = ((int)( tang_pts[cur_tang].x / BRICK_WIDTH )) * BRICK_WIDTH;
        /* get end line */
        if ( dir == DIR_LEFT )
            end = 0;
        else
            end = ( MAP_WIDTH - 1 ) * BRICK_WIDTH;
        /* adjust lines if ball moves up */
        if ( dir == DIR_LEFT ) {
            start += BRICK_WIDTH - 1;
            end += BRICK_WIDTH - 1;
        }
        /* get position change */
        change = BRICK_WIDTH;
        if ( dir == DIR_LEFT ) change = -change;
        /* we're at this brick so we can't reflect here */
        start += change;
        /* intersect */
        line_pos = start;
        /*  end specifies the last line to be checked too we have to add
            another line to state the break condition.
            this last line is not checked */
        end += change;
        while ( line_pos != end ) {
            line_set_vert( &cur_line, line_pos );
            if ( line_intersect( &cur_line, &tang, &pt ) && ( pt.y >= 0 && pt.y < sdl.screen->h ) )
                if ( bricks[(int)pt.x / BRICK_WIDTH][(int)pt.y / BRICK_HEIGHT].type != MAP_EMPTY ) {
                    /* we got our vertical target */
                    vert_target[cur_tang].exists = 1;
                    vert_target[cur_tang].x = pt.x;
                    vert_target[cur_tang].y = pt.y;
                    vert_target[cur_tang].mx = pt.x / BRICK_WIDTH;
                    vert_target[cur_tang].my = pt.y / BRICK_HEIGHT;
                    if ( ball->vel.x < 0 )
                        vert_target[cur_tang].side = SIDE_RIGHT;
                    else
                        vert_target[cur_tang].side = SIDE_LEFT;
                    break; /* we got our target for this tangent */
                }
            line_pos += change;
        }
        /* get closest target */
        if ( !hori_target[cur_tang].exists ) {
            targets[cur_tang] = vert_target[cur_tang];
#ifdef WITH_BUG_REPORT
			if ( !vert_target[cur_tang].exists )
				sprintf( tang_target_chosen_str[cur_tang], "No target chosen." );
			else
				sprintf( tang_target_chosen_str[cur_tang], "Vertical target chosen." );
#endif
		}	
        else
            if ( !vert_target[cur_tang].exists ) {
                targets[cur_tang] = hori_target[cur_tang];
#ifdef WITH_BUG_REPORT
				sprintf( tang_target_chosen_str[cur_tang], "Horizontal target chosen." );
#endif
			}
            else {
				/* check the relation and choose the correct target */
				/* if vertical and hori hit the same brick we have hit the corner */
				if ( hori_target[cur_tang].mx == vert_target[cur_tang].mx && hori_target[cur_tang].my == vert_target[cur_tang].my ) {
					/* congrats! we hit the exact corner pixel! now we have to decide by corner and 
					   tangent which target to use */
					if ( cur_tang == TANG_LEFT ) {
						/* left tangent */
						if ( ball->vel.y > 0 ) { 
							if ( ball->vel.x > 0 ) /* upper, right */
								targets[cur_tang] = vert_target[cur_tang];
							else
								targets[cur_tang] = hori_target[cur_tang];
						}
						else {
							if ( ball->vel.x > 0 ) /* lower, right */
								targets[cur_tang] = hori_target[cur_tang];
							else
								targets[cur_tang] = vert_target[cur_tang];
						}
					}
					else {
						/* right tangent */
						if ( ball->vel.y > 0 ) { 
							if ( ball->vel.x > 0 ) /* upper, right */
								targets[cur_tang] = hori_target[cur_tang];
							else
								targets[cur_tang] = vert_target[cur_tang];
						}
						else {
							if ( ball->vel.x > 0 ) /* lower, right */
								targets[cur_tang] = vert_target[cur_tang];
							else
								targets[cur_tang] = hori_target[cur_tang];
						}
					}
#ifdef WITH_BUG_REPORT
					if ( targets[cur_tang].x == hori_target[cur_tang].x && targets[cur_tang].y == hori_target[cur_tang].y )
						sprintf( tang_target_chosen_str[cur_tang], "(TRICKY) Horizontal target chosen." );
					else
						sprintf( tang_target_chosen_str[cur_tang], "(TRICKY) Vertical target chosen." );
#endif
				}
				else {
					if ( VEC_DIST( tang_pts[cur_tang], vector_get( hori_target[cur_tang].x, hori_target[cur_tang].y ) ) < VEC_DIST( tang_pts[cur_tang], vector_get( vert_target[cur_tang].x, vert_target[cur_tang].y ) ) ) {
                    	targets[cur_tang] = hori_target[cur_tang];
#ifdef WITH_BUG_REPORT					
						sprintf( tang_target_chosen_str[cur_tang], "Horizontal target chosen." );
#endif					
					}
                	else {
                    	targets[cur_tang] = vert_target[cur_tang];
#ifdef WITH_BUG_REPORT					
						sprintf( tang_target_chosen_str[cur_tang], "Vertical target chosen." );
#endif					
					}	
				}	
            }
    } /* now we have the two targets hit by the tangents */
	/* whatever's up the nearest brick is hit */
    if ( targets[TANG_LEFT].exists || targets[TANG_RIGHT].exists ) {
		prim = sec = 0;
        if ( !targets[TANG_LEFT].exists || !targets[TANG_RIGHT].exists ) {
			if ( targets[TANG_LEFT].exists )
				prim = &targets[TANG_LEFT];
			else 
				prim = &targets[TANG_RIGHT];
		}	
		else {
			if ( VEC_DIST( center, vector_get( targets[TANG_RIGHT].x, targets[TANG_RIGHT].y ) ) < VEC_DIST( center, vector_get( targets[TANG_LEFT].x, targets[TANG_LEFT].y ) ) )  {
				prim = &targets[TANG_RIGHT];
				sec  = &targets[TANG_LEFT];
			}
			else {
				prim = &targets[TANG_LEFT];
				sec  = &targets[TANG_RIGHT];
			}
		}	
		/* however, the primary target maybe be blocked by another brick or may be a corner */
		/* check if side of prim  target isn't blocked by a brick */
		switch ( prim->side ) {
			case SIDE_TOP: 
				if ( bricks[prim->mx][prim->my - 1].type != MAP_EMPTY ) {
					if ( ball->vel.x > 0 )
						prim->side = SIDE_LEFT;
					else
						prim->side = SIDE_RIGHT;
#ifdef WITH_BUG_REPORT
					printf( side_str, "Had to change side as TOP wasn't appropiate!" );
#endif							
				}
				break;
			case SIDE_BOTTOM: 
				if ( bricks[prim->mx][prim->my + 1].type != MAP_EMPTY ) {
					if ( ball->vel.x > 0 )
						prim->side = SIDE_LEFT;
					else
						prim->side = SIDE_RIGHT;
#ifdef WITH_BUG_REPORT
					printf( side_str, "Had to change side as BOTTOM wasn't appropiate!" );
#endif							
				}
				break;
			case SIDE_LEFT: 
				if ( bricks[prim->mx - 1][prim->my].type != MAP_EMPTY ) {
					if ( ball->vel.y > 0 )
						prim->side = SIDE_TOP;
					else
						prim->side = SIDE_BOTTOM;
#ifdef WITH_BUG_REPORT
					printf( side_str, "Had to change side as LEFT wasn't appropiate!" );
#endif							
				}
				break;
			case SIDE_RIGHT: 
				if ( bricks[prim->mx + 1][prim->my].type != MAP_EMPTY ) {
					if ( ball->vel.y > 0 )
						prim->side = SIDE_TOP;
					else
						prim->side = SIDE_BOTTOM;
#ifdef WITH_BUG_REPORT
					printf( side_str, "Had to change side as RIGHT wasn't appropiate!" );
#endif							
				}
				break;
		}
		/* now it still may be a corner */
		maybe_corner = 1;
		if ( ball->vel.y > 0 ) {
			if ( ball->vel.x > 0 ) {
				/* upper left corner */
				if ( bricks[prim->mx][prim->my - 1].type != MAP_EMPTY ) maybe_corner = 0;
				if ( bricks[prim->mx - 1][prim->my].type != MAP_EMPTY ) maybe_corner = 0;
			}
			else {
				/* upper right corner */
				if ( bricks[prim->mx][prim->my - 1].type != MAP_EMPTY ) maybe_corner = 0;
				if ( bricks[prim->mx + 1][prim->my].type != MAP_EMPTY ) maybe_corner = 0;
			}
		}
		else {
			if ( ball->vel.x > 0 ) {
				/* lower left corner */
				if ( bricks[prim->mx][prim->my + 1].type != MAP_EMPTY ) maybe_corner = 0;
				if ( bricks[prim->mx - 1][prim->my].type != MAP_EMPTY ) maybe_corner = 0;
			}
			else {
				/* lower right corner */
				if ( bricks[prim->mx][prim->my + 1].type != MAP_EMPTY ) maybe_corner = 0;
				if ( bricks[prim->mx + 1][prim->my].type != MAP_EMPTY ) maybe_corner = 0;
			}
		}
		if ( maybe_corner )
			corner_check( ball, &targets[TANG_LEFT], &targets[TANG_RIGHT], prim );
		/* we updated primary's side info correctly and may reflect now */
		ball->t = *prim;
		ball_reflect( ball );
        /* we got the reset position and the perpvector so finalize target */
        /* compute time: assume constant velocity: velocity change must not be too high! */
        dist = sqrt( square(center.x - ball->t.x) + square(center.y - ball->t.y) );
        ball->t.time = (int)floor(dist / ball_v);
        /* target's reset position is center position right now but
           we need the upper left corner of the ball */
        ball->t.x -= ball_rad; ball->t.y -= ball_rad;
        /* some error information */
		if ( ball->t.x <= 0.0 || ball->t.y <= 0.0 ) {
#ifdef WITH_BUG_REPORT
			printf( "*****\n" );
            printf( "2.2.2: Balls: %i\n", balls->count );
            printf( "Extras: " );
            for ( i = 0; i < EX_NUMBER; i++ )
                if ( active[i] )
                    printf( "%i ", i );
            printf( "\n" );
        	if ( targets[TANG_LEFT].exists ) {
                printf( "Left Tangent: Horizontal: %i,%i, %i (%4.2f,%4.2f)\n", 
						hori_target[TANG_LEFT].mx, hori_target[TANG_LEFT].my, hori_target[TANG_LEFT].side, 
						hori_target[TANG_LEFT].x, hori_target[TANG_LEFT].y );
                printf( "Left Tangent:   Vertical: %i,%i, %i (%4.2f,%4.2f)\n", 
						vert_target[TANG_LEFT].mx, vert_target[TANG_LEFT].my, vert_target[TANG_LEFT].side, 
						vert_target[TANG_LEFT].x, vert_target[TANG_LEFT].y );
				printf( "%s\n", tang_target_chosen_str[TANG_LEFT] );
			}
        	if ( targets[TANG_RIGHT].exists ) {
                printf( "Right Tangent: Horizontal: %i,%i, %i (%4.2f,%4.2f)\n", 
						hori_target[TANG_RIGHT].mx, hori_target[TANG_RIGHT].my, hori_target[TANG_RIGHT].side, 
						hori_target[TANG_RIGHT].x, hori_target[TANG_RIGHT].y );
                printf( "Right Tangent:   Vertical: %i,%i, %i (%4.2f,%4.2f)\n", 
						vert_target[TANG_RIGHT].mx, vert_target[TANG_RIGHT].my, vert_target[TANG_RIGHT].side, 
						vert_target[TANG_RIGHT].x, vert_target[TANG_RIGHT].y );
				printf( "%s\n", tang_target_chosen_str[TANG_RIGHT] );
			}
			if ( side_str[0] != 0 ) printf( "BTW: %s\n", side_str );
			printf( "-----\n" );
        	printf("Ball: %4.2f,%4.2f -> %4.2f,%4.2f (%4.2f)\n", ball->cur.x + ball_rad, ball->cur.y + ball_rad, ball->vel.x, ball->vel.y, mono );
			printf("Brick %i,%i: Side %i (%4.2f,%4.2f)\n", ball->t.mx, ball->t.my, ball->t.side, ball->t.x, ball->t.y );
        	printf("Perp Vector: %4.2f,%4.2f\n", ball->t.perp_vector.x, ball->t.perp_vector.y);
        	printf("Reset position %4.2f,%4.2f\n", ball->t.x, ball->t.y);
        	printf("Takes %i ms\n", ball->t.time);
			if ( corner != -1 ) printf( "%s\n", circle_msg );
        	printf("*****\n");
			printf( "\nYou encountered a bug! Please send this output to kulkanie@gmx.net. Thanks!\n" );
#endif		
            /* move ball back to paddle as the current target is nonsense */
            ball->t.exists = 0;
            ball->idle_time = 0;
			ball->moving_back = 1;
            ball->return_allowed = 0;
		}	
    }
}
/*
====================================================================
Increase velocity acording to vel_change
====================================================================
*/
void balls_inc_vel( int ms )
{
    List_Entry *entry;
    if (ball_v < ball_vm && !(balls->count == 1 && ((Ball*)(balls->head.next->item))->attached)) {
        ball_v += ball_v_add * ms;
        entry = balls->head.next;
        while ( entry != &balls->tail ) {
            ball_adjust_vel((Ball*)entry->item, ball_v);
            entry = entry->next;
        }
    }
}
/*
====================================================================
Return all balls that have ball->return_allowed True.
====================================================================
*/
void balls_return()
{
	Ball *ball;
    list_reset( balls );
    while ( ( ball = list_next( balls ) ) != 0 ) 
    if ( ball->return_allowed ) {
        ball->moving_back = 1;
        ball->t.exists = 0;
        ball->return_allowed = 0;
    }
}
