/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: waitall.c,v 6.4 1999/12/31 21:27:42 jsquyres Exp $
 *
 *	Function:	- wait for all requests to finish
 *	Accepts:	- # of requests
 *			- array of requests
 *			- array of status structures (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
    MPI_Waitall - Waits for all given communications to complete

Input Parameters:
+ count - lists length (integer) 
- reqs - array of requests (array of handles) 

Output Parameter:
. stats - array of status objects (array of Status), which may be the
MPI constant 'MPI_STATUSES_IGNORE'

.N waitstatus

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_REQUEST
.N MPI_ERR_ARG
.N MPI_ERR_COUNT
.N MPI_ERR_IN_STATUS
.N MPI_ERR_PENDING

.N ACK
@*/
int MPI_Waitall(int count, MPI_Request *reqs, 
		MPI_Status *stats)
{
	int		i;			/* favourite index */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	int		first;			/* first request ready */
	MPI_Status	status;
	MPI_Status	*firststat;		/* status of first request */
	MPI_Status	*p;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWAITALL);

	if (count < 0) {
	    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITALL,
		    lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((count > 0) && (reqs == 0)) {
	    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITALL,
		    lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Generate a start wrapper trace.
 */
	fl_trace = ((_kio.ki_rtf & RTF_TRON) == RTF_TRON);

	if (fl_trace) {
	    lam_tr_wrapstart(BLKMPIWAITALL);
	}
/*
 * Initialize all status members.
 */
	if (stats != MPI_STATUSES_IGNORE) {
	    for (i = 0, p = stats; i < count; ++i, ++p) {
		lam_emptystat(p);
	    }
	    firststat = &status;
	} else {
	    firststat = MPI_STATUS_IGNORE;
	}
/*
 * Loop waiting for all requests.
 */
	for (i = 0; i < count; ++i) {
	    err = MPI_Waitany(count, reqs, &first, firststat);

	    if (err != MPI_SUCCESS) {

		if (first == MPI_UNDEFINED) {
		    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITALL, err));
		}

		else {
		    if (stats != MPI_STATUSES_IGNORE) {
			stats[first].MPI_ERROR = err;
		    }

		    if (fl_trace) {
			lam_tr_wrapend(BLKMPIWAITALL);
		    }

		    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITALL,
			    lam_mkerr(MPI_ERR_IN_STATUS, 0)));
		}
	    }

	    if (first == MPI_UNDEFINED) {

		if (fl_trace) {
		    lam_tr_wrapend(BLKMPIWAITALL);
		}

		lam_resetfunc_m(BLKMPIWAITALL);
		return(MPI_SUCCESS);
	    }

	    if (stats != MPI_STATUSES_IGNORE) {
		stats[first] = *firststat;
		stats[first].MPI_ERROR = MPI_SUCCESS;
	    }
	}
/*
 * Generate an end wrapper trace.
 */
	if (fl_trace) {
	    lam_tr_wrapend(BLKMPIWAITALL);
	}

	lam_resetfunc_m(BLKMPIWAITALL);
	return(MPI_SUCCESS);
}
