/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN/RBD
 *
 *	$Id: tvector.c,v 6.4 1999/12/31 21:27:41 jsquyres Exp $
 *
 *	Function:	- create vector derived datatype
 *	Accepts:	- # blocks
 *			- block length
 *			- block stride
 *			- element datatype
 *			- new datatype (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

/*@
    MPI_Type_vector - Creates a vector (strided) datatype

Input Parameters:
+ count - number of blocks (nonnegative integer) 
. length - number of elements in each block 
(nonnegative integer) 
. stride - number of elements between start of each block (integer) 
- oldtype - old datatype (handle) 

Output Parameter:
. newtype - new datatype (handle) 

Notes:

Once a datatype is no longer being used, it should be freed with
'MPI_Type_free'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TYPE
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_EXHAUSTED

.seealso MPI_Type_free

.N ACK
@*/
int MPI_Type_vector(int count, int length, int stride, 
		    MPI_Datatype oldtype, MPI_Datatype *newtype)
{
	MPI_Datatype	ntype;			/* new datatype */
/*
 * Check the arguments.
 */
	if (oldtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITVECTOR, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITVECTOR, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((length < 0) || (newtype == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITVECTOR, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Create the new datatype.
 */
	ntype = (MPI_Datatype) malloc(sizeof(struct _dtype));
	if (ntype == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITVECTOR, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	oldtype->dt_refcount++;

	lam_type_setdefaults(ntype);
	
	ntype->dt_format = LAM_DTVECTOR;
	ntype->dt_count = count;
	ntype->dt_length = length;
	ntype->dt_stride = stride;
	ntype->dt_dtype = oldtype;

	if (count > 0) {
		lam_dtblock(ntype, oldtype, length, 0);

		if (count > 1 && stride != length) {
			ntype->dt_flags &= ~LAM_DTNOPACK;
		}

		ntype->dt_nelem *= count;
		ntype->dt_size *= count;

		stride *= (oldtype->dt_upper - oldtype->dt_lower) * (count - 1);

		if (stride < 0) {
			ntype->dt_datalow += stride;
			ntype->dt_lower += stride;
		} else {
			ntype->dt_dataup += stride;
			ntype->dt_upper += stride;
		}

		lam_dtalign(ntype);
	}
	else {
		ntype->dt_flags = (LAM_DTNOPACK | LAM_DTNOXADJ);
		ntype->dt_size = 0;
	}

	*newtype = ntype;

	lam_resetfunc(BLKMPITVECTOR);
	return(MPI_SUCCESS);
}
