/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: ssendinit.c,v 6.12 1999/12/31 21:27:38 jsquyres Exp $
 *
 *	Function:	- create a synchronous send request
 *	Accepts:	- buffer
 *			- count
 *			- datatype
 *			- destination
 *			- tag
 *			- communicator
 *			- request (out)
 *	Returns: 	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#if LAM_WANT_IMPI
#include <impi.h>
#include <debug.h>
#endif


/*@
    MPI_Ssend_init - Builds a handle for a synchronous send

Input Parameters:
+ buf - initial address of send buffer (choice) 
. count - number of elements sent (integer) 
. dtype - type of each element (handle) 
. dest - rank of destination (integer) 
. tag - message tag (integer) 
- comm - communicator (handle) 

Output Parameter:
. req - communication request (handle) 

Notes:

The communication initialized by this function will not be started
until 'req' is given to 'MPI_Start' or 'MPI_Startall'.  

The communication is then not guaranteed to progress or complete until
'req' has been given to one of the test or wait functions ('MPI_Test',
'MPI_Testall', 'MPI_Testany', 'MPI_Testsome', 'MPI_Wait',
'MPI_Waitall', 'MPI_Waitany', 'MPI_Waitsome').  The communication will
not complete until the destination rank `starts` to receive the
message.  Note that this is different than saying that the message has
been received by the destination rank.


Once any of the test or wait functions indicate that the communication
has finished, the communication cycle can be started again with
'MPI_Start' or 'MPI_Startall'.

However, once the communication has finished and it is no longer going
to be used, 'req' should be freed with 'MPI_Request_free'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_TAG
.N MPI_ERR_RANK

.seealso MPI_Ssend, MPI_Start, MPI_Startall, MPI_Test, MPI_Testall, MPI_Testany, MPI_Testsome, MPI_Wait, MPI_Waitall, MPI_Waitany, MPI_Waitsome

.N ACK
@*/
int MPI_Ssend_init(void *buf, int count, MPI_Datatype dtype, 
		   int dest, int tag, MPI_Comm comm, 
		   MPI_Request *req)
{
	int		err;			/* error code */
#if LAM_WANT_IMPI
	MPI_Request     lreq;
#endif

	lam_initerr_m();
	lam_setfunc_m(BLKMPISSINIT);

	if (tag < 0 || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPISSINIT,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}

	if (dest == MPI_ANY_SOURCE) {
		return(lam_errfunc(comm, BLKMPISSINIT,
				lam_mkerr(MPI_ERR_RANK, 0)));
	}

#if LAM_WANT_IMPI
	/* If IMPI_Isend_lamgiappe returns != MPI_REQUEST_NULL, this
           is an IMPI proxy send.  Need to re-route to the impid.  */

	IMPI_Isend_lamgiappe_init(LAM_RQISSEND, count, dtype, dest, tag, 
				  comm, &lreq);
	if (lreq != MPI_REQUEST_NULL) {
	  dest = impid_comm->c_group->g_nprocs - 1;
	  tag  = IMPI_MESSAGE_TAG;
	  comm = impid_comm;
	}
#endif

	*req = MPI_REQUEST_NULL;
	err = _mpi_req_build(buf, count, dtype, dest, tag, comm,
				LAM_RQISSEND, req);

	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPISSINIT, err));
	}

	(*req)->rq_marks |= (LAM_RQFPERSIST | LAM_RQFMAND);
#if LAM_WANT_IMPI
	(*req)->rq_shadow = lreq;
#endif

	lam_resetfunc_m(BLKMPISSINIT);
	return(MPI_SUCCESS);
}
