/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: irecv.c,v 6.19 2000/10/25 23:29:19 jsquyres Exp $
 *
 *	Function:	- immediate receive
 *	Accepts:	- buffer
 *			- count
 *			- datatype
 *			- source
 *			- tag
 *			- communicator
 *			- request (out)
 *	Returns: 	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>
#if LAM_WANT_IMPI
#include <impi.h>
#endif
#include <debug.h>


/*@
    MPI_Irecv - Begins a non-blocking receive

Input Parameters:
+ buf - initial address of receive buffer (choice) 
. count - number of elements in receive buffer (integer) 
. dtype - datatype of each receive buffer element (handle) 
. src - rank of source (integer) 
. tag - message tag (integer) 
- comm - communicator (handle) 

Output Parameter:
. req - communication request (handle) 

Notes:

The communication started with this function is not guaranteed to
progress or complete until 'req' has been given to one of the test or
wait functions ('MPI_Test', 'MPI_Testall', 'MPI_Testany',
'MPI_Testsome', 'MPI_Wait', 'MPI_Waitall', 'MPI_Waitany',
'MPI_Waitsome').

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_TYPE
.N MPI_ERR_COUNT
.N MPI_ERR_TAG
.N MPI_ERR_RANK

.seealso MPI_Recv, MPI_Test, MPI_Testall, MPI_Testany, MPI_Testsome, MPI_Wait, MPI_Waitall, MPI_Waitany, MPI_Waitsome

.N ACK
@*/
int MPI_Irecv(void *buf, int count, MPI_Datatype dtype, 
	      int src, int tag, MPI_Comm comm, 
	      MPI_Request *req)
{
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	double		startt = 0.0;		/* start time */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIIRECV);

	if ((tag < 0 && tag != MPI_ANY_TAG) || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPIIRECV,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}

	if ((fl_trace = LAM_TRACE_TOP())) startt = MPI_Wtime();

	*req = MPI_REQUEST_NULL;
	err = _mpi_req_build(buf, count, dtype, src, tag, comm,
			LAM_RQIRECV, req);
	if (err != MPI_SUCCESS) return(lam_errfunc(comm, BLKMPIIRECV, err));
	
	(*req)->rq_marks |= LAM_RQFMAND;

	err = _mpi_req_start(*req);
	if (err != MPI_SUCCESS) return(lam_errfunc(comm, BLKMPIIRECV, err));

	_mpi_req_add(*req);
	_mpi_req_blkclr();

#if LAM_WANT_IMPI
	IMPI_Register_datasync(src, tag, comm, &((*req)->rq_shadow));
#endif

	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) return(lam_errfunc(comm, BLKMPIIRECV, err));
/*
 * Generate a run time trace.  */
	if (fl_trace && src != MPI_PROC_NULL) {
		lam_tr_msg(TRTNOIO, startt, LAM_S2US(MPI_Wtime() - startt),
				0, src, tag, comm, dtype, count, 0, 0,
				(*req)->rq_seq, LAM_RQIRECV);
	}

	lam_resetfunc_m(BLKMPIIRECV);
	return(MPI_SUCCESS);
}
