/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: infodup.c,v 6.5 2001/02/03 20:24:51 jsquyres Exp $
 *
 *	Function:	- duplicate an info object
 *	Accepts:	- info
 *			- duplicate info (in-out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <string.h>

#include <all_list.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
  MPI_Info_dup - Duplicate an 'MPI_Info' object

Input Parameter:
. info - source info object (handle)

Output Parameters:
. newinfo - destination info object (handle)

Notes:

Not only will the (key, value) pairs be duplicated, the order of keys
will be the same in 'newinfo' as it is in 'info'.

When an info object is no longer being used, it should be freed with
'MPI_Info_free'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG

.seealso: MPI_Info_free

.N WEB
@*/
int MPI_Info_dup(MPI_Info info, MPI_Info *newinfo)
{
	struct _infoent	*p;			/* key/value pair */
	int		err;			/* error code */
	
	lam_initerr_m();
	lam_setfunc_m(BLKMPIINFODUP);

	if (info == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFODUP,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	err = MPI_Info_create(newinfo);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFODUP, err));
	}
/*
 * Duplicate all the entries.
 */
	for (p = al_top(info->info_list); p; p = al_next(info->info_list, p)) {

		err = MPI_Info_set(*newinfo, p->ie_key, p->ie_value);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(MPI_COMM_WORLD,
						BLKMPIINFODUP, err));
		}
	}

	lam_resetfunc(BLKMPIINFODUP);
	return(MPI_SUCCESS);
}
