/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN/RBD
 *
 *	$Id: getcount.c,v 6.5 1999/12/31 23:16:42 jsquyres Exp $
 *
 *	Function:	- get # top-level elements
 *	Accepts:	- status
 *			- datatype
 *			- count (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
  MPI_Get_count - Gets the number of "top level" elements

Input Parameters:
+ stat - return status of receive operation (Status) 
- dtype - datatype of each receive buffer element (handle) 

Output Parameter:
. count - number of received elements (integer) 

Notes:

It is `not` legal to pass 'MPI_STATUS_IGNORE' as 'stat' to this
function.

If the size of the datatype is zero, this routine will return a count
of zero.  If the amount of data in 'status' is not an exact multiple
of the size of 'datatype' (so that 'count' would not be integral), a
'count' of 'MPI_UNDEFINED' is returned instead.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_TYPE

.N ACK
@*/
int MPI_Get_count(MPI_Status *stat, MPI_Datatype dtype, 
		  int *count)
{
	lam_initerr_m();
	lam_setfunc_m(BLKMPIGETCOUNT);
/*
 * Check the arguments.
 */
	if ((count == 0) || (stat == 0) || (stat->st_length < 0)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGETCOUNT, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGETCOUNT, lam_mkerr(MPI_ERR_TYPE, 0)));
	}
/*
 * Find the count.
 */
	if (dtype->dt_size == 0 || (stat->st_length % dtype->dt_size) != 0) {
		*count = MPI_UNDEFINED;
	} else {
		*count = stat->st_length / dtype->dt_size;
	}

	lam_resetfunc_m(BLKMPIGETCOUNT);
	return(MPI_SUCCESS);
}
