/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: cdisconnect.c,v 6.4 2001/01/30 18:26:42 jsquyres Exp $
 *
 *	Function:	- completes all pending communication on
 *			  a communicator and frees it
 *	Accepts:	- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


/*@
   MPI_Comm_disconnect - Disconnect a client and server

Input Parameter:

. comm - communicator

.N fortran

.N IMPI_MPI2

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_OTHER

.seealso MPI_Close_port(3), MPI_Comm_accept(3), MPI_Comm_connect(3), MPI_Open_port(3)

.N WEB
@*/
int
MPI_Comm_disconnect(MPI_Comm *comm)
{
	MPI_Request 	*reqs;			/* array of pending requests */
	struct _req	*r;			/* request */
	int4		ccid;			/* collective context ID */
	int		npend = 0;		/* number of pending requests */
	int		err;			/* error return */
	int		i;
	
	lam_initerr();
	lam_setfunc(BLKMPICOMMDISCONNECT);

	LAM_TRACE(lam_tr_cffstart(BLKMPICOMMDISCONNECT));
/*
 * Find number of pending requests on the communicator.
 */
	ccid = lam_pt2coll((*comm)->c_contextid);
	
	for (r = lam_rq_top; r; r = r->rq_next) {
		if (r->rq_state != LAM_RQSDONE && (r->rq_cid == ccid
				|| r->rq_cid == (*comm)->c_contextid)) {
			npend++;
		}
	}
/*
 * Create array of pending requests to wait on.
 */
	reqs = (MPI_Request *) malloc((unsigned) npend * sizeof(MPI_Request));
	if (reqs == 0) {
		return(lam_errfunc(*comm, BLKMPICOMMDISCONNECT,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	for (i = 0, r = lam_rq_top; r; r = r->rq_next) {
		if (r->rq_state != LAM_RQSDONE && (r->rq_cid == ccid
				|| r->rq_cid == (*comm)->c_contextid)) {
			reqs[i++] = r;
		}
	}

	err = MPI_Waitall(npend, reqs, MPI_STATUSES_IGNORE);

	free((char *) reqs);
	if (err) {
		return(lam_errfunc(*comm, BLKMPICOMMDISCONNECT, err));
	}
	
	LAM_TRACE(lam_tr_cffend(BLKMPICOMMDISCONNECT, -1, *comm, 0, 0));

	if ((err = MPI_Comm_free(comm)) != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPICOMMDISCONNECT, err));
	}
	*comm = MPI_COMM_NULL;

        lam_resetfunc(BLKMPICOMMDISCONNECT);
	return(MPI_SUCCESS);
}
