/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: cdelattr.c,v 1.3 2001/02/03 21:56:03 bbarrett Exp $
 *
 *	Function:	- delete attribute
 *			- detach attribute from communicator
 *	Accepts:	- communicator
 *			- attribute key
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
    MPI_Comm_delete_attr - Delete an attribute

Input Parameters:
+ comm - Communicator from which the attribute is deleted
- key - Key value

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_KEYVAL
.N MPI_ERR_INTERN

.seealso MPI_Comm_create_keyval(3), MPI_Comm_free_keyval(3), MPI_Comm_set_attr(3), MPI_Comm_get_attr(3)

.N WEB
@*/
int
MPI_Comm_delete_attr(MPI_Comm comm, int key)
{
	struct _attrkey	*p;
	struct _attr	*pk;
	int		err;

	lam_initerr_m();
	lam_setfunc_m(BLKMPICOMMDELETEATTR);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_err_comm(MPI_COMM_WORLD,
				MPI_ERR_COMM, 0, "null handle"));
	}

	p = lam_getattr(key);
	if (p == 0) {
		return(lam_err_comm(comm, MPI_ERR_KEYVAL, 0, ""));
	}
/*
 * Check if the key is attached to the communicator.
 */
	pk = lam_getkey(comm->c_keys, key);
	if (pk == 0) {
		return(lam_err_comm(comm, MPI_ERR_KEYVAL, 0,
			"not on communicator"));
	}
/*
 * Delete the attribute via the callback function.
 */
	if (((MPI_Comm_delete_attr_function *) (p->ak_del))
			!= MPI_COMM_NULL_DELETE_FN) {

		if (p->ak_flags & LAM_LANGF77) {
			(*((MPI_F_delete_function *) (p->ak_del)))
				(&comm->c_f77handle, &key,
				(int *) &(pk->a_value),
				(int *) &(p->ak_extra), &err);
		} else {
			err = (*((MPI_Comm_delete_attr_function *) (p->ak_del)))
				(comm, key, pk->a_value, p->ak_extra);
		}

		if (err != MPI_SUCCESS) {
			return(lam_err_comm(comm, err, 0,
					"deleting attribute"));
		}
	}
/*
 * Detach the key from the communicator.
 */
	if (lam_delkey(comm->c_keys, key)) {
		return(lam_err_comm(comm, MPI_ERR_INTERN, errno,
			"detaching key"));
	}

	lam_resetfunc_m(BLKMPICOMMDELETEATTR);
	return(MPI_SUCCESS);
}
