/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 * $Id: ring.c,v 1.2 2001/01/22 18:03:35 jsquyres Exp $
 *
 * Simple example of MPI showing the ring message pattern.
 */

#include <stdio.h>
#include "mpi.h"


int 
main(int argc, char **argv)
{
  int message;
  int rank, size, tag, to, from;

  /* Start up MPI */

  MPI_Init(&argc, &argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  MPI_Comm_size(MPI_COMM_WORLD, &size);
 
  /* Arbitrarily choose 201 to be our tag.  Calculate the */
  /* rank of the to process in the ring.  Use the modulus */
  /* operator so that the last process "wraps around" to rank */
  /* zero. */

  message = 0;
  tag = 201;
  to = (rank + 1) % size;
  from = (rank + size - 1) % size;

  /* If we are the "console" process, get a integer from the */
  /* user to specify how many times we want to go around the */
  /* ring */

  if (rank == 0) {

    /* Send the message around the ring 5 times */

    message = 5;
    printf("Rank 0 starting message around the ring -- 1st of 5\n");
    MPI_Send(&message, 1, MPI_INT, to, tag, MPI_COMM_WORLD); 
  }

  /* Pass the message around the ring.  The exit mechanism works */
  /* as follows: the message (a positive integer) is passed */
  /* around the ring.  Each time is passes rank 0, it is decremented. */
  /* When each processes receives the 0 message, it passes it on */
  /* to the to process and then quits.  By passing the 0 first, */
  /* every process gets the 0 message and can quit normally. */

  /* Note our use of MPI_STATUS_IGNORE here.  Since we're not using
     the MPI_Status object at all, we can tell MPI to not even bother
     to fill one in with the special MPI_STATUS_IGNORE constant. */

  while (1) {
    MPI_Recv(&message, 1, MPI_INT, from, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);
    printf("Rank %d received message of \"%d\" from rank %d\n",
	   rank, message, from);

    if (rank == 0)
      message--;

    printf("Rank %d sending message of \"%d\" to rank %d\n",
	   rank, message, to);
    MPI_Send(&message, 1, MPI_INT, to, tag, MPI_COMM_WORLD);

    if (message == 0) {
      printf("Rank %d all done; quitting loop\n", rank);
      break;
    }
  }

  /* Do the final receive */

  if (rank == 0) {
    MPI_Recv(&message, 1, MPI_INT, from, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);
    printf("Rank 0 consumed final message\n");
  }

  /* Quit */

  MPI_Finalize();
  return 0;
}
