// =============================================================================
//
//      --- kvi_usertoolbar.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserToolbar"

#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_userparser.h"
#include "kvi_userpopupmenu.h"
#include "kvi_usertoolbar.h"
#include "kvi_usertoolbartemplate.h"
#include "kvi_usertoolbutton.h"

// TODO: Add an option to have the icons ALWAYS small

// From kvi_app.cpp:
extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

KviUserToolBar::KviUserToolBar(KviFrame *frame, Qt::Dock dock)
	: KviToolBar("KviUserToolBar", frame, frame, dock, true, "KviUserToolbar")
{
	initialize(frame, frame);
}

KviUserToolBar::KviUserToolBar(KviFrame *frame, QWidget *parent, Qt::Dock dock)
	: KviToolBar("KviTestToolBar", frame, parent, dock, false, "KviUserToolBar")
{
	initialize(frame, parent);
}

void KviUserToolBar::initialize(KviFrame *frame, QWidget *parent)
{
	m_pDataList   = new QPtrList<KviUserToolBarData>;
	m_pDataList->setAutoDelete(true);
	m_pFrame      = frame;
	m_pParent     = parent;
	m_bLocked     = false;
	m_id          = 1;

	setBackgroundMode(PaletteBackground);
	toggleBigIcons(g_pOptions->m_bUseBigUserToolbarPixmaps);
}

bool KviUserToolBar::isEmpty()
{
	return m_pDataList->isEmpty();
}

void KviUserToolBar::copyFromTemplate(KviUserToolBarTemplate *toolbar, bool /* unused */)
{
	__range_valid(toolbar);
	setUpdatesEnabled(false);
	clearAll();
	for( KviUserToolBarData *d = toolbar->m_pDataList->first(); d; d = toolbar->m_pDataList->next() ) {
		switch( d->type ) {
			case KviUserToolBar::Separator:
				insertSeparatorItem();
				break;
			case KviUserToolBar::Button:
				insertNormalItem(d->text.ptr(), d->sbar.ptr(), d->image.ptr(), d->buffer.ptr());
				break;
			case KviUserToolBar::Menu: {
				KviStr yourName(KviStr::Format, "usertoolbar_script_popup_%s", d->text.ptr());
				KviUserPopupMenu *m = new KviUserPopupMenu(0, yourName.ptr());
				m->copyFromPopup(d->menu);
				insertNormalItem(d->text.ptr(), d->sbar.ptr(), d->image.ptr(), 0, m);
				break;
			}
			default:
				break; // No other types
		}
	}
	setUpdatesEnabled(true);
}

void KviUserToolBar::insertSeparatorItem()
{
	insertLineSeparator();
	KviUserToolBarData *data = new KviUserToolBarData();
	data->id     = 0;
	data->buffer = " ";
	data->button = 0;
	data->menu   = 0;
	data->text   = "";
	data->image  = "";
	data->type   = KviUserToolBar::Separator;
	m_pDataList->append(data);
	m_id++;
}

int KviUserToolBar::insertNormalItem(
	const char *name, const char *sbar, const char *img, const char *buffer, KviUserPopupMenu *menu)
{
	KviUserToolBarData *data = new KviUserToolBarData();

	// Get image
	KviStr pName(name);
	m_pFrame->m_pUserParser->processString(name, (KviWindow *) m_pFrame->m_pConsole, pName);
	KviStr image(img);
	if( !image.hasData() )
		image = "0"; // Always have an image visible
	if( image.isNum() ) {
		bool bOk = false;
		int imId = image.toInt(&bOk);
		if( !bOk || (imId < 0) )
			imId = 0;
		else
			imId = imId % KVI_OUT_NUM_IMAGES;
		data->button = new KviUserToolButton(m_id, *g_pixViewOut[imId], pName.ptr(), sbar, this, 0, this);
	} else {
		KviStr imagePath;
		data->button = 0;
		if( g_pApp->findImage(imagePath, image.ptr()) ) {
			QPixmap pix(imagePath.ptr());
			if( !pix.isNull() ) {
				data->button = new KviUserToolButton(m_id, pix, pName.ptr(), sbar, this, 0, this);
			}
		}
		if( !data->button ) {
			data->button = new KviUserToolButton(m_id, *g_pixViewOut[0], pName.ptr(), sbar, this, 0, this);
		}
	}
	data->buffer = buffer;
	data->id     = m_id;
	data->image  = image;
	data->menu   = menu;
	data->sbar   = sbar;
	data->text   = name;
	data->buffer.stripWhiteSpace();
	connect(data->button, SIGNAL(idClicked(int)), this, SLOT(buttonClicked(int)));
	if( menu )
		data->type = KviUserToolBar::Menu;
	else
		data->type = KviUserToolBar::Button;
	m_pDataList->append(data);
	m_id++;
	return data->id;
}

void KviUserToolBar::buttonClicked(int id)
{
	for( KviUserToolBarData *d = m_pDataList->first(); d; d = m_pDataList->next() ) {
		if( d->id == id ) {
			if( d->menu ) {
				d->menu->doPopup(this, SLOT(popupClicked(const KviStr &)), d->button->mapToGlobal(QPoint(0, height())));
				return;
			}
			m_bLocked = true;
			m_pFrame->m_pUserParser->parseCommand(d->buffer.ptr(), (KviWindow *) (m_pFrame->m_pConsole));
			m_bLocked = false;
			return;
		}
	}
	debug("Oops! Cannot find button %d", id);
}

void KviUserToolBar::popupClicked(const KviStr &dataBuffer)
{
	m_pFrame->m_pUserParser->parseCommand(dataBuffer.ptr(), (KviWindow *) (m_pFrame->m_pConsole));
}

void KviUserToolBar::clearToolBar()
{
	clear();
}

void KviUserToolBar::clearAll()
{
	KviUserToolBarData *data;
	while( (data = m_pDataList->first()) ) {
		if( data->button )
			delete data->button;
		data->button = 0;
		if( data->menu )
			delete data->menu;
		data->menu = 0;
		m_pDataList->removeFirst();
	}
	clearToolBar();
	m_id = 1;
}

KviUserToolBar::~KviUserToolBar()
{
	clearAll();
	if( m_pDataList ) {
		delete m_pDataList;
		m_pDataList = 0;
	}
}

#include "m_kvi_usertoolbar.moc"
