// KreateCD - CD recording software for the K desktop environment
//
// 1998-2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include "CDWriter.h"
#include "CDTrack.h"
#include "ProgressDialog.h"
#include "IsoImage.h"
#include "PipeCopy.h"
#include "CDWriter.moc"

#include <qstring.h>
#include <qmessagebox.h>

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>

#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>

#include "appmacros.h"

CDWriter::CDWriter(void):ProcessInterface() {
  numTracks=0;
  pipeCopy=0;
}

void CDWriter::setupCD(int ntrack,CDTrack **track,bool simmode,int speed,Fixating fix,bool eject,bool onthefly) {
  int i;
  bool oldAudio=false,newAudio;

  numTracks=ntrack;
  simMode=simmode;
  writeSpeed=speed;
  fixMode=fix;
  ejectCD=eject;
  onTheFly=onthefly;
  delayedStart=false;

  for (i=0;i<numTracks;++i) {
    cdTracks[i]=track[i];
  }

  needSpace=0;
 
  for (i=0;i<numTracks;++i) {
    newAudio=cdTracks[i]->isAudio();
    if (i!=0) {
      needSpace+=(oldAudio==newAudio)?2*75:255;
    }
    trackSize[i]=cdTracks[i]->getRealDuration();
    blockSize[i]=newAudio?2352:2048;
    needSpace+=trackSize[i];
    oldAudio=newAudio;
    if (!newAudio) {
      needSpace+=2;
    }
  }
  if (onTheFly) {
    for (i=0;i<numTracks;++i) {
      if (cdTracks[i]->getTrackSource()!=CDTrack::Source_Filesystem) continue;
      if (cdTracks[i]->getISOImage()->getMSLBA()!=-1) {
        delayedStart=true;
        break;
      }
    }
  }
}

long int CDWriter::getNeedSpace(void) {
  return(needSpace);
}

int CDWriter::burnCD(void) {
  return(callRecord(false));
}

bool CDWriter::prepareBurn(ProgressDialog *prog) {
  int i;
  if (!onTheFly) {
    long int offset=0,total=0;
    for (i=0;i<numTracks;++i) {
      total+=trackSize[i]*blockSize[i]; 
    }
    offset=0;
    for (i=0;i<numTracks;++i) {
      prog->setupSecondProgress(total,offset);
      offset+=trackSize[i]*blockSize[i];
      if (!cdTracks[i]->validateTrack(prog)) {
        QMessageBox::critical(0,QString::null,
        	      i18n("Cannot create image files!"));
        return(false);
      }
    }
  } else {
    pipeCopy=new PipeCopy();
    pipeCopy->setupPipe();
    if (delayedStart) {
      pipeCopy->delayedOpen();
    }
    connect(pipeCopy,SIGNAL(firstOutput()),
            this,SLOT(pipeHasOutput()));

    for (i=0;i<numTracks;++i) {
      if (cdTracks[i]->needFile()) {
        if (!cdTracks[i]->prepareOnTheFly(pipeCopy)) {
          QMessageBox::critical(0,QString::null,
             i18n("Cannot prepare on the fly burning!"));
           return(false);
        }       
      }  
    }
    pipeCopy->startPipe();
    for (i=0;i<numTracks;++i) {
      flyTrack=i;
      if (!cdTracks[i]->checkOnTheFly()) continue;
      if (!cdTracks[i]->startOnTheFly(this)) {
        QMessageBox::critical(0,QString::null,
           i18n("Cannot start on the fly burning!"));
         return(false);
      }
      break;  
    }
  }

  return(true);
}

void CDWriter::cleanupBurn(void) {
  int i;
  if (pipeCopy!=0) {
    delete pipeCopy;
    pipeCopy=0;
  }
  if ((numTracks!=0) && (onTheFly)) {
    for (i=0;i<numTracks;++i) {
      cdTracks[i]->cleanupOnTheFly();
     }
  }
  numTracks=0;
}

long int CDWriter::getCDFree(void) {
 
  ProgressDialog *prog;
  int rc;

  prog=new ProgressDialog();
  prog->attachProcess(this);
  rc=callRecord(true);
  if (rc!=1) {
    delete prog;
    return(-1);
  } 
  if (prog->exec()!=1) {
    delete prog;
    printf("Error!\n");
    return(-1);
  }
  delete prog;
  printf("Free %ld Remain %ld\n",blocksFree,blocksRemain);
  return(blocksFree);
}

bool CDWriter::freshCD(void) {
  return (blocksFree==blocksTotal);
}

bool CDWriter::checkDiskSpace(void) {
  long int blcks;
  int ret;
  blcks=getCDFree();
  if (blcks==-1) return(false);
  printf("Calced : %ld\n",blcks-needSpace);
  blcks-=needSpace;
  if (blcks<-1) {
    QString buffer;
 
    buffer=i18n("WARNING: Data does not fit on disk:\n %1 blocks missing")
      .arg(-blcks);
    ret=QMessageBox::information(0,QString::null,buffer,
                                 i18n("Continue"),
                                 i18n("Cancel"));
    if (ret!=0) {
      return(false);
    }
  }
  return(true);
}

bool CDWriter::checkMSMismatch(void) {
  int ret,i;

  // check for data tracks behind track 1

  for (i=1;i<numTracks;++i) {
    if (!cdTracks[i]->isAudio()) {
      QString buffer;

      buffer=i18n("WARNING: You try to write a data track after track 1. This probably DOES NOT work for ISO 9660!");
      ret=QMessageBox::information(0,QString::null,buffer,
				   i18n("Continue"),
				   i18n("Cancel"));
      if (ret!=0) {
	return(false);
      }

    }
  }
  // check for multisession / non-multisession mismatch

  if (numTracks!=0) {
    ISOImage *isoim;
    if ( (cdTracks[0]->isAudio()) && 
	 (cdTracks[0]->getTrackSource()==CDTrack::Source_Filesystem)) {
      isoim=cdTracks[0]->getISOImage();
      if (freshCD()) {
	if (isoim->getMSWrite()!=0) {
	  QString buffer;

	  buffer=i18n("WARNING: You try to write a multisession ISO image\non a non-multisession CD-R. This probably DOES NOT work!");
	  ret=QMessageBox::information(0,QString::null,buffer,
				       i18n("Continue"),
				       i18n("Cancel"));
	  if (ret!=0) {
	    return(false);
	  }

	}
      } else { 
	if (isoim->getMSWrite()==0) {
	  QString buffer;

	  buffer=i18n("WARNING: You try to write a non-multisession ISO track \non a multisession CD-R. This probably DOES NOT work!");
	  ret=QMessageBox::information(0,QString::null,buffer,
				       i18n("Continue"),
				       i18n("Cancel"));
	  if (ret!=0) {
	    return(false);
	  }

	}
      }
    }
  }
  return(true);
}


int CDWriter::callRecord(bool capread) {
  int i;
  CDTrack *xtrack;
  KConfig *config;
  enum CDTrack::TrackType cmode,xmode;

  char devline[32],speedline[32];
  int unit,host;

  getCapacity=capread;
  currentTrack=0;
  trackOffset=0;

  prepareProcess();
  config=kapp->config();
  config->setGroup("SCSI");
  host=config->readNumEntry("SCSIWriterHost",0);
  unit=config->readNumEntry("SCSIWriterUnit",-1);
  if (unit==-1) return(0);

  sprintf(devline,"dev=%d,%d,0",host,unit);
  sprintf(speedline,"speed=%d",writeSpeed);
  config->setGroup("Path");
  if FREE_CDRECORD {
    *this<<config->readEntry("PathCdrecord",PATH_CDRECORD)<<"-v"<<
      devline<<speedline;
  } else {
    *this<<PATH_WRAPPER<<"cdrecord"<<"-v"<<devline<<speedline;
  }
  if (simMode) *this<<"-dummy";
  switch (fixMode) {
  case Fix_None:
    *this<<"-nofix";
    break;
  case Fix_Session:
    *this<<"-multi";
    break;
  case Fix_CD:
    break;
  }

  if (ejectCD) *this<<"-eject";



  cmode=CDTrack::Track_DataMode1;

  if (!getCapacity) {
    for (i=0;i<numTracks;++i) {
      char imagefile[1024];
      xtrack=cdTracks[i];

      if (!xtrack->checkOnTheFly()) {
        if (!xtrack->getBurnFile(imagefile)) {
          QMessageBox::critical(0,QString::null,i18n("Image file not found!"));
          return(0);
       }
      }  else {
        char tsizer[64];
        if (!xtrack->getOnTheFlyFile(imagefile)) {
          QMessageBox::critical(0,QString::null,i18n("Image file not found!"));
          return(0);
        }
        sprintf(tsizer,"tsize=%ld",getSize(xtrack));
        *this<<tsizer;
      }
      xmode=xtrack->getTrackType();
      if ( (xmode!=cmode) || (i==0) ) {
        if (cmode==CDTrack::Track_AudioPreEmph) *this<<"-nopreemp";
        cmode=xmode;
        switch(xmode) {
        case CDTrack::Track_Audio:
          *this<<"-audio";
	  break;
        case CDTrack::Track_AudioPreEmph: 
          *this<<"-audio"<<"-preemp";
	  break;
        case CDTrack::Track_DataMode1:
	  *this<<"-data";
          break;
        case CDTrack::Track_DataMode2:
          *this<<"-mode2";
          break;
        case CDTrack::Track_XA1:
          *this<<"-xa1";
          break;
        case CDTrack::Track_XA2:
          *this<<"-xa2";
          break;
        case CDTrack::Track_CDI:
          *this<<"-cdi";
          break;
        }
      }
      *this<<imagefile;
    }
  } else {
    FILE *duf;
    char dumpath[512];
    KConfig *mconfig;
    mconfig=kapp->config();
    mconfig->setGroup("Path");
    strcpy(dumpath,mconfig->readEntry("PathTemporary","/tmp"));
    strcat(dumpath,"/dummyfile");
    *this<<dumpath;
    duf=fopen(dumpath,"wb");
    fclose (duf);
  }


  reportAction(getCapacity?i18n("Reading capacity..."):i18n("Preparing burning..."));

  burnstate=0;
  burntrack=0;
  burnsize=100;

  if ( (getCapacity) || (!delayedStart) ) {
    startProcess();  // 1=ripping was OK  -1=abort -2=ripping error  0=window hard closed
  }
  return(1);
}

int CDWriter::processCleanup(int retval) {
  KConfig *config;
 
  config=kapp->config();
  closeProcess();

  if (retval==-2) {
    switch (burnstate) {
    case 10000:
      QMessageBox::critical(0,QString::null,i18n("Cannot access the drive! (no disc?)"));
      break;
    case 10001:
      QMessageBox::critical(0,QString::null,i18n("Cannot open a new session!"));
      break;
    default:
      QMessageBox::critical(0,QString::null,i18n("Unrecognized error while writing CD-R!!!"));
      break;
    }
  }

  if (getCapacity) {
    char dumpath[512];
    KConfig *mconfig;
    mconfig=kapp->config();
    mconfig->setGroup("Path");
    strcpy(dumpath,mconfig->readEntry("PathTemporary","/tmp"));
    strcat(dumpath,"/dummyfile");
    remove(dumpath);
  }

  if ( (retval!=1) || (getCapacity)) {
    // workaround : open close writer device to unlock tray
    FILE *xfile;
    QString qs;
    config->setGroup("SCSI");
    qs=config->readEntry("SCSIWriterDevice","/dev/null");
    xfile=fopen(qs.data(),"rb");
    if (xfile!=0) fclose(xfile);
  }

  if (retval!=1) return(0);

  return(1);
}

int CDWriter::processExited(void) {
  return((burnstate==99999)?1:-2);
}

bool CDWriter::processStdoutLine(char *linebuffer) {
  if (linebuffer[0]==0) return(true);
  if (strncmp("Track ",linebuffer,6)==0) {
    char *xptr,*yptr;
    long cur,max,fifo,track;
    xptr=linebuffer+6;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) {
      printStatusLine(QString(linebuffer));
      return(true);
    }
    track=strtol(xptr,&yptr,10);
    if (*yptr!=':') {
      printStatusLine(QString(linebuffer));
      return(true);
    }
    xptr=yptr+1;
    while (*xptr==' ') ++xptr;
    if ( (*xptr=='0') && (*(xptr+1)==' ') ) {
      cur=0;
      yptr=xptr+1;
    } else {
      cur=strtol(xptr,&yptr,10);
    }
    if (strncmp(" of ",yptr,4)!=0) {
      printStatusLine(QString(linebuffer));
      return(true);
    }
    xptr=yptr+4;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) {
      printStatusLine(QString(linebuffer));
      return(true);
    }
    max=strtol(xptr,&yptr,10);
    fifo=100;
    if (strncmp(" MB written (fifo ",yptr,18)!=0) { 
      if (strncmp(" MB written",yptr,11)==0) goto noFifo;
      printStatusLine(QString(linebuffer));
      return(true);
    }
    xptr=yptr+18;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) {
      printStatusLine(QString(linebuffer));
      return(true);
    }
    fifo=strtoul(xptr,&yptr,10);
    reportBuffer(fifo);
  noFifo:
    if (currentTrack==0) {
      trackOffset=track-1;
    }
    currentTrack=track-trackOffset;
    if (max==0) max=1;
    if ( (currentTrack<1) || (currentTrack>numTracks)) return(true);
    reportProgress(cur*1024*1024,trackSize[currentTrack-1]*blockSize[currentTrack-1]);
    if (burntrack==0) {
      char xwork[128];
      sprintf(xwork,i18n("Writing track %ld"),currentTrack);
      reportAction(xwork);
    }
    return(true);
  }

  printStatusLine(QString(linebuffer));
  if (strncmp("Starting to write CD",linebuffer,20)==0) {
    if (getCapacity) {
      burnstate=99999;
      usleep(1500);  // make sure cdrecord is interruptible
      return(false);
    }
    return(true);
  }
  if (strncmp("Starting new track at sector:",linebuffer,29)==0) {
    long int offset=0,total=0;
    int i;
    burntrack=0;
    if ( (currentTrack<0) || (currentTrack>=numTracks)) return(true);
    reportProgress(0,trackSize[currentTrack]*blockSize[currentTrack]); 
    for (i=0;i<numTracks;++i) {
      if (i==currentTrack) offset=total;
      total+=trackSize[i]*blockSize[i];
    }
    reportSecondProgress(total,offset);
    return(true);
  }
  if (strncmp("Fixating...",linebuffer,11)==0) {
    long int total=0;
    int i;
    total=trackSize[numTracks-1]*blockSize[numTracks-1]; 
    reportProgress(total,total);
    reportAction(i18n("Fixating..."));
    return(true);
  }
  if (strncmp("Blocks total:",linebuffer,13)==0) {
    char *xptr,*yptr;
    xptr=linebuffer+13;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksTotal=strtoul(xptr,&yptr,10);
    if (strncmp(" Blocks current:",yptr,16)!=0) return(true);
    xptr=yptr+16;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksFree=strtoul(xptr,&yptr,10);
    if (strncmp(" Blocks remaining:",yptr,18)!=0) return(true);
    xptr=yptr+18;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksRemain=strtoul(xptr,&yptr,10);
    if (xptr==yptr) return(true);
    return(true);
  }

  return(true);
}

bool CDWriter::processStderrLine(char *linebuffer) {
  printStatusLine(QString(linebuffer));
  {
    char *xptr;
    xptr=linebuffer;
    while ( (*xptr!=0) && (*xptr!=':') ) ++xptr;
    if (*xptr!=0) {
      xptr++;
      while (*xptr==' ') ++xptr;
      if (strncmp("fifo had ",xptr,9)==0) {
        if (burnstate<10000) burnstate=99999;
        return(true);
      }
      if (strncmp("No disk",xptr,7)==0) {
        burnstate=10000;
        return(true);
      }
      if (strncmp("Cannot open new session",xptr,23)==0) {
        burnstate=10001;
      }
    }
  }
  return(true);
}

long int CDWriter::getSize(CDTrack *trk) {
  if ( trk->isAudio()) {
    return (trk->getRealDuration()*2352);
  } else {
    return (trk->getRealDuration()*2048);
  }
}

void CDWriter::onTheFlyTerminated(ProcessInterface *,int ) {
  ++flyTrack;
  for (;flyTrack<numTracks;++flyTrack) {
    cdTracks[flyTrack]->startOnTheFly(this);  
    break;
  }
}

bool CDWriter::singleDrive(void) {
  int bhost,bunit,chost,cunit;
  KConfig *config;
  config=kapp->config();
  config->setGroup("SCSI");
  bhost=config->readNumEntry("SCSIWriterHost",0);
  bunit=config->readNumEntry("SCSIWriterUnit",-1);
  chost=config->readNumEntry("SCSICdromHost",0);
  cunit=config->readNumEntry("SCSICdromUnit",-1);
  if ( (bhost==chost) && (bunit==cunit) ) return(true);
  return(false);
}

void CDWriter::pipeHasOutput(void) {
  if (delayedStart) startProcess();
}
