// KreateCD - CD recording software for the K desktop environment
//
// 1999-2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include <qstring.h>
#include <qobject.h>

#include <kapp.h>
#include <kconfig.h>

#include <sys/stat.h>
#include <unistd.h>


#include "CDTrack.h"
#include "AudioFileInfo.h"
#include "AudioBuildImage.h"
#include "AudioFileFormat.h"
#include "DataRipper.h"
#include "AudioRipper.h"
#include "AudioConversion.h"
#include "AppChooser.h"
#include "IsoImage.h"
#include "ProgressDialog.h"
#include "PipeCopy.h"

#include <stdlib.h>
#include <strings.h>
#include <stdio.h>
#include <sys/stat.h>

static const char *TypeNames[]={"Audio","Audio PreEmph","Mode 1","Mode 2","XA Mode 1","XA Mode 2","CDI"};

CDTrack::CDTrack(void) {
  initObject();
}

CDTrack::~CDTrack(void) {
  clearImage();

  if (ThisAudio!=0) {
    delete(ThisAudio);
  }

  if (ThisISO!=0) {
    delete(ThisISO);
  }

  if (IsTemporary) {
    remove(SourceFile);
  }
}

CDTrack::CDTrack(KConfig *config)
 {
  QString tqs;

  initObject();
  TrackType=(enum TrackType) config->readNumEntry("TrackType",Track_DataMode1);
  TrackSource=(enum TrackSource) config->readNumEntry("TrackSource",Source_None);
  TrackImage=(enum TrackImage) config->readNumEntry("TrackImage",Image_None);
  //FIXME : for now we mark all loaded tracks as dirty
  if (TrackImage==Image_Clean) TrackImage=Image_Dirty;
  //IsTemporary=config->readBoolEntry("IsTemporary",false);
  IsTemporary=false;
  //FIXME : assume that there is no temporary CD image file
  CDTrackNum=config->readNumEntry("CDTrackNum",1);
  CDTocSum=config->readUnsignedLongNumEntry("CDTocSum",0);
  tqs=config->readEntry("Description",QString::null);
  strcpy(Description,tqs.data());
  tqs=config->readEntry("SourceFile",QString::null);
  strcpy(SourceFile,tqs.data());
  tqs=config->readEntry("ImageFile",QString::null);
  strcpy(ImageFile,tqs.data());

  TrackSize=config->readUnsignedLongNumEntry("TrackSize",0);
  TrackDuration=config->readUnsignedLongNumEntry("TrackDuration",0);
  TrackLBA=config->readUnsignedLongNumEntry("TrackLBA",0);

  if ( isAudio() && (TrackSource==Source_CD) )  {
    ThisAudio=new AudioFileInfo();
     {
      AudioFileFormat afifo(ThisAudio);
      if (!afifo.setupCDTrack(TrackDuration))
       {
        delete ThisAudio;
        ThisAudio=0;
        IsTemporary=false;
       }
     }
   }

  if ( isAudio() && (TrackSource==Source_File) )
   {
    ThisAudio=new AudioFileInfo();
     {
       AudioFileFormat afifo(ThisAudio);
       if (!afifo.identifyFile(SourceFile))
       {
        delete(ThisAudio);
        ThisAudio=0;
       }
     }
   }
  if ( isAudio() && (ThisAudio!=0) )
   {
    ThisAudio->setStartPos(config->readLongNumEntry("TrackStart",0));
    ThisAudio->setEndPos(config->readLongNumEntry("TrackEnd",
                         (ThisAudio->getAbsoluteDuration()*588)-1));
    ThisAudio->setBoost((float) config->readDoubleNumEntry("TrackBoost",1));
    ThisAudio->setBalance((float) config->readDoubleNumEntry("TrackBalance",0));
   }

  if ( !isAudio() && (TrackSource==Source_Filesystem) )
   {
    ThisISO=new ISOImage(config); 
    TrackDuration=ThisISO->calculateImage();
   }

  needImage();
 }

CDTrack::CDTrack(AudioFileInfo *afi) {
  initObject(); // I think this is missing here;
  TrackType=Track_Audio;
  TrackSource=Source_File;
  strcpy(SourceFile,afi->getSourceName());
  ImageFile[0]='\0';
  TrackSize=0;
  TrackDuration=0;
  ThisAudio=afi;
  needImage();
  TrackImage=NeedImage?Image_Dirty:Image_None;
  updateTrack();
}

CDTrack::CDTrack(const CDTrack &object) {
  initObject();

  TrackType=object.TrackType;
  TrackSource=object.TrackSource;
  TrackImage=object.TrackImage;
  NeedImage=object.NeedImage;
  IsTemporary=false;
  CDTrackNum=object.CDTrackNum;
  CDTocSum=object.CDTocSum;
  strcpy(Description,object.Description);
  strcpy(SourceFile,object.SourceFile);
  // FIXME check Source File and Image File
  ImageFile[0]=0;
  TrackSize=object.TrackSize;
  TrackDuration=object.TrackDuration;
  TrackLBA=object.TrackLBA;
  if (object.ThisAudio!=0) {
    ThisAudio=new AudioFileInfo(*object.ThisAudio);
  } else {
    ThisAudio=0;
  }
  if (object.ThisISO!=0) {
    ThisISO=new ISOImage(*object.ThisISO);
  } else {
    ThisISO=0;
  }
  if (TrackImage==Image_Clean) TrackImage=Image_Dirty;        
  needImage();
}

void CDTrack::saveTrack(KConfig *config)
 {
  config->writeEntry("TrackType",(int)TrackType);
  config->writeEntry("TrackSource",(int)TrackSource);
  config->writeEntry("TrackImage",(int)TrackImage);
  config->writeEntry("IsTemporary",IsTemporary);
  if (TrackSource==Source_CD)
   {
    config->writeEntry("CDTrackNum",CDTrackNum);
    config->writeEntry("CDTocSum",CDTocSum);
    config->writeEntry("TrackLBA",TrackLBA);
   }
  config->writeEntry("Description",Description);

  config->writeEntry("SourceFile",SourceFile);
  if (TrackImage==Image_Clean)
   {
    config->writeEntry("ImageFile",ImageFile);
   }
  config->writeEntry("TrackSize",TrackSize);
  config->writeEntry("TrackDuration",TrackDuration);
  if ( (isAudio()) && (ThisAudio!=0) )
   {
    config->writeEntry("TrackStart",ThisAudio->startPos());
    config->writeEntry("TrackEnd",ThisAudio->endPos());
    config->writeEntry("TrackBoost",(double) ThisAudio->boost());
    config->writeEntry("TrackBalance",(double) ThisAudio->balance());
   }
  if ( (!isAudio()) && (TrackSource==Source_Filesystem)) 
   {
    ThisISO->saveImage(config);
   }
 }

void CDTrack::initObject(void) {
  strcpy(Description,"New track");
  ImageFile[0]='\0';
  TrackType=Track_DataMode1;
  ThisAudio=0;
  ThisISO=0;
  TrackImage=Image_None;
  IsTemporary=false;

  onTheFlyRipper=0;
  onTheFlyDRipper=0;
  onTheFlyABuildImage=0;
  onTheFlyAudioConverter=0;
  onTheFlyTarget[0]=0;
  onTheFlyRipPipe[0]=0;
  onTheFlyAudioPipe[0]=0;
  onTheFlyDecoderPipe[0]=0;

  clearSource();
}

enum CDTrack::TrackType CDTrack::getTrackType(void)
 {
  return(TrackType);
 }

enum CDTrack::TrackSource CDTrack::getTrackSource(void)
 {
  return(TrackSource);
 }

enum CDTrack::TrackImage CDTrack::getTrackImageMode(void)
 {
  return(TrackImage);
 }

void CDTrack::getTrackDescription(char *desc)
 {
  strcpy(desc,Description);
 }

void CDTrack::getTrackDurationHMSB(char *duration,bool megabytes)
 {
  long int dura;
  dura=TrackDuration;
  if (ThisAudio)
   {
    dura=ThisAudio->getSelectedDuration();
   }
  getDurationHMSB(duration,dura,isAudio(),megabytes);
 }

long int CDTrack::getRealDuration(void)
 {
  long int dura;
  if ( (ThisISO!=0) && (!ThisISO->sizeValid())) {
        TrackDuration=ThisISO->calculateImage();
  }

  dura=TrackDuration;
  if (ThisAudio)
   {
    dura=ThisAudio->getSelectedDuration();
   }
  return(dura);
 }
 


void CDTrack::getTrackTypeText(char *type)
 {
  strcpy(type,TypeNames[TrackType]);
 }

bool CDTrack::isAudio(void)
 {
  if ( (TrackType==Track_Audio) || (TrackType==Track_AudioPreEmph) ) return(true);
  return(false);
 }

class AudioFileInfo *CDTrack::getAudioFile(void)
 {
  return(ThisAudio);
 }

class ISOImage * CDTrack::getISOImage(void)
 {
  return(ThisISO);
 }

bool CDTrack::getBurnFile(char *burnfile)
 {
  switch (TrackImage)
   {
    case Image_None:
      if (ThisAudio==0) {
        if (burnfile!=0) strcpy (burnfile,SourceFile);
      } else {
        const char *bfil;
        if (ThisAudio->needsRipping()) return(false);
        bfil=ThisAudio->getBurnFile();
        if ( (bfil==0) || (bfil[0]==0) ) return(false);
        strcpy (burnfile,bfil);
      }
      return(true);
    case Image_Dirty:
      return(false);
    case Image_Clean:
      if (burnfile==0) return(true);
      strcpy(burnfile,ImageFile);
      return(true);
   }
  return(false);
 }

bool CDTrack::getUseImage(void)
 {
  if (TrackImage==Image_None) return(false);
  return(true);
 }

void CDTrack::enableImage(void)
 {
  if (TrackImage==Image_None) TrackImage=Image_Dirty;
 }

void CDTrack::disableImage(void)
 {
  if (!NeedImage)
   {
    clearImage();
    TrackImage=Image_None;
   }
 }

void CDTrack::getTypeString(enum TrackType ttype,char *type)
 {
  strcpy(type,TypeNames[ttype]);
 }

void CDTrack::getDurationHMSB(char *durstr,long int duration,bool isaudio,bool megabytes)
 {
  int hour,min,sec,block;
  long int mega;

  block=duration%75;
  sec=duration/75;
  min=sec/60;
  sec%=60;
  hour=min/60;
  min%=60;
  mega=duration*(isaudio?2352:2048);
  mega+=512*1024;
  mega/=1024*1024;

  if (!megabytes) {
    sprintf(durstr,"%01d:%02d:%02d.%02d",hour,min,sec,block);
  } else {
    sprintf(durstr,"%01d:%02d:%02d.%02d (%ld MB)",hour,min,sec,block,mega);
  }
}      

void CDTrack::setDescription(const char *descr)
 {
  strncpy(Description,descr,39);
 }

void CDTrack::setTrackType(enum TrackType tracktype)
 {
  int oldaud,aud;
  oldaud=isAudio();
  TrackType=tracktype;
  aud=isAudio();
  if (oldaud!=aud) clearSource();
  updateTrack(); 
 }

void CDTrack::clearSource(void)
 {
  TrackSource=Source_None;
  if (ThisAudio!=0)
   {
    delete(ThisAudio);
    ThisAudio=0;
   }

  if (ThisISO!=0)
   {
    delete(ThisISO);
    ThisISO=0;
   }

  TrackDuration=0;
  TrackSize=0;
  if (IsTemporary)
   {
    remove(SourceFile);
    IsTemporary=false;
   }

  SourceFile[0]='\0';
  clearImage();
  TrackImage=Image_None;
  needImage();
 }

void CDTrack::setSourceFile(const char *filename)
 {
  clearSource();
  TrackSource=Source_File;
  strcpy(SourceFile,filename);
  ImageFile[0]='\0';
  TrackSize=0;
  TrackDuration=0;
  if (isAudio())
   {
    ThisAudio=new AudioFileInfo();
     {
      AudioFileFormat afifo(ThisAudio);
       if (!afifo.identifyFile(filename))
       {
        delete(ThisAudio);
        ThisAudio=0;
       }
      }
    needImage();
   }
  TrackImage=NeedImage?Image_Dirty:Image_None;
  updateTrack();
 }

void CDTrack::setSourceFileSystem(ISOImage *isoimg)
 {
  if (isoimg==ThisISO) ThisISO=0;
  clearSource();
  if (isAudio()) return;
  TrackSource=Source_Filesystem;
  SourceFile[0]='\0';
  ImageFile[0]='\0';
  TrackDuration=0;
  needImage();
  TrackImage=NeedImage?Image_Dirty:Image_None;
  if (isoimg==0)
   {
    ThisISO=new ISOImage();
   }
   else
   {
    ThisISO=isoimg;
   }
  updateTrack();

 }

void CDTrack::setSourceTrack(int track,long int tocsum,unsigned long duration,unsigned long lba)
 {
  clearSource();
  TrackSource=Source_CD;
  CDTrackNum=track;
  CDTocSum=tocsum;
 
  SourceFile[0]='\0';
  TrackDuration=duration;
  if (isAudio())
   {
    TrackSize=duration*2352;
   }
   else
   {
    TrackSize=duration*2048;
   }
  TrackLBA=lba;
  if (isAudio()) {
    AudioFileFormat *afifo;
    ThisAudio=new AudioFileInfo();
    afifo=new AudioFileFormat(ThisAudio);
    afifo->setupCDTrack(duration);
    delete afifo;    
  }
  needImage();
  TrackImage=NeedImage?Image_Dirty:Image_None;
  updateTrack();
 }

void CDTrack::updateTrack()
 {
  clearImage();
  if (IsTemporary==true)
   {
    remove(SourceFile);
    IsTemporary=false;
   }

  switch (TrackSource)
   {
    case Source_None:
     {
      break;
     }
    
    case Source_CD:
      break;
    case Source_File:
     {
      struct stat statbuf;
      if ((isAudio()) && (ThisAudio!=0) )
       {
        
        TrackDuration=ThisAudio->getSelectedDuration();
       }
       else
       { 
        if (stat(SourceFile,&statbuf))
         {
          clearSource();
          return;
         }      
        TrackSize=statbuf.st_size;
        if (isAudio())
         {
          TrackDuration=(TrackSize+2351)/2352;
         }
         else
         {
          TrackDuration=(TrackSize+2047)/2048;
         }     
        break;
       }
     }
    case Source_Filesystem:
      break;
   }
 }

bool CDTrack::validateTrack(ProgressDialog *xprog,const char *targetfilename)
 {
  if (TrackSource==Source_None) return(0);
  if ((TrackSource==Source_CD) && (isAudio()) && (IsTemporary!=true))
   {
    ProgressDialog *prog;
    AudioRipper *ripper;

    if (ThisAudio==0) return(false);
    ripper=AppChooser::getAudioRipper();
    if (ripper==0) return(false);
    clearImage();
    if (IsTemporary==true)
     {
      remove(SourceFile);
      IsTemporary=false;
     }
    createName(SourceFile,false,TrackDuration);
    if (xprog==0) {
      prog=new ProgressDialog();
    } else {
      prog=xprog;
    }
    prog->attachProcess(ripper);
    if (ripper->ripAudio(CDTrackNum,SourceFile)==0) {
      delete ripper;
      if (xprog==0) delete prog;
      return(false);
    }
    if (prog->exec()==0) {
      delete ripper;
      if (xprog==0) delete prog;
      return(false);
    }
    if (xprog==0) delete prog;
    delete ripper;
    IsTemporary=true;
    ThisAudio->setFilename(SourceFile);
    ThisAudio->setNeedsRipping(false);
    updateDuration(SourceFile);
    return(true);
   }
  

  if ( (TrackSource==Source_File) && (isAudio()) &&
       (ThisAudio!=0))
   {
	if (ThisAudio->validateFile()==0) {
          return(0);
        }
   }
 
  if (TrackImage==Image_None) return(1);
  if (TrackImage==Image_Clean) return(1);
  

  switch (TrackSource)
   {
    case Source_CD:
      if (isAudio())
       {
        // create audio image file
        if (ThisAudio==0) return(0);
        if (targetfilename==0) {
          createName(ImageFile,true,TrackDuration);
        } else {
          strcpy(ImageFile,targetfilename);
        }
        {
          ProgressDialog *prog;
          AudioBuildImage img(ThisAudio);

          if (xprog==0) {
            prog=new ProgressDialog();
          } else {
            prog=xprog;
          }
          prog->attachProcess(&img);
          if (!img.buildImage(ImageFile)) {
            if (xprog==0) delete prog;
            return(false);
           }
          if (!prog->exec()) {
            if (xprog==0) delete prog;
            return(false);
          }         
          if (xprog==0) delete prog;
        }
        TrackImage=Image_Clean;
        updateDuration(ImageFile);
        return(true);
       }
       else
       {
        // rip a datatrack from CD
        ProgressDialog *prog;
        DataRipper *datarip= AppChooser::getDataRipper();
        if (datarip==0) return(false);
        if (targetfilename==0) {
          createName(ImageFile,true,TrackDuration);
        } else {
          strcpy(ImageFile,targetfilename);
        }
        if (xprog==0) {
          prog=new ProgressDialog();
        } else {
          prog=xprog;
        }
        prog->attachProcess(datarip);
         
        if (!datarip->readDataTrack(CDTrackNum,ImageFile,TrackLBA,TrackDuration))
         {
          delete datarip;
          if (xprog==0) delete prog;
          return(false);
         }
        if (!prog->exec()) {
          delete datarip;
          if (xprog==0) delete prog;
          return(false);
        }
        delete(datarip);
        if (xprog==0) delete prog;
        updateDuration(ImageFile);
        TrackImage=Image_Clean;
        return(true);
       } 
    case Source_File:
      if (isAudio())
       {
        // create audio image file
        if (ThisAudio==0) return(0);
	ThisAudio->validateFile();
        if (targetfilename==0) {
          createName(ImageFile,true,TrackDuration);
        } else {
          strcpy(ImageFile,targetfilename);
        }
         {
          ProgressDialog *prog;
          AudioBuildImage img(ThisAudio);

          if (xprog==0) {
            prog=new ProgressDialog();
          } else {
            prog=xprog;
          }
          prog->attachProcess(&img);
          if (!img.buildImage(ImageFile)) {
            if (xprog==0) delete prog;
            return(false);
           }
          if (!prog->exec()) {
            if (xprog==0) delete prog;
            return(false);
          }         
          if (xprog==0) delete prog;
        }
        updateDuration(ImageFile);
        TrackImage=Image_Clean;
        return(true);
       }
       else
       {
        // copy a datatrack
        if (targetfilename==0) {
          createName(ImageFile,true,TrackDuration);
        } else {
          strcpy(ImageFile,targetfilename);
        }
        return(false);
        //updateDuration(ImageFile);
        //TrackImage=Image_Clean;
        //return(1);
       }
    case Source_Filesystem:
     {
      ProgressDialog *prog;
 
      if (ThisISO==0) return(0);
      if (targetfilename==0) {  
        createName(ImageFile,true,rand()%1000);
      } else {
        strcpy(ImageFile,targetfilename);
      }
      if (xprog==0) {
        prog=new ProgressDialog();
      } else {
        prog=xprog;
      }
      prog->attachProcess(ThisISO);
      if (!ThisISO->makeImage(ImageFile))
       {
        if (xprog==0) delete prog;
        return(false);
       }
      if (!prog->exec()) {
        if (xprog==0) delete prog;
        return(false);
      }
      if (xprog==0) delete prog;
      updateDuration(ImageFile);
      TrackImage=Image_Clean;
      return(true);
     }

    default:
      return(false);
   }

  return(false);
 }


void CDTrack::clearImage(void) {
  cleanupOnTheFly();
  needImage();
  if (NeedImage==true) {
    if (TrackImage==Image_None) TrackImage=Image_Dirty;
  } else {
    if (TrackImage!=Image_None) TrackImage=Image_None;
  }
  if (TrackImage!=Image_Clean) return;
  remove(ImageFile);
  TrackImage=Image_Dirty;
  ImageFile[0]='\0';
 }


bool  CDTrack::exportISO(class QString filename)
{
	
      ProgressDialog *prog;
 
      if (ThisISO==0) return(0);

      prog=new ProgressDialog();
      prog->attachProcess(ThisISO);
      if (!ThisISO->makeImage(filename))
       {
        delete prog;
        return(false);
       }
      if (!prog->exec()) {
        delete prog;
        return(false);
      }
      delete prog;
//      updateDuration(ImageFile);
//      TrackImage=Image_Clean;
      return(true);
}

void CDTrack::saveImage(void)
{
 TrackImage=Image_Dirty;
 clearImage();
}


void CDTrack::createName(char *fn,bool cdrimg,long int duration,bool iswav)
 {
  KConfig *config;
  QString tempdir;
  const char *directory;
  int random;

  config=kapp->config();

  config->setGroup("Path");
  tempdir=config->readEntry("PathTemporary","/tmp");
  directory=tempdir.data();
   
  while (1)
   {
    struct stat sbuf;
    random=rand()%1000;
    sprintf(fn,"%s/kreatecd_%ld_%d.%s",directory,duration,random,iswav?"cdr":(cdrimg?"img":"cdt"));
    if (stat(fn,&sbuf)!=0) break;
   }
 }

void CDTrack::updateDuration(const char *fn)
 {
  struct stat sb;
  if (stat(fn,&sb)==-1) return;
  TrackSize=sb.st_size;
  if (TrackSource==Source_CD)
   {
    if (isAudio())
     {
      TrackDuration=TrackSize/2352;
     }
     else
     {
      TrackDuration=TrackSize/2048;
     }
    }
  if (TrackSource==Source_Filesystem)
   {
    TrackDuration=TrackSize/2048;
   }
 }

void CDTrack::setTrackSize(long int size)
 {
   TrackSize=size;
   TrackDuration=TrackSize/2048;
 }


bool CDTrack::needImage(void) {
  switch (TrackSource) {
    case Source_None:
      NeedImage=false;
      break;
    case Source_File:
      if (ThisAudio==0) {
        NeedImage=false;
      } else {
        NeedImage=ThisAudio->needImage()?true:false;
      }
      break;
    case Source_Filesystem:
      NeedImage=true;
      break;
    case Source_CD:
      if (isAudio()) {
        if (ThisAudio==0) {
          NeedImage=false;
        } else {
          NeedImage=ThisAudio->needImage()?true:false;
        }
      } else {
        NeedImage=true;
      }
      break;
  }
  return(NeedImage);
}

bool CDTrack::needFile(void) {
  switch (TrackSource) {
    case Source_None:
      return(false);
    case Source_File:
      if (ThisAudio==0) {
        return(false);
      } else {
        return (ThisAudio->needFile());
      }
    case Source_Filesystem:
      return(true);
    case Source_CD:
      return(true);
  }
  return(false);
}

bool CDTrack::getOnTheFlyFile(char *burnfile) {
  if (onTheFlyTarget[0]!=0) {
    if (burnfile!=0) strcpy(burnfile,onTheFlyTarget);
    return(true);
  }
  return(false);
}

bool CDTrack::prepareOnTheFly(PipeCopy *pcop) {
  cleanupOnTheFly();
  switch (TrackSource) {
    case Source_None:
      return(false);
    case Source_File:
      if (isAudio()) {
        const char *bf;

        if (ThisAudio==0) return(false);
        bf=ThisAudio->getSourceName();

        createName(onTheFlyTarget,false,getRealDuration(),false);
        if (mkfifo(onTheFlyTarget,0600)!=0) { 
          return(false);
        }

        if (ThisAudio->converter()!=0) {
          createName(onTheFlyDecoderPipe,false,getRealDuration(),false);
          if (mkfifo(onTheFlyDecoderPipe,0600)!=0) { 
            return(false);
          }
          bf=onTheFlyDecoderPipe;       
          if (ThisAudio->converterFunction()==0) {
            onTheFlyAudioConverter=new AudioConversion(ThisAudio,ThisAudio->converter());         
          } else {
            onTheFlyAudioConverter=new AudioConversion(ThisAudio,ThisAudio->converter(),ThisAudio->converterFunction());         
          }
        }

        if (!ThisAudio->needImage()) {
          pcop->copyPipe(bf,onTheFlyTarget,getRealDuration()*2352);       
        } else {
          createName(onTheFlyAudioPipe,false,getRealDuration(),false);
          if (mkfifo(onTheFlyAudioPipe,0600)!=0) { 
            return(false);
          }
          pcop->copyPipe(onTheFlyAudioPipe,onTheFlyTarget,getRealDuration()*2352);                       
          onTheFlyABuildImage=new AudioBuildImage(ThisAudio,bf);
          onTheFlyABuildImage->disableAutoQuit();
        }
        break;        
      }
      return(false);
    case Source_Filesystem:
      if (!ThisISO->sizeValid()) {
        TrackDuration=ThisISO->calculateImage();
      }
      createName(onTheFlyRipPipe,false,TrackDuration,true);
      createName(onTheFlyTarget,false,TrackDuration,false);
      if (mkfifo(onTheFlyRipPipe,0600)!=0) { 
        return(false);
      }
      if (mkfifo(onTheFlyTarget,0600)!=0) { 
        return(false);
      }
      pcop->copyPipe(onTheFlyRipPipe,onTheFlyTarget,TrackDuration*2048);       
      break;
    case Source_CD:
      if (isAudio()) {
        AudioRipper *arip;
        const char *feedpipe;
        arip=AppChooser::getAudioRipper(true);
        if (arip==0) return(false);
        createName(onTheFlyRipPipe,false,getRealDuration(),true);
        createName(onTheFlyTarget,false,TrackDuration,false);
        if (mkfifo(onTheFlyRipPipe,0600)!=0) { 
          delete arip;
          return(false);
        }
        if (mkfifo(onTheFlyTarget,0600)!=0) { 
          delete arip;
          return(false);
        }
        feedpipe=onTheFlyRipPipe;
        if ( (ThisAudio!=0) && (ThisAudio->needImage()) )  {
          createName(onTheFlyAudioPipe,false,getRealDuration(),true);
          if (mkfifo(onTheFlyAudioPipe,0600)!=0) { 
            delete arip;
            return(false);
          }
          onTheFlyABuildImage=new AudioBuildImage(ThisAudio,onTheFlyRipPipe);
          onTheFlyABuildImage->disableAutoQuit();
          feedpipe=onTheFlyAudioPipe;
        }
        pcop->copyPipe(feedpipe,onTheFlyTarget,getRealDuration()*2352);       
        onTheFlyRipper=arip;
        break;
      } else {
        DataRipper *drip;
        drip=AppChooser::getDataRipper();
        if (drip==0) return(false);
        createName(onTheFlyRipPipe,false,TrackDuration,false);
        createName(onTheFlyTarget,false,TrackDuration,false);
        if (mkfifo(onTheFlyRipPipe,0600)!=0) { 
          delete drip;
          return(false);
        }
        if (mkfifo(onTheFlyTarget,0600)!=0) { 
          delete drip;
          return(false);
        }
        pcop->copyPipe(onTheFlyRipPipe,onTheFlyTarget,TrackDuration*2048);       
        onTheFlyDRipper=drip;
        break;
      }
      return(false);

  }
  return(true);
}

bool CDTrack::startOnTheFly(QObject *trigger) {
  if (onTheFlyTarget[0]==0) {
    return(false);
  }

  switch (TrackSource) {
    case Source_None:
      return(false);
    case Source_File:
      if (isAudio()) {
          
        if (onTheFlyAudioConverter!=0) {
          if ( (trigger!=0) && (onTheFlyABuildImage==0) ) {
            trigger->connect(onTheFlyAudioConverter,SIGNAL(processTerminated(ProcessInterface *,int)),
                    trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
          }
          onTheFlyAudioConverter->convertAudio(onTheFlyDecoderPipe);
        } 

        if (onTheFlyABuildImage!=0) {
          onTheFlyABuildImage->buildImage(onTheFlyAudioPipe);
          if ( (trigger!=0) ) {
            trigger->connect(onTheFlyABuildImage,SIGNAL(conversionDone(ProcessInterface *,int)),
                    trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
            trigger->connect(onTheFlyABuildImage,SIGNAL(processTerminated(ProcessInterface *,int)),
                    trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
          }
        }
        break;
      } else {
        return(false);
      }
    case Source_Filesystem:
      if (trigger!=0) {
        trigger->connect(ThisISO,SIGNAL(processTerminated(ProcessInterface *,int)),
                trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
      }
      if (ThisISO->makeImage(onTheFlyRipPipe)==0) {
        return(false);
      }
      break;
    case Source_CD:
      if (isAudio()) {
        if ( (trigger!=0) && (onTheFlyABuildImage==0) ) {
          trigger->connect(onTheFlyRipper,SIGNAL(processTerminated(ProcessInterface *,int)),
                  trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
        }
        if (onTheFlyRipper->ripAudio(CDTrackNum,onTheFlyRipPipe)==0) {
          return(false);
        }
        if (onTheFlyABuildImage!=0) {
          onTheFlyABuildImage->buildImage(onTheFlyAudioPipe);
          if ( (trigger!=0) ) {
            trigger->connect(onTheFlyABuildImage,SIGNAL(conversionDone(ProcessInterface *,int)),
                    trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
            trigger->connect(onTheFlyABuildImage,SIGNAL(processTerminated(ProcessInterface *,int)),
                    trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
          }
        }
        break;
      } else {
        if (trigger!=0) {
          trigger->connect(onTheFlyRipper,SIGNAL(processTerminated(ProcessInterface *,int)),
                  trigger,SLOT(onTheFlyTerminated(ProcessInterface *,int)));
        }
        if (onTheFlyDRipper->readDataTrack(CDTrackNum,onTheFlyRipPipe,TrackLBA,TrackDuration)==0) {
          return(false);
        }
       break;
      }
      return(false);

  }
  return(true);
}

bool CDTrack::checkOnTheFly(void) {
  return (onTheFlyTarget[0]!=0);
}

void CDTrack::cleanupOnTheFly(void) { 
  if (onTheFlyRipper!=0) {
    delete(onTheFlyRipper);
    onTheFlyRipper=0;
  }
  if (onTheFlyDRipper!=0) {
    delete(onTheFlyDRipper);
    onTheFlyDRipper=0;
  }
  if (onTheFlyAudioConverter!=0) {
    delete(onTheFlyAudioConverter);
    onTheFlyAudioConverter=0;
  }
  if (onTheFlyABuildImage!=0) {
    delete(onTheFlyABuildImage);
    onTheFlyABuildImage=0;
  }
  if (onTheFlyTarget[0]!=0) {
    remove(onTheFlyTarget);
    onTheFlyTarget[0]=0;
  }
  if (onTheFlyRipPipe[0]!=0) {
    remove(onTheFlyRipPipe);
    onTheFlyRipPipe[0]=0;
  }
  if (onTheFlyAudioPipe[0]!=0) {
    remove(onTheFlyAudioPipe);
    onTheFlyAudioPipe[0]=0;
  }
  if (onTheFlyDecoderPipe[0]!=0) {
    remove(onTheFlyDecoderPipe);
    onTheFlyDecoderPipe[0]=0;
  }

  if (ThisISO!=0) {
    ThisISO->cancelPressed();
  }
}

bool CDTrack::needDualDrive(void) {
  if (TrackSource==Source_CD) return(true);
  return(false);
}
