// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "AudioOptions.h"
#include "CDTrack.h"
#include "AudioFileInfo.h"
#include "AudioPlayImage.h"

#include "WaveView.h"
#include "AudioWaveData.h"
#include "AudioPosWidget.h"

#include "AudioOptions.moc"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kglobal.h>

#include <qwindowdefs.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qlabel.h>

#include <strings.h>
#include <stdio.h>

AudioOptions::AudioOptions( QWidget *parent, const char *name,
			    CDTrack *track,bool waveview):QWidget(parent,name) {
  QVBoxLayout *mainlayout;
  QHBoxLayout *controllayout;
  QHBoxLayout *phlayout;
  QLabel *ql1,*ql2;
  KConfig *config;

  config=kapp->config();
  advanced=0;

  ThisTrack=track;
  ThisAudio=track->getAudioFile();
  ThisPlayer=new AudioPlayImage(ThisAudio);
  absSize=ThisAudio->getAbsoluteDuration();
  frameScale=(double)(ThisAudio->rate())/(double)75;
  if (frameScale==0) frameScale=1;

  this->setCaption(i18n("KreateCD - audio options"));

  mainlayout=new QVBoxLayout(this);

  mainlayout->addStretch(1);
  PlaySlider=new QSlider(0,absSize,60*75,0,QSlider::Horizontal,this,0);
  PlaySlider->setSteps(5*75,60*75);
  PlaySlider->setTracking(true);
  {
    QSize qs;
    qs=PlaySlider->sizeHint();
    qs.setHeight(qs.height()+4);
    PlaySlider->setFixedHeight(qs.height());
  }
  mainlayout->addWidget(PlaySlider,1);
  mainlayout->addStretch(1);

  WaveViewer=0;
  startPosWidget=0;
  endPosWidget=0;


  if (waveview) {
    QGridLayout *lay1;
    QLabel *lab1,*lab2;

    WaveViewer=new WaveView(this,new AudioWaveData(ThisAudio));
    {
      QSize qs;
      qs=WaveViewer->sizeHint();
      WaveViewer->setMinimumHeight(WaveViewer->height());
    }
    WaveViewer->setStartMark(ThisAudio->startPos());
    WaveViewer->setEndMark(ThisAudio->endPos());
    WaveViewer->setPlayMark(0);

    mainlayout->addWidget(WaveViewer,10);
    mainlayout->addStretch(1);

    lay1=new QGridLayout(2,2);
    lab1=new QLabel(this);
    lab1->setText(i18n("Start position:"));
    lay1->addWidget(lab1,0,0);
    startPosWidget=new AudioPosWidget(this,ThisAudio->dataSamples(),ThisAudio->rate());
    {
      QSize qs;
      qs=startPosWidget->sizeHint();
      startPosWidget->setMinimumHeight(startPosWidget->height());
    }
    lay1->addWidget(startPosWidget,0,1);
    mainlayout->addLayout(lay1);
    mainlayout->addStretch(1);

    lab2=new QLabel(this);
    lab2->setText(i18n("End position:"));
    lay1->addWidget(lab2,1,0);
    endPosWidget=new AudioPosWidget(this,ThisAudio->dataSamples(),ThisAudio->rate());
    {
      QSize qs;
      qs=endPosWidget->sizeHint();
      endPosWidget->setMinimumHeight(endPosWidget->height());
    }
    lay1->addWidget(endPosWidget,1,1);

    startPosWidget->setPosition(ThisAudio->startPos());
    endPosWidget->setPosition(ThisAudio->endPos());
  }


  controllayout=new QHBoxLayout();
  mainlayout->addLayout(controllayout,1);
  mainlayout->addStretch(1);


  controllayout->addStretch(3);
  SetStartButton=new QPushButton(this);
  SetStartButton->setPixmap(BarIcon("kcd_set_start"));
  SetStartButton->setFixedSize(SetStartButton->sizeHint());
  controllayout->addWidget(SetStartButton,1,AlignCenter);

  controllayout->addStretch(2);
  BackwardButton=new QPushButton(this);
  BackwardButton->setPixmap(BarIcon("kcd_backward"));
  BackwardButton->setFixedSize(BackwardButton->sizeHint());
  controllayout->addWidget(BackwardButton,1,AlignCenter);

  controllayout->addStretch(1);
  PlayBeginButton=new QPushButton(this);
  PlayBeginButton->setPixmap(BarIcon("kcd_play_begin"));
  PlayBeginButton->setFixedSize(PlayBeginButton->sizeHint());
  controllayout->addWidget(PlayBeginButton,1,AlignCenter);

  controllayout->addStretch(1);
  PlayButton=new QPushButton(this);
  PlayButton->setPixmap(BarIcon("kcd_play"));
  PlayButton->setFixedSize(PlayButton->sizeHint());
  controllayout->addWidget(PlayButton,1,AlignCenter);

  controllayout->addStretch(1);
  StopButton=new QPushButton(this);
  StopButton->setPixmap(BarIcon("kcd_stop"));
  StopButton->setFixedSize(StopButton->sizeHint());
  controllayout->addWidget(StopButton,1,AlignCenter);

  controllayout->addStretch(1);
  ForwardButton=new QPushButton(this);
  ForwardButton->setPixmap(BarIcon("kcd_forward"));
  ForwardButton->setFixedSize(ForwardButton->sizeHint());
  controllayout->addWidget(ForwardButton,1,AlignCenter);
  controllayout->addStretch(2);

  SetStopButton=new QPushButton(this);
  SetStopButton->setPixmap(BarIcon("kcd_set_stop"));
  SetStopButton->setFixedSize(SetStopButton->sizeHint());
  controllayout->addWidget(SetStopButton,1);
  controllayout->addStretch(3);

  mainlayout->addStretch(1);
  ql1=new QLabel(this);
  ql1->setPixmap(BarIcon("kcd_balance"));
  ql1->setFixedSize(ql1->sizeHint());
  mainlayout->addWidget(ql1,1,AlignCenter);
  BalanceSlider=new QSlider(0,200,10,0,QSlider::Horizontal,this,0);
  BalanceSlider->setTracking(true);
  {
    QSize qs;
    qs=BalanceSlider->sizeHint();
    qs.setHeight(qs.height());
    BalanceSlider->setFixedHeight(qs.height());
  }
  mainlayout->addWidget(BalanceSlider,1);
  mainlayout->addStretch(1);

  ql2=new QLabel(this);
  ql2->setPixmap(BarIcon("kcd_volume"));
  ql2->setFixedSize(ql2->sizeHint());
  mainlayout->addWidget(ql2,1,AlignCenter);
  VolumeSlider=new QSlider(0,1000,2,0,QSlider::Horizontal,this,0);
  VolumeSlider->setTracking(true);
  {
    QSize qs;
    qs=VolumeSlider->sizeHint();
    qs.setHeight(qs.height());
    VolumeSlider->setFixedHeight(qs.height());
  }
  mainlayout->addWidget(VolumeSlider,1);
  mainlayout->addStretch(1);

  if (waveview) {
    mainlayout->addStretch(5);
    phlayout=new QHBoxLayout();
 
    advanced=new QPushButton(this);
    advanced->setText(i18n("Advanced view >>>"));
    advanced->setMinimumSize(advanced->sizeHint());
    phlayout->addStretch(10);
    phlayout->addWidget(advanced,1);
    phlayout->addStretch(1);
    mainlayout->addLayout(phlayout,1);

    config=kapp->config();
    config->setGroup("GUI");
    if (config->readBoolEntry("AudioOptionsAdvancedView",false)) {
      WaveViewer->show();
      startPosWidget->show();
      endPosWidget->show();
      PlaySlider->hide();
    } else {
      WaveViewer->hide();
      startPosWidget->hide();
      endPosWidget->hide();
      PlaySlider->show();
    }
 
    connect(advanced,SIGNAL(clicked()),this,SLOT(advancedView()));
  }

  BalanceSlider->setValue((int)((ThisAudio->balance()*100)+100));
  VolumeSlider->setValue((int)((ThisAudio->boost()*100)));

  connect(PlayButton,SIGNAL(clicked()),this,SLOT(doPlay()));
  connect(PlayBeginButton,SIGNAL(clicked()),this,SLOT(doPlayBegin()));
  connect(StopButton,SIGNAL(clicked()),ThisPlayer,SLOT(stopAudio()));
  connect(ForwardButton,SIGNAL(clicked()),this,SLOT(doForward()));
  connect(BackwardButton,SIGNAL(clicked()),this,SLOT(doBackward()));
  connect(SetStartButton,SIGNAL(clicked()),this,SLOT(setStart()));
  connect(SetStopButton,SIGNAL(clicked()),this,SLOT(setStop()));
  connect(ThisPlayer,SIGNAL(audioPlayPosition(int)),
          this,SLOT(positionChanged(int)));
  connect(PlaySlider,SIGNAL(valueChanged(int)),
          this,SLOT(sliderMove(int)));
  connect(PlaySlider,SIGNAL(sliderPressed(void)),
          this,SLOT(sliderPress(void)));
  connect(PlaySlider,SIGNAL(sliderReleased(void)),
          this,SLOT(sliderRelease(void)));
  connect(BalanceSlider,SIGNAL(valueChanged(int)),
          this,SLOT(changeBalance(int)));
  connect(VolumeSlider,SIGNAL(valueChanged(int)),
          this,SLOT(changeBoost(int)));

  if (WaveViewer!=0) { 
    connect(WaveViewer,SIGNAL(playChanged(long int)),
            this,SLOT(wavePlayChange(long int)));
    connect(WaveViewer,SIGNAL(startChanged(long int)),
            this,SLOT(waveStartChange(long int)));
    connect(WaveViewer,SIGNAL(endChanged(long int)),
            this,SLOT(waveEndChange(long int)));
    connect(startPosWidget,SIGNAL(positionChanged()),
            this,SLOT(aposStartChanged()));
    connect(endPosWidget,SIGNAL(positionChanged()),
            this,SLOT(aposEndChanged()));
  }


  isPressed=0;
  lastPosition=0;

  mainlayout->activate();
}


AudioOptions::~AudioOptions(void) {
  if (ThisPlayer!=0) {
    ThisPlayer->stopAudio();
    delete(ThisPlayer);
  }
}

void AudioOptions::positionChanged(int frame) {
  if (isPressed) return;
  if (frame>absSize) return;
  lastPosition=frame;
  PlaySlider->setValue(frame);
  if (WaveViewer!=0) WaveViewer->setPlayMark((long)(frame*frameScale));
}

void AudioOptions::sliderPress(void) {
  isPressed=1;
}

void AudioOptions::sliderRelease(void) {
  isPressed=0;
  ThisPlayer->setAudioPosition(lastPosition);
  if (WaveViewer!=0) WaveViewer->setPlayMark((long)(lastPosition*frameScale));
}


void AudioOptions::sliderMove(int frame) {
  lastPosition=frame;
}

void AudioOptions::doPlay(void) {
  if (lastPosition==absSize) lastPosition=-1;
  ThisPlayer->playAudio(lastPosition);
}

void AudioOptions::doPlayBegin(void) {
  ThisPlayer->playAudio(-1);
}

void AudioOptions::doForward(void) {
  if ((lastPosition+15*75)>=absSize) return;
  lastPosition+=15*75;
  PlaySlider->setValue(lastPosition);
  if (WaveViewer!=0) WaveViewer->setPlayMark((long)(lastPosition*frameScale));
  ThisPlayer->setAudioPosition(lastPosition);
}

void AudioOptions::doBackward(void) {
  if ((lastPosition<15*75)) {
    lastPosition=0;
  } else {
    lastPosition-=15*75;
  }
  PlaySlider->setValue(lastPosition);
  if (WaveViewer!=0) WaveViewer->setPlayMark((long)(lastPosition*frameScale));
  ThisPlayer->setAudioPosition(lastPosition);
}

void AudioOptions::changeBalance(int balance) {
  float rbalance;
  rbalance=(float)(balance-100)/100;
  ThisPlayer->setAudioBalance(rbalance);
  ThisAudio->setBalance(rbalance);
  emit(needUpdate());
}

void AudioOptions::changeBoost(int boost) {
  float rboost;
  rboost=(float) boost/100;
  ThisPlayer->setAudioBoost(rboost);
  ThisAudio->setBoost(rboost);
  emit(needUpdate());
}


void AudioOptions::setStart(void) {
  ThisAudio->setStartPos((long)(lastPosition*frameScale));
  if (WaveViewer!=0) WaveViewer->setStartMark(ThisAudio->startPos());
  if (startPosWidget!=0) startPosWidget->setPosition(ThisAudio->startPos());
  emit(needUpdate());
}

void AudioOptions::setStop(void) {
  ThisAudio->setEndPos((long)(lastPosition*frameScale));
  if (WaveViewer!=0) WaveViewer->setEndMark(ThisAudio->endPos());
  if (endPosWidget!=0) endPosWidget->setPosition(ThisAudio->endPos());
  emit(needUpdate());
}

void AudioOptions::waveStartChange(long sample) {
  ThisAudio->setStartPos(sample);
  if (startPosWidget!=0) startPosWidget->setPosition(ThisAudio->startPos());
  emit(needUpdate());
}

void AudioOptions::waveEndChange(long sample) {
  ThisAudio->setEndPos(sample);
  if (endPosWidget!=0) endPosWidget->setPosition(ThisAudio->endPos());
  emit(needUpdate());
}

void AudioOptions::wavePlayChange(long sample) {
  lastPosition=(long)(sample/frameScale);
  ThisPlayer->setAudioPosition(lastPosition);
  PlaySlider->setValue(lastPosition);
}

void AudioOptions::advancedView(void) {
  KConfig *config;
 
  WaveViewer->isHidden()?WaveViewer->show():WaveViewer->hide();
  WaveViewer->isHidden()?startPosWidget->hide():startPosWidget->show();
  WaveViewer->isHidden()?endPosWidget->hide():endPosWidget->show();
  WaveViewer->isHidden()?PlaySlider->show():PlaySlider->hide();
  config=kapp->config();
  config->setGroup("GUI");
  config->writeEntry("AudioOptionsAdvancedView",!WaveViewer->isHidden());
  updateGeometry();
}

void AudioOptions::aposStartChanged(void) {
  ThisAudio->setStartPos(startPosWidget->position());
  if (WaveViewer!=0) WaveViewer->setStartMark(ThisAudio->startPos());
  emit(needUpdate());
}

void AudioOptions::aposEndChanged(void) {
  ThisAudio->setEndPos(endPosWidget->position());
  if (WaveViewer!=0) WaveViewer->setEndMark(ThisAudio->endPos());
  emit(needUpdate());
}
