//
// C++ Interface: kpgfunction
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef KPGFUNCTION_H
#define KPGFUNCTION_H

// include files for Qt
#include <qvaluelist.h>

#include "kpgobject.h"

class KPGFunctionsFolder;


/**
  * Function argument
  *
  * @author Lumir Vanek
  */
class KPGFunctionArgument
{
public:
    
    // Argument mode
    enum EMode 
    { 	
        input = 0,
        output,
        inout,
        unknown
    };
    
	KPGFunctionArgument() {;}
	KPGFunctionArgument(unsigned int, const QString &, const QString &, const QString &);
    KPGFunctionArgument(unsigned int, const QString &, EMode, const QString &);
    
	unsigned int sequence() const { return m_iSequence; }
	const QString &name() const { return m_strName; }
    EMode mode() const { return m_eMode; }
	const QString &typName() const { return m_strTypName; }
    
    const QString modeAsString() const 
    { 
        switch(m_eMode)
        {
            case input: return "IN";
            case output: return "OUT";
            case inout: return "INOUT";
            default: return "UNKNOWN";
        }
    }

protected:
	unsigned int m_iSequence; // Starts from 1
	QString m_strName;
    EMode m_eMode;
	QString m_strTypName;
};

typedef QValueList<KPGFunctionArgument> ListFunctionArguments;

/**
  * Function argument vith value
  *
  * @author Lumir Vanek
  */
class KPGFunctionArgumentWithValue : public KPGFunctionArgument
{
public:	
	KPGFunctionArgumentWithValue() {;}
	KPGFunctionArgumentWithValue(unsigned int iSequence, const QString & strName, const QString & strMode, const QString & strTypName, const QString & strValue)
	: KPGFunctionArgument(iSequence, strName, strMode, strTypName) 
	{
		m_strValue = strValue;
	}
    
    KPGFunctionArgumentWithValue(unsigned int iSequence, const QString & strName, EMode eMode, const QString & strTypName, const QString & strValue)
    : KPGFunctionArgument(iSequence, strName, eMode, strTypName) 
    {
        m_strValue = strValue;
    }
	
	const QString &value() const { return m_strValue; }

protected:
	QString m_strValue;
};

typedef QValueList<KPGFunctionArgumentWithValue> ListFunctionArgumentsWithValues;


/**
  * Database function
  *
  * @author Lumir Vanek
  */
class KPGFunction : public KPGObject
{
public:
    KPGFunction(KPGFunctionsFolder *, const QString, pqxx::oid);
	KPGFunction(KPGFunctionsFolder *, KPGFunction *, const QString, pqxx::oid);
    ~KPGFunction();

	virtual eNodeType type() const { return nodeFunction; }
		
	// Get SQL object type 
  	virtual const char * getObjectType() const { return "FUNCTION"; }
  	
	void setProperties(const pqxx::result::tuple &, const KPGConnection *pConnection);
	void setArgTypes(const QString strArgTypes) { m_strArgTypes = strArgTypes; }
	void setSourceCode(const QString &strSourceCode);
	void setListArguments(const QString &, const QString &, const QString &, const QString &);
	
	// Refresh item, ald also childs objects
	void refresh() throw(const KPGSqlException &) { return refreshItem(); }
	
	// Refresh only this item, not childs
	void refreshItem() throw(const KPGSqlException &);
    
    /*
	 * Functions for accessing properties
	 */
	bool isAggregate() const { return m_bIsAgg; }	 
	bool isSecdef() const { return m_bIsSecdef; }
	bool isStrict() const { return m_bIsStrict; }
	bool returnSet() const { return m_bReturnSet; }
	const QString & getVolatile() const { return m_strVolatile; }
	unsigned int nArgs() const { return m_uiNArgs; }
	const QString & bin() const { return m_strBin; }
	const QString langName() const { return m_strLanName; }
	const QString & argTypes() const { return  m_strArgTypes; }
	const QString & sourceCode() const { return m_strSourceCode; }
	const QString & returnTypName() const { return m_strReturnTypName; }
	pqxx::oid oidReturnType() const { return m_oidReturnType; }
	const QString & acl() const { return m_strACL; }
	const QString & owner() const { return m_strFuncOwner; }
		    
    // Return arguments. Old variant, based on pg_proc.proargtypes
    const QString argumentTypes() const { return "(" + argTypes() + ")"; }
    
	// Return arguments, with names and IN/OUT info. 
	const QString arguments() const;
	
    // Return name with arguments. New variant, based on info since PostgreSQL 8.1
    const QString nameWithArguments() const;
    
    // Get list of arguments
    const ListFunctionArguments & listArguments() const { return m_listArguments; }
    
protected:
	
	/*
	 * Properties
	 */
	bool m_bIsAgg;
	bool m_bIsSecdef;
	bool m_bIsStrict;
	bool m_bReturnSet;
	QString m_strVolatile;
	unsigned int m_uiNArgs;
	QString m_strBin;
	QString m_strACL;
	pqxx::oid m_oidReturnType;
	QString m_strReturnTypName;
	QString m_strLanName;
	QString m_strFuncOwner;	
	
	// This includes only input arguments (including INOUT arguments), and thus represents the call signature of the function.
	QString m_strArgTypes;
	
	// It might be the actual source code of the function for interpreted languages, a link symbol, a file name, or just about anything else, depending on the implementation language/call convention.
	QString m_strSourceCode;
	
	// List of argument informations, available since PostgreSQL 8.1
    ListFunctionArguments m_listArguments;
};

#endif
