//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007,2008 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <sys/types.h>
#include <iostream>
#include <assert.h>

#include "silcaccount.h"
#include "silcchannelcontact.h"
#include "silcchannelcontactinfowidget.h"
#include "silcbuddycontact.h"
#include "silcmessagemanager.h"
#include "silcbuddycliententry.h"
#include "silcstatusfactory.h"

#include <kopetemetacontact.h>
#include <kopetemessage.h>
#include <kopetechatsession.h>
#include <kopetechatsessionmanager.h>
#include <kopeteuiglobal.h>
#include <ui/kopeteview.h>
#include <kdebug.h>
#include <qfileinfo.h>
#include <kfiledialog.h>

#include <klocale.h>
#include <kmessagebox.h>


// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef SilcTK::SilcUInt32 SilcUInt32;
typedef SilcTK::SilcUInt8 SilcUInt8;

SilcChannelContact::SilcChannelContact(SilcAccount *account,
				       const QString &channel, 
				       Kopete::MetaContact *meta,
				       const QString &icon)
  : SilcContact(account, QString("#%1").arg(channel), meta, icon),
    _channelEntry(NULL), _allowRichText(false), _autoJoin(false)
{
  modeToBeSet = 0;
  setNickName(channel);

  QObject::connect
    (this, SIGNAL(onlineStatusChanged(Kopete::Contact *,
				      const Kopete::OnlineStatus &, 
				      const Kopete::OnlineStatus &)),
     this, SLOT(slotOnlineStatusChanged(Kopete::Contact *,
					const Kopete::OnlineStatus &,
					const Kopete::OnlineStatus &)));

  if(account->isConnected())
    setOnlineStatus(SilcProtocol::protocol()->statusOnlineChannel);
}
 
SilcChannelContact::~SilcChannelContact() { }

Kopete::ChatSession *
SilcChannelContact::manager(Kopete::Contact::CanCreateFlags flags)
{
  Kopete::ChatSession *session =
    SilcContact::manager(Kopete::Contact::CannotCreate);

  if(! session && (flags & Kopete::Contact::CanCreate)) {
    session = SilcContact::manager(flags);

    if(_channelEntry)
      setNickNameForMode(_channelEntry->mode);

    // update myselves status, this is, show op crown, etc. if we got one
    SilcBuddyContact *b = static_cast<SilcBuddyContact *>(account()->myself());
    updateBuddyOnlineStatus(b);
  }

  return session;
}

/**
 * @brief join the channel
 *
 * We need this wrapper around the more specialized join function in order
 * to overwrite the join method from the SilcContact class.
 */
void
SilcChannelContact::join(void)
{
  join(false, false, QString::null);
}

void
SilcChannelContact::join(bool founder, bool auth, const QString &password)
{
  if(_joined) return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->isConnected()) {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("Sorry, you cannot join channels unless "
			    "you go online"), i18n("SILC Plugin"));
    return;
  }

  QString arg = QString("JOIN %1 %2").arg(nickName()).arg(password);

  if(founder)
    arg.append(" -founder");

  if(auth)
    arg.append(" -auth");

  account->sendSilcCommand(arg);
  setFileCapable(true);
}



void
SilcChannelContact::setChannelEntry(SilcTK::SilcChannelEntry e)
{
  _channelEntry = e;
  if(e) e->context = this;
  
  _joined = e != NULL;

  // We do this, because of inviting a buddy into a ad-hoc channel needs
  // channel modes
  if(modeToBeSet && e->mode != modeToBeSet) {
    _channelEntry->mode = modeToBeSet;
    commitModeChange();
    modeToBeSet = 0;
  }

  foreach(const SilcBuddyContact *buddy, toInvite)
    invite(buddy);
  toInvite.clear();
}


void
SilcChannelContact::slotSendMessage(Kopete::Message &msg,
				    Kopete::ChatSession *session)
{
  if(session != manager())
    return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->conn()) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
       i18n("Unable to send this message now. The protocol is currently "
	    "offline and does not support offline sending."),
       i18n( "User is Not Reachable"));
    return;
  }

  // get plain text message ...
  SilcTK::SilcMessageFlags flags = SILC_MESSAGE_FLAG_UTF8;
  unsigned char *buf = NULL;
  SilcTK::SilcUInt32 buflen = 0;
  QByteArray plaintext;

  if(account->signChannelMessages())
    flags |= SILC_MESSAGE_FLAG_SIGNED;

  if(allowRichText()) {
    SilcTK::SilcMime mime = getMessageAsMime(msg);
    buf = SilcTK::silc_mime_encode(mime, &buflen);
    SilcTK::silc_mime_free(mime);

    flags |= SILC_MESSAGE_FLAG_DATA;
  }
  else {
    plaintext = msg.plainBody().toUtf8();
    buf = (unsigned char *) plaintext.constData();
    buflen = plaintext.length();

    // use of rich text is forbidden, reset message to plain
    // (so our channel log doesn't show any markup as well)
    msg.setPlainBody(msg.plainBody());
  }

  prettyPrintMessage(msg, flags);

  // pass message to libsilc ...
  SilcTK::silc_client_send_channel_message
    (account->client(), account->conn(), channelEntry(), NULL, flags,
     account->sha1hash, buf, buflen);

  if(allowRichText())
    SilcTK::silc_free(buf);

  // append message locally ...
  session->appendMessage(msg);
  session->messageSucceeded();

  // reset idle time of ourself
  account->myself()->setIdleTime(1);
}

void
SilcChannelContact::slotOnlineStatusChanged(Kopete::Contact *,
					    const Kopete::OnlineStatus &status,
					    const Kopete::OnlineStatus &old)
{
  if(status == SilcProtocol::protocol()->statusOffline
     || status.status() == Kopete::OnlineStatus::Unknown) {
    // assume we're offline, thus left all channels ...
    setChannelEntry(NULL);
    return;
  }

  if(old != SilcProtocol::protocol()->statusOffline
     && old.status() != Kopete::OnlineStatus::Unknown) return;

  if(manager() || autoJoin())
    // Automatically join the channel, if there's either a window left
    // open for it or it is marked to be automatically joined by the user.
    join();
}


void
SilcChannelContact::leave(void)
{
  SilcAccount *account = static_cast<SilcAccount *>(this->account());
  account->sendSilcCommand(QString("LEAVE %1").arg(nickName()));
  setChannelEntry(NULL); // libsilc frees the channel entry after leaving
  setFileCapable(false);
}


bool
SilcChannelContact::isJoined(SilcBuddyContact *buddy)
{
  // channelEntry() is not set during the connection phase ...
  if(! channelEntry()) return false;

  return buddy->clientEntries()(this) != NULL;
}


bool
SilcChannelContact::isCUModeGeneric(SilcBuddyContact *buddy, int modemask)
{
  if(! channelEntry())
    return false;

  foreach(SilcTK::SilcClientEntry ce, buddy->clientEntries()) {
    SilcTK::SilcChannelUser cu = 
      SilcTK::silc_client_on_channel(channelEntry(), ce);
    if(! cu) continue;

    if(cu->mode & modemask)
      return true;
  }

  return false;
}


static SilcTK::SilcBool 
silc_reply_dummy(SilcTK::SilcClient, SilcTK::SilcClientConnection,
		 SilcTK::SilcCommand, SilcTK::SilcStatus, SilcTK::SilcStatus,
		 void *, va_list)
{
  return 0;
}


void
SilcChannelContact::setCUModeGeneric(SilcBuddyContact *buddy, bool status,
				   int mask)
{
  // @fixme don't just mark the first joined ce op, but mark 'em all
  SilcTK::SilcClientEntry ce = buddy->clientEntries()(this);
  SilcTK::SilcChannelUser cu =
    SilcTK::silc_client_on_channel(channelEntry(), ce);
  if(! cu) return;

  // get current mode ...
  SilcTK::SilcUInt32 mode = cu->mode;

  // update mode ...
  mode &= ~mask;
  if(status) mode |= mask;

  // prepare to send the command finally ...
  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
  SilcTK::SilcBuffer idp_client =
    SilcTK::silc_id_payload_encode(&ce->id, SILC_ID_CLIENT);

  unsigned char modebuf[4];
  SILC_PUT32_MSB(mode, modebuf);

  // finally send the command ...
  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_CUMODE, silc_reply_dummy,
				   NULL, 3,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, modebuf, sizeof(modebuf),
				   3, idp_client->data, 
                                   silc_buffer_len(idp_client));
}


void
SilcChannelContact::kick(SilcBuddyContact *buddy, QString kickMsg)
{
  if(! channelEntry())
    return;

  SilcAccount *acc = static_cast<SilcAccount *>(account());
  QByteArray msg = kickMsg.toUtf8();
  
  foreach(SilcTK::SilcClientEntry ce, buddy->clientEntries().joinedTo(this)) {
    // prepare to send the command finally ...
    SilcTK::SilcBuffer idp_channel = 
      SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
    SilcTK::SilcBuffer idp_client =
      SilcTK::silc_id_payload_encode(&ce->id, SILC_ID_CLIENT);

    // finally send the command ...
    if(msg.isEmpty())
      SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				       SILC_COMMAND_KICK, silc_reply_dummy,
				       NULL, 2,
				       1, silc_buffer_datalen(idp_channel), 
				       2, silc_buffer_datalen(idp_client));
    else
      SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				       SILC_COMMAND_KICK, silc_reply_dummy,
				       NULL, 3,
				       1, silc_buffer_datalen(idp_channel), 
				       2, silc_buffer_datalen(idp_client),
				       3, msg.data(), msg.size());

    // @fixme free idp_channel and idp_client with silc_buffer_free
  }
}


void
SilcChannelContact::updateBuddyOnlineStatus(SilcBuddyContact *buddy)
{
  // check whether the assigned status is correct for this channel,
  // i.e. check whether we don't have to use the Op'd variant for example
  const Kopete::OnlineStatus chuser_status =
    SilcStatusFactory::self()->getBuddyOnlineStatus(this, buddy);

  if(! manager()->members().contains(buddy)
     && buddy != account()->myself()) {
    manager()->addContact(buddy, chuser_status);
    return;
  }

  if(manager()->contactOnlineStatus(buddy) != chuser_status)
    manager()->setContactOnlineStatus(buddy, chuser_status);
}

void 
SilcChannelContact::silc_channel_message(SilcTK::SilcClient /* client */, 
					 SilcTK::SilcClientConnection /*conn*/,
					 SilcTK::SilcClientEntry sender, 
					 SilcTK::SilcChannelEntry channel,
					 SilcTK::SilcMessagePayload payload,
					 SilcTK::SilcMessageFlags flags, 
					 const unsigned char *message,
					 SilcTK::SilcUInt32 message_len)
{
  SilcChannelContact *ch = (SilcChannelContact *) channel->context;
  SilcBuddyContact *buddy = (SilcBuddyContact *) sender->context;

  if(! ch) {
    std::cerr << "cannot find SilcChannel structure for "
	      << channel->channel_name << std::endl;
    return;
  }

  // if we haven't heard of the buddy, use the channel name ... hmm.
  // @todo we ought to create a new buddy contact instead
  if(! buddy) {
    return;
  }

  // reset idletime of message sender
  buddy->setIdleTime(1);

  /* If the messages is digitally signed, verify it, if possible. */
  SignatureStatus sigstat = Unknown;
  if (flags & SILC_MESSAGE_FLAG_SIGNED)
    sigstat = buddy->verifySignature(payload);

  // convert Utf8 stuff depending on whether SILC_MESSAGE_FLAG_UTF8 is set
  QString text;
  if(flags & SILC_MESSAGE_FLAG_UTF8)
    text = QString::fromUtf8((const char *) message, message_len);
  else if (flags & SILC_MESSAGE_FLAG_DATA);
  else
    text = QString::fromLatin1((const char *) message, message_len);

  Kopete::Message msg;
  if(flags & SILC_MESSAGE_FLAG_NOTICE) {
    msg = Kopete::Message(buddy, ch->manager()->members());
    msg.setPlainBody(QString("%1 -*- %2").arg(buddy->nickName()).arg(text));
    msg.setDirection(Kopete::Message::Internal);
    msg.setType(Kopete::Message::TypeAction);
  }


  else if (flags & SILC_MESSAGE_FLAG_DATA) {
    /* SilcMimeMessage */
    QStringList *filenames;
    SilcTK::SilcMime tmp = SilcTK::silc_mime_decode(NULL, message, 
                                                    message_len);
    /* Assemble mime partials */
    SilcTK::SilcMime mime = buddy->mime_asm(tmp);
    if (!mime) return;

    QString type = SilcTK::silc_mime_get_field(mime, "Content-Type");
    if(type.isEmpty()) goto mimeout;

    if (type.left(21).compare("multipart/alternative") == 0) {
      msg = Kopete::Message(buddy, ch->manager()->members());
      msg.setDirection(Kopete::Message::Inbound);
      msg.setType(Kopete::Message::TypeNormal);

      buddy->mimeAlternateToMsg(msg, mime, ch->allowRichText());
      ch->manager()->appendMessage(msg);
    }
    else {
      filenames = buddy->saveMime(mime);
      for ( QStringList::Iterator it = filenames->begin(); 
            it != filenames->end(); ++it ) {
        msg = Kopete::Message(buddy, ch->manager()->members());
	msg.setHtmlBody(buddy->mimeDisplayMessage(*it));
	msg.setDirection(Kopete::Message::Inbound);
	msg.setType(Kopete::Message::TypeNormal);

        prettyPrintMessage(msg, flags, sigstat);
        ch->manager()->appendMessage(msg);
      }
      delete filenames;
    }
mimeout:
    SilcTK::silc_mime_free(mime);
    return;
  }
  else {
    msg = Kopete::Message(buddy, ch->manager()->members());
    msg.setPlainBody(text);
    msg.setDirection(Kopete::Message::Inbound);
    msg.setType((flags & SILC_MESSAGE_FLAG_ACTION)
		? Kopete::Message::TypeAction
		: Kopete::Message::TypeNormal);
  }

  prettyPrintMessage(msg, flags, sigstat);

  ch->manager()->appendMessage(msg);
}


void 
SilcChannelContact::silc_channel_message(SilcTK::SilcClient client,
					 SilcTK::SilcClientConnection conn,
					 SilcTK::SilcClientEntry sender,
					 SilcTK::SilcChannelEntry channel,
					 SilcTK::SilcMessagePayload payload,
					 SilcTK::SilcChannelPrivateKey,
					 SilcTK::SilcMessageFlags flags, 
					 const unsigned char *msg,
					 SilcTK::SilcUInt32 msg_len)
{
  SilcChannelContact::silc_channel_message(client, conn, sender, channel, 
					   payload, flags, msg, msg_len);
} 


void SilcChannelContact::setNickNameForMode(int mode) {
  QString _mode = "";
  
  if(mode & SILC_CHANNEL_MODE_PRIVATE) {
    _mode.append("p");
  }	

  if(mode & SILC_CHANNEL_MODE_SECRET) {
    _mode.append("s");
  }       

  if(mode & SILC_CHANNEL_MODE_PRIVKEY) {
    _mode.append("K");
  }      
  
  if(mode & SILC_CHANNEL_MODE_INVITE) {
    _mode.append("i");
  }
  
  if(mode & SILC_CHANNEL_MODE_CHANNEL_AUTH) {
    _mode.append("C");
  }
  
  if(_mode != "") {
    _mode = " ["+_mode+"]";
  }
 
 manager()->setDisplayName(nickName().append(_mode));
}



void 
SilcChannelContact::invite(const SilcBuddyContact *buddy)
{
  if(!_joined) {
    toInvite.append(buddy);
    return;
  }
  
  SilcTK::SilcClientEntry active_ce = buddy->clientEntries()();

  // prepare to send the command finally ...
  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
  SilcTK::SilcBuffer idp_client =
    SilcTK::silc_id_payload_encode(&active_ce->id, SILC_ID_CLIENT);

  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_INVITE, NULL, NULL, 2,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, idp_client->data, 
                                   silc_buffer_len(idp_client));
}



void 
SilcChannelContact::slotUserInfo(void)
{
  new SilcChannelContactInfoWidget(this);
}


void
SilcChannelContact::setTopic(QString &newTopic)
{
  QString oldTopic = topic();
  if(!oldTopic.isNull() && newTopic.compare(oldTopic) == 0)
    return;

  account()->sendSilcCommand
    (QString("TOPIC %1 %2").arg(nickName()).arg(newTopic));
}

unsigned int 
SilcChannelContact::userLimit(void) const
{
  if(_channelEntry)
    return (unsigned int) _channelEntry->user_limit;
  return 0;
}


void
SilcChannelContact::setPrivate(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_PRIVATE;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_PRIVATE;
    return;
  }
  else if (state == isPrivate())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_PRIVATE;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_PRIVATE;

  commitModeChange();
}


void
SilcChannelContact::setSecret(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SECRET;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SECRET;
    return;
  }
  else if (state == isSecret())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SECRET;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SECRET;

  commitModeChange();
}

void
SilcChannelContact::setTopicOperatorOnly(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_TOPIC;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_TOPIC;
    return;
  }
  else if (state == isTopicOperatorOnly())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_TOPIC;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_TOPIC;

  commitModeChange();
}

void
SilcChannelContact::setUserLimit(unsigned int userlimit)
{
  if(! channelEntry()) 
    return;

  if (userlimit) {
    if(userlimit != userLimit())
      account()->sendSilcCommand
        (QString("CMODE %1 +l %2").arg(nickName()).arg(userlimit));
  }
  else if (userLimit())
    account()->sendSilcCommand
      (QString("CMODE %1 -l").arg(nickName()));

}

void
SilcChannelContact::setInviteOnly(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_INVITE;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_INVITE;
    return;
  }
  else if (state == isInviteOnly())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_INVITE;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_INVITE;

  commitModeChange();
}

void 
SilcChannelContact::setPassphrase(QString passphrase)
{
  if (! channelEntry()) 
    return;

  account()->sendSilcCommand(QString("CMODE %1 -a").arg(nickName()));

  if(! passphrase.isEmpty()) 
    account()->sendSilcCommand
      (QString("CMODE %1 +a %2").arg(nickName()).arg(passphrase));

}

void
SilcChannelContact::setFounderAuth(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_FOUNDER_AUTH;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_FOUNDER_AUTH;
    return;
  }
  else if (state == isFounderAuth())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_FOUNDER_AUTH;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_FOUNDER_AUTH;

  commitModeChange();
}



void
SilcChannelContact::setSilenceUser(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SILENCE_USERS;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SILENCE_USERS;
    return;
  }
  else if (state == isSilenceUser())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SILENCE_USERS;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SILENCE_USERS;

  commitModeChange();
}


void
SilcChannelContact::setSilenceOperator(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SILENCE_OPERS;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SILENCE_OPERS;
    return;
  }
  else if (state == isSilenceOperator())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SILENCE_OPERS;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SILENCE_OPERS;

  commitModeChange();
}


void
SilcChannelContact::commitModeChange(void)
{
  assert(channelEntry());

  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);

  unsigned char modebuf[4];
  SILC_PUT32_MSB(channelEntry()->mode, modebuf);

  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_CMODE, NULL, NULL, 2,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, modebuf, sizeof(modebuf));
}

void 
SilcChannelContact::sendFile(const KUrl &sourceURL,
			   const QString & /* fileName */, uint /* fileSize */) 
{
  QString filePath;

  if(! sourceURL.isValid())
    filePath = KFileDialog::getOpenFileName(KUrl(), "*", 0L,
					    i18n("Kopete File Transfer"));
  else
    filePath = sourceURL.path();

  QFile file(filePath);
  if(! file.exists())
    return;
  sendFileAsMime(filePath);
}

void 
SilcChannelContact::sendFileAsMime(const QString &fileName)
{
  int chunks = 0;
  SilcTK::SilcBuffer buffer;
  QFile file(fileName);

  /* Sending Chunks */
  SilcTK::SilcDList parts = getFileAsMime(fileName);
  SilcTK::silc_dlist_start(parts);
  while ((buffer = (SilcTK::SilcBuffer)SilcTK::silc_dlist_get(parts))
	 != SILC_LIST_END) { 
    chunks ++;
    SilcTK::silc_client_send_channel_message
      (account()->client(), account()->conn(), channelEntry(), NULL, 
       SILC_MESSAGE_FLAG_DATA, account()->sha1hash, 
       (unsigned char*)buffer->data, silc_buffer_len(buffer));
  }
  SilcTK::silc_mime_partial_free(parts);
  
  Kopete::Message msg = 
    Kopete::Message(account()->myself(), manager()->members());
  msg.setHtmlBody(account()->myself()->mimeDisplayMessage(fileName, chunks));
  msg.setDirection(Kopete::Message::Outbound);
  msg.setType(Kopete::Message::TypeNormal);

  manager()->appendMessage(msg);
}

void 
SilcChannelContact::serialize(QMap<QString, QString> &serializedData,
			      QMap<QString, QString>&)
{
  serializedData["allowRichText"] = allowRichText() ? "yes" : "no";
  serializedData["autoJoin"] = autoJoin() ? "yes" : "no";
}

#include "silcchannelcontact.moc"

