#include <dlggrpchat.h>

#include <kmessagebox.h>

#include <agent.h>
#include <userpresence.h>

DlgGrpChat::DlgGrpChat(QString room, QString server, QString nick)
        : QWidget( 0, 0 )
{
        /* Store these, I will want to know them */
        _room = room;
        _server = server;
        _nick = nick;

        /* Lets setup the GUI first */
        setIcon( KGlobal::iconLoader()->loadIcon("groupchat_dlg", KIcon::User) );

        _layout = new QGridLayout( this, 1, 1, 5, -1 );

        _main = new QSplitter( this );
        _main->setOrientation( Vertical );
        _layout->addWidget( _main, 0, 0 );

        _top = new QSplitter( _main );
        _main->setResizeMode( _top, QSplitter::Stretch );

        _conversation = new ConversationView( _top );
        _top->setResizeMode( _conversation, QSplitter::Stretch );

        _members = new QListView( _top );
        _members->setMinimumWidth( 100 );
        _members->addColumn( "Members" );

        _input = new WrapEdit( _main );

        resize( 550, 400 );
        QValueList<int> sizes;
        sizes.append( 475 );
        sizes.append( 75 );
        _top->setSizes( sizes );
        sizes.clear();
        sizes.append( 350 );
        sizes.append( 50 );
        _main->setSizes( sizes );

        QString newCaption;
        newCaption = "Groupchat: " + _room + " on " + jabber->getAgent( _server )->name() + " - Konverse";
        setCaption( newCaption );

        show();


        /* Ok, now we can start talking to the jabber server */
        /* This will make it so we recieve presence info */
        _u = jabber->fetchUser( room + "@" + server );
        connect( _u, SIGNAL(presenceChanged(User *, UserPresence *)), this, SLOT(presenceChanged(User *, UserPresence *)) );

        /* Inform the message bin we want these messages */
        setJID( room + "@" + server );
        setType( "groupchat" );
        jabber->messageBin()->registerInterest( this );

        /* This joins us to the chat room */
        XMLNode joinRequest( "presence" );
        joinRequest.setAttribute( "to", room + "@" + server + "/" + nick );
        jabber->sendTag( joinRequest );

        QDateTime when;
        when.setTime_t( time( 0 ) );

        /* Let the user know what is going on */
        _conversation->systemAnnouncement( when, "*** Attempting to join...", "#808080" );

        connect( _input, SIGNAL( ctlEnter() ), this, SLOT(sendMessage()) );
        connect( jabber, SIGNAL(changedStatus()), this, SLOT(changedStatus()) );
        connect( _members, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(memberDblClicked(QListViewItem *)) );
        connect( _u, SIGNAL(userDeleted(User *)), this, SLOT(userDeleted()) );

}

DlgGrpChat::~DlgGrpChat()
{
        XMLNode setOffline("presence");
        setOffline.setAttribute( "to", _room + "@" + _server );
        setOffline.setAttribute( "type", "unavailable" );
        jabber->sendTag( setOffline );

        if ( !_u )
                delete _u;
}

void DlgGrpChat::newPacket( XMLNode *n )
{
        JabberID jid( n->getAttribute( "from" ) );

        /* changes the caption if the subject changes */
        if ( n->getChildNode( "subject", false ) != 0 )
                setCaption( "Groupchat: " + _room + " on " + jabber->getAgent( _server )->name() + ": " + n->getChildNode( "subject" )->getValue() + " - Konverse" );

        QDateTime when;
        when.setTime_t( time( 0 ) );

        /* See if we can find a timestamp */
        int i;
        for( i=0; i<n->numberChildNodes(); i++ )
        {
                if ( n->getChildNode( i )->getName() == "x" )
                {
                        if ( n->getChildNode( i )->getAttribute( "xmlns" ) == "jabber:x:delay" )
                        {
                                /* Found one! */
                                QString txtWhen = n->getChildNode( i )->getAttribute( "stamp" );
                                when = JabberCore::convertFromJabberTime( txtWhen );
                        }
                }
        }

        if ( jid.getResource().isEmpty() )
                _conversation->systemAnnouncement( when, "*** " + n->getChildNode("body")->getValue(), "#808080" );
        else
                if ( jid.getResource() != _nick )
                        _conversation->addLine( when, n->getChildNode("body")->getValue(), jid.getResource(), "auto" );
}

void DlgGrpChat::presenceChanged(User *, UserPresence *p)
{
        QListViewItem *item = _members->firstChild();
        bool found = false;

        while ( item != 0 )
        {
                if ( item->text( 0 ) == p->resourceName() )
                {
                        item->setPixmap( 0, p->getIcon() );
                        found = true;
                        break;
                }

                item = item->nextSibling();
        }

        if ( !found && p->available() )
        {
                item = new QListViewItem( _members, p->resourceName() );
                item->setPixmap( 0, p->getIcon() );
        }
        else if ( found && !p->available() )
        {
                delete item;
        }
}

void DlgGrpChat::sendMessage()
{
        XMLNode message("message");
        message.setAttribute( "type", "groupchat" );
        message.setAttribute( "to", _room + "@" + _server  );
        message.getChildNode( "body" )->setValue( _input->text() );
        jabber->sendTag( message );

        QDateTime when;
        when.setTime_t( time( 0 ) );

        _conversation->addLine( when, _input->text(), _nick, "red" );
        _input->setText("");
}

void DlgGrpChat::memberDblClicked(QListViewItem *item)
{
        if (item != 0)
                if ( item->text( 0 ) != _nick )
                        jabber->messageBin()->startChat( _room + "@" + _server + "/" + item->text( 0 ) )->show();
}

void DlgGrpChat::changedStatus()
{
        QString txtPriority;
        txtPriority.setNum( jabber->getPriority() );
        XMLNode presence;
        presence.setName( "presence" );
        presence.setAttribute( "to", _room + "@" + _server + "/" + _nick );
        presence.getChildNode( "status" )->setValue( jabber->getDescription() );
        presence.getChildNode( "priority" )->setValue( txtPriority );
        switch( jabber->getShow() )
        {
                case JabberCore::Chat:
                        presence.getChildNode( "show" )->setValue( "chat" );
                        break;
                case JabberCore::Away:
                        presence.getChildNode( "show" )->setValue( "away" );
                        break;
                case JabberCore::ExtendedAway:
                        presence.getChildNode( "show" )->setValue( "xa" );
                        break;
                case JabberCore::DND:
                        presence.getChildNode( "show" )->setValue( "dnd" );
                        break;
                default:
                        /* We dont send a show for normal */
                        break;
        }
        jabber->sendTag( presence );
}

void DlgGrpChat::userDeleted()
{
        _u = 0;
}

void DlgGrpChat::resizeEvent( QResizeEvent *e )
{
        QWidget::resizeEvent( e );

        _main->refresh();
}

/******************************************************************
                        DlgJoinGrpChat
******************************************************************/
DlgJoinGrpChat *DlgJoinGrpChat::_current = 0;

void DlgJoinGrpChat::showDialog()
{
        if ( _current == 0 )
                (void) new DlgJoinGrpChat( 0, 0 );
}

DlgJoinGrpChat::DlgJoinGrpChat( QWidget *parent, const char *name )
        : QDialog( parent, name )
{
        _layout = new QGridLayout( this, 4, 3, 5, -1 );

        _system = new QComboBox( this );
        /* Add in all the transports that support group chat */
        for( int i=0; i<jabber->numberAgents(); i++ )
                if ( jabber->getAgent( i )->supportsGroupchat() )
                        _system->insertItem( jabber->getAgent( i )->name() );
        _system->setMinimumSize( _system->sizeHint() );
        _system->setMinimumHeight( _system->sizeHint().height() );
        _layout->addMultiCellWidget( _system, 0, 0, 1, 2 );

        _nick = new QLineEdit( this );
        _nick->setMinimumSize( _nick->sizeHint() );
        _nick->setMaximumHeight( _nick->sizeHint().height() );
        _layout->addMultiCellWidget( _nick, 1, 1, 1, 2 );

        _room = new QLineEdit( this );
        _room->setMinimumSize( _room->sizeHint() );
        _room->setMaximumHeight( _room->sizeHint().height() );
        _layout->addMultiCellWidget( _room, 2, 2, 1, 2 );


        _lblSystem = new QLabel( "IM System", this );
        _lblSystem->setMinimumSize( _lblSystem->sizeHint() );
        _lblSystem->setMaximumHeight( _lblSystem->sizeHint().height() );
        _layout->addWidget( _lblSystem, 0, 0 );

        _lblNick = new QLabel( "Nick", this );
        _lblNick->setMinimumSize( _lblNick->sizeHint() );
        _lblNick->setMaximumHeight( _lblNick->sizeHint().height() );
        _layout->addWidget( _lblNick, 1, 0 );

        _lblRoom = new QLabel( "Room", this );
        _lblRoom->setMinimumSize( _lblRoom->sizeHint() );
        _lblRoom->setMaximumHeight( _lblRoom->sizeHint().height() );
        _layout->addWidget( _lblRoom, 2, 0 );

        _join = new QPushButton( "&Join", this );
        _join->setMinimumSize( _join->sizeHint() );
        _join->setMaximumHeight( _join->sizeHint().height() );
        _join->setDefault( true );
        _layout->addWidget( _join, 3, 1 );

        _cancel = new QPushButton( "&Cancel", this );
        _cancel->setMinimumSize( _cancel->sizeHint() );
        _cancel->setMaximumHeight( _cancel->sizeHint().height() );
        _layout->addWidget( _cancel, 3, 2 );

        _layout->activate();

        connect( jabber, SIGNAL(disconnected()), this, SLOT(close()) );
        connect( _cancel, SIGNAL(clicked()), this, SLOT(close()) );
        connect( _join, SIGNAL(clicked()), this, SLOT(joinClicked()) );

        _current = this;

        setCaption( "Join a Chatroom" );
        setIcon( KGlobal::iconLoader()->loadIcon("groupchat_dlg", KIcon::User) );

        show();
}

void DlgJoinGrpChat::joinClicked()
{
        if ( _nick->text().isEmpty() || _room->text().isEmpty() )
        {
                KMessageBox::error( this, "Please complete the entire form" );
                return;
        }

        /* find the servername */
        for ( int i=0; i<jabber->numberAgents(); i++ )
                if ( jabber->getAgent( i )->name() == _system->currentText() )
                        (void) new DlgGrpChat( _room->text(), jabber->getAgent( i )->jid(), _nick->text() );

        delete this;
}
