#!/usr/bin/perl

# $Id: jamo2hypua.pl,v 1.2 2007/07/22 04:12:15 karnes Exp $
#
### Copyrihgt (C) 2004-2007 Dohyun Kim <nomos at ktug.or.kr>
###
### -------------------------------------------------------------------
### part of ko.TeX
###
### This work may be distributed and/or modified under the
### conditions of the LaTeX Project Public License, either version 1.3c
### of this license or (at your option) any later version.
### The latest version of this license is in
###  http://www.latex-project.org/lppl.txt
### and version 1.3c or later is part of all distributions of LaTeX
### version 2006/05/20 or later.
### -------------------------------------------------------------------

#
# usage: jamo2hypua.pl < in_file > out_file
#

use 5.8.0;
use strict;

binmode (STDIN,":utf8");
binmode (STDOUT,":utf8");

my @jamo_cho_comp =
  qw/
     3131 3132 3134 3137 3138 3139 3141 3142
     3143 3145 3146 3147 3148 3149 314A 314B
     314C 314D 314E 1113 3165 3166 1116 1117
     1118 1119 3140 111B 316E 3171 3172 111F
     3173 3144 3174 3175 1124 1125 1126 3176
     1128 3177 112A 3178 3179 317A 317B 317C
     1130 1131 317D 1133 1134 1135 317E 1137
     1138 1139 113A 113B 113C 113D 113E 113F
     317F 1141 1142 1143 1144 1145 1146 3180
     1148 1149 114A 114B 3181 114D 114E 114F
     1150 1151 1152 1153 1154 1155 1156 3184
     3185 3186
     /;

my @jamo_jung_comp =
  qw/
	  314F 3150 3151 3152 3153 3154 3155
     3156 3157 3158 3159 315A 315B 315C 315D
     315E 315F 3160 3161 3162 3163 1176 1177
     1178 1179 117A 117B 117C 117D 117E 117F
     1180 1181 1182 1183 3187 3188 1186 1187
     3189 1189 118A 118B 118C 118D 118E 118F
     1190 318A 318B 1193 318C 1195 1196 1197
     1198 1199 119A 119B 119C 119D 318D 119F
     11A0 318E 11A2
     /;

my @jamo_jong_comp =
  qw/
     11A8 11A9 3133 11AB 3135 3136 11AE 11AF
     313A 313B 313C 313D 313E 313F 11B6 11B7
     11B8 11B9 11BA 11BB 11BC 11BD 11BE 11BF
     11C0 11C1 11C2 11C3 11C4 11C5 11C6 3167
     3168 11C9 11CA 11CB 3169 11CD 316A 11CF
     11D0 11D1 11D2 316B 11D4 11D5 11D6 316C
     11D8 316D 11DA 11DB 11DC 316F 11DE 3170
     11E0 11E1 11E2 11E3 11E4 11E5 11E6 11E7
     11E8 11E9 11EA 11EB 11EC 11ED 11EE 11EF
     11F0 3182 3183 11F3 11F4 11F5 11F6 11F7
     11F8 11F9
     /;

my @pua2jamo_cho = 
  qw/
     1100 1101 F787 1102 1113 1114 1115 1116
     F78D F78E F78F 1103 1117 1104 F793 F794
     F795 F796 F797 1105 F799 F79A 1118 F79C
     F79D 1119 F79F F7A0 F7A1 F7A2 F7A3 F7A4
     F7A5 111A 111B 1106 F7A9 F7AA 111C F7AC
     111D 1107 111E 111F 1120 1108 1121 1122
     1123 1124 1125 1126 F7B9 1127 1128 F7BC
     1129 112A F7BF 112B 112C 1109 112D 112E
     112F 1130 1131 1132 1133 110A F7CB 1134
     1135 1136 1137 1138 1139 113A 113B 113C
     113D 113E 113F 1140 110B 1141 1142 F7DC
     1143 1144 1145 1146 1147 1148 1149 114A
     114B F7E6 114C 110C 114D 110D F7EB 114E
     114F 1150 1151 110E 1152 1153 1154 1155
     110F 1110 F7F7 1111 1156 F7FA 1157 1112
     F7FD 1158 1159 F800
     /;

my @pua2jamo_jung = 
  qw/
     1161 1176 1177 F80A 1162 1163 1178 1179
     F80F 1164 1165 117A 117B 117C 1166 1167
     F817 117D 117E 1168 1169 116A 116B F81E
     F81F 117F 1180 F822 1181 1182 F825 1183
     116C 116D F829 F82A 1184 1185 F82D 1186
     1187 1188 116E 1189 118A 116F 118B 1170
     F837 118C 118D 1171 F83B 1172 118E F83E
     118F 1190 1191 1192 F843 1193 1194 1173
     F847 F848 F849 F84A 1195 1196 1174 1197
     1175 1198 1199 F852 F853 F854 F855 119A
     F857 F858 119B F85A 119C F85C 119D 119E
     F85F 119F F861 11A0 11A1 11A2
     /;

my @pua2jamo_jong = 
  qw/
     11A8 11A9 F86D 11C3 F86F 11AA 11C4 F872
     F873 F874 11AB 11C5 F877 11C6 F879 11C7
     11C8 11AC F87D 11C9 11AD 11AE 11CA F882
     F883 11CB F885 F886 F887 F888 F889 F88A
     11AF 11B0 F88D 11CC F88F 11CD 11CE 11CF
     11D0 F894 11B1 11D1 11D2 F898 11B2 F89A
     11D3 F89C 11D4 11D5 11B3 11D6 11D7 F8A2
     11D8 11B4 11B5 11B6 11D9 F8A8 F8A9 11B7
     11DA F8AC F8AD 11DB F8AF 11DC F8B1 11DD
     11DE 11DF F8B5 11E0 11E1 11E2 11B8 F8BA
     11E3 F8BC F8BD F8BE 11B9 F8C0 F8C1 F8C2
     11E4 11E5 11E6 11BA 11E7 11E8 11E9 F8CA
     11EA F8CC 11BB F8CE F8CF F8D0 F8D1 F8D2
     F8D3 F8D4 11EB F8D6 F8D7 11BC 11EC 11ED
     F8DB F8DC 11EE 11EF F8DF 11F0 F8E1 11F1
     11F2 F8E4 F8E5 11BD F8E7 F8E8 F8E9 11BE
     11BF 11C0 11C1 11F3 F8EF F8F0 11F4 11C2
     11F5 11F6 11F7 11F8 11F9
     /;

while (<>) {
  print STDERR ".";
  &decompose_jamo;
  &jamo2hypua;
  &compose_jamo;
  &jamo2syllable;
  &jamo2jamocomp;
  &jamo2puajamo;
  &single_puajamo2jamo;
  print;
}

print STDERR "\n";

##### subroutines #####

sub jamo2syllable {
  # 초성.중성.종성
  s/([\x{1100}-\x{1112}])
    ([\x{1161}-\x{1175}])
    ([\x{11A8}-\x{11C2}])
   /&do_jamo2syllable(ord $1, ord $2, ord $3)/gex;
  # 초성.중성
  s/([\x{1100}-\x{1112}])
    ([\x{1161}-\x{1175}])
   /&do_jamo2syllable(ord $1, ord $2, 0)/gex;
}

sub do_jamo2syllable {
  my ($cho,$jung,$jong) = @_;
  my $syllable = ($cho - 0x1100) * 21;
  $syllable = ($syllable + $jung - 0x1161) * 28;
  if ($jong > 0) {
    $syllable = $syllable + $jong - 0x11a7;
  }
  $syllable += 0xac00;
  return chr $syllable;
}

sub jamo2jamocomp {
  s/([\x{1100}-\x{1159}])\x{1160}/&cjamo_by_jamo_cho(ord $1)/ge;
  s/\x{115F}([\x{1161}-\x{11A2}])/&cjamo_by_jamo_jung(ord $1)/ge;
  s/\x{115F}\x{1160}([\x{11A8}-\x{11F9}])/&cjamo_by_jamo_jong(ord $1)/ge;
}

sub single_puajamo2jamo {
  s/([\x{F785}-\x{F800}])
    \x{F806}
   /chr hex $pua2jamo_cho[ord($1) - 0xF785]/gex;
  s/\x{F784}
    ([\x{F807}-\x{F864}])
   /chr hex $pua2jamo_jung[ord($1) - 0xF807]/gex;
  s/\x{F784}
    \x{F806}
    ([\x{F86B}-\x{F8F7}])
   /chr hex $pua2jamo_jong[ord($1) - 0xF86B]/gex;
}

sub cjamo_by_jamo_cho {
  my $jamo = shift;
  my $cjamo = hex $jamo_cho_comp[$jamo - 0x1100];
  if ($cjamo < 0x11FF) {
    return chr($cjamo) . chr(0x1160);
  }
  else {
    return chr($cjamo);
  }
}

sub cjamo_by_jamo_jung {
  my $jamo = shift;
  my $cjamo = hex $jamo_jung_comp[$jamo - 0x1161];
  if ($cjamo < 0x11FF) {
    return chr(0x115F) . chr($cjamo);
  }
  else {
    return chr($cjamo);
  }
}

sub cjamo_by_jamo_jong {
  my $jamo = shift;
  my $cjamo = hex $jamo_jong_comp[$jamo - 0x11A8];
  if ($cjamo < 0x11FF) {
    return chr(0x115F) . chr(0x1160) . chr($cjamo);
  }
  else {
    return chr($cjamo);
  }
}

sub compose_jamo {
    s/\x{1107}\x{1107}\x{110B}/\x{112C}/g;
    s/\x{1107}\x{1109}\x{1100}/\x{1122}/g;
    s/\x{1107}\x{1109}\x{1103}/\x{1123}/g;
    s/\x{1107}\x{1109}\x{1107}/\x{1124}/g;
    s/\x{1107}\x{1109}\x{1109}/\x{1125}/g;
    s/\x{1107}\x{1109}\x{110C}/\x{1126}/g;
    s/\x{1109}\x{1107}\x{1100}/\x{1133}/g;
    s/\x{1109}\x{1109}\x{1109}/\x{1134}/g;
    s/\x{1100}\x{1100}/\x{1101}/g;
    s/\x{1102}\x{1100}/\x{1113}/g;
    s/\x{1102}\x{1102}/\x{1114}/g;
    s/\x{1102}\x{1103}/\x{1115}/g;
    s/\x{1102}\x{1107}/\x{1116}/g;
    s/\x{1103}\x{1100}/\x{1117}/g;
    s/\x{1103}\x{1103}/\x{1104}/g;
    s/\x{1105}\x{1102}/\x{1118}/g;
    s/\x{1105}\x{1105}/\x{1119}/g;
    s/\x{1105}\x{110B}/\x{111B}/g;
    s/\x{1105}\x{1112}/\x{111A}/g;
    s/\x{1106}\x{1107}/\x{111C}/g;
    s/\x{1106}\x{110B}/\x{111D}/g;
    s/\x{1107}\x{1100}/\x{111E}/g;
    s/\x{1107}\x{1102}/\x{111F}/g;
    s/\x{1107}\x{1103}/\x{1120}/g;
    s/\x{1107}\x{1107}/\x{1108}/g;
    s/\x{1107}\x{1109}/\x{1121}/g;
    s/\x{1107}\x{110A}/\x{1125}/g;
    s/\x{1107}\x{110B}/\x{112B}/g;
    s/\x{1107}\x{110C}/\x{1127}/g;
    s/\x{1107}\x{110E}/\x{1128}/g;
    s/\x{1107}\x{1110}/\x{1129}/g;
    s/\x{1107}\x{1111}/\x{112A}/g;
    s/\x{1107}\x{112B}/\x{112C}/g;
    s/\x{1107}\x{112D}/\x{1122}/g;
    s/\x{1107}\x{112F}/\x{1123}/g;
    s/\x{1107}\x{1132}/\x{1124}/g;
    s/\x{1107}\x{1136}/\x{1126}/g;
    s/\x{1108}\x{110B}/\x{112C}/g;
    s/\x{1109}\x{1100}/\x{112D}/g;
    s/\x{1109}\x{1102}/\x{112E}/g;
    s/\x{1109}\x{1103}/\x{112F}/g;
    s/\x{1109}\x{1105}/\x{1130}/g;
    s/\x{1109}\x{1106}/\x{1131}/g;
    s/\x{1109}\x{1107}/\x{1132}/g;
    s/\x{1109}\x{1109}/\x{110A}/g;
    s/\x{1109}\x{110A}/\x{1134}/g;
    s/\x{1109}\x{110B}/\x{1135}/g;
    s/\x{1109}\x{110C}/\x{1136}/g;
    s/\x{1109}\x{110E}/\x{1137}/g;
    s/\x{1109}\x{110F}/\x{1138}/g;
    s/\x{1109}\x{1110}/\x{1139}/g;
    s/\x{1109}\x{1111}/\x{113A}/g;
    s/\x{1109}\x{1112}/\x{113B}/g;
    s/\x{1109}\x{111E}/\x{1133}/g;
    s/\x{110A}\x{1109}/\x{1134}/g;
    s/\x{110B}\x{1100}/\x{1141}/g;
    s/\x{110B}\x{1103}/\x{1142}/g;
    s/\x{110B}\x{1106}/\x{1143}/g;
    s/\x{110B}\x{1107}/\x{1144}/g;
    s/\x{110B}\x{1109}/\x{1145}/g;
    s/\x{110B}\x{110B}/\x{1147}/g;
    s/\x{110B}\x{110C}/\x{1148}/g;
    s/\x{110B}\x{110E}/\x{1149}/g;
    s/\x{110B}\x{1110}/\x{114A}/g;
    s/\x{110B}\x{1111}/\x{114B}/g;
    s/\x{110B}\x{1140}/\x{1146}/g;
    s/\x{110C}\x{110B}/\x{114D}/g;
    s/\x{110C}\x{110C}/\x{110D}/g;
    s/\x{110E}\x{110F}/\x{1152}/g;
    s/\x{110E}\x{1112}/\x{1153}/g;
    s/\x{1111}\x{1107}/\x{1156}/g;
    s/\x{1111}\x{110B}/\x{1157}/g;
    s/\x{1112}\x{1112}/\x{1158}/g;
    s/\x{1121}\x{1100}/\x{1122}/g;
    s/\x{1121}\x{1103}/\x{1123}/g;
    s/\x{1121}\x{1107}/\x{1124}/g;
    s/\x{1121}\x{1109}/\x{1125}/g;
    s/\x{1121}\x{110C}/\x{1126}/g;
    s/\x{1132}\x{1100}/\x{1133}/g;
    s/\x{113C}\x{113C}/\x{113D}/g;
    s/\x{113E}\x{113E}/\x{113F}/g;
    s/\x{114E}\x{114E}/\x{114F}/g;
    s/\x{1150}\x{1150}/\x{1151}/g;
    s/\x{1169}\x{1161}\x{1175}/\x{116B}/g;
    s/\x{1169}\x{1165}\x{1175}/\x{1180}/g;
    s/\x{1169}\x{1167}\x{1175}/\x{1181}/g;
    s/\x{116D}\x{1163}\x{1175}/\x{1185}/g;
    s/\x{116E}\x{1161}\x{1175}/\x{118A}/g;
    s/\x{116E}\x{1165}\x{1173}/\x{118B}/g;
    s/\x{116E}\x{1165}\x{1175}/\x{1170}/g;
    s/\x{116E}\x{1167}\x{1175}/\x{118C}/g;
    s/\x{1172}\x{1165}\x{1175}/\x{1190}/g;
    s/\x{1172}\x{1167}\x{1175}/\x{1192}/g;
    s/\x{1173}\x{1175}\x{116E}/\x{1197}/g;
    s/\x{1161}\x{1169}/\x{1176}/g;
    s/\x{1161}\x{116E}/\x{1177}/g;
    s/\x{1161}\x{1175}/\x{1162}/g;
    s/\x{1163}\x{1169}/\x{1178}/g;
    s/\x{1163}\x{116D}/\x{1179}/g;
    s/\x{1163}\x{1175}/\x{1164}/g;
    s/\x{1165}\x{1169}/\x{117A}/g;
    s/\x{1165}\x{116E}/\x{117B}/g;
    s/\x{1165}\x{1173}/\x{117C}/g;
    s/\x{1165}\x{1175}/\x{1166}/g;
    s/\x{1167}\x{1169}/\x{117D}/g;
    s/\x{1167}\x{116E}/\x{117E}/g;
    s/\x{1167}\x{1175}/\x{1168}/g;
    s/\x{1169}\x{1161}/\x{116A}/g;
    s/\x{1169}\x{1162}/\x{116B}/g;
    s/\x{1169}\x{1165}/\x{117F}/g;
    s/\x{1169}\x{1166}/\x{1180}/g;
    s/\x{1169}\x{1168}/\x{1181}/g;
    s/\x{1169}\x{1169}/\x{1182}/g;
    s/\x{1169}\x{116E}/\x{1183}/g;
    s/\x{1169}\x{1175}/\x{116C}/g;
    s/\x{116A}\x{1175}/\x{116B}/g;
    s/\x{116D}\x{1163}/\x{1184}/g;
    s/\x{116D}\x{1164}/\x{1185}/g;
    s/\x{116D}\x{1167}/\x{1186}/g;
    s/\x{116D}\x{1169}/\x{1187}/g;
    s/\x{116D}\x{1175}/\x{1188}/g;
    s/\x{116E}\x{1161}/\x{1189}/g;
    s/\x{116E}\x{1162}/\x{118A}/g;
    s/\x{116E}\x{1165}/\x{116F}/g;
    s/\x{116E}\x{1166}/\x{1170}/g;
    s/\x{116E}\x{1168}/\x{118C}/g;
    s/\x{116E}\x{116E}/\x{118D}/g;
    s/\x{116E}\x{1175}/\x{1171}/g;
    s/\x{116E}\x{117C}/\x{118B}/g;
    s/\x{116F}\x{1173}/\x{118B}/g;
    s/\x{116F}\x{1175}/\x{1170}/g;
    s/\x{1172}\x{1161}/\x{118E}/g;
    s/\x{1172}\x{1165}/\x{118F}/g;
    s/\x{1172}\x{1166}/\x{1190}/g;
    s/\x{1172}\x{1167}/\x{1191}/g;
    s/\x{1172}\x{1168}/\x{1192}/g;
    s/\x{1172}\x{116E}/\x{1193}/g;
    s/\x{1172}\x{1175}/\x{1194}/g;
    s/\x{1173}\x{116E}/\x{1195}/g;
    s/\x{1173}\x{1173}/\x{1196}/g;
    s/\x{1173}\x{1175}/\x{1174}/g;
    s/\x{1173}\x{119B}/\x{1197}/g;
    s/\x{1174}\x{116E}/\x{1197}/g;
    s/\x{1175}\x{1161}/\x{1198}/g;
    s/\x{1175}\x{1163}/\x{1199}/g;
    s/\x{1175}\x{1169}/\x{119A}/g;
    s/\x{1175}\x{116E}/\x{119B}/g;
    s/\x{1175}\x{1173}/\x{119C}/g;
    s/\x{1175}\x{119E}/\x{119D}/g;
    s/\x{117F}\x{1175}/\x{1180}/g;
    s/\x{1184}\x{1175}/\x{1185}/g;
    s/\x{1189}\x{1175}/\x{118A}/g;
    s/\x{118F}\x{1175}/\x{1190}/g;
    s/\x{1191}\x{1175}/\x{1192}/g;
    s/\x{119E}\x{1165}/\x{119F}/g;
    s/\x{119E}\x{116E}/\x{11A0}/g;
    s/\x{119E}\x{1175}/\x{11A1}/g;
    s/\x{119E}\x{119E}/\x{11A2}/g;
    s/\x{11A8}\x{11BA}\x{11A8}/\x{11C4}/g;
    s/\x{11AF}\x{11A8}\x{11BA}/\x{11CC}/g;
    s/\x{11AF}\x{11AE}\x{11C2}/\x{11CF}/g;
    s/\x{11AF}\x{11B7}\x{11A8}/\x{11D1}/g;
    s/\x{11AF}\x{11B7}\x{11BA}/\x{11D2}/g;
    s/\x{11AF}\x{11B8}\x{11BA}/\x{11D3}/g;
    s/\x{11AF}\x{11B8}\x{11BC}/\x{11D5}/g;
    s/\x{11AF}\x{11B8}\x{11C2}/\x{11D4}/g;
    s/\x{11AF}\x{11BA}\x{11BA}/\x{11D6}/g;
    s/\x{11B7}\x{11BA}\x{11BA}/\x{11DE}/g;
    s/\x{11BC}\x{11A8}\x{11A8}/\x{11ED}/g;
    s/\x{11A8}\x{11A8}/\x{11A9}/g;
    s/\x{11A8}\x{11AF}/\x{11C3}/g;
    s/\x{11A8}\x{11BA}/\x{11AA}/g;
    s/\x{11A8}\x{11E7}/\x{11C4}/g;
    s/\x{11AA}\x{11A8}/\x{11C4}/g;
    s/\x{11AB}\x{11A8}/\x{11C5}/g;
    s/\x{11AB}\x{11AE}/\x{11C6}/g;
    s/\x{11AB}\x{11BA}/\x{11C7}/g;
    s/\x{11AB}\x{11BD}/\x{11AC}/g;
    s/\x{11AB}\x{11C0}/\x{11C9}/g;
    s/\x{11AB}\x{11C2}/\x{11AD}/g;
    s/\x{11AB}\x{11EB}/\x{11C8}/g;
    s/\x{11AE}\x{11A8}/\x{11CA}/g;
    s/\x{11AE}\x{11AF}/\x{11CB}/g;
    s/\x{11AF}\x{11A8}/\x{11B0}/g;
    s/\x{11AF}\x{11AA}/\x{11CC}/g;
    s/\x{11AF}\x{11AB}/\x{11CD}/g;
    s/\x{11AF}\x{11AE}/\x{11CE}/g;
    s/\x{11AF}\x{11AF}/\x{11D0}/g;
    s/\x{11AF}\x{11B7}/\x{11B1}/g;
    s/\x{11AF}\x{11B8}/\x{11B2}/g;
    s/\x{11AF}\x{11B9}/\x{11D3}/g;
    s/\x{11AF}\x{11BA}/\x{11B3}/g;
    s/\x{11AF}\x{11BB}/\x{11D6}/g;
    s/\x{11AF}\x{11BF}/\x{11D8}/g;
    s/\x{11AF}\x{11C0}/\x{11B4}/g;
    s/\x{11AF}\x{11C1}/\x{11B5}/g;
    s/\x{11AF}\x{11C2}/\x{11B6}/g;
    s/\x{11AF}\x{11DA}/\x{11D1}/g;
    s/\x{11AF}\x{11DD}/\x{11D2}/g;
    s/\x{11AF}\x{11E5}/\x{11D4}/g;
    s/\x{11AF}\x{11E6}/\x{11D5}/g;
    s/\x{11AF}\x{11EB}/\x{11D7}/g;
    s/\x{11AF}\x{11F9}/\x{11D9}/g;
    s/\x{11B0}\x{11BA}/\x{11CC}/g;
    s/\x{11B1}\x{11A8}/\x{11D1}/g;
    s/\x{11B1}\x{11BA}/\x{11D2}/g;
    s/\x{11B2}\x{11BA}/\x{11D3}/g;
    s/\x{11B2}\x{11BC}/\x{11D5}/g;
    s/\x{11B2}\x{11C2}/\x{11D4}/g;
    s/\x{11B3}\x{11BA}/\x{11D6}/g;
    s/\x{11B7}\x{11A8}/\x{11DA}/g;
    s/\x{11B7}\x{11AF}/\x{11DB}/g;
    s/\x{11B7}\x{11B8}/\x{11DC}/g;
    s/\x{11B7}\x{11BA}/\x{11DD}/g;
    s/\x{11B7}\x{11BB}/\x{11DE}/g;
    s/\x{11B7}\x{11BC}/\x{11E2}/g;
    s/\x{11B7}\x{11BE}/\x{11E0}/g;
    s/\x{11B7}\x{11C2}/\x{11E1}/g;
    s/\x{11B7}\x{11EB}/\x{11DF}/g;
    s/\x{11B8}\x{11AF}/\x{11E3}/g;
    s/\x{11B8}\x{11BA}/\x{11B9}/g;
    s/\x{11B8}\x{11BC}/\x{11E6}/g;
    s/\x{11B8}\x{11C1}/\x{11E4}/g;
    s/\x{11B8}\x{11C2}/\x{11E5}/g;
    s/\x{11BA}\x{11A8}/\x{11E7}/g;
    s/\x{11BA}\x{11AE}/\x{11E8}/g;
    s/\x{11BA}\x{11AF}/\x{11E9}/g;
    s/\x{11BA}\x{11B8}/\x{11EA}/g;
    s/\x{11BA}\x{11BA}/\x{11BB}/g;
    s/\x{11BC}\x{11A8}/\x{11EC}/g;
    s/\x{11BC}\x{11A9}/\x{11ED}/g;
    s/\x{11BC}\x{11BC}/\x{11EE}/g;
    s/\x{11BC}\x{11BF}/\x{11EF}/g;
    s/\x{11C1}\x{11B8}/\x{11F3}/g;
    s/\x{11C1}\x{11BC}/\x{11F4}/g;
    s/\x{11C2}\x{11AB}/\x{11F5}/g;
    s/\x{11C2}\x{11AF}/\x{11F6}/g;
    s/\x{11C2}\x{11B7}/\x{11F7}/g;
    s/\x{11C2}\x{11B8}/\x{11F8}/g;
    s/\x{11CE}\x{11C2}/\x{11CF}/g;
    s/\x{11DD}\x{11BA}/\x{11DE}/g;
    s/\x{11EC}\x{11A8}/\x{11ED}/g;
    s/\x{11F0}\x{11BA}/\x{11F1}/g;
    s/\x{11F0}\x{11EB}/\x{11F2}/g;
}

sub decompose_jamo {
    s/\x{112C}/\x{1107}\x{1107}\x{110B}/g;
    s/\x{1122}/\x{1107}\x{1109}\x{1100}/g;
    s/\x{1123}/\x{1107}\x{1109}\x{1103}/g;
    s/\x{1124}/\x{1107}\x{1109}\x{1107}/g;
    s/\x{1125}/\x{1107}\x{1109}\x{1109}/g;
    s/\x{1126}/\x{1107}\x{1109}\x{110C}/g;
    s/\x{1133}/\x{1109}\x{1107}\x{1100}/g;
    s/\x{1134}/\x{1109}\x{1109}\x{1109}/g;
    s/\x{1101}/\x{1100}\x{1100}/g;
    s/\x{1113}/\x{1102}\x{1100}/g;
    s/\x{1114}/\x{1102}\x{1102}/g;
    s/\x{1115}/\x{1102}\x{1103}/g;
    s/\x{1116}/\x{1102}\x{1107}/g;
    s/\x{1117}/\x{1103}\x{1100}/g;
    s/\x{1104}/\x{1103}\x{1103}/g;
    s/\x{1118}/\x{1105}\x{1102}/g;
    s/\x{1119}/\x{1105}\x{1105}/g;
    s/\x{111B}/\x{1105}\x{110B}/g;
    s/\x{111A}/\x{1105}\x{1112}/g;
    s/\x{111C}/\x{1106}\x{1107}/g;
    s/\x{111D}/\x{1106}\x{110B}/g;
    s/\x{111E}/\x{1107}\x{1100}/g;
    s/\x{111F}/\x{1107}\x{1102}/g;
    s/\x{1120}/\x{1107}\x{1103}/g;
    s/\x{1108}/\x{1107}\x{1107}/g;
    s/\x{1121}/\x{1107}\x{1109}/g;
    s/\x{112B}/\x{1107}\x{110B}/g;
    s/\x{1127}/\x{1107}\x{110C}/g;
    s/\x{1128}/\x{1107}\x{110E}/g;
    s/\x{1129}/\x{1107}\x{1110}/g;
    s/\x{112A}/\x{1107}\x{1111}/g;
    s/\x{112D}/\x{1109}\x{1100}/g;
    s/\x{112E}/\x{1109}\x{1102}/g;
    s/\x{112F}/\x{1109}\x{1103}/g;
    s/\x{1130}/\x{1109}\x{1105}/g;
    s/\x{1131}/\x{1109}\x{1106}/g;
    s/\x{1132}/\x{1109}\x{1107}/g;
    s/\x{110A}/\x{1109}\x{1109}/g;
    s/\x{1135}/\x{1109}\x{110B}/g;
    s/\x{1136}/\x{1109}\x{110C}/g;
    s/\x{1137}/\x{1109}\x{110E}/g;
    s/\x{1138}/\x{1109}\x{110F}/g;
    s/\x{1139}/\x{1109}\x{1110}/g;
    s/\x{113A}/\x{1109}\x{1111}/g;
    s/\x{113B}/\x{1109}\x{1112}/g;
    s/\x{1141}/\x{110B}\x{1100}/g;
    s/\x{1142}/\x{110B}\x{1103}/g;
    s/\x{1143}/\x{110B}\x{1106}/g;
    s/\x{1144}/\x{110B}\x{1107}/g;
    s/\x{1145}/\x{110B}\x{1109}/g;
    s/\x{1147}/\x{110B}\x{110B}/g;
    s/\x{1148}/\x{110B}\x{110C}/g;
    s/\x{1149}/\x{110B}\x{110E}/g;
    s/\x{114A}/\x{110B}\x{1110}/g;
    s/\x{114B}/\x{110B}\x{1111}/g;
    s/\x{1146}/\x{110B}\x{1140}/g;
    s/\x{114D}/\x{110C}\x{110B}/g;
    s/\x{110D}/\x{110C}\x{110C}/g;
    s/\x{1152}/\x{110E}\x{110F}/g;
    s/\x{1153}/\x{110E}\x{1112}/g;
    s/\x{1156}/\x{1111}\x{1107}/g;
    s/\x{1157}/\x{1111}\x{110B}/g;
    s/\x{1158}/\x{1112}\x{1112}/g;
    s/\x{114F}/\x{114E}\x{114E}/g;
    s/\x{1151}/\x{1150}\x{1150}/g;
    s/\x{116B}/\x{1169}\x{1161}\x{1175}/g;
    s/\x{1180}/\x{1169}\x{1165}\x{1175}/g;
    s/\x{1181}/\x{1169}\x{1167}\x{1175}/g;
    s/\x{1185}/\x{116D}\x{1163}\x{1175}/g;
    s/\x{118A}/\x{116E}\x{1161}\x{1175}/g;
    s/\x{118B}/\x{116E}\x{1165}\x{1173}/g;
    s/\x{1170}/\x{116E}\x{1165}\x{1175}/g;
    s/\x{118C}/\x{116E}\x{1167}\x{1175}/g;
    s/\x{1190}/\x{1172}\x{1165}\x{1175}/g;
    s/\x{1192}/\x{1172}\x{1167}\x{1175}/g;
    s/\x{1197}/\x{1173}\x{1175}\x{116E}/g;
    s/\x{1176}/\x{1161}\x{1169}/g;
    s/\x{1177}/\x{1161}\x{116E}/g;
    s/\x{1162}/\x{1161}\x{1175}/g;
    s/\x{1178}/\x{1163}\x{1169}/g;
    s/\x{1179}/\x{1163}\x{116D}/g;
    s/\x{1164}/\x{1163}\x{1175}/g;
    s/\x{117A}/\x{1165}\x{1169}/g;
    s/\x{117B}/\x{1165}\x{116E}/g;
    s/\x{117C}/\x{1165}\x{1173}/g;
    s/\x{1166}/\x{1165}\x{1175}/g;
    s/\x{117D}/\x{1167}\x{1169}/g;
    s/\x{117E}/\x{1167}\x{116E}/g;
    s/\x{1168}/\x{1167}\x{1175}/g;
    s/\x{116A}/\x{1169}\x{1161}/g;
    s/\x{117F}/\x{1169}\x{1165}/g;
    s/\x{1182}/\x{1169}\x{1169}/g;
    s/\x{1183}/\x{1169}\x{116E}/g;
    s/\x{116C}/\x{1169}\x{1175}/g;
    s/\x{1184}/\x{116D}\x{1163}/g;
    s/\x{1186}/\x{116D}\x{1167}/g;
    s/\x{1187}/\x{116D}\x{1169}/g;
    s/\x{1188}/\x{116D}\x{1175}/g;
    s/\x{1189}/\x{116E}\x{1161}/g;
    s/\x{116F}/\x{116E}\x{1165}/g;
    s/\x{118D}/\x{116E}\x{116E}/g;
    s/\x{1171}/\x{116E}\x{1175}/g;
    s/\x{118E}/\x{1172}\x{1161}/g;
    s/\x{118F}/\x{1172}\x{1165}/g;
    s/\x{1191}/\x{1172}\x{1167}/g;
    s/\x{1193}/\x{1172}\x{116E}/g;
    s/\x{1194}/\x{1172}\x{1175}/g;
    s/\x{1195}/\x{1173}\x{116E}/g;
    s/\x{1196}/\x{1173}\x{1173}/g;
    s/\x{1174}/\x{1173}\x{1175}/g;
    s/\x{1198}/\x{1175}\x{1161}/g;
    s/\x{1199}/\x{1175}\x{1163}/g;
    s/\x{119A}/\x{1175}\x{1169}/g;
    s/\x{119B}/\x{1175}\x{116E}/g;
    s/\x{119C}/\x{1175}\x{1173}/g;
    s/\x{119D}/\x{1175}\x{119E}/g;
    s/\x{119F}/\x{119E}\x{1165}/g;
    s/\x{11A0}/\x{119E}\x{116E}/g;
    s/\x{11A1}/\x{119E}\x{1175}/g;
    s/\x{11A2}/\x{119E}\x{119E}/g;
    s/\x{11C4}/\x{11A8}\x{11BA}\x{11A8}/g;
    s/\x{11CC}/\x{11AF}\x{11A8}\x{11BA}/g;
    s/\x{11CF}/\x{11AF}\x{11AE}\x{11C2}/g;
    s/\x{11D1}/\x{11AF}\x{11B7}\x{11A8}/g;
    s/\x{11D2}/\x{11AF}\x{11B7}\x{11BA}/g;
    s/\x{11D3}/\x{11AF}\x{11B8}\x{11BA}/g;
    s/\x{11D5}/\x{11AF}\x{11B8}\x{11BC}/g;
    s/\x{11D4}/\x{11AF}\x{11B8}\x{11C2}/g;
    s/\x{11D6}/\x{11AF}\x{11BA}\x{11BA}/g;
    s/\x{11DE}/\x{11B7}\x{11BA}\x{11BA}/g;
    s/\x{11ED}/\x{11BC}\x{11A8}\x{11A8}/g;
    s/\x{11A9}/\x{11A8}\x{11A8}/g;
    s/\x{11C3}/\x{11A8}\x{11AF}/g;
    s/\x{11AA}/\x{11A8}\x{11BA}/g;
    s/\x{11C5}/\x{11AB}\x{11A8}/g;
    s/\x{11C6}/\x{11AB}\x{11AE}/g;
    s/\x{11C7}/\x{11AB}\x{11BA}/g;
    s/\x{11AC}/\x{11AB}\x{11BD}/g;
    s/\x{11C9}/\x{11AB}\x{11C0}/g;
    s/\x{11AD}/\x{11AB}\x{11C2}/g;
    s/\x{11C8}/\x{11AB}\x{11EB}/g;
    s/\x{11CA}/\x{11AE}\x{11A8}/g;
    s/\x{11CB}/\x{11AE}\x{11AF}/g;
    s/\x{11B0}/\x{11AF}\x{11A8}/g;
    s/\x{11CD}/\x{11AF}\x{11AB}/g;
    s/\x{11CE}/\x{11AF}\x{11AE}/g;
    s/\x{11D0}/\x{11AF}\x{11AF}/g;
    s/\x{11B1}/\x{11AF}\x{11B7}/g;
    s/\x{11B2}/\x{11AF}\x{11B8}/g;
    s/\x{11B3}/\x{11AF}\x{11BA}/g;
    s/\x{11D8}/\x{11AF}\x{11BF}/g;
    s/\x{11B4}/\x{11AF}\x{11C0}/g;
    s/\x{11B5}/\x{11AF}\x{11C1}/g;
    s/\x{11B6}/\x{11AF}\x{11C2}/g;
    s/\x{11D7}/\x{11AF}\x{11EB}/g;
    s/\x{11D9}/\x{11AF}\x{11F9}/g;
    s/\x{11DA}/\x{11B7}\x{11A8}/g;
    s/\x{11DB}/\x{11B7}\x{11AF}/g;
    s/\x{11DC}/\x{11B7}\x{11B8}/g;
    s/\x{11DD}/\x{11B7}\x{11BA}/g;
    s/\x{11E2}/\x{11B7}\x{11BC}/g;
    s/\x{11E0}/\x{11B7}\x{11BE}/g;
    s/\x{11E1}/\x{11B7}\x{11C2}/g;
    s/\x{11DF}/\x{11B7}\x{11EB}/g;
    s/\x{11E3}/\x{11B8}\x{11AF}/g;
    s/\x{11B9}/\x{11B8}\x{11BA}/g;
    s/\x{11E6}/\x{11B8}\x{11BC}/g;
    s/\x{11E4}/\x{11B8}\x{11C1}/g;
    s/\x{11E5}/\x{11B8}\x{11C2}/g;
    s/\x{11E7}/\x{11BA}\x{11A8}/g;
    s/\x{11E8}/\x{11BA}\x{11AE}/g;
    s/\x{11E9}/\x{11BA}\x{11AF}/g;
    s/\x{11EA}/\x{11BA}\x{11B8}/g;
    s/\x{11BB}/\x{11BA}\x{11BA}/g;
    s/\x{11EC}/\x{11BC}\x{11A8}/g;
    s/\x{11EE}/\x{11BC}\x{11BC}/g;
    s/\x{11EF}/\x{11BC}\x{11BF}/g;
    s/\x{11F3}/\x{11C1}\x{11B8}/g;
    s/\x{11F4}/\x{11C1}\x{11BC}/g;
    s/\x{11F5}/\x{11C2}\x{11AB}/g;
    s/\x{11F6}/\x{11C2}\x{11AF}/g;
    s/\x{11F7}/\x{11C2}\x{11B7}/g;
    s/\x{11F8}/\x{11C2}\x{11B8}/g;
    s/\x{11F1}/\x{11F0}\x{11BA}/g;
    s/\x{11F2}/\x{11F0}\x{11EB}/g;
}

sub jamo2puajamo {
  s/\x{11BD}\x{11B8}\x{11B8}/\x{F8E8}/g;
  s/\x{11AE}\x{11AE}\x{11B8}/\x{F883}/g;
  s/\x{11B7}\x{11AB}\x{11AB}/\x{F8AD}/g;
  s/\x{1100}\x{1103}/\x{F787}/g;
  s/\x{1102}\x{1109}/\x{F78D}/g;
  s/\x{1102}\x{110C}/\x{F78E}/g;
  s/\x{1102}\x{1112}/\x{F78F}/g;
  s/\x{1103}\x{1105}/\x{F793}/g;
  s/\x{1103}\x{1106}/\x{F794}/g;
  s/\x{1103}\x{1107}/\x{F795}/g;
  s/\x{1103}\x{1109}/\x{F796}/g;
  s/\x{1103}\x{110C}/\x{F797}/g;
  s/\x{1105}\x{1100}/\x{F799}/g;
  s/\x{1105}\x{1101}/\x{F79A}/g;
  s/\x{1105}\x{1103}/\x{F79C}/g;
  s/\x{1105}\x{1104}/\x{F79D}/g;
  s/\x{1105}\x{1106}/\x{F79F}/g;
  s/\x{1105}\x{1107}/\x{F7A0}/g;
  s/\x{1105}\x{1108}/\x{F7A1}/g;
  s/\x{1105}\x{112B}/\x{F7A2}/g;
  s/\x{1105}\x{1109}/\x{F7A3}/g;
  s/\x{1105}\x{110C}/\x{F7A4}/g;
  s/\x{1105}\x{110F}/\x{F7A5}/g;
  s/\x{1106}\x{1100}/\x{F7A9}/g;
  s/\x{1106}\x{1103}/\x{F7AA}/g;
  s/\x{1106}\x{1109}/\x{F7AC}/g;
  s/\x{1121}\x{1110}/\x{F7B9}/g;
  s/\x{1107}\x{110F}/\x{F7BC}/g;
  s/\x{1107}\x{1112}/\x{F7BF}/g;
  s/\x{1109}\x{1132}/\x{F7CB}/g;
  s/\x{110B}\x{1105}/\x{F7DC}/g;
  s/\x{110B}\x{1112}/\x{F7E6}/g;
  s/\x{110D}\x{1112}/\x{F7EB}/g;
  s/\x{1110}\x{1110}/\x{F7F7}/g;
  s/\x{1111}\x{1112}/\x{F7FA}/g;
  s/\x{1112}\x{1109}/\x{F7FD}/g;
  s/\x{1159}\x{1159}/\x{F800}/g;
  s/\x{1161}\x{1173}/\x{F80A}/g;
  s/\x{1163}\x{116E}/\x{F80F}/g;
  s/\x{1167}\x{1163}/\x{F817}/g;
  s/\x{1169}\x{1163}/\x{F81E}/g;
  s/\x{1169}\x{1164}/\x{F81F}/g;
  s/\x{116E}\x{1167}/\x{F837}/g;
  s/\x{1169}\x{1167}/\x{F822}/g;
  s/\x{1182}\x{116E}/\x{F825}/g;
  s/\x{116D}\x{1161}/\x{F829}/g;
  s/\x{116D}\x{1162}/\x{F82A}/g;
  s/\x{116D}\x{1165}/\x{F82D}/g;
  s/\x{1171}\x{1175}/\x{F83B}/g;
  s/\x{1172}\x{1162}/\x{F83E}/g;
  s/\x{1172}\x{1169}/\x{F843}/g;
  s/\x{1173}\x{1161}/\x{F847}/g;
  s/\x{1173}\x{1165}/\x{F848}/g;
  s/\x{1173}\x{1166}/\x{F849}/g;
  s/\x{1173}\x{1169}/\x{F84A}/g;
  s/\x{1175}\x{1178}/\x{F852}/g;
  s/\x{1175}\x{1164}/\x{F853}/g;
  s/\x{1175}\x{1167}/\x{F854}/g;
  s/\x{1175}\x{1168}/\x{F855}/g;
  s/\x{1175}\x{1183}/\x{F857}/g;
  s/\x{1175}\x{116D}/\x{F858}/g;
  s/\x{1175}\x{1172}/\x{F85A}/g;
  s/\x{1175}\x{1175}/\x{F85C}/g;
  s/\x{119E}\x{1161}/\x{F85F}/g;
  s/\x{119E}\x{1166}/\x{F861}/g;
  s/\x{11A8}\x{11AB}/\x{F86D}/g;
  s/\x{11A8}\x{11B8}/\x{F86F}/g;
  s/\x{11A8}\x{11BE}/\x{F872}/g;
  s/\x{11A8}\x{11BF}/\x{F873}/g;
  s/\x{11A8}\x{11C2}/\x{F874}/g;
  s/\x{11AB}\x{11AB}/\x{F877}/g;
  s/\x{11AB}\x{11AF}/\x{F879}/g;
  s/\x{11AB}\x{11BE}/\x{F87D}/g;
  s/\x{11AE}\x{11AE}/\x{F882}/g;
  s/\x{11AE}\x{11B8}/\x{F885}/g;
  s/\x{11AE}\x{11BA}/\x{F886}/g;
  s/\x{11AE}\x{11E7}/\x{F887}/g;
  s/\x{11AE}\x{11BD}/\x{F888}/g;
  s/\x{11AE}\x{11BE}/\x{F889}/g;
  s/\x{11AE}\x{11C0}/\x{F88A}/g;
  s/\x{11AF}\x{11A9}/\x{F88D}/g;
  s/\x{11B0}\x{11C2}/\x{F88F}/g;
  s/\x{11D0}\x{11BF}/\x{F894}/g;
  s/\x{11B1}\x{11C2}/\x{F898}/g;
  s/\x{11B2}\x{11AE}/\x{F89A}/g;
  s/\x{11B2}\x{11C1}/\x{F89C}/g;
  s/\x{11AF}\x{11F0}/\x{F8A2}/g;
  s/\x{11D9}\x{11C2}/\x{F8A8}/g;
  s/\x{11AF}\x{11BC}/\x{F8A9}/g;
  s/\x{11B7}\x{11AB}/\x{F8AC}/g;
  s/\x{11B7}\x{11B7}/\x{F8AF}/g;
  s/\x{11DC}\x{11BA}/\x{F8B1}/g;
  s/\x{11B7}\x{11BD}/\x{F8B5}/g;
  s/\x{11B8}\x{11AE}/\x{F8BA}/g;
  s/\x{11B8}\x{11B5}/\x{F8BC}/g;
  s/\x{11B8}\x{11B7}/\x{F8BD}/g;
  s/\x{11B8}\x{11B8}/\x{F8BE}/g;
  s/\x{11B9}\x{11AE}/\x{F8C0}/g;
  s/\x{11B8}\x{11BD}/\x{F8C1}/g;
  s/\x{11B8}\x{11BE}/\x{F8C2}/g;
  s/\x{11BA}\x{11B7}/\x{F8CA}/g;
  s/\x{11BA}\x{11E6}/\x{F8CC}/g;
  s/\x{11BA}\x{11E7}/\x{F8CE}/g;
  s/\x{11BA}\x{11E8}/\x{F8CF}/g;
  s/\x{11BA}\x{11EB}/\x{F8D0}/g;
  s/\x{11BA}\x{11BD}/\x{F8D1}/g;
  s/\x{11BA}\x{11BE}/\x{F8D2}/g;
  s/\x{11BA}\x{11C0}/\x{F8D3}/g;
  s/\x{11BA}\x{11C2}/\x{F8D4}/g;
  s/\x{11EB}\x{11B8}/\x{F8D6}/g;
  s/\x{11EB}\x{11E6}/\x{F8D7}/g;
  s/\x{11BC}\x{11B7}/\x{F8DB}/g;
  s/\x{11BC}\x{11BA}/\x{F8DC}/g;
  s/\x{11BC}\x{11C2}/\x{F8DF}/g;
  s/\x{11F0}\x{11A8}/\x{F8E1}/g;
  s/\x{11F0}\x{11BF}/\x{F8E4}/g;
  s/\x{11F0}\x{11C2}/\x{F8E5}/g;
  s/\x{11BD}\x{11B8}/\x{F8E7}/g;
  s/\x{11BD}\x{11BD}/\x{F8E9}/g;
  s/\x{11C1}\x{11BA}/\x{F8EF}/g;
  s/\x{11C1}\x{11C0}/\x{F8F0}/g;
  s/\x{115F}/\x{F784}/g;
  s/\x{1100}/\x{F785}/g;
  s/\x{1101}/\x{F786}/g;
  s/\x{1102}/\x{F788}/g;
  s/\x{1113}/\x{F789}/g;
  s/\x{1114}/\x{F78A}/g;
  s/\x{1115}/\x{F78B}/g;
  s/\x{1116}/\x{F78C}/g;
  s/\x{1103}/\x{F790}/g;
  s/\x{1117}/\x{F791}/g;
  s/\x{1104}/\x{F792}/g;
  s/\x{1105}/\x{F798}/g;
  s/\x{1118}/\x{F79B}/g;
  s/\x{1119}/\x{F79E}/g;
  s/\x{111A}/\x{F7A6}/g;
  s/\x{111B}/\x{F7A7}/g;
  s/\x{1106}/\x{F7A8}/g;
  s/\x{111C}/\x{F7AB}/g;
  s/\x{111D}/\x{F7AD}/g;
  s/\x{1107}/\x{F7AE}/g;
  s/\x{111E}/\x{F7AF}/g;
  s/\x{111F}/\x{F7B0}/g;
  s/\x{1120}/\x{F7B1}/g;
  s/\x{1108}/\x{F7B2}/g;
  s/\x{1121}/\x{F7B3}/g;
  s/\x{1122}/\x{F7B4}/g;
  s/\x{1123}/\x{F7B5}/g;
  s/\x{1124}/\x{F7B6}/g;
  s/\x{1125}/\x{F7B7}/g;
  s/\x{1126}/\x{F7B8}/g;
  s/\x{1127}/\x{F7BA}/g;
  s/\x{1128}/\x{F7BB}/g;
  s/\x{1129}/\x{F7BD}/g;
  s/\x{112A}/\x{F7BE}/g;
  s/\x{112B}/\x{F7C0}/g;
  s/\x{112C}/\x{F7C1}/g;
  s/\x{1109}/\x{F7C2}/g;
  s/\x{112D}/\x{F7C3}/g;
  s/\x{112E}/\x{F7C4}/g;
  s/\x{112F}/\x{F7C5}/g;
  s/\x{1130}/\x{F7C6}/g;
  s/\x{1131}/\x{F7C7}/g;
  s/\x{1132}/\x{F7C8}/g;
  s/\x{1133}/\x{F7C9}/g;
  s/\x{110A}/\x{F7CA}/g;
  s/\x{1134}/\x{F7CC}/g;
  s/\x{1135}/\x{F7CD}/g;
  s/\x{1136}/\x{F7CE}/g;
  s/\x{1137}/\x{F7CF}/g;
  s/\x{1138}/\x{F7D0}/g;
  s/\x{1139}/\x{F7D1}/g;
  s/\x{113A}/\x{F7D2}/g;
  s/\x{113B}/\x{F7D3}/g;
  s/\x{113C}/\x{F7D4}/g;
  s/\x{113D}/\x{F7D5}/g;
  s/\x{113E}/\x{F7D6}/g;
  s/\x{113F}/\x{F7D7}/g;
  s/\x{1140}/\x{F7D8}/g;
  s/\x{110B}/\x{F7D9}/g;
  s/\x{1141}/\x{F7DA}/g;
  s/\x{1142}/\x{F7DB}/g;
  s/\x{1143}/\x{F7DD}/g;
  s/\x{1144}/\x{F7DE}/g;
  s/\x{1145}/\x{F7DF}/g;
  s/\x{1146}/\x{F7E0}/g;
  s/\x{1147}/\x{F7E1}/g;
  s/\x{1148}/\x{F7E2}/g;
  s/\x{1149}/\x{F7E3}/g;
  s/\x{114A}/\x{F7E4}/g;
  s/\x{114B}/\x{F7E5}/g;
  s/\x{114C}/\x{F7E7}/g;
  s/\x{110C}/\x{F7E8}/g;
  s/\x{114D}/\x{F7E9}/g;
  s/\x{110D}/\x{F7EA}/g;
  s/\x{114E}/\x{F7EC}/g;
  s/\x{114F}/\x{F7ED}/g;
  s/\x{1150}/\x{F7EE}/g;
  s/\x{1151}/\x{F7EF}/g;
  s/\x{110E}/\x{F7F0}/g;
  s/\x{1152}/\x{F7F1}/g;
  s/\x{1153}/\x{F7F2}/g;
  s/\x{1154}/\x{F7F3}/g;
  s/\x{1155}/\x{F7F4}/g;
  s/\x{110F}/\x{F7F5}/g;
  s/\x{1110}/\x{F7F6}/g;
  s/\x{1111}/\x{F7F8}/g;
  s/\x{1156}/\x{F7F9}/g;
  s/\x{1157}/\x{F7FB}/g;
  s/\x{1112}/\x{F7FC}/g;
  s/\x{1158}/\x{F7FE}/g;
  s/\x{1159}/\x{F7FF}/g;
  s/\x{1160}/\x{F806}/g;
  s/\x{1161}/\x{F807}/g;
  s/\x{1176}/\x{F808}/g;
  s/\x{1177}/\x{F809}/g;
  s/\x{1162}/\x{F80B}/g;
  s/\x{1163}/\x{F80C}/g;
  s/\x{1178}/\x{F80D}/g;
  s/\x{1179}/\x{F80E}/g;
  s/\x{1164}/\x{F810}/g;
  s/\x{1165}/\x{F811}/g;
  s/\x{117A}/\x{F812}/g;
  s/\x{117B}/\x{F813}/g;
  s/\x{117C}/\x{F814}/g;
  s/\x{1166}/\x{F815}/g;
  s/\x{1167}/\x{F816}/g;
  s/\x{117D}/\x{F818}/g;
  s/\x{117E}/\x{F819}/g;
  s/\x{1168}/\x{F81A}/g;
  s/\x{1169}/\x{F81B}/g;
  s/\x{116A}/\x{F81C}/g;
  s/\x{116B}/\x{F81D}/g;
  s/\x{117F}/\x{F820}/g;
  s/\x{1180}/\x{F821}/g;
  s/\x{1181}/\x{F823}/g;
  s/\x{1182}/\x{F824}/g;
  s/\x{1183}/\x{F826}/g;
  s/\x{116C}/\x{F827}/g;
  s/\x{116D}/\x{F828}/g;
  s/\x{1184}/\x{F82B}/g;
  s/\x{1185}/\x{F82C}/g;
  s/\x{1186}/\x{F82E}/g;
  s/\x{1187}/\x{F82F}/g;
  s/\x{1188}/\x{F830}/g;
  s/\x{116E}/\x{F831}/g;
  s/\x{1189}/\x{F832}/g;
  s/\x{118A}/\x{F833}/g;
  s/\x{116F}/\x{F834}/g;
  s/\x{118B}/\x{F835}/g;
  s/\x{1170}/\x{F836}/g;
  s/\x{118C}/\x{F838}/g;
  s/\x{118D}/\x{F839}/g;
  s/\x{1171}/\x{F83A}/g;
  s/\x{1172}/\x{F83C}/g;
  s/\x{118E}/\x{F83D}/g;
  s/\x{118F}/\x{F83F}/g;
  s/\x{1190}/\x{F840}/g;
  s/\x{1191}/\x{F841}/g;
  s/\x{1192}/\x{F842}/g;
  s/\x{1193}/\x{F844}/g;
  s/\x{1194}/\x{F845}/g;
  s/\x{1173}/\x{F846}/g;
  s/\x{1195}/\x{F84B}/g;
  s/\x{1196}/\x{F84C}/g;
  s/\x{1174}/\x{F84D}/g;
  s/\x{1197}/\x{F84E}/g;
  s/\x{1175}/\x{F84F}/g;
  s/\x{1198}/\x{F850}/g;
  s/\x{1199}/\x{F851}/g;
  s/\x{119A}/\x{F856}/g;
  s/\x{119B}/\x{F859}/g;
  s/\x{119C}/\x{F85B}/g;
  s/\x{119D}/\x{F85D}/g;
  s/\x{119E}/\x{F85E}/g;
  s/\x{119F}/\x{F860}/g;
  s/\x{11A0}/\x{F862}/g;
  s/\x{11A1}/\x{F863}/g;
  s/\x{11A2}/\x{F864}/g;
  s/\x{11A8}/\x{F86B}/g;
  s/\x{11A9}/\x{F86C}/g;
  s/\x{11C3}/\x{F86E}/g;
  s/\x{11AA}/\x{F870}/g;
  s/\x{11C4}/\x{F871}/g;
  s/\x{11AB}/\x{F875}/g;
  s/\x{11C5}/\x{F876}/g;
  s/\x{11C6}/\x{F878}/g;
  s/\x{11C7}/\x{F87A}/g;
  s/\x{11C8}/\x{F87B}/g;
  s/\x{11AC}/\x{F87C}/g;
  s/\x{11C9}/\x{F87E}/g;
  s/\x{11AD}/\x{F87F}/g;
  s/\x{11AE}/\x{F880}/g;
  s/\x{11CA}/\x{F881}/g;
  s/\x{11CB}/\x{F884}/g;
  s/\x{11AF}/\x{F88B}/g;
  s/\x{11B0}/\x{F88C}/g;
  s/\x{11CC}/\x{F88E}/g;
  s/\x{11CD}/\x{F890}/g;
  s/\x{11CE}/\x{F891}/g;
  s/\x{11CF}/\x{F892}/g;
  s/\x{11D0}/\x{F893}/g;
  s/\x{11B1}/\x{F895}/g;
  s/\x{11D1}/\x{F896}/g;
  s/\x{11D2}/\x{F897}/g;
  s/\x{11B2}/\x{F899}/g;
  s/\x{11D3}/\x{F89B}/g;
  s/\x{11D4}/\x{F89D}/g;
  s/\x{11D5}/\x{F89E}/g;
  s/\x{11B3}/\x{F89F}/g;
  s/\x{11D6}/\x{F8A0}/g;
  s/\x{11D7}/\x{F8A1}/g;
  s/\x{11D8}/\x{F8A3}/g;
  s/\x{11B4}/\x{F8A4}/g;
  s/\x{11B5}/\x{F8A5}/g;
  s/\x{11B6}/\x{F8A6}/g;
  s/\x{11D9}/\x{F8A7}/g;
  s/\x{11B7}/\x{F8AA}/g;
  s/\x{11DA}/\x{F8AB}/g;
  s/\x{11DB}/\x{F8AE}/g;
  s/\x{11DC}/\x{F8B0}/g;
  s/\x{11DD}/\x{F8B2}/g;
  s/\x{11DE}/\x{F8B3}/g;
  s/\x{11DF}/\x{F8B4}/g;
  s/\x{11E0}/\x{F8B6}/g;
  s/\x{11E1}/\x{F8B7}/g;
  s/\x{11E2}/\x{F8B8}/g;
  s/\x{11B8}/\x{F8B9}/g;
  s/\x{11E3}/\x{F8BB}/g;
  s/\x{11B9}/\x{F8BF}/g;
  s/\x{11E4}/\x{F8C3}/g;
  s/\x{11E5}/\x{F8C4}/g;
  s/\x{11E6}/\x{F8C5}/g;
  s/\x{11BA}/\x{F8C6}/g;
  s/\x{11E7}/\x{F8C7}/g;
  s/\x{11E8}/\x{F8C8}/g;
  s/\x{11E9}/\x{F8C9}/g;
  s/\x{11EA}/\x{F8CB}/g;
  s/\x{11BB}/\x{F8CD}/g;
  s/\x{11EB}/\x{F8D5}/g;
  s/\x{11BC}/\x{F8D8}/g;
  s/\x{11EC}/\x{F8D9}/g;
  s/\x{11ED}/\x{F8DA}/g;
  s/\x{11EE}/\x{F8DD}/g;
  s/\x{11EF}/\x{F8DE}/g;
  s/\x{11F0}/\x{F8E0}/g;
  s/\x{11F1}/\x{F8E2}/g;
  s/\x{11F2}/\x{F8E3}/g;
  s/\x{11BD}/\x{F8E6}/g;
  s/\x{11BE}/\x{F8EA}/g;
  s/\x{11BF}/\x{F8EB}/g;
  s/\x{11C0}/\x{F8EC}/g;
  s/\x{11C1}/\x{F8ED}/g;
  s/\x{11F3}/\x{F8EE}/g;
  s/\x{11F4}/\x{F8F1}/g;
  s/\x{11C2}/\x{F8F2}/g;
  s/\x{11F5}/\x{F8F3}/g;
  s/\x{11F6}/\x{F8F4}/g;
  s/\x{11F7}/\x{F8F5}/g;
  s/\x{11F8}/\x{F8F6}/g;
  s/\x{11F9}/\x{F8F7}/g;
}

sub jamo2hypua {
  s/\x{110C}\x{110C}\x{1112}\x{1165}\x{1175}\x{11AF}\x{11AF}\x{11BF}/\x{F267}/g;
  s/\x{110C}\x{110C}\x{1112}\x{1165}\x{1175}\x{11AF}\x{11AF}/\x{F266}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{1175}\x{11AF}\x{11A8}/\x{E9F8}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F5CA}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F5CE}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F5A8}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{1175}\x{11BC}/\x{F5AD}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{F25E}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F241}/g;
  s/\x{110C}\x{110C}\x{1112}\x{1165}\x{1175}\x{11AF}/\x{F265}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1165}\x{1175}\x{11AB}/\x{EFBE}/g;
  s/\x{110B}\x{1111}\x{1169}\x{1161}\x{1175}\x{11AB}/\x{EFB3}/g;
  s/\x{110B}\x{1107}\x{116E}\x{1165}\x{1175}\x{11AF}/\x{EF05}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{EBC7}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11BA}\x{11A8}/\x{EB9D}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1173}\x{11BA}\x{11BA}/\x{EA9C}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{EA65}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}\x{11AF}\x{11AF}/\x{EA48}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{1175}\x{11AF}/\x{EA1F}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{1175}\x{11AB}/\x{EA1E}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{1175}\x{11BC}/\x{E9F9}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{1175}\x{11AF}/\x{E9F7}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11AF}\x{11B8}\x{11BC}/\x{E9C5}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11AF}\x{11B8}\x{11BC}/\x{E830}/g;
  s/\x{1107}\x{110B}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{E892}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}\x{1175}\x{11BA}/\x{E75A}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1165}\x{11AF}\x{11B8}/\x{E7C3}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{1175}\x{11BA}/\x{E7B8}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{1175}\x{11B7}/\x{E7B7}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{1175}\x{11AF}/\x{E7B6}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{1175}\x{11AB}/\x{E7B5}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{1175}\x{11AB}/\x{E7AE}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{1165}\x{1175}/\x{E7AC}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1169}\x{1175}\x{11BA}/\x{E7A5}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1167}\x{1175}\x{11BA}/\x{E7A2}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}\x{11B8}\x{11BC}/\x{E79A}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}\x{1175}\x{11AF}/\x{E79D}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1161}\x{1175}\x{11BA}/\x{E797}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1161}\x{1175}\x{11AB}/\x{E796}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}\x{11B7}\x{11BC}/\x{E8C9}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1169}\x{1161}\x{11AB}/\x{E8BE}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11AF}\x{11B8}\x{11BC}/\x{E6B2}/g;
  s/\x{1103}\x{1103}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E3CB}/g;
  s/\x{1103}\x{1103}\x{1165}\x{1173}\x{11AF}\x{11B8}/\x{E3C4}/g;
  s/\x{1100}\x{1172}\x{1167}\x{1175}\x{11AF}\x{11A8}/\x{E168}/g;
  s/\x{1100}\x{116D}\x{1163}\x{1175}\x{11AF}\x{11A8}/\x{E135}/g;
  s/\x{1100}\x{1100}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{E1EC}/g;
  s/\x{1100}\x{1100}\x{1165}\x{11BC}\x{11A8}\x{11A8}/\x{E1DE}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11AF}\x{11B7}/\x{F66C}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11AF}\x{11AB}/\x{F66B}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F64A}/g;
  s/\x{1159}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F64E}/g;
  s/\x{1159}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{F63D}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11B8}\x{11BC}/\x{F621}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F61F}/g;
  s/\x{1159}\x{1169}\x{1161}\x{1175}\x{11BC}/\x{F626}/g;
  s/\x{1150}\x{1150}\x{1173}\x{11B7}\x{11BC}/\x{F2D9}/g;
  s/\x{1150}\x{1150}\x{1172}\x{1167}\x{11AB}/\x{F2D6}/g;
  s/\x{1150}\x{1150}\x{1163}\x{11B7}\x{11BC}/\x{F2CC}/g;
  s/\x{114E}\x{114E}\x{1172}\x{1167}\x{11AB}/\x{F29E}/g;
  s/\x{114E}\x{114E}\x{1161}\x{11B7}\x{11BC}/\x{F28E}/g;
  s/\x{114C}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{F139}/g;
  s/\x{114C}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{F133}/g;
  s/\x{114C}\x{116D}\x{1163}\x{1175}\x{11BC}/\x{F127}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11B7}\x{11BA}/\x{F118}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F116}/g;
  s/\x{114C}\x{1169}\x{1161}\x{1175}\x{11BC}/\x{F11C}/g;
  s/\x{1140}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{ED66}/g;
  s/\x{1140}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{ED68}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11B8}\x{11BD}/\x{F559}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{F556}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11AF}\x{11A8}/\x{F555}/g;
  s/\x{1112}\x{1175}\x{11B7}\x{11BA}\x{11BA}/\x{F528}/g;
  s/\x{1112}\x{1175}\x{11B7}\x{11B8}\x{11BA}/\x{F526}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F50D}/g;
  s/\x{1112}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F513}/g;
  s/\x{1112}\x{1172}\x{1167}\x{1175}\x{11AF}/\x{F512}/g;
  s/\x{1112}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{F4FA}/g;
  s/\x{1112}\x{116E}\x{1165}\x{1173}\x{11AF}/\x{F4FF}/g;
  s/\x{1112}\x{116E}\x{1165}\x{1173}\x{11AB}/\x{F4FE}/g;
  s/\x{1112}\x{1169}\x{1161}\x{11F0}\x{11BA}/\x{F4DD}/g;
  s/\x{1112}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F4DB}/g;
  s/\x{1112}\x{1169}\x{1161}\x{1175}\x{11F9}/\x{F4DE}/g;
  s/\x{1112}\x{1167}\x{1175}\x{11AF}\x{11F9}/\x{F4D1}/g;
  s/\x{1112}\x{1161}\x{11AF}\x{11B7}\x{11C2}/\x{F4B2}/g;
  s/\x{1112}\x{1161}\x{11AF}\x{11AE}\x{11C2}/\x{F4B0}/g;
  s/\x{1112}\x{1161}\x{1175}\x{11F0}\x{11C2}/\x{F4BC}/g;
  s/\x{1112}\x{1112}\x{119E}\x{1175}\x{11F0}/\x{F5E7}/g;
  s/\x{1112}\x{1112}\x{119E}\x{1175}\x{11BC}/\x{F5E6}/g;
  s/\x{1112}\x{1112}\x{119E}\x{1175}\x{11A8}/\x{F5E5}/g;
  s/\x{1112}\x{1112}\x{1175}\x{11B7}\x{11BC}/\x{F5DF}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11B7}\x{11BC}/\x{F5D3}/g;
  s/\x{1112}\x{1112}\x{1173}\x{1175}\x{11F9}/\x{F5DB}/g;
  s/\x{1112}\x{1112}\x{1173}\x{1175}\x{11F0}/\x{F5D9}/g;
  s/\x{1112}\x{1112}\x{1173}\x{1175}\x{11C2}/\x{F5DA}/g;
  s/\x{1112}\x{1112}\x{1173}\x{1175}\x{11A8}/\x{F5D8}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{11F0}/\x{F5CC}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{11BC}/\x{F5CB}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{11AF}/\x{F5C9}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{11AB}/\x{F5C8}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}\x{1175}/\x{F5CD}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1165}\x{11AB}/\x{F5C6}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11B8}\x{11BC}/\x{F5BA}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11B7}\x{11BC}/\x{F5B8}/g;
  s/\x{1112}\x{1112}\x{116E}\x{1175}\x{11F9}/\x{F5C2}/g;
  s/\x{1112}\x{1112}\x{116E}\x{1165}\x{11F9}/\x{F5C0}/g;
  s/\x{1112}\x{1112}\x{116E}\x{1165}\x{11AB}/\x{F5BF}/g;
  s/\x{1112}\x{1112}\x{116D}\x{11B7}\x{11BC}/\x{F5B4}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11B7}\x{11BC}/\x{F5A1}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11AF}\x{11F9}/\x{F59F}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1175}\x{11F0}/\x{F5B1}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1175}\x{11BC}/\x{F5B0}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1175}\x{11A8}/\x{F5AF}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11F9}/\x{F5AB}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11F0}/\x{F5AA}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11BC}/\x{F5A9}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11AF}/\x{F5A7}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11AB}/\x{F5A6}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{11A8}/\x{F5A5}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}\x{1175}/\x{F5AC}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11AF}\x{11F9}/\x{F594}/g;
  s/\x{1112}\x{1112}\x{1167}\x{1175}\x{11BC}/\x{F59A}/g;
  s/\x{1112}\x{1112}\x{1165}\x{11B7}\x{11BC}/\x{F58C}/g;
  s/\x{1112}\x{1112}\x{1165}\x{1175}\x{11BC}/\x{F58F}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11B8}\x{11BC}/\x{F584}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11B7}\x{11BC}/\x{F582}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11B8}\x{11BC}/\x{F579}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11B7}\x{11BC}/\x{F577}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11AF}\x{11F9}/\x{F575}/g;
  s/\x{1112}\x{1112}\x{1161}\x{1175}\x{11F0}/\x{F57E}/g;
  s/\x{1112}\x{1112}\x{1161}\x{1175}\x{11BC}/\x{F57D}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1175}\x{11AB}/\x{F56D}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1165}\x{1175}/\x{F56B}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1161}\x{11AB}/\x{F569}/g;
  s/\x{1112}\x{1109}\x{116E}\x{1165}\x{1175}/\x{F564}/g;
  s/\x{1112}\x{1109}\x{1169}\x{1161}\x{11AB}/\x{F561}/g;
  s/\x{1111}\x{1165}\x{1175}\x{11AB}\x{11BA}/\x{F465}/g;
  s/\x{1110}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F415}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F3CD}/g;
  s/\x{110F}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F3D1}/g;
  s/\x{110F}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{F3C0}/g;
  s/\x{110F}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F3B1}/g;
  s/\x{110E}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F321}/g;
  s/\x{110E}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F324}/g;
  s/\x{110E}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{F313}/g;
  s/\x{110E}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F2FF}/g;
  s/\x{110E}\x{1167}\x{11AE}\x{11AE}\x{11B8}/\x{F2F1}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11AF}\x{11F9}/\x{F1DA}/g;
  s/\x{110C}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{F1D0}/g;
  s/\x{110C}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{F1D5}/g;
  s/\x{110C}\x{1172}\x{1167}\x{1175}\x{11B8}/\x{F1D4}/g;
  s/\x{110C}\x{1172}\x{1167}\x{1175}\x{11AB}/\x{F1D3}/g;
  s/\x{110C}\x{116E}\x{1167}\x{11AF}\x{11F9}/\x{F1C0}/g;
  s/\x{110C}\x{116D}\x{1175}\x{11AF}\x{11F9}/\x{F1B1}/g;
  s/\x{110C}\x{1169}\x{1161}\x{11B7}\x{11BC}/\x{F1A6}/g;
  s/\x{110C}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{F1A5}/g;
  s/\x{110C}\x{1167}\x{11AE}\x{11AE}\x{11B8}/\x{F193}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11B8}\x{11BC}/\x{F256}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11F9}/\x{F262}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11F0}/\x{F261}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11BC}/\x{F260}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11B7}/\x{F25F}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11AF}/\x{F25D}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11AB}/\x{F25C}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}\x{11A8}/\x{F25B}/g;
  s/\x{110C}\x{110C}\x{1175}\x{11B7}\x{11BC}/\x{F24C}/g;
  s/\x{110C}\x{110C}\x{1175}\x{11AF}\x{11F9}/\x{F24B}/g;
  s/\x{110C}\x{110C}\x{1173}\x{11BD}\x{11BD}/\x{F248}/g;
  s/\x{110C}\x{110C}\x{1173}\x{11B7}\x{11BC}/\x{F245}/g;
  s/\x{110C}\x{110C}\x{1173}\x{1175}\x{11F9}/\x{F24A}/g;
  s/\x{110C}\x{110C}\x{1173}\x{1175}\x{11F0}/\x{F249}/g;
  s/\x{110C}\x{110C}\x{1172}\x{11B7}\x{11BC}/\x{F23D}/g;
  s/\x{110C}\x{110C}\x{1172}\x{11AF}\x{11F9}/\x{F23C}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1175}\x{11BC}/\x{F244}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1167}\x{11F9}/\x{F242}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1167}\x{11AF}/\x{F240}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1167}\x{11AB}/\x{F23F}/g;
  s/\x{110C}\x{110C}\x{116E}\x{11B7}\x{11BC}/\x{F238}/g;
  s/\x{110C}\x{110C}\x{116E}\x{11AF}\x{11F9}/\x{F237}/g;
  s/\x{110C}\x{110C}\x{116E}\x{1175}\x{11F9}/\x{F23B}/g;
  s/\x{110C}\x{110C}\x{116D}\x{11B7}\x{11BC}/\x{F235}/g;
  s/\x{110C}\x{110C}\x{1169}\x{11BD}\x{11BD}/\x{F230}/g;
  s/\x{110C}\x{110C}\x{1169}\x{11B7}\x{11BC}/\x{F22E}/g;
  s/\x{110C}\x{110C}\x{1169}\x{11AF}\x{11F9}/\x{F22D}/g;
  s/\x{110C}\x{110C}\x{1169}\x{1175}\x{11F9}/\x{F234}/g;
  s/\x{110C}\x{110C}\x{1169}\x{1161}\x{11F0}/\x{F232}/g;
  s/\x{110C}\x{110C}\x{1167}\x{11AF}\x{11F9}/\x{F22B}/g;
  s/\x{110C}\x{110C}\x{1163}\x{11B8}\x{11BC}/\x{F227}/g;
  s/\x{110C}\x{110C}\x{1163}\x{11B7}\x{11BC}/\x{F226}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11B8}\x{11B8}/\x{F223}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11B7}\x{11BC}/\x{F222}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11AF}\x{11F9}/\x{F221}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11AF}\x{11AF}/\x{F220}/g;
  s/\x{110C}\x{110C}\x{1112}\x{1165}\x{1175}/\x{F264}/g;
  s/\x{110B}\x{119E}\x{1165}\x{11AF}\x{11B8}/\x{EEA1}/g;
  s/\x{110B}\x{1173}\x{11AF}\x{11B8}\x{11C2}/\x{EE62}/g;
  s/\x{110B}\x{1173}\x{11AF}\x{11B8}\x{11C1}/\x{EE61}/g;
  s/\x{110B}\x{1173}\x{1175}\x{11AF}\x{11F9}/\x{EE74}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{EE52}/g;
  s/\x{110B}\x{1172}\x{1167}\x{1175}\x{11F0}/\x{EE57}/g;
  s/\x{110B}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{EE56}/g;
  s/\x{110B}\x{1172}\x{1165}\x{1175}\x{11AB}/\x{EE4D}/g;
  s/\x{110B}\x{116E}\x{11EB}\x{11B8}\x{11BC}/\x{EE31}/g;
  s/\x{110B}\x{116E}\x{11BA}\x{11B8}\x{11BC}/\x{EE2D}/g;
  s/\x{110B}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{EE42}/g;
  s/\x{110B}\x{116E}\x{1165}\x{11F0}\x{11BA}/\x{EE3D}/g;
  s/\x{110B}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{EE3A}/g;
  s/\x{110B}\x{116E}\x{1165}\x{11AF}\x{11BF}/\x{EE39}/g;
  s/\x{110B}\x{116E}\x{1165}\x{1173}\x{11AB}/\x{EE3F}/g;
  s/\x{110B}\x{116D}\x{1161}\x{1175}\x{11AB}/\x{EE1C}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11BA}\x{11BA}/\x{EE01}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11A8}\x{11C2}/\x{EDFE}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11A8}\x{11A8}/\x{EDFD}/g;
  s/\x{110B}\x{1169}\x{1175}\x{11AF}\x{11F9}/\x{EE15}/g;
  s/\x{110B}\x{1169}\x{1175}\x{11AF}\x{11AF}/\x{EE14}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11BA}\x{11B8}/\x{EE0C}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11B7}\x{11BA}/\x{EE0B}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11AE}\x{11C0}/\x{EE0A}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11AE}\x{11B8}/\x{EE09}/g;
  s/\x{110B}\x{1167}\x{11BC}\x{11A8}\x{11A8}/\x{EDF7}/g;
  s/\x{110B}\x{1167}\x{11AF}\x{11B8}\x{11BC}/\x{EDEB}/g;
  s/\x{110B}\x{1167}\x{11A8}\x{11BA}\x{11A8}/\x{EDE4}/g;
  s/\x{110B}\x{1165}\x{11B7}\x{11BA}\x{11BA}/\x{EDD1}/g;
  s/\x{110B}\x{1165}\x{1175}\x{11AB}\x{11AE}/\x{EDE1}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11B8}\x{11BA}/\x{EDDE}/g;
  s/\x{110B}\x{1161}\x{11B8}\x{11AF}\x{11C1}/\x{EDB2}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11BA}\x{11BA}/\x{EDAC}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11B8}\x{11C2}/\x{EDAB}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11B8}\x{11C1}/\x{EDAA}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11B8}\x{11BA}/\x{EDA9}/g;
  s/\x{110B}\x{1140}\x{116E}\x{1167}\x{1175}/\x{EF4C}/g;
  s/\x{110B}\x{1140}\x{116E}\x{1165}\x{1175}/\x{EF4A}/g;
  s/\x{110B}\x{1140}\x{1169}\x{1161}\x{1175}/\x{EF45}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1175}\x{11AF}/\x{EFC2}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1175}\x{11AB}/\x{EFC1}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1175}\x{11A8}/\x{EFC0}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1165}\x{11AB}/\x{EFBC}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1165}\x{1175}/\x{EFBD}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1161}\x{11AB}/\x{EFBA}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11AF}\x{11BF}/\x{EFAE}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11AF}\x{11B7}/\x{EFAD}/g;
  s/\x{110B}\x{1111}\x{1169}\x{1161}\x{1175}/\x{EFB2}/g;
  s/\x{110B}\x{1111}\x{1165}\x{1175}\x{11BF}/\x{EFA9}/g;
  s/\x{110B}\x{1111}\x{1165}\x{1175}\x{11AF}/\x{EFA8}/g;
  s/\x{110B}\x{1111}\x{1165}\x{1175}\x{11AB}/\x{EFA7}/g;
  s/\x{110B}\x{1111}\x{1165}\x{1175}\x{11A8}/\x{EFA6}/g;
  s/\x{110B}\x{1111}\x{1161}\x{11AF}\x{11BF}/\x{EF9B}/g;
  s/\x{110B}\x{1111}\x{1161}\x{1175}\x{11BF}/\x{EF9F}/g;
  s/\x{110B}\x{1111}\x{1161}\x{1175}\x{11AB}/\x{EF9E}/g;
  s/\x{110B}\x{110E}\x{1172}\x{1161}\x{11AB}/\x{EF93}/g;
  s/\x{110B}\x{110E}\x{116E}\x{1165}\x{1175}/\x{EF90}/g;
  s/\x{110B}\x{110E}\x{116E}\x{1161}\x{11AB}/\x{EF8E}/g;
  s/\x{110B}\x{110C}\x{116E}\x{1165}\x{1175}/\x{EF7D}/g;
  s/\x{110B}\x{110C}\x{116E}\x{1161}\x{11AB}/\x{EF7B}/g;
  s/\x{110B}\x{110B}\x{1172}\x{1161}\x{11AB}/\x{EF68}/g;
  s/\x{110B}\x{110B}\x{116D}\x{11B7}\x{11BC}/\x{EF60}/g;
  s/\x{110B}\x{110B}\x{1167}\x{1175}\x{11BA}/\x{EF5C}/g;
  s/\x{110B}\x{1109}\x{1173}\x{1175}\x{11AB}/\x{EF37}/g;
  s/\x{110B}\x{1109}\x{1172}\x{1167}\x{1175}/\x{EF32}/g;
  s/\x{110B}\x{1109}\x{1172}\x{1161}\x{11AB}/\x{EF30}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1165}\x{1175}/\x{EF2D}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1161}\x{11BC}/\x{EF2A}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1161}\x{11AB}/\x{EF29}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1161}\x{1175}/\x{EF2B}/g;
  s/\x{110B}\x{1109}\x{1167}\x{1175}\x{11AB}/\x{EF23}/g;
  s/\x{110B}\x{1107}\x{116E}\x{1175}\x{11BC}/\x{EF07}/g;
  s/\x{110B}\x{1107}\x{116E}\x{1165}\x{1175}/\x{EF04}/g;
  s/\x{110B}\x{1107}\x{1169}\x{1161}\x{11B7}/\x{EEFE}/g;
  s/\x{110B}\x{1107}\x{1167}\x{1175}\x{11AB}/\x{EEF8}/g;
  s/\x{110B}\x{1107}\x{1165}\x{1175}\x{11C0}/\x{EEF5}/g;
  s/\x{110B}\x{1107}\x{1165}\x{1175}\x{11AF}/\x{EEF4}/g;
  s/\x{110B}\x{1107}\x{1165}\x{1175}\x{11AB}/\x{EEF3}/g;
  s/\x{110B}\x{1107}\x{1161}\x{1175}\x{11B7}/\x{EEEE}/g;
  s/\x{110B}\x{1107}\x{1161}\x{1175}\x{11AF}/\x{EEED}/g;
  s/\x{110B}\x{1103}\x{1165}\x{1175}\x{11AB}/\x{EED7}/g;
  s/\x{110B}\x{1100}\x{116E}\x{1167}\x{1175}/\x{EEC7}/g;
  s/\x{110B}\x{1100}\x{116E}\x{1165}\x{1175}/\x{EEC5}/g;
  s/\x{110B}\x{1100}\x{1169}\x{1161}\x{11AB}/\x{EEBE}/g;
  s/\x{110B}\x{1100}\x{1169}\x{1161}\x{1175}/\x{EEBF}/g;
  s/\x{110B}\x{1100}\x{1167}\x{1175}\x{11AB}/\x{EEB9}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11B8}\x{11BC}/\x{E987}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11F0}\x{11BA}/\x{E9A4}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11B7}\x{11BA}/\x{E99F}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{E99D}/g;
  s/\x{1109}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{E944}/g;
  s/\x{1109}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{E948}/g;
  s/\x{1109}\x{116E}\x{1175}\x{11BA}\x{11A8}/\x{E936}/g;
  s/\x{1109}\x{116E}\x{1175}\x{11B8}\x{11BC}/\x{E935}/g;
  s/\x{1109}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{E934}/g;
  s/\x{1109}\x{1169}\x{11B8}\x{11BA}\x{11AE}/\x{E90D}/g;
  s/\x{1109}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E912}/g;
  s/\x{1109}\x{1169}\x{1161}\x{1175}\x{11F9}/\x{E915}/g;
  s/\x{1109}\x{1167}\x{11AF}\x{11B8}\x{11BC}/\x{E8FC}/g;
  s/\x{1109}\x{1165}\x{11BC}\x{11A8}\x{11A8}/\x{E8EF}/g;
  s/\x{1109}\x{1165}\x{1175}\x{11C1}\x{11C0}/\x{E8FB}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11BA}\x{11BA}/\x{E8F9}/g;
  s/\x{1109}\x{1161}\x{1175}\x{11B8}\x{11BC}/\x{E8E1}/g;
  s/\x{1109}\x{110F}\x{1169}\x{1175}\x{11F0}/\x{EC72}/g;
  s/\x{1109}\x{110F}\x{1169}\x{1175}\x{11BC}/\x{EC71}/g;
  s/\x{1109}\x{110F}\x{1169}\x{1161}\x{1175}/\x{EC6F}/g;
  s/\x{1109}\x{110C}\x{119E}\x{1175}\x{11F0}/\x{EC5D}/g;
  s/\x{1109}\x{110C}\x{119E}\x{1175}\x{11BC}/\x{EC5C}/g;
  s/\x{1109}\x{110C}\x{119E}\x{1175}\x{11AF}/\x{EC5B}/g;
  s/\x{1109}\x{110C}\x{119E}\x{1175}\x{11A8}/\x{EC5A}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11BA}\x{11C2}/\x{EC4D}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11AF}\x{11F9}/\x{EC49}/g;
  s/\x{1109}\x{110C}\x{1173}\x{1175}\x{11AF}/\x{EC44}/g;
  s/\x{1109}\x{110C}\x{116E}\x{1175}\x{11A8}/\x{EC37}/g;
  s/\x{1109}\x{110C}\x{116E}\x{1165}\x{1175}/\x{EC35}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1175}\x{11B7}/\x{EC26}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1175}\x{11AF}/\x{EC25}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1175}\x{11AB}/\x{EC24}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1175}\x{11A8}/\x{EC23}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1161}\x{11A8}/\x{EC21}/g;
  s/\x{1109}\x{110C}\x{1167}\x{1175}\x{11AB}/\x{EC15}/g;
  s/\x{1109}\x{110C}\x{1165}\x{1175}\x{11BA}/\x{EC0D}/g;
  s/\x{1109}\x{110C}\x{1165}\x{1175}\x{11B8}/\x{EC0C}/g;
  s/\x{1109}\x{110C}\x{1165}\x{1175}\x{11AF}/\x{EC0B}/g;
  s/\x{1109}\x{110C}\x{1165}\x{1175}\x{11AB}/\x{EC0A}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11BA}\x{11AF}/\x{EBF3}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11AF}\x{11B8}/\x{EBEF}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11F0}/\x{EBFC}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11BC}/\x{EBFB}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11BA}/\x{EBFA}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11AF}/\x{EBF9}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11AB}/\x{EBF8}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}\x{11A8}/\x{EBF7}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11B8}\x{11BC}/\x{EBDC}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11B7}\x{11BA}/\x{EBDA}/g;
  s/\x{1109}\x{1109}\x{119E}\x{1175}\x{11F0}/\x{EBE3}/g;
  s/\x{1109}\x{1109}\x{119E}\x{1175}\x{11AB}/\x{EBE2}/g;
  s/\x{1109}\x{1109}\x{1175}\x{11B7}\x{11BC}/\x{EBCF}/g;
  s/\x{1109}\x{1109}\x{1175}\x{11AF}\x{11F9}/\x{EBCE}/g;
  s/\x{1109}\x{1109}\x{1173}\x{11AF}\x{11F9}/\x{EBCB}/g;
  s/\x{1109}\x{1109}\x{1172}\x{11B7}\x{11BC}/\x{EBC0}/g;
  s/\x{1109}\x{1109}\x{1172}\x{11AF}\x{11F9}/\x{EBBF}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1175}\x{11F0}/\x{EBCA}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1175}\x{11BC}/\x{EBC9}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1167}\x{11BC}/\x{EBC5}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1167}\x{11AB}/\x{EBC4}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1167}\x{1175}/\x{EBC6}/g;
  s/\x{1109}\x{1109}\x{116E}\x{11B7}\x{11BC}/\x{EBB7}/g;
  s/\x{1109}\x{1109}\x{116E}\x{11AF}\x{11F9}/\x{EBB6}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1167}\x{11AB}/\x{EBBD}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1167}\x{1175}/\x{EBBE}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1161}\x{11BC}/\x{EBBB}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1161}\x{11AB}/\x{EBBA}/g;
  s/\x{1109}\x{1109}\x{116D}\x{11B7}\x{11BC}/\x{EBB3}/g;
  s/\x{1109}\x{1109}\x{1169}\x{11B7}\x{11BC}/\x{EBB1}/g;
  s/\x{1109}\x{1109}\x{1167}\x{11AF}\x{11F9}/\x{EBAE}/g;
  s/\x{1109}\x{1109}\x{1163}\x{11B7}\x{11BC}/\x{EBAB}/g;
  s/\x{1109}\x{1109}\x{1161}\x{11BA}\x{11C2}/\x{EBA9}/g;
  s/\x{1109}\x{1109}\x{1161}\x{11B7}\x{11BC}/\x{EBA8}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11AF}\x{11AF}/\x{EB92}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11F0}/\x{EB9F}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11BC}/\x{EB9E}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11BA}/\x{EB9C}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11B7}/\x{EB9B}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11AF}/\x{EB9A}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11AB}/\x{EB99}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}\x{11A8}/\x{EB98}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11BA}\x{11C2}/\x{EB89}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11BA}\x{11A8}/\x{EB88}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11AF}\x{11BA}/\x{EB7C}/g;
  s/\x{1109}\x{1107}\x{1173}\x{1175}\x{11AF}/\x{EB81}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11AF}\x{11B7}/\x{EB6B}/g;
  s/\x{1109}\x{1107}\x{116E}\x{1167}\x{1175}/\x{EB74}/g;
  s/\x{1109}\x{1107}\x{116E}\x{1165}\x{1175}/\x{EB72}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11F0}\x{11BA}/\x{EB60}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11BA}\x{11A8}/\x{EB5D}/g;
  s/\x{1109}\x{1107}\x{1169}\x{1161}\x{1175}/\x{EB63}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11BA}\x{11C2}/\x{EB52}/g;
  s/\x{1109}\x{1107}\x{1165}\x{1175}\x{11BA}/\x{EB4B}/g;
  s/\x{1109}\x{1107}\x{1165}\x{1175}\x{11B7}/\x{EB4A}/g;
  s/\x{1109}\x{1107}\x{1165}\x{1175}\x{11AF}/\x{EB49}/g;
  s/\x{1109}\x{1107}\x{1165}\x{1175}\x{11AB}/\x{EB48}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11AF}\x{11AF}/\x{EB2D}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11AF}\x{11A8}/\x{EB2C}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11F0}/\x{EB3A}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11BA}/\x{EB39}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11B7}/\x{EB38}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11AF}/\x{EB37}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11AB}/\x{EB36}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}\x{11A8}/\x{EB35}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1173}\x{11AF}/\x{EBA5}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1173}\x{1175}/\x{EBA6}/g;
  s/\x{1109}\x{1107}\x{1100}\x{116E}\x{11AF}/\x{EBA3}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1161}\x{1175}/\x{EBA1}/g;
  s/\x{1109}\x{1105}\x{1161}\x{1175}\x{11AB}/\x{EB0F}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11AF}\x{11C2}/\x{EAFD}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11F0}/\x{EB0C}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11BC}/\x{EB0B}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11BA}/\x{EB0A}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11B8}/\x{EB09}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11B7}/\x{EB08}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11AF}/\x{EB07}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11AB}/\x{EB06}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}\x{11A8}/\x{EB05}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11BA}\x{11C2}/\x{EAF4}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11F0}\x{11BA}/\x{EAE5}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11AF}\x{11C2}/\x{EADF}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11AF}\x{11B8}/\x{EADE}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}\x{11EB}/\x{EAEB}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}\x{11BE}/\x{EAEC}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}\x{11BA}/\x{EAEA}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}\x{11AF}/\x{EAE9}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}\x{11AB}/\x{EAE8}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11AF}\x{11B8}/\x{EAC4}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11AF}\x{11A8}/\x{EAC3}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1175}\x{11BA}/\x{EAD1}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1175}\x{11B7}/\x{EAD0}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1175}\x{11AF}/\x{EACF}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1175}\x{11AB}/\x{EACE}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1167}\x{1175}/\x{EACC}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1165}\x{1175}/\x{EACA}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11F0}\x{11BA}/\x{EAB5}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11A8}\x{11BA}/\x{EAAD}/g;
  s/\x{1109}\x{1103}\x{1169}\x{1175}\x{11AF}/\x{EABA}/g;
  s/\x{1109}\x{1103}\x{1169}\x{1175}\x{11AB}/\x{EAB9}/g;
  s/\x{1109}\x{1103}\x{1169}\x{1161}\x{1175}/\x{EAB7}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11AF}\x{11B8}/\x{EA93}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11AF}\x{11A8}/\x{EA92}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11BC}/\x{EAA4}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11BA}/\x{EAA3}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11B7}/\x{EAA2}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11AF}/\x{EAA1}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11AB}/\x{EAA0}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}\x{11A8}/\x{EA9F}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1173}\x{11BF}/\x{EA9D}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1173}\x{11BA}/\x{EA9B}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11F0}\x{11C2}/\x{EA7D}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11BC}\x{11C2}/\x{EA7B}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11AF}\x{11C2}/\x{EA76}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11F0}/\x{EA87}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11BC}/\x{EA86}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11BA}/\x{EA85}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11B8}/\x{EA84}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11B7}/\x{EA83}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11AF}/\x{EA82}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11AB}/\x{EA81}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}\x{11A8}/\x{EA80}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11BC}/\x{EA69}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11BA}/\x{EA68}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11B8}/\x{EA67}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11B7}/\x{EA66}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11AF}/\x{EA64}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11AB}/\x{EA63}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}\x{11A8}/\x{EA62}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11AF}\x{11C2}/\x{EA51}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11AB}\x{11C2}/\x{EA4E}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BA}\x{11C2}/\x{EA3F}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BA}\x{11C0}/\x{EA3E}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BA}\x{11BE}/\x{EA3D}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}\x{11F9}/\x{EA3A}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}\x{11C2}/\x{EA39}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}\x{11C0}/\x{EA38}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}\x{11BA}/\x{EA37}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}\x{11AF}/\x{EA36}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AE}\x{11BE}/\x{EA34}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AB}\x{11C2}/\x{EA32}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AB}\x{11BE}/\x{EA31}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}\x{11BA}/\x{EA4A}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}\x{11B7}/\x{EA49}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}\x{11AF}/\x{EA47}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}\x{11AB}/\x{EA46}/g;
  s/\x{1109}\x{1100}\x{1172}\x{1165}\x{11BC}/\x{EA2C}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11BC}\x{11A8}/\x{EA12}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11B7}\x{11BA}/\x{EA0E}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11B7}\x{11B7}/\x{EA0D}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11B7}\x{11A8}/\x{EA0C}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11AF}\x{11F9}/\x{EA0A}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1175}\x{11F0}/\x{EA26}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1175}\x{11B7}/\x{EA25}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1175}\x{11AF}/\x{EA24}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1175}\x{11AB}/\x{EA23}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1167}\x{1175}/\x{EA21}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11F0}/\x{EA1C}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11BC}/\x{EA1B}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11B7}/\x{EA1A}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11AF}/\x{EA19}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11AB}/\x{EA18}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{11A8}/\x{EA17}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}\x{1175}/\x{EA1D}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11BA}\x{11BE}/\x{E9EB}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11AF}\x{11BA}/\x{E9E7}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11A8}\x{11BA}/\x{E9E3}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11F0}/\x{EA00}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11BA}/\x{E9FF}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11B7}/\x{E9FE}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11AF}/\x{E9FD}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11AB}/\x{E9FC}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}\x{11A8}/\x{E9FB}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{11F0}/\x{E9F5}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{11BC}/\x{E9F4}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{11AF}/\x{E9F3}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{11AB}/\x{E9F2}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{11A8}/\x{E9F1}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}\x{1175}/\x{E9F6}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11BA}\x{11A8}/\x{E9CA}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11B8}\x{11BC}/\x{E9C8}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11AF}\x{11B8}/\x{E9C4}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11A8}\x{11A8}/\x{E9C1}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11F0}/\x{E9D4}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11BA}/\x{E9D3}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11B8}/\x{E9D2}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11B7}/\x{E9D1}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11AF}/\x{E9D0}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11AB}/\x{E9CF}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}\x{11A8}/\x{E9CE}/g;
  s/\x{1109}\x{1100}\x{1163}\x{1175}\x{11BA}/\x{E9BE}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11BA}\x{11A8}/\x{E9B1}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11B8}\x{11BC}/\x{E9AF}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11B8}\x{11BA}/\x{E9AE}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11AF}\x{11C2}/\x{E9AB}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11A8}\x{11BA}/\x{E9A8}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11A8}\x{11A8}/\x{E9A7}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11F0}/\x{E9BA}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11BA}/\x{E9B9}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11B7}/\x{E9B8}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11AF}/\x{E9B7}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11AB}/\x{E9B6}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}\x{11A8}/\x{E9B5}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11B8}\x{11BC}/\x{E654}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11F0}\x{11BA}/\x{E674}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{E66E}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11AF}\x{11A8}/\x{E66D}/g;
  s/\x{1107}\x{116E}\x{11AF}\x{11B7}\x{11BA}/\x{E618}/g;
  s/\x{1107}\x{116E}\x{11AF}\x{11B7}\x{11A8}/\x{E617}/g;
  s/\x{1107}\x{116E}\x{1175}\x{11AF}\x{11F9}/\x{E626}/g;
  s/\x{1107}\x{1169}\x{11BA}\x{11BA}\x{11A8}/\x{E609}/g;
  s/\x{1107}\x{1169}\x{1175}\x{11BA}\x{11BE}/\x{E60F}/g;
  s/\x{1107}\x{1169}\x{1175}\x{11AF}\x{11F9}/\x{E60E}/g;
  s/\x{1107}\x{1161}\x{11AE}\x{11BA}\x{11A8}/\x{E5D5}/g;
  s/\x{1107}\x{1110}\x{1173}\x{1175}\x{11B7}/\x{E842}/g;
  s/\x{1107}\x{1110}\x{116E}\x{11A8}\x{11B8}/\x{E839}/g;
  s/\x{1107}\x{1110}\x{116E}\x{1175}\x{11B7}/\x{E83C}/g;
  s/\x{1107}\x{1110}\x{116E}\x{1175}\x{11AF}/\x{E83B}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11AF}\x{11B8}/\x{E82F}/g;
  s/\x{1107}\x{110F}\x{1169}\x{1161}\x{1175}/\x{E829}/g;
  s/\x{1107}\x{110C}\x{119E}\x{1175}\x{11BC}/\x{E822}/g;
  s/\x{1107}\x{110C}\x{119E}\x{1175}\x{11AF}/\x{E821}/g;
  s/\x{1107}\x{110C}\x{119E}\x{1175}\x{11AB}/\x{E820}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11BA}\x{11C2}/\x{E817}/g;
  s/\x{1107}\x{110C}\x{1173}\x{1175}\x{11F0}/\x{E80F}/g;
  s/\x{1107}\x{110C}\x{1173}\x{1175}\x{11BC}/\x{E80E}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1175}\x{11AF}/\x{E7FE}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1175}\x{11AB}/\x{E7FD}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1161}\x{11F0}/\x{E7FA}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1161}\x{11BC}/\x{E7F9}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1161}\x{1175}/\x{E7FB}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AF}\x{11F9}/\x{E8AE}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AF}\x{11BA}/\x{E8AD}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AF}\x{11B8}/\x{E8AC}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AF}\x{11A8}/\x{E8AB}/g;
  s/\x{1107}\x{110B}\x{119E}\x{1175}\x{11AF}/\x{E8B5}/g;
  s/\x{1107}\x{110B}\x{119E}\x{1175}\x{11AB}/\x{E8B4}/g;
  s/\x{1107}\x{110B}\x{119E}\x{1175}\x{11A8}/\x{E8B3}/g;
  s/\x{1107}\x{110B}\x{1175}\x{11AF}\x{11F9}/\x{E8A5}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11B7}\x{11BC}/\x{E89C}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11AF}\x{11F9}/\x{E89A}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11AF}\x{11BA}/\x{E899}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11AB}\x{11BA}/\x{E897}/g;
  s/\x{1107}\x{110B}\x{116E}\x{1175}\x{11AF}/\x{E891}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11B7}\x{11BA}/\x{E87C}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11AF}\x{11F9}/\x{E87A}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11AF}\x{11BA}/\x{E879}/g;
  s/\x{1107}\x{110B}\x{1169}\x{1161}\x{11AB}/\x{E881}/g;
  s/\x{1107}\x{110B}\x{1167}\x{11AF}\x{11F9}/\x{E872}/g;
  s/\x{1107}\x{110B}\x{1167}\x{1175}\x{11BA}/\x{E875}/g;
  s/\x{1107}\x{110B}\x{1165}\x{1175}\x{11BA}/\x{E86E}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11AF}\x{11F9}/\x{E85D}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11AF}\x{11A8}/\x{E85C}/g;
  s/\x{1107}\x{110B}\x{1161}\x{1175}\x{11BA}/\x{E867}/g;
  s/\x{1107}\x{110B}\x{1161}\x{1175}\x{11AB}/\x{E866}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11BA}\x{11C2}/\x{E78F}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11AF}\x{11F9}/\x{E78B}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11AF}\x{11BA}/\x{E78A}/g;
  s/\x{1107}\x{1109}\x{119E}\x{1175}\x{11BA}/\x{E793}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11AF}\x{11F9}/\x{E776}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11AF}\x{11C2}/\x{E775}/g;
  s/\x{1107}\x{1109}\x{1173}\x{1175}\x{11A8}/\x{E77B}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11AF}\x{11F9}/\x{E761}/g;
  s/\x{1107}\x{1109}\x{116E}\x{1167}\x{1175}/\x{E769}/g;
  s/\x{1107}\x{1109}\x{116E}\x{1165}\x{1175}/\x{E767}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}\x{11F9}/\x{E758}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}\x{11F0}/\x{E757}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}\x{11BC}/\x{E756}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}\x{1175}/\x{E759}/g;
  s/\x{1107}\x{1109}\x{1167}\x{1175}\x{11AB}/\x{E74B}/g;
  s/\x{1107}\x{1109}\x{1165}\x{1175}\x{11AF}/\x{E743}/g;
  s/\x{1107}\x{1109}\x{1165}\x{1175}\x{11AB}/\x{E742}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11BA}\x{11C2}/\x{E72F}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11BA}\x{11C0}/\x{E72E}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11AF}\x{11B8}/\x{E72A}/g;
  s/\x{1107}\x{1109}\x{1161}\x{1175}\x{11B8}/\x{E734}/g;
  s/\x{1107}\x{1109}\x{110C}\x{119E}\x{11AB}/\x{E7DD}/g;
  s/\x{1107}\x{1109}\x{1107}\x{1167}\x{1175}/\x{E7D8}/g;
  s/\x{1107}\x{1109}\x{1103}\x{119E}\x{11AF}/\x{E7D5}/g;
  s/\x{1107}\x{1109}\x{1103}\x{119E}\x{1175}/\x{E7D6}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1175}\x{11AF}/\x{E7D3}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1173}\x{11BA}/\x{E7D0}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1173}\x{11AB}/\x{E7CF}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1173}\x{1175}/\x{E7D1}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1172}\x{11B7}/\x{E7CD}/g;
  s/\x{1107}\x{1109}\x{1103}\x{116E}\x{11B7}/\x{E7CB}/g;
  s/\x{1107}\x{1109}\x{1103}\x{116E}\x{11AF}/\x{E7CA}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1169}\x{1175}/\x{E7C8}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1167}\x{11BA}/\x{E7C6}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1165}\x{11BA}/\x{E7C4}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1165}\x{11AF}/\x{E7C2}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1161}\x{11B8}/\x{E7BF}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1161}\x{1175}/\x{E7C0}/g;
  s/\x{1107}\x{1109}\x{1100}\x{119E}\x{11AB}/\x{E7BD}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1175}\x{11B7}/\x{E7BB}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1175}\x{11AF}/\x{E7BA}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{11BA}/\x{E7B3}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{11B7}/\x{E7B2}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{11AF}/\x{E7B1}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{11AB}/\x{E7B0}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}\x{1175}/\x{E7B4}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{11BA}/\x{E7AA}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{11B7}/\x{E7A9}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{11AF}/\x{E7A8}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{11AB}/\x{E7A7}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{1175}/\x{E7AD}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}\x{1165}/\x{E7AB}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1169}\x{1175}/\x{E7A4}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1167}\x{11BA}/\x{E7A0}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1167}\x{11A8}/\x{E79F}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1167}\x{1175}/\x{E7A1}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}\x{11BA}/\x{E79B}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}\x{11B8}/\x{E799}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}\x{1175}/\x{E79C}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1161}\x{1175}/\x{E795}/g;
  s/\x{1107}\x{1107}\x{119E}\x{11AF}\x{11F9}/\x{E720}/g;
  s/\x{1107}\x{1107}\x{119E}\x{11AF}\x{11BA}/\x{E71F}/g;
  s/\x{1107}\x{1107}\x{119E}\x{1175}\x{11F0}/\x{E724}/g;
  s/\x{1107}\x{1107}\x{119E}\x{1175}\x{11BC}/\x{E723}/g;
  s/\x{1107}\x{1107}\x{119E}\x{1175}\x{11A8}/\x{E722}/g;
  s/\x{1107}\x{1107}\x{1175}\x{11B7}\x{11BC}/\x{E71A}/g;
  s/\x{1107}\x{1107}\x{1175}\x{11AF}\x{11F9}/\x{E719}/g;
  s/\x{1107}\x{1107}\x{1173}\x{11B8}\x{11B8}/\x{E714}/g;
  s/\x{1107}\x{1107}\x{1173}\x{11B7}\x{11BC}/\x{E713}/g;
  s/\x{1107}\x{1107}\x{1173}\x{11AF}\x{11F9}/\x{E712}/g;
  s/\x{1107}\x{1107}\x{1173}\x{1175}\x{11F9}/\x{E718}/g;
  s/\x{1107}\x{1107}\x{1173}\x{1175}\x{11F0}/\x{E717}/g;
  s/\x{1107}\x{1107}\x{1172}\x{11B7}\x{11BC}/\x{E711}/g;
  s/\x{1107}\x{1107}\x{116E}\x{11B7}\x{11BC}/\x{E70B}/g;
  s/\x{1107}\x{1107}\x{116E}\x{11AF}\x{11F9}/\x{E70A}/g;
  s/\x{1107}\x{1107}\x{116E}\x{1167}\x{11F9}/\x{E710}/g;
  s/\x{1107}\x{1107}\x{116E}\x{1165}\x{11F9}/\x{E70E}/g;
  s/\x{1107}\x{1107}\x{116D}\x{11B7}\x{11BC}/\x{E709}/g;
  s/\x{1107}\x{1107}\x{1169}\x{11B7}\x{11BC}/\x{E707}/g;
  s/\x{1107}\x{1107}\x{1169}\x{11AF}\x{11F9}/\x{E706}/g;
  s/\x{1107}\x{1107}\x{1167}\x{11B7}\x{11BC}/\x{E703}/g;
  s/\x{1107}\x{1107}\x{1167}\x{11AF}\x{11F9}/\x{E702}/g;
  s/\x{1107}\x{1107}\x{1165}\x{11AF}\x{11F9}/\x{E700}/g;
  s/\x{1107}\x{1107}\x{1163}\x{11B7}\x{11BC}/\x{E6FE}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11B8}\x{11BC}/\x{E6FB}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11B7}\x{11BC}/\x{E6FA}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11AF}\x{11F9}/\x{E6F9}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11AF}\x{11AF}/\x{E6F8}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}\x{11BC}/\x{E8CA}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}\x{11B7}/\x{E8C8}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}\x{11AB}/\x{E8C7}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}\x{1175}/\x{E8CB}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1172}\x{11AB}/\x{E8C5}/g;
  s/\x{1107}\x{1107}\x{110B}\x{116E}\x{11F9}/\x{E8C3}/g;
  s/\x{1107}\x{1107}\x{110B}\x{116E}\x{11F0}/\x{E8C2}/g;
  s/\x{1107}\x{1107}\x{110B}\x{116E}\x{11BC}/\x{E8C1}/g;
  s/\x{1107}\x{1107}\x{110B}\x{116E}\x{11AB}/\x{E8C0}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1169}\x{11AB}/\x{E8BC}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1169}\x{1161}/\x{E8BD}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1161}\x{11F9}/\x{E8BA}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1161}\x{11F0}/\x{E8B9}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1161}\x{11BC}/\x{E8B8}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1161}\x{11AB}/\x{E8B7}/g;
  s/\x{1107}\x{1103}\x{119E}\x{1175}\x{11BA}/\x{E6F7}/g;
  s/\x{1107}\x{1103}\x{119E}\x{1175}\x{11AB}/\x{E6F6}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11AF}\x{11AF}/\x{E6EB}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AF}\x{11C2}/\x{E6DF}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AF}\x{11BA}/\x{E6DE}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AF}\x{11B8}/\x{E6DD}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AF}\x{11A8}/\x{E6DC}/g;
  s/\x{1107}\x{1103}\x{1173}\x{1175}\x{11BA}/\x{E6E6}/g;
  s/\x{1107}\x{1103}\x{1173}\x{1175}\x{11AF}/\x{E6E5}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11AF}\x{11B8}/\x{E6C9}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1175}\x{11BA}/\x{E6D4}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1175}\x{11AF}/\x{E6D3}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1167}\x{1175}/\x{E6D1}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1165}\x{1175}/\x{E6CF}/g;
  s/\x{1107}\x{1103}\x{1169}\x{1175}\x{11BA}/\x{E6C4}/g;
  s/\x{1107}\x{1103}\x{1169}\x{1161}\x{1175}/\x{E6C2}/g;
  s/\x{1107}\x{1103}\x{1167}\x{1175}\x{11BA}/\x{E6BB}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11AF}\x{11B8}/\x{E6B1}/g;
  s/\x{1107}\x{1103}\x{1165}\x{1175}\x{11BA}/\x{E6B7}/g;
  s/\x{1107}\x{1103}\x{1165}\x{1175}\x{11AB}/\x{E6B6}/g;
  s/\x{1107}\x{1100}\x{119E}\x{1175}\x{11BA}/\x{E6A1}/g;
  s/\x{1107}\x{1100}\x{119E}\x{1175}\x{11AB}/\x{E6A0}/g;
  s/\x{1107}\x{1100}\x{1173}\x{1175}\x{11BA}/\x{E698}/g;
  s/\x{1107}\x{1100}\x{116E}\x{1165}\x{1175}/\x{E690}/g;
  s/\x{1107}\x{1100}\x{1169}\x{1175}\x{11BA}/\x{E68A}/g;
  s/\x{1107}\x{1100}\x{1167}\x{1175}\x{11AF}/\x{E686}/g;
  s/\x{1107}\x{1100}\x{1165}\x{1175}\x{11AF}/\x{E681}/g;
  s/\x{1107}\x{1100}\x{1161}\x{1175}\x{11AB}/\x{E67D}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11B8}\x{11BC}/\x{E56A}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11B8}\x{11BC}/\x{E585}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{E582}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11AF}\x{11BA}/\x{E581}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11A8}\x{11A8}/\x{E57D}/g;
  s/\x{1106}\x{1173}\x{11A8}\x{11BA}\x{11A8}/\x{E53C}/g;
  s/\x{1106}\x{1173}\x{1175}\x{11B8}\x{11BC}/\x{E548}/g;
  s/\x{1106}\x{1173}\x{1175}\x{11AF}\x{11F9}/\x{E547}/g;
  s/\x{1106}\x{116E}\x{11BC}\x{11A8}\x{11A8}/\x{E532}/g;
  s/\x{1106}\x{116E}\x{11BA}\x{11BA}\x{11A8}/\x{E52E}/g;
  s/\x{1106}\x{1165}\x{1175}\x{11AF}\x{11F9}/\x{E510}/g;
  s/\x{1106}\x{1161}\x{11BA}\x{11BA}\x{11AE}/\x{E4FC}/g;
  s/\x{1106}\x{110B}\x{1173}\x{1161}\x{11AB}/\x{E5CB}/g;
  s/\x{1106}\x{110B}\x{1169}\x{1161}\x{11F9}/\x{E5C0}/g;
  s/\x{1106}\x{110B}\x{1169}\x{1161}\x{11AB}/\x{E5BF}/g;
  s/\x{1106}\x{1107}\x{116E}\x{1167}\x{1175}/\x{E5AC}/g;
  s/\x{1106}\x{1107}\x{116E}\x{1165}\x{1175}/\x{E5AA}/g;
  s/\x{1106}\x{1107}\x{1169}\x{1161}\x{1175}/\x{E5A3}/g;
  s/\x{1106}\x{1107}\x{1167}\x{1175}\x{11AE}/\x{E59D}/g;
  s/\x{1106}\x{1107}\x{1167}\x{1175}\x{11AB}/\x{E59C}/g;
  s/\x{1105}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{E44A}/g;
  s/\x{1105}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E42E}/g;
  s/\x{1105}\x{1161}\x{11BA}\x{11BA}\x{11A8}/\x{E409}/g;
  s/\x{1105}\x{1107}\x{110B}\x{119E}\x{1175}/\x{E4E1}/g;
  s/\x{1105}\x{1105}\x{1173}\x{11AF}\x{11AF}/\x{E4D6}/g;
  s/\x{1105}\x{1105}\x{1172}\x{1165}\x{1175}/\x{E4D3}/g;
  s/\x{1105}\x{1105}\x{116E}\x{1165}\x{1175}/\x{E4CF}/g;
  s/\x{1105}\x{1105}\x{116E}\x{1161}\x{11AB}/\x{E4CD}/g;
  s/\x{1105}\x{1105}\x{1167}\x{1175}\x{11AB}/\x{E4C5}/g;
  s/\x{1105}\x{1105}\x{1165}\x{1175}\x{11A8}/\x{E4C1}/g;
  s/\x{1105}\x{1105}\x{1161}\x{1175}\x{11AB}/\x{E4B8}/g;
  s/\x{1105}\x{1102}\x{119E}\x{1175}\x{11B7}/\x{E4B0}/g;
  s/\x{1105}\x{1102}\x{119E}\x{1175}\x{11A8}/\x{E4AF}/g;
  s/\x{1105}\x{1102}\x{1173}\x{1175}\x{11B8}/\x{E4AA}/g;
  s/\x{1105}\x{1102}\x{1173}\x{1175}\x{11AB}/\x{E4A9}/g;
  s/\x{1105}\x{1102}\x{1173}\x{1175}\x{11A8}/\x{E4A8}/g;
  s/\x{1105}\x{1102}\x{1165}\x{1175}\x{11B8}/\x{E49B}/g;
  s/\x{1105}\x{1102}\x{1165}\x{1175}\x{11AF}/\x{E49A}/g;
  s/\x{1103}\x{119E}\x{11B7}\x{11AB}\x{11AB}/\x{E39A}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11A8}\x{11BA}/\x{E391}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11AF}\x{11F9}/\x{E3AD}/g;
  s/\x{1103}\x{1173}\x{1175}\x{11AF}\x{11F9}/\x{E378}/g;
  s/\x{1103}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{E364}/g;
  s/\x{1103}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{E367}/g;
  s/\x{1103}\x{116E}\x{1175}\x{11B8}\x{11BC}/\x{E35B}/g;
  s/\x{1103}\x{1169}\x{11BA}\x{11BA}\x{11A8}/\x{E341}/g;
  s/\x{1103}\x{1169}\x{11AF}\x{11BA}\x{11BA}/\x{E33B}/g;
  s/\x{1103}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E346}/g;
  s/\x{1103}\x{1165}\x{1175}\x{11B8}\x{11AE}/\x{E332}/g;
  s/\x{1103}\x{1165}\x{1175}\x{11AB}\x{11AE}/\x{E331}/g;
  s/\x{1103}\x{1105}\x{1175}\x{11BC}\x{11BF}/\x{E3F2}/g;
  s/\x{1103}\x{1103}\x{119E}\x{1175}\x{11F0}/\x{E3EF}/g;
  s/\x{1103}\x{1103}\x{119E}\x{1175}\x{11BC}/\x{E3EE}/g;
  s/\x{1103}\x{1103}\x{119E}\x{1175}\x{11AF}/\x{E3ED}/g;
  s/\x{1103}\x{1103}\x{119E}\x{1175}\x{11A8}/\x{E3EC}/g;
  s/\x{1103}\x{1103}\x{1175}\x{11AF}\x{11F9}/\x{E3E1}/g;
  s/\x{1103}\x{1103}\x{1173}\x{11B7}\x{11BC}/\x{E3DD}/g;
  s/\x{1103}\x{1103}\x{1173}\x{11AE}\x{11AE}/\x{E3DC}/g;
  s/\x{1103}\x{1103}\x{1173}\x{1175}\x{11F9}/\x{E3E0}/g;
  s/\x{1103}\x{1103}\x{1172}\x{11B7}\x{11BC}/\x{E3D6}/g;
  s/\x{1103}\x{1103}\x{1172}\x{11AF}\x{11F9}/\x{E3D5}/g;
  s/\x{1103}\x{1103}\x{1172}\x{1175}\x{11BC}/\x{E3DB}/g;
  s/\x{1103}\x{1103}\x{1172}\x{1167}\x{11AB}/\x{E3D9}/g;
  s/\x{1103}\x{1103}\x{116E}\x{11B7}\x{11BC}/\x{E3CF}/g;
  s/\x{1103}\x{1103}\x{116E}\x{1175}\x{11F9}/\x{E3D4}/g;
  s/\x{1103}\x{1103}\x{116E}\x{1167}\x{1175}/\x{E3D3}/g;
  s/\x{1103}\x{1103}\x{116D}\x{11B7}\x{11BC}/\x{E3CE}/g;
  s/\x{1103}\x{1103}\x{1169}\x{11B7}\x{11BC}/\x{E3C8}/g;
  s/\x{1103}\x{1103}\x{1169}\x{11AF}\x{11F9}/\x{E3C7}/g;
  s/\x{1103}\x{1103}\x{1169}\x{1161}\x{11F0}/\x{E3CC}/g;
  s/\x{1103}\x{1103}\x{1167}\x{11AF}\x{11F9}/\x{E3C5}/g;
  s/\x{1103}\x{1103}\x{1165}\x{1173}\x{11AF}/\x{E3C3}/g;
  s/\x{1103}\x{1103}\x{1163}\x{11B7}\x{11BC}/\x{E3BF}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11B8}\x{11BC}/\x{E3BC}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11B7}\x{11BC}/\x{E3BB}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11AF}\x{11F9}/\x{E3BA}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11AF}\x{11BF}/\x{E3B9}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11AF}\x{11AF}/\x{E3B8}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11BA}\x{11AE}/\x{E2A1}/g;
  s/\x{1102}\x{1175}\x{11AF}\x{11B8}\x{11AE}/\x{E272}/g;
  s/\x{1102}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E249}/g;
  s/\x{1102}\x{1161}\x{11BD}\x{11B8}\x{11B8}/\x{E219}/g;
  s/\x{1102}\x{1161}\x{11A8}\x{11BA}\x{11A8}/\x{E206}/g;
  s/\x{1102}\x{1161}\x{1175}\x{11B8}\x{11AF}/\x{E221}/g;
  s/\x{1102}\x{1161}\x{1175}\x{11B8}\x{11AE}/\x{E220}/g;
  s/\x{1102}\x{1161}\x{1175}\x{11AF}\x{11F9}/\x{E21F}/g;
  s/\x{1102}\x{1161}\x{1175}\x{11AE}\x{11B8}/\x{E21E}/g;
  s/\x{1102}\x{1107}\x{116E}\x{1167}\x{1175}/\x{E2F8}/g;
  s/\x{1102}\x{1107}\x{116E}\x{1165}\x{1175}/\x{E2F6}/g;
  s/\x{1102}\x{1107}\x{1169}\x{1161}\x{1175}/\x{E2F1}/g;
  s/\x{1102}\x{1103}\x{116E}\x{1167}\x{1175}/\x{E2DE}/g;
  s/\x{1102}\x{1103}\x{116E}\x{1165}\x{1175}/\x{E2DC}/g;
  s/\x{1102}\x{1103}\x{1169}\x{1161}\x{1175}/\x{E2D7}/g;
  s/\x{1102}\x{1103}\x{1165}\x{1175}\x{11AB}/\x{E2D1}/g;
  s/\x{1102}\x{1100}\x{116E}\x{1167}\x{1175}/\x{E2B6}/g;
  s/\x{1102}\x{1100}\x{116E}\x{1165}\x{1175}/\x{E2B4}/g;
  s/\x{1102}\x{1100}\x{1169}\x{1161}\x{1175}/\x{E2AF}/g;
  s/\x{1102}\x{1100}\x{1167}\x{1175}\x{11AB}/\x{E2AC}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11B8}\x{11C1}/\x{E1B4}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11B8}\x{11BC}/\x{E1B5}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11B8}\x{11BA}/\x{E1B3}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11F0}\x{11BA}/\x{E1D6}/g;
  s/\x{1100}\x{1175}\x{11BA}\x{11BA}\x{11A8}/\x{E195}/g;
  s/\x{1100}\x{1175}\x{11AF}\x{11F9}\x{11C2}/\x{E18C}/g;
  s/\x{1100}\x{1173}\x{1175}\x{11AF}\x{11F9}/\x{E185}/g;
  s/\x{1100}\x{1172}\x{1175}\x{11AF}\x{11F9}/\x{E16D}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11AF}\x{11F9}/\x{E162}/g;
  s/\x{1100}\x{1172}\x{1167}\x{1175}\x{11BC}/\x{E169}/g;
  s/\x{1100}\x{1172}\x{1167}\x{1175}\x{11AF}/\x{E167}/g;
  s/\x{1100}\x{116E}\x{1165}\x{11AF}\x{11F9}/\x{E14C}/g;
  s/\x{1100}\x{116E}\x{1165}\x{11AB}\x{11BA}/\x{E14B}/g;
  s/\x{1100}\x{116E}\x{1165}\x{1173}\x{11AB}/\x{E14F}/g;
  s/\x{1100}\x{116D}\x{1163}\x{1175}\x{11AF}/\x{E134}/g;
  s/\x{1100}\x{116D}\x{1161}\x{1175}\x{11AF}/\x{E131}/g;
  s/\x{1100}\x{1169}\x{11AF}\x{11BA}\x{11BA}/\x{E114}/g;
  s/\x{1100}\x{1169}\x{11AF}\x{11B8}\x{11C1}/\x{E113}/g;
  s/\x{1100}\x{1169}\x{1161}\x{11AF}\x{11F9}/\x{E121}/g;
  s/\x{1100}\x{1169}\x{1161}\x{11AB}\x{11BA}/\x{E120}/g;
  s/\x{1100}\x{1161}\x{1175}\x{11AF}\x{11F0}/\x{E0EC}/g;
  s/\x{1100}\x{1161}\x{1175}\x{11AF}\x{11AB}/\x{E0EB}/g;
  s/\x{1100}\x{1100}\x{1175}\x{11B7}\x{11BC}/\x{E1FE}/g;
  s/\x{1100}\x{1100}\x{1173}\x{11C2}\x{11AF}/\x{E1FC}/g;
  s/\x{1100}\x{1100}\x{1173}\x{11AF}\x{11F9}/\x{E1FA}/g;
  s/\x{1100}\x{1100}\x{1173}\x{1175}\x{11F0}/\x{E1FD}/g;
  s/\x{1100}\x{1100}\x{1172}\x{11B7}\x{11BC}/\x{E1F1}/g;
  s/\x{1100}\x{1100}\x{1172}\x{11AF}\x{11F9}/\x{E1F0}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1175}\x{11BC}/\x{E1F9}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1167}\x{11F9}/\x{E1F7}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1167}\x{11F0}/\x{E1F6}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1167}\x{11BC}/\x{E1F5}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1167}\x{11AB}/\x{E1F4}/g;
  s/\x{1100}\x{1100}\x{116E}\x{11B7}\x{11BC}/\x{E1E9}/g;
  s/\x{1100}\x{1100}\x{116E}\x{11B7}\x{11A8}/\x{E1E8}/g;
  s/\x{1100}\x{1100}\x{116E}\x{11AF}\x{11F9}/\x{E1E7}/g;
  s/\x{1100}\x{1100}\x{116E}\x{1167}\x{1175}/\x{E1EF}/g;
  s/\x{1100}\x{1100}\x{116E}\x{1165}\x{11F0}/\x{E1ED}/g;
  s/\x{1100}\x{1100}\x{116D}\x{11B7}\x{11BC}/\x{E1E4}/g;
  s/\x{1100}\x{1100}\x{1169}\x{11B7}\x{11BC}/\x{E1E1}/g;
  s/\x{1100}\x{1100}\x{1169}\x{1161}\x{11F0}/\x{E1E3}/g;
  s/\x{1100}\x{1100}\x{1165}\x{11BC}\x{11A8}/\x{E1DD}/g;
  s/\x{1100}\x{1100}\x{1165}\x{11B8}\x{11BC}/\x{E1DC}/g;
  s/\x{1100}\x{1100}\x{1165}\x{11B7}\x{11BC}/\x{E1DB}/g;
  s/\x{1100}\x{1100}\x{1165}\x{11AF}\x{11F9}/\x{E1DA}/g;
  s/\x{1100}\x{1100}\x{1163}\x{11B7}\x{11BC}/\x{E1D9}/g;
  s/\x{115F}\x{1161}\x{11BD}\x{11BD}/\x{E0C5}/g;
  s/\x{115F}\x{1161}\x{11BA}\x{11BA}/\x{E0C3}/g;
  s/\x{115F}\x{1161}\x{11AF}\x{11B8}/\x{E0C1}/g;
  s/\x{115F}\x{1161}\x{11AF}\x{11B7}/\x{E0C0}/g;
  s/\x{115F}\x{1161}\x{11AF}\x{11A8}/\x{E0BF}/g;
  s/\x{115F}\x{1161}\x{11AE}\x{11AE}/\x{E0BD}/g;
  s/\x{115F}\x{1161}\x{1173}\x{11AE}/\x{E0C7}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11F0}/\x{F66E}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11BC}/\x{F66D}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11AF}/\x{F66A}/g;
  s/\x{1159}\x{119E}\x{1175}\x{11A8}/\x{F669}/g;
  s/\x{1159}\x{1175}\x{11B7}\x{11BC}/\x{F661}/g;
  s/\x{1159}\x{1175}\x{11AF}\x{11F9}/\x{F65F}/g;
  s/\x{1159}\x{1173}\x{11B7}\x{11BC}/\x{F655}/g;
  s/\x{1159}\x{1173}\x{1175}\x{11BC}/\x{F65B}/g;
  s/\x{1159}\x{1173}\x{1175}\x{11AB}/\x{F65A}/g;
  s/\x{1159}\x{1172}\x{11B7}\x{11BC}/\x{F644}/g;
  s/\x{1159}\x{1172}\x{1175}\x{11BC}/\x{F650}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11F0}/\x{F64C}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11BC}/\x{F64B}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11AF}/\x{F649}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11AB}/\x{F648}/g;
  s/\x{1159}\x{1172}\x{1167}\x{11A8}/\x{F647}/g;
  s/\x{1159}\x{1172}\x{1167}\x{1175}/\x{F64D}/g;
  s/\x{1159}\x{116E}\x{11B7}\x{11BC}/\x{F636}/g;
  s/\x{1159}\x{116E}\x{11AF}\x{11F9}/\x{F634}/g;
  s/\x{1159}\x{116E}\x{1175}\x{11BC}/\x{F640}/g;
  s/\x{1159}\x{116E}\x{1165}\x{11BC}/\x{F63E}/g;
  s/\x{1159}\x{116E}\x{1165}\x{11AF}/\x{F63C}/g;
  s/\x{1159}\x{116E}\x{1165}\x{11AB}/\x{F63B}/g;
  s/\x{1159}\x{116D}\x{11B7}\x{11BC}/\x{F62E}/g;
  s/\x{1159}\x{1169}\x{11B7}\x{11BC}/\x{F618}/g;
  s/\x{1159}\x{1169}\x{1175}\x{11F0}/\x{F62A}/g;
  s/\x{1159}\x{1169}\x{1175}\x{11BC}/\x{F629}/g;
  s/\x{1159}\x{1169}\x{1175}\x{11A8}/\x{F628}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11F9}/\x{F624}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11F0}/\x{F623}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11BC}/\x{F622}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11B8}/\x{F620}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11AF}/\x{F61E}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11AB}/\x{F61D}/g;
  s/\x{1159}\x{1169}\x{1161}\x{11A8}/\x{F61C}/g;
  s/\x{1159}\x{1169}\x{1161}\x{1175}/\x{F625}/g;
  s/\x{1159}\x{1167}\x{11AF}\x{11F9}/\x{F60D}/g;
  s/\x{1159}\x{1167}\x{1175}\x{11BC}/\x{F613}/g;
  s/\x{1159}\x{1165}\x{11AF}\x{11F9}/\x{F603}/g;
  s/\x{1159}\x{1165}\x{1175}\x{11BC}/\x{F608}/g;
  s/\x{1159}\x{1163}\x{11B8}\x{11BC}/\x{F5FC}/g;
  s/\x{1159}\x{1163}\x{11B7}\x{11BC}/\x{F5FA}/g;
  s/\x{1159}\x{1161}\x{11B8}\x{11BC}/\x{F5F0}/g;
  s/\x{1159}\x{1161}\x{11B7}\x{11BC}/\x{F5EE}/g;
  s/\x{1159}\x{1161}\x{11AF}\x{11F9}/\x{F5EC}/g;
  s/\x{1159}\x{1161}\x{1175}\x{11F0}/\x{F5F5}/g;
  s/\x{1159}\x{1161}\x{1175}\x{11BC}/\x{F5F4}/g;
  s/\x{1155}\x{1175}\x{11B7}\x{11BC}/\x{F397}/g;
  s/\x{1155}\x{1172}\x{1167}\x{11AB}/\x{F390}/g;
  s/\x{1155}\x{1169}\x{1161}\x{11AB}/\x{F386}/g;
  s/\x{1155}\x{1169}\x{1161}\x{1175}/\x{F387}/g;
  s/\x{1155}\x{1161}\x{11B7}\x{11BC}/\x{F37D}/g;
  s/\x{1154}\x{1172}\x{1167}\x{11AB}/\x{F373}/g;
  s/\x{1150}\x{1172}\x{1167}\x{11AB}/\x{F2BE}/g;
  s/\x{1150}\x{1169}\x{1161}\x{11BC}/\x{F2B5}/g;
  s/\x{1150}\x{1163}\x{11B7}\x{11BC}/\x{F2AF}/g;
  s/\x{1150}\x{1161}\x{11B7}\x{11BC}/\x{F2AA}/g;
  s/\x{1150}\x{1150}\x{1175}\x{11F0}/\x{F2DE}/g;
  s/\x{1150}\x{1150}\x{1175}\x{11EB}/\x{F2DD}/g;
  s/\x{1150}\x{1150}\x{1175}\x{11B7}/\x{F2DC}/g;
  s/\x{1150}\x{1150}\x{1175}\x{11AB}/\x{F2DB}/g;
  s/\x{1150}\x{1150}\x{1173}\x{11B7}/\x{F2D8}/g;
  s/\x{1150}\x{1150}\x{1172}\x{11F0}/\x{F2D4}/g;
  s/\x{1150}\x{1150}\x{1172}\x{1167}/\x{F2D5}/g;
  s/\x{1150}\x{1150}\x{116E}\x{1175}/\x{F2D2}/g;
  s/\x{1150}\x{1150}\x{1167}\x{11AB}/\x{F2D0}/g;
  s/\x{1150}\x{1150}\x{1163}\x{11F0}/\x{F2CE}/g;
  s/\x{1150}\x{1150}\x{1163}\x{11B8}/\x{F2CD}/g;
  s/\x{1150}\x{1150}\x{1163}\x{11B7}/\x{F2CB}/g;
  s/\x{1150}\x{1150}\x{1161}\x{11C2}/\x{F2C8}/g;
  s/\x{1150}\x{1150}\x{1161}\x{11AB}/\x{F2C7}/g;
  s/\x{1150}\x{1150}\x{1161}\x{1175}/\x{F2C9}/g;
  s/\x{114E}\x{1175}\x{11B7}\x{11BC}/\x{F289}/g;
  s/\x{114E}\x{1172}\x{1167}\x{11AB}/\x{F281}/g;
  s/\x{114E}\x{1161}\x{11B7}\x{11BC}/\x{F26B}/g;
  s/\x{114E}\x{114E}\x{119E}\x{1175}/\x{F2A5}/g;
  s/\x{114E}\x{114E}\x{1175}\x{11AB}/\x{F2A3}/g;
  s/\x{114E}\x{114E}\x{1173}\x{11F0}/\x{F2A1}/g;
  s/\x{114E}\x{114E}\x{1173}\x{11EB}/\x{F2A0}/g;
  s/\x{114E}\x{114E}\x{1172}\x{11F0}/\x{F29C}/g;
  s/\x{114E}\x{114E}\x{1172}\x{1167}/\x{F29D}/g;
  s/\x{114E}\x{114E}\x{116E}\x{1175}/\x{F29A}/g;
  s/\x{114E}\x{114E}\x{1167}\x{11AB}/\x{F297}/g;
  s/\x{114E}\x{114E}\x{1167}\x{1175}/\x{F298}/g;
  s/\x{114E}\x{114E}\x{1165}\x{11AB}/\x{F295}/g;
  s/\x{114E}\x{114E}\x{1163}\x{11F0}/\x{F293}/g;
  s/\x{114E}\x{114E}\x{1163}\x{11B7}/\x{F292}/g;
  s/\x{114E}\x{114E}\x{1161}\x{11F0}/\x{F28F}/g;
  s/\x{114E}\x{114E}\x{1161}\x{11B7}/\x{F28D}/g;
  s/\x{114E}\x{114E}\x{1161}\x{11AB}/\x{F28C}/g;
  s/\x{114E}\x{114E}\x{1161}\x{1175}/\x{F290}/g;
  s/\x{114C}\x{119E}\x{1175}\x{11A8}/\x{F163}/g;
  s/\x{114C}\x{1173}\x{11AF}\x{11F9}/\x{F14D}/g;
  s/\x{114C}\x{1173}\x{1175}\x{11BC}/\x{F153}/g;
  s/\x{114C}\x{1172}\x{11B7}\x{11BC}/\x{F142}/g;
  s/\x{114C}\x{1172}\x{11AF}\x{11F9}/\x{F140}/g;
  s/\x{114C}\x{1172}\x{1167}\x{11F9}/\x{F147}/g;
  s/\x{114C}\x{1172}\x{1167}\x{11F0}/\x{F146}/g;
  s/\x{114C}\x{1172}\x{1167}\x{11AB}/\x{F145}/g;
  s/\x{114C}\x{1172}\x{1167}\x{11A8}/\x{F144}/g;
  s/\x{114C}\x{116E}\x{11B7}\x{11BC}/\x{F12E}/g;
  s/\x{114C}\x{116E}\x{11AF}\x{11F9}/\x{F12C}/g;
  s/\x{114C}\x{116E}\x{1175}\x{11F0}/\x{F13B}/g;
  s/\x{114C}\x{116E}\x{1175}\x{11BC}/\x{F13A}/g;
  s/\x{114C}\x{116E}\x{1175}\x{11AF}/\x{F138}/g;
  s/\x{114C}\x{116E}\x{1165}\x{11F0}/\x{F135}/g;
  s/\x{114C}\x{116E}\x{1165}\x{11BC}/\x{F134}/g;
  s/\x{114C}\x{116E}\x{1165}\x{11AF}/\x{F132}/g;
  s/\x{114C}\x{116E}\x{1165}\x{11AB}/\x{F131}/g;
  s/\x{114C}\x{116E}\x{1165}\x{1175}/\x{F136}/g;
  s/\x{114C}\x{116D}\x{11B7}\x{11BC}/\x{F123}/g;
  s/\x{114C}\x{116D}\x{1163}\x{1175}/\x{F126}/g;
  s/\x{114C}\x{1169}\x{11B7}\x{11BC}/\x{F10E}/g;
  s/\x{114C}\x{1169}\x{11AF}\x{11F9}/\x{F10C}/g;
  s/\x{114C}\x{1169}\x{11AF}\x{11BA}/\x{F10B}/g;
  s/\x{114C}\x{1169}\x{1175}\x{11F0}/\x{F120}/g;
  s/\x{114C}\x{1169}\x{1175}\x{11BC}/\x{F11F}/g;
  s/\x{114C}\x{1169}\x{1175}\x{11AB}/\x{F11E}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11F0}/\x{F11A}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11BC}/\x{F119}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11B7}/\x{F117}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11AF}/\x{F115}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11AB}/\x{F114}/g;
  s/\x{114C}\x{1169}\x{1161}\x{11A8}/\x{F113}/g;
  s/\x{114C}\x{1169}\x{1161}\x{1175}/\x{F11B}/g;
  s/\x{114C}\x{1167}\x{11C1}\x{11BC}/\x{F103}/g;
  s/\x{114C}\x{1167}\x{11B7}\x{11BC}/\x{EFFD}/g;
  s/\x{114C}\x{1167}\x{1175}\x{11BC}/\x{F106}/g;
  s/\x{114C}\x{1167}\x{1175}\x{11BA}/\x{F105}/g;
  s/\x{114C}\x{1165}\x{11AF}\x{11F9}/\x{EFEB}/g;
  s/\x{114C}\x{1165}\x{1175}\x{11F0}/\x{EFF7}/g;
  s/\x{114C}\x{1165}\x{1175}\x{11EB}/\x{EFF5}/g;
  s/\x{114C}\x{1165}\x{1175}\x{11BC}/\x{EFF6}/g;
  s/\x{114C}\x{1165}\x{1175}\x{11BA}/\x{EFF4}/g;
  s/\x{114C}\x{1165}\x{1175}\x{11AB}/\x{EFF3}/g;
  s/\x{114C}\x{1163}\x{1175}\x{11BA}/\x{EFE6}/g;
  s/\x{114C}\x{1161}\x{11AF}\x{11F9}/\x{EFD6}/g;
  s/\x{114C}\x{1161}\x{1175}\x{11BC}/\x{EFDF}/g;
  s/\x{114C}\x{1161}\x{1175}\x{11BA}/\x{EFDE}/g;
  s/\x{1140}\x{119E}\x{11B8}\x{11BC}/\x{ED9B}/g;
  s/\x{1140}\x{119E}\x{11B7}\x{11BA}/\x{ED99}/g;
  s/\x{1140}\x{119E}\x{11AF}\x{11F9}/\x{ED97}/g;
  s/\x{1140}\x{119E}\x{11AF}\x{11C2}/\x{ED96}/g;
  s/\x{1140}\x{119E}\x{11AF}\x{11BA}/\x{ED95}/g;
  s/\x{1140}\x{119E}\x{11AF}\x{11A8}/\x{ED94}/g;
  s/\x{1140}\x{119E}\x{1175}\x{11BA}/\x{EDA4}/g;
  s/\x{1140}\x{119E}\x{1175}\x{11B8}/\x{EDA3}/g;
  s/\x{1140}\x{119E}\x{1175}\x{11AF}/\x{EDA2}/g;
  s/\x{1140}\x{119E}\x{1175}\x{11AB}/\x{EDA1}/g;
  s/\x{1140}\x{1175}\x{11B8}\x{11BC}/\x{ED88}/g;
  s/\x{1140}\x{1175}\x{11AF}\x{11F9}/\x{ED85}/g;
  s/\x{1140}\x{1175}\x{11AF}\x{11BA}/\x{ED84}/g;
  s/\x{1140}\x{1173}\x{11BA}\x{11BE}/\x{ED78}/g;
  s/\x{1140}\x{1173}\x{11B7}\x{11BE}/\x{ED75}/g;
  s/\x{1140}\x{1173}\x{11B7}\x{11BA}/\x{ED74}/g;
  s/\x{1140}\x{1173}\x{11AF}\x{11F9}/\x{ED72}/g;
  s/\x{1140}\x{1173}\x{11AF}\x{11C2}/\x{ED71}/g;
  s/\x{1140}\x{1173}\x{11AF}\x{11BA}/\x{ED70}/g;
  s/\x{1140}\x{1173}\x{11AF}\x{11B7}/\x{ED6F}/g;
  s/\x{1140}\x{1173}\x{1175}\x{11BA}/\x{ED7E}/g;
  s/\x{1140}\x{1173}\x{1175}\x{11AB}/\x{ED7D}/g;
  s/\x{1140}\x{1172}\x{11B7}\x{11BC}/\x{ED5A}/g;
  s/\x{1140}\x{1172}\x{1175}\x{11BC}/\x{ED6A}/g;
  s/\x{1140}\x{1172}\x{1167}\x{11AF}/\x{ED65}/g;
  s/\x{1140}\x{1172}\x{1167}\x{11AB}/\x{ED64}/g;
  s/\x{1140}\x{1172}\x{1167}\x{1175}/\x{ED67}/g;
  s/\x{1140}\x{1172}\x{1165}\x{11AB}/\x{ED62}/g;
  s/\x{1140}\x{116E}\x{11B7}\x{11BA}/\x{ED47}/g;
  s/\x{1140}\x{116E}\x{11AF}\x{11F9}/\x{ED45}/g;
  s/\x{1140}\x{116E}\x{1175}\x{11AB}/\x{ED55}/g;
  s/\x{1140}\x{116E}\x{1167}\x{1175}/\x{ED53}/g;
  s/\x{1140}\x{116E}\x{1165}\x{1175}/\x{ED51}/g;
  s/\x{1140}\x{116E}\x{1161}\x{11AB}/\x{ED4F}/g;
  s/\x{1140}\x{116D}\x{11B7}\x{11BC}/\x{ED3C}/g;
  s/\x{1140}\x{1169}\x{11AF}\x{11F9}/\x{ED30}/g;
  s/\x{1140}\x{1169}\x{1161}\x{1175}/\x{ED37}/g;
  s/\x{1140}\x{1167}\x{11AF}\x{11F9}/\x{ED21}/g;
  s/\x{1140}\x{1167}\x{11AF}\x{11BA}/\x{ED20}/g;
  s/\x{1140}\x{1167}\x{1175}\x{11BC}/\x{ED2C}/g;
  s/\x{1140}\x{1167}\x{1175}\x{11BA}/\x{ED2B}/g;
  s/\x{1140}\x{1167}\x{1175}\x{11B7}/\x{ED2A}/g;
  s/\x{1140}\x{1167}\x{1175}\x{11AE}/\x{ED29}/g;
  s/\x{1140}\x{1167}\x{1175}\x{11AB}/\x{ED28}/g;
  s/\x{1140}\x{1165}\x{11B7}\x{11BA}/\x{ED11}/g;
  s/\x{1140}\x{1165}\x{1175}\x{11BA}/\x{ED1B}/g;
  s/\x{1140}\x{1165}\x{1175}\x{11AB}/\x{ED1A}/g;
  s/\x{1140}\x{1163}\x{11B8}\x{11BC}/\x{ED07}/g;
  s/\x{1140}\x{1163}\x{11B7}\x{11BC}/\x{ED05}/g;
  s/\x{1140}\x{1161}\x{11B7}\x{11BA}/\x{ECF6}/g;
  s/\x{1140}\x{1161}\x{1175}\x{11F0}/\x{ED00}/g;
  s/\x{1140}\x{1161}\x{1175}\x{11EB}/\x{ECFE}/g;
  s/\x{1140}\x{1161}\x{1175}\x{11BC}/\x{ECFF}/g;
  s/\x{1140}\x{1161}\x{1175}\x{11BA}/\x{ECFD}/g;
  s/\x{1140}\x{1161}\x{1175}\x{11AB}/\x{ECFC}/g;
  s/\x{113E}\x{1175}\x{11B7}\x{11BC}/\x{ECE0}/g;
  s/\x{113E}\x{1172}\x{1167}\x{11AB}/\x{ECD9}/g;
  s/\x{113E}\x{1169}\x{1161}\x{1175}/\x{ECD1}/g;
  s/\x{113E}\x{1163}\x{11B7}\x{11BC}/\x{ECCC}/g;
  s/\x{113D}\x{1172}\x{1167}\x{11AB}/\x{ECBE}/g;
  s/\x{113C}\x{1172}\x{1167}\x{11AB}/\x{ECAD}/g;
  s/\x{113C}\x{1163}\x{11B7}\x{11BC}/\x{EC9F}/g;
  s/\x{1112}\x{119E}\x{11B8}\x{11BC}/\x{F547}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11F9}/\x{F544}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11C2}/\x{F543}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11BA}/\x{F542}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11B7}/\x{F541}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11AF}/\x{F540}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11AE}/\x{F53F}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11AB}/\x{F53E}/g;
  s/\x{1112}\x{119E}\x{11AF}\x{11A8}/\x{F53D}/g;
  s/\x{1112}\x{119E}\x{11A8}\x{11AF}/\x{F539}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11F0}/\x{F55C}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11BC}/\x{F55B}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11BA}/\x{F55A}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11B8}/\x{F558}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11B7}/\x{F557}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11AF}/\x{F554}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11AE}/\x{F553}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11AB}/\x{F552}/g;
  s/\x{1112}\x{119E}\x{1175}\x{11A8}/\x{F551}/g;
  s/\x{1112}\x{119E}\x{1165}\x{11AB}/\x{F54F}/g;
  s/\x{1112}\x{1175}\x{11F0}\x{11BA}/\x{F52C}/g;
  s/\x{1112}\x{1175}\x{11BC}\x{11BC}/\x{F52A}/g;
  s/\x{1112}\x{1175}\x{11B7}\x{11BC}/\x{F529}/g;
  s/\x{1112}\x{1175}\x{11B7}\x{11BA}/\x{F527}/g;
  s/\x{1112}\x{1175}\x{11B7}\x{11B8}/\x{F525}/g;
  s/\x{1112}\x{1175}\x{11AF}\x{11F9}/\x{F524}/g;
  s/\x{1112}\x{1175}\x{11AB}\x{11AB}/\x{F523}/g;
  s/\x{1112}\x{1175}\x{1172}\x{11AB}/\x{F536}/g;
  s/\x{1112}\x{1175}\x{1167}\x{1175}/\x{F532}/g;
  s/\x{1112}\x{1175}\x{1163}\x{11BC}/\x{F530}/g;
  s/\x{1112}\x{1175}\x{1163}\x{11AB}/\x{F52F}/g;
  s/\x{1112}\x{1173}\x{11B7}\x{11BC}/\x{F51A}/g;
  s/\x{1112}\x{1173}\x{11AF}\x{11F9}/\x{F519}/g;
  s/\x{1112}\x{1173}\x{11AF}\x{11AF}/\x{F518}/g;
  s/\x{1112}\x{1173}\x{11AE}\x{11C0}/\x{F517}/g;
  s/\x{1112}\x{1173}\x{11A8}\x{11AF}/\x{F516}/g;
  s/\x{1112}\x{1173}\x{1175}\x{11F9}/\x{F522}/g;
  s/\x{1112}\x{1173}\x{1175}\x{11F0}/\x{F521}/g;
  s/\x{1112}\x{1173}\x{1175}\x{11EB}/\x{F520}/g;
  s/\x{1112}\x{1172}\x{11B7}\x{11BC}/\x{F505}/g;
  s/\x{1112}\x{1172}\x{11AF}\x{11F9}/\x{F504}/g;
  s/\x{1112}\x{1172}\x{1175}\x{11BC}/\x{F515}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11F9}/\x{F510}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11F0}/\x{F50F}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11BC}/\x{F50E}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11AF}/\x{F50C}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11AB}/\x{F50B}/g;
  s/\x{1112}\x{1172}\x{1167}\x{11A8}/\x{F50A}/g;
  s/\x{1112}\x{1172}\x{1167}\x{1175}/\x{F511}/g;
  s/\x{1112}\x{1172}\x{1161}\x{11AB}/\x{F508}/g;
  s/\x{1112}\x{116E}\x{11B7}\x{11BC}/\x{F4F2}/g;
  s/\x{1112}\x{116E}\x{11B7}\x{11BA}/\x{F4F1}/g;
  s/\x{1112}\x{116E}\x{11AF}\x{11F9}/\x{F4F0}/g;
  s/\x{1112}\x{116E}\x{1175}\x{11F9}/\x{F503}/g;
  s/\x{1112}\x{116E}\x{1175}\x{11F0}/\x{F502}/g;
  s/\x{1112}\x{116E}\x{1167}\x{1175}/\x{F501}/g;
  s/\x{1112}\x{116E}\x{1165}\x{11F9}/\x{F4FC}/g;
  s/\x{1112}\x{116E}\x{1165}\x{11F0}/\x{F4FB}/g;
  s/\x{1112}\x{116E}\x{1165}\x{1173}/\x{F4FD}/g;
  s/\x{1112}\x{116E}\x{1161}\x{11BC}/\x{F4F8}/g;
  s/\x{1112}\x{116E}\x{1161}\x{11AB}/\x{F4F7}/g;
  s/\x{1112}\x{116E}\x{1161}\x{1175}/\x{F4F9}/g;
  s/\x{1112}\x{116D}\x{11B7}\x{11BC}/\x{F4E8}/g;
  s/\x{1112}\x{116D}\x{11AF}\x{11F9}/\x{F4E7}/g;
  s/\x{1112}\x{116D}\x{1175}\x{11AB}/\x{F4EF}/g;
  s/\x{1112}\x{116D}\x{1163}\x{11BC}/\x{F4EC}/g;
  s/\x{1112}\x{116D}\x{1161}\x{11AB}/\x{F4EA}/g;
  s/\x{1112}\x{1169}\x{11F0}\x{11BA}/\x{F4DA}/g;
  s/\x{1112}\x{1169}\x{11B7}\x{11BC}/\x{F4D7}/g;
  s/\x{1112}\x{1169}\x{11B7}\x{11BA}/\x{F4D6}/g;
  s/\x{1112}\x{1169}\x{11B7}\x{11A8}/\x{F4D5}/g;
  s/\x{1112}\x{1169}\x{11AF}\x{11F9}/\x{F4D4}/g;
  s/\x{1112}\x{1169}\x{11AF}\x{11AB}/\x{F4D3}/g;
  s/\x{1112}\x{1169}\x{11AB}\x{11BA}/\x{F4D2}/g;
  s/\x{1112}\x{1169}\x{1175}\x{11F9}/\x{F4E6}/g;
  s/\x{1112}\x{1169}\x{1175}\x{11F0}/\x{F4E5}/g;
  s/\x{1112}\x{1169}\x{1165}\x{11F9}/\x{F4E2}/g;
  s/\x{1112}\x{1169}\x{1165}\x{11AB}/\x{F4E1}/g;
  s/\x{1112}\x{1169}\x{1165}\x{1175}/\x{F4E3}/g;
  s/\x{1112}\x{1169}\x{1161}\x{11F0}/\x{F4DC}/g;
  s/\x{1112}\x{1167}\x{11B7}\x{11BC}/\x{F4CE}/g;
  s/\x{1112}\x{1167}\x{11AF}\x{11F9}/\x{F4CD}/g;
  s/\x{1112}\x{1165}\x{11F0}\x{11BA}/\x{F4C6}/g;
  s/\x{1112}\x{1165}\x{11AF}\x{11F9}/\x{F4C4}/g;
  s/\x{1112}\x{1165}\x{11AF}\x{11AF}/\x{F4C3}/g;
  s/\x{1112}\x{1165}\x{1173}\x{11B7}/\x{F4CC}/g;
  s/\x{1112}\x{1165}\x{1173}\x{11AB}/\x{F4CB}/g;
  s/\x{1112}\x{1163}\x{11B7}\x{11BC}/\x{F4BE}/g;
  s/\x{1112}\x{1163}\x{11AB}\x{11BA}/\x{F4BD}/g;
  s/\x{1112}\x{1163}\x{1175}\x{11F0}/\x{F4C2}/g;
  s/\x{1112}\x{1161}\x{11F0}\x{11BA}/\x{F4B7}/g;
  s/\x{1112}\x{1161}\x{11B7}\x{11BD}/\x{F4B4}/g;
  s/\x{1112}\x{1161}\x{11B7}\x{11BC}/\x{F4B5}/g;
  s/\x{1112}\x{1161}\x{11AF}\x{11F9}/\x{F4B3}/g;
  s/\x{1112}\x{1161}\x{11AF}\x{11AF}/\x{F4B1}/g;
  s/\x{1112}\x{1161}\x{11AF}\x{11AE}/\x{F4AF}/g;
  s/\x{1112}\x{1161}\x{1175}\x{11F0}/\x{F4BB}/g;
  s/\x{1112}\x{1161}\x{1175}\x{11EB}/\x{F4BA}/g;
  s/\x{1112}\x{1112}\x{119E}\x{11AB}/\x{F5E3}/g;
  s/\x{1112}\x{1112}\x{119E}\x{1175}/\x{F5E4}/g;
  s/\x{1112}\x{1112}\x{1175}\x{11F9}/\x{F5E1}/g;
  s/\x{1112}\x{1112}\x{1175}\x{11F0}/\x{F5E0}/g;
  s/\x{1112}\x{1112}\x{1175}\x{11B7}/\x{F5DE}/g;
  s/\x{1112}\x{1112}\x{1175}\x{11AB}/\x{F5DD}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11F0}/\x{F5D5}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11BC}/\x{F5D4}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11B7}/\x{F5D2}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11AB}/\x{F5D1}/g;
  s/\x{1112}\x{1112}\x{1173}\x{11A8}/\x{F5D0}/g;
  s/\x{1112}\x{1112}\x{1173}\x{1175}/\x{F5D7}/g;
  s/\x{1112}\x{1112}\x{1173}\x{116E}/\x{F5D6}/g;
  s/\x{1112}\x{1112}\x{1172}\x{11B7}/\x{F5C4}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1167}/\x{F5C7}/g;
  s/\x{1112}\x{1112}\x{1172}\x{1165}/\x{F5C5}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11F9}/\x{F5BD}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11F0}/\x{F5BC}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11BC}/\x{F5BB}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11B8}/\x{F5B9}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11B7}/\x{F5B7}/g;
  s/\x{1112}\x{1112}\x{116E}\x{11AB}/\x{F5B6}/g;
  s/\x{1112}\x{1112}\x{116E}\x{1175}/\x{F5C1}/g;
  s/\x{1112}\x{1112}\x{116E}\x{1165}/\x{F5BE}/g;
  s/\x{1112}\x{1112}\x{116D}\x{11B7}/\x{F5B3}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11F0}/\x{F5A3}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11BC}/\x{F5A2}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11B7}/\x{F5A0}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11AF}/\x{F59E}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11AB}/\x{F59D}/g;
  s/\x{1112}\x{1112}\x{1169}\x{11A8}/\x{F59C}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1175}/\x{F5AE}/g;
  s/\x{1112}\x{1112}\x{1169}\x{1161}/\x{F5A4}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11F9}/\x{F598}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11F0}/\x{F597}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11B8}/\x{F596}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11B7}/\x{F595}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11AF}/\x{F593}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11AB}/\x{F592}/g;
  s/\x{1112}\x{1112}\x{1167}\x{11A8}/\x{F591}/g;
  s/\x{1112}\x{1112}\x{1167}\x{1175}/\x{F599}/g;
  s/\x{1112}\x{1112}\x{1165}\x{11F9}/\x{F58D}/g;
  s/\x{1112}\x{1112}\x{1165}\x{11B7}/\x{F58B}/g;
  s/\x{1112}\x{1112}\x{1165}\x{11AF}/\x{F58A}/g;
  s/\x{1112}\x{1112}\x{1165}\x{11AB}/\x{F589}/g;
  s/\x{1112}\x{1112}\x{1165}\x{1175}/\x{F58E}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11F9}/\x{F586}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11F0}/\x{F585}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11B8}/\x{F583}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11B7}/\x{F581}/g;
  s/\x{1112}\x{1112}\x{1163}\x{11AB}/\x{F580}/g;
  s/\x{1112}\x{1112}\x{1163}\x{1175}/\x{F587}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11F0}/\x{F57B}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11BC}/\x{F57A}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11B8}/\x{F578}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11B7}/\x{F576}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11AF}/\x{F574}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11AB}/\x{F573}/g;
  s/\x{1112}\x{1112}\x{1161}\x{11A8}/\x{F572}/g;
  s/\x{1112}\x{1112}\x{1161}\x{1175}/\x{F57C}/g;
  s/\x{1112}\x{1109}\x{1175}\x{11BC}/\x{F570}/g;
  s/\x{1112}\x{1109}\x{1175}\x{11AB}/\x{F56F}/g;
  s/\x{1112}\x{1109}\x{1172}\x{11BC}/\x{F567}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1175}/\x{F56C}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1165}/\x{F56A}/g;
  s/\x{1112}\x{1109}\x{1172}\x{1161}/\x{F568}/g;
  s/\x{1112}\x{1109}\x{116E}\x{1175}/\x{F565}/g;
  s/\x{1112}\x{1109}\x{116E}\x{1165}/\x{F563}/g;
  s/\x{1112}\x{1109}\x{1169}\x{1161}/\x{F560}/g;
  s/\x{1112}\x{1109}\x{1163}\x{11BC}/\x{F55E}/g;
  s/\x{1111}\x{119E}\x{11BA}\x{11A8}/\x{F499}/g;
  s/\x{1111}\x{119E}\x{11AF}\x{11C2}/\x{F495}/g;
  s/\x{1111}\x{119E}\x{11AF}\x{11BA}/\x{F494}/g;
  s/\x{1111}\x{119E}\x{11AF}\x{11A8}/\x{F493}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11F0}/\x{F4A8}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11BC}/\x{F4A7}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11BA}/\x{F4A6}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11B8}/\x{F4A5}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11B7}/\x{F4A4}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11AF}/\x{F4A3}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11AB}/\x{F4A2}/g;
  s/\x{1111}\x{119E}\x{1175}\x{11A8}/\x{F4A1}/g;
  s/\x{1111}\x{119E}\x{1165}\x{11AB}/\x{F49F}/g;
  s/\x{1111}\x{1175}\x{11AF}\x{11F9}/\x{F485}/g;
  s/\x{1111}\x{1175}\x{1167}\x{1175}/\x{F48D}/g;
  s/\x{1111}\x{1175}\x{1163}\x{11AB}/\x{F48A}/g;
  s/\x{1111}\x{1175}\x{1163}\x{1169}/\x{F48B}/g;
  s/\x{1111}\x{1173}\x{11AF}\x{11F9}/\x{F480}/g;
  s/\x{1111}\x{1173}\x{1175}\x{11F9}/\x{F484}/g;
  s/\x{1111}\x{1173}\x{1175}\x{11F0}/\x{F483}/g;
  s/\x{1111}\x{1172}\x{11B7}\x{11BC}/\x{F47B}/g;
  s/\x{1111}\x{1172}\x{1161}\x{11AB}/\x{F47E}/g;
  s/\x{1111}\x{116E}\x{11F0}\x{11BA}/\x{F476}/g;
  s/\x{1111}\x{116E}\x{11B7}\x{11BC}/\x{F474}/g;
  s/\x{1111}\x{116E}\x{11AF}\x{11F9}/\x{F473}/g;
  s/\x{1111}\x{116E}\x{11AF}\x{11AF}/\x{F472}/g;
  s/\x{1111}\x{116E}\x{1175}\x{11F9}/\x{F47A}/g;
  s/\x{1111}\x{116E}\x{1167}\x{1175}/\x{F479}/g;
  s/\x{1111}\x{116E}\x{1165}\x{11F9}/\x{F477}/g;
  s/\x{1111}\x{116D}\x{11B8}\x{11BC}/\x{F470}/g;
  s/\x{1111}\x{116D}\x{11B7}\x{11BC}/\x{F46F}/g;
  s/\x{1111}\x{1169}\x{11B7}\x{11BC}/\x{F46C}/g;
  s/\x{1111}\x{1169}\x{11AF}\x{11F9}/\x{F46B}/g;
  s/\x{1111}\x{1167}\x{11B7}\x{11BC}/\x{F468}/g;
  s/\x{1111}\x{1167}\x{11AF}\x{11F9}/\x{F467}/g;
  s/\x{1111}\x{1167}\x{11AB}\x{11BA}/\x{F466}/g;
  s/\x{1111}\x{1165}\x{11AF}\x{11AF}/\x{F460}/g;
  s/\x{1111}\x{1165}\x{1173}\x{11AF}/\x{F464}/g;
  s/\x{1111}\x{1163}\x{11B7}\x{11BC}/\x{F45E}/g;
  s/\x{1111}\x{1161}\x{11BA}\x{11A8}/\x{F459}/g;
  s/\x{1111}\x{1161}\x{11B8}\x{11BC}/\x{F458}/g;
  s/\x{1111}\x{1161}\x{11B7}\x{11BC}/\x{F457}/g;
  s/\x{1111}\x{1161}\x{11AF}\x{11F9}/\x{F456}/g;
  s/\x{1111}\x{1161}\x{11AF}\x{11BF}/\x{F455}/g;
  s/\x{1111}\x{1161}\x{11AF}\x{11AF}/\x{F454}/g;
  s/\x{1111}\x{1161}\x{1175}\x{11F9}/\x{F45D}/g;
  s/\x{1111}\x{1112}\x{1175}\x{11AB}/\x{F4AC}/g;
  s/\x{1111}\x{110B}\x{116E}\x{11AB}/\x{F4AE}/g;
  s/\x{1111}\x{1107}\x{1161}\x{11AB}/\x{F4AA}/g;
  s/\x{1110}\x{119E}\x{11AF}\x{11BA}/\x{F442}/g;
  s/\x{1110}\x{119E}\x{11AF}\x{11A8}/\x{F441}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11F9}/\x{F450}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11F0}/\x{F44F}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11BC}/\x{F44E}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11BA}/\x{F44D}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11B7}/\x{F44C}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11AF}/\x{F44B}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11AB}/\x{F44A}/g;
  s/\x{1110}\x{119E}\x{1175}\x{11A8}/\x{F449}/g;
  s/\x{1110}\x{1175}\x{11B8}\x{11BC}/\x{F435}/g;
  s/\x{1110}\x{1175}\x{11AF}\x{11F9}/\x{F434}/g;
  s/\x{1110}\x{1175}\x{1167}\x{1175}/\x{F43B}/g;
  s/\x{1110}\x{1175}\x{1163}\x{11AB}/\x{F439}/g;
  s/\x{1110}\x{1173}\x{11B7}\x{11BC}/\x{F42F}/g;
  s/\x{1110}\x{1173}\x{11AF}\x{11F9}/\x{F42E}/g;
  s/\x{1110}\x{1173}\x{11AF}\x{11EB}/\x{F42D}/g;
  s/\x{1110}\x{1173}\x{11AF}\x{11AF}/\x{F42C}/g;
  s/\x{1110}\x{1173}\x{1175}\x{11F9}/\x{F433}/g;
  s/\x{1110}\x{1172}\x{11B7}\x{11BC}/\x{F429}/g;
  s/\x{1110}\x{1172}\x{11AF}\x{11F9}/\x{F428}/g;
  s/\x{1110}\x{116E}\x{11B7}\x{11BC}/\x{F41F}/g;
  s/\x{1110}\x{116E}\x{11AF}\x{11F9}/\x{F41E}/g;
  s/\x{1110}\x{116E}\x{11AF}\x{11AF}/\x{F41D}/g;
  s/\x{1110}\x{116E}\x{1167}\x{1175}/\x{F426}/g;
  s/\x{1110}\x{116E}\x{1165}\x{11F9}/\x{F424}/g;
  s/\x{1110}\x{116E}\x{1161}\x{11AB}/\x{F423}/g;
  s/\x{1110}\x{116D}\x{11B7}\x{11BC}/\x{F41A}/g;
  s/\x{1110}\x{116D}\x{11AF}\x{11F9}/\x{F419}/g;
  s/\x{1110}\x{1169}\x{11BA}\x{11A8}/\x{F413}/g;
  s/\x{1110}\x{1169}\x{11B8}\x{11B7}/\x{F412}/g;
  s/\x{1110}\x{1169}\x{11B7}\x{11BC}/\x{F411}/g;
  s/\x{1110}\x{1169}\x{11AF}\x{11F9}/\x{F410}/g;
  s/\x{1110}\x{1169}\x{11AF}\x{11AF}/\x{F40F}/g;
  s/\x{1110}\x{1169}\x{1169}\x{11AF}/\x{F418}/g;
  s/\x{1110}\x{1169}\x{1161}\x{11F0}/\x{F416}/g;
  s/\x{1110}\x{1167}\x{11AF}\x{11F9}/\x{F40B}/g;
  s/\x{1110}\x{1167}\x{1175}\x{11F0}/\x{F40E}/g;
  s/\x{1110}\x{1165}\x{11F0}\x{11BA}/\x{F407}/g;
  s/\x{1110}\x{1165}\x{11B8}\x{11BC}/\x{F405}/g;
  s/\x{1110}\x{1165}\x{11B7}\x{11BA}/\x{F404}/g;
  s/\x{1110}\x{1165}\x{116E}\x{11AF}/\x{F40A}/g;
  s/\x{1110}\x{1165}\x{116E}\x{11AE}/\x{F409}/g;
  s/\x{1110}\x{1163}\x{11B7}\x{11BC}/\x{F401}/g;
  s/\x{1110}\x{1161}\x{11B7}\x{11BC}/\x{F3FC}/g;
  s/\x{1110}\x{1161}\x{11AF}\x{11F9}/\x{F3FB}/g;
  s/\x{1110}\x{1110}\x{119E}\x{11AF}/\x{F453}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11F0}/\x{F3FA}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11BC}/\x{F3F9}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11BA}/\x{F3F8}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11B7}/\x{F3F7}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11AF}/\x{F3F6}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11AB}/\x{F3F5}/g;
  s/\x{110F}\x{119E}\x{1175}\x{11A8}/\x{F3F4}/g;
  s/\x{110F}\x{1175}\x{11AF}\x{11F9}/\x{F3DA}/g;
  s/\x{110F}\x{1175}\x{1172}\x{11AB}/\x{F3E9}/g;
  s/\x{110F}\x{1175}\x{116E}\x{11AF}/\x{F3E7}/g;
  s/\x{110F}\x{1175}\x{1167}\x{1175}/\x{F3E3}/g;
  s/\x{110F}\x{1175}\x{1163}\x{11BC}/\x{F3E0}/g;
  s/\x{110F}\x{1175}\x{1163}\x{11AB}/\x{F3DF}/g;
  s/\x{110F}\x{1175}\x{1163}\x{1169}/\x{F3E1}/g;
  s/\x{110F}\x{1173}\x{11B7}\x{11BC}/\x{F3D5}/g;
  s/\x{110F}\x{1173}\x{11AF}\x{11F9}/\x{F3D4}/g;
  s/\x{110F}\x{1173}\x{1175}\x{11F9}/\x{F3D9}/g;
  s/\x{110F}\x{1172}\x{1175}\x{11BC}/\x{F3D3}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11F9}/\x{F3CF}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11F0}/\x{F3CE}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11AF}/\x{F3CC}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11AB}/\x{F3CB}/g;
  s/\x{110F}\x{1172}\x{1167}\x{11A8}/\x{F3CA}/g;
  s/\x{110F}\x{1172}\x{1167}\x{1175}/\x{F3D0}/g;
  s/\x{110F}\x{1172}\x{1165}\x{1175}/\x{F3C8}/g;
  s/\x{110F}\x{116E}\x{11B7}\x{11BC}/\x{F3B9}/g;
  s/\x{110F}\x{116E}\x{11AF}\x{11F9}/\x{F3B8}/g;
  s/\x{110F}\x{116E}\x{1167}\x{11AB}/\x{F3C3}/g;
  s/\x{110F}\x{116E}\x{1167}\x{1175}/\x{F3C4}/g;
  s/\x{110F}\x{116E}\x{1165}\x{11F9}/\x{F3C1}/g;
  s/\x{110F}\x{116E}\x{1161}\x{11BC}/\x{F3BE}/g;
  s/\x{110F}\x{116E}\x{1161}\x{11AB}/\x{F3BD}/g;
  s/\x{110F}\x{116E}\x{1161}\x{1175}/\x{F3BF}/g;
  s/\x{110F}\x{116D}\x{11B7}\x{11BC}/\x{F3B6}/g;
  s/\x{110F}\x{116D}\x{11AF}\x{11F9}/\x{F3B5}/g;
  s/\x{110F}\x{1169}\x{11B7}\x{11BC}/\x{F3AF}/g;
  s/\x{110F}\x{1169}\x{11AF}\x{11F9}/\x{F3AE}/g;
  s/\x{110F}\x{1169}\x{1175}\x{11F0}/\x{F3B4}/g;
  s/\x{110F}\x{1169}\x{1161}\x{11F0}/\x{F3B2}/g;
  s/\x{110F}\x{1167}\x{11B7}\x{11BC}/\x{F3AA}/g;
  s/\x{110F}\x{1167}\x{11AF}\x{11F9}/\x{F3A9}/g;
  s/\x{110F}\x{1165}\x{11AF}\x{11F9}/\x{F3A5}/g;
  s/\x{110F}\x{1163}\x{11B7}\x{11BC}/\x{F3A1}/g;
  s/\x{110F}\x{1161}\x{11B8}\x{11BC}/\x{F39E}/g;
  s/\x{110F}\x{1161}\x{11B7}\x{11BC}/\x{F39D}/g;
  s/\x{110F}\x{1161}\x{11AF}\x{11F9}/\x{F39C}/g;
  s/\x{110F}\x{1161}\x{11AF}\x{11AF}/\x{F39B}/g;
  s/\x{110E}\x{119E}\x{11BA}\x{11BA}/\x{F34F}/g;
  s/\x{110E}\x{119E}\x{11B7}\x{11BA}/\x{F34C}/g;
  s/\x{110E}\x{119E}\x{11AF}\x{11F9}/\x{F34A}/g;
  s/\x{110E}\x{119E}\x{11AF}\x{11C2}/\x{F349}/g;
  s/\x{110E}\x{119E}\x{11AF}\x{11BA}/\x{F348}/g;
  s/\x{110E}\x{119E}\x{11AF}\x{11B7}/\x{F347}/g;
  s/\x{110E}\x{119E}\x{11AF}\x{11AE}/\x{F346}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11F0}/\x{F35A}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11BC}/\x{F359}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11BA}/\x{F358}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11B7}/\x{F357}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11AF}/\x{F356}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11AB}/\x{F355}/g;
  s/\x{110E}\x{119E}\x{1175}\x{11A8}/\x{F354}/g;
  s/\x{110E}\x{1175}\x{11B8}\x{11BC}/\x{F335}/g;
  s/\x{110E}\x{1175}\x{11B8}\x{11AE}/\x{F334}/g;
  s/\x{110E}\x{1175}\x{11B7}\x{11BC}/\x{F333}/g;
  s/\x{110E}\x{1175}\x{11AF}\x{11F9}/\x{F332}/g;
  s/\x{110E}\x{1175}\x{1167}\x{1175}/\x{F33E}/g;
  s/\x{110E}\x{1175}\x{1163}\x{11BC}/\x{F33C}/g;
  s/\x{110E}\x{1175}\x{1163}\x{11AB}/\x{F33B}/g;
  s/\x{110E}\x{1173}\x{11B7}\x{11BC}/\x{F32C}/g;
  s/\x{110E}\x{1173}\x{11AF}\x{11F9}/\x{F32A}/g;
  s/\x{110E}\x{1173}\x{11AF}\x{11F0}/\x{F329}/g;
  s/\x{110E}\x{1173}\x{11AF}\x{11BC}/\x{F32B}/g;
  s/\x{110E}\x{1173}\x{1175}\x{11F9}/\x{F331}/g;
  s/\x{110E}\x{1172}\x{11B7}\x{11BC}/\x{F316}/g;
  s/\x{110E}\x{1172}\x{11AF}\x{11F9}/\x{F315}/g;
  s/\x{110E}\x{1172}\x{1175}\x{11BC}/\x{F328}/g;
  s/\x{110E}\x{1172}\x{1175}\x{11AF}/\x{F327}/g;
  s/\x{110E}\x{1172}\x{1175}\x{11AB}/\x{F326}/g;
  s/\x{110E}\x{1172}\x{1167}\x{11B7}/\x{F322}/g;
  s/\x{110E}\x{1172}\x{1167}\x{11AF}/\x{F320}/g;
  s/\x{110E}\x{1172}\x{1167}\x{11AB}/\x{F31F}/g;
  s/\x{110E}\x{1172}\x{1167}\x{1175}/\x{F323}/g;
  s/\x{110E}\x{1172}\x{1165}\x{11B7}/\x{F31C}/g;
  s/\x{110E}\x{1172}\x{1165}\x{1175}/\x{F31D}/g;
  s/\x{110E}\x{1172}\x{1161}\x{11AB}/\x{F31A}/g;
  s/\x{110E}\x{116E}\x{11B7}\x{11BE}/\x{F308}/g;
  s/\x{110E}\x{116E}\x{11B7}\x{11BC}/\x{F309}/g;
  s/\x{110E}\x{116E}\x{11AF}\x{11F9}/\x{F307}/g;
  s/\x{110E}\x{116E}\x{1175}\x{11F9}/\x{F314}/g;
  s/\x{110E}\x{116E}\x{1167}\x{1175}/\x{F312}/g;
  s/\x{110E}\x{116E}\x{1165}\x{11F9}/\x{F310}/g;
  s/\x{110E}\x{116E}\x{1161}\x{11BC}/\x{F30E}/g;
  s/\x{110E}\x{116E}\x{1161}\x{11AB}/\x{F30D}/g;
  s/\x{110E}\x{116E}\x{1161}\x{1175}/\x{F30F}/g;
  s/\x{110E}\x{116D}\x{11B7}\x{11BC}/\x{F302}/g;
  s/\x{110E}\x{116D}\x{1175}\x{11AB}/\x{F306}/g;
  s/\x{110E}\x{1169}\x{11B8}\x{11BD}/\x{F2FC}/g;
  s/\x{110E}\x{1169}\x{11B7}\x{11BC}/\x{F2FB}/g;
  s/\x{110E}\x{1169}\x{11AF}\x{11F9}/\x{F2FA}/g;
  s/\x{110E}\x{1169}\x{1161}\x{11F0}/\x{F300}/g;
  s/\x{110E}\x{1167}\x{11F0}\x{11BA}/\x{F2F6}/g;
  s/\x{110E}\x{1167}\x{11B7}\x{11BC}/\x{F2F4}/g;
  s/\x{110E}\x{1167}\x{11AF}\x{11F9}/\x{F2F3}/g;
  s/\x{110E}\x{1167}\x{11AE}\x{11B8}/\x{F2F2}/g;
  s/\x{110E}\x{1167}\x{11AE}\x{11AE}/\x{F2F0}/g;
  s/\x{110E}\x{1165}\x{1173}\x{11AB}/\x{F2EF}/g;
  s/\x{110E}\x{1163}\x{11F0}\x{11BA}/\x{F2E8}/g;
  s/\x{110E}\x{1163}\x{11B8}\x{11BC}/\x{F2E6}/g;
  s/\x{110E}\x{1163}\x{11AF}\x{11F9}/\x{F2E5}/g;
  s/\x{110E}\x{1163}\x{1169}\x{11AF}/\x{F2EA}/g;
  s/\x{110E}\x{1161}\x{11B8}\x{11BC}/\x{F2E1}/g;
  s/\x{110E}\x{1161}\x{11B7}\x{11BC}/\x{F2E0}/g;
  s/\x{110E}\x{1161}\x{11AF}\x{11F9}/\x{F2DF}/g;
  s/\x{110E}\x{1112}\x{1175}\x{11AF}/\x{F35E}/g;
  s/\x{110E}\x{110F}\x{116E}\x{1165}/\x{F35C}/g;
  s/\x{110C}\x{119E}\x{11B8}\x{11BC}/\x{F208}/g;
  s/\x{110C}\x{119E}\x{11B7}\x{11BA}/\x{F206}/g;
  s/\x{110C}\x{119E}\x{11B7}\x{11A8}/\x{F205}/g;
  s/\x{110C}\x{119E}\x{11AF}\x{11F9}/\x{F203}/g;
  s/\x{110C}\x{119E}\x{11AF}\x{11AF}/\x{F202}/g;
  s/\x{110C}\x{119E}\x{11AF}\x{11AB}/\x{F201}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11F0}/\x{F21B}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11BC}/\x{F21A}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11BA}/\x{F219}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11B8}/\x{F218}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11B7}/\x{F217}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11AF}/\x{F216}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11AE}/\x{F215}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11AB}/\x{F214}/g;
  s/\x{110C}\x{119E}\x{1175}\x{11A8}/\x{F213}/g;
  s/\x{110C}\x{119E}\x{1161}\x{11BC}/\x{F211}/g;
  s/\x{110C}\x{1175}\x{11BA}\x{11C2}/\x{F1F0}/g;
  s/\x{110C}\x{1175}\x{11BA}\x{11C0}/\x{F1EF}/g;
  s/\x{110C}\x{1175}\x{11BA}\x{11BE}/\x{F1EE}/g;
  s/\x{110C}\x{1175}\x{11B8}\x{11BC}/\x{F1ED}/g;
  s/\x{110C}\x{1175}\x{11B7}\x{11BC}/\x{F1EC}/g;
  s/\x{110C}\x{1175}\x{11B7}\x{11BA}/\x{F1EB}/g;
  s/\x{110C}\x{1175}\x{11AF}\x{11F9}/\x{F1EA}/g;
  s/\x{110C}\x{1175}\x{11AF}\x{11AB}/\x{F1E9}/g;
  s/\x{110C}\x{1175}\x{11AE}\x{11BE}/\x{F1E8}/g;
  s/\x{110C}\x{1175}\x{11A8}\x{11C2}/\x{F1E7}/g;
  s/\x{110C}\x{1175}\x{1169}\x{11AF}/\x{F1FA}/g;
  s/\x{110C}\x{1175}\x{1167}\x{1175}/\x{F1F8}/g;
  s/\x{110C}\x{1175}\x{1163}\x{11AB}/\x{F1F6}/g;
  s/\x{110C}\x{1173}\x{11B8}\x{11BC}/\x{F1E0}/g;
  s/\x{110C}\x{1173}\x{11B7}\x{11BC}/\x{F1DF}/g;
  s/\x{110C}\x{1173}\x{11AF}\x{11F9}/\x{F1DE}/g;
  s/\x{110C}\x{1173}\x{11AF}\x{11AF}/\x{F1DD}/g;
  s/\x{110C}\x{1173}\x{1175}\x{11F9}/\x{F1E6}/g;
  s/\x{110C}\x{1173}\x{1175}\x{11F0}/\x{F1E5}/g;
  s/\x{110C}\x{1172}\x{11F0}\x{11A8}/\x{F1C5}/g;
  s/\x{110C}\x{1172}\x{11B7}\x{11BC}/\x{F1C3}/g;
  s/\x{110C}\x{1172}\x{11AF}\x{11F9}/\x{F1C2}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11BC}/\x{F1DC}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11BA}/\x{F1DB}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11AF}/\x{F1D9}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11AB}/\x{F1D8}/g;
  s/\x{110C}\x{1172}\x{1175}\x{11A8}/\x{F1D7}/g;
  s/\x{110C}\x{1172}\x{1167}\x{11F9}/\x{F1D1}/g;
  s/\x{110C}\x{1172}\x{1167}\x{11AF}/\x{F1CF}/g;
  s/\x{110C}\x{1172}\x{1167}\x{11AB}/\x{F1CE}/g;
  s/\x{110C}\x{1172}\x{1167}\x{1175}/\x{F1D2}/g;
  s/\x{110C}\x{1172}\x{1165}\x{11AF}/\x{F1CB}/g;
  s/\x{110C}\x{1172}\x{1165}\x{11AB}/\x{F1CA}/g;
  s/\x{110C}\x{1172}\x{1165}\x{1175}/\x{F1CC}/g;
  s/\x{110C}\x{1172}\x{1161}\x{11AB}/\x{F1C8}/g;
  s/\x{110C}\x{116E}\x{11B7}\x{11BC}/\x{F1B6}/g;
  s/\x{110C}\x{116E}\x{11B7}\x{11BA}/\x{F1B5}/g;
  s/\x{110C}\x{116E}\x{11AF}\x{11F9}/\x{F1B4}/g;
  s/\x{110C}\x{116E}\x{11AF}\x{11AB}/\x{F1B3}/g;
  s/\x{110C}\x{116E}\x{11AB}\x{11BA}/\x{F1B2}/g;
  s/\x{110C}\x{116E}\x{1167}\x{11AF}/\x{F1BF}/g;
  s/\x{110C}\x{116E}\x{1167}\x{11AB}/\x{F1BE}/g;
  s/\x{110C}\x{116E}\x{1167}\x{1175}/\x{F1C1}/g;
  s/\x{110C}\x{116E}\x{1161}\x{11BC}/\x{F1BB}/g;
  s/\x{110C}\x{116E}\x{1161}\x{1175}/\x{F1BC}/g;
  s/\x{110C}\x{116D}\x{11F0}\x{11BA}/\x{F1AD}/g;
  s/\x{110C}\x{116D}\x{11BC}\x{11BA}/\x{F1AB}/g;
  s/\x{110C}\x{116D}\x{11B8}\x{11BC}/\x{F1AA}/g;
  s/\x{110C}\x{116D}\x{11B7}\x{11BC}/\x{F1A9}/g;
  s/\x{110C}\x{116D}\x{11AF}\x{11F9}/\x{F1A8}/g;
  s/\x{110C}\x{116D}\x{1175}\x{11AF}/\x{F1B0}/g;
  s/\x{110C}\x{1169}\x{11B7}\x{11BC}/\x{F1A2}/g;
  s/\x{110C}\x{1169}\x{11B7}\x{11BA}/\x{F1A1}/g;
  s/\x{110C}\x{1169}\x{11AF}\x{11F9}/\x{F1A0}/g;
  s/\x{110C}\x{1169}\x{11AB}\x{11BA}/\x{F19F}/g;
  s/\x{110C}\x{1169}\x{1161}\x{11F0}/\x{F1A7}/g;
  s/\x{110C}\x{1167}\x{11B7}\x{11BC}/\x{F197}/g;
  s/\x{110C}\x{1167}\x{11AF}\x{11F9}/\x{F196}/g;
  s/\x{110C}\x{1167}\x{11AF}\x{11AF}/\x{F195}/g;
  s/\x{110C}\x{1167}\x{11AE}\x{11BD}/\x{F194}/g;
  s/\x{110C}\x{1167}\x{11AE}\x{11AE}/\x{F192}/g;
  s/\x{110C}\x{1167}\x{11AE}\x{11A8}/\x{F191}/g;
  s/\x{110C}\x{1167}\x{11AB}\x{11BA}/\x{F190}/g;
  s/\x{110C}\x{1167}\x{1163}\x{11BC}/\x{F19C}/g;
  s/\x{110C}\x{1165}\x{11F0}\x{11C2}/\x{F188}/g;
  s/\x{110C}\x{1165}\x{11BA}\x{11C2}/\x{F185}/g;
  s/\x{110C}\x{1165}\x{11B8}\x{11C1}/\x{F184}/g;
  s/\x{110C}\x{1165}\x{11B7}\x{11BA}/\x{F183}/g;
  s/\x{110C}\x{1165}\x{11AF}\x{11F9}/\x{F182}/g;
  s/\x{110C}\x{1165}\x{11AF}\x{11AF}/\x{F181}/g;
  s/\x{110C}\x{1165}\x{11AF}\x{11AB}/\x{F180}/g;
  s/\x{110C}\x{1165}\x{11A8}\x{11C2}/\x{F17F}/g;
  s/\x{110C}\x{1165}\x{1173}\x{11F9}/\x{F18F}/g;
  s/\x{110C}\x{1165}\x{1173}\x{11BC}/\x{F18E}/g;
  s/\x{110C}\x{1165}\x{1173}\x{11AB}/\x{F18D}/g;
  s/\x{110C}\x{1165}\x{1173}\x{11A8}/\x{F18C}/g;
  s/\x{110C}\x{1163}\x{11F0}\x{11BA}/\x{F17B}/g;
  s/\x{110C}\x{1163}\x{11BC}\x{11BA}/\x{F179}/g;
  s/\x{110C}\x{1163}\x{11B7}\x{11BC}/\x{F177}/g;
  s/\x{110C}\x{1163}\x{11AF}\x{11AF}/\x{F176}/g;
  s/\x{110C}\x{1163}\x{11AF}\x{11AB}/\x{F175}/g;
  s/\x{110C}\x{1163}\x{11AB}\x{11BA}/\x{F174}/g;
  s/\x{110C}\x{1163}\x{116E}\x{11AF}/\x{F17E}/g;
  s/\x{110C}\x{1161}\x{11F0}\x{11BA}/\x{F170}/g;
  s/\x{110C}\x{1161}\x{11F0}\x{11A8}/\x{F16F}/g;
  s/\x{110C}\x{1161}\x{11B8}\x{11BC}/\x{F16C}/g;
  s/\x{110C}\x{1161}\x{11B8}\x{11B8}/\x{F16B}/g;
  s/\x{110C}\x{1161}\x{11B7}\x{11BC}/\x{F16A}/g;
  s/\x{110C}\x{1161}\x{11B7}\x{11BA}/\x{F169}/g;
  s/\x{110C}\x{1161}\x{11B7}\x{11A8}/\x{F168}/g;
  s/\x{110C}\x{1161}\x{11AF}\x{11F9}/\x{F167}/g;
  s/\x{110C}\x{1161}\x{11AF}\x{11AF}/\x{F166}/g;
  s/\x{110C}\x{1161}\x{11AF}\x{11AB}/\x{F165}/g;
  s/\x{110C}\x{1161}\x{11AB}\x{11BA}/\x{F164}/g;
  s/\x{110C}\x{1161}\x{1175}\x{11F0}/\x{F173}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11F0}/\x{F259}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11BC}/\x{F258}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11BA}/\x{F257}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11B8}/\x{F255}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11B7}/\x{F254}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11AF}/\x{F253}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11AB}/\x{F252}/g;
  s/\x{110C}\x{110C}\x{119E}\x{11A8}/\x{F251}/g;
  s/\x{110C}\x{110C}\x{119E}\x{1175}/\x{F25A}/g;
  s/\x{110C}\x{110C}\x{1175}\x{11F9}/\x{F24F}/g;
  s/\x{110C}\x{110C}\x{1175}\x{11F0}/\x{F24E}/g;
  s/\x{110C}\x{110C}\x{1175}\x{11EB}/\x{F24D}/g;
  s/\x{110C}\x{110C}\x{1173}\x{11F0}/\x{F247}/g;
  s/\x{110C}\x{110C}\x{1173}\x{11EB}/\x{F246}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1175}/\x{F243}/g;
  s/\x{110C}\x{110C}\x{1172}\x{1167}/\x{F23E}/g;
  s/\x{110C}\x{110C}\x{116E}\x{11F9}/\x{F23A}/g;
  s/\x{110C}\x{110C}\x{116E}\x{11F0}/\x{F239}/g;
  s/\x{110C}\x{110C}\x{116D}\x{11F0}/\x{F236}/g;
  s/\x{110C}\x{110C}\x{1169}\x{11F9}/\x{F231}/g;
  s/\x{110C}\x{110C}\x{1169}\x{11F0}/\x{F22F}/g;
  s/\x{110C}\x{110C}\x{1169}\x{1165}/\x{F233}/g;
  s/\x{110C}\x{110C}\x{1167}\x{11F0}/\x{F22C}/g;
  s/\x{110C}\x{110C}\x{1165}\x{11F9}/\x{F22A}/g;
  s/\x{110C}\x{110C}\x{1165}\x{11F0}/\x{F229}/g;
  s/\x{110C}\x{110C}\x{1163}\x{11F0}/\x{F228}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11F9}/\x{F225}/g;
  s/\x{110C}\x{110C}\x{1161}\x{11F0}/\x{F224}/g;
  s/\x{110C}\x{110C}\x{1112}\x{1165}/\x{F263}/g;
  s/\x{110C}\x{110B}\x{1175}\x{11AF}/\x{F21F}/g;
  s/\x{110C}\x{110B}\x{1173}\x{11AF}/\x{F21D}/g;
  s/\x{110B}\x{119E}\x{11B8}\x{11BA}/\x{EE96}/g;
  s/\x{110B}\x{119E}\x{11B7}\x{11BA}/\x{EE94}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11F9}/\x{EE92}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11C2}/\x{EE91}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11BA}/\x{EE90}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11B8}/\x{EE8F}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11B7}/\x{EE8E}/g;
  s/\x{110B}\x{119E}\x{11AF}\x{11A8}/\x{EE8D}/g;
  s/\x{110B}\x{119E}\x{119E}\x{11AF}/\x{EEAF}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11F0}/\x{EEAD}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11EB}/\x{EEAB}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11BC}/\x{EEAC}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11BA}/\x{EEAA}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11B8}/\x{EEA9}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11B7}/\x{EEA8}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11AF}/\x{EEA7}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11AB}/\x{EEA6}/g;
  s/\x{110B}\x{119E}\x{1175}\x{11A8}/\x{EEA5}/g;
  s/\x{110B}\x{119E}\x{1165}\x{11BC}/\x{EEA3}/g;
  s/\x{110B}\x{119E}\x{1165}\x{11B7}/\x{EEA2}/g;
  s/\x{110B}\x{119E}\x{1165}\x{11AF}/\x{EEA0}/g;
  s/\x{110B}\x{119E}\x{1165}\x{11AB}/\x{EE9F}/g;
  s/\x{110B}\x{1175}\x{11BA}\x{11B8}/\x{EE7F}/g;
  s/\x{110B}\x{1175}\x{11BA}\x{11AF}/\x{EE7E}/g;
  s/\x{110B}\x{1175}\x{11BA}\x{11A8}/\x{EE7D}/g;
  s/\x{110B}\x{1175}\x{11B8}\x{11C2}/\x{EE7B}/g;
  s/\x{110B}\x{1175}\x{11B8}\x{11BC}/\x{EE7C}/g;
  s/\x{110B}\x{1175}\x{11B7}\x{11BC}/\x{EE7A}/g;
  s/\x{110B}\x{1175}\x{11AF}\x{11F9}/\x{EE79}/g;
  s/\x{110B}\x{1175}\x{11AF}\x{11AF}/\x{EE78}/g;
  s/\x{110B}\x{1175}\x{11AF}\x{11AB}/\x{EE77}/g;
  s/\x{110B}\x{1175}\x{11AB}\x{11BA}/\x{EE76}/g;
  s/\x{110B}\x{1175}\x{1173}\x{11AB}/\x{EE87}/g;
  s/\x{110B}\x{1175}\x{1169}\x{116E}/\x{EE84}/g;
  s/\x{110B}\x{1173}\x{11BD}\x{11BD}/\x{EE6C}/g;
  s/\x{110B}\x{1173}\x{11BA}\x{11BE}/\x{EE69}/g;
  s/\x{110B}\x{1173}\x{11B8}\x{11C1}/\x{EE68}/g;
  s/\x{110B}\x{1173}\x{11B8}\x{11B8}/\x{EE67}/g;
  s/\x{110B}\x{1173}\x{11B8}\x{11AF}/\x{EE66}/g;
  s/\x{110B}\x{1173}\x{11B7}\x{11BC}/\x{EE65}/g;
  s/\x{110B}\x{1173}\x{11B7}\x{11BA}/\x{EE64}/g;
  s/\x{110B}\x{1173}\x{11AF}\x{11F9}/\x{EE63}/g;
  s/\x{110B}\x{1173}\x{11AF}\x{11AF}/\x{EE60}/g;
  s/\x{110B}\x{1173}\x{11AE}\x{11AE}/\x{EE5F}/g;
  s/\x{110B}\x{1173}\x{11AB}\x{11BA}/\x{EE5E}/g;
  s/\x{110B}\x{1173}\x{11AB}\x{11AB}/\x{EE5D}/g;
  s/\x{110B}\x{1173}\x{1175}\x{11F9}/\x{EE75}/g;
  s/\x{110B}\x{1173}\x{1173}\x{11F9}/\x{EE73}/g;
  s/\x{110B}\x{1173}\x{1173}\x{11F0}/\x{EE72}/g;
  s/\x{110B}\x{1173}\x{1173}\x{11BC}/\x{EE71}/g;
  s/\x{110B}\x{1173}\x{1173}\x{11AB}/\x{EE70}/g;
  s/\x{110B}\x{1172}\x{11B7}\x{11BC}/\x{EE45}/g;
  s/\x{110B}\x{1172}\x{11AF}\x{11F9}/\x{EE44}/g;
  s/\x{110B}\x{1172}\x{1175}\x{11F0}/\x{EE5C}/g;
  s/\x{110B}\x{1172}\x{1175}\x{11BC}/\x{EE5B}/g;
  s/\x{110B}\x{1172}\x{1175}\x{11AB}/\x{EE5A}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11F9}/\x{EE54}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11F0}/\x{EE53}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11AF}/\x{EE51}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11AB}/\x{EE50}/g;
  s/\x{110B}\x{1172}\x{1167}\x{11A8}/\x{EE4F}/g;
  s/\x{110B}\x{1172}\x{1167}\x{1175}/\x{EE55}/g;
  s/\x{110B}\x{1172}\x{1165}\x{1175}/\x{EE4C}/g;
  s/\x{110B}\x{1172}\x{1161}\x{11B7}/\x{EE4A}/g;
  s/\x{110B}\x{1172}\x{1161}\x{11AB}/\x{EE49}/g;
  s/\x{110B}\x{116E}\x{11EB}\x{11B8}/\x{EE30}/g;
  s/\x{110B}\x{116E}\x{11BA}\x{11BE}/\x{EE2E}/g;
  s/\x{110B}\x{116E}\x{11BA}\x{11B8}/\x{EE2C}/g;
  s/\x{110B}\x{116E}\x{11B8}\x{11C1}/\x{EE2B}/g;
  s/\x{110B}\x{116E}\x{11B7}\x{11C2}/\x{EE29}/g;
  s/\x{110B}\x{116E}\x{11B7}\x{11BE}/\x{EE28}/g;
  s/\x{110B}\x{116E}\x{11B7}\x{11BC}/\x{EE2A}/g;
  s/\x{110B}\x{116E}\x{11B7}\x{11BA}/\x{EE27}/g;
  s/\x{110B}\x{116E}\x{11AF}\x{11F9}/\x{EE26}/g;
  s/\x{110B}\x{116E}\x{11AF}\x{11AF}/\x{EE25}/g;
  s/\x{110B}\x{116E}\x{11AF}\x{11AB}/\x{EE24}/g;
  s/\x{110B}\x{116E}\x{11AE}\x{11BE}/\x{EE23}/g;
  s/\x{110B}\x{116E}\x{11AB}\x{11C0}/\x{EE22}/g;
  s/\x{110B}\x{116E}\x{1175}\x{11F0}/\x{EE43}/g;
  s/\x{110B}\x{116E}\x{1167}\x{1175}/\x{EE41}/g;
  s/\x{110B}\x{116E}\x{1165}\x{11F0}/\x{EE3C}/g;
  s/\x{110B}\x{116E}\x{1165}\x{11EB}/\x{EE3B}/g;
  s/\x{110B}\x{116E}\x{1165}\x{1173}/\x{EE3E}/g;
  s/\x{110B}\x{116E}\x{1161}\x{11F0}/\x{EE37}/g;
  s/\x{110B}\x{116E}\x{1161}\x{11BC}/\x{EE36}/g;
  s/\x{110B}\x{116E}\x{1161}\x{11AB}/\x{EE35}/g;
  s/\x{110B}\x{116E}\x{1161}\x{1175}/\x{EE38}/g;
  s/\x{110B}\x{116D}\x{11B7}\x{11BC}/\x{EE18}/g;
  s/\x{110B}\x{116D}\x{11AF}\x{11F9}/\x{EE17}/g;
  s/\x{110B}\x{116D}\x{1175}\x{11AF}/\x{EE21}/g;
  s/\x{110B}\x{116D}\x{1165}\x{11AB}/\x{EE1E}/g;
  s/\x{110B}\x{116D}\x{1161}\x{1175}/\x{EE1B}/g;
  s/\x{110B}\x{1169}\x{11F0}\x{11BA}/\x{EE08}/g;
  s/\x{110B}\x{1169}\x{11BA}\x{11BE}/\x{EE05}/g;
  s/\x{110B}\x{1169}\x{11B7}\x{11BC}/\x{EE04}/g;
  s/\x{110B}\x{1169}\x{11B7}\x{11BA}/\x{EE03}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11F9}/\x{EE02}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11AF}/\x{EE00}/g;
  s/\x{110B}\x{1169}\x{11AF}\x{11AB}/\x{EDFF}/g;
  s/\x{110B}\x{1169}\x{11AB}\x{11BA}/\x{EDFC}/g;
  s/\x{110B}\x{1169}\x{1175}\x{11F0}/\x{EE16}/g;
  s/\x{110B}\x{1169}\x{1169}\x{11AF}/\x{EE13}/g;
  s/\x{110B}\x{1169}\x{1165}\x{11AF}/\x{EE11}/g;
  s/\x{110B}\x{1169}\x{1165}\x{11AB}/\x{EE10}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11F9}/\x{EE0E}/g;
  s/\x{110B}\x{1169}\x{1161}\x{11F0}/\x{EE0D}/g;
  s/\x{110B}\x{1167}\x{11BC}\x{11A8}/\x{EDF6}/g;
  s/\x{110B}\x{1167}\x{11BA}\x{11C2}/\x{EDF4}/g;
  s/\x{110B}\x{1167}\x{11BA}\x{11C0}/\x{EDF3}/g;
  s/\x{110B}\x{1167}\x{11BA}\x{11B8}/\x{EDF2}/g;
  s/\x{110B}\x{1167}\x{11BA}\x{11A8}/\x{EDF1}/g;
  s/\x{110B}\x{1167}\x{11B8}\x{11BC}/\x{EDF0}/g;
  s/\x{110B}\x{1167}\x{11B8}\x{11B8}/\x{EDEF}/g;
  s/\x{110B}\x{1167}\x{11B8}\x{11AF}/\x{EDEE}/g;
  s/\x{110B}\x{1167}\x{11B7}\x{11BC}/\x{EDED}/g;
  s/\x{110B}\x{1167}\x{11AF}\x{11F9}/\x{EDEC}/g;
  s/\x{110B}\x{1167}\x{11AF}\x{11AF}/\x{EDEA}/g;
  s/\x{110B}\x{1167}\x{11AE}\x{11C0}/\x{EDE9}/g;
  s/\x{110B}\x{1167}\x{11AB}\x{11C0}/\x{EDE8}/g;
  s/\x{110B}\x{1167}\x{11AB}\x{11BA}/\x{EDE7}/g;
  s/\x{110B}\x{1167}\x{11AB}\x{11AE}/\x{EDE6}/g;
  s/\x{110B}\x{1167}\x{11A8}\x{11BF}/\x{EDE5}/g;
  s/\x{110B}\x{1165}\x{11C1}\x{11BA}/\x{EDD7}/g;
  s/\x{110B}\x{1165}\x{11BA}\x{11BD}/\x{EDD4}/g;
  s/\x{110B}\x{1165}\x{11B8}\x{11C1}/\x{EDD2}/g;
  s/\x{110B}\x{1165}\x{11B8}\x{11BC}/\x{EDD3}/g;
  s/\x{110B}\x{1165}\x{11B7}\x{11BA}/\x{EDD0}/g;
  s/\x{110B}\x{1165}\x{11AF}\x{11F9}/\x{EDCF}/g;
  s/\x{110B}\x{1165}\x{11AF}\x{11AF}/\x{EDCE}/g;
  s/\x{110B}\x{1165}\x{11AF}\x{11AB}/\x{EDCD}/g;
  s/\x{110B}\x{1165}\x{11AB}\x{11BA}/\x{EDCC}/g;
  s/\x{110B}\x{1165}\x{11AB}\x{11AE}/\x{EDCB}/g;
  s/\x{110B}\x{1165}\x{11A8}\x{11AF}/\x{EDCA}/g;
  s/\x{110B}\x{1165}\x{1175}\x{11F0}/\x{EDE3}/g;
  s/\x{110B}\x{1165}\x{1175}\x{11EB}/\x{EDE2}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11BC}/\x{EDE0}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11BA}/\x{EDDF}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11B8}/\x{EDDD}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11B7}/\x{EDDC}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11AF}/\x{EDDB}/g;
  s/\x{110B}\x{1165}\x{1173}\x{11AB}/\x{EDDA}/g;
  s/\x{110B}\x{1163}\x{11C2}\x{11AB}/\x{EDC6}/g;
  s/\x{110B}\x{1163}\x{11BC}\x{11BA}/\x{EDC4}/g;
  s/\x{110B}\x{1163}\x{11BA}\x{11B8}/\x{EDC3}/g;
  s/\x{110B}\x{1163}\x{11B8}\x{11BC}/\x{EDC2}/g;
  s/\x{110B}\x{1163}\x{11B7}\x{11BC}/\x{EDC1}/g;
  s/\x{110B}\x{1163}\x{11B7}\x{11BA}/\x{EDC0}/g;
  s/\x{110B}\x{1163}\x{11B7}\x{11B8}/\x{EDBF}/g;
  s/\x{110B}\x{1163}\x{11AF}\x{11F9}/\x{EDBE}/g;
  s/\x{110B}\x{1161}\x{11C2}\x{11AB}/\x{EDB9}/g;
  s/\x{110B}\x{1161}\x{11BA}\x{11B8}/\x{EDB6}/g;
  s/\x{110B}\x{1161}\x{11B8}\x{11C2}/\x{EDB4}/g;
  s/\x{110B}\x{1161}\x{11B8}\x{11C1}/\x{EDB3}/g;
  s/\x{110B}\x{1161}\x{11B8}\x{11BC}/\x{EDB5}/g;
  s/\x{110B}\x{1161}\x{11B8}\x{11AF}/\x{EDB1}/g;
  s/\x{110B}\x{1161}\x{11B7}\x{11C2}/\x{EDAF}/g;
  s/\x{110B}\x{1161}\x{11B7}\x{11BC}/\x{EDB0}/g;
  s/\x{110B}\x{1161}\x{11B7}\x{11BA}/\x{EDAE}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11F9}/\x{EDAD}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11AF}/\x{EDA8}/g;
  s/\x{110B}\x{1161}\x{11AF}\x{11AB}/\x{EDA7}/g;
  s/\x{110B}\x{1161}\x{11AB}\x{11C0}/\x{EDA6}/g;
  s/\x{110B}\x{1161}\x{11AB}\x{11BA}/\x{EDA5}/g;
  s/\x{110B}\x{1161}\x{1175}\x{11F0}/\x{EDBD}/g;
  s/\x{110B}\x{1161}\x{1175}\x{11EB}/\x{EDBC}/g;
  s/\x{110B}\x{1161}\x{1169}\x{11AF}/\x{EDBB}/g;
  s/\x{110B}\x{1140}\x{119E}\x{1175}/\x{EF55}/g;
  s/\x{110B}\x{1140}\x{1175}\x{1161}/\x{EF53}/g;
  s/\x{110B}\x{1140}\x{1173}\x{1175}/\x{EF51}/g;
  s/\x{110B}\x{1140}\x{1172}\x{1175}/\x{EF4F}/g;
  s/\x{110B}\x{1140}\x{116E}\x{1175}/\x{EF4D}/g;
  s/\x{110B}\x{1140}\x{116E}\x{1167}/\x{EF4B}/g;
  s/\x{110B}\x{1140}\x{116E}\x{1165}/\x{EF49}/g;
  s/\x{110B}\x{1140}\x{1169}\x{1175}/\x{EF46}/g;
  s/\x{110B}\x{1140}\x{1169}\x{1161}/\x{EF44}/g;
  s/\x{110B}\x{1140}\x{1167}\x{1175}/\x{EF42}/g;
  s/\x{110B}\x{1140}\x{1165}\x{1175}/\x{EF40}/g;
  s/\x{110B}\x{1140}\x{1161}\x{1175}/\x{EF3D}/g;
  s/\x{110B}\x{1111}\x{119E}\x{1175}/\x{EFCF}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11C1}/\x{EFCD}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11C0}/\x{EFCC}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11BF}/\x{EFCB}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11BC}/\x{EFCA}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11AF}/\x{EFC9}/g;
  s/\x{110B}\x{1111}\x{1175}\x{11AB}/\x{EFC8}/g;
  s/\x{110B}\x{1111}\x{1173}\x{11AF}/\x{EFC6}/g;
  s/\x{110B}\x{1111}\x{1172}\x{11B7}/\x{EFC4}/g;
  s/\x{110B}\x{1111}\x{116E}\x{11C0}/\x{EFB8}/g;
  s/\x{110B}\x{1111}\x{116E}\x{11BF}/\x{EFB7}/g;
  s/\x{110B}\x{1111}\x{116E}\x{11AF}/\x{EFB6}/g;
  s/\x{110B}\x{1111}\x{116E}\x{11AB}/\x{EFB5}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1175}/\x{EFBF}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1165}/\x{EFBB}/g;
  s/\x{110B}\x{1111}\x{116E}\x{1161}/\x{EFB9}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11BF}/\x{EFB0}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11B7}/\x{EFAF}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11AF}/\x{EFAC}/g;
  s/\x{110B}\x{1111}\x{1169}\x{11AB}/\x{EFAB}/g;
  s/\x{110B}\x{1111}\x{1169}\x{1161}/\x{EFB1}/g;
  s/\x{110B}\x{1111}\x{1165}\x{11BC}/\x{EFA4}/g;
  s/\x{110B}\x{1111}\x{1165}\x{11B7}/\x{EFA3}/g;
  s/\x{110B}\x{1111}\x{1165}\x{11AF}/\x{EFA2}/g;
  s/\x{110B}\x{1111}\x{1165}\x{11AB}/\x{EFA1}/g;
  s/\x{110B}\x{1111}\x{1165}\x{1175}/\x{EFA5}/g;
  s/\x{110B}\x{1111}\x{1161}\x{11BF}/\x{EF9C}/g;
  s/\x{110B}\x{1111}\x{1161}\x{11AF}/\x{EF9A}/g;
  s/\x{110B}\x{1111}\x{1161}\x{11AB}/\x{EF99}/g;
  s/\x{110B}\x{1111}\x{1161}\x{11A8}/\x{EF98}/g;
  s/\x{110B}\x{1111}\x{1161}\x{1175}/\x{EF9D}/g;
  s/\x{110B}\x{110E}\x{1172}\x{1175}/\x{EF94}/g;
  s/\x{110B}\x{110E}\x{1172}\x{1161}/\x{EF92}/g;
  s/\x{110B}\x{110E}\x{116E}\x{11BC}/\x{EF8C}/g;
  s/\x{110B}\x{110E}\x{116E}\x{11AB}/\x{EF8B}/g;
  s/\x{110B}\x{110E}\x{116E}\x{1165}/\x{EF8F}/g;
  s/\x{110B}\x{110E}\x{116E}\x{1161}/\x{EF8D}/g;
  s/\x{110B}\x{110E}\x{1165}\x{11BC}/\x{EF89}/g;
  s/\x{110B}\x{110E}\x{1165}\x{11AB}/\x{EF88}/g;
  s/\x{110B}\x{110E}\x{1161}\x{11BC}/\x{EF85}/g;
  s/\x{110B}\x{110E}\x{1161}\x{11AB}/\x{EF84}/g;
  s/\x{110B}\x{110E}\x{1161}\x{1175}/\x{EF86}/g;
  s/\x{110B}\x{110C}\x{1172}\x{1175}/\x{EF80}/g;
  s/\x{110B}\x{110C}\x{116E}\x{11BC}/\x{EF79}/g;
  s/\x{110B}\x{110C}\x{116E}\x{11AB}/\x{EF78}/g;
  s/\x{110B}\x{110C}\x{116E}\x{1175}/\x{EF7E}/g;
  s/\x{110B}\x{110C}\x{116E}\x{1165}/\x{EF7C}/g;
  s/\x{110B}\x{110C}\x{116E}\x{1161}/\x{EF7A}/g;
  s/\x{110B}\x{110C}\x{1165}\x{11BC}/\x{EF73}/g;
  s/\x{110B}\x{110C}\x{1165}\x{11B7}/\x{EF72}/g;
  s/\x{110B}\x{110C}\x{1165}\x{11AB}/\x{EF71}/g;
  s/\x{110B}\x{110C}\x{1165}\x{116E}/\x{EF74}/g;
  s/\x{110B}\x{110C}\x{1161}\x{11BC}/\x{EF6D}/g;
  s/\x{110B}\x{110C}\x{1161}\x{11AB}/\x{EF6C}/g;
  s/\x{110B}\x{110C}\x{1161}\x{1175}/\x{EF6F}/g;
  s/\x{110B}\x{110C}\x{1161}\x{1169}/\x{EF6E}/g;
  s/\x{110B}\x{110B}\x{1175}\x{11AB}/\x{EF6A}/g;
  s/\x{110B}\x{110B}\x{1172}\x{11B7}/\x{EF66}/g;
  s/\x{110B}\x{110B}\x{1172}\x{11AB}/\x{EF65}/g;
  s/\x{110B}\x{110B}\x{1172}\x{1161}/\x{EF67}/g;
  s/\x{110B}\x{110B}\x{116E}\x{1165}/\x{EF63}/g;
  s/\x{110B}\x{110B}\x{116D}\x{11BA}/\x{EF61}/g;
  s/\x{110B}\x{110B}\x{116D}\x{11B7}/\x{EF5F}/g;
  s/\x{110B}\x{110B}\x{116D}\x{11AB}/\x{EF5E}/g;
  s/\x{110B}\x{110B}\x{1167}\x{11BA}/\x{EF5A}/g;
  s/\x{110B}\x{110B}\x{1167}\x{11AB}/\x{EF59}/g;
  s/\x{110B}\x{110B}\x{1167}\x{1175}/\x{EF5B}/g;
  s/\x{110B}\x{110B}\x{1163}\x{11AB}/\x{EF57}/g;
  s/\x{110B}\x{1109}\x{119E}\x{11AF}/\x{EF3B}/g;
  s/\x{110B}\x{1109}\x{1175}\x{11AF}/\x{EF39}/g;
  s/\x{110B}\x{1109}\x{1173}\x{11AB}/\x{EF35}/g;
  s/\x{110B}\x{1109}\x{1173}\x{1175}/\x{EF36}/g;
  s/\x{110B}\x{1109}\x{1172}\x{1175}/\x{EF33}/g;
  s/\x{110B}\x{1109}\x{1172}\x{1167}/\x{EF31}/g;
  s/\x{110B}\x{1109}\x{1172}\x{1161}/\x{EF2F}/g;
  s/\x{110B}\x{1109}\x{116E}\x{11AB}/\x{EF27}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1165}/\x{EF2C}/g;
  s/\x{110B}\x{1109}\x{116E}\x{1161}/\x{EF28}/g;
  s/\x{110B}\x{1109}\x{1169}\x{11AB}/\x{EF25}/g;
  s/\x{110B}\x{1109}\x{1167}\x{1175}/\x{EF22}/g;
  s/\x{110B}\x{1109}\x{1165}\x{11BC}/\x{EF1E}/g;
  s/\x{110B}\x{1109}\x{1165}\x{11B7}/\x{EF1D}/g;
  s/\x{110B}\x{1109}\x{1165}\x{11AF}/\x{EF1C}/g;
  s/\x{110B}\x{1109}\x{1165}\x{11AB}/\x{EF1B}/g;
  s/\x{110B}\x{1109}\x{1165}\x{1175}/\x{EF20}/g;
  s/\x{110B}\x{1109}\x{1165}\x{116E}/\x{EF1F}/g;
  s/\x{110B}\x{1109}\x{1161}\x{11BC}/\x{EF17}/g;
  s/\x{110B}\x{1109}\x{1161}\x{11AB}/\x{EF16}/g;
  s/\x{110B}\x{1109}\x{1161}\x{1175}/\x{EF19}/g;
  s/\x{110B}\x{1109}\x{1161}\x{1169}/\x{EF18}/g;
  s/\x{110B}\x{1107}\x{119E}\x{1175}/\x{EF14}/g;
  s/\x{110B}\x{1107}\x{1175}\x{11BF}/\x{EF12}/g;
  s/\x{110B}\x{1107}\x{1175}\x{11BC}/\x{EF11}/g;
  s/\x{110B}\x{1107}\x{1175}\x{11AF}/\x{EF10}/g;
  s/\x{110B}\x{1107}\x{1175}\x{11AB}/\x{EF0F}/g;
  s/\x{110B}\x{1107}\x{1175}\x{11A8}/\x{EF0E}/g;
  s/\x{110B}\x{1107}\x{1173}\x{11AF}/\x{EF0B}/g;
  s/\x{110B}\x{1107}\x{1173}\x{11AB}/\x{EF0A}/g;
  s/\x{110B}\x{1107}\x{1173}\x{1175}/\x{EF0C}/g;
  s/\x{110B}\x{1107}\x{116E}\x{11AF}/\x{EF02}/g;
  s/\x{110B}\x{1107}\x{116E}\x{11AB}/\x{EF01}/g;
  s/\x{110B}\x{1107}\x{116E}\x{1175}/\x{EF06}/g;
  s/\x{110B}\x{1107}\x{116E}\x{1165}/\x{EF03}/g;
  s/\x{110B}\x{1107}\x{1169}\x{11C0}/\x{EEFC}/g;
  s/\x{110B}\x{1107}\x{1169}\x{11AF}/\x{EEFB}/g;
  s/\x{110B}\x{1107}\x{1169}\x{11AB}/\x{EEFA}/g;
  s/\x{110B}\x{1107}\x{1169}\x{1161}/\x{EEFD}/g;
  s/\x{110B}\x{1107}\x{1167}\x{1175}/\x{EEF7}/g;
  s/\x{110B}\x{1107}\x{1165}\x{11C0}/\x{EEF1}/g;
  s/\x{110B}\x{1107}\x{1165}\x{11AF}/\x{EEF0}/g;
  s/\x{110B}\x{1107}\x{1165}\x{1175}/\x{EEF2}/g;
  s/\x{110B}\x{1107}\x{1161}\x{11B7}/\x{EEEB}/g;
  s/\x{110B}\x{1107}\x{1161}\x{11AF}/\x{EEEA}/g;
  s/\x{110B}\x{1107}\x{1161}\x{11AB}/\x{EEE9}/g;
  s/\x{110B}\x{1107}\x{1161}\x{1175}/\x{EEEC}/g;
  s/\x{110B}\x{1106}\x{119E}\x{1175}/\x{EEE7}/g;
  s/\x{110B}\x{1103}\x{119E}\x{11B7}/\x{EEE1}/g;
  s/\x{110B}\x{1103}\x{1175}\x{11BC}/\x{EEDF}/g;
  s/\x{110B}\x{1103}\x{1173}\x{11B7}/\x{EEDC}/g;
  s/\x{110B}\x{1103}\x{1173}\x{1175}/\x{EEDD}/g;
  s/\x{110B}\x{1103}\x{1169}\x{11AF}/\x{EEDA}/g;
  s/\x{110B}\x{1103}\x{1169}\x{11AB}/\x{EED9}/g;
  s/\x{110B}\x{1103}\x{1165}\x{11AF}/\x{EED5}/g;
  s/\x{110B}\x{1103}\x{1165}\x{1175}/\x{EED6}/g;
  s/\x{110B}\x{1103}\x{1161}\x{11AB}/\x{EED3}/g;
  s/\x{110B}\x{1100}\x{119E}\x{1175}/\x{EED1}/g;
  s/\x{110B}\x{1100}\x{1175}\x{11AB}/\x{EECE}/g;
  s/\x{110B}\x{1100}\x{1175}\x{1161}/\x{EECF}/g;
  s/\x{110B}\x{1100}\x{1173}\x{1175}/\x{EECC}/g;
  s/\x{110B}\x{1100}\x{1172}\x{1175}/\x{EECA}/g;
  s/\x{110B}\x{1100}\x{116E}\x{11AB}/\x{EEC3}/g;
  s/\x{110B}\x{1100}\x{116E}\x{1175}/\x{EEC8}/g;
  s/\x{110B}\x{1100}\x{116E}\x{1167}/\x{EEC6}/g;
  s/\x{110B}\x{1100}\x{116E}\x{1165}/\x{EEC4}/g;
  s/\x{110B}\x{1100}\x{1169}\x{11AB}/\x{EEBC}/g;
  s/\x{110B}\x{1100}\x{1169}\x{11A8}/\x{EEBB}/g;
  s/\x{110B}\x{1100}\x{1169}\x{1175}/\x{EEC0}/g;
  s/\x{110B}\x{1100}\x{1169}\x{1161}/\x{EEBD}/g;
  s/\x{110B}\x{1100}\x{1167}\x{1175}/\x{EEB8}/g;
  s/\x{110B}\x{1100}\x{1165}\x{1175}/\x{EEB6}/g;
  s/\x{110B}\x{1100}\x{1163}\x{11AB}/\x{EEB4}/g;
  s/\x{110B}\x{1100}\x{1161}\x{11A8}/\x{EEB1}/g;
  s/\x{110B}\x{1100}\x{1161}\x{1175}/\x{EEB2}/g;
  s/\x{1109}\x{119E}\x{11BA}\x{11BE}/\x{E992}/g;
  s/\x{1109}\x{119E}\x{11B8}\x{11BC}/\x{E990}/g;
  s/\x{1109}\x{119E}\x{11B8}\x{11AF}/\x{E98F}/g;
  s/\x{1109}\x{119E}\x{11B7}\x{11BA}/\x{E98D}/g;
  s/\x{1109}\x{119E}\x{11B7}\x{11B7}/\x{E98C}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11F9}/\x{E98A}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11C2}/\x{E989}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11BA}/\x{E988}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11B8}/\x{E986}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11B7}/\x{E985}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11AF}/\x{E984}/g;
  s/\x{1109}\x{119E}\x{11AF}\x{11A8}/\x{E983}/g;
  s/\x{1109}\x{119E}\x{11AE}\x{11A8}/\x{E981}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11F0}/\x{E9A3}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11BC}/\x{E9A2}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11BA}/\x{E9A1}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11B8}/\x{E9A0}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11B7}/\x{E99E}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11AF}/\x{E99C}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11AB}/\x{E99B}/g;
  s/\x{1109}\x{119E}\x{1175}\x{11A8}/\x{E99A}/g;
  s/\x{1109}\x{1175}\x{11B8}\x{11BC}/\x{E96B}/g;
  s/\x{1109}\x{1175}\x{11B7}\x{11C2}/\x{E969}/g;
  s/\x{1109}\x{1175}\x{11B7}\x{11BC}/\x{E96A}/g;
  s/\x{1109}\x{1175}\x{11B7}\x{11A8}/\x{E968}/g;
  s/\x{1109}\x{1175}\x{11AF}\x{11F9}/\x{E967}/g;
  s/\x{1109}\x{1175}\x{11AF}\x{11AF}/\x{E966}/g;
  s/\x{1109}\x{1175}\x{11AF}\x{11AE}/\x{E965}/g;
  s/\x{1109}\x{1175}\x{11AB}\x{11BA}/\x{E964}/g;
  s/\x{1109}\x{1175}\x{11AB}\x{11AB}/\x{E963}/g;
  s/\x{1109}\x{1175}\x{11A8}\x{11AB}/\x{E962}/g;
  s/\x{1109}\x{1175}\x{1172}\x{11AB}/\x{E97C}/g;
  s/\x{1109}\x{1175}\x{1169}\x{116E}/\x{E978}/g;
  s/\x{1109}\x{1175}\x{1167}\x{1175}/\x{E976}/g;
  s/\x{1109}\x{1175}\x{1163}\x{11BC}/\x{E972}/g;
  s/\x{1109}\x{1175}\x{1163}\x{11AB}/\x{E971}/g;
  s/\x{1109}\x{1175}\x{1163}\x{1175}/\x{E974}/g;
  s/\x{1109}\x{1175}\x{1163}\x{1169}/\x{E973}/g;
  s/\x{1109}\x{1173}\x{11F0}\x{11BA}/\x{E95C}/g;
  s/\x{1109}\x{1173}\x{11BC}\x{11C2}/\x{E95A}/g;
  s/\x{1109}\x{1173}\x{11BC}\x{11BA}/\x{E959}/g;
  s/\x{1109}\x{1173}\x{11BA}\x{11C2}/\x{E957}/g;
  s/\x{1109}\x{1173}\x{11BA}\x{11BE}/\x{E956}/g;
  s/\x{1109}\x{1173}\x{11B8}\x{11AF}/\x{E955}/g;
  s/\x{1109}\x{1173}\x{11B7}\x{11BC}/\x{E954}/g;
  s/\x{1109}\x{1173}\x{11AF}\x{11F9}/\x{E953}/g;
  s/\x{1109}\x{1173}\x{1175}\x{11F9}/\x{E961}/g;
  s/\x{1109}\x{1173}\x{1175}\x{11F0}/\x{E960}/g;
  s/\x{1109}\x{1172}\x{11B8}\x{11BC}/\x{E939}/g;
  s/\x{1109}\x{1172}\x{11B7}\x{11BC}/\x{E938}/g;
  s/\x{1109}\x{1172}\x{11AF}\x{11F9}/\x{E937}/g;
  s/\x{1109}\x{1172}\x{1175}\x{11BC}/\x{E952}/g;
  s/\x{1109}\x{1172}\x{1175}\x{11B8}/\x{E951}/g;
  s/\x{1109}\x{1172}\x{1175}\x{11B7}/\x{E950}/g;
  s/\x{1109}\x{1172}\x{1175}\x{11AF}/\x{E94F}/g;
  s/\x{1109}\x{1172}\x{1175}\x{11AB}/\x{E94E}/g;
  s/\x{1109}\x{1172}\x{116E}\x{11AF}/\x{E94C}/g;
  s/\x{1109}\x{1172}\x{1169}\x{11AF}/\x{E94A}/g;
  s/\x{1109}\x{1172}\x{1167}\x{11F9}/\x{E946}/g;
  s/\x{1109}\x{1172}\x{1167}\x{11BC}/\x{E945}/g;
  s/\x{1109}\x{1172}\x{1167}\x{11AF}/\x{E943}/g;
  s/\x{1109}\x{1172}\x{1167}\x{11AB}/\x{E942}/g;
  s/\x{1109}\x{1172}\x{1167}\x{1175}/\x{E947}/g;
  s/\x{1109}\x{1172}\x{1165}\x{1175}/\x{E940}/g;
  s/\x{1109}\x{1172}\x{1161}\x{11F0}/\x{E93D}/g;
  s/\x{1109}\x{1172}\x{1161}\x{11AB}/\x{E93C}/g;
  s/\x{1109}\x{1172}\x{1161}\x{1175}/\x{E93E}/g;
  s/\x{1109}\x{116E}\x{11BA}\x{11BE}/\x{E92A}/g;
  s/\x{1109}\x{116E}\x{11BA}\x{11A8}/\x{E929}/g;
  s/\x{1109}\x{116E}\x{11B8}\x{11C1}/\x{E927}/g;
  s/\x{1109}\x{116E}\x{11B8}\x{11BC}/\x{E928}/g;
  s/\x{1109}\x{116E}\x{11B7}\x{11BC}/\x{E926}/g;
  s/\x{1109}\x{116E}\x{11AF}\x{11F9}/\x{E925}/g;
  s/\x{1109}\x{116E}\x{11AF}\x{11AF}/\x{E924}/g;
  s/\x{1109}\x{116E}\x{1167}\x{11AB}/\x{E932}/g;
  s/\x{1109}\x{116E}\x{1167}\x{1175}/\x{E933}/g;
  s/\x{1109}\x{116E}\x{1161}\x{11B7}/\x{E930}/g;
  s/\x{1109}\x{116E}\x{1161}\x{11AB}/\x{E92F}/g;
  s/\x{1109}\x{116D}\x{11F0}\x{11BA}/\x{E91D}/g;
  s/\x{1109}\x{116D}\x{11B7}\x{11BC}/\x{E91B}/g;
  s/\x{1109}\x{116D}\x{11AF}\x{11F9}/\x{E91A}/g;
  s/\x{1109}\x{116D}\x{1175}\x{11BA}/\x{E923}/g;
  s/\x{1109}\x{116D}\x{1175}\x{11AF}/\x{E922}/g;
  s/\x{1109}\x{116D}\x{1175}\x{11AB}/\x{E921}/g;
  s/\x{1109}\x{116D}\x{1169}\x{11AF}/\x{E91F}/g;
  s/\x{1109}\x{1169}\x{11BA}\x{11BE}/\x{E90F}/g;
  s/\x{1109}\x{1169}\x{11BA}\x{11AE}/\x{E90E}/g;
  s/\x{1109}\x{1169}\x{11B7}\x{11BC}/\x{E90C}/g;
  s/\x{1109}\x{1169}\x{11AF}\x{11F9}/\x{E90B}/g;
  s/\x{1109}\x{1169}\x{11AB}\x{11BA}/\x{E90A}/g;
  s/\x{1109}\x{1169}\x{1175}\x{11F0}/\x{E919}/g;
  s/\x{1109}\x{1169}\x{1163}\x{1175}/\x{E917}/g;
  s/\x{1109}\x{1169}\x{1161}\x{11F9}/\x{E914}/g;
  s/\x{1109}\x{1169}\x{1161}\x{11F0}/\x{E913}/g;
  s/\x{1109}\x{1167}\x{11F0}\x{11BA}/\x{E905}/g;
  s/\x{1109}\x{1167}\x{11BA}\x{11A8}/\x{E903}/g;
  s/\x{1109}\x{1167}\x{11B8}\x{11C2}/\x{E902}/g;
  s/\x{1109}\x{1167}\x{11B8}\x{11C1}/\x{E901}/g;
  s/\x{1109}\x{1167}\x{11B8}\x{11AF}/\x{E900}/g;
  s/\x{1109}\x{1167}\x{11B7}\x{11BC}/\x{E8FF}/g;
  s/\x{1109}\x{1167}\x{11B7}\x{11BA}/\x{E8FE}/g;
  s/\x{1109}\x{1167}\x{11AF}\x{11F9}/\x{E8FD}/g;
  s/\x{1109}\x{1167}\x{1175}\x{11F0}/\x{E909}/g;
  s/\x{1109}\x{1165}\x{11BC}\x{11A8}/\x{E8EE}/g;
  s/\x{1109}\x{1165}\x{11BA}\x{11A8}/\x{E8EC}/g;
  s/\x{1109}\x{1165}\x{11B8}\x{11C2}/\x{E8EB}/g;
  s/\x{1109}\x{1165}\x{11AF}\x{11F9}/\x{E8EA}/g;
  s/\x{1109}\x{1165}\x{11A8}\x{11AF}/\x{E8E9}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11BC}/\x{E8FA}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11BA}/\x{E8F8}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11B7}/\x{E8F7}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11AF}/\x{E8F6}/g;
  s/\x{1109}\x{1165}\x{1173}\x{11A8}/\x{E8F5}/g;
  s/\x{1109}\x{1163}\x{11F0}\x{11BA}/\x{E8E7}/g;
  s/\x{1109}\x{1163}\x{11B7}\x{11BC}/\x{E8E4}/g;
  s/\x{1109}\x{1163}\x{11AF}\x{11F9}/\x{E8E3}/g;
  s/\x{1109}\x{1161}\x{11F0}\x{11BA}/\x{E8DB}/g;
  s/\x{1109}\x{1161}\x{11BC}\x{11BA}/\x{E8D9}/g;
  s/\x{1109}\x{1161}\x{11BA}\x{11C2}/\x{E8D7}/g;
  s/\x{1109}\x{1161}\x{11BA}\x{11AE}/\x{E8D6}/g;
  s/\x{1109}\x{1161}\x{11BA}\x{11A8}/\x{E8D5}/g;
  s/\x{1109}\x{1161}\x{11B7}\x{11BC}/\x{E8D4}/g;
  s/\x{1109}\x{1161}\x{11B7}\x{11BA}/\x{E8D3}/g;
  s/\x{1109}\x{1161}\x{11AF}\x{11F9}/\x{E8D2}/g;
  s/\x{1109}\x{1161}\x{11AF}\x{11AF}/\x{E8D1}/g;
  s/\x{1109}\x{1161}\x{11AF}\x{11AB}/\x{E8D0}/g;
  s/\x{1109}\x{1161}\x{11AE}\x{11C0}/\x{E8CF}/g;
  s/\x{1109}\x{1161}\x{11AE}\x{11A8}/\x{E8CE}/g;
  s/\x{1109}\x{1161}\x{11AB}\x{11BA}/\x{E8CD}/g;
  s/\x{1109}\x{1161}\x{1175}\x{11F0}/\x{E8E2}/g;
  s/\x{1109}\x{1161}\x{1173}\x{11B7}/\x{E8E0}/g;
  s/\x{1109}\x{1112}\x{1173}\x{11AF}/\x{EC98}/g;
  s/\x{1109}\x{1112}\x{1165}\x{11AF}/\x{EC95}/g;
  s/\x{1109}\x{1111}\x{119E}\x{11AF}/\x{EC93}/g;
  s/\x{1109}\x{1111}\x{116E}\x{11AB}/\x{EC90}/g;
  s/\x{1109}\x{1111}\x{1167}\x{11AB}/\x{EC8E}/g;
  s/\x{1109}\x{1110}\x{119E}\x{11B7}/\x{EC8C}/g;
  s/\x{1109}\x{1110}\x{119E}\x{11AF}/\x{EC8B}/g;
  s/\x{1109}\x{1110}\x{1175}\x{11BF}/\x{EC89}/g;
  s/\x{1109}\x{1110}\x{1173}\x{11BA}/\x{EC87}/g;
  s/\x{1109}\x{1110}\x{1173}\x{11B7}/\x{EC86}/g;
  s/\x{1109}\x{1110}\x{1173}\x{11AF}/\x{EC85}/g;
  s/\x{1109}\x{1110}\x{116E}\x{11A8}/\x{EC82}/g;
  s/\x{1109}\x{1110}\x{116E}\x{1175}/\x{EC83}/g;
  s/\x{1109}\x{1110}\x{1169}\x{11B8}/\x{EC80}/g;
  s/\x{1109}\x{1110}\x{1165}\x{11AF}/\x{EC7D}/g;
  s/\x{1109}\x{1110}\x{1165}\x{1175}/\x{EC7E}/g;
  s/\x{1109}\x{1110}\x{1161}\x{11AF}/\x{EC7B}/g;
  s/\x{1109}\x{110F}\x{1173}\x{11AB}/\x{EC76}/g;
  s/\x{1109}\x{110F}\x{1173}\x{1175}/\x{EC77}/g;
  s/\x{1109}\x{110F}\x{116E}\x{1165}/\x{EC74}/g;
  s/\x{1109}\x{110F}\x{1169}\x{11F0}/\x{EC6D}/g;
  s/\x{1109}\x{110F}\x{1169}\x{11BC}/\x{EC6C}/g;
  s/\x{1109}\x{110F}\x{1169}\x{11BA}/\x{EC6B}/g;
  s/\x{1109}\x{110F}\x{1169}\x{11AB}/\x{EC6A}/g;
  s/\x{1109}\x{110F}\x{1169}\x{1175}/\x{EC70}/g;
  s/\x{1109}\x{110F}\x{1169}\x{1161}/\x{EC6E}/g;
  s/\x{1109}\x{110F}\x{1165}\x{11AF}/\x{EC67}/g;
  s/\x{1109}\x{110F}\x{1165}\x{11AB}/\x{EC66}/g;
  s/\x{1109}\x{110F}\x{1161}\x{11BC}/\x{EC64}/g;
  s/\x{1109}\x{110E}\x{119E}\x{11B8}/\x{EC62}/g;
  s/\x{1109}\x{110E}\x{119E}\x{11AF}/\x{EC61}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11BC}/\x{EC58}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11BA}/\x{EC57}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11B8}/\x{EC56}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11B7}/\x{EC55}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11AF}/\x{EC54}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11AB}/\x{EC53}/g;
  s/\x{1109}\x{110C}\x{119E}\x{11A8}/\x{EC52}/g;
  s/\x{1109}\x{110C}\x{119E}\x{1175}/\x{EC59}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11F0}/\x{EC4F}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11C2}/\x{EC50}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11BC}/\x{EC4E}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11BA}/\x{EC4C}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11B8}/\x{EC4B}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11B7}/\x{EC4A}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11AF}/\x{EC48}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11AB}/\x{EC47}/g;
  s/\x{1109}\x{110C}\x{1175}\x{11A8}/\x{EC46}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11BC}/\x{EC42}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11BA}/\x{EC41}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11B7}/\x{EC40}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11AF}/\x{EC3F}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11AB}/\x{EC3E}/g;
  s/\x{1109}\x{110C}\x{1173}\x{11A8}/\x{EC3D}/g;
  s/\x{1109}\x{110C}\x{1173}\x{1175}/\x{EC43}/g;
  s/\x{1109}\x{110C}\x{1172}\x{11B7}/\x{EC3B}/g;
  s/\x{1109}\x{110C}\x{1172}\x{11AF}/\x{EC3A}/g;
  s/\x{1109}\x{110C}\x{1172}\x{11A8}/\x{EC39}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11BC}/\x{EC33}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11B8}/\x{EC32}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11B7}/\x{EC31}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11AF}/\x{EC30}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11AB}/\x{EC2F}/g;
  s/\x{1109}\x{110C}\x{116E}\x{11A8}/\x{EC2E}/g;
  s/\x{1109}\x{110C}\x{116E}\x{1175}/\x{EC36}/g;
  s/\x{1109}\x{110C}\x{116E}\x{1165}/\x{EC34}/g;
  s/\x{1109}\x{110C}\x{116D}\x{11BC}/\x{EC2C}/g;
  s/\x{1109}\x{110C}\x{116D}\x{11BA}/\x{EC2B}/g;
  s/\x{1109}\x{110C}\x{116D}\x{11B7}/\x{EC2A}/g;
  s/\x{1109}\x{110C}\x{116D}\x{11AE}/\x{EC29}/g;
  s/\x{1109}\x{110C}\x{116D}\x{11A8}/\x{EC28}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11BE}/\x{EC1F}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11BD}/\x{EC1E}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11BC}/\x{EC1D}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11BA}/\x{EC1C}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11B8}/\x{EC1B}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11B7}/\x{EC1A}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11AF}/\x{EC19}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11AB}/\x{EC18}/g;
  s/\x{1109}\x{110C}\x{1169}\x{11A8}/\x{EC17}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1175}/\x{EC22}/g;
  s/\x{1109}\x{110C}\x{1169}\x{1161}/\x{EC20}/g;
  s/\x{1109}\x{110C}\x{1167}\x{11F0}/\x{EC13}/g;
  s/\x{1109}\x{110C}\x{1167}\x{11BA}/\x{EC12}/g;
  s/\x{1109}\x{110C}\x{1167}\x{11AF}/\x{EC11}/g;
  s/\x{1109}\x{110C}\x{1167}\x{11AB}/\x{EC10}/g;
  s/\x{1109}\x{110C}\x{1167}\x{11A8}/\x{EC0F}/g;
  s/\x{1109}\x{110C}\x{1167}\x{1175}/\x{EC14}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11BC}/\x{EC08}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11BA}/\x{EC07}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11B8}/\x{EC06}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11B7}/\x{EC05}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11AF}/\x{EC04}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11AB}/\x{EC03}/g;
  s/\x{1109}\x{110C}\x{1165}\x{11A8}/\x{EC02}/g;
  s/\x{1109}\x{110C}\x{1165}\x{1175}/\x{EC09}/g;
  s/\x{1109}\x{110C}\x{1163}\x{11F0}/\x{EC00}/g;
  s/\x{1109}\x{110C}\x{1163}\x{11BC}/\x{EBFF}/g;
  s/\x{1109}\x{110C}\x{1163}\x{11A8}/\x{EBFE}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11F0}/\x{EBF5}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11BC}/\x{EBF4}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11BA}/\x{EBF2}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11B8}/\x{EBF1}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11B7}/\x{EBF0}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11AF}/\x{EBEE}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11AB}/\x{EBED}/g;
  s/\x{1109}\x{110C}\x{1161}\x{11A8}/\x{EBEC}/g;
  s/\x{1109}\x{110C}\x{1161}\x{1175}/\x{EBF6}/g;
  s/\x{1109}\x{110B}\x{1175}\x{11AF}/\x{EBEA}/g;
  s/\x{1109}\x{110B}\x{1173}\x{11B7}/\x{EBE8}/g;
  s/\x{1109}\x{110B}\x{1173}\x{11AF}/\x{EBE7}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11F0}/\x{EBDF}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11C1}/\x{EBE0}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11BC}/\x{EBDE}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11BA}/\x{EBDD}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11B8}/\x{EBDB}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11B7}/\x{EBD9}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11AF}/\x{EBD8}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11AE}/\x{EBD7}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11AB}/\x{EBD6}/g;
  s/\x{1109}\x{1109}\x{119E}\x{11A8}/\x{EBD5}/g;
  s/\x{1109}\x{1109}\x{119E}\x{1175}/\x{EBE1}/g;
  s/\x{1109}\x{1109}\x{1175}\x{11F9}/\x{EBD2}/g;
  s/\x{1109}\x{1109}\x{1175}\x{11F0}/\x{EBD1}/g;
  s/\x{1109}\x{1109}\x{1175}\x{11EB}/\x{EBD0}/g;
  s/\x{1109}\x{1109}\x{1175}\x{1161}/\x{EBD3}/g;
  s/\x{1109}\x{1109}\x{1173}\x{11F0}/\x{EBCD}/g;
  s/\x{1109}\x{1109}\x{1173}\x{11EB}/\x{EBCC}/g;
  s/\x{1109}\x{1109}\x{1172}\x{11F9}/\x{EBC2}/g;
  s/\x{1109}\x{1109}\x{1172}\x{11F0}/\x{EBC1}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1175}/\x{EBC8}/g;
  s/\x{1109}\x{1109}\x{1172}\x{1167}/\x{EBC3}/g;
  s/\x{1109}\x{1109}\x{116E}\x{11F9}/\x{EBB8}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1167}/\x{EBBC}/g;
  s/\x{1109}\x{1109}\x{116E}\x{1161}/\x{EBB9}/g;
  s/\x{1109}\x{1109}\x{116D}\x{11F0}/\x{EBB4}/g;
  s/\x{1109}\x{1109}\x{116D}\x{1175}/\x{EBB5}/g;
  s/\x{1109}\x{1109}\x{1169}\x{11F0}/\x{EBB2}/g;
  s/\x{1109}\x{1109}\x{1167}\x{11F9}/\x{EBB0}/g;
  s/\x{1109}\x{1109}\x{1167}\x{11F0}/\x{EBAF}/g;
  s/\x{1109}\x{1109}\x{1165}\x{11F0}/\x{EBAD}/g;
  s/\x{1109}\x{1109}\x{1163}\x{11F0}/\x{EBAC}/g;
  s/\x{1109}\x{1109}\x{1161}\x{11F0}/\x{EBAA}/g;
  s/\x{1109}\x{1109}\x{1109}\x{1173}/\x{EBE5}/g;
  s/\x{1109}\x{1109}\x{1107}\x{1169}/\x{EBE4}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11BC}/\x{EB96}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11BA}/\x{EB95}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11B8}/\x{EB94}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11B7}/\x{EB93}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11AF}/\x{EB91}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11AB}/\x{EB90}/g;
  s/\x{1109}\x{1107}\x{119E}\x{11A8}/\x{EB8F}/g;
  s/\x{1109}\x{1107}\x{119E}\x{1175}/\x{EB97}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11F0}/\x{EB8A}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11C2}/\x{EB8C}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11BE}/\x{EB8B}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11BA}/\x{EB87}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11B7}/\x{EB86}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11AF}/\x{EB85}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11AB}/\x{EB84}/g;
  s/\x{1109}\x{1107}\x{1175}\x{11A8}/\x{EB83}/g;
  s/\x{1109}\x{1107}\x{1175}\x{1161}/\x{EB8D}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11BC}/\x{EB7F}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11BA}/\x{EB7E}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11B7}/\x{EB7D}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11AF}/\x{EB7B}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11AB}/\x{EB7A}/g;
  s/\x{1109}\x{1107}\x{1173}\x{11A8}/\x{EB79}/g;
  s/\x{1109}\x{1107}\x{1173}\x{1175}/\x{EB80}/g;
  s/\x{1109}\x{1107}\x{1172}\x{1175}/\x{EB77}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11BE}/\x{EB70}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11BD}/\x{EB6F}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11BC}/\x{EB6E}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11BA}/\x{EB6D}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11B7}/\x{EB6C}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11AF}/\x{EB6A}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11AE}/\x{EB69}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11AB}/\x{EB68}/g;
  s/\x{1109}\x{1107}\x{116E}\x{11A8}/\x{EB67}/g;
  s/\x{1109}\x{1107}\x{116E}\x{1175}/\x{EB75}/g;
  s/\x{1109}\x{1107}\x{116E}\x{1167}/\x{EB73}/g;
  s/\x{1109}\x{1107}\x{116E}\x{1165}/\x{EB71}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11F0}/\x{EB5F}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11C2}/\x{EB61}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11BC}/\x{EB5E}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11BA}/\x{EB5C}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11B8}/\x{EB5B}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11B7}/\x{EB5A}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11AF}/\x{EB59}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11AB}/\x{EB58}/g;
  s/\x{1109}\x{1107}\x{1169}\x{11A8}/\x{EB57}/g;
  s/\x{1109}\x{1107}\x{1169}\x{1175}/\x{EB64}/g;
  s/\x{1109}\x{1107}\x{1169}\x{1161}/\x{EB62}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11F0}/\x{EB53}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11C2}/\x{EB54}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11BA}/\x{EB51}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11B7}/\x{EB50}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11AF}/\x{EB4F}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11AB}/\x{EB4E}/g;
  s/\x{1109}\x{1107}\x{1167}\x{11A8}/\x{EB4D}/g;
  s/\x{1109}\x{1107}\x{1167}\x{1175}/\x{EB55}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11F0}/\x{EB46}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11BC}/\x{EB45}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11BA}/\x{EB44}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11B8}/\x{EB43}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11B7}/\x{EB42}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11AF}/\x{EB41}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11AE}/\x{EB40}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11AB}/\x{EB3F}/g;
  s/\x{1109}\x{1107}\x{1165}\x{11A8}/\x{EB3E}/g;
  s/\x{1109}\x{1107}\x{1165}\x{1175}/\x{EB47}/g;
  s/\x{1109}\x{1107}\x{1163}\x{11B7}/\x{EB3C}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11F0}/\x{EB32}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11C2}/\x{EB33}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11BC}/\x{EB31}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11BA}/\x{EB30}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11B8}/\x{EB2F}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11B7}/\x{EB2E}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11AF}/\x{EB2B}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11AE}/\x{EB2A}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11AB}/\x{EB29}/g;
  s/\x{1109}\x{1107}\x{1161}\x{11A8}/\x{EB28}/g;
  s/\x{1109}\x{1107}\x{1161}\x{1175}/\x{EB34}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1175}/\x{EBA7}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1173}/\x{EBA4}/g;
  s/\x{1109}\x{1107}\x{1100}\x{116E}/\x{EBA2}/g;
  s/\x{1109}\x{1107}\x{1100}\x{1161}/\x{EBA0}/g;
  s/\x{1109}\x{1106}\x{119E}\x{11AF}/\x{EB26}/g;
  s/\x{1109}\x{1106}\x{1173}\x{11AF}/\x{EB24}/g;
  s/\x{1109}\x{1106}\x{1161}\x{11AF}/\x{EB21}/g;
  s/\x{1109}\x{1105}\x{1175}\x{11AB}/\x{EB1F}/g;
  s/\x{1109}\x{1105}\x{1173}\x{11B7}/\x{EB1D}/g;
  s/\x{1109}\x{1105}\x{1173}\x{11AF}/\x{EB1C}/g;
  s/\x{1109}\x{1105}\x{1173}\x{11AB}/\x{EB1B}/g;
  s/\x{1109}\x{1105}\x{1172}\x{11BC}/\x{EB19}/g;
  s/\x{1109}\x{1105}\x{1172}\x{11AB}/\x{EB18}/g;
  s/\x{1109}\x{1105}\x{1167}\x{11BA}/\x{EB15}/g;
  s/\x{1109}\x{1105}\x{1165}\x{11BA}/\x{EB12}/g;
  s/\x{1109}\x{1105}\x{1165}\x{1175}/\x{EB13}/g;
  s/\x{1109}\x{1105}\x{1161}\x{1175}/\x{EB0E}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11F0}/\x{EB02}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11C2}/\x{EB03}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11BC}/\x{EB01}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11BA}/\x{EB00}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11B8}/\x{EAFF}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11B7}/\x{EAFE}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11AF}/\x{EAFC}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11AE}/\x{EAFB}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11AB}/\x{EAFA}/g;
  s/\x{1109}\x{1103}\x{119E}\x{11A8}/\x{EAF9}/g;
  s/\x{1109}\x{1103}\x{119E}\x{1175}/\x{EB04}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11C2}/\x{EAF6}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11BC}/\x{EAF5}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11BA}/\x{EAF3}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11B7}/\x{EAF2}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11AF}/\x{EAF1}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11AE}/\x{EAF0}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11AB}/\x{EAEF}/g;
  s/\x{1109}\x{1103}\x{1175}\x{11A8}/\x{EAEE}/g;
  s/\x{1109}\x{1103}\x{1175}\x{1161}/\x{EAF7}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11F0}/\x{EAE4}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11BD}/\x{EAE6}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11BC}/\x{EAE3}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11BA}/\x{EAE2}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11B8}/\x{EAE1}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11B7}/\x{EAE0}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11AF}/\x{EADD}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11AE}/\x{EADC}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11AB}/\x{EADB}/g;
  s/\x{1109}\x{1103}\x{1173}\x{11A8}/\x{EADA}/g;
  s/\x{1109}\x{1103}\x{1173}\x{1175}/\x{EAE7}/g;
  s/\x{1109}\x{1103}\x{1172}\x{11F0}/\x{EAD7}/g;
  s/\x{1109}\x{1103}\x{1172}\x{11BC}/\x{EAD6}/g;
  s/\x{1109}\x{1103}\x{1172}\x{11B7}/\x{EAD5}/g;
  s/\x{1109}\x{1103}\x{1172}\x{11AF}/\x{EAD4}/g;
  s/\x{1109}\x{1103}\x{1172}\x{11A8}/\x{EAD3}/g;
  s/\x{1109}\x{1103}\x{1172}\x{1175}/\x{EAD8}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11F0}/\x{EAC8}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11BC}/\x{EAC7}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11BA}/\x{EAC6}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11B7}/\x{EAC5}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11AF}/\x{EAC2}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11AB}/\x{EAC1}/g;
  s/\x{1109}\x{1103}\x{116E}\x{11A8}/\x{EAC0}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1175}/\x{EACD}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1167}/\x{EACB}/g;
  s/\x{1109}\x{1103}\x{116E}\x{1165}/\x{EAC9}/g;
  s/\x{1109}\x{1103}\x{116D}\x{11BC}/\x{EABE}/g;
  s/\x{1109}\x{1103}\x{116D}\x{11BA}/\x{EABD}/g;
  s/\x{1109}\x{1103}\x{116D}\x{11A8}/\x{EABC}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11F0}/\x{EAB4}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11BC}/\x{EAB3}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11BA}/\x{EAB2}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11B8}/\x{EAB1}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11B7}/\x{EAB0}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11AF}/\x{EAAF}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11AB}/\x{EAAE}/g;
  s/\x{1109}\x{1103}\x{1169}\x{11A8}/\x{EAAC}/g;
  s/\x{1109}\x{1103}\x{1169}\x{1175}/\x{EAB8}/g;
  s/\x{1109}\x{1103}\x{1169}\x{1161}/\x{EAB6}/g;
  s/\x{1109}\x{1103}\x{1167}\x{11BC}/\x{EAA9}/g;
  s/\x{1109}\x{1103}\x{1167}\x{11BA}/\x{EAA8}/g;
  s/\x{1109}\x{1103}\x{1167}\x{11AB}/\x{EAA7}/g;
  s/\x{1109}\x{1103}\x{1167}\x{11A8}/\x{EAA6}/g;
  s/\x{1109}\x{1103}\x{1167}\x{1175}/\x{EAAA}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11F0}/\x{EA98}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11BF}/\x{EA99}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11BC}/\x{EA97}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11BA}/\x{EA96}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11B8}/\x{EA95}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11B7}/\x{EA94}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11AF}/\x{EA91}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11AE}/\x{EA90}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11AB}/\x{EA8F}/g;
  s/\x{1109}\x{1103}\x{1165}\x{11A8}/\x{EA8E}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1175}/\x{EA9E}/g;
  s/\x{1109}\x{1103}\x{1165}\x{1173}/\x{EA9A}/g;
  s/\x{1109}\x{1103}\x{1163}\x{11F0}/\x{EA8B}/g;
  s/\x{1109}\x{1103}\x{1163}\x{11BC}/\x{EA8A}/g;
  s/\x{1109}\x{1103}\x{1163}\x{11A8}/\x{EA89}/g;
  s/\x{1109}\x{1103}\x{1163}\x{1175}/\x{EA8C}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11F0}/\x{EA7C}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11C2}/\x{EA7E}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11BC}/\x{EA7A}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11BA}/\x{EA79}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11B8}/\x{EA78}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11B7}/\x{EA77}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11AF}/\x{EA75}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11AE}/\x{EA74}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11AB}/\x{EA73}/g;
  s/\x{1109}\x{1103}\x{1161}\x{11A8}/\x{EA72}/g;
  s/\x{1109}\x{1103}\x{1161}\x{1175}/\x{EA7F}/g;
  s/\x{1109}\x{1102}\x{1167}\x{11A8}/\x{EA6D}/g;
  s/\x{1109}\x{1102}\x{1161}\x{11AF}/\x{EA6B}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11BC}/\x{EA60}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11BA}/\x{EA5F}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11B8}/\x{EA5E}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11B7}/\x{EA5D}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11AF}/\x{EA5C}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11AE}/\x{EA5B}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11AB}/\x{EA5A}/g;
  s/\x{1109}\x{1100}\x{119E}\x{11A8}/\x{EA59}/g;
  s/\x{1109}\x{1100}\x{119E}\x{1175}/\x{EA61}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11C2}/\x{EA56}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11BC}/\x{EA55}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11BA}/\x{EA54}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11B8}/\x{EA53}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11B7}/\x{EA52}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11AF}/\x{EA50}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11AE}/\x{EA4F}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11AB}/\x{EA4D}/g;
  s/\x{1109}\x{1100}\x{1175}\x{11A8}/\x{EA4C}/g;
  s/\x{1109}\x{1100}\x{1175}\x{1161}/\x{EA57}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11F0}/\x{EA42}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11EB}/\x{EA40}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11C0}/\x{EA44}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BE}/\x{EA43}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BC}/\x{EA41}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11BA}/\x{EA3C}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11B7}/\x{EA3B}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AF}/\x{EA35}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AE}/\x{EA33}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11AB}/\x{EA30}/g;
  s/\x{1109}\x{1100}\x{1173}\x{11A8}/\x{EA2F}/g;
  s/\x{1109}\x{1100}\x{1173}\x{1175}/\x{EA45}/g;
  s/\x{1109}\x{1100}\x{1172}\x{11B7}/\x{EA2A}/g;
  s/\x{1109}\x{1100}\x{1172}\x{11AF}/\x{EA29}/g;
  s/\x{1109}\x{1100}\x{1172}\x{11AB}/\x{EA28}/g;
  s/\x{1109}\x{1100}\x{1172}\x{1175}/\x{EA2D}/g;
  s/\x{1109}\x{1100}\x{1172}\x{1165}/\x{EA2B}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11F0}/\x{EA13}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11BE}/\x{EA15}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11BD}/\x{EA14}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11BC}/\x{EA11}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11BA}/\x{EA10}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11B8}/\x{EA0F}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11B7}/\x{EA0B}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11AF}/\x{EA09}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11AE}/\x{EA08}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11AB}/\x{EA07}/g;
  s/\x{1109}\x{1100}\x{116E}\x{11A8}/\x{EA06}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1175}/\x{EA22}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1167}/\x{EA20}/g;
  s/\x{1109}\x{1100}\x{116E}\x{1165}/\x{EA16}/g;
  s/\x{1109}\x{1100}\x{116D}\x{11B7}/\x{EA04}/g;
  s/\x{1109}\x{1100}\x{116D}\x{11AF}/\x{EA03}/g;
  s/\x{1109}\x{1100}\x{116D}\x{11AB}/\x{EA02}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11F0}/\x{E9ED}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11BE}/\x{E9EF}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11BD}/\x{E9EE}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11BC}/\x{E9EC}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11BA}/\x{E9EA}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11B8}/\x{E9E9}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11B7}/\x{E9E8}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11AF}/\x{E9E6}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11AE}/\x{E9E5}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11AB}/\x{E9E4}/g;
  s/\x{1109}\x{1100}\x{1169}\x{11A8}/\x{E9E2}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1175}/\x{E9FA}/g;
  s/\x{1109}\x{1100}\x{1169}\x{1161}/\x{E9F0}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11F0}/\x{E9DE}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11C0}/\x{E9DF}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11BC}/\x{E9DD}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11BA}/\x{E9DC}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11B8}/\x{E9DB}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11B7}/\x{E9DA}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11AF}/\x{E9D9}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11AE}/\x{E9D8}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11AB}/\x{E9D7}/g;
  s/\x{1109}\x{1100}\x{1167}\x{11A8}/\x{E9D6}/g;
  s/\x{1109}\x{1100}\x{1167}\x{1175}/\x{E9E0}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11F0}/\x{E9CC}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11EB}/\x{E9CB}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11BA}/\x{E9C9}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11B8}/\x{E9C7}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11B7}/\x{E9C6}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11AF}/\x{E9C3}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11AB}/\x{E9C2}/g;
  s/\x{1109}\x{1100}\x{1165}\x{11A8}/\x{E9C0}/g;
  s/\x{1109}\x{1100}\x{1165}\x{1175}/\x{E9CD}/g;
  s/\x{1109}\x{1100}\x{1163}\x{11A8}/\x{E9BC}/g;
  s/\x{1109}\x{1100}\x{1163}\x{1175}/\x{E9BD}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11F0}/\x{E9B3}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11BC}/\x{E9B2}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11BA}/\x{E9B0}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11B8}/\x{E9AD}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11B7}/\x{E9AC}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11AF}/\x{E9AA}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11AB}/\x{E9A9}/g;
  s/\x{1109}\x{1100}\x{1161}\x{11A8}/\x{E9A6}/g;
  s/\x{1109}\x{1100}\x{1161}\x{1175}/\x{E9B4}/g;
  s/\x{1107}\x{119E}\x{11BA}\x{11C0}/\x{E65D}/g;
  s/\x{1107}\x{119E}\x{11B8}\x{11AF}/\x{E65B}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11F9}/\x{E658}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11C2}/\x{E657}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11C0}/\x{E656}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11BA}/\x{E655}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11B8}/\x{E653}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11B7}/\x{E652}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11AF}/\x{E651}/g;
  s/\x{1107}\x{119E}\x{11AF}\x{11A8}/\x{E650}/g;
  s/\x{1107}\x{119E}\x{11A8}\x{11AF}/\x{E64C}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11F0}/\x{E673}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11C2}/\x{E678}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11C0}/\x{E677}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11BE}/\x{E676}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11BD}/\x{E675}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11BC}/\x{E672}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11BA}/\x{E671}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11B8}/\x{E670}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11B7}/\x{E66F}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11AF}/\x{E66C}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11AE}/\x{E66B}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11AB}/\x{E66A}/g;
  s/\x{1107}\x{119E}\x{1175}\x{11A8}/\x{E669}/g;
  s/\x{1107}\x{119E}\x{1165}\x{11BC}/\x{E666}/g;
  s/\x{1107}\x{119E}\x{1165}\x{11AF}/\x{E665}/g;
  s/\x{1107}\x{119E}\x{1165}\x{11AB}/\x{E664}/g;
  s/\x{1107}\x{119E}\x{1165}\x{1175}/\x{E667}/g;
  s/\x{1107}\x{1175}\x{11F0}\x{11BA}/\x{E641}/g;
  s/\x{1107}\x{1175}\x{11BA}\x{11EB}/\x{E63C}/g;
  s/\x{1107}\x{1175}\x{11BA}\x{11C2}/\x{E63E}/g;
  s/\x{1107}\x{1175}\x{11BA}\x{11BE}/\x{E63D}/g;
  s/\x{1107}\x{1175}\x{11BA}\x{11A8}/\x{E63B}/g;
  s/\x{1107}\x{1175}\x{11B7}\x{11BC}/\x{E63A}/g;
  s/\x{1107}\x{1175}\x{11AF}\x{11F9}/\x{E639}/g;
  s/\x{1107}\x{1175}\x{11AE}\x{11BA}/\x{E638}/g;
  s/\x{1107}\x{1175}\x{11AE}\x{11A8}/\x{E637}/g;
  s/\x{1107}\x{1175}\x{1167}\x{1175}/\x{E647}/g;
  s/\x{1107}\x{1175}\x{1163}\x{11AB}/\x{E645}/g;
  s/\x{1107}\x{1173}\x{11BA}\x{11C0}/\x{E631}/g;
  s/\x{1107}\x{1173}\x{11BA}\x{11A8}/\x{E630}/g;
  s/\x{1107}\x{1173}\x{11B7}\x{11BA}/\x{E62F}/g;
  s/\x{1107}\x{1173}\x{11AF}\x{11F9}/\x{E62E}/g;
  s/\x{1107}\x{1173}\x{11AF}\x{11AF}/\x{E62D}/g;
  s/\x{1107}\x{1173}\x{11AE}\x{11C0}/\x{E62C}/g;
  s/\x{1107}\x{1173}\x{11A8}\x{11AF}/\x{E62B}/g;
  s/\x{1107}\x{1173}\x{1175}\x{11F9}/\x{E636}/g;
  s/\x{1107}\x{1172}\x{11B7}\x{11BC}/\x{E629}/g;
  s/\x{1107}\x{1172}\x{11AF}\x{11F9}/\x{E628}/g;
  s/\x{1107}\x{116E}\x{11BC}\x{11BF}/\x{E61F}/g;
  s/\x{1107}\x{116E}\x{11BA}\x{11BE}/\x{E61D}/g;
  s/\x{1107}\x{116E}\x{11BA}\x{11A8}/\x{E61C}/g;
  s/\x{1107}\x{116E}\x{11B8}\x{11BC}/\x{E61B}/g;
  s/\x{1107}\x{116E}\x{11B7}\x{11BC}/\x{E61A}/g;
  s/\x{1107}\x{116E}\x{11AF}\x{11F9}/\x{E619}/g;
  s/\x{1107}\x{116E}\x{11AF}\x{11AF}/\x{E616}/g;
  s/\x{1107}\x{116E}\x{11AB}\x{11BA}/\x{E615}/g;
  s/\x{1107}\x{116E}\x{1175}\x{11F9}/\x{E627}/g;
  s/\x{1107}\x{116E}\x{1167}\x{1175}/\x{E624}/g;
  s/\x{1107}\x{116E}\x{1165}\x{11F9}/\x{E622}/g;
  s/\x{1107}\x{116D}\x{11B7}\x{11BC}/\x{E611}/g;
  s/\x{1107}\x{116D}\x{11AF}\x{11F9}/\x{E610}/g;
  s/\x{1107}\x{1169}\x{11BA}\x{11A8}/\x{E608}/g;
  s/\x{1107}\x{1169}\x{11B7}\x{11BC}/\x{E607}/g;
  s/\x{1107}\x{1169}\x{11B7}\x{11BA}/\x{E606}/g;
  s/\x{1107}\x{1169}\x{11B7}\x{11B8}/\x{E605}/g;
  s/\x{1107}\x{1169}\x{11B7}\x{11B7}/\x{E604}/g;
  s/\x{1107}\x{1169}\x{11AF}\x{11F9}/\x{E603}/g;
  s/\x{1107}\x{1169}\x{11AF}\x{11AF}/\x{E602}/g;
  s/\x{1107}\x{1169}\x{1169}\x{116E}/\x{E60D}/g;
  s/\x{1107}\x{1167}\x{11F0}\x{11BA}/\x{E600}/g;
  s/\x{1107}\x{1167}\x{11BA}\x{11C0}/\x{E5FE}/g;
  s/\x{1107}\x{1167}\x{11B7}\x{11BC}/\x{E5FD}/g;
  s/\x{1107}\x{1167}\x{11AF}\x{11F9}/\x{E5FC}/g;
  s/\x{1107}\x{1165}\x{11AF}\x{11F9}/\x{E5F2}/g;
  s/\x{1107}\x{1165}\x{11AF}\x{11AB}/\x{E5F1}/g;
  s/\x{1107}\x{1165}\x{11AB}\x{11BA}/\x{E5F0}/g;
  s/\x{1107}\x{1165}\x{1173}\x{11BC}/\x{E5FB}/g;
  s/\x{1107}\x{1165}\x{1173}\x{11B7}/\x{E5FA}/g;
  s/\x{1107}\x{1165}\x{1173}\x{11AF}/\x{E5F9}/g;
  s/\x{1107}\x{1165}\x{1173}\x{11A8}/\x{E5F8}/g;
  s/\x{1107}\x{1165}\x{116E}\x{11AF}/\x{E5F6}/g;
  s/\x{1107}\x{1163}\x{11B7}\x{11BC}/\x{E5EE}/g;
  s/\x{1107}\x{1161}\x{11F0}\x{11BA}/\x{E5E6}/g;
  s/\x{1107}\x{1161}\x{11C2}\x{11AF}/\x{E5E7}/g;
  s/\x{1107}\x{1161}\x{11BC}\x{11BA}/\x{E5E4}/g;
  s/\x{1107}\x{1161}\x{11BA}\x{11BE}/\x{E5E2}/g;
  s/\x{1107}\x{1161}\x{11BA}\x{11B8}/\x{E5E1}/g;
  s/\x{1107}\x{1161}\x{11BA}\x{11AE}/\x{E5E0}/g;
  s/\x{1107}\x{1161}\x{11BA}\x{11A8}/\x{E5DF}/g;
  s/\x{1107}\x{1161}\x{11B8}\x{11BC}/\x{E5DE}/g;
  s/\x{1107}\x{1161}\x{11B8}\x{11AF}/\x{E5DD}/g;
  s/\x{1107}\x{1161}\x{11B7}\x{11BC}/\x{E5DC}/g;
  s/\x{1107}\x{1161}\x{11B7}\x{11BA}/\x{E5DB}/g;
  s/\x{1107}\x{1161}\x{11B7}\x{11B8}/\x{E5DA}/g;
  s/\x{1107}\x{1161}\x{11B7}\x{11B7}/\x{E5D9}/g;
  s/\x{1107}\x{1161}\x{11AF}\x{11F9}/\x{E5D8}/g;
  s/\x{1107}\x{1161}\x{11AF}\x{11AF}/\x{E5D7}/g;
  s/\x{1107}\x{1161}\x{11AE}\x{11C0}/\x{E5D6}/g;
  s/\x{1107}\x{1161}\x{11AE}\x{11BA}/\x{E5D4}/g;
  s/\x{1107}\x{1161}\x{11AE}\x{11A8}/\x{E5D3}/g;
  s/\x{1107}\x{1161}\x{11AB}\x{11C0}/\x{E5D2}/g;
  s/\x{1107}\x{1161}\x{11AB}\x{11BA}/\x{E5D1}/g;
  s/\x{1107}\x{1161}\x{11AB}\x{11AF}/\x{E5D0}/g;
  s/\x{1107}\x{1161}\x{11A8}\x{11BE}/\x{E5CF}/g;
  s/\x{1107}\x{1161}\x{11A8}\x{11AF}/\x{E5CE}/g;
  s/\x{1107}\x{1161}\x{1175}\x{11F0}/\x{E5ED}/g;
  s/\x{1107}\x{1161}\x{116E}\x{11AF}/\x{E5EC}/g;
  s/\x{1107}\x{1161}\x{1169}\x{11AF}/\x{E5EA}/g;
  s/\x{1107}\x{1112}\x{1161}\x{11AB}/\x{E856}/g;
  s/\x{1107}\x{1111}\x{119E}\x{11AF}/\x{E854}/g;
  s/\x{1107}\x{1111}\x{1173}\x{11AF}/\x{E851}/g;
  s/\x{1107}\x{1111}\x{1169}\x{11AB}/\x{E84E}/g;
  s/\x{1107}\x{1111}\x{1161}\x{11AB}/\x{E84C}/g;
  s/\x{1107}\x{1110}\x{119E}\x{11AF}/\x{E849}/g;
  s/\x{1107}\x{1110}\x{119E}\x{11AB}/\x{E848}/g;
  s/\x{1107}\x{1110}\x{119E}\x{1175}/\x{E84A}/g;
  s/\x{1107}\x{1110}\x{1175}\x{11B7}/\x{E846}/g;
  s/\x{1107}\x{1110}\x{1175}\x{11AF}/\x{E845}/g;
  s/\x{1107}\x{1110}\x{1175}\x{11AB}/\x{E844}/g;
  s/\x{1107}\x{1110}\x{1173}\x{11BA}/\x{E840}/g;
  s/\x{1107}\x{1110}\x{1173}\x{11AE}/\x{E83F}/g;
  s/\x{1107}\x{1110}\x{1173}\x{11AB}/\x{E83E}/g;
  s/\x{1107}\x{1110}\x{1173}\x{1175}/\x{E841}/g;
  s/\x{1107}\x{1110}\x{116E}\x{11A8}/\x{E838}/g;
  s/\x{1107}\x{1110}\x{116E}\x{1175}/\x{E83A}/g;
  s/\x{1107}\x{1110}\x{1169}\x{11B8}/\x{E836}/g;
  s/\x{1107}\x{1110}\x{1169}\x{11B7}/\x{E835}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11BA}/\x{E832}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11B7}/\x{E831}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11AF}/\x{E82E}/g;
  s/\x{1107}\x{1110}\x{1165}\x{11A8}/\x{E82D}/g;
  s/\x{1107}\x{110F}\x{1169}\x{1161}/\x{E828}/g;
  s/\x{1107}\x{110E}\x{119E}\x{11AF}/\x{E826}/g;
  s/\x{1107}\x{110E}\x{1169}\x{11BC}/\x{E824}/g;
  s/\x{1107}\x{110C}\x{119E}\x{11B7}/\x{E81E}/g;
  s/\x{1107}\x{110C}\x{119E}\x{11AF}/\x{E81D}/g;
  s/\x{1107}\x{110C}\x{119E}\x{11AB}/\x{E81C}/g;
  s/\x{1107}\x{110C}\x{119E}\x{11A8}/\x{E81B}/g;
  s/\x{1107}\x{110C}\x{119E}\x{1175}/\x{E81F}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11F0}/\x{E819}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11BC}/\x{E818}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11BA}/\x{E816}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11B7}/\x{E815}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11AF}/\x{E814}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11AE}/\x{E813}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11AB}/\x{E812}/g;
  s/\x{1107}\x{110C}\x{1175}\x{11A8}/\x{E811}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11BD}/\x{E80C}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11BA}/\x{E80B}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11B7}/\x{E80A}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11AF}/\x{E809}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11AE}/\x{E808}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11AB}/\x{E807}/g;
  s/\x{1107}\x{110C}\x{1173}\x{11A8}/\x{E806}/g;
  s/\x{1107}\x{110C}\x{1173}\x{1175}/\x{E80D}/g;
  s/\x{1107}\x{110C}\x{116E}\x{11B7}/\x{E803}/g;
  s/\x{1107}\x{110C}\x{116E}\x{11AF}/\x{E802}/g;
  s/\x{1107}\x{110C}\x{116E}\x{11A8}/\x{E801}/g;
  s/\x{1107}\x{110C}\x{116E}\x{1175}/\x{E804}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11F0}/\x{E7F5}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11BE}/\x{E7F7}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11BD}/\x{E7F6}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11BC}/\x{E7F4}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11BA}/\x{E7F3}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11B7}/\x{E7F2}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11AB}/\x{E7F1}/g;
  s/\x{1107}\x{110C}\x{1169}\x{11A8}/\x{E7F0}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1175}/\x{E7FC}/g;
  s/\x{1107}\x{110C}\x{1169}\x{1161}/\x{E7F8}/g;
  s/\x{1107}\x{110C}\x{1165}\x{11A8}/\x{E7EC}/g;
  s/\x{1107}\x{110C}\x{1165}\x{1175}/\x{E7ED}/g;
  s/\x{1107}\x{110C}\x{1163}\x{11F0}/\x{E7EA}/g;
  s/\x{1107}\x{110C}\x{1163}\x{11BC}/\x{E7E9}/g;
  s/\x{1107}\x{110C}\x{1163}\x{11A8}/\x{E7E8}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11F0}/\x{E7E5}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11BC}/\x{E7E4}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11BA}/\x{E7E3}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11B7}/\x{E7E2}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11AB}/\x{E7E1}/g;
  s/\x{1107}\x{110C}\x{1161}\x{11A8}/\x{E7E0}/g;
  s/\x{1107}\x{110C}\x{1161}\x{1175}/\x{E7E6}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11EB}/\x{E8B0}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11BC}/\x{E8B1}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11B7}/\x{E8AF}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AF}/\x{E8AA}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11AB}/\x{E8A9}/g;
  s/\x{1107}\x{110B}\x{119E}\x{11A8}/\x{E8A8}/g;
  s/\x{1107}\x{110B}\x{119E}\x{1175}/\x{E8B2}/g;
  s/\x{1107}\x{110B}\x{1175}\x{11BC}/\x{E8A6}/g;
  s/\x{1107}\x{110B}\x{1175}\x{11AF}/\x{E8A4}/g;
  s/\x{1107}\x{110B}\x{1175}\x{11AB}/\x{E8A3}/g;
  s/\x{1107}\x{110B}\x{1175}\x{11A8}/\x{E8A2}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11BC}/\x{E89E}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11BA}/\x{E89D}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11B7}/\x{E89B}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11AF}/\x{E898}/g;
  s/\x{1107}\x{110B}\x{1173}\x{11AB}/\x{E896}/g;
  s/\x{1107}\x{110B}\x{1173}\x{1175}/\x{E8A0}/g;
  s/\x{1107}\x{110B}\x{1173}\x{116E}/\x{E89F}/g;
  s/\x{1107}\x{110B}\x{1172}\x{11F9}/\x{E894}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11F9}/\x{E88E}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11F0}/\x{E88C}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11C2}/\x{E88D}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11BC}/\x{E88B}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11BA}/\x{E88A}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11B8}/\x{E889}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11B7}/\x{E888}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11AF}/\x{E887}/g;
  s/\x{1107}\x{110B}\x{116E}\x{11AB}/\x{E886}/g;
  s/\x{1107}\x{110B}\x{116E}\x{1175}/\x{E890}/g;
  s/\x{1107}\x{110B}\x{116E}\x{1165}/\x{E88F}/g;
  s/\x{1107}\x{110B}\x{116D}\x{1175}/\x{E884}/g;
  s/\x{1107}\x{110B}\x{116D}\x{1169}/\x{E883}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11BC}/\x{E87F}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11BA}/\x{E87E}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11B8}/\x{E87D}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11B7}/\x{E87B}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11AF}/\x{E878}/g;
  s/\x{1107}\x{110B}\x{1169}\x{11AB}/\x{E877}/g;
  s/\x{1107}\x{110B}\x{1169}\x{1161}/\x{E880}/g;
  s/\x{1107}\x{110B}\x{1167}\x{11BA}/\x{E873}/g;
  s/\x{1107}\x{110B}\x{1167}\x{11AF}/\x{E871}/g;
  s/\x{1107}\x{110B}\x{1167}\x{11A8}/\x{E870}/g;
  s/\x{1107}\x{110B}\x{1167}\x{1175}/\x{E874}/g;
  s/\x{1107}\x{110B}\x{1165}\x{11BC}/\x{E86C}/g;
  s/\x{1107}\x{110B}\x{1165}\x{11B8}/\x{E86B}/g;
  s/\x{1107}\x{110B}\x{1165}\x{11B7}/\x{E86A}/g;
  s/\x{1107}\x{110B}\x{1165}\x{11AB}/\x{E869}/g;
  s/\x{1107}\x{110B}\x{1165}\x{1175}/\x{E86D}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11F9}/\x{E864}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11F0}/\x{E862}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11EB}/\x{E860}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11C0}/\x{E863}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11BC}/\x{E861}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11BA}/\x{E85F}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11B7}/\x{E85E}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11AF}/\x{E85B}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11AE}/\x{E85A}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11AB}/\x{E859}/g;
  s/\x{1107}\x{110B}\x{1161}\x{11A8}/\x{E858}/g;
  s/\x{1107}\x{110B}\x{1161}\x{1175}/\x{E865}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11F0}/\x{E790}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11C2}/\x{E791}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11BA}/\x{E78E}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11B8}/\x{E78D}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11B7}/\x{E78C}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11AF}/\x{E789}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11AE}/\x{E788}/g;
  s/\x{1107}\x{1109}\x{119E}\x{11AB}/\x{E787}/g;
  s/\x{1107}\x{1109}\x{119E}\x{1175}/\x{E792}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11F0}/\x{E784}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11BC}/\x{E783}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11BA}/\x{E782}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11B8}/\x{E781}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11B7}/\x{E780}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11AF}/\x{E77F}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11AB}/\x{E77E}/g;
  s/\x{1107}\x{1109}\x{1175}\x{11A8}/\x{E77D}/g;
  s/\x{1107}\x{1109}\x{1175}\x{1161}/\x{E785}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11BE}/\x{E779}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11BA}/\x{E778}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11B7}/\x{E777}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11AF}/\x{E774}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11AE}/\x{E773}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11AB}/\x{E772}/g;
  s/\x{1107}\x{1109}\x{1173}\x{11A8}/\x{E771}/g;
  s/\x{1107}\x{1109}\x{1173}\x{1175}/\x{E77A}/g;
  s/\x{1107}\x{1109}\x{1172}\x{11BD}/\x{E76E}/g;
  s/\x{1107}\x{1109}\x{1172}\x{11BA}/\x{E76D}/g;
  s/\x{1107}\x{1109}\x{1172}\x{11B8}/\x{E76C}/g;
  s/\x{1107}\x{1109}\x{1172}\x{1175}/\x{E76F}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11BE}/\x{E765}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11BD}/\x{E764}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11BA}/\x{E763}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11B7}/\x{E762}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11AF}/\x{E760}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11AB}/\x{E75F}/g;
  s/\x{1107}\x{1109}\x{116E}\x{11A8}/\x{E75E}/g;
  s/\x{1107}\x{1109}\x{116E}\x{1175}/\x{E76A}/g;
  s/\x{1107}\x{1109}\x{116E}\x{1167}/\x{E768}/g;
  s/\x{1107}\x{1109}\x{116E}\x{1165}/\x{E766}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11F0}/\x{E754}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11BC}/\x{E753}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11BA}/\x{E752}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11B7}/\x{E751}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11AF}/\x{E750}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11AE}/\x{E74F}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11AB}/\x{E74E}/g;
  s/\x{1107}\x{1109}\x{1169}\x{11A8}/\x{E74D}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1175}/\x{E75B}/g;
  s/\x{1107}\x{1109}\x{1169}\x{1161}/\x{E755}/g;
  s/\x{1107}\x{1109}\x{1167}\x{11BA}/\x{E749}/g;
  s/\x{1107}\x{1109}\x{1167}\x{11B7}/\x{E748}/g;
  s/\x{1107}\x{1109}\x{1167}\x{11AF}/\x{E747}/g;
  s/\x{1107}\x{1109}\x{1167}\x{11AB}/\x{E746}/g;
  s/\x{1107}\x{1109}\x{1167}\x{11A8}/\x{E745}/g;
  s/\x{1107}\x{1109}\x{1167}\x{1175}/\x{E74A}/g;
  s/\x{1107}\x{1109}\x{1165}\x{11BA}/\x{E740}/g;
  s/\x{1107}\x{1109}\x{1165}\x{11B7}/\x{E73F}/g;
  s/\x{1107}\x{1109}\x{1165}\x{11AF}/\x{E73E}/g;
  s/\x{1107}\x{1109}\x{1165}\x{11AB}/\x{E73D}/g;
  s/\x{1107}\x{1109}\x{1165}\x{11A8}/\x{E73C}/g;
  s/\x{1107}\x{1109}\x{1165}\x{1175}/\x{E741}/g;
  s/\x{1107}\x{1109}\x{1163}\x{11F0}/\x{E737}/g;
  s/\x{1107}\x{1109}\x{1163}\x{11C2}/\x{E739}/g;
  s/\x{1107}\x{1109}\x{1163}\x{11C1}/\x{E738}/g;
  s/\x{1107}\x{1109}\x{1163}\x{11BC}/\x{E736}/g;
  s/\x{1107}\x{1109}\x{1163}\x{1175}/\x{E73A}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11F0}/\x{E731}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11C2}/\x{E732}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11BC}/\x{E730}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11BA}/\x{E72D}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11B8}/\x{E72C}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11B7}/\x{E72B}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11AF}/\x{E729}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11AE}/\x{E728}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11AB}/\x{E727}/g;
  s/\x{1107}\x{1109}\x{1161}\x{11A8}/\x{E726}/g;
  s/\x{1107}\x{1109}\x{1161}\x{1175}/\x{E733}/g;
  s/\x{1107}\x{1109}\x{1110}\x{1161}/\x{E7DE}/g;
  s/\x{1107}\x{1109}\x{110C}\x{119E}/\x{E7DC}/g;
  s/\x{1107}\x{1109}\x{1109}\x{1173}/\x{E7DB}/g;
  s/\x{1107}\x{1109}\x{1109}\x{1169}/\x{E7DA}/g;
  s/\x{1107}\x{1109}\x{1109}\x{1165}/\x{E7D9}/g;
  s/\x{1107}\x{1109}\x{1107}\x{1167}/\x{E7D7}/g;
  s/\x{1107}\x{1109}\x{1103}\x{119E}/\x{E7D4}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1175}/\x{E7D2}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1173}/\x{E7CE}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1172}/\x{E7CC}/g;
  s/\x{1107}\x{1109}\x{1103}\x{116E}/\x{E7C9}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1169}/\x{E7C7}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1167}/\x{E7C5}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1165}/\x{E7C1}/g;
  s/\x{1107}\x{1109}\x{1103}\x{1161}/\x{E7BE}/g;
  s/\x{1107}\x{1109}\x{1100}\x{119E}/\x{E7BC}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1175}/\x{E7B9}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1173}/\x{E7AF}/g;
  s/\x{1107}\x{1109}\x{1100}\x{116E}/\x{E7A6}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1169}/\x{E7A3}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1167}/\x{E79E}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1165}/\x{E798}/g;
  s/\x{1107}\x{1109}\x{1100}\x{1161}/\x{E794}/g;
  s/\x{1107}\x{1107}\x{119E}\x{11AF}/\x{E71E}/g;
  s/\x{1107}\x{1107}\x{119E}\x{1175}/\x{E721}/g;
  s/\x{1107}\x{1107}\x{1175}\x{11F9}/\x{E71C}/g;
  s/\x{1107}\x{1107}\x{1175}\x{11F0}/\x{E71B}/g;
  s/\x{1107}\x{1107}\x{1173}\x{11F9}/\x{E716}/g;
  s/\x{1107}\x{1107}\x{1173}\x{11F0}/\x{E715}/g;
  s/\x{1107}\x{1107}\x{116E}\x{11F9}/\x{E70D}/g;
  s/\x{1107}\x{1107}\x{116E}\x{11F0}/\x{E70C}/g;
  s/\x{1107}\x{1107}\x{116E}\x{1167}/\x{E70F}/g;
  s/\x{1107}\x{1107}\x{1169}\x{11F0}/\x{E708}/g;
  s/\x{1107}\x{1107}\x{1167}\x{11F9}/\x{E705}/g;
  s/\x{1107}\x{1107}\x{1167}\x{11F0}/\x{E704}/g;
  s/\x{1107}\x{1107}\x{1165}\x{11F9}/\x{E701}/g;
  s/\x{1107}\x{1107}\x{1163}\x{11F0}/\x{E6FF}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11F9}/\x{E6FD}/g;
  s/\x{1107}\x{1107}\x{1161}\x{11F0}/\x{E6FC}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1175}/\x{E8CC}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1173}/\x{E8C6}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1172}/\x{E8C4}/g;
  s/\x{1107}\x{1107}\x{110B}\x{116E}/\x{E8BF}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1169}/\x{E8BB}/g;
  s/\x{1107}\x{1107}\x{110B}\x{1161}/\x{E8B6}/g;
  s/\x{1107}\x{1103}\x{119E}\x{11BA}/\x{E6F4}/g;
  s/\x{1107}\x{1103}\x{119E}\x{11B7}/\x{E6F3}/g;
  s/\x{1107}\x{1103}\x{119E}\x{11AF}/\x{E6F2}/g;
  s/\x{1107}\x{1103}\x{119E}\x{11AE}/\x{E6F1}/g;
  s/\x{1107}\x{1103}\x{119E}\x{11AB}/\x{E6F0}/g;
  s/\x{1107}\x{1103}\x{119E}\x{1175}/\x{E6F5}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11BA}/\x{E6ED}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11B7}/\x{E6EC}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11AF}/\x{E6EA}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11AB}/\x{E6E9}/g;
  s/\x{1107}\x{1103}\x{1175}\x{11A8}/\x{E6E8}/g;
  s/\x{1107}\x{1103}\x{1175}\x{1161}/\x{E6EE}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11BD}/\x{E6E3}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11BC}/\x{E6E2}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11BA}/\x{E6E1}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11B7}/\x{E6E0}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AF}/\x{E6DB}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AE}/\x{E6DA}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11AB}/\x{E6D9}/g;
  s/\x{1107}\x{1103}\x{1173}\x{11A8}/\x{E6D8}/g;
  s/\x{1107}\x{1103}\x{1173}\x{1175}/\x{E6E4}/g;
  s/\x{1107}\x{1103}\x{1172}\x{1175}/\x{E6D6}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11F0}/\x{E6CD}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11BC}/\x{E6CC}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11BA}/\x{E6CB}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11B7}/\x{E6CA}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11AF}/\x{E6C8}/g;
  s/\x{1107}\x{1103}\x{116E}\x{11A8}/\x{E6C7}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1175}/\x{E6D2}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1167}/\x{E6D0}/g;
  s/\x{1107}\x{1103}\x{116E}\x{1165}/\x{E6CE}/g;
  s/\x{1107}\x{1103}\x{1169}\x{11BC}/\x{E6C0}/g;
  s/\x{1107}\x{1103}\x{1169}\x{11B7}/\x{E6BF}/g;
  s/\x{1107}\x{1103}\x{1169}\x{11AF}/\x{E6BE}/g;
  s/\x{1107}\x{1103}\x{1169}\x{11A8}/\x{E6BD}/g;
  s/\x{1107}\x{1103}\x{1169}\x{1175}/\x{E6C3}/g;
  s/\x{1107}\x{1103}\x{1169}\x{1161}/\x{E6C1}/g;
  s/\x{1107}\x{1103}\x{1167}\x{11BA}/\x{E6B9}/g;
  s/\x{1107}\x{1103}\x{1167}\x{1175}/\x{E6BA}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11BA}/\x{E6B4}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11B7}/\x{E6B3}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11AF}/\x{E6B0}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11AE}/\x{E6AF}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11AB}/\x{E6AE}/g;
  s/\x{1107}\x{1103}\x{1165}\x{11A8}/\x{E6AD}/g;
  s/\x{1107}\x{1103}\x{1165}\x{1175}/\x{E6B5}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11BA}/\x{E6A9}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11B8}/\x{E6A8}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11B7}/\x{E6A7}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11AF}/\x{E6A6}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11AB}/\x{E6A5}/g;
  s/\x{1107}\x{1103}\x{1161}\x{11A8}/\x{E6A4}/g;
  s/\x{1107}\x{1103}\x{1161}\x{1175}/\x{E6AA}/g;
  s/\x{1107}\x{1100}\x{119E}\x{11AF}/\x{E69E}/g;
  s/\x{1107}\x{1100}\x{119E}\x{1175}/\x{E69F}/g;
  s/\x{1107}\x{1100}\x{1175}\x{11BA}/\x{E69C}/g;
  s/\x{1107}\x{1100}\x{1175}\x{11B7}/\x{E69B}/g;
  s/\x{1107}\x{1100}\x{1175}\x{11AF}/\x{E69A}/g;
  s/\x{1107}\x{1100}\x{1173}\x{11BA}/\x{E696}/g;
  s/\x{1107}\x{1100}\x{1173}\x{11AF}/\x{E695}/g;
  s/\x{1107}\x{1100}\x{1173}\x{11AE}/\x{E694}/g;
  s/\x{1107}\x{1100}\x{1173}\x{11AB}/\x{E693}/g;
  s/\x{1107}\x{1100}\x{1173}\x{1175}/\x{E697}/g;
  s/\x{1107}\x{1100}\x{116E}\x{11BA}/\x{E68E}/g;
  s/\x{1107}\x{1100}\x{116E}\x{11B7}/\x{E68D}/g;
  s/\x{1107}\x{1100}\x{116E}\x{11AF}/\x{E68C}/g;
  s/\x{1107}\x{1100}\x{116E}\x{1175}/\x{E691}/g;
  s/\x{1107}\x{1100}\x{116E}\x{1165}/\x{E68F}/g;
  s/\x{1107}\x{1100}\x{1169}\x{11BA}/\x{E688}/g;
  s/\x{1107}\x{1100}\x{1169}\x{1175}/\x{E689}/g;
  s/\x{1107}\x{1100}\x{1167}\x{11BA}/\x{E684}/g;
  s/\x{1107}\x{1100}\x{1167}\x{11AB}/\x{E683}/g;
  s/\x{1107}\x{1100}\x{1167}\x{1175}/\x{E685}/g;
  s/\x{1107}\x{1100}\x{1165}\x{11BA}/\x{E67F}/g;
  s/\x{1107}\x{1100}\x{1165}\x{1175}/\x{E680}/g;
  s/\x{1107}\x{1100}\x{1161}\x{11B7}/\x{E67B}/g;
  s/\x{1107}\x{1100}\x{1161}\x{11A8}/\x{E67A}/g;
  s/\x{1107}\x{1100}\x{1161}\x{1175}/\x{E67C}/g;
  s/\x{1106}\x{119E}\x{11BA}\x{11BE}/\x{E571}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11F9}/\x{E56D}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11C2}/\x{E56C}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11BA}/\x{E56B}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11B8}/\x{E569}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11AF}/\x{E568}/g;
  s/\x{1106}\x{119E}\x{11AF}\x{11A8}/\x{E567}/g;
  s/\x{1106}\x{119E}\x{11AE}\x{11BE}/\x{E565}/g;
  s/\x{1106}\x{119E}\x{11A8}\x{11AF}/\x{E562}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11F0}/\x{E588}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11C2}/\x{E58B}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11BE}/\x{E58A}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11BD}/\x{E589}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11BC}/\x{E587}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11BA}/\x{E586}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11B8}/\x{E584}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11B7}/\x{E583}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11AF}/\x{E580}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11AE}/\x{E57F}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11AB}/\x{E57E}/g;
  s/\x{1106}\x{119E}\x{1175}\x{11A8}/\x{E57C}/g;
  s/\x{1106}\x{119E}\x{1165}\x{11BC}/\x{E57A}/g;
  s/\x{1106}\x{119E}\x{1165}\x{11AB}/\x{E579}/g;
  s/\x{1106}\x{1175}\x{11C2}\x{11AF}/\x{E557}/g;
  s/\x{1106}\x{1175}\x{11BA}\x{11C2}/\x{E554}/g;
  s/\x{1106}\x{1175}\x{11BA}\x{11C0}/\x{E553}/g;
  s/\x{1106}\x{1175}\x{11BA}\x{11BE}/\x{E552}/g;
  s/\x{1106}\x{1175}\x{11BA}\x{11A8}/\x{E551}/g;
  s/\x{1106}\x{1175}\x{11B8}\x{11BC}/\x{E550}/g;
  s/\x{1106}\x{1175}\x{11AF}\x{11F9}/\x{E54F}/g;
  s/\x{1106}\x{1175}\x{11AF}\x{11AF}/\x{E54E}/g;
  s/\x{1106}\x{1175}\x{11AF}\x{11AB}/\x{E54D}/g;
  s/\x{1106}\x{1175}\x{11AE}\x{11C0}/\x{E54C}/g;
  s/\x{1106}\x{1175}\x{11AE}\x{11BE}/\x{E54B}/g;
  s/\x{1106}\x{1175}\x{11AE}\x{11B8}/\x{E54A}/g;
  s/\x{1106}\x{1175}\x{1173}\x{11BC}/\x{E55F}/g;
  s/\x{1106}\x{1175}\x{1167}\x{1175}/\x{E55D}/g;
  s/\x{1106}\x{1175}\x{1163}\x{11AB}/\x{E55B}/g;
  s/\x{1106}\x{1173}\x{11BA}\x{11C2}/\x{E541}/g;
  s/\x{1106}\x{1173}\x{11BA}\x{11A8}/\x{E540}/g;
  s/\x{1106}\x{1173}\x{11B7}\x{11BC}/\x{E53F}/g;
  s/\x{1106}\x{1173}\x{11AF}\x{11F9}/\x{E53E}/g;
  s/\x{1106}\x{1173}\x{11AF}\x{11AF}/\x{E53D}/g;
  s/\x{1106}\x{1173}\x{1175}\x{11F9}/\x{E549}/g;
  s/\x{1106}\x{1173}\x{1173}\x{11B7}/\x{E546}/g;
  s/\x{1106}\x{1172}\x{11B7}\x{11BC}/\x{E53A}/g;
  s/\x{1106}\x{116E}\x{11BC}\x{11A8}/\x{E531}/g;
  s/\x{1106}\x{116E}\x{11BA}\x{11C0}/\x{E52F}/g;
  s/\x{1106}\x{116E}\x{11BA}\x{11A8}/\x{E52D}/g;
  s/\x{1106}\x{116E}\x{11B7}\x{11BC}/\x{E52C}/g;
  s/\x{1106}\x{116E}\x{11B7}\x{11BA}/\x{E52B}/g;
  s/\x{1106}\x{116E}\x{11AF}\x{11F9}/\x{E52A}/g;
  s/\x{1106}\x{116E}\x{11AF}\x{11AF}/\x{E529}/g;
  s/\x{1106}\x{116E}\x{11AB}\x{11BA}/\x{E528}/g;
  s/\x{1106}\x{116E}\x{11AB}\x{11AE}/\x{E527}/g;
  s/\x{1106}\x{116E}\x{1175}\x{11F9}/\x{E539}/g;
  s/\x{1106}\x{116E}\x{1167}\x{1175}/\x{E537}/g;
  s/\x{1106}\x{116E}\x{1165}\x{11F9}/\x{E535}/g;
  s/\x{1106}\x{116D}\x{11B7}\x{11BC}/\x{E524}/g;
  s/\x{1106}\x{1169}\x{11BA}\x{11BE}/\x{E51C}/g;
  s/\x{1106}\x{1169}\x{11B7}\x{11BC}/\x{E51B}/g;
  s/\x{1106}\x{1169}\x{11B7}\x{11BA}/\x{E51A}/g;
  s/\x{1106}\x{1169}\x{11AF}\x{11F9}/\x{E519}/g;
  s/\x{1106}\x{1169}\x{11AF}\x{11AF}/\x{E518}/g;
  s/\x{1106}\x{1169}\x{11AB}\x{11AE}/\x{E517}/g;
  s/\x{1106}\x{1169}\x{1169}\x{116E}/\x{E522}/g;
  s/\x{1106}\x{1169}\x{1165}\x{11AB}/\x{E520}/g;
  s/\x{1106}\x{1167}\x{11F0}\x{11C2}/\x{E515}/g;
  s/\x{1106}\x{1167}\x{11B7}\x{11BC}/\x{E512}/g;
  s/\x{1106}\x{1167}\x{11AF}\x{11F9}/\x{E511}/g;
  s/\x{1106}\x{1165}\x{11AF}\x{11F9}/\x{E50A}/g;
  s/\x{1106}\x{1165}\x{11AF}\x{11AF}/\x{E509}/g;
  s/\x{1106}\x{1165}\x{11AF}\x{11AB}/\x{E508}/g;
  s/\x{1106}\x{1165}\x{11AB}\x{11AE}/\x{E507}/g;
  s/\x{1106}\x{1165}\x{1173}\x{11AF}/\x{E50F}/g;
  s/\x{1106}\x{1163}\x{11B7}\x{11BC}/\x{E505}/g;
  s/\x{1106}\x{1161}\x{11BA}\x{11C0}/\x{E4FE}/g;
  s/\x{1106}\x{1161}\x{11BA}\x{11BE}/\x{E4FD}/g;
  s/\x{1106}\x{1161}\x{11BA}\x{11AE}/\x{E4FB}/g;
  s/\x{1106}\x{1161}\x{11B8}\x{11BC}/\x{E4FA}/g;
  s/\x{1106}\x{1161}\x{11B7}\x{11BC}/\x{E4F9}/g;
  s/\x{1106}\x{1161}\x{11AF}\x{11F9}/\x{E4F8}/g;
  s/\x{1106}\x{1161}\x{11AF}\x{11BF}/\x{E4F7}/g;
  s/\x{1106}\x{1161}\x{11AF}\x{11AF}/\x{E4F6}/g;
  s/\x{1106}\x{1161}\x{11AF}\x{11AB}/\x{E4F5}/g;
  s/\x{1106}\x{1161}\x{11AE}\x{11C0}/\x{E4F4}/g;
  s/\x{1106}\x{1161}\x{11AB}\x{11BA}/\x{E4F3}/g;
  s/\x{1106}\x{1161}\x{11AB}\x{11AE}/\x{E4F2}/g;
  s/\x{1106}\x{1161}\x{11A8}\x{11C2}/\x{E4F1}/g;
  s/\x{1106}\x{1161}\x{1175}\x{11F0}/\x{E504}/g;
  s/\x{1106}\x{110B}\x{1175}\x{11BC}/\x{E5CD}/g;
  s/\x{1106}\x{110B}\x{1173}\x{11F9}/\x{E5CA}/g;
  s/\x{1106}\x{110B}\x{1173}\x{11AB}/\x{E5C9}/g;
  s/\x{1106}\x{110B}\x{1172}\x{11AB}/\x{E5C7}/g;
  s/\x{1106}\x{110B}\x{116E}\x{11F9}/\x{E5C4}/g;
  s/\x{1106}\x{110B}\x{116E}\x{11BC}/\x{E5C3}/g;
  s/\x{1106}\x{110B}\x{116E}\x{11AB}/\x{E5C2}/g;
  s/\x{1106}\x{110B}\x{116E}\x{1175}/\x{E5C5}/g;
  s/\x{1106}\x{110B}\x{1169}\x{1161}/\x{E5BE}/g;
  s/\x{1106}\x{110B}\x{1161}\x{11F0}/\x{E5BC}/g;
  s/\x{1106}\x{110B}\x{1161}\x{11BC}/\x{E5BB}/g;
  s/\x{1106}\x{110B}\x{1161}\x{11AB}/\x{E5BA}/g;
  s/\x{1106}\x{1109}\x{1169}\x{11A8}/\x{E5B8}/g;
  s/\x{1106}\x{1107}\x{119E}\x{1175}/\x{E5B6}/g;
  s/\x{1106}\x{1107}\x{1175}\x{11AB}/\x{E5B3}/g;
  s/\x{1106}\x{1107}\x{1175}\x{1161}/\x{E5B4}/g;
  s/\x{1106}\x{1107}\x{1173}\x{1175}/\x{E5B1}/g;
  s/\x{1106}\x{1107}\x{1172}\x{1175}/\x{E5AF}/g;
  s/\x{1106}\x{1107}\x{116E}\x{11BC}/\x{E5A8}/g;
  s/\x{1106}\x{1107}\x{116E}\x{11AB}/\x{E5A7}/g;
  s/\x{1106}\x{1107}\x{116E}\x{1175}/\x{E5AD}/g;
  s/\x{1106}\x{1107}\x{116E}\x{1167}/\x{E5AB}/g;
  s/\x{1106}\x{1107}\x{116E}\x{1165}/\x{E5A9}/g;
  s/\x{1106}\x{1107}\x{1169}\x{11AE}/\x{E5A1}/g;
  s/\x{1106}\x{1107}\x{1169}\x{11AB}/\x{E5A0}/g;
  s/\x{1106}\x{1107}\x{1169}\x{11A8}/\x{E59F}/g;
  s/\x{1106}\x{1107}\x{1169}\x{1175}/\x{E5A4}/g;
  s/\x{1106}\x{1107}\x{1169}\x{1161}/\x{E5A2}/g;
  s/\x{1106}\x{1107}\x{1167}\x{1175}/\x{E59B}/g;
  s/\x{1106}\x{1107}\x{1165}\x{1175}/\x{E599}/g;
  s/\x{1106}\x{1107}\x{1161}\x{11AE}/\x{E595}/g;
  s/\x{1106}\x{1107}\x{1161}\x{11AB}/\x{E594}/g;
  s/\x{1106}\x{1107}\x{1161}\x{1175}/\x{E596}/g;
  s/\x{1106}\x{1103}\x{1167}\x{11AF}/\x{E591}/g;
  s/\x{1106}\x{1100}\x{1167}\x{1175}/\x{E58F}/g;
  s/\x{1106}\x{1100}\x{1163}\x{1169}/\x{E58D}/g;
  s/\x{1105}\x{119E}\x{11BA}\x{11BA}/\x{E47F}/g;
  s/\x{1105}\x{119E}\x{11B8}\x{11BC}/\x{E47D}/g;
  s/\x{1105}\x{119E}\x{11B8}\x{11BA}/\x{E47C}/g;
  s/\x{1105}\x{119E}\x{11B7}\x{11BA}/\x{E47A}/g;
  s/\x{1105}\x{119E}\x{11B7}\x{11B8}/\x{E479}/g;
  s/\x{1105}\x{119E}\x{11B7}\x{11B7}/\x{E478}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11F9}/\x{E476}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11EB}/\x{E474}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11C2}/\x{E475}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11BA}/\x{E473}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11B7}/\x{E472}/g;
  s/\x{1105}\x{119E}\x{11AF}\x{11A8}/\x{E471}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11F0}/\x{E48F}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11BC}/\x{E48E}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11BA}/\x{E48D}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11B7}/\x{E48C}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11AF}/\x{E48B}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11AE}/\x{E48A}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11AB}/\x{E489}/g;
  s/\x{1105}\x{119E}\x{1175}\x{11A8}/\x{E488}/g;
  s/\x{1105}\x{1175}\x{11B8}\x{11BC}/\x{E45C}/g;
  s/\x{1105}\x{1175}\x{11B7}\x{11BC}/\x{E45B}/g;
  s/\x{1105}\x{1175}\x{11B7}\x{11BA}/\x{E45A}/g;
  s/\x{1105}\x{1175}\x{11AF}\x{11F9}/\x{E459}/g;
  s/\x{1105}\x{1175}\x{11AB}\x{11BA}/\x{E458}/g;
  s/\x{1105}\x{1175}\x{116E}\x{11AF}/\x{E46A}/g;
  s/\x{1105}\x{1175}\x{1169}\x{11AF}/\x{E467}/g;
  s/\x{1105}\x{1175}\x{1167}\x{1175}/\x{E465}/g;
  s/\x{1105}\x{1175}\x{1163}\x{11BC}/\x{E462}/g;
  s/\x{1105}\x{1175}\x{1163}\x{11AB}/\x{E461}/g;
  s/\x{1105}\x{1175}\x{1163}\x{1169}/\x{E463}/g;
  s/\x{1105}\x{1173}\x{11B8}\x{11BC}/\x{E453}/g;
  s/\x{1105}\x{1173}\x{11B7}\x{11BC}/\x{E452}/g;
  s/\x{1105}\x{1173}\x{11B7}\x{11BA}/\x{E451}/g;
  s/\x{1105}\x{1173}\x{11B7}\x{11B7}/\x{E450}/g;
  s/\x{1105}\x{1173}\x{11AF}\x{11F9}/\x{E44F}/g;
  s/\x{1105}\x{1173}\x{11AF}\x{11AF}/\x{E44E}/g;
  s/\x{1105}\x{1173}\x{11AE}\x{11BE}/\x{E44D}/g;
  s/\x{1105}\x{1173}\x{1175}\x{11F9}/\x{E457}/g;
  s/\x{1105}\x{1172}\x{11B7}\x{11BC}/\x{E444}/g;
  s/\x{1105}\x{1172}\x{11AF}\x{11F9}/\x{E443}/g;
  s/\x{1105}\x{1172}\x{1175}\x{11BC}/\x{E44C}/g;
  s/\x{1105}\x{1172}\x{1167}\x{11AF}/\x{E449}/g;
  s/\x{1105}\x{1172}\x{1167}\x{11AB}/\x{E448}/g;
  s/\x{1105}\x{116E}\x{11B8}\x{11C1}/\x{E43B}/g;
  s/\x{1105}\x{116E}\x{11B7}\x{11BC}/\x{E43A}/g;
  s/\x{1105}\x{116E}\x{11B7}\x{11BA}/\x{E439}/g;
  s/\x{1105}\x{116E}\x{11AF}\x{11F9}/\x{E438}/g;
  s/\x{1105}\x{116E}\x{1167}\x{1175}/\x{E441}/g;
  s/\x{1105}\x{116E}\x{1161}\x{11AB}/\x{E43F}/g;
  s/\x{1105}\x{116D}\x{11B7}\x{11BC}/\x{E433}/g;
  s/\x{1105}\x{116D}\x{11B7}\x{11BA}/\x{E432}/g;
  s/\x{1105}\x{116D}\x{11AF}\x{11F9}/\x{E431}/g;
  s/\x{1105}\x{1169}\x{11F0}\x{11BA}/\x{E42D}/g;
  s/\x{1105}\x{1169}\x{11BA}\x{11BE}/\x{E42B}/g;
  s/\x{1105}\x{1169}\x{11B8}\x{11BC}/\x{E42A}/g;
  s/\x{1105}\x{1169}\x{11B8}\x{11B7}/\x{E429}/g;
  s/\x{1105}\x{1169}\x{11B7}\x{11BC}/\x{E428}/g;
  s/\x{1105}\x{1169}\x{11B7}\x{11BA}/\x{E427}/g;
  s/\x{1105}\x{1169}\x{11AF}\x{11F9}/\x{E426}/g;
  s/\x{1105}\x{1169}\x{11AE}\x{11BA}/\x{E425}/g;
  s/\x{1105}\x{1169}\x{1161}\x{11F0}/\x{E42F}/g;
  s/\x{1105}\x{1167}\x{11B8}\x{11BC}/\x{E422}/g;
  s/\x{1105}\x{1167}\x{11B7}\x{11BC}/\x{E421}/g;
  s/\x{1105}\x{1167}\x{11AF}\x{11F9}/\x{E420}/g;
  s/\x{1105}\x{1167}\x{11AB}\x{11BA}/\x{E41F}/g;
  s/\x{1105}\x{1165}\x{11C2}\x{11AB}/\x{E41A}/g;
  s/\x{1105}\x{1165}\x{11BA}\x{11C2}/\x{E418}/g;
  s/\x{1105}\x{1165}\x{11B8}\x{11BC}/\x{E417}/g;
  s/\x{1105}\x{1165}\x{11B7}\x{11B7}/\x{E416}/g;
  s/\x{1105}\x{1165}\x{1175}\x{11F0}/\x{E41E}/g;
  s/\x{1105}\x{1165}\x{116E}\x{11AF}/\x{E41D}/g;
  s/\x{1105}\x{1163}\x{11B7}\x{11BC}/\x{E413}/g;
  s/\x{1105}\x{1161}\x{11F0}\x{11A8}/\x{E40C}/g;
  s/\x{1105}\x{1161}\x{11C2}\x{11AB}/\x{E40D}/g;
  s/\x{1105}\x{1161}\x{11BA}\x{11A8}/\x{E408}/g;
  s/\x{1105}\x{1161}\x{11B8}\x{11BC}/\x{E407}/g;
  s/\x{1105}\x{1161}\x{11B7}\x{11BC}/\x{E406}/g;
  s/\x{1105}\x{1161}\x{11B7}\x{11BA}/\x{E405}/g;
  s/\x{1105}\x{1161}\x{11B7}\x{11B7}/\x{E404}/g;
  s/\x{1105}\x{1161}\x{11AF}\x{11F9}/\x{E403}/g;
  s/\x{1105}\x{1161}\x{11AF}\x{11AF}/\x{E402}/g;
  s/\x{1105}\x{1161}\x{1175}\x{11F0}/\x{E412}/g;
  s/\x{1105}\x{1161}\x{1169}\x{11AF}/\x{E410}/g;
  s/\x{1105}\x{1112}\x{1169}\x{1161}/\x{E4EF}/g;
  s/\x{1105}\x{1112}\x{1163}\x{11BC}/\x{E4ED}/g;
  s/\x{1105}\x{1112}\x{1161}\x{11AB}/\x{E4EB}/g;
  s/\x{1105}\x{110C}\x{1172}\x{11AF}/\x{E4E8}/g;
  s/\x{1105}\x{110C}\x{1165}\x{1175}/\x{E4E6}/g;
  s/\x{1105}\x{1109}\x{1169}\x{11A8}/\x{E4E3}/g;
  s/\x{1105}\x{1107}\x{110B}\x{119E}/\x{E4E0}/g;
  s/\x{1105}\x{1107}\x{1107}\x{1161}/\x{E4DF}/g;
  s/\x{1105}\x{1106}\x{1161}\x{11AB}/\x{E4DD}/g;
  s/\x{1105}\x{1105}\x{1175}\x{11BC}/\x{E4DA}/g;
  s/\x{1105}\x{1105}\x{1175}\x{11AB}/\x{E4D9}/g;
  s/\x{1105}\x{1105}\x{1175}\x{116E}/\x{E4DB}/g;
  s/\x{1105}\x{1105}\x{1173}\x{11BC}/\x{E4D7}/g;
  s/\x{1105}\x{1105}\x{1173}\x{11AF}/\x{E4D5}/g;
  s/\x{1105}\x{1105}\x{1172}\x{1165}/\x{E4D2}/g;
  s/\x{1105}\x{1105}\x{116E}\x{11BC}/\x{E4CB}/g;
  s/\x{1105}\x{1105}\x{116E}\x{11AB}/\x{E4CA}/g;
  s/\x{1105}\x{1105}\x{116E}\x{1175}/\x{E4D0}/g;
  s/\x{1105}\x{1105}\x{116E}\x{1165}/\x{E4CE}/g;
  s/\x{1105}\x{1105}\x{116E}\x{1161}/\x{E4CC}/g;
  s/\x{1105}\x{1105}\x{1169}\x{11BC}/\x{E4C7}/g;
  s/\x{1105}\x{1105}\x{1167}\x{11A8}/\x{E4C3}/g;
  s/\x{1105}\x{1105}\x{1167}\x{1175}/\x{E4C4}/g;
  s/\x{1105}\x{1105}\x{1165}\x{11BC}/\x{E4BE}/g;
  s/\x{1105}\x{1105}\x{1165}\x{11AB}/\x{E4BD}/g;
  s/\x{1105}\x{1105}\x{1165}\x{1175}/\x{E4C0}/g;
  s/\x{1105}\x{1105}\x{1165}\x{116E}/\x{E4BF}/g;
  s/\x{1105}\x{1105}\x{1163}\x{11BC}/\x{E4BA}/g;
  s/\x{1105}\x{1105}\x{1163}\x{1169}/\x{E4BB}/g;
  s/\x{1105}\x{1105}\x{1161}\x{11BC}/\x{E4B5}/g;
  s/\x{1105}\x{1105}\x{1161}\x{11AB}/\x{E4B4}/g;
  s/\x{1105}\x{1105}\x{1161}\x{11A8}/\x{E4B3}/g;
  s/\x{1105}\x{1105}\x{1161}\x{1175}/\x{E4B7}/g;
  s/\x{1105}\x{1105}\x{1161}\x{1169}/\x{E4B6}/g;
  s/\x{1105}\x{1103}\x{1103}\x{1161}/\x{E4B1}/g;
  s/\x{1105}\x{1102}\x{119E}\x{11BC}/\x{E4AD}/g;
  s/\x{1105}\x{1102}\x{119E}\x{11AB}/\x{E4AC}/g;
  s/\x{1105}\x{1102}\x{119E}\x{1175}/\x{E4AE}/g;
  s/\x{1105}\x{1102}\x{1173}\x{1175}/\x{E4A7}/g;
  s/\x{1105}\x{1102}\x{1172}\x{11A8}/\x{E4A5}/g;
  s/\x{1105}\x{1102}\x{116D}\x{11AB}/\x{E4A2}/g;
  s/\x{1105}\x{1102}\x{1169}\x{11BA}/\x{E4A0}/g;
  s/\x{1105}\x{1102}\x{1169}\x{11B7}/\x{E49F}/g;
  s/\x{1105}\x{1102}\x{1169}\x{11AB}/\x{E49E}/g;
  s/\x{1105}\x{1102}\x{1169}\x{11A8}/\x{E49D}/g;
  s/\x{1105}\x{1102}\x{1165}\x{1175}/\x{E499}/g;
  s/\x{1105}\x{1102}\x{1161}\x{11BC}/\x{E497}/g;
  s/\x{1105}\x{1102}\x{1161}\x{11AB}/\x{E496}/g;
  s/\x{1105}\x{1102}\x{1161}\x{11A8}/\x{E495}/g;
  s/\x{1105}\x{1100}\x{1169}\x{11BA}/\x{E492}/g;
  s/\x{1105}\x{1100}\x{1100}\x{1161}/\x{E493}/g;
  s/\x{1103}\x{119E}\x{11BA}\x{11B8}/\x{E3A2}/g;
  s/\x{1103}\x{119E}\x{11BA}\x{11AF}/\x{E3A1}/g;
  s/\x{1103}\x{119E}\x{11BA}\x{11AE}/\x{E3A0}/g;
  s/\x{1103}\x{119E}\x{11BA}\x{11A8}/\x{E39F}/g;
  s/\x{1103}\x{119E}\x{11B8}\x{11BC}/\x{E39D}/g;
  s/\x{1103}\x{119E}\x{11B8}\x{11AF}/\x{E39C}/g;
  s/\x{1103}\x{119E}\x{11B7}\x{11AB}/\x{E399}/g;
  s/\x{1103}\x{119E}\x{11B7}\x{11A8}/\x{E398}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11F9}/\x{E396}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11C2}/\x{E395}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11BA}/\x{E394}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11B8}/\x{E393}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11B7}/\x{E392}/g;
  s/\x{1103}\x{119E}\x{11AF}\x{11A8}/\x{E390}/g;
  s/\x{1103}\x{119E}\x{11A8}\x{11AF}/\x{E38C}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11F0}/\x{E3B3}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11EB}/\x{E3B1}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11BC}/\x{E3B2}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11BA}/\x{E3B0}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11B8}/\x{E3AF}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11B7}/\x{E3AE}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11AF}/\x{E3AC}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11AE}/\x{E3AB}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11AB}/\x{E3AA}/g;
  s/\x{1103}\x{119E}\x{1175}\x{11A8}/\x{E3A9}/g;
  s/\x{1103}\x{1175}\x{11BA}\x{11C0}/\x{E37F}/g;
  s/\x{1103}\x{1175}\x{11B8}\x{11C2}/\x{E37E}/g;
  s/\x{1103}\x{1175}\x{11B8}\x{11C1}/\x{E37D}/g;
  s/\x{1103}\x{1175}\x{11B7}\x{11BC}/\x{E37C}/g;
  s/\x{1103}\x{1175}\x{11AF}\x{11F9}/\x{E37B}/g;
  s/\x{1103}\x{1175}\x{11AF}\x{11AF}/\x{E37A}/g;
  s/\x{1103}\x{1175}\x{1167}\x{1175}/\x{E388}/g;
  s/\x{1103}\x{1175}\x{1163}\x{11BC}/\x{E386}/g;
  s/\x{1103}\x{1175}\x{1163}\x{11AB}/\x{E385}/g;
  s/\x{1103}\x{1173}\x{11F0}\x{11BA}/\x{E375}/g;
  s/\x{1103}\x{1173}\x{11B8}\x{11BC}/\x{E372}/g;
  s/\x{1103}\x{1173}\x{11B8}\x{11AF}/\x{E371}/g;
  s/\x{1103}\x{1173}\x{11B7}\x{11BC}/\x{E370}/g;
  s/\x{1103}\x{1173}\x{11B7}\x{11A8}/\x{E36F}/g;
  s/\x{1103}\x{1173}\x{11AF}\x{11F9}/\x{E36E}/g;
  s/\x{1103}\x{1173}\x{11AF}\x{11AF}/\x{E36D}/g;
  s/\x{1103}\x{1173}\x{11AF}\x{11AE}/\x{E36C}/g;
  s/\x{1103}\x{1173}\x{11AF}\x{11AB}/\x{E36B}/g;
  s/\x{1103}\x{1172}\x{11F0}\x{11BA}/\x{E360}/g;
  s/\x{1103}\x{1172}\x{11B7}\x{11BC}/\x{E35E}/g;
  s/\x{1103}\x{1172}\x{11AF}\x{11F9}/\x{E35D}/g;
  s/\x{1103}\x{1172}\x{1175}\x{11BC}/\x{E36A}/g;
  s/\x{1103}\x{1172}\x{1175}\x{11AB}/\x{E369}/g;
  s/\x{1103}\x{1172}\x{1167}\x{11BC}/\x{E365}/g;
  s/\x{1103}\x{1172}\x{1167}\x{11AF}/\x{E363}/g;
  s/\x{1103}\x{1172}\x{1167}\x{11AB}/\x{E362}/g;
  s/\x{1103}\x{1172}\x{1167}\x{1175}/\x{E366}/g;
  s/\x{1103}\x{116E}\x{11B7}\x{11BC}/\x{E354}/g;
  s/\x{1103}\x{116E}\x{11B7}\x{11A8}/\x{E353}/g;
  s/\x{1103}\x{116E}\x{11AF}\x{11F9}/\x{E352}/g;
  s/\x{1103}\x{116E}\x{11AF}\x{11AF}/\x{E351}/g;
  s/\x{1103}\x{116E}\x{11AF}\x{11AE}/\x{E350}/g;
  s/\x{1103}\x{116E}\x{1175}\x{11F9}/\x{E35C}/g;
  s/\x{1103}\x{116E}\x{1167}\x{1175}/\x{E35A}/g;
  s/\x{1103}\x{116E}\x{1165}\x{11F9}/\x{E358}/g;
  s/\x{1103}\x{116E}\x{1161}\x{11AB}/\x{E357}/g;
  s/\x{1103}\x{116D}\x{11BC}\x{11B7}/\x{E34B}/g;
  s/\x{1103}\x{116D}\x{11B7}\x{11BC}/\x{E34A}/g;
  s/\x{1103}\x{116D}\x{1175}\x{11AF}/\x{E34F}/g;
  s/\x{1103}\x{116D}\x{1175}\x{11AB}/\x{E34E}/g;
  s/\x{1103}\x{1169}\x{11F0}\x{11BA}/\x{E345}/g;
  s/\x{1103}\x{1169}\x{11BA}\x{11C2}/\x{E342}/g;
  s/\x{1103}\x{1169}\x{11BA}\x{11A8}/\x{E340}/g;
  s/\x{1103}\x{1169}\x{11B8}\x{11BC}/\x{E33F}/g;
  s/\x{1103}\x{1169}\x{11B7}\x{11BC}/\x{E33E}/g;
  s/\x{1103}\x{1169}\x{11AF}\x{11F9}/\x{E33D}/g;
  s/\x{1103}\x{1169}\x{11AF}\x{11BF}/\x{E33C}/g;
  s/\x{1103}\x{1169}\x{11AF}\x{11AF}/\x{E33A}/g;
  s/\x{1103}\x{1169}\x{11AE}\x{11C0}/\x{E339}/g;
  s/\x{1103}\x{1169}\x{11AE}\x{11A8}/\x{E338}/g;
  s/\x{1103}\x{1169}\x{11AB}\x{11AE}/\x{E337}/g;
  s/\x{1103}\x{1169}\x{1169}\x{11B7}/\x{E349}/g;
  s/\x{1103}\x{1169}\x{1161}\x{11F0}/\x{E347}/g;
  s/\x{1103}\x{1167}\x{11B7}\x{11BC}/\x{E334}/g;
  s/\x{1103}\x{1167}\x{11AF}\x{11F9}/\x{E333}/g;
  s/\x{1103}\x{1165}\x{11B8}\x{11C1}/\x{E327}/g;
  s/\x{1103}\x{1165}\x{11B8}\x{11BC}/\x{E328}/g;
  s/\x{1103}\x{1165}\x{11B7}\x{11BA}/\x{E326}/g;
  s/\x{1103}\x{1165}\x{1173}\x{11B7}/\x{E330}/g;
  s/\x{1103}\x{1165}\x{1173}\x{11AF}/\x{E32F}/g;
  s/\x{1103}\x{1165}\x{116E}\x{11AF}/\x{E32D}/g;
  s/\x{1103}\x{1163}\x{11F0}\x{11BA}/\x{E324}/g;
  s/\x{1103}\x{1163}\x{11B7}\x{11BC}/\x{E322}/g;
  s/\x{1103}\x{1161}\x{11F0}\x{11C2}/\x{E31D}/g;
  s/\x{1103}\x{1161}\x{11F0}\x{11BA}/\x{E31C}/g;
  s/\x{1103}\x{1161}\x{11BA}\x{11C2}/\x{E319}/g;
  s/\x{1103}\x{1161}\x{11BA}\x{11A8}/\x{E318}/g;
  s/\x{1103}\x{1161}\x{11B8}\x{11C2}/\x{E316}/g;
  s/\x{1103}\x{1161}\x{11B8}\x{11BC}/\x{E317}/g;
  s/\x{1103}\x{1161}\x{11B7}\x{11BC}/\x{E315}/g;
  s/\x{1103}\x{1161}\x{11B7}\x{11BA}/\x{E314}/g;
  s/\x{1103}\x{1161}\x{11B7}\x{11AF}/\x{E313}/g;
  s/\x{1103}\x{1161}\x{11AF}\x{11F9}/\x{E312}/g;
  s/\x{1103}\x{1161}\x{11AF}\x{11AF}/\x{E311}/g;
  s/\x{1103}\x{1161}\x{11AF}\x{11AB}/\x{E310}/g;
  s/\x{1103}\x{1161}\x{11AE}\x{11AE}/\x{E30F}/g;
  s/\x{1103}\x{1161}\x{11AE}\x{11A8}/\x{E30E}/g;
  s/\x{1103}\x{1161}\x{11AB}\x{11BA}/\x{E30D}/g;
  s/\x{1103}\x{1161}\x{1175}\x{11F0}/\x{E321}/g;
  s/\x{1103}\x{1161}\x{1169}\x{11AF}/\x{E320}/g;
  s/\x{1103}\x{110C}\x{1163}\x{11BC}/\x{E400}/g;
  s/\x{1103}\x{1109}\x{1175}\x{11A8}/\x{E3FE}/g;
  s/\x{1103}\x{1109}\x{1172}\x{11BC}/\x{E3FB}/g;
  s/\x{1103}\x{1109}\x{1172}\x{11AF}/\x{E3FA}/g;
  s/\x{1103}\x{1107}\x{116E}\x{11AF}/\x{E3F8}/g;
  s/\x{1103}\x{1107}\x{1161}\x{1175}/\x{E3F6}/g;
  s/\x{1103}\x{1106}\x{1161}\x{11AF}/\x{E3F4}/g;
  s/\x{1103}\x{1105}\x{1175}\x{11BC}/\x{E3F1}/g;
  s/\x{1103}\x{1103}\x{119E}\x{11BA}/\x{E3EA}/g;
  s/\x{1103}\x{1103}\x{119E}\x{11B7}/\x{E3E9}/g;
  s/\x{1103}\x{1103}\x{119E}\x{11AF}/\x{E3E8}/g;
  s/\x{1103}\x{1103}\x{119E}\x{11AB}/\x{E3E7}/g;
  s/\x{1103}\x{1103}\x{119E}\x{11A8}/\x{E3E6}/g;
  s/\x{1103}\x{1103}\x{119E}\x{1175}/\x{E3EB}/g;
  s/\x{1103}\x{1103}\x{1175}\x{11F9}/\x{E3E3}/g;
  s/\x{1103}\x{1103}\x{1175}\x{11F0}/\x{E3E2}/g;
  s/\x{1103}\x{1103}\x{1175}\x{1161}/\x{E3E4}/g;
  s/\x{1103}\x{1103}\x{1173}\x{11F9}/\x{E3DF}/g;
  s/\x{1103}\x{1103}\x{1173}\x{11F0}/\x{E3DE}/g;
  s/\x{1103}\x{1103}\x{1172}\x{11F0}/\x{E3D7}/g;
  s/\x{1103}\x{1103}\x{1172}\x{1175}/\x{E3DA}/g;
  s/\x{1103}\x{1103}\x{1172}\x{1167}/\x{E3D8}/g;
  s/\x{1103}\x{1103}\x{116E}\x{11F9}/\x{E3D1}/g;
  s/\x{1103}\x{1103}\x{116E}\x{11F0}/\x{E3D0}/g;
  s/\x{1103}\x{1103}\x{116E}\x{1167}/\x{E3D2}/g;
  s/\x{1103}\x{1103}\x{1169}\x{11F9}/\x{E3CA}/g;
  s/\x{1103}\x{1103}\x{1169}\x{11F0}/\x{E3C9}/g;
  s/\x{1103}\x{1103}\x{1169}\x{1165}/\x{E3CD}/g;
  s/\x{1103}\x{1103}\x{1167}\x{11F0}/\x{E3C6}/g;
  s/\x{1103}\x{1103}\x{1165}\x{1173}/\x{E3C2}/g;
  s/\x{1103}\x{1103}\x{1165}\x{116E}/\x{E3C1}/g;
  s/\x{1103}\x{1103}\x{1163}\x{11F0}/\x{E3C0}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11F9}/\x{E3BE}/g;
  s/\x{1103}\x{1103}\x{1161}\x{11F0}/\x{E3BD}/g;
  s/\x{1103}\x{1100}\x{1175}\x{11AF}/\x{E3B7}/g;
  s/\x{1103}\x{1100}\x{1173}\x{11AF}/\x{E3B5}/g;
  s/\x{1102}\x{119E}\x{11BA}\x{11BE}/\x{E293}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11F9}/\x{E28F}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11EB}/\x{E28D}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11C2}/\x{E28E}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11BA}/\x{E28C}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11B8}/\x{E28B}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11B7}/\x{E28A}/g;
  s/\x{1102}\x{119E}\x{11AF}\x{11A8}/\x{E289}/g;
  s/\x{1102}\x{119E}\x{11AE}\x{11BE}/\x{E287}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11F0}/\x{E2A3}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11C2}/\x{E2A4}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11BC}/\x{E2A2}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11BA}/\x{E2A0}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11B8}/\x{E29F}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11B7}/\x{E29E}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11AF}/\x{E29D}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11AE}/\x{E29C}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11AB}/\x{E29B}/g;
  s/\x{1102}\x{119E}\x{1175}\x{11A8}/\x{E29A}/g;
  s/\x{1102}\x{1175}\x{11B8}\x{11C2}/\x{E277}/g;
  s/\x{1102}\x{1175}\x{11B8}\x{11C1}/\x{E276}/g;
  s/\x{1102}\x{1175}\x{11B8}\x{11BC}/\x{E278}/g;
  s/\x{1102}\x{1175}\x{11B7}\x{11BC}/\x{E275}/g;
  s/\x{1102}\x{1175}\x{11B7}\x{11BA}/\x{E274}/g;
  s/\x{1102}\x{1175}\x{11AF}\x{11F9}/\x{E273}/g;
  s/\x{1102}\x{1175}\x{11AF}\x{11AF}/\x{E271}/g;
  s/\x{1102}\x{1175}\x{11AF}\x{11AE}/\x{E270}/g;
  s/\x{1102}\x{1175}\x{11A8}\x{11AF}/\x{E26F}/g;
  s/\x{1102}\x{1175}\x{1167}\x{1175}/\x{E281}/g;
  s/\x{1102}\x{1175}\x{1163}\x{11BC}/\x{E27F}/g;
  s/\x{1102}\x{1175}\x{1163}\x{11AB}/\x{E27E}/g;
  s/\x{1102}\x{1173}\x{11F0}\x{11C2}/\x{E26B}/g;
  s/\x{1102}\x{1173}\x{11AF}\x{11F9}/\x{E269}/g;
  s/\x{1102}\x{1173}\x{11AF}\x{11AB}/\x{E268}/g;
  s/\x{1102}\x{1173}\x{11A8}\x{11AF}/\x{E267}/g;
  s/\x{1102}\x{1173}\x{1175}\x{11F0}/\x{E26D}/g;
  s/\x{1102}\x{1173}\x{1175}\x{116E}/\x{E26E}/g;
  s/\x{1102}\x{1172}\x{11B7}\x{11BC}/\x{E261}/g;
  s/\x{1102}\x{1172}\x{11AF}\x{11F9}/\x{E260}/g;
  s/\x{1102}\x{1172}\x{1175}\x{11BC}/\x{E266}/g;
  s/\x{1102}\x{1172}\x{1175}\x{11AB}/\x{E265}/g;
  s/\x{1102}\x{116E}\x{11C2}\x{11AB}/\x{E259}/g;
  s/\x{1102}\x{116E}\x{11B8}\x{11BC}/\x{E258}/g;
  s/\x{1102}\x{116E}\x{11B7}\x{11BC}/\x{E257}/g;
  s/\x{1102}\x{116E}\x{11B7}\x{11BA}/\x{E256}/g;
  s/\x{1102}\x{116E}\x{11AF}\x{11AF}/\x{E255}/g;
  s/\x{1102}\x{116E}\x{11AB}\x{11EB}/\x{E254}/g;
  s/\x{1102}\x{116E}\x{11AB}\x{11BA}/\x{E253}/g;
  s/\x{1102}\x{116E}\x{11AB}\x{11AE}/\x{E252}/g;
  s/\x{1102}\x{116E}\x{1175}\x{11EB}/\x{E25E}/g;
  s/\x{1102}\x{116E}\x{1175}\x{1175}/\x{E25F}/g;
  s/\x{1102}\x{116E}\x{1167}\x{1175}/\x{E25D}/g;
  s/\x{1102}\x{116E}\x{1161}\x{11AB}/\x{E25B}/g;
  s/\x{1102}\x{116D}\x{11B7}\x{11BC}/\x{E24D}/g;
  s/\x{1102}\x{116D}\x{11AF}\x{11F9}/\x{E24C}/g;
  s/\x{1102}\x{116D}\x{1161}\x{11BC}/\x{E250}/g;
  s/\x{1102}\x{1169}\x{11F0}\x{11C2}/\x{E247}/g;
  s/\x{1102}\x{1169}\x{11C1}\x{11B8}/\x{E248}/g;
  s/\x{1102}\x{1169}\x{11BA}\x{11C2}/\x{E244}/g;
  s/\x{1102}\x{1169}\x{11B8}\x{11C2}/\x{E242}/g;
  s/\x{1102}\x{1169}\x{11B8}\x{11C1}/\x{E241}/g;
  s/\x{1102}\x{1169}\x{11B8}\x{11BC}/\x{E243}/g;
  s/\x{1102}\x{1169}\x{11B7}\x{11BC}/\x{E240}/g;
  s/\x{1102}\x{1169}\x{11AF}\x{11F9}/\x{E23F}/g;
  s/\x{1102}\x{1169}\x{11AF}\x{11AF}/\x{E23E}/g;
  s/\x{1102}\x{1169}\x{11AF}\x{11AB}/\x{E23D}/g;
  s/\x{1102}\x{1169}\x{1175}\x{11F0}/\x{E24B}/g;
  s/\x{1102}\x{1167}\x{11BA}\x{11C2}/\x{E23A}/g;
  s/\x{1102}\x{1167}\x{11BA}\x{11A8}/\x{E239}/g;
  s/\x{1102}\x{1167}\x{11AF}\x{11F9}/\x{E238}/g;
  s/\x{1102}\x{1167}\x{11AB}\x{11BA}/\x{E237}/g;
  s/\x{1102}\x{1167}\x{11AB}\x{11AB}/\x{E236}/g;
  s/\x{1102}\x{1167}\x{11AB}\x{11A8}/\x{E235}/g;
  s/\x{1102}\x{1167}\x{11A8}\x{11C2}/\x{E234}/g;
  s/\x{1102}\x{1167}\x{11A8}\x{11BF}/\x{E233}/g;
  s/\x{1102}\x{1165}\x{11BA}\x{11C2}/\x{E22C}/g;
  s/\x{1102}\x{1165}\x{11BA}\x{11AF}/\x{E22B}/g;
  s/\x{1102}\x{1165}\x{11B8}\x{11C2}/\x{E22A}/g;
  s/\x{1102}\x{1165}\x{11B8}\x{11AF}/\x{E229}/g;
  s/\x{1102}\x{1165}\x{11B7}\x{11BA}/\x{E228}/g;
  s/\x{1102}\x{1165}\x{11B7}\x{11AF}/\x{E227}/g;
  s/\x{1102}\x{1165}\x{11B7}\x{11A8}/\x{E226}/g;
  s/\x{1102}\x{1165}\x{11AF}\x{11AF}/\x{E225}/g;
  s/\x{1102}\x{1165}\x{1173}\x{11C2}/\x{E232}/g;
  s/\x{1102}\x{1165}\x{1173}\x{11A8}/\x{E231}/g;
  s/\x{1102}\x{1165}\x{116E}\x{11B7}/\x{E22F}/g;
  s/\x{1102}\x{1163}\x{11B7}\x{11BC}/\x{E222}/g;
  s/\x{1102}\x{1161}\x{11C2}\x{11AB}/\x{E21A}/g;
  s/\x{1102}\x{1161}\x{11BD}\x{11B8}/\x{E218}/g;
  s/\x{1102}\x{1161}\x{11BC}\x{11A8}/\x{E216}/g;
  s/\x{1102}\x{1161}\x{11BA}\x{11C2}/\x{E214}/g;
  s/\x{1102}\x{1161}\x{11BA}\x{11BD}/\x{E213}/g;
  s/\x{1102}\x{1161}\x{11BA}\x{11B8}/\x{E212}/g;
  s/\x{1102}\x{1161}\x{11BA}\x{11A8}/\x{E211}/g;
  s/\x{1102}\x{1161}\x{11B8}\x{11BC}/\x{E210}/g;
  s/\x{1102}\x{1161}\x{11B7}\x{11BC}/\x{E20F}/g;
  s/\x{1102}\x{1161}\x{11B7}\x{11BA}/\x{E20E}/g;
  s/\x{1102}\x{1161}\x{11B7}\x{11B8}/\x{E20D}/g;
  s/\x{1102}\x{1161}\x{11B7}\x{11A8}/\x{E20C}/g;
  s/\x{1102}\x{1161}\x{11AF}\x{11F9}/\x{E20B}/g;
  s/\x{1102}\x{1161}\x{11AF}\x{11AF}/\x{E20A}/g;
  s/\x{1102}\x{1161}\x{11AF}\x{11AB}/\x{E209}/g;
  s/\x{1102}\x{1161}\x{11AE}\x{11B8}/\x{E208}/g;
  s/\x{1102}\x{1161}\x{11AB}\x{11BA}/\x{E207}/g;
  s/\x{1102}\x{1112}\x{1175}\x{11BC}/\x{E30C}/g;
  s/\x{1102}\x{1112}\x{1172}\x{11AB}/\x{E30A}/g;
  s/\x{1102}\x{1112}\x{1163}\x{11BC}/\x{E307}/g;
  s/\x{1102}\x{1109}\x{1175}\x{1161}/\x{E303}/g;
  s/\x{1102}\x{1107}\x{119E}\x{1175}/\x{E301}/g;
  s/\x{1102}\x{1107}\x{1175}\x{1161}/\x{E2FF}/g;
  s/\x{1102}\x{1107}\x{1173}\x{1175}/\x{E2FD}/g;
  s/\x{1102}\x{1107}\x{1172}\x{1175}/\x{E2FB}/g;
  s/\x{1102}\x{1107}\x{116E}\x{1175}/\x{E2F9}/g;
  s/\x{1102}\x{1107}\x{116E}\x{1167}/\x{E2F7}/g;
  s/\x{1102}\x{1107}\x{116E}\x{1165}/\x{E2F5}/g;
  s/\x{1102}\x{1107}\x{1169}\x{1175}/\x{E2F2}/g;
  s/\x{1102}\x{1107}\x{1169}\x{1161}/\x{E2F0}/g;
  s/\x{1102}\x{1107}\x{1167}\x{1175}/\x{E2EE}/g;
  s/\x{1102}\x{1107}\x{1165}\x{1175}/\x{E2EC}/g;
  s/\x{1102}\x{1107}\x{1161}\x{1175}/\x{E2E9}/g;
  s/\x{1102}\x{1103}\x{119E}\x{1175}/\x{E2E7}/g;
  s/\x{1102}\x{1103}\x{1175}\x{1161}/\x{E2E5}/g;
  s/\x{1102}\x{1103}\x{1173}\x{1175}/\x{E2E3}/g;
  s/\x{1102}\x{1103}\x{1172}\x{1175}/\x{E2E1}/g;
  s/\x{1102}\x{1103}\x{116E}\x{1175}/\x{E2DF}/g;
  s/\x{1102}\x{1103}\x{116E}\x{1167}/\x{E2DD}/g;
  s/\x{1102}\x{1103}\x{116E}\x{1165}/\x{E2DB}/g;
  s/\x{1102}\x{1103}\x{1169}\x{11AB}/\x{E2D5}/g;
  s/\x{1102}\x{1103}\x{1169}\x{1175}/\x{E2D8}/g;
  s/\x{1102}\x{1103}\x{1169}\x{1161}/\x{E2D6}/g;
  s/\x{1102}\x{1103}\x{1167}\x{1175}/\x{E2D3}/g;
  s/\x{1102}\x{1103}\x{1165}\x{1175}/\x{E2D0}/g;
  s/\x{1102}\x{1103}\x{1161}\x{11AB}/\x{E2CC}/g;
  s/\x{1102}\x{1103}\x{1161}\x{1175}/\x{E2CD}/g;
  s/\x{1102}\x{1102}\x{119E}\x{11AB}/\x{E2CA}/g;
  s/\x{1102}\x{1102}\x{1173}\x{11AB}/\x{E2C8}/g;
  s/\x{1102}\x{1102}\x{1165}\x{1175}/\x{E2C4}/g;
  s/\x{1102}\x{1102}\x{1163}\x{11BC}/\x{E2C2}/g;
  s/\x{1102}\x{1100}\x{119E}\x{1175}/\x{E2BF}/g;
  s/\x{1102}\x{1100}\x{1175}\x{1161}/\x{E2BD}/g;
  s/\x{1102}\x{1100}\x{1173}\x{1175}/\x{E2BB}/g;
  s/\x{1102}\x{1100}\x{1172}\x{1175}/\x{E2B9}/g;
  s/\x{1102}\x{1100}\x{116E}\x{1175}/\x{E2B7}/g;
  s/\x{1102}\x{1100}\x{116E}\x{1167}/\x{E2B5}/g;
  s/\x{1102}\x{1100}\x{116E}\x{1165}/\x{E2B3}/g;
  s/\x{1102}\x{1100}\x{1169}\x{1175}/\x{E2B0}/g;
  s/\x{1102}\x{1100}\x{1169}\x{1161}/\x{E2AE}/g;
  s/\x{1102}\x{1100}\x{1167}\x{1175}/\x{E2AB}/g;
  s/\x{1102}\x{1100}\x{1165}\x{1175}/\x{E2A9}/g;
  s/\x{1102}\x{1100}\x{1161}\x{1175}/\x{E2A6}/g;
  s/\x{1100}\x{119E}\x{11F0}\x{11BA}/\x{E1C5}/g;
  s/\x{1100}\x{119E}\x{11BA}\x{11B8}/\x{E1C1}/g;
  s/\x{1100}\x{119E}\x{11BA}\x{11A8}/\x{E1C0}/g;
  s/\x{1100}\x{119E}\x{11B8}\x{11BC}/\x{E1BE}/g;
  s/\x{1100}\x{119E}\x{11B8}\x{11AF}/\x{E1BD}/g;
  s/\x{1100}\x{119E}\x{11B7}\x{11BC}/\x{E1BB}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11F9}/\x{E1B9}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11C2}/\x{E1B8}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11C1}/\x{E1B7}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11BA}/\x{E1B6}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11B8}/\x{E1B2}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11B7}/\x{E1B1}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11AF}/\x{E1B0}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11AB}/\x{E1AF}/g;
  s/\x{1100}\x{119E}\x{11AF}\x{11A8}/\x{E1AE}/g;
  s/\x{1100}\x{119E}\x{11AE}\x{11B8}/\x{E1AC}/g;
  s/\x{1100}\x{119E}\x{11A8}\x{11A8}/\x{E1A9}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11F0}/\x{E1D5}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11BC}/\x{E1D4}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11BA}/\x{E1D3}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11B8}/\x{E1D2}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11B7}/\x{E1D1}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11AF}/\x{E1D0}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11AB}/\x{E1CF}/g;
  s/\x{1100}\x{119E}\x{1175}\x{11A8}/\x{E1CE}/g;
  s/\x{1100}\x{119E}\x{1165}\x{11BC}/\x{E1CC}/g;
  s/\x{1100}\x{119E}\x{1165}\x{11AB}/\x{E1CB}/g;
  s/\x{1100}\x{1175}\x{11BA}\x{11BE}/\x{E196}/g;
  s/\x{1100}\x{1175}\x{11BA}\x{11A8}/\x{E194}/g;
  s/\x{1100}\x{1175}\x{11B8}\x{11C2}/\x{E192}/g;
  s/\x{1100}\x{1175}\x{11B8}\x{11C1}/\x{E191}/g;
  s/\x{1100}\x{1175}\x{11B8}\x{11BD}/\x{E190}/g;
  s/\x{1100}\x{1175}\x{11B8}\x{11BC}/\x{E193}/g;
  s/\x{1100}\x{1175}\x{11B8}\x{11B8}/\x{E18F}/g;
  s/\x{1100}\x{1175}\x{11B7}\x{11BE}/\x{E18D}/g;
  s/\x{1100}\x{1175}\x{11B7}\x{11BC}/\x{E18E}/g;
  s/\x{1100}\x{1175}\x{11AF}\x{11F9}/\x{E18B}/g;
  s/\x{1100}\x{1175}\x{11AF}\x{11AF}/\x{E18A}/g;
  s/\x{1100}\x{1175}\x{11AF}\x{11AB}/\x{E189}/g;
  s/\x{1100}\x{1175}\x{11AE}\x{11AE}/\x{E188}/g;
  s/\x{1100}\x{1175}\x{11AE}\x{11A8}/\x{E187}/g;
  s/\x{1100}\x{1175}\x{1172}\x{11AB}/\x{E1A3}/g;
  s/\x{1100}\x{1175}\x{1167}\x{1175}/\x{E19F}/g;
  s/\x{1100}\x{1175}\x{1163}\x{11BC}/\x{E19D}/g;
  s/\x{1100}\x{1175}\x{1163}\x{11AB}/\x{E19C}/g;
  s/\x{1100}\x{1173}\x{11BA}\x{11BE}/\x{E17D}/g;
  s/\x{1100}\x{1173}\x{11B7}\x{11EB}/\x{E17A}/g;
  s/\x{1100}\x{1173}\x{11B7}\x{11C2}/\x{E17B}/g;
  s/\x{1100}\x{1173}\x{11B7}\x{11BC}/\x{E17C}/g;
  s/\x{1100}\x{1173}\x{11B7}\x{11BA}/\x{E179}/g;
  s/\x{1100}\x{1173}\x{11B7}\x{11A8}/\x{E178}/g;
  s/\x{1100}\x{1173}\x{11AF}\x{11F9}/\x{E177}/g;
  s/\x{1100}\x{1173}\x{11AF}\x{11AF}/\x{E176}/g;
  s/\x{1100}\x{1173}\x{11AF}\x{11AB}/\x{E175}/g;
  s/\x{1100}\x{1173}\x{11AE}\x{11C0}/\x{E174}/g;
  s/\x{1100}\x{1173}\x{11AE}\x{11BE}/\x{E173}/g;
  s/\x{1100}\x{1173}\x{11AB}\x{11C0}/\x{E172}/g;
  s/\x{1100}\x{1173}\x{11AB}\x{11BE}/\x{E171}/g;
  s/\x{1100}\x{1173}\x{11AB}\x{11BA}/\x{E170}/g;
  s/\x{1100}\x{1173}\x{1175}\x{11F9}/\x{E186}/g;
  s/\x{1100}\x{1173}\x{1165}\x{1175}/\x{E182}/g;
  s/\x{1100}\x{1172}\x{11B7}\x{11BC}/\x{E157}/g;
  s/\x{1100}\x{1172}\x{11AF}\x{11F9}/\x{E156}/g;
  s/\x{1100}\x{1172}\x{1175}\x{11F0}/\x{E16F}/g;
  s/\x{1100}\x{1172}\x{1175}\x{11BC}/\x{E16E}/g;
  s/\x{1100}\x{1172}\x{1175}\x{11AF}/\x{E16C}/g;
  s/\x{1100}\x{1172}\x{1175}\x{11A8}/\x{E16B}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11F9}/\x{E165}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11F0}/\x{E164}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11BC}/\x{E163}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11AF}/\x{E161}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11AB}/\x{E160}/g;
  s/\x{1100}\x{1172}\x{1167}\x{11A8}/\x{E15F}/g;
  s/\x{1100}\x{1172}\x{1167}\x{1175}/\x{E166}/g;
  s/\x{1100}\x{1172}\x{1165}\x{11AB}/\x{E15D}/g;
  s/\x{1100}\x{1172}\x{1161}\x{11AB}/\x{E15B}/g;
  s/\x{1100}\x{116E}\x{11F0}\x{11A8}/\x{E143}/g;
  s/\x{1100}\x{116E}\x{11C2}\x{11AF}/\x{E144}/g;
  s/\x{1100}\x{116E}\x{11BC}\x{11A8}/\x{E141}/g;
  s/\x{1100}\x{116E}\x{11B8}\x{11BC}/\x{E140}/g;
  s/\x{1100}\x{116E}\x{11B7}\x{11BC}/\x{E13F}/g;
  s/\x{1100}\x{116E}\x{11B7}\x{11BA}/\x{E13E}/g;
  s/\x{1100}\x{116E}\x{11B7}\x{11A8}/\x{E13D}/g;
  s/\x{1100}\x{116E}\x{11AF}\x{11F9}/\x{E13C}/g;
  s/\x{1100}\x{116E}\x{11AF}\x{11AF}/\x{E13B}/g;
  s/\x{1100}\x{116E}\x{11AF}\x{11AB}/\x{E13A}/g;
  s/\x{1100}\x{116E}\x{11AB}\x{11BA}/\x{E139}/g;
  s/\x{1100}\x{116E}\x{11A8}\x{11BF}/\x{E138}/g;
  s/\x{1100}\x{116E}\x{1175}\x{11F9}/\x{E155}/g;
  s/\x{1100}\x{116E}\x{1175}\x{11F0}/\x{E154}/g;
  s/\x{1100}\x{116E}\x{116E}\x{11AF}/\x{E153}/g;
  s/\x{1100}\x{116E}\x{1167}\x{1175}/\x{E151}/g;
  s/\x{1100}\x{116E}\x{1165}\x{11F9}/\x{E14D}/g;
  s/\x{1100}\x{116E}\x{1165}\x{1173}/\x{E14E}/g;
  s/\x{1100}\x{116E}\x{1161}\x{11BC}/\x{E149}/g;
  s/\x{1100}\x{116E}\x{1161}\x{11B7}/\x{E148}/g;
  s/\x{1100}\x{116E}\x{1161}\x{11AB}/\x{E147}/g;
  s/\x{1100}\x{116E}\x{1161}\x{1175}/\x{E14A}/g;
  s/\x{1100}\x{116D}\x{11B7}\x{11BC}/\x{E12D}/g;
  s/\x{1100}\x{116D}\x{11AF}\x{11F9}/\x{E12C}/g;
  s/\x{1100}\x{116D}\x{1175}\x{11AB}/\x{E137}/g;
  s/\x{1100}\x{116D}\x{1163}\x{1175}/\x{E133}/g;
  s/\x{1100}\x{116D}\x{1161}\x{1175}/\x{E130}/g;
  s/\x{1100}\x{1169}\x{11F0}\x{11BA}/\x{E11D}/g;
  s/\x{1100}\x{1169}\x{11C2}\x{11B8}/\x{E11F}/g;
  s/\x{1100}\x{1169}\x{11C2}\x{11B7}/\x{E11E}/g;
  s/\x{1100}\x{1169}\x{11BA}\x{11BE}/\x{E11A}/g;
  s/\x{1100}\x{1169}\x{11B8}\x{11BC}/\x{E119}/g;
  s/\x{1100}\x{1169}\x{11B7}\x{11BC}/\x{E118}/g;
  s/\x{1100}\x{1169}\x{11B7}\x{11BA}/\x{E117}/g;
  s/\x{1100}\x{1169}\x{11B7}\x{11A8}/\x{E116}/g;
  s/\x{1100}\x{1169}\x{11AF}\x{11F9}/\x{E115}/g;
  s/\x{1100}\x{1169}\x{11AF}\x{11AF}/\x{E112}/g;
  s/\x{1100}\x{1169}\x{1175}\x{11F0}/\x{E12B}/g;
  s/\x{1100}\x{1169}\x{1169}\x{11AF}/\x{E12A}/g;
  s/\x{1100}\x{1169}\x{1165}\x{11AF}/\x{E128}/g;
  s/\x{1100}\x{1169}\x{1165}\x{11AB}/\x{E127}/g;
  s/\x{1100}\x{1169}\x{1163}\x{11BC}/\x{E125}/g;
  s/\x{1100}\x{1169}\x{1161}\x{11F9}/\x{E123}/g;
  s/\x{1100}\x{1169}\x{1161}\x{11F0}/\x{E122}/g;
  s/\x{1100}\x{1167}\x{11F0}\x{11BA}/\x{E10F}/g;
  s/\x{1100}\x{1167}\x{11BA}\x{11C2}/\x{E10C}/g;
  s/\x{1100}\x{1167}\x{11BA}\x{11C0}/\x{E10B}/g;
  s/\x{1100}\x{1167}\x{11BA}\x{11BE}/\x{E10A}/g;
  s/\x{1100}\x{1167}\x{11BA}\x{11A8}/\x{E109}/g;
  s/\x{1100}\x{1167}\x{11B8}\x{11BE}/\x{E108}/g;
  s/\x{1100}\x{1167}\x{11B7}\x{11BC}/\x{E107}/g;
  s/\x{1100}\x{1167}\x{11AF}\x{11F9}/\x{E106}/g;
  s/\x{1100}\x{1167}\x{11AE}\x{11C0}/\x{E105}/g;
  s/\x{1100}\x{1167}\x{1175}\x{11F0}/\x{E111}/g;
  s/\x{1100}\x{1165}\x{11BA}\x{11C2}/\x{E0FC}/g;
  s/\x{1100}\x{1165}\x{11BA}\x{11BE}/\x{E0FB}/g;
  s/\x{1100}\x{1165}\x{11BA}\x{11A8}/\x{E0FA}/g;
  s/\x{1100}\x{1165}\x{11B8}\x{11BC}/\x{E0F9}/g;
  s/\x{1100}\x{1165}\x{11B8}\x{11AF}/\x{E0F8}/g;
  s/\x{1100}\x{1165}\x{11B7}\x{11BC}/\x{E0F7}/g;
  s/\x{1100}\x{1165}\x{11B7}\x{11BA}/\x{E0F6}/g;
  s/\x{1100}\x{1165}\x{11AF}\x{11F9}/\x{E0F5}/g;
  s/\x{1100}\x{1165}\x{11AF}\x{11AF}/\x{E0F4}/g;
  s/\x{1100}\x{1165}\x{11AB}\x{11BA}/\x{E0F3}/g;
  s/\x{1100}\x{1165}\x{1175}\x{11F0}/\x{E104}/g;
  s/\x{1100}\x{1165}\x{1173}\x{11B7}/\x{E103}/g;
  s/\x{1100}\x{1165}\x{1173}\x{11AF}/\x{E102}/g;
  s/\x{1100}\x{1165}\x{1173}\x{11AB}/\x{E101}/g;
  s/\x{1100}\x{1163}\x{11B8}\x{11BC}/\x{E0EF}/g;
  s/\x{1100}\x{1163}\x{11B7}\x{11BC}/\x{E0EE}/g;
  s/\x{1100}\x{1161}\x{11F0}\x{11BA}/\x{E0E7}/g;
  s/\x{1100}\x{1161}\x{11BC}\x{11BA}/\x{E0E5}/g;
  s/\x{1100}\x{1161}\x{11BA}\x{11C0}/\x{E0E3}/g;
  s/\x{1100}\x{1161}\x{11BA}\x{11BE}/\x{E0E2}/g;
  s/\x{1100}\x{1161}\x{11BA}\x{11B8}/\x{E0E1}/g;
  s/\x{1100}\x{1161}\x{11BA}\x{11A8}/\x{E0E0}/g;
  s/\x{1100}\x{1161}\x{11B8}\x{11C2}/\x{E0DE}/g;
  s/\x{1100}\x{1161}\x{11B8}\x{11C1}/\x{E0DD}/g;
  s/\x{1100}\x{1161}\x{11B8}\x{11BC}/\x{E0DF}/g;
  s/\x{1100}\x{1161}\x{11B7}\x{11C2}/\x{E0DB}/g;
  s/\x{1100}\x{1161}\x{11B7}\x{11BC}/\x{E0DC}/g;
  s/\x{1100}\x{1161}\x{11B7}\x{11BA}/\x{E0DA}/g;
  s/\x{1100}\x{1161}\x{11AF}\x{11F9}/\x{E0D9}/g;
  s/\x{1100}\x{1161}\x{11AF}\x{11F0}/\x{E0D8}/g;
  s/\x{1100}\x{1161}\x{11AF}\x{11AF}/\x{E0D7}/g;
  s/\x{1100}\x{1161}\x{11AF}\x{11AE}/\x{E0D6}/g;
  s/\x{1100}\x{1161}\x{11AF}\x{11AB}/\x{E0D5}/g;
  s/\x{1100}\x{1161}\x{11AE}\x{11A8}/\x{E0D4}/g;
  s/\x{1100}\x{1161}\x{11AB}\x{11BA}/\x{E0D3}/g;
  s/\x{1100}\x{1161}\x{11AB}\x{11AB}/\x{E0D2}/g;
  s/\x{1100}\x{1161}\x{1175}\x{11F0}/\x{E0ED}/g;
  s/\x{1100}\x{1100}\x{119E}\x{1175}/\x{E203}/g;
  s/\x{1100}\x{1100}\x{1175}\x{11F9}/\x{E200}/g;
  s/\x{1100}\x{1100}\x{1175}\x{11F0}/\x{E1FF}/g;
  s/\x{1100}\x{1100}\x{1175}\x{1161}/\x{E201}/g;
  s/\x{1100}\x{1100}\x{1173}\x{11F0}/\x{E1FB}/g;
  s/\x{1100}\x{1100}\x{1172}\x{11F0}/\x{E1F2}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1175}/\x{E1F8}/g;
  s/\x{1100}\x{1100}\x{1172}\x{1167}/\x{E1F3}/g;
  s/\x{1100}\x{1100}\x{116E}\x{11F0}/\x{E1EA}/g;
  s/\x{1100}\x{1100}\x{116E}\x{1167}/\x{E1EE}/g;
  s/\x{1100}\x{1100}\x{116E}\x{1161}/\x{E1EB}/g;
  s/\x{1100}\x{1100}\x{116D}\x{11F0}/\x{E1E5}/g;
  s/\x{1100}\x{1100}\x{116D}\x{1175}/\x{E1E6}/g;
  s/\x{1100}\x{1100}\x{1169}\x{11F0}/\x{E1E2}/g;
  s/\x{1100}\x{1100}\x{1167}\x{11F0}/\x{E1E0}/g;
  s/\x{1100}\x{1100}\x{1165}\x{116E}/\x{E1DF}/g;
  s/\x{1100}\x{1100}\x{1161}\x{11F0}/\x{E1D7}/g;
  s/\x{1100}\x{1100}\x{1161}\x{1169}/\x{E1D8}/g;
  s/\x{115F}\x{119E}\x{11AF}/\x{E0D1}/g;
  s/\x{115F}\x{1175}\x{11BC}/\x{E0D0}/g;
  s/\x{115F}\x{1175}\x{11BA}/\x{E0CF}/g;
  s/\x{115F}\x{1175}\x{11AF}/\x{E0CE}/g;
  s/\x{115F}\x{1175}\x{11AB}/\x{E0CD}/g;
  s/\x{115F}\x{1175}\x{11A8}/\x{E0CC}/g;
  s/\x{115F}\x{116E}\x{11AB}/\x{E0CB}/g;
  s/\x{115F}\x{1165}\x{11B8}/\x{E0CA}/g;
  s/\x{115F}\x{1165}\x{11AF}/\x{E0C9}/g;
  s/\x{115F}\x{1163}\x{11AB}/\x{E0C8}/g;
  s/\x{115F}\x{1161}\x{11BD}/\x{E0C4}/g;
  s/\x{115F}\x{1161}\x{11BA}/\x{E0C2}/g;
  s/\x{115F}\x{1161}\x{11AF}/\x{E0BE}/g;
  s/\x{115F}\x{1161}\x{11AE}/\x{E0BC}/g;
  s/\x{115F}\x{1161}\x{1173}/\x{E0C6}/g;
  s/\x{1159}\x{119E}\x{11B7}/\x{F667}/g;
  s/\x{1159}\x{119E}\x{11AB}/\x{F666}/g;
  s/\x{1159}\x{119E}\x{1175}/\x{F668}/g;
  s/\x{1159}\x{1175}\x{11F9}/\x{F664}/g;
  s/\x{1159}\x{1175}\x{11F0}/\x{F663}/g;
  s/\x{1159}\x{1175}\x{11BC}/\x{F662}/g;
  s/\x{1159}\x{1175}\x{11B7}/\x{F660}/g;
  s/\x{1159}\x{1175}\x{11AF}/\x{F65E}/g;
  s/\x{1159}\x{1175}\x{11AB}/\x{F65D}/g;
  s/\x{1159}\x{1173}\x{11F9}/\x{F658}/g;
  s/\x{1159}\x{1173}\x{11F0}/\x{F657}/g;
  s/\x{1159}\x{1173}\x{11B8}/\x{F656}/g;
  s/\x{1159}\x{1173}\x{11B7}/\x{F654}/g;
  s/\x{1159}\x{1173}\x{11AB}/\x{F653}/g;
  s/\x{1159}\x{1173}\x{11A8}/\x{F652}/g;
  s/\x{1159}\x{1173}\x{1175}/\x{F659}/g;
  s/\x{1159}\x{1172}\x{11F9}/\x{F645}/g;
  s/\x{1159}\x{1172}\x{11B7}/\x{F643}/g;
  s/\x{1159}\x{1172}\x{11AB}/\x{F642}/g;
  s/\x{1159}\x{1172}\x{1175}/\x{F64F}/g;
  s/\x{1159}\x{1172}\x{1167}/\x{F646}/g;
  s/\x{1159}\x{116E}\x{11F9}/\x{F639}/g;
  s/\x{1159}\x{116E}\x{11F0}/\x{F638}/g;
  s/\x{1159}\x{116E}\x{11BC}/\x{F637}/g;
  s/\x{1159}\x{116E}\x{11B7}/\x{F635}/g;
  s/\x{1159}\x{116E}\x{11AF}/\x{F633}/g;
  s/\x{1159}\x{116E}\x{11AB}/\x{F632}/g;
  s/\x{1159}\x{116E}\x{11A8}/\x{F631}/g;
  s/\x{1159}\x{116E}\x{1175}/\x{F63F}/g;
  s/\x{1159}\x{116E}\x{1165}/\x{F63A}/g;
  s/\x{1159}\x{116D}\x{11BC}/\x{F62F}/g;
  s/\x{1159}\x{116D}\x{11B7}/\x{F62D}/g;
  s/\x{1159}\x{116D}\x{11A8}/\x{F62C}/g;
  s/\x{1159}\x{1169}\x{11F0}/\x{F61A}/g;
  s/\x{1159}\x{1169}\x{11BC}/\x{F619}/g;
  s/\x{1159}\x{1169}\x{11B7}/\x{F617}/g;
  s/\x{1159}\x{1169}\x{11AB}/\x{F616}/g;
  s/\x{1159}\x{1169}\x{11A8}/\x{F615}/g;
  s/\x{1159}\x{1169}\x{1175}/\x{F627}/g;
  s/\x{1159}\x{1169}\x{1161}/\x{F61B}/g;
  s/\x{1159}\x{1167}\x{11F0}/\x{F610}/g;
  s/\x{1159}\x{1167}\x{11C2}/\x{F611}/g;
  s/\x{1159}\x{1167}\x{11B8}/\x{F60F}/g;
  s/\x{1159}\x{1167}\x{11B7}/\x{F60E}/g;
  s/\x{1159}\x{1167}\x{11AF}/\x{F60C}/g;
  s/\x{1159}\x{1167}\x{11AB}/\x{F60B}/g;
  s/\x{1159}\x{1167}\x{11A8}/\x{F60A}/g;
  s/\x{1159}\x{1167}\x{1175}/\x{F612}/g;
  s/\x{1159}\x{1165}\x{11BC}/\x{F606}/g;
  s/\x{1159}\x{1165}\x{11B8}/\x{F605}/g;
  s/\x{1159}\x{1165}\x{11B7}/\x{F604}/g;
  s/\x{1159}\x{1165}\x{11AF}/\x{F602}/g;
  s/\x{1159}\x{1165}\x{11AB}/\x{F601}/g;
  s/\x{1159}\x{1165}\x{1175}/\x{F607}/g;
  s/\x{1159}\x{1163}\x{11F9}/\x{F5FE}/g;
  s/\x{1159}\x{1163}\x{11F0}/\x{F5FD}/g;
  s/\x{1159}\x{1163}\x{11B8}/\x{F5FB}/g;
  s/\x{1159}\x{1163}\x{11B7}/\x{F5F9}/g;
  s/\x{1159}\x{1163}\x{11AB}/\x{F5F8}/g;
  s/\x{1159}\x{1163}\x{11A8}/\x{F5F7}/g;
  s/\x{1159}\x{1163}\x{1175}/\x{F5FF}/g;
  s/\x{1159}\x{1161}\x{11F0}/\x{F5F2}/g;
  s/\x{1159}\x{1161}\x{11BC}/\x{F5F1}/g;
  s/\x{1159}\x{1161}\x{11B8}/\x{F5EF}/g;
  s/\x{1159}\x{1161}\x{11B7}/\x{F5ED}/g;
  s/\x{1159}\x{1161}\x{11AF}/\x{F5EB}/g;
  s/\x{1159}\x{1161}\x{11AB}/\x{F5EA}/g;
  s/\x{1159}\x{1161}\x{11A8}/\x{F5E9}/g;
  s/\x{1159}\x{1161}\x{1175}/\x{F5F3}/g;
  s/\x{1155}\x{1175}\x{11F9}/\x{F399}/g;
  s/\x{1155}\x{1175}\x{11EB}/\x{F398}/g;
  s/\x{1155}\x{1175}\x{11B7}/\x{F396}/g;
  s/\x{1155}\x{1175}\x{11AB}/\x{F395}/g;
  s/\x{1155}\x{1175}\x{116E}/\x{F39A}/g;
  s/\x{1155}\x{1173}\x{11EB}/\x{F393}/g;
  s/\x{1155}\x{1173}\x{11AB}/\x{F392}/g;
  s/\x{1155}\x{1172}\x{11F0}/\x{F38E}/g;
  s/\x{1155}\x{1172}\x{11BC}/\x{F38D}/g;
  s/\x{1155}\x{1172}\x{11AB}/\x{F38C}/g;
  s/\x{1155}\x{1172}\x{11A8}/\x{F38B}/g;
  s/\x{1155}\x{1172}\x{1167}/\x{F38F}/g;
  s/\x{1155}\x{116E}\x{1175}/\x{F389}/g;
  s/\x{1155}\x{1169}\x{1161}/\x{F385}/g;
  s/\x{1155}\x{1167}\x{11AB}/\x{F383}/g;
  s/\x{1155}\x{1163}\x{11F0}/\x{F381}/g;
  s/\x{1155}\x{1161}\x{11F9}/\x{F37E}/g;
  s/\x{1155}\x{1161}\x{11B7}/\x{F37C}/g;
  s/\x{1155}\x{1161}\x{11AB}/\x{F37B}/g;
  s/\x{1155}\x{1161}\x{1175}/\x{F37F}/g;
  s/\x{1154}\x{1175}\x{11F9}/\x{F379}/g;
  s/\x{1154}\x{1175}\x{11F0}/\x{F378}/g;
  s/\x{1154}\x{1175}\x{11EB}/\x{F377}/g;
  s/\x{1154}\x{1175}\x{11AB}/\x{F376}/g;
  s/\x{1154}\x{1172}\x{11AB}/\x{F371}/g;
  s/\x{1154}\x{1172}\x{1167}/\x{F372}/g;
  s/\x{1154}\x{116E}\x{11BC}/\x{F36E}/g;
  s/\x{1154}\x{116E}\x{11AB}/\x{F36D}/g;
  s/\x{1154}\x{116E}\x{1175}/\x{F36F}/g;
  s/\x{1154}\x{116D}\x{1163}/\x{F36B}/g;
  s/\x{1154}\x{116D}\x{1161}/\x{F36A}/g;
  s/\x{1154}\x{1167}\x{11AB}/\x{F366}/g;
  s/\x{1154}\x{1167}\x{1175}/\x{F367}/g;
  s/\x{1154}\x{1165}\x{11AB}/\x{F364}/g;
  s/\x{1154}\x{1161}\x{11AB}/\x{F360}/g;
  s/\x{1154}\x{1161}\x{1175}/\x{F362}/g;
  s/\x{1154}\x{1161}\x{1169}/\x{F361}/g;
  s/\x{1150}\x{1175}\x{11F0}/\x{F2C5}/g;
  s/\x{1150}\x{1175}\x{11EB}/\x{F2C4}/g;
  s/\x{1150}\x{1175}\x{11AB}/\x{F2C3}/g;
  s/\x{1150}\x{1173}\x{11EB}/\x{F2C1}/g;
  s/\x{1150}\x{1173}\x{11B7}/\x{F2C0}/g;
  s/\x{1150}\x{1172}\x{11F0}/\x{F2BC}/g;
  s/\x{1150}\x{1172}\x{11AB}/\x{F2BB}/g;
  s/\x{1150}\x{1172}\x{11A8}/\x{F2BA}/g;
  s/\x{1150}\x{1172}\x{1167}/\x{F2BD}/g;
  s/\x{1150}\x{116E}\x{11F9}/\x{F2B8}/g;
  s/\x{1150}\x{116E}\x{11BC}/\x{F2B7}/g;
  s/\x{1150}\x{1169}\x{1161}/\x{F2B4}/g;
  s/\x{1150}\x{1167}\x{11AB}/\x{F2B2}/g;
  s/\x{1150}\x{1163}\x{11F0}/\x{F2B0}/g;
  s/\x{1150}\x{1163}\x{11B7}/\x{F2AE}/g;
  s/\x{1150}\x{1161}\x{11B8}/\x{F2AB}/g;
  s/\x{1150}\x{1161}\x{11B7}/\x{F2A9}/g;
  s/\x{1150}\x{1161}\x{11AE}/\x{F2A8}/g;
  s/\x{1150}\x{1161}\x{11AB}/\x{F2A7}/g;
  s/\x{1150}\x{1161}\x{1175}/\x{F2AC}/g;
  s/\x{1150}\x{1150}\x{1175}/\x{F2DA}/g;
  s/\x{1150}\x{1150}\x{1173}/\x{F2D7}/g;
  s/\x{1150}\x{1150}\x{1172}/\x{F2D3}/g;
  s/\x{1150}\x{1150}\x{116E}/\x{F2D1}/g;
  s/\x{1150}\x{1150}\x{1167}/\x{F2CF}/g;
  s/\x{1150}\x{1150}\x{1163}/\x{F2CA}/g;
  s/\x{1150}\x{1150}\x{1161}/\x{F2C6}/g;
  s/\x{114E}\x{1175}\x{11EB}/\x{F28A}/g;
  s/\x{114E}\x{1175}\x{11B7}/\x{F288}/g;
  s/\x{114E}\x{1175}\x{11AB}/\x{F287}/g;
  s/\x{114E}\x{1173}\x{11EB}/\x{F285}/g;
  s/\x{114E}\x{1173}\x{11BA}/\x{F284}/g;
  s/\x{114E}\x{1173}\x{11AB}/\x{F283}/g;
  s/\x{114E}\x{1172}\x{11BC}/\x{F27F}/g;
  s/\x{114E}\x{1172}\x{11AB}/\x{F27E}/g;
  s/\x{114E}\x{1172}\x{1167}/\x{F280}/g;
  s/\x{114E}\x{116E}\x{11BC}/\x{F27B}/g;
  s/\x{114E}\x{116E}\x{11AB}/\x{F27A}/g;
  s/\x{114E}\x{116E}\x{1175}/\x{F27C}/g;
  s/\x{114E}\x{116D}\x{1163}/\x{F278}/g;
  s/\x{114E}\x{1169}\x{1161}/\x{F276}/g;
  s/\x{114E}\x{1167}\x{11F9}/\x{F273}/g;
  s/\x{114E}\x{1167}\x{11AB}/\x{F272}/g;
  s/\x{114E}\x{1167}\x{1175}/\x{F274}/g;
  s/\x{114E}\x{1165}\x{11AB}/\x{F270}/g;
  s/\x{114E}\x{1163}\x{11BC}/\x{F26E}/g;
  s/\x{114E}\x{1161}\x{11B7}/\x{F26A}/g;
  s/\x{114E}\x{1161}\x{11AB}/\x{F269}/g;
  s/\x{114E}\x{1161}\x{1175}/\x{F26C}/g;
  s/\x{114E}\x{114E}\x{119E}/\x{F2A4}/g;
  s/\x{114E}\x{114E}\x{1175}/\x{F2A2}/g;
  s/\x{114E}\x{114E}\x{1173}/\x{F29F}/g;
  s/\x{114E}\x{114E}\x{1172}/\x{F29B}/g;
  s/\x{114E}\x{114E}\x{116E}/\x{F299}/g;
  s/\x{114E}\x{114E}\x{1167}/\x{F296}/g;
  s/\x{114E}\x{114E}\x{1165}/\x{F294}/g;
  s/\x{114E}\x{114E}\x{1163}/\x{F291}/g;
  s/\x{114E}\x{114E}\x{1161}/\x{F28B}/g;
  s/\x{114C}\x{119E}\x{11BA}/\x{F161}/g;
  s/\x{114C}\x{119E}\x{11AF}/\x{F160}/g;
  s/\x{114C}\x{119E}\x{11AE}/\x{F15F}/g;
  s/\x{114C}\x{119E}\x{11AB}/\x{F15E}/g;
  s/\x{114C}\x{119E}\x{11A8}/\x{F15D}/g;
  s/\x{114C}\x{119E}\x{1175}/\x{F162}/g;
  s/\x{114C}\x{1175}\x{11F0}/\x{F15B}/g;
  s/\x{114C}\x{1175}\x{11EB}/\x{F159}/g;
  s/\x{114C}\x{1175}\x{11BC}/\x{F15A}/g;
  s/\x{114C}\x{1175}\x{11BA}/\x{F158}/g;
  s/\x{114C}\x{1175}\x{11AF}/\x{F157}/g;
  s/\x{114C}\x{1175}\x{11AE}/\x{F156}/g;
  s/\x{114C}\x{1175}\x{11AB}/\x{F155}/g;
  s/\x{114C}\x{1173}\x{11F0}/\x{F151}/g;
  s/\x{114C}\x{1173}\x{11BC}/\x{F150}/g;
  s/\x{114C}\x{1173}\x{11B8}/\x{F14F}/g;
  s/\x{114C}\x{1173}\x{11B7}/\x{F14E}/g;
  s/\x{114C}\x{1173}\x{11AF}/\x{F14C}/g;
  s/\x{114C}\x{1173}\x{11AB}/\x{F14B}/g;
  s/\x{114C}\x{1173}\x{11A8}/\x{F14A}/g;
  s/\x{114C}\x{1173}\x{1175}/\x{F152}/g;
  s/\x{114C}\x{1172}\x{11B7}/\x{F141}/g;
  s/\x{114C}\x{1172}\x{11AF}/\x{F13F}/g;
  s/\x{114C}\x{1172}\x{11AB}/\x{F13E}/g;
  s/\x{114C}\x{1172}\x{11A8}/\x{F13D}/g;
  s/\x{114C}\x{1172}\x{1175}/\x{F148}/g;
  s/\x{114C}\x{1172}\x{1167}/\x{F143}/g;
  s/\x{114C}\x{116E}\x{11BC}/\x{F12F}/g;
  s/\x{114C}\x{116E}\x{11B7}/\x{F12D}/g;
  s/\x{114C}\x{116E}\x{11AF}/\x{F12B}/g;
  s/\x{114C}\x{116E}\x{11AB}/\x{F12A}/g;
  s/\x{114C}\x{116E}\x{11A8}/\x{F129}/g;
  s/\x{114C}\x{116E}\x{1175}/\x{F137}/g;
  s/\x{114C}\x{116E}\x{1165}/\x{F130}/g;
  s/\x{114C}\x{116D}\x{11F0}/\x{F124}/g;
  s/\x{114C}\x{116D}\x{11B7}/\x{F122}/g;
  s/\x{114C}\x{116D}\x{1163}/\x{F125}/g;
  s/\x{114C}\x{1169}\x{11F0}/\x{F111}/g;
  s/\x{114C}\x{1169}\x{11BC}/\x{F110}/g;
  s/\x{114C}\x{1169}\x{11BA}/\x{F10F}/g;
  s/\x{114C}\x{1169}\x{11B7}/\x{F10D}/g;
  s/\x{114C}\x{1169}\x{11AF}/\x{F10A}/g;
  s/\x{114C}\x{1169}\x{11AB}/\x{F109}/g;
  s/\x{114C}\x{1169}\x{11A8}/\x{F108}/g;
  s/\x{114C}\x{1169}\x{1175}/\x{F11D}/g;
  s/\x{114C}\x{1169}\x{1161}/\x{F112}/g;
  s/\x{114C}\x{1167}\x{11F0}/\x{F101}/g;
  s/\x{114C}\x{1167}\x{11C1}/\x{F102}/g;
  s/\x{114C}\x{1167}\x{11BC}/\x{F100}/g;
  s/\x{114C}\x{1167}\x{11BA}/\x{EFFF}/g;
  s/\x{114C}\x{1167}\x{11B8}/\x{EFFE}/g;
  s/\x{114C}\x{1167}\x{11B7}/\x{EFFC}/g;
  s/\x{114C}\x{1167}\x{11AF}/\x{EFFB}/g;
  s/\x{114C}\x{1167}\x{11AB}/\x{EFFA}/g;
  s/\x{114C}\x{1167}\x{11A8}/\x{EFF9}/g;
  s/\x{114C}\x{1167}\x{1175}/\x{F104}/g;
  s/\x{114C}\x{1165}\x{11F0}/\x{EFF1}/g;
  s/\x{114C}\x{1165}\x{11EB}/\x{EFEF}/g;
  s/\x{114C}\x{1165}\x{11BC}/\x{EFF0}/g;
  s/\x{114C}\x{1165}\x{11BA}/\x{EFEE}/g;
  s/\x{114C}\x{1165}\x{11B8}/\x{EFED}/g;
  s/\x{114C}\x{1165}\x{11B7}/\x{EFEC}/g;
  s/\x{114C}\x{1165}\x{11AF}/\x{EFEA}/g;
  s/\x{114C}\x{1165}\x{11AB}/\x{EFE9}/g;
  s/\x{114C}\x{1165}\x{11A8}/\x{EFE8}/g;
  s/\x{114C}\x{1165}\x{1175}/\x{EFF2}/g;
  s/\x{114C}\x{1163}\x{11F0}/\x{EFE4}/g;
  s/\x{114C}\x{1163}\x{11BC}/\x{EFE3}/g;
  s/\x{114C}\x{1163}\x{11AB}/\x{EFE2}/g;
  s/\x{114C}\x{1163}\x{11A8}/\x{EFE1}/g;
  s/\x{114C}\x{1163}\x{1175}/\x{EFE5}/g;
  s/\x{114C}\x{1161}\x{11F9}/\x{EFDC}/g;
  s/\x{114C}\x{1161}\x{11F0}/\x{EFDB}/g;
  s/\x{114C}\x{1161}\x{11EB}/\x{EFD9}/g;
  s/\x{114C}\x{1161}\x{11BC}/\x{EFDA}/g;
  s/\x{114C}\x{1161}\x{11B8}/\x{EFD8}/g;
  s/\x{114C}\x{1161}\x{11B7}/\x{EFD7}/g;
  s/\x{114C}\x{1161}\x{11AF}/\x{EFD5}/g;
  s/\x{114C}\x{1161}\x{11AB}/\x{EFD4}/g;
  s/\x{114C}\x{1161}\x{11A8}/\x{EFD3}/g;
  s/\x{114C}\x{1161}\x{1175}/\x{EFDD}/g;
  s/\x{1140}\x{119E}\x{11EB}/\x{ED9D}/g;
  s/\x{1140}\x{119E}\x{11BD}/\x{ED9F}/g;
  s/\x{1140}\x{119E}\x{11BC}/\x{ED9E}/g;
  s/\x{1140}\x{119E}\x{11BA}/\x{ED9C}/g;
  s/\x{1140}\x{119E}\x{11B8}/\x{ED9A}/g;
  s/\x{1140}\x{119E}\x{11B7}/\x{ED98}/g;
  s/\x{1140}\x{119E}\x{11AF}/\x{ED93}/g;
  s/\x{1140}\x{119E}\x{11AE}/\x{ED92}/g;
  s/\x{1140}\x{119E}\x{11AB}/\x{ED91}/g;
  s/\x{1140}\x{119E}\x{11A8}/\x{ED90}/g;
  s/\x{1140}\x{119E}\x{1175}/\x{EDA0}/g;
  s/\x{1140}\x{1175}\x{11F9}/\x{ED8D}/g;
  s/\x{1140}\x{1175}\x{11F0}/\x{ED8C}/g;
  s/\x{1140}\x{1175}\x{11EB}/\x{ED8A}/g;
  s/\x{1140}\x{1175}\x{11BC}/\x{ED8B}/g;
  s/\x{1140}\x{1175}\x{11BA}/\x{ED89}/g;
  s/\x{1140}\x{1175}\x{11B8}/\x{ED87}/g;
  s/\x{1140}\x{1175}\x{11B7}/\x{ED86}/g;
  s/\x{1140}\x{1175}\x{11AF}/\x{ED83}/g;
  s/\x{1140}\x{1175}\x{11AE}/\x{ED82}/g;
  s/\x{1140}\x{1175}\x{11AB}/\x{ED81}/g;
  s/\x{1140}\x{1175}\x{11A8}/\x{ED80}/g;
  s/\x{1140}\x{1175}\x{1161}/\x{ED8E}/g;
  s/\x{1140}\x{1173}\x{11F0}/\x{ED7A}/g;
  s/\x{1140}\x{1173}\x{11EB}/\x{ED79}/g;
  s/\x{1140}\x{1173}\x{11BE}/\x{ED7B}/g;
  s/\x{1140}\x{1173}\x{11BA}/\x{ED77}/g;
  s/\x{1140}\x{1173}\x{11B8}/\x{ED76}/g;
  s/\x{1140}\x{1173}\x{11B7}/\x{ED73}/g;
  s/\x{1140}\x{1173}\x{11AF}/\x{ED6E}/g;
  s/\x{1140}\x{1173}\x{11AB}/\x{ED6D}/g;
  s/\x{1140}\x{1173}\x{11A8}/\x{ED6C}/g;
  s/\x{1140}\x{1173}\x{1175}/\x{ED7C}/g;
  s/\x{1140}\x{1172}\x{11F9}/\x{ED60}/g;
  s/\x{1140}\x{1172}\x{11F0}/\x{ED5D}/g;
  s/\x{1140}\x{1172}\x{11BE}/\x{ED5F}/g;
  s/\x{1140}\x{1172}\x{11BD}/\x{ED5E}/g;
  s/\x{1140}\x{1172}\x{11BC}/\x{ED5C}/g;
  s/\x{1140}\x{1172}\x{11BA}/\x{ED5B}/g;
  s/\x{1140}\x{1172}\x{11B7}/\x{ED59}/g;
  s/\x{1140}\x{1172}\x{11AB}/\x{ED58}/g;
  s/\x{1140}\x{1172}\x{11A8}/\x{ED57}/g;
  s/\x{1140}\x{1172}\x{1175}/\x{ED69}/g;
  s/\x{1140}\x{1172}\x{1167}/\x{ED63}/g;
  s/\x{1140}\x{1172}\x{1165}/\x{ED61}/g;
  s/\x{1140}\x{116E}\x{11F0}/\x{ED4B}/g;
  s/\x{1140}\x{116E}\x{11C2}/\x{ED4D}/g;
  s/\x{1140}\x{116E}\x{11BE}/\x{ED4C}/g;
  s/\x{1140}\x{116E}\x{11BC}/\x{ED4A}/g;
  s/\x{1140}\x{116E}\x{11BA}/\x{ED49}/g;
  s/\x{1140}\x{116E}\x{11B8}/\x{ED48}/g;
  s/\x{1140}\x{116E}\x{11B7}/\x{ED46}/g;
  s/\x{1140}\x{116E}\x{11AF}/\x{ED44}/g;
  s/\x{1140}\x{116E}\x{11AB}/\x{ED43}/g;
  s/\x{1140}\x{116E}\x{11A8}/\x{ED42}/g;
  s/\x{1140}\x{116E}\x{1175}/\x{ED54}/g;
  s/\x{1140}\x{116E}\x{1167}/\x{ED52}/g;
  s/\x{1140}\x{116E}\x{1165}/\x{ED50}/g;
  s/\x{1140}\x{116E}\x{1161}/\x{ED4E}/g;
  s/\x{1140}\x{116D}\x{11F0}/\x{ED3F}/g;
  s/\x{1140}\x{116D}\x{11C2}/\x{ED40}/g;
  s/\x{1140}\x{116D}\x{11BC}/\x{ED3E}/g;
  s/\x{1140}\x{116D}\x{11BA}/\x{ED3D}/g;
  s/\x{1140}\x{116D}\x{11B7}/\x{ED3B}/g;
  s/\x{1140}\x{116D}\x{11A8}/\x{ED3A}/g;
  s/\x{1140}\x{1169}\x{11F0}/\x{ED35}/g;
  s/\x{1140}\x{1169}\x{11EB}/\x{ED33}/g;
  s/\x{1140}\x{1169}\x{11BC}/\x{ED34}/g;
  s/\x{1140}\x{1169}\x{11B8}/\x{ED32}/g;
  s/\x{1140}\x{1169}\x{11B7}/\x{ED31}/g;
  s/\x{1140}\x{1169}\x{11AF}/\x{ED2F}/g;
  s/\x{1140}\x{1169}\x{11AB}/\x{ED2E}/g;
  s/\x{1140}\x{1169}\x{1175}/\x{ED38}/g;
  s/\x{1140}\x{1169}\x{1161}/\x{ED36}/g;
  s/\x{1140}\x{1167}\x{11F9}/\x{ED26}/g;
  s/\x{1140}\x{1167}\x{11F0}/\x{ED25}/g;
  s/\x{1140}\x{1167}\x{11BC}/\x{ED24}/g;
  s/\x{1140}\x{1167}\x{11B8}/\x{ED23}/g;
  s/\x{1140}\x{1167}\x{11B7}/\x{ED22}/g;
  s/\x{1140}\x{1167}\x{11AF}/\x{ED1F}/g;
  s/\x{1140}\x{1167}\x{11AB}/\x{ED1E}/g;
  s/\x{1140}\x{1167}\x{11A8}/\x{ED1D}/g;
  s/\x{1140}\x{1167}\x{1175}/\x{ED27}/g;
  s/\x{1140}\x{1165}\x{11F0}/\x{ED16}/g;
  s/\x{1140}\x{1165}\x{11EB}/\x{ED14}/g;
  s/\x{1140}\x{1165}\x{11C2}/\x{ED17}/g;
  s/\x{1140}\x{1165}\x{11BC}/\x{ED15}/g;
  s/\x{1140}\x{1165}\x{11BA}/\x{ED13}/g;
  s/\x{1140}\x{1165}\x{11B8}/\x{ED12}/g;
  s/\x{1140}\x{1165}\x{11B7}/\x{ED10}/g;
  s/\x{1140}\x{1165}\x{11AF}/\x{ED0F}/g;
  s/\x{1140}\x{1165}\x{11AB}/\x{ED0E}/g;
  s/\x{1140}\x{1165}\x{11A8}/\x{ED0D}/g;
  s/\x{1140}\x{1165}\x{1175}/\x{ED19}/g;
  s/\x{1140}\x{1165}\x{116E}/\x{ED18}/g;
  s/\x{1140}\x{1163}\x{11F9}/\x{ED0A}/g;
  s/\x{1140}\x{1163}\x{11F0}/\x{ED09}/g;
  s/\x{1140}\x{1163}\x{11BC}/\x{ED08}/g;
  s/\x{1140}\x{1163}\x{11B8}/\x{ED06}/g;
  s/\x{1140}\x{1163}\x{11B7}/\x{ED04}/g;
  s/\x{1140}\x{1163}\x{11AB}/\x{ED03}/g;
  s/\x{1140}\x{1163}\x{11A8}/\x{ED02}/g;
  s/\x{1140}\x{1163}\x{1169}/\x{ED0B}/g;
  s/\x{1140}\x{1161}\x{11F0}/\x{ECFA}/g;
  s/\x{1140}\x{1161}\x{11BC}/\x{ECF9}/g;
  s/\x{1140}\x{1161}\x{11BA}/\x{ECF8}/g;
  s/\x{1140}\x{1161}\x{11B8}/\x{ECF7}/g;
  s/\x{1140}\x{1161}\x{11B7}/\x{ECF5}/g;
  s/\x{1140}\x{1161}\x{11AF}/\x{ECF4}/g;
  s/\x{1140}\x{1161}\x{11AE}/\x{ECF3}/g;
  s/\x{1140}\x{1161}\x{11AB}/\x{ECF2}/g;
  s/\x{1140}\x{1161}\x{11A8}/\x{ECF1}/g;
  s/\x{1140}\x{1161}\x{1175}/\x{ECFB}/g;
  s/\x{113F}\x{1175}\x{11F9}/\x{ECEF}/g;
  s/\x{113F}\x{1175}\x{11AB}/\x{ECEE}/g;
  s/\x{113F}\x{1173}\x{11EB}/\x{ECEC}/g;
  s/\x{113F}\x{1172}\x{11F0}/\x{ECEA}/g;
  s/\x{113F}\x{1172}\x{11AB}/\x{ECE9}/g;
  s/\x{113F}\x{1172}\x{11A8}/\x{ECE8}/g;
  s/\x{113F}\x{116E}\x{1175}/\x{ECE6}/g;
  s/\x{113F}\x{1167}\x{11AB}/\x{ECE4}/g;
  s/\x{113F}\x{1163}\x{11F0}/\x{ECE2}/g;
  s/\x{113E}\x{1175}\x{11B7}/\x{ECDF}/g;
  s/\x{113E}\x{1175}\x{11AB}/\x{ECDE}/g;
  s/\x{113E}\x{1173}\x{11EB}/\x{ECDB}/g;
  s/\x{113E}\x{1173}\x{11BC}/\x{ECDC}/g;
  s/\x{113E}\x{1172}\x{11AB}/\x{ECD7}/g;
  s/\x{113E}\x{1172}\x{1167}/\x{ECD8}/g;
  s/\x{113E}\x{116E}\x{11BC}/\x{ECD4}/g;
  s/\x{113E}\x{116E}\x{11A8}/\x{ECD3}/g;
  s/\x{113E}\x{116E}\x{1175}/\x{ECD5}/g;
  s/\x{113E}\x{1169}\x{1161}/\x{ECD0}/g;
  s/\x{113E}\x{1167}\x{11AB}/\x{ECCE}/g;
  s/\x{113E}\x{1163}\x{11B7}/\x{ECCB}/g;
  s/\x{113E}\x{1161}\x{11AE}/\x{ECC8}/g;
  s/\x{113E}\x{1161}\x{11AB}/\x{ECC7}/g;
  s/\x{113E}\x{1161}\x{1175}/\x{ECC9}/g;
  s/\x{113D}\x{1175}\x{11F9}/\x{ECC5}/g;
  s/\x{113D}\x{1175}\x{11EB}/\x{ECC4}/g;
  s/\x{113D}\x{1175}\x{11AB}/\x{ECC3}/g;
  s/\x{113D}\x{1173}\x{11EB}/\x{ECC1}/g;
  s/\x{113D}\x{1172}\x{11F0}/\x{ECBC}/g;
  s/\x{113D}\x{1172}\x{11AB}/\x{ECBB}/g;
  s/\x{113D}\x{1172}\x{1175}/\x{ECBF}/g;
  s/\x{113D}\x{1172}\x{1167}/\x{ECBD}/g;
  s/\x{113D}\x{116E}\x{1175}/\x{ECB9}/g;
  s/\x{113D}\x{1167}\x{11AB}/\x{ECB7}/g;
  s/\x{113D}\x{1163}\x{11F0}/\x{ECB5}/g;
  s/\x{113D}\x{1163}\x{11BC}/\x{ECB4}/g;
  s/\x{113C}\x{1175}\x{11BC}/\x{ECB2}/g;
  s/\x{113C}\x{1173}\x{11EB}/\x{ECB0}/g;
  s/\x{113C}\x{1173}\x{11AB}/\x{ECAF}/g;
  s/\x{113C}\x{1172}\x{11AB}/\x{ECAB}/g;
  s/\x{113C}\x{1172}\x{1167}/\x{ECAC}/g;
  s/\x{113C}\x{116E}\x{11BC}/\x{ECA8}/g;
  s/\x{113C}\x{116E}\x{11AB}/\x{ECA7}/g;
  s/\x{113C}\x{116E}\x{1175}/\x{ECA9}/g;
  s/\x{113C}\x{1167}\x{11AB}/\x{ECA4}/g;
  s/\x{113C}\x{1167}\x{1175}/\x{ECA5}/g;
  s/\x{113C}\x{1165}\x{11AB}/\x{ECA2}/g;
  s/\x{113C}\x{1163}\x{11F0}/\x{ECA0}/g;
  s/\x{113C}\x{1163}\x{11B7}/\x{EC9E}/g;
  s/\x{113C}\x{1161}\x{11F0}/\x{EC9B}/g;
  s/\x{113C}\x{1161}\x{11AB}/\x{EC9A}/g;
  s/\x{113C}\x{1161}\x{1175}/\x{EC9C}/g;
  s/\x{1112}\x{119E}\x{11F9}/\x{F54D}/g;
  s/\x{1112}\x{119E}\x{11F0}/\x{F54A}/g;
  s/\x{1112}\x{119E}\x{11C2}/\x{F54C}/g;
  s/\x{1112}\x{119E}\x{11C0}/\x{F54B}/g;
  s/\x{1112}\x{119E}\x{11BC}/\x{F549}/g;
  s/\x{1112}\x{119E}\x{11BA}/\x{F548}/g;
  s/\x{1112}\x{119E}\x{11B8}/\x{F546}/g;
  s/\x{1112}\x{119E}\x{11B7}/\x{F545}/g;
  s/\x{1112}\x{119E}\x{11AF}/\x{F53C}/g;
  s/\x{1112}\x{119E}\x{11AE}/\x{F53B}/g;
  s/\x{1112}\x{119E}\x{11AB}/\x{F53A}/g;
  s/\x{1112}\x{119E}\x{11A8}/\x{F538}/g;
  s/\x{1112}\x{119E}\x{1175}/\x{F550}/g;
  s/\x{1112}\x{119E}\x{1165}/\x{F54E}/g;
  s/\x{1112}\x{1175}\x{11F0}/\x{F52B}/g;
  s/\x{1112}\x{1175}\x{1172}/\x{F535}/g;
  s/\x{1112}\x{1175}\x{116E}/\x{F534}/g;
  s/\x{1112}\x{1175}\x{116D}/\x{F533}/g;
  s/\x{1112}\x{1175}\x{1167}/\x{F531}/g;
  s/\x{1112}\x{1175}\x{1163}/\x{F52E}/g;
  s/\x{1112}\x{1175}\x{1161}/\x{F52D}/g;
  s/\x{1112}\x{1173}\x{11F0}/\x{F51B}/g;
  s/\x{1112}\x{1173}\x{1173}/\x{F51F}/g;
  s/\x{1112}\x{1173}\x{116E}/\x{F51E}/g;
  s/\x{1112}\x{1173}\x{1169}/\x{F51D}/g;
  s/\x{1112}\x{1173}\x{1161}/\x{F51C}/g;
  s/\x{1112}\x{1172}\x{11F0}/\x{F506}/g;
  s/\x{1112}\x{1172}\x{1175}/\x{F514}/g;
  s/\x{1112}\x{1172}\x{1167}/\x{F509}/g;
  s/\x{1112}\x{1172}\x{1161}/\x{F507}/g;
  s/\x{1112}\x{116E}\x{11F9}/\x{F4F5}/g;
  s/\x{1112}\x{116E}\x{11F0}/\x{F4F4}/g;
  s/\x{1112}\x{116E}\x{11EB}/\x{F4F3}/g;
  s/\x{1112}\x{116E}\x{1167}/\x{F500}/g;
  s/\x{1112}\x{116E}\x{1161}/\x{F4F6}/g;
  s/\x{1112}\x{116D}\x{1175}/\x{F4EE}/g;
  s/\x{1112}\x{116D}\x{1169}/\x{F4ED}/g;
  s/\x{1112}\x{116D}\x{1163}/\x{F4EB}/g;
  s/\x{1112}\x{116D}\x{1161}/\x{F4E9}/g;
  s/\x{1112}\x{1169}\x{11F0}/\x{F4D9}/g;
  s/\x{1112}\x{1169}\x{11EB}/\x{F4D8}/g;
  s/\x{1112}\x{1169}\x{1169}/\x{F4E4}/g;
  s/\x{1112}\x{1169}\x{1165}/\x{F4E0}/g;
  s/\x{1112}\x{1169}\x{1163}/\x{F4DF}/g;
  s/\x{1112}\x{1167}\x{11F9}/\x{F4D0}/g;
  s/\x{1112}\x{1167}\x{11F0}/\x{F4CF}/g;
  s/\x{1112}\x{1165}\x{11F9}/\x{F4C7}/g;
  s/\x{1112}\x{1165}\x{11F0}/\x{F4C5}/g;
  s/\x{1112}\x{1165}\x{1173}/\x{F4CA}/g;
  s/\x{1112}\x{1165}\x{116E}/\x{F4C9}/g;
  s/\x{1112}\x{1165}\x{1169}/\x{F4C8}/g;
  s/\x{1112}\x{1163}\x{11F9}/\x{F4C0}/g;
  s/\x{1112}\x{1163}\x{11F0}/\x{F4BF}/g;
  s/\x{1112}\x{1163}\x{1169}/\x{F4C1}/g;
  s/\x{1112}\x{1161}\x{11F0}/\x{F4B6}/g;
  s/\x{1112}\x{1161}\x{116E}/\x{F4B9}/g;
  s/\x{1112}\x{1161}\x{1169}/\x{F4B8}/g;
  s/\x{1112}\x{1112}\x{119E}/\x{F5E2}/g;
  s/\x{1112}\x{1112}\x{1175}/\x{F5DC}/g;
  s/\x{1112}\x{1112}\x{1173}/\x{F5CF}/g;
  s/\x{1112}\x{1112}\x{1172}/\x{F5C3}/g;
  s/\x{1112}\x{1112}\x{116E}/\x{F5B5}/g;
  s/\x{1112}\x{1112}\x{116D}/\x{F5B2}/g;
  s/\x{1112}\x{1112}\x{1169}/\x{F59B}/g;
  s/\x{1112}\x{1112}\x{1167}/\x{F590}/g;
  s/\x{1112}\x{1112}\x{1165}/\x{F588}/g;
  s/\x{1112}\x{1112}\x{1163}/\x{F57F}/g;
  s/\x{1112}\x{1112}\x{1161}/\x{F571}/g;
  s/\x{1112}\x{1109}\x{1175}/\x{F56E}/g;
  s/\x{1112}\x{1109}\x{1172}/\x{F566}/g;
  s/\x{1112}\x{1109}\x{116E}/\x{F562}/g;
  s/\x{1112}\x{1109}\x{1169}/\x{F55F}/g;
  s/\x{1112}\x{1109}\x{1163}/\x{F55D}/g;
  s/\x{1111}\x{119E}\x{11C0}/\x{F49D}/g;
  s/\x{1111}\x{119E}\x{11BE}/\x{F49C}/g;
  s/\x{1111}\x{119E}\x{11BD}/\x{F49B}/g;
  s/\x{1111}\x{119E}\x{11BC}/\x{F49A}/g;
  s/\x{1111}\x{119E}\x{11BA}/\x{F498}/g;
  s/\x{1111}\x{119E}\x{11B8}/\x{F497}/g;
  s/\x{1111}\x{119E}\x{11B7}/\x{F496}/g;
  s/\x{1111}\x{119E}\x{11AF}/\x{F492}/g;
  s/\x{1111}\x{119E}\x{11AE}/\x{F491}/g;
  s/\x{1111}\x{119E}\x{11AB}/\x{F490}/g;
  s/\x{1111}\x{119E}\x{11A8}/\x{F48F}/g;
  s/\x{1111}\x{119E}\x{1175}/\x{F4A0}/g;
  s/\x{1111}\x{119E}\x{1165}/\x{F49E}/g;
  s/\x{1111}\x{1175}\x{11F9}/\x{F487}/g;
  s/\x{1111}\x{1175}\x{11F0}/\x{F486}/g;
  s/\x{1111}\x{1175}\x{1167}/\x{F48C}/g;
  s/\x{1111}\x{1175}\x{1163}/\x{F489}/g;
  s/\x{1111}\x{1175}\x{1161}/\x{F488}/g;
  s/\x{1111}\x{1173}\x{11F0}/\x{F481}/g;
  s/\x{1111}\x{1173}\x{1173}/\x{F482}/g;
  s/\x{1111}\x{1172}\x{11F0}/\x{F47C}/g;
  s/\x{1111}\x{1172}\x{1175}/\x{F47F}/g;
  s/\x{1111}\x{1172}\x{1161}/\x{F47D}/g;
  s/\x{1111}\x{116E}\x{11F0}/\x{F475}/g;
  s/\x{1111}\x{116E}\x{1167}/\x{F478}/g;
  s/\x{1111}\x{116D}\x{1175}/\x{F471}/g;
  s/\x{1111}\x{1169}\x{11F0}/\x{F46D}/g;
  s/\x{1111}\x{1169}\x{1169}/\x{F46E}/g;
  s/\x{1111}\x{1167}\x{11F9}/\x{F46A}/g;
  s/\x{1111}\x{1167}\x{11F0}/\x{F469}/g;
  s/\x{1111}\x{1165}\x{11F0}/\x{F461}/g;
  s/\x{1111}\x{1165}\x{1173}/\x{F463}/g;
  s/\x{1111}\x{1165}\x{1169}/\x{F462}/g;
  s/\x{1111}\x{1163}\x{1169}/\x{F45F}/g;
  s/\x{1111}\x{1161}\x{11F9}/\x{F45B}/g;
  s/\x{1111}\x{1161}\x{11F0}/\x{F45A}/g;
  s/\x{1111}\x{1161}\x{1169}/\x{F45C}/g;
  s/\x{1111}\x{1112}\x{1175}/\x{F4AB}/g;
  s/\x{1111}\x{110B}\x{116E}/\x{F4AD}/g;
  s/\x{1111}\x{1107}\x{1161}/\x{F4A9}/g;
  s/\x{1110}\x{119E}\x{11BD}/\x{F447}/g;
  s/\x{1110}\x{119E}\x{11BC}/\x{F446}/g;
  s/\x{1110}\x{119E}\x{11BA}/\x{F445}/g;
  s/\x{1110}\x{119E}\x{11B8}/\x{F444}/g;
  s/\x{1110}\x{119E}\x{11B7}/\x{F443}/g;
  s/\x{1110}\x{119E}\x{11AF}/\x{F440}/g;
  s/\x{1110}\x{119E}\x{11AE}/\x{F43F}/g;
  s/\x{1110}\x{119E}\x{11AB}/\x{F43E}/g;
  s/\x{1110}\x{119E}\x{11A8}/\x{F43D}/g;
  s/\x{1110}\x{119E}\x{1175}/\x{F448}/g;
  s/\x{1110}\x{1175}\x{11F0}/\x{F436}/g;
  s/\x{1110}\x{1175}\x{1167}/\x{F43A}/g;
  s/\x{1110}\x{1175}\x{1163}/\x{F438}/g;
  s/\x{1110}\x{1175}\x{1161}/\x{F437}/g;
  s/\x{1110}\x{1173}\x{11F0}/\x{F430}/g;
  s/\x{1110}\x{1173}\x{1173}/\x{F432}/g;
  s/\x{1110}\x{1173}\x{116E}/\x{F431}/g;
  s/\x{1110}\x{1172}\x{11F0}/\x{F42A}/g;
  s/\x{1110}\x{1172}\x{1175}/\x{F42B}/g;
  s/\x{1110}\x{116E}\x{11F9}/\x{F421}/g;
  s/\x{1110}\x{116E}\x{11F0}/\x{F420}/g;
  s/\x{1110}\x{116E}\x{116E}/\x{F427}/g;
  s/\x{1110}\x{116E}\x{1167}/\x{F425}/g;
  s/\x{1110}\x{116E}\x{1161}/\x{F422}/g;
  s/\x{1110}\x{116D}\x{11F0}/\x{F41B}/g;
  s/\x{1110}\x{116D}\x{1175}/\x{F41C}/g;
  s/\x{1110}\x{1169}\x{11F0}/\x{F414}/g;
  s/\x{1110}\x{1169}\x{1169}/\x{F417}/g;
  s/\x{1110}\x{1167}\x{11F9}/\x{F40D}/g;
  s/\x{1110}\x{1167}\x{11F0}/\x{F40C}/g;
  s/\x{1110}\x{1165}\x{11F0}/\x{F406}/g;
  s/\x{1110}\x{1165}\x{116E}/\x{F408}/g;
  s/\x{1110}\x{1163}\x{11F0}/\x{F402}/g;
  s/\x{1110}\x{1163}\x{1169}/\x{F403}/g;
  s/\x{1110}\x{1161}\x{11F9}/\x{F3FE}/g;
  s/\x{1110}\x{1161}\x{11F0}/\x{F3FD}/g;
  s/\x{1110}\x{1161}\x{116E}/\x{F400}/g;
  s/\x{1110}\x{1161}\x{1169}/\x{F3FF}/g;
  s/\x{1110}\x{1110}\x{119E}/\x{F452}/g;
  s/\x{1110}\x{1110}\x{116E}/\x{F451}/g;
  s/\x{110F}\x{119E}\x{11C0}/\x{F3F2}/g;
  s/\x{110F}\x{119E}\x{11BA}/\x{F3F1}/g;
  s/\x{110F}\x{119E}\x{11B8}/\x{F3F0}/g;
  s/\x{110F}\x{119E}\x{11B7}/\x{F3EF}/g;
  s/\x{110F}\x{119E}\x{11AF}/\x{F3EE}/g;
  s/\x{110F}\x{119E}\x{11AE}/\x{F3ED}/g;
  s/\x{110F}\x{119E}\x{11AB}/\x{F3EC}/g;
  s/\x{110F}\x{119E}\x{11A8}/\x{F3EB}/g;
  s/\x{110F}\x{119E}\x{1175}/\x{F3F3}/g;
  s/\x{110F}\x{1175}\x{11F9}/\x{F3DC}/g;
  s/\x{110F}\x{1175}\x{11F0}/\x{F3DB}/g;
  s/\x{110F}\x{1175}\x{1172}/\x{F3E8}/g;
  s/\x{110F}\x{1175}\x{116E}/\x{F3E6}/g;
  s/\x{110F}\x{1175}\x{116D}/\x{F3E5}/g;
  s/\x{110F}\x{1175}\x{1169}/\x{F3E4}/g;
  s/\x{110F}\x{1175}\x{1167}/\x{F3E2}/g;
  s/\x{110F}\x{1175}\x{1163}/\x{F3DE}/g;
  s/\x{110F}\x{1175}\x{1161}/\x{F3DD}/g;
  s/\x{110F}\x{1173}\x{11F0}/\x{F3D6}/g;
  s/\x{110F}\x{1173}\x{1173}/\x{F3D8}/g;
  s/\x{110F}\x{1173}\x{116E}/\x{F3D7}/g;
  s/\x{110F}\x{1172}\x{11F9}/\x{F3C6}/g;
  s/\x{110F}\x{1172}\x{11F0}/\x{F3C5}/g;
  s/\x{110F}\x{1172}\x{1175}/\x{F3D2}/g;
  s/\x{110F}\x{1172}\x{1167}/\x{F3C9}/g;
  s/\x{110F}\x{1172}\x{1165}/\x{F3C7}/g;
  s/\x{110F}\x{116E}\x{11F9}/\x{F3BB}/g;
  s/\x{110F}\x{116E}\x{11F0}/\x{F3BA}/g;
  s/\x{110F}\x{116E}\x{1167}/\x{F3C2}/g;
  s/\x{110F}\x{116E}\x{1161}/\x{F3BC}/g;
  s/\x{110F}\x{116D}\x{1169}/\x{F3B7}/g;
  s/\x{110F}\x{1169}\x{11F0}/\x{F3B0}/g;
  s/\x{110F}\x{1169}\x{1169}/\x{F3B3}/g;
  s/\x{110F}\x{1167}\x{11F9}/\x{F3AC}/g;
  s/\x{110F}\x{1167}\x{11F0}/\x{F3AB}/g;
  s/\x{110F}\x{1167}\x{116E}/\x{F3AD}/g;
  s/\x{110F}\x{1165}\x{11F9}/\x{F3A6}/g;
  s/\x{110F}\x{1165}\x{116E}/\x{F3A8}/g;
  s/\x{110F}\x{1165}\x{1169}/\x{F3A7}/g;
  s/\x{110F}\x{1163}\x{11F9}/\x{F3A3}/g;
  s/\x{110F}\x{1163}\x{11F0}/\x{F3A2}/g;
  s/\x{110F}\x{1163}\x{1169}/\x{F3A4}/g;
  s/\x{110F}\x{1161}\x{11F0}/\x{F39F}/g;
  s/\x{110F}\x{1161}\x{1169}/\x{F3A0}/g;
  s/\x{110E}\x{119E}\x{11BE}/\x{F352}/g;
  s/\x{110E}\x{119E}\x{11BD}/\x{F351}/g;
  s/\x{110E}\x{119E}\x{11BC}/\x{F350}/g;
  s/\x{110E}\x{119E}\x{11BA}/\x{F34E}/g;
  s/\x{110E}\x{119E}\x{11B8}/\x{F34D}/g;
  s/\x{110E}\x{119E}\x{11B7}/\x{F34B}/g;
  s/\x{110E}\x{119E}\x{11AF}/\x{F345}/g;
  s/\x{110E}\x{119E}\x{11AE}/\x{F344}/g;
  s/\x{110E}\x{119E}\x{11AB}/\x{F343}/g;
  s/\x{110E}\x{119E}\x{11A8}/\x{F342}/g;
  s/\x{110E}\x{119E}\x{1175}/\x{F353}/g;
  s/\x{110E}\x{1175}\x{11F9}/\x{F338}/g;
  s/\x{110E}\x{1175}\x{11F0}/\x{F337}/g;
  s/\x{110E}\x{1175}\x{11EB}/\x{F336}/g;
  s/\x{110E}\x{1175}\x{116E}/\x{F340}/g;
  s/\x{110E}\x{1175}\x{116D}/\x{F33F}/g;
  s/\x{110E}\x{1175}\x{1167}/\x{F33D}/g;
  s/\x{110E}\x{1175}\x{1163}/\x{F33A}/g;
  s/\x{110E}\x{1175}\x{1161}/\x{F339}/g;
  s/\x{110E}\x{1173}\x{11F0}/\x{F32E}/g;
  s/\x{110E}\x{1173}\x{11EB}/\x{F32D}/g;
  s/\x{110E}\x{1173}\x{1173}/\x{F330}/g;
  s/\x{110E}\x{1173}\x{116E}/\x{F32F}/g;
  s/\x{110E}\x{1172}\x{11F9}/\x{F318}/g;
  s/\x{110E}\x{1172}\x{11F0}/\x{F317}/g;
  s/\x{110E}\x{1172}\x{1175}/\x{F325}/g;
  s/\x{110E}\x{1172}\x{1167}/\x{F31E}/g;
  s/\x{110E}\x{1172}\x{1165}/\x{F31B}/g;
  s/\x{110E}\x{1172}\x{1161}/\x{F319}/g;
  s/\x{110E}\x{116E}\x{11F9}/\x{F30B}/g;
  s/\x{110E}\x{116E}\x{11F0}/\x{F30A}/g;
  s/\x{110E}\x{116E}\x{1167}/\x{F311}/g;
  s/\x{110E}\x{116E}\x{1161}/\x{F30C}/g;
  s/\x{110E}\x{116D}\x{11F0}/\x{F303}/g;
  s/\x{110E}\x{116D}\x{1175}/\x{F305}/g;
  s/\x{110E}\x{116D}\x{1169}/\x{F304}/g;
  s/\x{110E}\x{1169}\x{11F0}/\x{F2FE}/g;
  s/\x{110E}\x{1169}\x{11EB}/\x{F2FD}/g;
  s/\x{110E}\x{1169}\x{1169}/\x{F301}/g;
  s/\x{110E}\x{1167}\x{11F9}/\x{F2F7}/g;
  s/\x{110E}\x{1167}\x{11F0}/\x{F2F5}/g;
  s/\x{110E}\x{1167}\x{116E}/\x{F2F9}/g;
  s/\x{110E}\x{1167}\x{1169}/\x{F2F8}/g;
  s/\x{110E}\x{1165}\x{11F0}/\x{F2ED}/g;
  s/\x{110E}\x{1165}\x{11EB}/\x{F2EC}/g;
  s/\x{110E}\x{1165}\x{1173}/\x{F2EE}/g;
  s/\x{110E}\x{1163}\x{11F0}/\x{F2E7}/g;
  s/\x{110E}\x{1163}\x{116E}/\x{F2EB}/g;
  s/\x{110E}\x{1163}\x{1169}/\x{F2E9}/g;
  s/\x{110E}\x{1161}\x{11F9}/\x{F2E3}/g;
  s/\x{110E}\x{1161}\x{11F0}/\x{F2E2}/g;
  s/\x{110E}\x{1161}\x{1169}/\x{F2E4}/g;
  s/\x{110E}\x{1112}\x{1175}/\x{F35D}/g;
  s/\x{110E}\x{110F}\x{116E}/\x{F35B}/g;
  s/\x{110C}\x{119E}\x{11F0}/\x{F20C}/g;
  s/\x{110C}\x{119E}\x{11EB}/\x{F20A}/g;
  s/\x{110C}\x{119E}\x{11C0}/\x{F20F}/g;
  s/\x{110C}\x{119E}\x{11BE}/\x{F20E}/g;
  s/\x{110C}\x{119E}\x{11BD}/\x{F20D}/g;
  s/\x{110C}\x{119E}\x{11BC}/\x{F20B}/g;
  s/\x{110C}\x{119E}\x{11BA}/\x{F209}/g;
  s/\x{110C}\x{119E}\x{11B8}/\x{F207}/g;
  s/\x{110C}\x{119E}\x{11B7}/\x{F204}/g;
  s/\x{110C}\x{119E}\x{11AF}/\x{F200}/g;
  s/\x{110C}\x{119E}\x{11AE}/\x{F1FF}/g;
  s/\x{110C}\x{119E}\x{11AB}/\x{F1FE}/g;
  s/\x{110C}\x{119E}\x{11A8}/\x{F1FD}/g;
  s/\x{110C}\x{119E}\x{1175}/\x{F212}/g;
  s/\x{110C}\x{119E}\x{1161}/\x{F210}/g;
  s/\x{110C}\x{1175}\x{11F9}/\x{F1F3}/g;
  s/\x{110C}\x{1175}\x{11F0}/\x{F1F2}/g;
  s/\x{110C}\x{1175}\x{11EB}/\x{F1F1}/g;
  s/\x{110C}\x{1175}\x{116E}/\x{F1FB}/g;
  s/\x{110C}\x{1175}\x{1169}/\x{F1F9}/g;
  s/\x{110C}\x{1175}\x{1167}/\x{F1F7}/g;
  s/\x{110C}\x{1175}\x{1163}/\x{F1F5}/g;
  s/\x{110C}\x{1175}\x{1161}/\x{F1F4}/g;
  s/\x{110C}\x{1173}\x{11F0}/\x{F1E2}/g;
  s/\x{110C}\x{1173}\x{11EB}/\x{F1E1}/g;
  s/\x{110C}\x{1173}\x{1173}/\x{F1E4}/g;
  s/\x{110C}\x{1173}\x{116E}/\x{F1E3}/g;
  s/\x{110C}\x{1172}\x{11F9}/\x{F1C6}/g;
  s/\x{110C}\x{1172}\x{11F0}/\x{F1C4}/g;
  s/\x{110C}\x{1172}\x{1175}/\x{F1D6}/g;
  s/\x{110C}\x{1172}\x{1167}/\x{F1CD}/g;
  s/\x{110C}\x{1172}\x{1165}/\x{F1C9}/g;
  s/\x{110C}\x{1172}\x{1161}/\x{F1C7}/g;
  s/\x{110C}\x{116E}\x{11F9}/\x{F1B9}/g;
  s/\x{110C}\x{116E}\x{11F0}/\x{F1B8}/g;
  s/\x{110C}\x{116E}\x{11EB}/\x{F1B7}/g;
  s/\x{110C}\x{116E}\x{1167}/\x{F1BD}/g;
  s/\x{110C}\x{116E}\x{1161}/\x{F1BA}/g;
  s/\x{110C}\x{116D}\x{11F0}/\x{F1AC}/g;
  s/\x{110C}\x{116D}\x{1175}/\x{F1AF}/g;
  s/\x{110C}\x{116D}\x{1169}/\x{F1AE}/g;
  s/\x{110C}\x{1169}\x{11F0}/\x{F1A4}/g;
  s/\x{110C}\x{1169}\x{11EB}/\x{F1A3}/g;
  s/\x{110C}\x{1167}\x{11F9}/\x{F19A}/g;
  s/\x{110C}\x{1167}\x{11F0}/\x{F199}/g;
  s/\x{110C}\x{1167}\x{11EB}/\x{F198}/g;
  s/\x{110C}\x{1167}\x{116E}/\x{F19E}/g;
  s/\x{110C}\x{1167}\x{1169}/\x{F19D}/g;
  s/\x{110C}\x{1167}\x{1163}/\x{F19B}/g;
  s/\x{110C}\x{1165}\x{11F9}/\x{F189}/g;
  s/\x{110C}\x{1165}\x{11F0}/\x{F187}/g;
  s/\x{110C}\x{1165}\x{11EB}/\x{F186}/g;
  s/\x{110C}\x{1165}\x{1173}/\x{F18B}/g;
  s/\x{110C}\x{1165}\x{116E}/\x{F18A}/g;
  s/\x{110C}\x{1163}\x{11F0}/\x{F17A}/g;
  s/\x{110C}\x{1163}\x{11EB}/\x{F178}/g;
  s/\x{110C}\x{1163}\x{116E}/\x{F17D}/g;
  s/\x{110C}\x{1163}\x{1169}/\x{F17C}/g;
  s/\x{110C}\x{1161}\x{11F9}/\x{F171}/g;
  s/\x{110C}\x{1161}\x{11F0}/\x{F16E}/g;
  s/\x{110C}\x{1161}\x{11EB}/\x{F16D}/g;
  s/\x{110C}\x{1161}\x{1169}/\x{F172}/g;
  s/\x{110C}\x{110C}\x{119E}/\x{F250}/g;
  s/\x{110C}\x{110B}\x{1175}/\x{F21E}/g;
  s/\x{110C}\x{110B}\x{1173}/\x{F21C}/g;
  s/\x{110B}\x{119E}\x{11F0}/\x{EE9A}/g;
  s/\x{110B}\x{119E}\x{11EB}/\x{EE98}/g;
  s/\x{110B}\x{119E}\x{11C0}/\x{EE9D}/g;
  s/\x{110B}\x{119E}\x{11BE}/\x{EE9C}/g;
  s/\x{110B}\x{119E}\x{11BD}/\x{EE9B}/g;
  s/\x{110B}\x{119E}\x{11BC}/\x{EE99}/g;
  s/\x{110B}\x{119E}\x{11BA}/\x{EE97}/g;
  s/\x{110B}\x{119E}\x{11B8}/\x{EE95}/g;
  s/\x{110B}\x{119E}\x{11B7}/\x{EE93}/g;
  s/\x{110B}\x{119E}\x{11AF}/\x{EE8C}/g;
  s/\x{110B}\x{119E}\x{11AE}/\x{EE8B}/g;
  s/\x{110B}\x{119E}\x{11AB}/\x{EE8A}/g;
  s/\x{110B}\x{119E}\x{11A8}/\x{EE89}/g;
  s/\x{110B}\x{119E}\x{119E}/\x{EEAE}/g;
  s/\x{110B}\x{119E}\x{1175}/\x{EEA4}/g;
  s/\x{110B}\x{119E}\x{1165}/\x{EE9E}/g;
  s/\x{110B}\x{1175}\x{11F9}/\x{EE81}/g;
  s/\x{110B}\x{1175}\x{11F0}/\x{EE80}/g;
  s/\x{110B}\x{1175}\x{1173}/\x{EE86}/g;
  s/\x{110B}\x{1175}\x{116E}/\x{EE85}/g;
  s/\x{110B}\x{1175}\x{1169}/\x{EE83}/g;
  s/\x{110B}\x{1175}\x{1161}/\x{EE82}/g;
  s/\x{110B}\x{1173}\x{11F9}/\x{EE6D}/g;
  s/\x{110B}\x{1173}\x{11F0}/\x{EE6B}/g;
  s/\x{110B}\x{1173}\x{11EB}/\x{EE6A}/g;
  s/\x{110B}\x{1173}\x{1173}/\x{EE6F}/g;
  s/\x{110B}\x{1173}\x{116E}/\x{EE6E}/g;
  s/\x{110B}\x{1172}\x{11F9}/\x{EE47}/g;
  s/\x{110B}\x{1172}\x{11F0}/\x{EE46}/g;
  s/\x{110B}\x{1172}\x{1175}/\x{EE59}/g;
  s/\x{110B}\x{1172}\x{116E}/\x{EE58}/g;
  s/\x{110B}\x{1172}\x{1167}/\x{EE4E}/g;
  s/\x{110B}\x{1172}\x{1165}/\x{EE4B}/g;
  s/\x{110B}\x{1172}\x{1161}/\x{EE48}/g;
  s/\x{110B}\x{116E}\x{11F9}/\x{EE33}/g;
  s/\x{110B}\x{116E}\x{11F0}/\x{EE32}/g;
  s/\x{110B}\x{116E}\x{11EB}/\x{EE2F}/g;
  s/\x{110B}\x{116E}\x{1167}/\x{EE40}/g;
  s/\x{110B}\x{116E}\x{1161}/\x{EE34}/g;
  s/\x{110B}\x{116D}\x{11F0}/\x{EE19}/g;
  s/\x{110B}\x{116D}\x{1175}/\x{EE20}/g;
  s/\x{110B}\x{116D}\x{1169}/\x{EE1F}/g;
  s/\x{110B}\x{116D}\x{1165}/\x{EE1D}/g;
  s/\x{110B}\x{116D}\x{1161}/\x{EE1A}/g;
  s/\x{110B}\x{1169}\x{11F0}/\x{EE07}/g;
  s/\x{110B}\x{1169}\x{11EB}/\x{EE06}/g;
  s/\x{110B}\x{1169}\x{1169}/\x{EE12}/g;
  s/\x{110B}\x{1169}\x{1165}/\x{EE0F}/g;
  s/\x{110B}\x{1167}\x{11F9}/\x{EDF9}/g;
  s/\x{110B}\x{1167}\x{11F0}/\x{EDF8}/g;
  s/\x{110B}\x{1167}\x{11EB}/\x{EDF5}/g;
  s/\x{110B}\x{1167}\x{116E}/\x{EDFB}/g;
  s/\x{110B}\x{1167}\x{1169}/\x{EDFA}/g;
  s/\x{110B}\x{1165}\x{11F0}/\x{EDD6}/g;
  s/\x{110B}\x{1165}\x{11EB}/\x{EDD5}/g;
  s/\x{110B}\x{1165}\x{1173}/\x{EDD9}/g;
  s/\x{110B}\x{1165}\x{1169}/\x{EDD8}/g;
  s/\x{110B}\x{1163}\x{11F9}/\x{EDC7}/g;
  s/\x{110B}\x{1163}\x{11F0}/\x{EDC5}/g;
  s/\x{110B}\x{1163}\x{116E}/\x{EDC9}/g;
  s/\x{110B}\x{1163}\x{1169}/\x{EDC8}/g;
  s/\x{110B}\x{1161}\x{11F0}/\x{EDB8}/g;
  s/\x{110B}\x{1161}\x{11EB}/\x{EDB7}/g;
  s/\x{110B}\x{1161}\x{1169}/\x{EDBA}/g;
  s/\x{110B}\x{1140}\x{119E}/\x{EF54}/g;
  s/\x{110B}\x{1140}\x{1175}/\x{EF52}/g;
  s/\x{110B}\x{1140}\x{1173}/\x{EF50}/g;
  s/\x{110B}\x{1140}\x{1172}/\x{EF4E}/g;
  s/\x{110B}\x{1140}\x{116E}/\x{EF48}/g;
  s/\x{110B}\x{1140}\x{116D}/\x{EF47}/g;
  s/\x{110B}\x{1140}\x{1169}/\x{EF43}/g;
  s/\x{110B}\x{1140}\x{1167}/\x{EF41}/g;
  s/\x{110B}\x{1140}\x{1165}/\x{EF3F}/g;
  s/\x{110B}\x{1140}\x{1163}/\x{EF3E}/g;
  s/\x{110B}\x{1140}\x{1161}/\x{EF3C}/g;
  s/\x{110B}\x{1112}\x{116E}/\x{EFD1}/g;
  s/\x{110B}\x{1112}\x{1169}/\x{EFD0}/g;
  s/\x{110B}\x{1111}\x{119E}/\x{EFCE}/g;
  s/\x{110B}\x{1111}\x{1175}/\x{EFC7}/g;
  s/\x{110B}\x{1111}\x{1173}/\x{EFC5}/g;
  s/\x{110B}\x{1111}\x{1172}/\x{EFC3}/g;
  s/\x{110B}\x{1111}\x{116E}/\x{EFB4}/g;
  s/\x{110B}\x{1111}\x{1169}/\x{EFAA}/g;
  s/\x{110B}\x{1111}\x{1165}/\x{EFA0}/g;
  s/\x{110B}\x{1111}\x{1161}/\x{EF97}/g;
  s/\x{110B}\x{1110}\x{1173}/\x{EF96}/g;
  s/\x{110B}\x{110E}\x{1173}/\x{EF95}/g;
  s/\x{110B}\x{110E}\x{1172}/\x{EF91}/g;
  s/\x{110B}\x{110E}\x{116E}/\x{EF8A}/g;
  s/\x{110B}\x{110E}\x{1165}/\x{EF87}/g;
  s/\x{110B}\x{110E}\x{1161}/\x{EF83}/g;
  s/\x{110B}\x{110C}\x{1175}/\x{EF82}/g;
  s/\x{110B}\x{110C}\x{1173}/\x{EF81}/g;
  s/\x{110B}\x{110C}\x{1172}/\x{EF7F}/g;
  s/\x{110B}\x{110C}\x{116E}/\x{EF77}/g;
  s/\x{110B}\x{110C}\x{116D}/\x{EF76}/g;
  s/\x{110B}\x{110C}\x{1169}/\x{EF75}/g;
  s/\x{110B}\x{110C}\x{1165}/\x{EF70}/g;
  s/\x{110B}\x{110C}\x{1161}/\x{EF6B}/g;
  s/\x{110B}\x{110B}\x{1175}/\x{EF69}/g;
  s/\x{110B}\x{110B}\x{1172}/\x{EF64}/g;
  s/\x{110B}\x{110B}\x{116E}/\x{EF62}/g;
  s/\x{110B}\x{110B}\x{116D}/\x{EF5D}/g;
  s/\x{110B}\x{110B}\x{1167}/\x{EF58}/g;
  s/\x{110B}\x{110B}\x{1163}/\x{EF56}/g;
  s/\x{110B}\x{1109}\x{119E}/\x{EF3A}/g;
  s/\x{110B}\x{1109}\x{1175}/\x{EF38}/g;
  s/\x{110B}\x{1109}\x{1173}/\x{EF34}/g;
  s/\x{110B}\x{1109}\x{1172}/\x{EF2E}/g;
  s/\x{110B}\x{1109}\x{116E}/\x{EF26}/g;
  s/\x{110B}\x{1109}\x{1169}/\x{EF24}/g;
  s/\x{110B}\x{1109}\x{1167}/\x{EF21}/g;
  s/\x{110B}\x{1109}\x{1165}/\x{EF1A}/g;
  s/\x{110B}\x{1109}\x{1161}/\x{EF15}/g;
  s/\x{110B}\x{1107}\x{119E}/\x{EF13}/g;
  s/\x{110B}\x{1107}\x{1175}/\x{EF0D}/g;
  s/\x{110B}\x{1107}\x{1173}/\x{EF09}/g;
  s/\x{110B}\x{1107}\x{1172}/\x{EF08}/g;
  s/\x{110B}\x{1107}\x{116E}/\x{EF00}/g;
  s/\x{110B}\x{1107}\x{116D}/\x{EEFF}/g;
  s/\x{110B}\x{1107}\x{1169}/\x{EEF9}/g;
  s/\x{110B}\x{1107}\x{1167}/\x{EEF6}/g;
  s/\x{110B}\x{1107}\x{1165}/\x{EEEF}/g;
  s/\x{110B}\x{1107}\x{1161}/\x{EEE8}/g;
  s/\x{110B}\x{1106}\x{119E}/\x{EEE6}/g;
  s/\x{110B}\x{1106}\x{116E}/\x{EEE5}/g;
  s/\x{110B}\x{1105}\x{1175}/\x{EEE4}/g;
  s/\x{110B}\x{1105}\x{1169}/\x{EEE3}/g;
  s/\x{110B}\x{1105}\x{1165}/\x{EEE2}/g;
  s/\x{110B}\x{1103}\x{119E}/\x{EEE0}/g;
  s/\x{110B}\x{1103}\x{1175}/\x{EEDE}/g;
  s/\x{110B}\x{1103}\x{1173}/\x{EEDB}/g;
  s/\x{110B}\x{1103}\x{1169}/\x{EED8}/g;
  s/\x{110B}\x{1103}\x{1165}/\x{EED4}/g;
  s/\x{110B}\x{1103}\x{1161}/\x{EED2}/g;
  s/\x{110B}\x{1100}\x{119E}/\x{EED0}/g;
  s/\x{110B}\x{1100}\x{1175}/\x{EECD}/g;
  s/\x{110B}\x{1100}\x{1173}/\x{EECB}/g;
  s/\x{110B}\x{1100}\x{1172}/\x{EEC9}/g;
  s/\x{110B}\x{1100}\x{116E}/\x{EEC2}/g;
  s/\x{110B}\x{1100}\x{116D}/\x{EEC1}/g;
  s/\x{110B}\x{1100}\x{1169}/\x{EEBA}/g;
  s/\x{110B}\x{1100}\x{1167}/\x{EEB7}/g;
  s/\x{110B}\x{1100}\x{1165}/\x{EEB5}/g;
  s/\x{110B}\x{1100}\x{1163}/\x{EEB3}/g;
  s/\x{110B}\x{1100}\x{1161}/\x{EEB0}/g;
  s/\x{1109}\x{119E}\x{11F0}/\x{E995}/g;
  s/\x{1109}\x{119E}\x{11EB}/\x{E993}/g;
  s/\x{1109}\x{119E}\x{11C0}/\x{E998}/g;
  s/\x{1109}\x{119E}\x{11BE}/\x{E997}/g;
  s/\x{1109}\x{119E}\x{11BD}/\x{E996}/g;
  s/\x{1109}\x{119E}\x{11BC}/\x{E994}/g;
  s/\x{1109}\x{119E}\x{11BA}/\x{E991}/g;
  s/\x{1109}\x{119E}\x{11B8}/\x{E98E}/g;
  s/\x{1109}\x{119E}\x{11B7}/\x{E98B}/g;
  s/\x{1109}\x{119E}\x{11AF}/\x{E982}/g;
  s/\x{1109}\x{119E}\x{11AE}/\x{E980}/g;
  s/\x{1109}\x{119E}\x{11AB}/\x{E97F}/g;
  s/\x{1109}\x{119E}\x{11A8}/\x{E97E}/g;
  s/\x{1109}\x{119E}\x{1175}/\x{E999}/g;
  s/\x{1109}\x{1175}\x{11F9}/\x{E96E}/g;
  s/\x{1109}\x{1175}\x{11F0}/\x{E96D}/g;
  s/\x{1109}\x{1175}\x{11EB}/\x{E96C}/g;
  s/\x{1109}\x{1175}\x{1172}/\x{E97B}/g;
  s/\x{1109}\x{1175}\x{116E}/\x{E97A}/g;
  s/\x{1109}\x{1175}\x{116D}/\x{E979}/g;
  s/\x{1109}\x{1175}\x{1169}/\x{E977}/g;
  s/\x{1109}\x{1175}\x{1167}/\x{E975}/g;
  s/\x{1109}\x{1175}\x{1163}/\x{E970}/g;
  s/\x{1109}\x{1175}\x{1161}/\x{E96F}/g;
  s/\x{1109}\x{1173}\x{11F9}/\x{E95D}/g;
  s/\x{1109}\x{1173}\x{11F0}/\x{E95B}/g;
  s/\x{1109}\x{1173}\x{11EB}/\x{E958}/g;
  s/\x{1109}\x{1173}\x{1173}/\x{E95F}/g;
  s/\x{1109}\x{1173}\x{116E}/\x{E95E}/g;
  s/\x{1109}\x{1172}\x{11F0}/\x{E93A}/g;
  s/\x{1109}\x{1172}\x{1175}/\x{E94D}/g;
  s/\x{1109}\x{1172}\x{116E}/\x{E94B}/g;
  s/\x{1109}\x{1172}\x{1169}/\x{E949}/g;
  s/\x{1109}\x{1172}\x{1167}/\x{E941}/g;
  s/\x{1109}\x{1172}\x{1165}/\x{E93F}/g;
  s/\x{1109}\x{1172}\x{1161}/\x{E93B}/g;
  s/\x{1109}\x{116E}\x{11F9}/\x{E92D}/g;
  s/\x{1109}\x{116E}\x{11F0}/\x{E92C}/g;
  s/\x{1109}\x{116E}\x{11EB}/\x{E92B}/g;
  s/\x{1109}\x{116E}\x{1167}/\x{E931}/g;
  s/\x{1109}\x{116E}\x{1161}/\x{E92E}/g;
  s/\x{1109}\x{116D}\x{11F0}/\x{E91C}/g;
  s/\x{1109}\x{116D}\x{1175}/\x{E920}/g;
  s/\x{1109}\x{116D}\x{1169}/\x{E91E}/g;
  s/\x{1109}\x{1169}\x{11F0}/\x{E911}/g;
  s/\x{1109}\x{1169}\x{11EB}/\x{E910}/g;
  s/\x{1109}\x{1169}\x{1169}/\x{E918}/g;
  s/\x{1109}\x{1169}\x{1163}/\x{E916}/g;
  s/\x{1109}\x{1167}\x{11F9}/\x{E906}/g;
  s/\x{1109}\x{1167}\x{11F0}/\x{E904}/g;
  s/\x{1109}\x{1167}\x{116E}/\x{E908}/g;
  s/\x{1109}\x{1167}\x{1169}/\x{E907}/g;
  s/\x{1109}\x{1165}\x{11F9}/\x{E8F1}/g;
  s/\x{1109}\x{1165}\x{11F0}/\x{E8F0}/g;
  s/\x{1109}\x{1165}\x{11EB}/\x{E8ED}/g;
  s/\x{1109}\x{1165}\x{1173}/\x{E8F4}/g;
  s/\x{1109}\x{1165}\x{116E}/\x{E8F3}/g;
  s/\x{1109}\x{1165}\x{1169}/\x{E8F2}/g;
  s/\x{1109}\x{1163}\x{11F0}/\x{E8E6}/g;
  s/\x{1109}\x{1163}\x{11EB}/\x{E8E5}/g;
  s/\x{1109}\x{1163}\x{1169}/\x{E8E8}/g;
  s/\x{1109}\x{1161}\x{11F9}/\x{E8DC}/g;
  s/\x{1109}\x{1161}\x{11F0}/\x{E8DA}/g;
  s/\x{1109}\x{1161}\x{11EB}/\x{E8D8}/g;
  s/\x{1109}\x{1161}\x{1173}/\x{E8DF}/g;
  s/\x{1109}\x{1161}\x{116E}/\x{E8DE}/g;
  s/\x{1109}\x{1161}\x{1169}/\x{E8DD}/g;
  s/\x{1109}\x{1112}\x{1173}/\x{EC97}/g;
  s/\x{1109}\x{1112}\x{1167}/\x{EC96}/g;
  s/\x{1109}\x{1112}\x{1165}/\x{EC94}/g;
  s/\x{1109}\x{1111}\x{119E}/\x{EC92}/g;
  s/\x{1109}\x{1111}\x{1175}/\x{EC91}/g;
  s/\x{1109}\x{1111}\x{116E}/\x{EC8F}/g;
  s/\x{1109}\x{1111}\x{1167}/\x{EC8D}/g;
  s/\x{1109}\x{1110}\x{119E}/\x{EC8A}/g;
  s/\x{1109}\x{1110}\x{1175}/\x{EC88}/g;
  s/\x{1109}\x{1110}\x{1173}/\x{EC84}/g;
  s/\x{1109}\x{1110}\x{116E}/\x{EC81}/g;
  s/\x{1109}\x{1110}\x{1169}/\x{EC7F}/g;
  s/\x{1109}\x{1110}\x{1165}/\x{EC7C}/g;
  s/\x{1109}\x{1110}\x{1161}/\x{EC7A}/g;
  s/\x{1109}\x{110F}\x{119E}/\x{EC79}/g;
  s/\x{1109}\x{110F}\x{1175}/\x{EC78}/g;
  s/\x{1109}\x{110F}\x{1173}/\x{EC75}/g;
  s/\x{1109}\x{110F}\x{116E}/\x{EC73}/g;
  s/\x{1109}\x{110F}\x{1169}/\x{EC69}/g;
  s/\x{1109}\x{110F}\x{1167}/\x{EC68}/g;
  s/\x{1109}\x{110F}\x{1165}/\x{EC65}/g;
  s/\x{1109}\x{110F}\x{1161}/\x{EC63}/g;
  s/\x{1109}\x{110E}\x{119E}/\x{EC60}/g;
  s/\x{1109}\x{110E}\x{1175}/\x{EC5F}/g;
  s/\x{1109}\x{110E}\x{1169}/\x{EC5E}/g;
  s/\x{1109}\x{110C}\x{119E}/\x{EC51}/g;
  s/\x{1109}\x{110C}\x{1175}/\x{EC45}/g;
  s/\x{1109}\x{110C}\x{1173}/\x{EC3C}/g;
  s/\x{1109}\x{110C}\x{1172}/\x{EC38}/g;
  s/\x{1109}\x{110C}\x{116E}/\x{EC2D}/g;
  s/\x{1109}\x{110C}\x{116D}/\x{EC27}/g;
  s/\x{1109}\x{110C}\x{1169}/\x{EC16}/g;
  s/\x{1109}\x{110C}\x{1167}/\x{EC0E}/g;
  s/\x{1109}\x{110C}\x{1165}/\x{EC01}/g;
  s/\x{1109}\x{110C}\x{1163}/\x{EBFD}/g;
  s/\x{1109}\x{110C}\x{1161}/\x{EBEB}/g;
  s/\x{1109}\x{110B}\x{1175}/\x{EBE9}/g;
  s/\x{1109}\x{110B}\x{1173}/\x{EBE6}/g;
  s/\x{1109}\x{1109}\x{119E}/\x{EBD4}/g;
  s/\x{1109}\x{1107}\x{119E}/\x{EB8E}/g;
  s/\x{1109}\x{1107}\x{1175}/\x{EB82}/g;
  s/\x{1109}\x{1107}\x{1173}/\x{EB78}/g;
  s/\x{1109}\x{1107}\x{1172}/\x{EB76}/g;
  s/\x{1109}\x{1107}\x{116E}/\x{EB66}/g;
  s/\x{1109}\x{1107}\x{116D}/\x{EB65}/g;
  s/\x{1109}\x{1107}\x{1169}/\x{EB56}/g;
  s/\x{1109}\x{1107}\x{1167}/\x{EB4C}/g;
  s/\x{1109}\x{1107}\x{1165}/\x{EB3D}/g;
  s/\x{1109}\x{1107}\x{1163}/\x{EB3B}/g;
  s/\x{1109}\x{1107}\x{1161}/\x{EB27}/g;
  s/\x{1109}\x{1106}\x{119E}/\x{EB25}/g;
  s/\x{1109}\x{1106}\x{1173}/\x{EB23}/g;
  s/\x{1109}\x{1106}\x{1169}/\x{EB22}/g;
  s/\x{1109}\x{1106}\x{1161}/\x{EB20}/g;
  s/\x{1109}\x{1105}\x{1175}/\x{EB1E}/g;
  s/\x{1109}\x{1105}\x{1173}/\x{EB1A}/g;
  s/\x{1109}\x{1105}\x{1172}/\x{EB17}/g;
  s/\x{1109}\x{1105}\x{116E}/\x{EB16}/g;
  s/\x{1109}\x{1105}\x{1167}/\x{EB14}/g;
  s/\x{1109}\x{1105}\x{1165}/\x{EB11}/g;
  s/\x{1109}\x{1105}\x{1163}/\x{EB10}/g;
  s/\x{1109}\x{1105}\x{1161}/\x{EB0D}/g;
  s/\x{1109}\x{1103}\x{119E}/\x{EAF8}/g;
  s/\x{1109}\x{1103}\x{1175}/\x{EAED}/g;
  s/\x{1109}\x{1103}\x{1173}/\x{EAD9}/g;
  s/\x{1109}\x{1103}\x{1172}/\x{EAD2}/g;
  s/\x{1109}\x{1103}\x{116E}/\x{EABF}/g;
  s/\x{1109}\x{1103}\x{116D}/\x{EABB}/g;
  s/\x{1109}\x{1103}\x{1169}/\x{EAAB}/g;
  s/\x{1109}\x{1103}\x{1167}/\x{EAA5}/g;
  s/\x{1109}\x{1103}\x{1165}/\x{EA8D}/g;
  s/\x{1109}\x{1103}\x{1163}/\x{EA88}/g;
  s/\x{1109}\x{1103}\x{1161}/\x{EA71}/g;
  s/\x{1109}\x{1102}\x{119E}/\x{EA70}/g;
  s/\x{1109}\x{1102}\x{1175}/\x{EA6F}/g;
  s/\x{1109}\x{1102}\x{1169}/\x{EA6E}/g;
  s/\x{1109}\x{1102}\x{1167}/\x{EA6C}/g;
  s/\x{1109}\x{1102}\x{1161}/\x{EA6A}/g;
  s/\x{1109}\x{1100}\x{119E}/\x{EA58}/g;
  s/\x{1109}\x{1100}\x{1175}/\x{EA4B}/g;
  s/\x{1109}\x{1100}\x{1173}/\x{EA2E}/g;
  s/\x{1109}\x{1100}\x{1172}/\x{EA27}/g;
  s/\x{1109}\x{1100}\x{116E}/\x{EA05}/g;
  s/\x{1109}\x{1100}\x{116D}/\x{EA01}/g;
  s/\x{1109}\x{1100}\x{1169}/\x{E9E1}/g;
  s/\x{1109}\x{1100}\x{1167}/\x{E9D5}/g;
  s/\x{1109}\x{1100}\x{1165}/\x{E9BF}/g;
  s/\x{1109}\x{1100}\x{1163}/\x{E9BB}/g;
  s/\x{1109}\x{1100}\x{1161}/\x{E9A5}/g;
  s/\x{1107}\x{119E}\x{11F0}/\x{E65F}/g;
  s/\x{1107}\x{119E}\x{11EB}/\x{E65E}/g;
  s/\x{1107}\x{119E}\x{11C0}/\x{E662}/g;
  s/\x{1107}\x{119E}\x{11BE}/\x{E661}/g;
  s/\x{1107}\x{119E}\x{11BD}/\x{E660}/g;
  s/\x{1107}\x{119E}\x{11BA}/\x{E65C}/g;
  s/\x{1107}\x{119E}\x{11B8}/\x{E65A}/g;
  s/\x{1107}\x{119E}\x{11B7}/\x{E659}/g;
  s/\x{1107}\x{119E}\x{11AF}/\x{E64F}/g;
  s/\x{1107}\x{119E}\x{11AE}/\x{E64E}/g;
  s/\x{1107}\x{119E}\x{11AB}/\x{E64D}/g;
  s/\x{1107}\x{119E}\x{11A8}/\x{E64B}/g;
  s/\x{1107}\x{119E}\x{1175}/\x{E668}/g;
  s/\x{1107}\x{119E}\x{1165}/\x{E663}/g;
  s/\x{1107}\x{1175}\x{11F9}/\x{E642}/g;
  s/\x{1107}\x{1175}\x{11F0}/\x{E640}/g;
  s/\x{1107}\x{1175}\x{11EB}/\x{E63F}/g;
  s/\x{1107}\x{1175}\x{116E}/\x{E649}/g;
  s/\x{1107}\x{1175}\x{1169}/\x{E648}/g;
  s/\x{1107}\x{1175}\x{1167}/\x{E646}/g;
  s/\x{1107}\x{1175}\x{1163}/\x{E644}/g;
  s/\x{1107}\x{1175}\x{1161}/\x{E643}/g;
  s/\x{1107}\x{1173}\x{11F9}/\x{E634}/g;
  s/\x{1107}\x{1173}\x{11F0}/\x{E633}/g;
  s/\x{1107}\x{1173}\x{11EB}/\x{E632}/g;
  s/\x{1107}\x{1173}\x{1173}/\x{E635}/g;
  s/\x{1107}\x{1172}\x{1175}/\x{E62A}/g;
  s/\x{1107}\x{116E}\x{11F9}/\x{E621}/g;
  s/\x{1107}\x{116E}\x{11F0}/\x{E620}/g;
  s/\x{1107}\x{116E}\x{11EB}/\x{E61E}/g;
  s/\x{1107}\x{116E}\x{116E}/\x{E625}/g;
  s/\x{1107}\x{116E}\x{1167}/\x{E623}/g;
  s/\x{1107}\x{116D}\x{11F0}/\x{E612}/g;
  s/\x{1107}\x{116D}\x{1175}/\x{E614}/g;
  s/\x{1107}\x{116D}\x{1169}/\x{E613}/g;
  s/\x{1107}\x{1169}\x{11F0}/\x{E60B}/g;
  s/\x{1107}\x{1169}\x{11EB}/\x{E60A}/g;
  s/\x{1107}\x{1169}\x{1169}/\x{E60C}/g;
  s/\x{1107}\x{1167}\x{11F9}/\x{E601}/g;
  s/\x{1107}\x{1167}\x{11F0}/\x{E5FF}/g;
  s/\x{1107}\x{1165}\x{11F0}/\x{E5F4}/g;
  s/\x{1107}\x{1165}\x{11EB}/\x{E5F3}/g;
  s/\x{1107}\x{1165}\x{1173}/\x{E5F7}/g;
  s/\x{1107}\x{1165}\x{116E}/\x{E5F5}/g;
  s/\x{1107}\x{1163}\x{1169}/\x{E5EF}/g;
  s/\x{1107}\x{1161}\x{11F9}/\x{E5E8}/g;
  s/\x{1107}\x{1161}\x{11F0}/\x{E5E5}/g;
  s/\x{1107}\x{1161}\x{11EB}/\x{E5E3}/g;
  s/\x{1107}\x{1161}\x{116E}/\x{E5EB}/g;
  s/\x{1107}\x{1161}\x{1169}/\x{E5E9}/g;
  s/\x{1107}\x{1112}\x{1161}/\x{E855}/g;
  s/\x{1107}\x{1111}\x{119E}/\x{E853}/g;
  s/\x{1107}\x{1111}\x{1175}/\x{E852}/g;
  s/\x{1107}\x{1111}\x{1173}/\x{E850}/g;
  s/\x{1107}\x{1111}\x{116D}/\x{E84F}/g;
  s/\x{1107}\x{1111}\x{1169}/\x{E84D}/g;
  s/\x{1107}\x{1111}\x{1161}/\x{E84B}/g;
  s/\x{1107}\x{1110}\x{119E}/\x{E847}/g;
  s/\x{1107}\x{1110}\x{1175}/\x{E843}/g;
  s/\x{1107}\x{1110}\x{1173}/\x{E83D}/g;
  s/\x{1107}\x{1110}\x{116E}/\x{E837}/g;
  s/\x{1107}\x{1110}\x{1169}/\x{E834}/g;
  s/\x{1107}\x{1110}\x{1167}/\x{E833}/g;
  s/\x{1107}\x{1110}\x{1165}/\x{E82C}/g;
  s/\x{1107}\x{1110}\x{1161}/\x{E82B}/g;
  s/\x{1107}\x{110F}\x{1173}/\x{E82A}/g;
  s/\x{1107}\x{110F}\x{1169}/\x{E827}/g;
  s/\x{1107}\x{110E}\x{119E}/\x{E825}/g;
  s/\x{1107}\x{110E}\x{1169}/\x{E823}/g;
  s/\x{1107}\x{110C}\x{119E}/\x{E81A}/g;
  s/\x{1107}\x{110C}\x{1175}/\x{E810}/g;
  s/\x{1107}\x{110C}\x{1173}/\x{E805}/g;
  s/\x{1107}\x{110C}\x{116E}/\x{E800}/g;
  s/\x{1107}\x{110C}\x{116D}/\x{E7FF}/g;
  s/\x{1107}\x{110C}\x{1169}/\x{E7EF}/g;
  s/\x{1107}\x{110C}\x{1167}/\x{E7EE}/g;
  s/\x{1107}\x{110C}\x{1165}/\x{E7EB}/g;
  s/\x{1107}\x{110C}\x{1163}/\x{E7E7}/g;
  s/\x{1107}\x{110C}\x{1161}/\x{E7DF}/g;
  s/\x{1107}\x{110B}\x{119E}/\x{E8A7}/g;
  s/\x{1107}\x{110B}\x{1175}/\x{E8A1}/g;
  s/\x{1107}\x{110B}\x{1173}/\x{E895}/g;
  s/\x{1107}\x{110B}\x{1172}/\x{E893}/g;
  s/\x{1107}\x{110B}\x{116E}/\x{E885}/g;
  s/\x{1107}\x{110B}\x{116D}/\x{E882}/g;
  s/\x{1107}\x{110B}\x{1169}/\x{E876}/g;
  s/\x{1107}\x{110B}\x{1167}/\x{E86F}/g;
  s/\x{1107}\x{110B}\x{1165}/\x{E868}/g;
  s/\x{1107}\x{110B}\x{1161}/\x{E857}/g;
  s/\x{1107}\x{1109}\x{119E}/\x{E786}/g;
  s/\x{1107}\x{1109}\x{1175}/\x{E77C}/g;
  s/\x{1107}\x{1109}\x{1173}/\x{E770}/g;
  s/\x{1107}\x{1109}\x{1172}/\x{E76B}/g;
  s/\x{1107}\x{1109}\x{116E}/\x{E75D}/g;
  s/\x{1107}\x{1109}\x{116D}/\x{E75C}/g;
  s/\x{1107}\x{1109}\x{1169}/\x{E74C}/g;
  s/\x{1107}\x{1109}\x{1167}/\x{E744}/g;
  s/\x{1107}\x{1109}\x{1165}/\x{E73B}/g;
  s/\x{1107}\x{1109}\x{1163}/\x{E735}/g;
  s/\x{1107}\x{1109}\x{1161}/\x{E725}/g;
  s/\x{1107}\x{1107}\x{119E}/\x{E71D}/g;
  s/\x{1107}\x{1103}\x{119E}/\x{E6EF}/g;
  s/\x{1107}\x{1103}\x{1175}/\x{E6E7}/g;
  s/\x{1107}\x{1103}\x{1173}/\x{E6D7}/g;
  s/\x{1107}\x{1103}\x{1172}/\x{E6D5}/g;
  s/\x{1107}\x{1103}\x{116E}/\x{E6C6}/g;
  s/\x{1107}\x{1103}\x{116D}/\x{E6C5}/g;
  s/\x{1107}\x{1103}\x{1169}/\x{E6BC}/g;
  s/\x{1107}\x{1103}\x{1167}/\x{E6B8}/g;
  s/\x{1107}\x{1103}\x{1165}/\x{E6AC}/g;
  s/\x{1107}\x{1103}\x{1163}/\x{E6AB}/g;
  s/\x{1107}\x{1103}\x{1161}/\x{E6A3}/g;
  s/\x{1107}\x{1102}\x{1167}/\x{E6A2}/g;
  s/\x{1107}\x{1100}\x{119E}/\x{E69D}/g;
  s/\x{1107}\x{1100}\x{1175}/\x{E699}/g;
  s/\x{1107}\x{1100}\x{1173}/\x{E692}/g;
  s/\x{1107}\x{1100}\x{116E}/\x{E68B}/g;
  s/\x{1107}\x{1100}\x{1169}/\x{E687}/g;
  s/\x{1107}\x{1100}\x{1167}/\x{E682}/g;
  s/\x{1107}\x{1100}\x{1165}/\x{E67E}/g;
  s/\x{1107}\x{1100}\x{1161}/\x{E679}/g;
  s/\x{1106}\x{119E}\x{11F0}/\x{E574}/g;
  s/\x{1106}\x{119E}\x{11EB}/\x{E572}/g;
  s/\x{1106}\x{119E}\x{11C0}/\x{E577}/g;
  s/\x{1106}\x{119E}\x{11BE}/\x{E576}/g;
  s/\x{1106}\x{119E}\x{11BD}/\x{E575}/g;
  s/\x{1106}\x{119E}\x{11BC}/\x{E573}/g;
  s/\x{1106}\x{119E}\x{11BA}/\x{E570}/g;
  s/\x{1106}\x{119E}\x{11B8}/\x{E56F}/g;
  s/\x{1106}\x{119E}\x{11B7}/\x{E56E}/g;
  s/\x{1106}\x{119E}\x{11AF}/\x{E566}/g;
  s/\x{1106}\x{119E}\x{11AE}/\x{E564}/g;
  s/\x{1106}\x{119E}\x{11AB}/\x{E563}/g;
  s/\x{1106}\x{119E}\x{11A8}/\x{E561}/g;
  s/\x{1106}\x{119E}\x{1175}/\x{E57B}/g;
  s/\x{1106}\x{119E}\x{1165}/\x{E578}/g;
  s/\x{1106}\x{1175}\x{11F9}/\x{E558}/g;
  s/\x{1106}\x{1175}\x{11F0}/\x{E556}/g;
  s/\x{1106}\x{1175}\x{11EB}/\x{E555}/g;
  s/\x{1106}\x{1175}\x{1173}/\x{E55E}/g;
  s/\x{1106}\x{1175}\x{1167}/\x{E55C}/g;
  s/\x{1106}\x{1175}\x{1163}/\x{E55A}/g;
  s/\x{1106}\x{1175}\x{1161}/\x{E559}/g;
  s/\x{1106}\x{1173}\x{11F0}/\x{E543}/g;
  s/\x{1106}\x{1173}\x{11EB}/\x{E542}/g;
  s/\x{1106}\x{1173}\x{1173}/\x{E545}/g;
  s/\x{1106}\x{1173}\x{116E}/\x{E544}/g;
  s/\x{1106}\x{1172}\x{1175}/\x{E53B}/g;
  s/\x{1106}\x{116E}\x{11F9}/\x{E534}/g;
  s/\x{1106}\x{116E}\x{11F0}/\x{E533}/g;
  s/\x{1106}\x{116E}\x{11EB}/\x{E530}/g;
  s/\x{1106}\x{116E}\x{116E}/\x{E538}/g;
  s/\x{1106}\x{116E}\x{1167}/\x{E536}/g;
  s/\x{1106}\x{116D}\x{1175}/\x{E526}/g;
  s/\x{1106}\x{116D}\x{1169}/\x{E525}/g;
  s/\x{1106}\x{1169}\x{11F0}/\x{E51E}/g;
  s/\x{1106}\x{1169}\x{11EB}/\x{E51D}/g;
  s/\x{1106}\x{1169}\x{116E}/\x{E523}/g;
  s/\x{1106}\x{1169}\x{1169}/\x{E521}/g;
  s/\x{1106}\x{1169}\x{1165}/\x{E51F}/g;
  s/\x{1106}\x{1167}\x{11F9}/\x{E516}/g;
  s/\x{1106}\x{1167}\x{11F0}/\x{E514}/g;
  s/\x{1106}\x{1167}\x{11EB}/\x{E513}/g;
  s/\x{1106}\x{1165}\x{11F9}/\x{E50C}/g;
  s/\x{1106}\x{1165}\x{11F0}/\x{E50B}/g;
  s/\x{1106}\x{1165}\x{1173}/\x{E50E}/g;
  s/\x{1106}\x{1165}\x{1169}/\x{E50D}/g;
  s/\x{1106}\x{1163}\x{1169}/\x{E506}/g;
  s/\x{1106}\x{1161}\x{11F9}/\x{E501}/g;
  s/\x{1106}\x{1161}\x{11F0}/\x{E500}/g;
  s/\x{1106}\x{1161}\x{11EB}/\x{E4FF}/g;
  s/\x{1106}\x{1161}\x{116E}/\x{E503}/g;
  s/\x{1106}\x{1161}\x{1169}/\x{E502}/g;
  s/\x{1106}\x{110B}\x{1175}/\x{E5CC}/g;
  s/\x{1106}\x{110B}\x{1173}/\x{E5C8}/g;
  s/\x{1106}\x{110B}\x{1172}/\x{E5C6}/g;
  s/\x{1106}\x{110B}\x{116E}/\x{E5C1}/g;
  s/\x{1106}\x{110B}\x{1169}/\x{E5BD}/g;
  s/\x{1106}\x{110B}\x{1161}/\x{E5B9}/g;
  s/\x{1106}\x{1109}\x{1169}/\x{E5B7}/g;
  s/\x{1106}\x{1107}\x{119E}/\x{E5B5}/g;
  s/\x{1106}\x{1107}\x{1175}/\x{E5B2}/g;
  s/\x{1106}\x{1107}\x{1173}/\x{E5B0}/g;
  s/\x{1106}\x{1107}\x{1172}/\x{E5AE}/g;
  s/\x{1106}\x{1107}\x{116E}/\x{E5A6}/g;
  s/\x{1106}\x{1107}\x{116D}/\x{E5A5}/g;
  s/\x{1106}\x{1107}\x{1169}/\x{E59E}/g;
  s/\x{1106}\x{1107}\x{1167}/\x{E59A}/g;
  s/\x{1106}\x{1107}\x{1165}/\x{E598}/g;
  s/\x{1106}\x{1107}\x{1163}/\x{E597}/g;
  s/\x{1106}\x{1107}\x{1161}/\x{E593}/g;
  s/\x{1106}\x{1103}\x{1169}/\x{E592}/g;
  s/\x{1106}\x{1103}\x{1167}/\x{E590}/g;
  s/\x{1106}\x{1100}\x{1167}/\x{E58E}/g;
  s/\x{1106}\x{1100}\x{1163}/\x{E58C}/g;
  s/\x{1105}\x{119E}\x{11F0}/\x{E481}/g;
  s/\x{1105}\x{119E}\x{11EB}/\x{E480}/g;
  s/\x{1105}\x{119E}\x{11C2}/\x{E485}/g;
  s/\x{1105}\x{119E}\x{11C0}/\x{E484}/g;
  s/\x{1105}\x{119E}\x{11BE}/\x{E483}/g;
  s/\x{1105}\x{119E}\x{11BD}/\x{E482}/g;
  s/\x{1105}\x{119E}\x{11BA}/\x{E47E}/g;
  s/\x{1105}\x{119E}\x{11B8}/\x{E47B}/g;
  s/\x{1105}\x{119E}\x{11B7}/\x{E477}/g;
  s/\x{1105}\x{119E}\x{11AF}/\x{E470}/g;
  s/\x{1105}\x{119E}\x{11AE}/\x{E46F}/g;
  s/\x{1105}\x{119E}\x{11AB}/\x{E46E}/g;
  s/\x{1105}\x{119E}\x{11A8}/\x{E46D}/g;
  s/\x{1105}\x{119E}\x{1175}/\x{E487}/g;
  s/\x{1105}\x{119E}\x{1161}/\x{E486}/g;
  s/\x{1105}\x{1175}\x{11F9}/\x{E45E}/g;
  s/\x{1105}\x{1175}\x{11F0}/\x{E45D}/g;
  s/\x{1105}\x{1175}\x{1175}/\x{E46B}/g;
  s/\x{1105}\x{1175}\x{116E}/\x{E469}/g;
  s/\x{1105}\x{1175}\x{116D}/\x{E468}/g;
  s/\x{1105}\x{1175}\x{1169}/\x{E466}/g;
  s/\x{1105}\x{1175}\x{1167}/\x{E464}/g;
  s/\x{1105}\x{1175}\x{1163}/\x{E460}/g;
  s/\x{1105}\x{1175}\x{1161}/\x{E45F}/g;
  s/\x{1105}\x{1173}\x{11F0}/\x{E454}/g;
  s/\x{1105}\x{1173}\x{1173}/\x{E456}/g;
  s/\x{1105}\x{1173}\x{116E}/\x{E455}/g;
  s/\x{1105}\x{1172}\x{11F9}/\x{E446}/g;
  s/\x{1105}\x{1172}\x{11F0}/\x{E445}/g;
  s/\x{1105}\x{1172}\x{1175}/\x{E44B}/g;
  s/\x{1105}\x{1172}\x{1167}/\x{E447}/g;
  s/\x{1105}\x{116E}\x{11F9}/\x{E43D}/g;
  s/\x{1105}\x{116E}\x{11F0}/\x{E43C}/g;
  s/\x{1105}\x{116E}\x{116E}/\x{E442}/g;
  s/\x{1105}\x{116E}\x{1167}/\x{E440}/g;
  s/\x{1105}\x{116E}\x{1161}/\x{E43E}/g;
  s/\x{1105}\x{116D}\x{11F0}/\x{E434}/g;
  s/\x{1105}\x{116D}\x{1175}/\x{E437}/g;
  s/\x{1105}\x{116D}\x{1169}/\x{E436}/g;
  s/\x{1105}\x{116D}\x{1167}/\x{E435}/g;
  s/\x{1105}\x{1169}\x{11F0}/\x{E42C}/g;
  s/\x{1105}\x{1169}\x{1169}/\x{E430}/g;
  s/\x{1105}\x{1167}\x{11F9}/\x{E424}/g;
  s/\x{1105}\x{1167}\x{11F0}/\x{E423}/g;
  s/\x{1105}\x{1165}\x{11F0}/\x{E419}/g;
  s/\x{1105}\x{1165}\x{116E}/\x{E41C}/g;
  s/\x{1105}\x{1165}\x{1169}/\x{E41B}/g;
  s/\x{1105}\x{1163}\x{11F0}/\x{E414}/g;
  s/\x{1105}\x{1163}\x{1169}/\x{E415}/g;
  s/\x{1105}\x{1161}\x{11F9}/\x{E40E}/g;
  s/\x{1105}\x{1161}\x{11F0}/\x{E40B}/g;
  s/\x{1105}\x{1161}\x{11EB}/\x{E40A}/g;
  s/\x{1105}\x{1161}\x{116E}/\x{E411}/g;
  s/\x{1105}\x{1161}\x{1169}/\x{E40F}/g;
  s/\x{1105}\x{1112}\x{1175}/\x{E4F0}/g;
  s/\x{1105}\x{1112}\x{1169}/\x{E4EE}/g;
  s/\x{1105}\x{1112}\x{1163}/\x{E4EC}/g;
  s/\x{1105}\x{1112}\x{1161}/\x{E4EA}/g;
  s/\x{1105}\x{110F}\x{1161}/\x{E4E9}/g;
  s/\x{1105}\x{110C}\x{1172}/\x{E4E7}/g;
  s/\x{1105}\x{110C}\x{1165}/\x{E4E5}/g;
  s/\x{1105}\x{1109}\x{1173}/\x{E4E4}/g;
  s/\x{1105}\x{1109}\x{1169}/\x{E4E2}/g;
  s/\x{1105}\x{1107}\x{1161}/\x{E4DE}/g;
  s/\x{1105}\x{1106}\x{1161}/\x{E4DC}/g;
  s/\x{1105}\x{1105}\x{1175}/\x{E4D8}/g;
  s/\x{1105}\x{1105}\x{1173}/\x{E4D4}/g;
  s/\x{1105}\x{1105}\x{1172}/\x{E4D1}/g;
  s/\x{1105}\x{1105}\x{116E}/\x{E4C9}/g;
  s/\x{1105}\x{1105}\x{116D}/\x{E4C8}/g;
  s/\x{1105}\x{1105}\x{1169}/\x{E4C6}/g;
  s/\x{1105}\x{1105}\x{1167}/\x{E4C2}/g;
  s/\x{1105}\x{1105}\x{1165}/\x{E4BC}/g;
  s/\x{1105}\x{1105}\x{1163}/\x{E4B9}/g;
  s/\x{1105}\x{1105}\x{1161}/\x{E4B2}/g;
  s/\x{1105}\x{1102}\x{119E}/\x{E4AB}/g;
  s/\x{1105}\x{1102}\x{1173}/\x{E4A6}/g;
  s/\x{1105}\x{1102}\x{1172}/\x{E4A4}/g;
  s/\x{1105}\x{1102}\x{116E}/\x{E4A3}/g;
  s/\x{1105}\x{1102}\x{116D}/\x{E4A1}/g;
  s/\x{1105}\x{1102}\x{1169}/\x{E49C}/g;
  s/\x{1105}\x{1102}\x{1165}/\x{E498}/g;
  s/\x{1105}\x{1102}\x{1161}/\x{E494}/g;
  s/\x{1105}\x{1100}\x{1169}/\x{E491}/g;
  s/\x{1105}\x{1100}\x{1161}/\x{E490}/g;
  s/\x{1103}\x{119E}\x{11F0}/\x{E3A5}/g;
  s/\x{1103}\x{119E}\x{11EB}/\x{E3A3}/g;
  s/\x{1103}\x{119E}\x{11BE}/\x{E3A7}/g;
  s/\x{1103}\x{119E}\x{11BD}/\x{E3A6}/g;
  s/\x{1103}\x{119E}\x{11BC}/\x{E3A4}/g;
  s/\x{1103}\x{119E}\x{11BA}/\x{E39E}/g;
  s/\x{1103}\x{119E}\x{11B8}/\x{E39B}/g;
  s/\x{1103}\x{119E}\x{11B7}/\x{E397}/g;
  s/\x{1103}\x{119E}\x{11AF}/\x{E38F}/g;
  s/\x{1103}\x{119E}\x{11AE}/\x{E38E}/g;
  s/\x{1103}\x{119E}\x{11AB}/\x{E38D}/g;
  s/\x{1103}\x{119E}\x{11A8}/\x{E38B}/g;
  s/\x{1103}\x{119E}\x{1175}/\x{E3A8}/g;
  s/\x{1103}\x{1175}\x{11F9}/\x{E382}/g;
  s/\x{1103}\x{1175}\x{11F0}/\x{E381}/g;
  s/\x{1103}\x{1175}\x{11EB}/\x{E380}/g;
  s/\x{1103}\x{1175}\x{116E}/\x{E389}/g;
  s/\x{1103}\x{1175}\x{1167}/\x{E387}/g;
  s/\x{1103}\x{1175}\x{1163}/\x{E384}/g;
  s/\x{1103}\x{1175}\x{1161}/\x{E383}/g;
  s/\x{1103}\x{1173}\x{11F9}/\x{E379}/g;
  s/\x{1103}\x{1173}\x{11F0}/\x{E374}/g;
  s/\x{1103}\x{1173}\x{11EB}/\x{E373}/g;
  s/\x{1103}\x{1173}\x{1173}/\x{E377}/g;
  s/\x{1103}\x{1173}\x{116E}/\x{E376}/g;
  s/\x{1103}\x{1172}\x{11F0}/\x{E35F}/g;
  s/\x{1103}\x{1172}\x{1175}/\x{E368}/g;
  s/\x{1103}\x{1172}\x{1167}/\x{E361}/g;
  s/\x{1103}\x{116E}\x{11F0}/\x{E355}/g;
  s/\x{1103}\x{116E}\x{1167}/\x{E359}/g;
  s/\x{1103}\x{116E}\x{1161}/\x{E356}/g;
  s/\x{1103}\x{116D}\x{11F0}/\x{E34C}/g;
  s/\x{1103}\x{116D}\x{1175}/\x{E34D}/g;
  s/\x{1103}\x{1169}\x{11F0}/\x{E344}/g;
  s/\x{1103}\x{1169}\x{11EB}/\x{E343}/g;
  s/\x{1103}\x{1169}\x{1169}/\x{E348}/g;
  s/\x{1103}\x{1167}\x{11F9}/\x{E336}/g;
  s/\x{1103}\x{1167}\x{11F0}/\x{E335}/g;
  s/\x{1103}\x{1165}\x{11F0}/\x{E32A}/g;
  s/\x{1103}\x{1165}\x{11EB}/\x{E329}/g;
  s/\x{1103}\x{1165}\x{1173}/\x{E32E}/g;
  s/\x{1103}\x{1165}\x{116E}/\x{E32C}/g;
  s/\x{1103}\x{1165}\x{1169}/\x{E32B}/g;
  s/\x{1103}\x{1163}\x{11F0}/\x{E323}/g;
  s/\x{1103}\x{1163}\x{1169}/\x{E325}/g;
  s/\x{1103}\x{1161}\x{11F9}/\x{E31E}/g;
  s/\x{1103}\x{1161}\x{11F0}/\x{E31B}/g;
  s/\x{1103}\x{1161}\x{11EB}/\x{E31A}/g;
  s/\x{1103}\x{1161}\x{1169}/\x{E31F}/g;
  s/\x{1103}\x{110C}\x{119E}/\x{E401}/g;
  s/\x{1103}\x{110C}\x{1163}/\x{E3FF}/g;
  s/\x{1103}\x{1109}\x{1175}/\x{E3FD}/g;
  s/\x{1103}\x{1109}\x{1173}/\x{E3FC}/g;
  s/\x{1103}\x{1109}\x{1172}/\x{E3F9}/g;
  s/\x{1103}\x{1107}\x{116E}/\x{E3F7}/g;
  s/\x{1103}\x{1107}\x{1161}/\x{E3F5}/g;
  s/\x{1103}\x{1106}\x{1161}/\x{E3F3}/g;
  s/\x{1103}\x{1105}\x{1175}/\x{E3F0}/g;
  s/\x{1103}\x{1103}\x{119E}/\x{E3E5}/g;
  s/\x{1103}\x{1100}\x{1175}/\x{E3B6}/g;
  s/\x{1103}\x{1100}\x{1173}/\x{E3B4}/g;
  s/\x{1102}\x{119E}\x{11F0}/\x{E296}/g;
  s/\x{1102}\x{119E}\x{11EB}/\x{E294}/g;
  s/\x{1102}\x{119E}\x{11BE}/\x{E298}/g;
  s/\x{1102}\x{119E}\x{11BD}/\x{E297}/g;
  s/\x{1102}\x{119E}\x{11BC}/\x{E295}/g;
  s/\x{1102}\x{119E}\x{11BA}/\x{E292}/g;
  s/\x{1102}\x{119E}\x{11B8}/\x{E291}/g;
  s/\x{1102}\x{119E}\x{11B7}/\x{E290}/g;
  s/\x{1102}\x{119E}\x{11AF}/\x{E288}/g;
  s/\x{1102}\x{119E}\x{11AE}/\x{E286}/g;
  s/\x{1102}\x{119E}\x{11AB}/\x{E285}/g;
  s/\x{1102}\x{119E}\x{11A8}/\x{E284}/g;
  s/\x{1102}\x{119E}\x{1175}/\x{E299}/g;
  s/\x{1102}\x{1175}\x{11F9}/\x{E27B}/g;
  s/\x{1102}\x{1175}\x{11F0}/\x{E27A}/g;
  s/\x{1102}\x{1175}\x{11EB}/\x{E279}/g;
  s/\x{1102}\x{1175}\x{116E}/\x{E282}/g;
  s/\x{1102}\x{1175}\x{1167}/\x{E280}/g;
  s/\x{1102}\x{1175}\x{1163}/\x{E27D}/g;
  s/\x{1102}\x{1175}\x{1161}/\x{E27C}/g;
  s/\x{1102}\x{1173}\x{11F0}/\x{E26A}/g;
  s/\x{1102}\x{1173}\x{1173}/\x{E26C}/g;
  s/\x{1102}\x{1172}\x{11F0}/\x{E262}/g;
  s/\x{1102}\x{1172}\x{1175}/\x{E264}/g;
  s/\x{1102}\x{1172}\x{1165}/\x{E263}/g;
  s/\x{1102}\x{116E}\x{1167}/\x{E25C}/g;
  s/\x{1102}\x{116E}\x{1161}/\x{E25A}/g;
  s/\x{1102}\x{116D}\x{11F0}/\x{E24E}/g;
  s/\x{1102}\x{116D}\x{1175}/\x{E251}/g;
  s/\x{1102}\x{116D}\x{1161}/\x{E24F}/g;
  s/\x{1102}\x{1169}\x{11F0}/\x{E246}/g;
  s/\x{1102}\x{1169}\x{11EB}/\x{E245}/g;
  s/\x{1102}\x{1169}\x{1165}/\x{E24A}/g;
  s/\x{1102}\x{1167}\x{11F9}/\x{E23C}/g;
  s/\x{1102}\x{1167}\x{11F0}/\x{E23B}/g;
  s/\x{1102}\x{1165}\x{11F0}/\x{E22D}/g;
  s/\x{1102}\x{1165}\x{1173}/\x{E230}/g;
  s/\x{1102}\x{1165}\x{116E}/\x{E22E}/g;
  s/\x{1102}\x{1163}\x{11F0}/\x{E223}/g;
  s/\x{1102}\x{1163}\x{1169}/\x{E224}/g;
  s/\x{1102}\x{1161}\x{11F9}/\x{E21B}/g;
  s/\x{1102}\x{1161}\x{11F0}/\x{E217}/g;
  s/\x{1102}\x{1161}\x{11EB}/\x{E215}/g;
  s/\x{1102}\x{1161}\x{116E}/\x{E21D}/g;
  s/\x{1102}\x{1161}\x{1169}/\x{E21C}/g;
  s/\x{1102}\x{1112}\x{1175}/\x{E30B}/g;
  s/\x{1102}\x{1112}\x{1172}/\x{E309}/g;
  s/\x{1102}\x{1112}\x{1169}/\x{E308}/g;
  s/\x{1102}\x{1112}\x{1163}/\x{E306}/g;
  s/\x{1102}\x{110C}\x{1175}/\x{E305}/g;
  s/\x{1102}\x{110C}\x{1173}/\x{E304}/g;
  s/\x{1102}\x{1109}\x{1175}/\x{E302}/g;
  s/\x{1102}\x{1107}\x{119E}/\x{E300}/g;
  s/\x{1102}\x{1107}\x{1175}/\x{E2FE}/g;
  s/\x{1102}\x{1107}\x{1173}/\x{E2FC}/g;
  s/\x{1102}\x{1107}\x{1172}/\x{E2FA}/g;
  s/\x{1102}\x{1107}\x{116E}/\x{E2F4}/g;
  s/\x{1102}\x{1107}\x{116D}/\x{E2F3}/g;
  s/\x{1102}\x{1107}\x{1169}/\x{E2EF}/g;
  s/\x{1102}\x{1107}\x{1167}/\x{E2ED}/g;
  s/\x{1102}\x{1107}\x{1165}/\x{E2EB}/g;
  s/\x{1102}\x{1107}\x{1163}/\x{E2EA}/g;
  s/\x{1102}\x{1107}\x{1161}/\x{E2E8}/g;
  s/\x{1102}\x{1103}\x{119E}/\x{E2E6}/g;
  s/\x{1102}\x{1103}\x{1175}/\x{E2E4}/g;
  s/\x{1102}\x{1103}\x{1173}/\x{E2E2}/g;
  s/\x{1102}\x{1103}\x{1172}/\x{E2E0}/g;
  s/\x{1102}\x{1103}\x{116E}/\x{E2DA}/g;
  s/\x{1102}\x{1103}\x{116D}/\x{E2D9}/g;
  s/\x{1102}\x{1103}\x{1169}/\x{E2D4}/g;
  s/\x{1102}\x{1103}\x{1167}/\x{E2D2}/g;
  s/\x{1102}\x{1103}\x{1165}/\x{E2CF}/g;
  s/\x{1102}\x{1103}\x{1163}/\x{E2CE}/g;
  s/\x{1102}\x{1103}\x{1161}/\x{E2CB}/g;
  s/\x{1102}\x{1102}\x{119E}/\x{E2C9}/g;
  s/\x{1102}\x{1102}\x{1173}/\x{E2C7}/g;
  s/\x{1102}\x{1102}\x{116E}/\x{E2C6}/g;
  s/\x{1102}\x{1102}\x{1169}/\x{E2C5}/g;
  s/\x{1102}\x{1102}\x{1165}/\x{E2C3}/g;
  s/\x{1102}\x{1102}\x{1163}/\x{E2C1}/g;
  s/\x{1102}\x{1102}\x{1161}/\x{E2C0}/g;
  s/\x{1102}\x{1100}\x{119E}/\x{E2BE}/g;
  s/\x{1102}\x{1100}\x{1175}/\x{E2BC}/g;
  s/\x{1102}\x{1100}\x{1173}/\x{E2BA}/g;
  s/\x{1102}\x{1100}\x{1172}/\x{E2B8}/g;
  s/\x{1102}\x{1100}\x{116E}/\x{E2B2}/g;
  s/\x{1102}\x{1100}\x{116D}/\x{E2B1}/g;
  s/\x{1102}\x{1100}\x{1169}/\x{E2AD}/g;
  s/\x{1102}\x{1100}\x{1167}/\x{E2AA}/g;
  s/\x{1102}\x{1100}\x{1165}/\x{E2A8}/g;
  s/\x{1102}\x{1100}\x{1163}/\x{E2A7}/g;
  s/\x{1102}\x{1100}\x{1161}/\x{E2A5}/g;
  s/\x{1100}\x{119E}\x{11F0}/\x{E1C4}/g;
  s/\x{1100}\x{119E}\x{11EB}/\x{E1C2}/g;
  s/\x{1100}\x{119E}\x{11C2}/\x{E1C9}/g;
  s/\x{1100}\x{119E}\x{11C0}/\x{E1C8}/g;
  s/\x{1100}\x{119E}\x{11BE}/\x{E1C7}/g;
  s/\x{1100}\x{119E}\x{11BD}/\x{E1C6}/g;
  s/\x{1100}\x{119E}\x{11BC}/\x{E1C3}/g;
  s/\x{1100}\x{119E}\x{11BA}/\x{E1BF}/g;
  s/\x{1100}\x{119E}\x{11B8}/\x{E1BC}/g;
  s/\x{1100}\x{119E}\x{11B7}/\x{E1BA}/g;
  s/\x{1100}\x{119E}\x{11AF}/\x{E1AD}/g;
  s/\x{1100}\x{119E}\x{11AE}/\x{E1AB}/g;
  s/\x{1100}\x{119E}\x{11AB}/\x{E1AA}/g;
  s/\x{1100}\x{119E}\x{11A8}/\x{E1A8}/g;
  s/\x{1100}\x{119E}\x{1175}/\x{E1CD}/g;
  s/\x{1100}\x{119E}\x{1165}/\x{E1CA}/g;
  s/\x{1100}\x{1175}\x{11F9}/\x{E199}/g;
  s/\x{1100}\x{1175}\x{11F0}/\x{E198}/g;
  s/\x{1100}\x{1175}\x{11EB}/\x{E197}/g;
  s/\x{1100}\x{1175}\x{119E}/\x{E1A6}/g;
  s/\x{1100}\x{1175}\x{1175}/\x{E1A5}/g;
  s/\x{1100}\x{1175}\x{1173}/\x{E1A4}/g;
  s/\x{1100}\x{1175}\x{1172}/\x{E1A2}/g;
  s/\x{1100}\x{1175}\x{116E}/\x{E1A1}/g;
  s/\x{1100}\x{1175}\x{116D}/\x{E1A0}/g;
  s/\x{1100}\x{1175}\x{1167}/\x{E19E}/g;
  s/\x{1100}\x{1175}\x{1163}/\x{E19B}/g;
  s/\x{1100}\x{1175}\x{1161}/\x{E19A}/g;
  s/\x{1100}\x{1173}\x{11F9}/\x{E180}/g;
  s/\x{1100}\x{1173}\x{11F0}/\x{E17F}/g;
  s/\x{1100}\x{1173}\x{11EB}/\x{E17E}/g;
  s/\x{1100}\x{1173}\x{1173}/\x{E184}/g;
  s/\x{1100}\x{1173}\x{116E}/\x{E183}/g;
  s/\x{1100}\x{1173}\x{1165}/\x{E181}/g;
  s/\x{1100}\x{1172}\x{11F9}/\x{E159}/g;
  s/\x{1100}\x{1172}\x{11F0}/\x{E158}/g;
  s/\x{1100}\x{1172}\x{1175}/\x{E16A}/g;
  s/\x{1100}\x{1172}\x{1167}/\x{E15E}/g;
  s/\x{1100}\x{1172}\x{1165}/\x{E15C}/g;
  s/\x{1100}\x{1172}\x{1161}/\x{E15A}/g;
  s/\x{1100}\x{116E}\x{11F9}/\x{E145}/g;
  s/\x{1100}\x{116E}\x{11F0}/\x{E142}/g;
  s/\x{1100}\x{116E}\x{116E}/\x{E152}/g;
  s/\x{1100}\x{116E}\x{1167}/\x{E150}/g;
  s/\x{1100}\x{116E}\x{1161}/\x{E146}/g;
  s/\x{1100}\x{116D}\x{11F0}/\x{E12E}/g;
  s/\x{1100}\x{116D}\x{1175}/\x{E136}/g;
  s/\x{1100}\x{116D}\x{1163}/\x{E132}/g;
  s/\x{1100}\x{116D}\x{1161}/\x{E12F}/g;
  s/\x{1100}\x{1169}\x{11F0}/\x{E11C}/g;
  s/\x{1100}\x{1169}\x{11EB}/\x{E11B}/g;
  s/\x{1100}\x{1169}\x{1169}/\x{E129}/g;
  s/\x{1100}\x{1169}\x{1165}/\x{E126}/g;
  s/\x{1100}\x{1169}\x{1163}/\x{E124}/g;
  s/\x{1100}\x{1167}\x{11F9}/\x{E110}/g;
  s/\x{1100}\x{1167}\x{11F0}/\x{E10E}/g;
  s/\x{1100}\x{1167}\x{11EB}/\x{E10D}/g;
  s/\x{1100}\x{1165}\x{11F9}/\x{E0FF}/g;
  s/\x{1100}\x{1165}\x{11F0}/\x{E0FE}/g;
  s/\x{1100}\x{1165}\x{11EB}/\x{E0FD}/g;
  s/\x{1100}\x{1165}\x{1173}/\x{E100}/g;
  s/\x{1100}\x{1163}\x{11F9}/\x{E0F1}/g;
  s/\x{1100}\x{1163}\x{11F0}/\x{E0F0}/g;
  s/\x{1100}\x{1163}\x{1169}/\x{E0F2}/g;
  s/\x{1100}\x{1161}\x{11F9}/\x{E0E8}/g;
  s/\x{1100}\x{1161}\x{11F0}/\x{E0E6}/g;
  s/\x{1100}\x{1161}\x{11EB}/\x{E0E4}/g;
  s/\x{1100}\x{1161}\x{116E}/\x{E0EA}/g;
  s/\x{1100}\x{1161}\x{1169}/\x{E0E9}/g;
  s/\x{1100}\x{1103}\x{1173}/\x{E205}/g;
  s/\x{1100}\x{1103}\x{1165}/\x{E204}/g;
  s/\x{1100}\x{1100}\x{119E}/\x{E202}/g;
  s/\x{1159}\x{119E}/\x{F665}/g;
  s/\x{1159}\x{1175}/\x{F65C}/g;
  s/\x{1159}\x{1173}/\x{F651}/g;
  s/\x{1159}\x{1172}/\x{F641}/g;
  s/\x{1159}\x{116E}/\x{F630}/g;
  s/\x{1159}\x{116D}/\x{F62B}/g;
  s/\x{1159}\x{1169}/\x{F614}/g;
  s/\x{1159}\x{1167}/\x{F609}/g;
  s/\x{1159}\x{1165}/\x{F600}/g;
  s/\x{1159}\x{1163}/\x{F5F6}/g;
  s/\x{1159}\x{1161}/\x{F5E8}/g;
  s/\x{1155}\x{1175}/\x{F394}/g;
  s/\x{1155}\x{1173}/\x{F391}/g;
  s/\x{1155}\x{1172}/\x{F38A}/g;
  s/\x{1155}\x{116E}/\x{F388}/g;
  s/\x{1155}\x{1169}/\x{F384}/g;
  s/\x{1155}\x{1167}/\x{F382}/g;
  s/\x{1155}\x{1163}/\x{F380}/g;
  s/\x{1155}\x{1161}/\x{F37A}/g;
  s/\x{1154}\x{1175}/\x{F375}/g;
  s/\x{1154}\x{1173}/\x{F374}/g;
  s/\x{1154}\x{1172}/\x{F370}/g;
  s/\x{1154}\x{116E}/\x{F36C}/g;
  s/\x{1154}\x{116D}/\x{F369}/g;
  s/\x{1154}\x{1169}/\x{F368}/g;
  s/\x{1154}\x{1167}/\x{F365}/g;
  s/\x{1154}\x{1165}/\x{F363}/g;
  s/\x{1154}\x{1161}/\x{F35F}/g;
  s/\x{1150}\x{1175}/\x{F2C2}/g;
  s/\x{1150}\x{1173}/\x{F2BF}/g;
  s/\x{1150}\x{1172}/\x{F2B9}/g;
  s/\x{1150}\x{116E}/\x{F2B6}/g;
  s/\x{1150}\x{1169}/\x{F2B3}/g;
  s/\x{1150}\x{1167}/\x{F2B1}/g;
  s/\x{1150}\x{1163}/\x{F2AD}/g;
  s/\x{1150}\x{1161}/\x{F2A6}/g;
  s/\x{114E}\x{1175}/\x{F286}/g;
  s/\x{114E}\x{1173}/\x{F282}/g;
  s/\x{114E}\x{1172}/\x{F27D}/g;
  s/\x{114E}\x{116E}/\x{F279}/g;
  s/\x{114E}\x{116D}/\x{F277}/g;
  s/\x{114E}\x{1169}/\x{F275}/g;
  s/\x{114E}\x{1167}/\x{F271}/g;
  s/\x{114E}\x{1165}/\x{F26F}/g;
  s/\x{114E}\x{1163}/\x{F26D}/g;
  s/\x{114E}\x{1161}/\x{F268}/g;
  s/\x{114C}\x{119E}/\x{F15C}/g;
  s/\x{114C}\x{1175}/\x{F154}/g;
  s/\x{114C}\x{1173}/\x{F149}/g;
  s/\x{114C}\x{1172}/\x{F13C}/g;
  s/\x{114C}\x{116E}/\x{F128}/g;
  s/\x{114C}\x{116D}/\x{F121}/g;
  s/\x{114C}\x{1169}/\x{F107}/g;
  s/\x{114C}\x{1167}/\x{EFF8}/g;
  s/\x{114C}\x{1165}/\x{EFE7}/g;
  s/\x{114C}\x{1163}/\x{EFE0}/g;
  s/\x{114C}\x{1161}/\x{EFD2}/g;
  s/\x{1140}\x{119E}/\x{ED8F}/g;
  s/\x{1140}\x{1175}/\x{ED7F}/g;
  s/\x{1140}\x{1173}/\x{ED6B}/g;
  s/\x{1140}\x{1172}/\x{ED56}/g;
  s/\x{1140}\x{116E}/\x{ED41}/g;
  s/\x{1140}\x{116D}/\x{ED39}/g;
  s/\x{1140}\x{1169}/\x{ED2D}/g;
  s/\x{1140}\x{1167}/\x{ED1C}/g;
  s/\x{1140}\x{1165}/\x{ED0C}/g;
  s/\x{1140}\x{1163}/\x{ED01}/g;
  s/\x{1140}\x{1161}/\x{ECF0}/g;
  s/\x{113F}\x{1175}/\x{ECED}/g;
  s/\x{113F}\x{1173}/\x{ECEB}/g;
  s/\x{113F}\x{1172}/\x{ECE7}/g;
  s/\x{113F}\x{116E}/\x{ECE5}/g;
  s/\x{113F}\x{1167}/\x{ECE3}/g;
  s/\x{113F}\x{1163}/\x{ECE1}/g;
  s/\x{113E}\x{1175}/\x{ECDD}/g;
  s/\x{113E}\x{1173}/\x{ECDA}/g;
  s/\x{113E}\x{1172}/\x{ECD6}/g;
  s/\x{113E}\x{116E}/\x{ECD2}/g;
  s/\x{113E}\x{1169}/\x{ECCF}/g;
  s/\x{113E}\x{1167}/\x{ECCD}/g;
  s/\x{113E}\x{1163}/\x{ECCA}/g;
  s/\x{113E}\x{1161}/\x{ECC6}/g;
  s/\x{113D}\x{1175}/\x{ECC2}/g;
  s/\x{113D}\x{1173}/\x{ECC0}/g;
  s/\x{113D}\x{1172}/\x{ECBA}/g;
  s/\x{113D}\x{116E}/\x{ECB8}/g;
  s/\x{113D}\x{1167}/\x{ECB6}/g;
  s/\x{113D}\x{1163}/\x{ECB3}/g;
  s/\x{113C}\x{1175}/\x{ECB1}/g;
  s/\x{113C}\x{1173}/\x{ECAE}/g;
  s/\x{113C}\x{1172}/\x{ECAA}/g;
  s/\x{113C}\x{116E}/\x{ECA6}/g;
  s/\x{113C}\x{1167}/\x{ECA3}/g;
  s/\x{113C}\x{1165}/\x{ECA1}/g;
  s/\x{113C}\x{1163}/\x{EC9D}/g;
  s/\x{113C}\x{1161}/\x{EC99}/g;
  s/\x{1112}\x{119E}/\x{F537}/g;
  s/\x{1111}\x{119E}/\x{F48E}/g;
  s/\x{1110}\x{119E}/\x{F43C}/g;
  s/\x{110F}\x{119E}/\x{F3EA}/g;
  s/\x{110E}\x{119E}/\x{F341}/g;
  s/\x{110C}\x{119E}/\x{F1FC}/g;
  s/\x{110B}\x{119E}/\x{EE88}/g;
  s/\x{1109}\x{119E}/\x{E97D}/g;
  s/\x{1107}\x{119E}/\x{E64A}/g;
  s/\x{1106}\x{119E}/\x{E560}/g;
  s/\x{1105}\x{119E}/\x{E46C}/g;
  s/\x{1103}\x{119E}/\x{E38A}/g;
  s/\x{1102}\x{119E}/\x{E283}/g;
  s/\x{1100}\x{119E}/\x{E1A7}/g;
}
